import logging
from dataclasses import dataclass
from typing import Callable, Coroutine, Optional

from telebot import AsyncTeleBot
from telebot import types as tg
from telebot.callback_data import CallbackData

ROUTE_MENU_CALLBACK_DATA = CallbackData("route_to", prefix="menu")
TERMINATE_MENU_CALLBACK_DATA = CallbackData("id", prefix="terminator")
INACTIVE_BUTTON_CALLBACK_DATA = CallbackData(prefix="inactive_button")


class MenuItem:
    def __init__(self, label: str, submenu: Optional["Menu"] = None, terminator: Optional[str] = None):
        self.label = label
        self.submenu = submenu
        self.terminator = terminator

        if not ((self.submenu is None) ^ (self.terminator is None)):
            raise RuntimeError(
                "MenuItem is not configured correctly. Only submenu OR terminator can be specified in one MenuItem."
            )

        self._id: Optional[str] = None
        self._parent_menu: Optional["Menu"] = None

    @property
    def id(self) -> str:
        if self._id is None:
            raise RuntimeError("MenuItem object was not properly initialized.")
        return self._id

    @id.setter
    def id(self, id: str):
        self._id = id

    @property
    def parent_menu(self) -> "Menu":
        if self._parent_menu is None:
            raise RuntimeError("MenuItem object was not properly initialized.")
        return self._parent_menu

    @parent_menu.setter
    def parent_menu(self, parent_menu: "Menu"):
        self._parent_menu = parent_menu

    def get_inline_button(self):
        if self.submenu is not None:

            return tg.InlineKeyboardButton(
                text=self.label,
                callback_data=ROUTE_MENU_CALLBACK_DATA.new(self.submenu.id),
            )
        else:
            return tg.InlineKeyboardButton(
                text=self.label,
                callback_data=TERMINATE_MENU_CALLBACK_DATA.new(self.id),
            )

    def get_blocked_inline_button(self, selected_item_id: str):
        button_text = self.label
        if self.id == selected_item_id:
            button_text = "✅ " + button_text
        return tg.InlineKeyboardButton(
            text=button_text,
            callback_data=INACTIVE_BUTTON_CALLBACK_DATA.new(),
        )


class Menu:
    def __init__(
        self,
        text: str,
        menu_items: list[MenuItem],
    ):
        self._id: Optional[str] = None
        self.parent_menu: Optional["Menu"] = None
        self.text = text
        self.menu_items = menu_items

    @property
    def id(self) -> str:
        if self._id is None:
            raise RuntimeError("Menu object was not properly initialized.")
        return self._id

    @id.setter
    def id(self, id: str):
        self._id = id

    def descendants(self) -> list["Menu"]:
        children = [mi.submenu for mi in self.menu_items if mi.submenu is not None]
        grandchildren: list[Menu] = []
        for menu in children:
            grandchildren.extend(menu.descendants())
        return children + grandchildren

    def get_keyboard_markup(self):
        keyboard = [[menu_item.get_inline_button()] for menu_item in self.menu_items]
        if self.parent_menu is not None:
            keyboard.append([MenuItem(label="back", submenu=self.parent_menu).get_inline_button()])
        return tg.InlineKeyboardMarkup(keyboard=keyboard)

    def get_inactive_keyboard_markup(self, selected_item_id: str):
        return tg.InlineKeyboardMarkup(
            keyboard=[[menu_item.get_blocked_inline_button(selected_item_id)] for menu_item in self.menu_items]
        )


@dataclass
class TerminatorContext:
    bot: AsyncTeleBot
    user: tg.User
    menu_message: tg.Message
    terminator: str


class MenuHandler:
    def __init__(
        self,
        bot_prefix: str,
        menu_tree: Menu,
    ):
        self.menus_list: list[Menu] = [menu_tree]
        self.menus_list.extend(menu_tree.descendants())

        self.init_menu_ids()
        self.init_parent_menu()

        self.menu_by_id: dict[str, Menu] = {m.id: m for m in self.menus_list}

        self.menu_items_list = self.init_item_ids_and_get_item_list()
        self.menu_item_by_id: dict[str, MenuItem] = {mi.id: mi for mi in self.menu_items_list}

        self.logger = logging.getLogger(f"{__name__}.{bot_prefix}")

    def init_item_ids_and_get_item_list(self) -> list[MenuItem]:
        item_list: list[MenuItem] = []
        for menu in self.menus_list:
            for menu_item in menu.menu_items:
                item_list.append(menu_item)
        for i, menu_item in enumerate(item_list):
            menu_item.id = str(i)
        return item_list

    def init_menu_ids(self):
        for i, menu in enumerate(self.menus_list):
            menu.id = str(i)

    def init_parent_menu(self):
        for menu in self.menus_list:
            for menu_item in menu.menu_items:
                menu_item.parent_menu = menu
                if menu_item.submenu is not None:
                    menu_item.submenu.parent_menu = menu

    def get_main_menu(self):
        return self.menu_by_id["0"]

    def setup(
        self,
        bot: AsyncTeleBot,
        on_terminal_menu_option_selected: Callable[[TerminatorContext], Coroutine[None, None, None]],
    ):
        @bot.callback_query_handler(callback_data=ROUTE_MENU_CALLBACK_DATA)
        async def handle_menu(call: tg.CallbackQuery):
            user = call.from_user
            data = ROUTE_MENU_CALLBACK_DATA.parse(call.data)
            route_to = data["route_to"]

            await bot.answer_callback_query(call.id)

            menu = self.menu_by_id[route_to]
            await bot.edit_message_text(
                text=menu.text,
                chat_id=user.id,
                message_id=call.message.id,
                reply_markup=menu.get_keyboard_markup(),
            )

        @bot.callback_query_handler(callback_data=INACTIVE_BUTTON_CALLBACK_DATA)
        async def handle_inactive_menu(call: tg.CallbackQuery):
            await bot.answer_callback_query(call.id)

        @bot.callback_query_handler(callback_data=TERMINATE_MENU_CALLBACK_DATA)
        async def handle_terminator(call: tg.CallbackQuery):
            user = call.from_user
            data = TERMINATE_MENU_CALLBACK_DATA.parse(call.data)
            selected_menu_item_id = data["id"]

            selected_menu_item = self.menu_item_by_id[selected_menu_item_id]
            terminator = selected_menu_item.terminator
            if terminator is None:
                raise RuntimeError("Got MenuItem that is not terminating item.")
            current_menu = self.menu_by_id[selected_menu_item.parent_menu.id]

            await bot.answer_callback_query(call.id)
            await bot.edit_message_reply_markup(
                chat_id=user.id,
                message_id=call.message.id,
                reply_markup=current_menu.get_inactive_keyboard_markup(selected_menu_item_id),
            )

            await on_terminal_menu_option_selected(TerminatorContext(bot, user, call.message, terminator))
