# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pyathenajdbc']

package_data = \
{'': ['*']}

install_requires = \
['future', 'jpype1>=0.6.0,<=0.7.0']

extras_require = \
{'Pandas': ['pandas>=0.19.0'], 'SQLAlchemy': ['sqlalchemy>=1.0.0,<2.0.0']}

entry_points = \
{'sqlalchemy.dialects': ['awsathena.jdbc = '
                         'pyathenajdbc.sqlalchemy_athena:AthenaDialect']}

setup_kwargs = {
    'name': 'pyathenajdbc',
    'version': '2.1.1',
    'description': 'Amazon Athena JDBC driver wrapper for the Python DB API 2.0 (PEP 249)',
    'long_description': '.. image:: https://img.shields.io/pypi/pyversions/PyAthenaJDBC.svg\n    :target: https://pypi.org/project/PyAthenaJDBC/\n\n.. image:: https://github.com/laughingman7743/PyAthenaJDBC/workflows/test/badge.svg\n    :target: https://github.com/laughingman7743/PyAthenaJDBC/actions\n\n.. image:: https://codecov.io/gh/laughingman7743/PyAthenaJDBC/branch/master/graph/badge.svg\n    :target: https://codecov.io/gh/laughingman7743/PyAthenaJDBC\n\n.. image:: https://img.shields.io/pypi/l/PyAthenaJDBC.svg\n    :target: https://github.com/laughingman7743/PyAthenaJDBC/blob/master/LICENSE\n\n.. image:: https://pepy.tech/badge/pyathenajdbc/month\n    :target: https://pepy.tech/project/pyathenajdbc\n\n.. image:: https://img.shields.io/badge/code%20style-black-000000.svg\n    :target: https://github.com/psf/black\n\nPyAthenaJDBC\n============\n\nPyAthenaJDBC is an `Amazon Athena JDBC driver`_ wrapper for the Python `DB API 2.0 (PEP 249)`_.\n\n.. _`DB API 2.0 (PEP 249)`: https://www.python.org/dev/peps/pep-0249/\n.. _`Amazon Athena JDBC driver`: https://docs.aws.amazon.com/athena/latest/ug/connect-with-jdbc.html\n\nRequirements\n------------\n\n* Python\n\n  - CPython 2,7, 3.5, 3.6, 3.7\n\n* Java\n\n  - Java >= 8 (JDBC 4.2)\n\nJDBC driver compatibility\n-------------------------\n\n+---------------+---------------------+-------------------------------------------------------------------------------+\n| Version       | JDBC driver version | Vendor                                                                        |\n+===============+=====================+===============================================================================+\n| < 2.0.0       | == 1.1.0            | AWS (Early released JDBC driver. It is incompatible with Simba\'s JDBC driver) |\n+---------------+---------------------+-------------------------------------------------------------------------------+\n| >= 2.0.0      | >= 2.0.5            | Simba                                                                         |\n+---------------+---------------------+-------------------------------------------------------------------------------+\n\nInstallation\n------------\n\n.. code:: bash\n\n    $ pip install PyAthenaJDBC\n\nExtra packages:\n\n+---------------+------------------------------------------+-----------------+\n| Package       | Install command                          | Version         |\n+===============+==========================================+=================+\n| Pandas        | ``pip install PyAthenaJDBC[Pandas]``     | >=0.19.0        |\n+---------------+------------------------------------------+-----------------+\n| SQLAlchemy    | ``pip install PyAthenaJDBC[SQLAlchemy]`` | >=1.0.0, <2.0.0 |\n+---------------+------------------------------------------+-----------------+\n\nUsage\n-----\n\nBasic usage\n~~~~~~~~~~~\n\n.. code:: python\n\n    from pyathenajdbc import connect\n\n    conn = connect(S3OutputLocation=\'s3://YOUR_S3_BUCKET/path/to/\',\n                   AwsRegion=\'us-west-2\')\n    try:\n        with conn.cursor() as cursor:\n            cursor.execute("""\n            SELECT * FROM one_row\n            """)\n            print(cursor.description)\n            print(cursor.fetchall())\n    finally:\n        conn.close()\n\nCursor iteration\n~~~~~~~~~~~~~~~~\n\n.. code:: python\n\n    from pyathenajdbc import connect\n\n    conn = connect(S3OutputLocation=\'s3://YOUR_S3_BUCKET/path/to/\',\n                   AwsRegion=\'us-west-2\')\n    try:\n        with conn.cursor() as cursor:\n            cursor.execute("""\n            SELECT * FROM many_rows LIMIT 10\n            """)\n            for row in cursor:\n                print(row)\n    finally:\n        conn.close()\n\nQuery with parameter\n~~~~~~~~~~~~~~~~~~~~\n\nSupported `DB API paramstyle`_ is only ``PyFormat``.\n``PyFormat`` only supports `named placeholders`_ with old ``%`` operator style and parameters specify dictionary format.\n\n.. code:: python\n\n    from pyathenajdbc import connect\n\n    conn = connect(S3OutputLocation=\'s3://YOUR_S3_BUCKET/path/to/\',\n                   AwsRegion=\'us-west-2\')\n    try:\n        with conn.cursor() as cursor:\n            cursor.execute("""\n            SELECT col_string FROM one_row_complex\n            WHERE col_string = %(param)s\n            """, {\'param\': \'a string\'})\n            print(cursor.fetchall())\n    finally:\n        conn.close()\n\nif ``%`` character is contained in your query, it must be escaped with ``%%`` like the following:\n\n.. code:: sql\n\n    SELECT col_string FROM one_row_complex\n    WHERE col_string = %(param)s OR col_string LIKE \'a%%\'\n\n.. _`DB API paramstyle`: https://www.python.org/dev/peps/pep-0249/#paramstyle\n.. _`named placeholders`: https://pyformat.info/#named_placeholders\n\nJVM options\n~~~~~~~~~~~\n\nIn the connect method or connection object, you can specify JVM options with a string array.\n\nYou can increase the JVM heap size like the following:\n\n.. code:: python\n\n    from pyathenajdbc import connect\n\n    conn = connect(S3OutputLocation=\'s3://YOUR_S3_BUCKET/path/to/\',\n                   AwsRegion=\'us-west-2\',\n                   jvm_options=[\'-Xms1024m\', \'-Xmx4096m\'])\n    try:\n        with conn.cursor() as cursor:\n            cursor.execute("""\n            SELECT * FROM many_rows\n            """)\n            print(cursor.fetchall())\n    finally:\n        conn.close()\n\nJDBC 4.1\n~~~~~~~~\n\nIf you want to use JDBC 4.1, download the corresponding JDBC driver\nand specify the path of the downloaded JDBC driver as the argument ``driver_path`` of the connect method or connection object.\n\n* The `AthenaJDBC41-2.0.7.jar`_ is compatible with JDBC 4.1 and requires JDK 7.0 or later.\n\n.. _`AthenaJDBC41-2.0.7.jar`: https://s3.amazonaws.com/athena-downloads/drivers/JDBC/SimbaAthenaJDBC_2.0.7/AthenaJDBC41_2.0.7.jar\n\n.. code:: python\n\n    from pyathenajdbc import connect\n\n    conn = connect(S3OutputLocation=\'s3://YOUR_S3_BUCKET/path/to/\',\n                   AwsRegion=\'us-west-2\',\n                   driver_path=\'/path/to/AthenaJDBC41_2.0.7.jar\')\n\nJDBC driver configuration options\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n\nThe connect method or connection object pass keyword arguments as options to the JDBC driver.\nIf you want to change the behavior of the JDBC driver,\nspecify the option as a keyword argument in the connect method or connection object.\n\n.. code:: python\n\n    from pyathenajdbc import connect\n\n    conn = connect(S3OutputLocation=\'s3://YOUR_S3_BUCKET/path/to/\',\n                   AwsRegion=\'us-west-2\',\n                   LogPath=\'/path/to/pyathenajdbc/log/\',\n                   LogLevel=\'6\')\n\nFor details of the JDBC driver options refer to the official documentation.\n\n* `JDBC Driver Installation and Configuration Guide`_.\n\n.. _`JDBC Driver Installation and Configuration Guide`: https://s3.amazonaws.com/athena-downloads/drivers/JDBC/SimbaAthenaJDBC_2.0.7/docs/Simba+Athena+JDBC+Driver+Install+and+Configuration+Guide.pdf\n\nNOTE: Option names and values are case-sensitive. The option value is specified as a character string.\n\nSQLAlchemy\n~~~~~~~~~~\n\nInstall SQLAlchemy with ``pip install SQLAlchemy>=1.0.0`` or ``pip install PyAthenaJDBC[SQLAlchemy]``.\nSupported SQLAlchemy is 1.0.0 or higher and less than 2.0.0.\n\n.. code:: python\n\n    import contextlib\n    from urllib.parse import quote_plus  # PY2: from urllib import quote_plus\n    from sqlalchemy.engine import create_engine\n    from sqlalchemy.sql.expression import select\n    from sqlalchemy.sql.functions import func\n    from sqlalchemy.sql.schema import Table, MetaData\n\n    conn_str = \'awsathena+jdbc://{User}:{Password}@athena.{AwsRegion}.amazonaws.com:443/\'\\\n               \'{Schema}?S3OutputLocation={S3OutputLocation}\'\n    engine = create_engine(conn_str.format(\n        User=quote_plus(\'YOUR_ACCESS_KEY\'),\n        Password=quote_plus(\'YOUR_SECRET_ACCESS_KEY\'),\n        AwsRegion=\'us-west-2\',\n        Schema=\'default\',\n        S3OutputLocation=quote_plus(\'s3://YOUR_S3_BUCKET/path/to/\')))\n    try:\n        with contextlib.closing(engine.connect()) as conn:\n            many_rows = Table(\'many_rows\', MetaData(bind=engine), autoload=True)\n            print(select([func.count(\'*\')], from_obj=many_rows).scalar())\n    finally:\n        engine.dispose()\n\nThe connection string has the following format:\n\n.. code:: text\n\n    awsathena+jdbc://{User}:{Password}@athena.{AwsRegion}.amazonaws.com:443/{Schema}?S3OutputLocation={S3OutputLocation}&driver_path={driver_path}&...\n\nIf you do not specify ``User`` (i.e. AWSAccessKeyID) and ``Password`` (i.e. AWSSecretAccessKey) using instance profile credentials or credential profiles file:\n\n.. code:: text\n\n    awsathena+jdbc://:@athena.{Region}.amazonaws.com:443/{Schema}?S3OutputLocation={S3OutputLocation}&driver_path={driver_path}&...\n\nNOTE: ``S3OutputLocation`` requires quote. If ``User``, ``Password`` and other parameter contain special characters, quote is also required.\n\nPandas\n~~~~~~\n\nAs DataFrame\n^^^^^^^^^^^^\n\nYou can use the `pandas.read_sql`_ to handle the query results as a `DataFrame object`_.\n\n.. code:: python\n\n    from pyathenajdbc import connect\n    import pandas as pd\n\n    conn = connect(User=\'YOUR_ACCESS_KEY_ID\',\n                   Password=\'YOUR_SECRET_ACCESS_KEY\',\n                   S3OutputLocation=\'s3://YOUR_S3_BUCKET/path/to/\',\n                   AwsRegion=\'us-west-2\',\n                   jvm_path=\'/path/to/jvm\')\n    df = pd.read_sql("SELECT * FROM many_rows LIMIT 10", conn)\n\nThe ``pyathena.util`` package also has helper methods.\n\n.. code:: python\n\n    import contextlib\n    from pyathenajdbc import connect\n    from pyathenajdbc.util import as_pandas\n\n    with contextlib.closing(\n            connect(S3OutputLocation=\'s3://YOUR_S3_BUCKET/path/to/\'\n                    AwsRegion=\'us-west-2\'))) as conn:\n        with conn.cursor() as cursor:\n            cursor.execute("""\n            SELECT * FROM many_rows\n            """)\n            df = as_pandas(cursor)\n    print(df.describe())\n\n.. _`pandas.read_sql`: https://pandas.pydata.org/pandas-docs/stable/reference/api/pandas.read_sql.html\n.. _`DataFrame object`: https://pandas.pydata.org/pandas-docs/stable/reference/api/pandas.DataFrame.html\n\nTo SQL\n^^^^^^\n\nYou can use `pandas.DataFrame.to_sql`_ to write records stored in DataFrame to Amazon Athena.\n`pandas.DataFrame.to_sql`_ uses `SQLAlchemy`_, so you need to install it.\n\n.. code:: python\n\n    import pandas as pd\n    from urllib.parse import quote_plus\n    from sqlalchemy import create_engine\n    conn_str = \'awsathena+jdbc://:@athena.{AwsRegion}.amazonaws.com:443/\'\\\n               \'{Schema}?S3OutputLocation={S3OutputLocation}&S3Location={S3Location}&compression=snappy\'\n    engine = create_engine(conn_str.format(\n        AwsRegion=\'us-west-2\',\n        Schema_name=\'YOUR_SCHEMA\',\n        S3OutputLocation=quote_plus(\'s3://YOUR_S3_BUCKET/path/to/\'),\n        S3Location=quote_plus(\'s3://YOUR_S3_BUCKET/path/to/\')))\n    df = pd.DataFrame({\'a\': [1, 2, 3, 4, 5]})\n    df.to_sql(\'YOUR_TABLE\', engine, schema="YOUR_SCHEMA", index=False, if_exists=\'replace\', method=\'multi\')\n\nThe location of the Amazon S3 table is specified by the ``S3Location`` parameter in the connection string.\nIf ``S3Location`` is not specified, ``S3OutputLocation`` parameter will be used. The following rules apply.\n\n.. code:: text\n\n    s3://{S3Location or S3OutputLocation}/{schema}/{table}/\n\nThe data format only supports Parquet. The compression format is specified by the ``compression`` parameter in the connection string.\n\n.. _`pandas.DataFrame.to_sql`: https://pandas.pydata.org/pandas-docs/stable/reference/api/pandas.DataFrame.to_sql.html\n\nCredential\n----------\n\nAWS credentials provider chain\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n\nSee `AWS credentials provider chain`_\n\n    AWS credentials provider chain that looks for credentials in this order:\n\n        * Environment Variables - AWS_ACCESS_KEY_ID and AWS_SECRET_ACCESS_KEY (RECOMMENDED since they are recognized by all the AWS SDKs and CLI except for .NET), or AWS_ACCESS_KEY and AWS_SECRET_KEY (only recognized by Java SDK)\n        * Java System Properties - aws.accessKeyId and aws.secretKey\n        * Credential profiles file at the default location (~/.aws/credentials) shared by all AWS SDKs and the AWS CLI\n        * Credentials delivered through the Amazon EC2 container service if AWS_CONTAINER_CREDENTIALS_RELATIVE_URI" environment variable is set and security manager has permission to access the variable,\n        * Instance profile credentials delivered through the Amazon EC2 metadata service\n\nIn the connect method or connection object, you can connect by specifying at least ``S3OutputLocation`` and ``AwsRegion``.\nIt is not necessary to specify ``User`` and ``Password``.\n\n.. code:: python\n\n    from pyathenajdbc import connect\n\n    conn = connect(S3OutputLocation=\'s3://YOUR_S3_BUCKET/path/to/\',\n                   AwsRegion=\'us-west-2\')\n\nTesting\n-------\n\nDepends on the following environment variables:\n\n.. code:: bash\n\n    $ export AWS_ACCESS_KEY_ID=YOUR_ACCESS_KEY_ID\n    $ export AWS_SECRET_ACCESS_KEY=YOUR_SECRET_ACCESS_KEY\n    $ export AWS_DEFAULT_REGION=us-west-2\n    $ export AWS_ATHENA_S3_STAGING_DIR=s3://YOUR_S3_BUCKET/path/to/\n\nAnd you need to create a workgroup named ``test-pyathena-jdbc``.\n\nRun test\n~~~~~~~~\n\n.. code:: bash\n\n    $ pip install poetry\n    $ poetry install -v\n    $ poetry run scripts/test_data/upload_test_data.sh\n    $ poetry run pytest\n    $ poetry run scripts/test_data/delete_test_data.sh\n\nRun test multiple Python versions\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n\n.. code:: bash\n\n    $ pip install poetry\n    $ poetry install -v\n    $ poetry run scripts/test_data/upload_test_data.sh\n    $ pyenv local 3.7.2 3.6.8 3.5.7 2.7.16\n    $ poetry run tox\n    $ poetry run scripts/test_data/delete_test_data.sh\n\nCode formatting\n---------------\n\nThe code formatting uses `black`_ and `isort`_.\n\nAppy format\n~~~~~~~~~~~\n\n.. code:: bash\n\n    $ make fmt\n\nCheck format\n~~~~~~~~~~~~\n\n.. code:: bash\n\n    $ make chk\n\n.. _`black`: https://github.com/psf/black\n.. _`isort`: https://github.com/timothycrosley/isort\n\nLicense\n-------\n\nThe license of all Python code except JDBC driver is `MIT license`_.\n\n.. _`MIT license`: LICENSE\n\nJDBC driver\n~~~~~~~~~~~\n\nFor the license of JDBC driver, please check the following link.\n\n* `JDBC driver release notes`_\n* `JDBC driver license`_\n* `JDBC driver notices`_\n* `JDBC driver third-party licenses`_\n\n.. _`JDBC driver release notes`: https://s3.amazonaws.com/athena-downloads/drivers/JDBC/SimbaAthenaJDBC_2.0.7/docs/release-notes.txt\n.. _`JDBC driver License`: https://s3.amazonaws.com/athena-downloads/drivers/JDBC/SimbaAthenaJDBC_2.0.7/docs/LICENSE.txt\n.. _`JDBC driver notices`: https://s3.amazonaws.com/athena-downloads/drivers/JDBC/SimbaAthenaJDBC_2.0.7/docs/NOTICES.txt\n.. _`JDBC driver third-party licenses`: https://s3.amazonaws.com/athena-downloads/drivers/JDBC/SimbaAthenaJDBC_2.0.7/docs/third-party-licenses.txt\n',
    'author': 'laughingman7743',
    'author_email': 'laughingman7743@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/laughingman7743/PyAthenaJDBC/',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'entry_points': entry_points,
    'python_requires': '>=2.7, !=3.0.*, !=3.1.*, !=3.2.*, !=3.3.*, !=3.4.*',
}


setup(**setup_kwargs)
