from .web import download
from .io import load

import xarray
from pathlib import Path
from argparse import ArgumentParser


def dascasi_download():
    """
    Download DASC ASI data for various sites.

    site VEE for one minute:

        python DownloadDASC.py VEE 2017-02-13T06:59 2017-02-13T07:00 ~/data
    """
    p = ArgumentParser(description="download DASC all-sky camera data")
    p.add_argument("site", choices=["EAA", "FYU", "KAK", "PKR", "TOO", "VEE"])
    p.add_argument("startend", help="start/end times UTC e.g. 2012-11-03T06:23 2012-11-03T07", nargs=2)
    p.add_argument("odir", help="directory to write downloaded FITS to")
    p.add_argument("-w", "--wavelen", help="request specific wavelength(s)", nargs="+")
    p.add_argument("-host", default="ftp://optics.gi.alaska.edu")
    p = p.parse_args()

    # host = "ftp://mirrors.arsc.edu/AMISR/PKR/DASC/RAW/"
    download(p.startend, p.site, p.odir, p.host, p.wavelen)


def _plotdasc(img: xarray.Dataset, outdir: Path, cadence: float):
    from .plots import histogram_dasc, moviedasc

    histogram_dasc(img, outdir)

    moviedasc(img, outdir, cadence)


def dascasi_movie():
    from matplotlib.pyplot import show

    p = ArgumentParser(description="for DASC all sky camera, read and play az/el mapping and images")
    p.add_argument("indir", help="directory of .fits or specific .fits file")
    p.add_argument("-o", "--outdir", help="directory to write plots to")
    p.add_argument("-t", "--tlim", help="only plot data in this range", nargs=2)
    p.add_argument("-a", "--azelfn", help="stem for DASC .fits azimuth calibration")
    p.add_argument("-w", "--wavelength", help="select wavelength(s) to plot simultaneously [428 558 630]", nargs="+")
    p.add_argument("-c", "--cadence", help="set playback cadence to request times [sec]", type=float, default=5.0)
    p = p.parse_args()

    imgs = load(p.indir, p.azelfn, treq=p.tlim, wavelenreq=p.wavelength)

    _plotdasc(imgs, p.outdir, p.cadence)

    show()
