"""
proxy for database connection

"""
import re
import os
import sys
import json
import time
import math
import copy
import glob
import uuid
import random
import urllib
import socket
import logging
import datetime
import traceback
import warnings
import operator
import atexit
from pandaserver.taskbuffer import ErrorCode
from pandaserver.taskbuffer import SiteSpec
from pandaserver.taskbuffer import CloudSpec
from pandaserver.taskbuffer import PrioUtil
from pandaserver.taskbuffer import ProcessGroups
from pandaserver.taskbuffer import JobUtils
from pandaserver.taskbuffer import EventServiceUtils
from pandaserver.taskbuffer import GlobalShares
from pandaserver.taskbuffer.DdmSpec  import DdmSpec
from pandaserver.taskbuffer.JobSpec  import JobSpec, push_status_changes
from pandaserver.taskbuffer.FileSpec import FileSpec
from pandaserver.taskbuffer.WorkerSpec import WorkerSpec
from pandaserver.taskbuffer.DatasetSpec import DatasetSpec
from pandaserver.taskbuffer.ResourceSpec import ResourceSpec
from pandaserver.taskbuffer.CloudTaskSpec import CloudTaskSpec
from pandaserver.taskbuffer.HarvesterMetricsSpec import HarvesterMetricsSpec
from pandaserver.taskbuffer.WrappedCursor import WrappedCursor
from pandaserver.taskbuffer.Utils import create_shards
from pandacommon.pandalogger.PandaLogger import PandaLogger
from pandacommon.pandalogger.LogWrapper import LogWrapper
from pandaserver.config import panda_config
from pandaserver.taskbuffer.SupErrors import SupErrors
from pandaserver.srvcore import CoreUtils, srv_msg_utils

try:
    from idds.client.client import Client as iDDS_Client
    import idds.common.constants
    import idds.common.utils
except ImportError:
    pass

try:
    long
except NameError:
    long = int
try:
    unicode
except NameError:
    unicode = str

if panda_config.backend == 'oracle':
    import cx_Oracle
    varNUMBER = cx_Oracle.NUMBER
elif panda_config.backend == 'postgres':
    import psycopg2 as psycopg
    from . import WrappedPostgresConn
    varNUMBER = long
else:
    import MySQLdb
    varNUMBER = long

warnings.filterwarnings('ignore')

# logger
_logger = PandaLogger().getLogger('DBProxy')
_loggerFiltered = PandaLogger().getLogger('DBProxyFiltered')
# add handlers
for hdr in _loggerFiltered.handlers:
    hdr.setLevel(logging.INFO)
    _logger.addHandler(hdr)
    _loggerFiltered.removeHandler(hdr)

# get mb proxies used in DBProxy methods
def get_mb_proxy_dict():
    if hasattr(panda_config, 'mbproxy_configFile') \
            and panda_config.mbproxy_configFile:
        # delay import to open logger file inside python daemon
        from pandaserver.taskbuffer.PanDAMsgProcessor import MsgProcAgent
        out_q_list = ['panda_jobstatus', 'panda_jedi', 'panda_pilot_topic', 'panda_pilot_queue']
        mp_agent = MsgProcAgent(config_file=panda_config.mbproxy_configFile)
        mb_proxy_dict = mp_agent.start_passive_mode(in_q_list=[], out_q_list=out_q_list)
        return mb_proxy_dict
    return {}


# proxy
class DBProxy:

    # constructor
    def __init__(self,useOtherError=False):

        # connection object
        self.conn = None
        # cursor object
        self.cur = None
        # host name
        self.hostname = None
        # retry count
        self.nTry = 5
        # use special error codes for reconnection in querySQL
        self.useOtherError = useOtherError
        # pledge resource ratio
        self.beyondPledgeRatio = {}
        # update time for pledge resource ratio
        self.updateTimeForPledgeRatio = None
        # fairshare policy
        self.fairsharePolicy = {}
        # update time for fairshare policy
        self.updateTimeForFairsharePolicy = None
        # hostname
        self.myHostName = socket.getfqdn()
        self.backend = panda_config.backend

        # global share variables
        self.tree = None  # Pointer to the root of the global shares tree
        self.leave_shares = None  # Pointer to the list with leave shares
        self.__t_update_shares = None  # Timestamp when the shares were last updated
        self.__hs_distribution = None  # HS06s distribution of sites
        self.__t_update_distribution = None  # Timestamp when the HS06s distribution was last updated

        # priority boost
        self.job_prio_boost_dict = None
        self.job_prio_boost_dict_update_time = None

        # keep type
        self.__orig_type = type

        # mb proxy
        self.mb_proxy_dict = None

        # self.__reload_shares()
        # self.__reload_hs_distribution()

    # connect to DB
    def connect(self,dbhost=panda_config.dbhost,dbpasswd=panda_config.dbpasswd,
                dbuser=panda_config.dbuser,dbname=panda_config.dbname,
                dbtimeout=panda_config.dbtimeout,reconnect=False,dbport=panda_config.dbport):
        _logger.debug("connect : re=%s" % reconnect)
        # keep parameters for reconnect
        if not reconnect:
            self.dbhost    = dbhost
            self.dbpasswd  = dbpasswd
            self.dbuser    = dbuser
            self.dbname    = dbname
            self.dbtimeout = dbtimeout
            self.dbport    = dbport
        # close old connection
        if reconnect:
            _logger.debug("closing old connection")
            try:
                self.conn.close()
            except Exception:
                _logger.debug("failed to close old connection")
        # connect
        try:

            if self.backend == 'oracle':
                self.conn = cx_Oracle.connect(dsn=self.dbhost, user=self.dbuser,
                                              password=self.dbpasswd, threaded=True,
                                              encoding='UTF-8')
                def OutputTypeHandler(cursor, name, defaultType, size, precision, scale):
                    if defaultType == cx_Oracle.CLOB:
                        return cursor.var(cx_Oracle.LONG_STRING, arraysize = cursor.arraysize)
                self.conn.outputtypehandler = OutputTypeHandler
            elif self.backend == 'postgres':
                dsn = {'dbname': self.dbname, 'user': self.dbuser}
                if self.dbpasswd:
                    dsn['password'] = self.dbpasswd
                if self.dbhost:
                    dsn['host'] = self.dbhost
                if self.dbport:
                    dsn['port'] = self.dbport
                conn = psycopg.connect(**dsn)
                self.conn = WrappedPostgresConn.WrappedPostgresConn(conn)
            else:
                self.conn = MySQLdb.connect(host=self.dbhost, db=self.dbname,
                                            port=self.dbport, connect_timeout=self.dbtimeout,
                                            user=self.dbuser, passwd=self.dbpasswd,
                                            charset='utf8')
            self.cur = WrappedCursor(self.conn)
            try:
                # use SQL dumper
                if panda_config.dump_sql:
                    import SQLDumper
                    self.cur = SQLDumper.SQLDumper(self.cur)
            except Exception:
                pass
            self.hostname = self.cur.initialize()
            if not reconnect:
                atexit.register(self.close_connection)
            _logger.debug("connect : re=%s ready" % reconnect)
            return True
        except Exception as e:
            _logger.error("connect : %s" % str(e))
            return False

    # close connection
    def close_connection(self):
        if self.conn:
            try:
                self.conn.close()
            except Exception:
                pass
        return

    # cleanup
    def cleanup(self):
        _logger.debug("cleanup start")
        self.close_connection()
        atexit.unregister(self.close_connection)
        _logger.debug("cleanup done")

    def getvalue_corrector(self, value):
        """
        Needed to support old and new versions of cx_Oracle
        :return:
        """
        if isinstance(value, list): # cx_Oracle version >= 6.3
            return value[0]
        else: # cx_Oracle version < 6.3
            return value

    # Internal caching of a result. Use only for information with low update frequency and low memory footprint
    def memoize(f):
        memo = {}
        kwd_mark = object()
        def helper(self, *args, **kwargs):
            now = datetime.datetime.now()
            key = args + (kwd_mark,) + tuple(sorted(kwargs.items()))
            if key not in memo or memo[key]['timestamp'] < now - datetime.timedelta(hours=1):
                memo[key] = {}
                memo[key]['value'] = f(self, *args, **kwargs)
                memo[key]['timestamp'] = now
            return memo[key]['value']
        return helper

    # query an SQL
    def querySQL(self,sql,arraySize=1000):
        comment = ' /* DBProxy.querySQL */'
        try:
            _logger.debug("querySQL : %s " % sql)
            # begin transaction
            self.conn.begin()
            self.cur.arraysize = arraySize
            self.cur.execute(sql+comment)
            res = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            return res
        except Exception:
            # roll back
            self._rollback(self.useOtherError)
            type, value, traceBack = sys.exc_info()
            _logger.error("querySQL : %s " % sql)
            _logger.error("querySQL : %s %s" % (type,value))
            return None


    # query an SQL return Status
    def querySQLS(self,sql,varMap,arraySize=1000):
        comment = ' /* DBProxy.querySQLS */'
        try:
            # begin transaction
            self.conn.begin()
            self.cur.arraysize = arraySize
            ret = self.cur.execute(sql+comment,varMap)
            if ret:
                ret = True
            if sql.startswith('INSERT') or sql.startswith('UPDATE') or \
                   sql.startswith('DELETE'):
                res = self.cur.rowcount
            else:
                res = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            return ret,res
        except Exception as e:
            # roll back
            self._rollback(self.useOtherError)
            _logger.error("querySQLS : %s %s" % (sql,str(varMap)))
            _logger.error("querySQLS : %s" % str(e))
            return -1,None


    # execute an SQL return with executemany
    def executemanySQL(self, sql, varMaps, arraySize=1000):
        comment = ' /* DBProxy.executemanySQL */'
        try:
            # begin transaction
            self.conn.begin()
            self.cur.arraysize = arraySize
            ret = self.cur.executemany(sql+comment, varMaps)
            if ret:
                ret = True
            if sql.startswith('INSERT') or sql.startswith('UPDATE') or \
                   sql.startswith('DELETE'):
                res = self.cur.rowcount
            else:
                raise RuntimeError('Operation unsupported. Only INSERT, UPDATE, DELETE are allowed')
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            return res
        except Exception as e:
            # roll back
            self._rollback(self.useOtherError)
            _logger.error("executemanySQL : %s %s" % (sql, str(varMaps)))
            _logger.error("executemanySQL : %s" % str(e))
            return None


    # get CLOB
    def getClobObj(self, sql, varMap, arraySize=10000, use_commit=True):
        comment = ' /* DBProxy.getClobObj */'
        try:
            # begin transaction
            if use_commit:
                self.conn.begin()
                self.cur.arraysize = arraySize
            ret = self.cur.execute(sql+comment,varMap)
            if ret:
                ret = True
            res = []
            for items in self.cur:
                resItem = []
                for item in items:
                    # read CLOB
                    try:
                        itemRead = item.read()
                    except AttributeError:
                        itemRead = item
                    resItem.append(itemRead)
                # append
                res.append(resItem)
            # commit
            if use_commit:
                if not self._commit():
                    raise RuntimeError('Commit error')
            return ret, res
        except Exception as e:
            # roll back
            if use_commit:
                self._rollback()
            type, value, traceBack = sys.exc_info()
            _logger.error("getClobObj : %s %s" % (sql,str(varMap)))
            _logger.error("getClobObj : {}".format(str(e)))
            return -1, None


    # get configuration value. cached for an hour
    @memoize
    def getConfigValue(self, component, key, app='pandaserver', vo=None):
        comment = ' /* DBProxy.getConfigValue */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        varMap = {':component': component, ':key': key, ':app': app}
        sql = """
        SELECT value, type FROM ATLAS_PANDA.CONFIG
        WHERE component=:component
        AND key=:key
        AND app=:app
        """

        # If VO is specified, select only the config values for this VO or VO independent values
        if vo:
            varMap[':vo'] = vo
            sql += "AND (vo=:vo or vo IS NULL)"

        self.cur.execute(sql+comment, varMap)

        try:
            value_str, type = self.cur.fetchone()
        except TypeError:
            error_message = 'Specified key={0} not found for component={1} app={2}'.format(key,component,app)
            _logger.debug(error_message)
            return None

        try:

            if type.lower() in ('str', 'string'):
                return value_str
            elif type.lower() in ('int', 'integer'):
                return int(value_str)
            elif type.lower() == 'float':
                return float(value_str)
            elif type.lower() in ('bool', 'boolean'):
                if value_str.lower() == 'true':
                    return True
                else:
                    return False
            else:
                raise ValueError
        except ValueError:
            error_message = 'Wrong value/type pair. Value: {0}, Type: {1}'.format(value_str, type)
            _logger.debug(error_message)
            return None



    # insert job to jobsDefined
    def insertNewJob(self, job, user, serNum, weight=0.0, priorityOffset=0, userVO=None, groupJobSN=0, toPending=False,
                     origEsJob=False, eventServiceInfo=None, oldPandaIDs=None, relationType=None, fileIDPool=[],
                     origSpecialHandling=None, unprocessedMap=None, prio_reduction=True):
        comment = ' /* DBProxy.insertNewJob */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += ' <JediTaskID={0} idPool={1}>'.format(job.jediTaskID, len(fileIDPool))
        tmp_log = LogWrapper(_logger, methodName)

        if not toPending:
            sql1 = "INSERT INTO ATLAS_PANDA.jobsDefined4 (%s) " % JobSpec.columnNames()
        else:
            sql1 = "INSERT INTO ATLAS_PANDA.jobsWaiting4 (%s) " % JobSpec.columnNames()
        sql1 += JobSpec.bindValuesExpression(useSeq=True)
        sql1 += " RETURNING PandaID INTO :newPandaID"

        # make sure PandaID is NULL
        job.PandaID = None
        # job status
        if not toPending:
            job.jobStatus = 'defined'
        else:
            if job.computingSite == EventServiceUtils.siteIdForWaitingCoJumboJobs:
                # put waiting co-jumbo jobs to waiting
                job.jobStatus = 'waiting'
            else:
                job.jobStatus = 'pending'
        
        # host and time information
        job.modificationHost = self.hostname
        job.creationTime = datetime.datetime.utcnow()
        job.modificationTime = job.creationTime
        job.stateChangeTime = job.creationTime
        job.prodDBUpdateTime = job.creationTime
        # DN
        if job.prodUserID == "NULL" or job.prodSourceLabel in ['user', 'panda']:
            job.prodUserID = user
        
        # compact user name
        job.prodUserName = self.cleanUserID(job.prodUserID)
        if job.prodUserName in ['', 'NULL']:
            # use prodUserID as compact user name
            job.prodUserName = job.prodUserID
        
        # VO
        job.VO = userVO
        
        # priority
        if job.assignedPriority != 'NULL':
            job.currentPriority = job.assignedPriority
        if job.prodSourceLabel == 'install':
            job.currentPriority = 4100
        elif job.prodUserName in ['artprod'] and job.prodSourceLabel in ['user', 'panda']:
            job.currentPriority = 7000
        elif job.prodSourceLabel == 'user':
            if job.processingType in ['usermerge', 'pmerge'] and job.currentPriority not in ['NULL', None]:
                # avoid prio reduction for merge jobs
                pass
            else:
                if not prio_reduction:
                    job.currentPriority = priorityOffset
                    if job.isScoutJob():
                        job.currentPriority += 1
                elif job.currentPriority not in ['NULL', None] and \
                     (job.isScoutJob() or job.currentPriority >= JobUtils.priorityTasksToJumpOver):
                    pass
                else:
                    job.currentPriority = PrioUtil.calculatePriority(priorityOffset,serNum,weight)
                    if 'express' in job.specialHandling:
                        job.currentPriority = 6000
        elif job.prodSourceLabel == 'panda':
            job.currentPriority = 2000 + priorityOffset
            if 'express' in job.specialHandling:
                job.currentPriority = 6500

        # usergroup
        if job.prodSourceLabel == 'regional':
            job.computingSite = "BNLPROD"

        # group job SN
        groupJobSN = "%05d" % groupJobSN
        # set attempt numbers
        if job.prodSourceLabel in ['user', 'panda'] + JobUtils.list_ptest_prod_sources:
            if job.attemptNr in [None, 'NULL', '']:
                job.attemptNr = 0
            if job.maxAttempt in [None, 'NULL', '']:
                job.maxAttempt = 0
            # set maxAttempt to attemptNr to disable server/pilot retry
            if job.maxAttempt == -1:
                job.maxAttempt = job.attemptNr
            else:
                # set maxAttempt to have server/pilot retries for retried jobs
                if job.maxAttempt <= job.attemptNr:
                    job.maxAttempt = job.attemptNr + 2

        # obtain the share and resource type
        if job.gshare in ('NULL', None, ''):
            job.gshare = self.get_share_for_job(job)
        tmp_log.debug('resource_type is set to {0}'.format(job.resource_type))
        tmp_log.debug("jediTaskID={0} SH={1} origEsJob={2} eInfo={3}".format(job.jediTaskID, origSpecialHandling,
                                                                                            origEsJob, eventServiceInfo))
        if job.resource_type in ('NULL', None, ''):
            try:
                job.resource_type = self.get_resource_type_job(job)
                tmp_log.debug('reset resource_type to {0}'.format(job.resource_type))
            except Exception:
                job.resource_type = 'Undefined'
                tmp_log.error('reset resource_type excepted with: {0}'.format(traceback.format_exc()))

        try:
            # use JEDI
            if hasattr(panda_config, 'useJEDI') and panda_config.useJEDI is True and \
                    job.lockedby == 'jedi':
                useJEDI = True
            else:
                useJEDI = False
            
            # begin transaction
            self.conn.begin()
            
            # get jobsetID for event service
            if origEsJob:
                if self.backend == 'oracle':
                    sqlESS = "SELECT ATLAS_PANDA.JOBSDEFINED4_PANDAID_SEQ.nextval FROM dual "
                    self.cur.arraysize = 10
                    self.cur.execute(sqlESS+comment, {})
                    job.jobsetID, = self.cur.fetchone()
                else:
                    #panda_config.backend == 'mysql':
                    ### fake sequence
                    sql = " INSERT INTO ATLAS_PANDA.JOBSDEFINED4_PANDAID_SEQ (col) VALUES (NULL) "
                    self.cur.arraysize = 10
                    self.cur.execute(sql + comment, {})
                    sql2 = """ SELECT LAST_INSERT_ID() """
                    self.cur.execute(sql2 + comment, {})
                    job.jobsetID, = self.cur.fetchone()

            # check input
            if useJEDI:
                allInputOK = True
                if eventServiceInfo is None or eventServiceInfo == {} or origEsJob:
                    sqlCheckJediFile  = "SELECT status,keepTrack,attemptNr,type "
                    sqlCheckJediFile += "FROM ATLAS_PANDA.JEDI_Dataset_Contents "
                    sqlCheckJediFile += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                    sqlCheckJediFile += "FOR UPDATE "
                    for file in job.Files:
                        # skip if no JEDI
                        if file.fileID == 'NULL':
                            continue
                        # only input
                        if file.type not in ['input','pseudo_input']:
                            continue
                        varMap = {}
                        varMap[':fileID'] = file.fileID
                        varMap[':datasetID'] = file.datasetID
                        varMap[':jediTaskID'] = file.jediTaskID
                        self.cur.execute(sqlCheckJediFile+comment, varMap)
                        retFC = self.cur.fetchone()
                        if retFC is None:
                            allInputOK = False
                            tmp_log.debug("input check failed - missing jediTaskID:%s datasetID=%s fileID=%s" % (file.jediTaskID, file.datasetID, file.fileID))
                            break
                        tmpStatus,tmpKeepTrack,tmpAttemptNr,tmpType = retFC
                        # only keep track
                        if tmpKeepTrack != 1:
                            continue
                        # ignore lib
                        if tmpType in ['lib']:
                            continue
                        # check attemptNr
                        if tmpAttemptNr != file.attemptNr:
                            allInputOK = False
                            tmp_log.debug("input check failed - bad attemptNr %s:%s jediTaskID:%s datasetID=%s fileID=%s" % (tmpAttemptNr, file.attemptNr,
                                                                                                                             file.jediTaskID, file.datasetID, file.fileID))
                            break
                        # check status
                        if tmpStatus != 'picked':
                            allInputOK = False
                            tmp_log.debug("input check failed - bad status %s jediTaskID:%s datasetID=%s fileID=%s" % (tmpStatus, file.jediTaskID,
                                                                                                                       file.datasetID, file.fileID))
                            break
                if not allInputOK:
                    # commit
                    if not self._commit():
                        raise RuntimeError('Commit error')
                    if unprocessedMap is not None:
                        return (False, unprocessedMap)
                    return False
            
            # insert job
            varMap = job.valuesMap(useSeq=True)
            varMap[':newPandaID'] = self.cur.var(varNUMBER)
            retI = self.cur.execute(sql1+comment, varMap)
            
            # set PandaID
            val = self.getvalue_corrector(self.cur.getvalue(varMap[':newPandaID']))
            job.PandaID = long(val)
            
            # get jobsetID
            if job.jobsetID in [None, 'NULL', -1]:
                jobsetID = 0
            else:
                jobsetID = job.jobsetID
            jobsetID = '%06d' % jobsetID
            try:
                strJediTaskID = str(job.jediTaskID)
            except Exception:
                strJediTaskID = ''
            
            # get originPandaID
            originPandaID = None
            if oldPandaIDs is not None and len(oldPandaIDs) > 0:
                varMap = {}
                varMap[':jediTaskID'] = job.jediTaskID
                varMap[':pandaID'] = oldPandaIDs[0]
                sqlOrigin  = "SELECT originPandaID FROM {0}.JEDI_Job_Retry_History ".format(panda_config.schemaJEDI)
                sqlOrigin += "WHERE jediTaskID=:jediTaskID AND newPandaID=:pandaID "
                sqlOrigin += "AND (relationType IS NULL OR NOT relationType IN ("
                for tmpType in EventServiceUtils.relationTypesForJS:
                    tmpKey = ':{0}'.format(tmpType)
                    sqlOrigin += '{0},'.format(tmpKey)
                    varMap[tmpKey] = tmpType
                sqlOrigin = sqlOrigin[:-1]
                sqlOrigin  += ')) '
                self.cur.execute(sqlOrigin+comment,varMap)
                resOrigin = self.cur.fetchone()
                if resOrigin is not None:
                    originPandaID, = resOrigin
                else:
                    originPandaID = oldPandaIDs[0]
            if originPandaID is None:
                originPandaID = job.PandaID
            newJobName = re.sub('\$ORIGINPANDAID',str(originPandaID),job.jobName)
            # update jobName
            if newJobName != job.jobName:
                job.jobName = newJobName
                if not toPending:
                    sqlJobName  = "UPDATE ATLAS_PANDA.jobsDefined4 "
                else:
                    sqlJobName  = "UPDATE ATLAS_PANDA.jobsWaiting4 "
                sqlJobName += "SET jobName=:jobName WHERE PandaID=:pandaID "
                varMap ={}
                varMap[':jobName'] = job.jobName
                varMap[':pandaID'] = job.PandaID
                self.cur.execute(sqlJobName+comment,varMap)
            # reset changed attribute list
            job.resetChangedList()
            # insert files
            tmp_log.debug("inserted %s label:%s prio:%s jediTaskID:%s" % (job.PandaID,job.prodSourceLabel,
                                                                                         job.currentPriority,
                                                                                         job.jediTaskID))
            # sql with SEQ
            sqlFile = "INSERT INTO ATLAS_PANDA.filesTable4 (%s) " % FileSpec.columnNames()
            sqlFile += FileSpec.bindValuesExpression(useSeq=True)
            sqlFile += " RETURNING row_ID INTO :newRowID"
            # sql without SEQ
            sqlFileW = "INSERT INTO ATLAS_PANDA.filesTable4 (%s) " % FileSpec.columnNames()
            sqlFileW += FileSpec.bindValuesExpression(useSeq=False)
            dynNumEvents = EventServiceUtils.isDynNumEventsSH(job.specialHandling)
            dynFileMap = {}
            dynLfnIdMap = {}
            totalInputEvents = 0
            indexFileID = 0
            varMapsForFile = []
            nFilesWaitingMap = {}
            nEventsToProcess = 0

            # failed related ES jobs
            if origEsJob and eventServiceInfo is not None and not job.notDiscardEvents():
                self.updateRelatedEventServiceJobs(job, killEvents=False, forceFailed=True)
            for file in job.Files:
                file.row_ID = None
                if file.status not in ['ready','cached']:
                    file.status='unknown'
                # replace $PANDAID with real PandaID
                file.lfn = re.sub('\$PANDAID', '%05d' % job.PandaID, file.lfn)
                # replace $JOBSETID with real jobsetID
                if job.prodSourceLabel not in ['managed']:
                    file.lfn = re.sub('\$JOBSETID', jobsetID, file.lfn)
                    file.lfn = re.sub('\$GROUPJOBSN', groupJobSN, file.lfn)
                    try:
                        file.lfn = re.sub('\$JEDITASKID', strJediTaskID, file.lfn)
                    except Exception:
                        pass
                # avoid duplicated files for dynamic number of events
                toSkipInsert = False
                if dynNumEvents and file.type in ['input','pseudo_input']:
                    if file.lfn not in dynFileMap:
                        dynFileMap[file.lfn] = set()
                    else:
                        toSkipInsert = True
                        dynFileMap[file.lfn].add((file.jediTaskID,file.datasetID,file.fileID,file.attemptNr))
                # set scope
                if file.type in ['output','log'] and job.VO in ['atlas']:
                    file.scope = self.extractScope(file.dataset)
                # insert
                if not toSkipInsert:
                    if indexFileID < len(fileIDPool):
                        file.row_ID = fileIDPool[indexFileID]
                        varMap = file.valuesMap(useSeq=False)
                        varMapsForFile.append(varMap)
                        indexFileID += 1
                    else:
                        varMap = file.valuesMap(useSeq=True)
                        varMap[':newRowID'] = self.cur.var(varNUMBER)
                        self.cur.execute(sqlFile+comment, varMap)
                        # get rowID
                        val = self.getvalue_corrector(self.cur.getvalue(varMap[':newRowID']))
                        file.row_ID = long(val)
                    dynLfnIdMap[file.lfn] = file.row_ID
                    # reset changed attribute list
                    file.resetChangedList()
                # update JEDI table
                if useJEDI:
                    # skip if no JEDI
                    if file.fileID == 'NULL':
                        continue
                    # input for waiting co-jumbo jobs
                    isWaiting = None
                    isFileForWaitingCoJumbo = False
                    if file.type not in ['output','log']:
                        if job.computingSite == EventServiceUtils.siteIdForWaitingCoJumboJobs:
                            isFileForWaitingCoJumbo = True
                        # check is_waiting
                        sqlJediFileIsW = "SELECT is_waiting FROM ATLAS_PANDA.JEDI_Dataset_Contents "
                        sqlJediFileIsW += " WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                        varMap = {}
                        varMap[':fileID'] = file.fileID
                        varMap[':jediTaskID'] = file.jediTaskID
                        varMap[':datasetID']  = file.datasetID
                        self.cur.execute(sqlJediFileIsW+comment, varMap)
                        resJediFileIsW = self.cur.fetchone()
                        if resJediFileIsW is not None:
                            isWaiting, = resJediFileIsW
                    # update Dataset_Contents table
                    varMap = {}
                    varMap[':fileID'] = file.fileID
                    if isFileForWaitingCoJumbo:
                        # not change status for wating co-jumbo jobs to allow new jobs to pickup files
                        varMap[':status'] = 'picked'
                        varMap[':is_waiting'] = 'Y'
                    else:
                        varMap[':status'] = 'running'
                    varMap[':oldStatusI'] = 'picked'
                    varMap[':oldStatusO'] = 'defined'
                    varMap[':attemptNr']  = file.attemptNr
                    varMap[':datasetID']  = file.datasetID
                    varMap[':keepTrack']  = 1
                    varMap[':jediTaskID'] = file.jediTaskID
                    if isFileForWaitingCoJumbo:
                        varMap[':PandaID'] = job.jobsetID
                    else:
                        varMap[':PandaID'] = file.PandaID
                    varMap[':jobsetID'] = job.jobsetID
                    sqlJediFile  = "UPDATE /*+ INDEX_RS_ASC(JEDI_DATASET_CONTENTS (JEDI_DATASET_CONTENTS.JEDITASKID JEDI_DATASET_CONTENTS.DATASETID JEDI_DATASET_CONTENTS.FILEID)) */ ATLAS_PANDA.JEDI_Dataset_Contents SET status=:status,PandaID=:PandaID,jobsetID=:jobsetID"
                    if file.type in ['output','log']:
                        sqlJediFile += ",outPandaID=:PandaID"
                    if isFileForWaitingCoJumbo:
                        sqlJediFile += ",is_waiting=:is_waiting"
                    else:
                        sqlJediFile += ",is_waiting=NULL"
                    sqlJediFile += " WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                    sqlJediFile += "AND attemptNr=:attemptNr AND status IN (:oldStatusI,:oldStatusO) AND keepTrack=:keepTrack "
                    self.cur.execute(sqlJediFile+comment, varMap)
                    # get number of inputs for waiting co-jumbo jobs
                    if (isFileForWaitingCoJumbo or isWaiting is not None) and self.cur.rowcount > 0:
                        if file.datasetID not in nFilesWaitingMap:
                            nFilesWaitingMap[file.datasetID] = 0
                        if isFileForWaitingCoJumbo and isWaiting is None:
                            nFilesWaitingMap[file.datasetID] += 1
                        elif not isFileForWaitingCoJumbo and isWaiting is not None:
                            nFilesWaitingMap[file.datasetID] -= 1
                    # no insert for dynamic number of events
                    if toSkipInsert:
                        continue
                    # insert events for ES
                    if origEsJob and eventServiceInfo is not None and file.lfn in eventServiceInfo:
                        # discard old successful event ranges
                        sqlJediOdEvt  = "UPDATE /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
                        sqlJediOdEvt += "{0}.JEDI_Events tab ".format(panda_config.schemaJEDI)
                        sqlJediOdEvt += "SET status=:newStatus "
                        sqlJediOdEvt += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                        sqlJediOdEvt += "AND status IN (:esFinished,:esDone) "
                        varMap = {}
                        varMap[':jediTaskID']  = file.jediTaskID
                        varMap[':datasetID']   = file.datasetID
                        varMap[':fileID']      = file.fileID
                        if not job.notDiscardEvents():
                            varMap[':newStatus']   = EventServiceUtils.ST_discarded
                        else:
                            varMap[':newStatus']   = EventServiceUtils.ST_done
                        varMap[':esDone']      = EventServiceUtils.ST_done
                        varMap[':esFinished']  = EventServiceUtils.ST_finished
                        _logger.debug(sqlJediOdEvt+comment+str(varMap))
                        self.cur.execute(sqlJediOdEvt+comment, varMap)
                        # cancel old unprocessed event ranges
                        sqlJediCEvt  = "UPDATE /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
                        sqlJediCEvt += "{0}.JEDI_Events tab ".format(panda_config.schemaJEDI)
                        sqlJediCEvt += "SET status=:newStatus "
                        sqlJediCEvt += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                        sqlJediCEvt += "AND NOT status IN (:esFinished,:esDone,:esDiscarded,:esCancelled,:esFailed,:esFatal,:esCorrupted) "
                        sqlJediCEvt += "AND (is_jumbo IS NULL OR (is_jumbo=:isJumbo AND status NOT IN (:esSent,:esRunning))) "
                        varMap[':newStatus']   = EventServiceUtils.ST_cancelled
                        varMap[':esDiscarded'] = EventServiceUtils.ST_discarded
                        varMap[':esCancelled'] = EventServiceUtils.ST_cancelled
                        varMap[':esCorrupted'] = EventServiceUtils.ST_corrupted
                        varMap[':esFatal']     = EventServiceUtils.ST_fatal
                        varMap[':esFailed']    = EventServiceUtils.ST_failed
                        varMap[':esSent']      = EventServiceUtils.ST_sent
                        varMap[':esRunning']   = EventServiceUtils.ST_running
                        varMap[':isJumbo'] = EventServiceUtils.eventTableIsJumbo
                        tmp_log.debug(sqlJediCEvt+comment+str(varMap))
                        self.cur.execute(sqlJediCEvt+comment, varMap)
                        # unset processed_upto for old failed events
                        sqlJediFEvt  = "UPDATE /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
                        sqlJediFEvt += "{0}.JEDI_Events tab ".format(panda_config.schemaJEDI)
                        sqlJediFEvt += "SET processed_upto_eventID=NULL "
                        sqlJediFEvt += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                        sqlJediFEvt += "AND status=:esFailed AND processed_upto_eventID IS NOT NULL "
                        varMap = {}
                        varMap[':jediTaskID']  = file.jediTaskID
                        varMap[':datasetID']   = file.datasetID
                        varMap[':fileID']      = file.fileID
                        varMap[':esFailed']    = EventServiceUtils.ST_failed
                        tmp_log.debug(sqlJediFEvt+comment+str(varMap))
                        self.cur.execute(sqlJediFEvt+comment, varMap)
                        # event range with offset
                        withOffset = False
                        if 'offset' in eventServiceInfo[file.lfn] and eventServiceInfo[file.lfn]['offset'] != -1:
                            withOffset = True
                        # get sucessful event ranges
                        okRanges = set()
                        if job.notDiscardEvents():
                            sqlJediOks = "SELECT /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
                            sqlJediOks += "jediTaskID,fileID,job_processID FROM {0}.JEDI_Events tab ".format(panda_config.schemaJEDI)
                            sqlJediOks += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                            sqlJediOks += "AND (status=:esDone OR (is_jumbo=:isJumbo AND status IN (:esSent,:esRunning))) "
                            varMap = {}
                            varMap[':jediTaskID']  = file.jediTaskID
                            varMap[':datasetID']   = file.datasetID
                            varMap[':fileID']      = file.fileID
                            varMap[':esDone']      = EventServiceUtils.ST_done
                            varMap[':esSent']      = EventServiceUtils.ST_sent
                            varMap[':esRunning']   = EventServiceUtils.ST_running
                            varMap[':isJumbo'] = EventServiceUtils.eventTableIsJumbo
                            self.cur.execute(sqlJediOks+comment, varMap)
                            resOks = self.cur.fetchall()
                            for tmpOk_jediTaskID,tmpOk_fileID,tmpOk_job_processID in resOks:
                                okRanges.add('{0}-{1}-{2}'.format(tmpOk_jediTaskID,tmpOk_fileID,tmpOk_job_processID))
                        # insert new ranges
                        sqlJediEvent  = "INSERT INTO {0}.JEDI_Events ".format(panda_config.schemaJEDI)
                        sqlJediEvent += "(jediTaskID,datasetID,PandaID,fileID,attemptNr,status,"
                        sqlJediEvent += "job_processID,def_min_eventID,def_max_eventID,processed_upto_eventID,"
                        sqlJediEvent += "event_offset"
                        sqlJediEvent += ") "
                        sqlJediEvent += "VALUES(:jediTaskID,:datasetID,:pandaID,:fileID,:attemptNr,:eventStatus,"
                        sqlJediEvent += ":startEvent,:startEvent,:lastEvent,:processedEvent,"
                        sqlJediEvent += ":eventOffset"
                        sqlJediEvent += ") "
                        varMaps = []
                        iEvent = 1
                        while iEvent <= eventServiceInfo[file.lfn]['nEvents']:
                            varMap = {}
                            varMap[':jediTaskID'] = file.jediTaskID
                            varMap[':datasetID'] = file.datasetID
                            varMap[':pandaID'] = job.jobsetID
                            varMap[':fileID'] = file.fileID
                            varMap[':attemptNr'] = eventServiceInfo[file.lfn]['maxAttempt']
                            varMap[':eventStatus'] = EventServiceUtils.ST_ready
                            varMap[':processedEvent'] = 0
                            varMap[':startEvent'] = eventServiceInfo[file.lfn]['startEvent'] + iEvent
                            iEvent += eventServiceInfo[file.lfn]['nEventsPerRange']
                            if iEvent > eventServiceInfo[file.lfn]['nEvents']:
                                iEvent = eventServiceInfo[file.lfn]['nEvents'] + 1
                            lastEvent = eventServiceInfo[file.lfn]['startEvent'] + iEvent -1
                            varMap[':lastEvent'] = lastEvent
                            # add offset for positional event numbers
                            if not job.inFilePosEvtNum():
                                varMap[':startEvent'] += totalInputEvents
                                varMap[':lastEvent'] += totalInputEvents
                            # keep jobsetID
                            varMap[':eventOffset'] = job.jobsetID
                            # skip if already succeeded
                            tmpKey = '{0}-{1}-{2}'.format(varMap[':jediTaskID'], varMap[':fileID'], varMap[':startEvent'])
                            if tmpKey in okRanges:
                                continue
                            varMaps.append(varMap)
                            nEventsToProcess += 1
                        tmp_log.debug("%s insert %s event ranges jediTaskID:%s" % (job.PandaID,len(varMaps),
                                                                                                  job.jediTaskID))
                        self.cur.executemany(sqlJediEvent+comment, varMaps)
                        tmp_log.debug("%s inserted %s event ranges jediTaskID:%s" % (job.PandaID,len(varMaps),
                                                                                                    job.jediTaskID))
                        totalInputEvents += eventServiceInfo[file.lfn]['nEvents']
            if job.notDiscardEvents() and origEsJob and nEventsToProcess == 0:
                job.setAllOkEvents()
                if not toPending:
                    sqlJediJSH = "UPDATE ATLAS_PANDA.jobsDefined4 "
                else:
                    sqlJediJSH = "UPDATE ATLAS_PANDA.jobsWaiting4 "
                sqlJediJSH += "SET specialHandling=:specialHandling WHERE PandaID=:PandaID "
                varMap = dict()
                varMap[':specialHandling'] = job.specialHandling
                varMap[':PandaID'] = job.PandaID
                self.cur.execute(sqlJediJSH+comment, varMap)
            # use score if not so many events are available
            if origEsJob and unprocessedMap is not None:
                unprocessedMap[job.jobsetID] = nEventsToProcess
            if EventServiceUtils.isEventServiceJob(job) and not EventServiceUtils.isJobCloningJob(job) and unprocessedMap is not None:
                if job.coreCount not in [None, '', 'NULL'] and job.coreCount > 1:
                    minUnprocessed = self.getConfigValue('dbproxy', 'AES_MINEVENTSFORMCORE')
                    if minUnprocessed is not None:
                        minUnprocessed = max(minUnprocessed, job.coreCount)
                        if unprocessedMap[job.jobsetID] < minUnprocessed and unprocessedMap[job.jobsetID] > 0:
                            self.setScoreSiteToEs(job, "insertNewJob : {0}".format(job.PandaID), comment)
            # bulk insert files
            if len(varMapsForFile) > 0:
                tmp_log.debug("{0} bulk insert {1} files for jediTaskID:{2}".format(job.PandaID,len(varMapsForFile),
                                                                                                   job.jediTaskID))
                self.cur.executemany(sqlFileW+comment,varMapsForFile)
            # update nFilesWaiting
            if len(nFilesWaitingMap) > 0:
                sqlJediNFW  = "UPDATE {0}.JEDI_Datasets ".format(panda_config.schemaJEDI)
                sqlJediNFW += "SET nFilesWaiting=nFilesWaiting+:nDiff "
                sqlJediNFW += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID "
                for tmpDatasetID in nFilesWaitingMap:
                    nDiff = nFilesWaitingMap[tmpDatasetID]
                    varMap = {}
                    varMap[':jediTaskID'] = job.jediTaskID
                    varMap[':datasetID'] = tmpDatasetID
                    varMap[':nDiff'] = nDiff
                    self.cur.execute(sqlJediNFW+comment, varMap)
            # insert events for dynamic number of events
            if dynFileMap != {}:
                # insert new ranges
                sqlJediEvent  = "INSERT INTO {0}.JEDI_Events ".format(panda_config.schemaJEDI)
                sqlJediEvent += "(jediTaskID,datasetID,PandaID,fileID,attemptNr,status,"
                sqlJediEvent += "job_processID,def_min_eventID,def_max_eventID,processed_upto_eventID) "
                sqlJediEvent += "VALUES(:jediTaskID,:datasetID,:pandaID,:fileID,:attemptNr,:eventStatus,"
                sqlJediEvent += ":processID,:startEvent,:lastEvent,:processedEvent) "
                varMaps = []
                for tmpLFN in dynFileMap:
                    dynFiles = dynFileMap[tmpLFN]
                    for tmpJediTaskID,tmpDatasetID,tmpFileID,tmpAttemptNr in dynFiles:
                        varMap = {}
                        varMap[':jediTaskID'] = tmpJediTaskID
                        varMap[':datasetID'] = tmpDatasetID
                        varMap[':pandaID'] = job.PandaID
                        varMap[':fileID'] = tmpFileID
                        varMap[':attemptNr'] = tmpAttemptNr+1 # to avoid 0
                        varMap[':eventStatus'] = EventServiceUtils.ST_discarded
                        varMap[':processID'] = dynLfnIdMap[tmpLFN]
                        varMap[':processedEvent'] = -1
                        varMap[':startEvent'] = 0
                        varMap[':lastEvent'] = 0
                        varMaps.append(varMap)
                self.cur.executemany(sqlJediEvent+comment, varMaps)
                tmp_log.debug("%s inserted %s dyn events jediTaskID:%s" % (job.PandaID,len(varMaps),
                                                                                          job.jediTaskID))
            # update t_task
            if useJEDI and job.prodSourceLabel not in ['panda'] and job.processingType != 'pmerge':
                varMap = {}
                varMap[':jediTaskID'] = job.jediTaskID
                schemaDEFT = self.getSchemaDEFT()
                sqlTtask  = "UPDATE {0}.T_TASK ".format(schemaDEFT)
                sqlTtask += "SET total_req_jobs=total_req_jobs+1,timestamp=CURRENT_DATE "
                sqlTtask += "WHERE taskid=:jediTaskID "
                tmp_log.debug(sqlTtask+comment+str(varMap))
                self.cur.execute(sqlTtask+comment,varMap)
                tmp_log.debug("%s updated T_TASK jediTaskID:%s" % (job.PandaID,job.jediTaskID))
            # metadata
            if job.prodSourceLabel in ['user','panda'] and job.metadata != '':
                sqlMeta = "INSERT INTO ATLAS_PANDA.metaTable (PandaID,metaData) VALUES (:PandaID,:metaData)"
                varMap = {}
                varMap[':PandaID']  = job.PandaID
                varMap[':metaData'] = job.metadata
                tmp_log.debug("%s inserting meta jediTaskID:%s" % (job.PandaID,job.jediTaskID))
                self.cur.execute(sqlMeta+comment, varMap)
                tmp_log.debug("%s inserted meta jediTaskID:%s" % (job.PandaID,job.jediTaskID))
            # job parameters
            if job.prodSourceLabel not in ['managed']:
                job.jobParameters = re.sub('\$JOBSETID', jobsetID, job.jobParameters)
                job.jobParameters = re.sub('\$GROUPJOBSN', groupJobSN, job.jobParameters)
                try:
                    job.jobParameters = re.sub('\$JEDITASKID', strJediTaskID, job.jobParameters)
                except Exception:
                    pass
            sqlJob = "INSERT INTO ATLAS_PANDA.jobParamsTable (PandaID,jobParameters) VALUES (:PandaID,:param)"
            varMap = {}
            varMap[':PandaID'] = job.PandaID
            varMap[':param']   = job.jobParameters
            tmp_log.debug("%s inserting jobParam jediTaskID:%s" % (job.PandaID,job.jediTaskID))
            self.cur.execute(sqlJob+comment, varMap)
            tmp_log.debug("%s inserted jobParam jediTaskID:%s" % (job.PandaID,job.jediTaskID))
            # update input
            if useJEDI and not EventServiceUtils.isJumboJob(job) and job.computingSite != EventServiceUtils.siteIdForWaitingCoJumboJobs and \
                    not (EventServiceUtils.isEventServiceJob(job) and not origEsJob):
                self.updateInputStatusJedi(job.jediTaskID, job.PandaID, 'queued')
            # record retry history
            if oldPandaIDs is not None and len(oldPandaIDs) > 0:
                tmp_log.debug("%s recording history nOld=%s jediTaskID:%s" % (job.PandaID,len(oldPandaIDs),job.jediTaskID))
                self.recordRetryHistoryJEDI(job.jediTaskID,job.PandaID,oldPandaIDs,relationType)
                tmp_log.debug("%s recorded history jediTaskID:%s" % (job.PandaID,job.jediTaskID))
            # record jobset
            if origEsJob:
                self.recordRetryHistoryJEDI(job.jediTaskID,job.PandaID,[job.jobsetID],EventServiceUtils.relationTypeJS_ID)
                # record jobset history
                if oldPandaIDs is not None and len(oldPandaIDs) > 0:
                    # get old jobsetID
                    for oldPandaID in oldPandaIDs:
                        oldJobsetID = self.getJobsetIDforPandaID(oldPandaID,job.jediTaskID)
                        if oldJobsetID is not None:
                            self.recordRetryHistoryJEDI(job.jediTaskID,job.jobsetID,[oldJobsetID],EventServiceUtils.relationTypeJS_Retry)
            # record jobset mapping for event service
            if EventServiceUtils.isEventServiceJob(job) and EventServiceUtils.isResurrectConsumers(job.specialHandling):
                self.recordRetryHistoryJEDI(job.jediTaskID, job.jobsetID, [job.PandaID], EventServiceUtils.relationTypeJS_Map)
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            tmp_log.debug("%s all OK jediTaskID:%s" % (job.PandaID,job.jediTaskID))
            # record status change
            try:
                self.recordStatusChange(job.PandaID, job.jobStatus, jobInfo=job)
            except Exception:
                tmp_log.error('recordStatusChange in insertNewJob')
            self.push_job_status_message(job, job.PandaID, job.jobStatus, job.jediTaskID, origSpecialHandling)
            if unprocessedMap is not None:
                return True, unprocessedMap
            return True
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            if unprocessedMap is not None:
                return False, unprocessedMap
            return False


    # simply insert job to a table
    def insertJobSimple(self,job,table,fileTable,jobParamsTable,metaTable):
        comment = ' /* DBProxy.insertJobSimple */'
        _logger.debug("insertJobSimple : %s" % job.PandaID)
        sql1 = "INSERT INTO %s (%s) " % (table,JobSpec.columnNames())
        sql1+= JobSpec.bindValuesExpression()
        try:
            # begin transaction
            self.conn.begin()
            # insert
            self.cur.execute(sql1+comment, job.valuesMap())
            # files
            sqlFile = "INSERT INTO %s " % fileTable
            sqlFile+= "(%s) " % FileSpec.columnNames(withMod=True)
            sqlFile+= FileSpec.bindValuesExpression(withMod=True)
            for file in job.Files:
                varMap = file.valuesMap()
                varMap[':modificationTime'] = job.modificationTime
                self.cur.execute(sqlFile+comment, varMap)
            # job parameters
            sqlJob = "INSERT INTO %s (PandaID,jobParameters,modificationTime) VALUES (:PandaID,:param,:modificationTime)" \
                     % jobParamsTable
            varMap = {}
            varMap[':PandaID'] = job.PandaID
            varMap[':param']   = job.jobParameters
            varMap[':modificationTime'] = job.modificationTime
            self.cur.execute(sqlJob+comment, varMap)
            # metadata
            if job.metadata not in [None,'NULL','']:
                sqlMeta = "INSERT INTO %s (PandaID,metaData,modificationTime) VALUES(:PandaID,:metaData,:modificationTime)" \
                          % metaTable
                varMap = {}
                varMap[':PandaID']  = job.PandaID
                varMap[':metaData'] = job.metadata
                varMap[':modificationTime'] = job.modificationTime
                self.cur.execute(sqlMeta+comment,varMap)
            # set flag to avoid duplicated insertion attempts
            varMap = {}
            varMap[':PandaID']      = job.PandaID
            varMap[':archivedFlag'] = 1
            sqlArch = "UPDATE ATLAS_PANDA.jobsArchived4 SET archivedFlag=:archivedFlag WHERE PandaID=:PandaID"
            self.cur.execute(sqlArch+comment, varMap)
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            return True
        except Exception:
            type, value, traceBack = sys.exc_info()
            _logger.error("insertJobSimple : %s %s" % (type,value))
            # roll back
            self._rollback()
            return False


    # simply insert job to a table without reading
    def insertJobSimpleUnread(self,pandaID,modTime):
        comment = ' /* DBProxy.insertJobSimpleUnread */'
        _logger.debug("insertJobSimpleUnread : %s" % pandaID)
        # check
        sqlC = "SELECT archivedFlag FROM ATLAS_PANDA.jobsArchived4 "
        sqlC += "WHERE PandaID=:pandaID "
        # job
        sqlJ  = "INSERT INTO ATLAS_PANDAARCH.jobsArchived (%s) " % JobSpec.columnNames()
        sqlJ += "SELECT %s FROM ATLAS_PANDA.jobsArchived4 " % JobSpec.columnNames()
        sqlJ += "WHERE PandaID=:pandaID "
        # file
        sqlF  = "INSERT INTO ATLAS_PANDAARCH.filesTable_ARCH (%s) " % FileSpec.columnNames(withMod=True)
        sqlF += "SELECT %s,:modTime FROM ATLAS_PANDA.filesTable4 " % FileSpec.columnNames(withMod=False)
        sqlF += "WHERE PandaID=:pandaID "
        # parameters
        sqlP  = "INSERT INTO ATLAS_PANDAARCH.jobParamsTable_ARCH (PandaID,jobParameters,modificationTime) "
        sqlP += "SELECT PandaID,jobParameters,:modTime FROM ATLAS_PANDA.jobParamsTable "
        sqlP += "WHERE PandaID=:pandaID "
        # metadata
        sqlM1  = "SELECT PandaID FROM ATLAS_PANDA.metaTable "
        sqlM1 += "WHERE PandaID=:pandaID AND rownum<=1 "
        sqlM2  = "INSERT INTO ATLAS_PANDAARCH.metaTable_ARCH (PandaID,metaData,modificationTime) "
        sqlM2 += "SELECT PandaID,metaData,:modTime FROM ATLAS_PANDA.metaTable "
        sqlM2 += "WHERE PandaID=:pandaID "
        try:
            # begin transaction
            self.conn.begin()
            # check
            varMap = {}
            varMap[':pandaID'] = pandaID
            self.cur.execute(sqlC+comment,varMap)
            res = self.cur.fetchone()
            if res is None or res[0] == 1:
                if res is None:
                    _logger.error("insertJobSimpleUnread : %s cannot get archivedFlag" % pandaID)
                else:
                    _logger.debug("insertJobSimpleUnread : %s skip" % pandaID)
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                return True
            # insert
            varMap = {}
            varMap[':pandaID'] = pandaID
            self.cur.execute(sqlJ+comment,varMap)
            varMap = {}
            varMap[':pandaID'] = pandaID
            varMap[':modTime'] = modTime
            self.cur.execute(sqlF+comment,varMap)
            varMap = {}
            varMap[':pandaID'] = pandaID
            varMap[':modTime'] = modTime
            self.cur.execute(sqlP+comment,varMap)
            varMap = {}
            varMap[':pandaID'] = pandaID
            self.cur.execute(sqlM1+comment,varMap)
            res = self.cur.fetchone()
            if res is not None:
                varMap = {}
                varMap[':pandaID'] = pandaID
                varMap[':modTime'] = modTime
                self.cur.execute(sqlM2+comment,varMap)
            # set flag to avoid duplicated insertion attempts
            varMap = {}
            varMap[':PandaID']      = pandaID
            varMap[':archivedFlag'] = 1
            sqlArch = "UPDATE ATLAS_PANDA.jobsArchived4 SET archivedFlag=:archivedFlag WHERE PandaID=:PandaID"
            self.cur.execute(sqlArch+comment, varMap)
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            return True
        except Exception:
            type, value, traceBack = sys.exc_info()
            _logger.error("insertJobSimpleUnread %s : %s %s" % (pandaID,type,value))
            # roll back
            self._rollback()
            return False


    # delete job
    def deleteJobSimple(self,pandaID):
        comment = ' /* DBProxy.deleteJobSimple */'
        _logger.debug("deleteJobSimple : %s" % pandaID)
        try:
            # begin transaction
            self.conn.begin()
            # delete
            varMap = {}
            varMap[':PandaID']  = pandaID
            sql = 'DELETE from ATLAS_PANDA.jobsArchived4 WHERE PandaID=:PandaID'
            self.cur.execute(sql+comment, varMap)
            sql = "DELETE from ATLAS_PANDA.filesTable4 WHERE PandaID=:PandaID"
            self.cur.execute(sql+comment, varMap)
            sql = "DELETE from ATLAS_PANDA.metaTable WHERE PandaID=:PandaID"
            self.cur.execute(sql+comment, varMap)
            sql = "DELETE from ATLAS_PANDA.jobParamsTable WHERE PandaID=:PandaID"
            self.cur.execute(sql+comment, varMap)
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            return True
        except Exception:
            type, value = sys.exc_info()[:2]
            _logger.error("deleteJobSimple %s : %s %s" % (pandaID,type,value))
            # roll back
            self._rollback()
            return False


    # activate job. move job from jobsDefined to jobsActive
    def activateJob(self,job):
        comment = ' /* DBProxy.activateJob */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        if job is None:
            tmp_id = None
        else:
            tmp_id = job.PandaID
        tmpLog = LogWrapper(_logger, methodName + " < PandaID={} >".format(tmp_id))
        updatedFlag = False
        if job is None:
            tmpLog.debug('skip job=None')
            return True
        tmpLog.debug("start")
        sql0 = "SELECT row_ID FROM ATLAS_PANDA.filesTable4 WHERE PandaID=:PandaID AND type=:type AND NOT status IN (:status1,:status2) "
        sql1 = "DELETE FROM ATLAS_PANDA.jobsDefined4 "
        sql1+= "WHERE PandaID=:PandaID AND (jobStatus=:oldJobStatus1 OR jobStatus=:oldJobStatus2) AND commandToPilot IS NULL"
        sql2 = "INSERT INTO ATLAS_PANDA.jobsActive4 (%s) " % JobSpec.columnNames()
        sql2+= JobSpec.bindValuesExpression()
        # host and time information
        job.modificationTime = datetime.datetime.utcnow()
        # set stateChangeTime for defined->activated but not for assigned->activated
        if job.jobStatus in ['defined']:
            job.stateChangeTime = job.modificationTime
        nTry=3
        to_push = False
        for iTry in range(nTry):
            try:
                # check if all files are ready
                allOK = True
                for file in job.Files:
                    if file.type == 'input' and file.status not in ['ready','cached']:
                        allOK = False
                        break
                # begin transaction
                self.conn.begin()
                # check all inputs are ready
                varMap = {}
                varMap[':type']    = 'input'
                varMap[':status1'] = 'ready'
                varMap[':status2'] = 'cached'
                varMap[':PandaID'] = job.PandaID
                self.cur.arraysize = 100
                self.cur.execute(sql0+comment, varMap)
                res = self.cur.fetchall()
                if len(res) == 0 or allOK:
                    # check resource share
                    job.jobStatus = "activated"

                    # delete
                    varMap = {}
                    varMap[':PandaID']       = job.PandaID
                    varMap[':oldJobStatus1'] = 'assigned'
                    varMap[':oldJobStatus2'] = 'defined'
                    self.cur.execute(sql1+comment, varMap)
                    n = self.cur.rowcount
                    if n==0:
                        # already killed or activated
                        tmpLog.debug("Job not found to activate")
                    else:
                        # insert
                        self.cur.execute(sql2+comment, job.valuesMap())
                        # update files
                        datasetContentsStat = {}
                        for file in job.Files:
                            sqlF = ("UPDATE ATLAS_PANDA.filesTable4 SET %s" % file.bindUpdateChangesExpression()) + "WHERE row_ID=:row_ID"
                            varMap = file.valuesMap(onlyChanged=True)
                            if varMap != {}:
                                varMap[':row_ID'] = file.row_ID
                                _logger.debug(sqlF+comment+str(varMap))
                                self.cur.execute(sqlF+comment, varMap)
                        # job parameters
                        sqlJob = "UPDATE ATLAS_PANDA.jobParamsTable SET jobParameters=:param WHERE PandaID=:PandaID"
                        varMap = {}
                        varMap[':PandaID'] = job.PandaID
                        varMap[':param']   = job.jobParameters
                        self.cur.execute(sqlJob+comment, varMap)
                        updatedFlag = True
                        to_push = job.is_push_job()
                else:
                    # update job
                    sqlJ = ("UPDATE ATLAS_PANDA.jobsDefined4 SET %s " % job.bindUpdateChangesExpression()) + \
                           "WHERE PandaID=:PandaID AND (jobStatus=:oldJobStatus1 OR jobStatus=:oldJobStatus2)"
                    varMap = job.valuesMap(onlyChanged=True)
                    varMap[':PandaID']       = job.PandaID
                    varMap[':oldJobStatus1'] = 'assigned'
                    varMap[':oldJobStatus2'] = 'defined'
                    _logger.debug(sqlJ+comment+str(varMap))
                    self.cur.execute(sqlJ+comment, varMap)
                    n = self.cur.rowcount
                    if n==0:
                        # already killed or activated
                        tmpLog.debug("Job not found to update")
                    else:
                        # update files
                        for file in job.Files:
                            sqlF = ("UPDATE ATLAS_PANDA.filesTable4 SET %s" % file.bindUpdateChangesExpression()) + "WHERE row_ID=:row_ID"
                            varMap = file.valuesMap(onlyChanged=True)
                            if varMap != {}:
                                varMap[':row_ID'] = file.row_ID
                                _logger.debug(sqlF+comment+str(varMap))
                                self.cur.execute(sqlF+comment, varMap)
                        # job parameters
                        sqlJob = "UPDATE ATLAS_PANDA.jobParamsTable SET jobParameters=:param WHERE PandaID=:PandaID"
                        varMap = {}
                        varMap[':PandaID'] = job.PandaID
                        varMap[':param']   = job.jobParameters
                        self.cur.execute(sqlJob+comment, varMap)
                        updatedFlag = True
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                # record status change
                try:
                    if updatedFlag:
                        self.recordStatusChange(job.PandaID,job.jobStatus,jobInfo=job)
                except Exception:
                    tmpLog.error('recordStatusChange failed')
                self.push_job_status_message(job, job.PandaID, job.jobStatus)
                # push job
                if to_push:
                    mb_proxy_queue = self.get_mb_proxy('panda_pilot_queue')
                    mb_proxy_topic = self.get_mb_proxy('panda_pilot_topic')
                    if mb_proxy_queue and mb_proxy_topic:
                        tmpLog.debug('push job')
                        srv_msg_utils.send_job_message(mb_proxy_queue, mb_proxy_topic,
                                                       job.jediTaskID, job.PandaID)
                    else:
                        tmpLog.debug('message queue/topic not configured')
                tmpLog.debug('done')
                return True
            except Exception as e:
                # roll back
                self._rollback()
                if iTry+1 < nTry:
                    tmpLog.debug("retry: {}".format(iTry))
                    time.sleep(random.randint(10,20))
                    continue
                tmpLog.error("failed with {}".format(str(e)))
                return False


    # send job to jobsWaiting
    def keepJob(self,job):
        comment = ' /* DBProxy.keepJob */'
        _logger.debug("keepJob : %s" % job.PandaID)
        sql1 = "DELETE FROM ATLAS_PANDA.jobsDefined4 "
        sql1+= "WHERE PandaID=:PandaID AND (jobStatus=:oldJobStatus1 OR jobStatus=:oldJobStatus2) AND commandToPilot IS NULL"
        sql2 = "INSERT INTO ATLAS_PANDA.jobsWaiting4 (%s) " % JobSpec.columnNames()
        sql2+= JobSpec.bindValuesExpression()
        # time information
        job.modificationTime = datetime.datetime.utcnow()
        job.stateChangeTime  = job.modificationTime
        updatedFlag = False
        nTry=3
        for iTry in range(nTry):
            try:
                # begin transaction
                self.conn.begin()
                # delete
                varMap = {}
                varMap[':PandaID']       = job.PandaID
                varMap[':oldJobStatus1'] = 'assigned'
                varMap[':oldJobStatus2'] = 'defined'
                self.cur.execute(sql1+comment, varMap)
                n = self.cur.rowcount
                if n==0:
                    # already killed
                    _logger.debug("keepJob : Not found %s" % job.PandaID)
                else:
                    # set status
                    job.jobStatus = 'waiting'
                    # insert
                    self.cur.execute(sql2+comment, job.valuesMap())
                    # update files
                    for file in job.Files:
                        sqlF = ("UPDATE ATLAS_PANDA.filesTable4 SET %s" % file.bindUpdateChangesExpression()) + "WHERE row_ID=:row_ID"
                        varMap = file.valuesMap(onlyChanged=True)
                        if varMap != {}:
                            varMap[':row_ID'] = file.row_ID
                            _logger.debug(sqlF+comment+str(varMap))
                            self.cur.execute(sqlF+comment, varMap)
                    # update parameters
                    sqlJob = "UPDATE ATLAS_PANDA.jobParamsTable SET jobParameters=:param WHERE PandaID=:PandaID"
                    varMap = {}
                    varMap[':PandaID'] = job.PandaID
                    varMap[':param']   = job.jobParameters
                    self.cur.execute(sqlJob+comment, varMap)
                    updatedFlag = True
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                # record status change
                try:
                    if updatedFlag:
                        self.recordStatusChange(job.PandaID,job.jobStatus,jobInfo=job)
                        self.push_job_status_message(job, job.PandaID, job.jobStatus)
                except Exception:
                    _logger.error('recordStatusChange in keepJob')
                return True
            except Exception:
                # roll back
                self._rollback()
                if iTry+1 < nTry:
                    _logger.debug("keepJob : %s retry : %s" % (job.PandaID,iTry))
                    time.sleep(random.randint(10,20))
                    continue
                type, value, traceBack = sys.exc_info()
                _logger.error("keepJob : %s %s" % (type,value))
                return False

    # archive job to jobArchived and remove the job from jobsActive or jobsDefined
    def archiveJob(self,job,fromJobsDefined,useCommit=True,extraInfo=None,fromJobsWaiting=False):
        comment = ' /* DBProxy.archiveJob */'
        _logger.debug("archiveJob : %s %s" % (job.PandaID, job.jobStatus))
        if fromJobsDefined:
            sql0 = "SELECT jobStatus FROM ATLAS_PANDA.jobsDefined4 WHERE PandaID=:PandaID "
            sql1 = "DELETE FROM ATLAS_PANDA.jobsDefined4 WHERE PandaID=:PandaID AND (jobStatus=:oldJobStatus1 OR jobStatus=:oldJobStatus2)"
        elif fromJobsWaiting:
            sql0 = "SELECT jobStatus FROM ATLAS_PANDA.jobsWaiting4 WHERE PandaID=:PandaID "
            sql1 = "DELETE FROM ATLAS_PANDA.jobsWaiting4 WHERE PandaID=:PandaID"
        else:
            sql0 = "SELECT jobStatus FROM ATLAS_PANDA.jobsActive4 WHERE PandaID=:PandaID FOR UPDATE "
            sql1 = "DELETE FROM ATLAS_PANDA.jobsActive4 WHERE PandaID=:PandaID"
        sql2 = "INSERT INTO ATLAS_PANDA.jobsArchived4 (%s) " % JobSpec.columnNames()
        sql2+= JobSpec.bindValuesExpression()
        updatedJobList = []
        nTry=1
        for iTry in range(nTry):
            try:
                # begin transaction
                if useCommit:
                    self.conn.begin()
                # check if JEDI is used
                useJEDI = False
                if hasattr(panda_config,'useJEDI') and panda_config.useJEDI is True and \
                        job.lockedby == 'jedi' and self.checkTaskStatusJEDI(job.jediTaskID,self.cur):
                    useJEDI = True
                if useCommit:
                    if not self._commit():
                        raise RuntimeError('Commit error')
                # delete downstream jobs first
                ddmIDs     = []
                newJob     = None
                ddmAttempt = 0
                if job.prodSourceLabel == 'panda' and job.jobStatus == 'failed':
                    # look for outputs
                    upOutputs = []
                    for file in job.Files:
                        if file.type == 'output':
                            upOutputs.append(file.lfn)
                    toBeClosedSubList = {}
                    topUserDsList = []
                    # look for downstream jobs
                    sqlD   = "SELECT PandaID FROM ATLAS_PANDA.filesTable4 WHERE type=:type AND lfn=:lfn GROUP BY PandaID"
                    sqlDJS = "SELECT %s " % JobSpec.columnNames()
                    sqlDJS+= "FROM ATLAS_PANDA.jobsDefined4 WHERE PandaID=:PandaID"
                    sqlDJD = "DELETE FROM ATLAS_PANDA.jobsDefined4 WHERE PandaID=:PandaID"
                    sqlDJI = "INSERT INTO ATLAS_PANDA.jobsArchived4 (%s) " % JobSpec.columnNames()
                    sqlDJI+= JobSpec.bindValuesExpression()
                    sqlDFup = "UPDATE ATLAS_PANDA.filesTable4 SET status=:status WHERE PandaID=:PandaID AND type IN (:type1,:type2)"
                    sqlFMod = "UPDATE ATLAS_PANDA.filesTable4 SET modificationTime=:modificationTime WHERE PandaID=:PandaID"
                    sqlMMod = "UPDATE ATLAS_PANDA.metaTable SET modificationTime=:modificationTime WHERE PandaID=:PandaID"
                    sqlPMod = "UPDATE ATLAS_PANDA.jobParamsTable SET modificationTime=:modificationTime WHERE PandaID=:PandaID"
                    sqlGetSub = "SELECT DISTINCT destinationDBlock FROM ATLAS_PANDA.filesTable4 WHERE type=:type AND PandaID=:PandaID"
                    sqlCloseSub  = 'UPDATE /*+ INDEX_RS_ASC(TAB("DATASETS"."NAME")) */ ATLAS_PANDA.Datasets tab '
                    sqlCloseSub += 'SET status=:status,modificationDate=CURRENT_DATE WHERE name=:name'
                    sqlDFile = "SELECT %s FROM ATLAS_PANDA.filesTable4 " % FileSpec.columnNames()
                    sqlDFile+= "WHERE PandaID=:PandaID"
                    for upFile in upOutputs:
                        _logger.debug("look for downstream jobs for %s" % upFile)
                        if useCommit:
                            self.conn.begin()
                        # select PandaID
                        varMap = {}
                        varMap[':lfn']  = upFile
                        varMap[':type'] = 'input'
                        self.cur.arraysize = 100000
                        self.cur.execute(sqlD+comment, varMap)
                        res = self.cur.fetchall()
                        if useCommit:
                            if not self._commit():
                                raise RuntimeError('Commit error')
                        iDownJobs = 0
                        nDownJobs = len(res)
                        nDownChunk = 20
                        inTransaction = False
                        _logger.debug("archiveJob : {0} found {1} downstream jobs for {2}".format(job.PandaID,nDownJobs,upFile))
                        # loop over all downstream IDs
                        for downID, in res:
                            if useCommit:
                                if not inTransaction:
                                    self.conn.begin()
                                    inTransaction = True
                            _logger.debug("archiveJob : {0} delete : {1} ({2}/{3})".format(job.PandaID,downID,iDownJobs,nDownJobs))
                            iDownJobs += 1
                            # select jobs
                            varMap = {}
                            varMap[':PandaID'] = downID
                            self.cur.arraysize = 10
                            self.cur.execute(sqlDJS+comment, varMap)
                            resJob = self.cur.fetchall()
                            if len(resJob) == 0:
                                if useCommit and (iDownJobs % nDownChunk) == 0:
                                    if not self._commit():
                                        raise RuntimeError('Commit error')
                                    inTransaction = False
                                continue
                            # instantiate JobSpec
                            dJob = JobSpec()
                            dJob.pack(resJob[0])
                            # delete
                            varMap = {}
                            varMap[':PandaID'] = downID
                            self.cur.execute(sqlDJD+comment, varMap)
                            retD = self.cur.rowcount
                            if retD == 0:
                                if useCommit and (iDownJobs % nDownChunk) == 0:
                                    if not self._commit():
                                        raise RuntimeError('Commit error')
                                    inTransaction = False
                                continue
                            # error code
                            dJob.jobStatus = 'cancelled'
                            dJob.endTime   = datetime.datetime.utcnow()
                            dJob.taskBufferErrorCode = ErrorCode.EC_Kill
                            dJob.taskBufferErrorDiag = 'killed by Panda server : upstream job failed'
                            dJob.modificationTime = dJob.endTime
                            dJob.stateChangeTime  = dJob.endTime
                            # insert
                            self.cur.execute(sqlDJI+comment, dJob.valuesMap())
                            # update file status
                            varMap = {}
                            varMap[':PandaID'] = downID
                            varMap[':status']  = 'failed'
                            varMap[':type1']   = 'output'
                            varMap[':type2']   = 'log'
                            self.cur.execute(sqlDFup+comment, varMap)
                            # update files,metadata,parametes
                            varMap = {}
                            varMap[':PandaID'] = downID
                            varMap[':modificationTime'] = dJob.modificationTime
                            self.cur.execute(sqlFMod+comment,varMap)
                            self.cur.execute(sqlMMod+comment,varMap)
                            self.cur.execute(sqlPMod+comment,varMap)
                            # collect to record state change
                            updatedJobList.append(dJob)
                            # update JEDI tables
                            if useJEDI:
                                # read files
                                varMap = {}
                                varMap[':PandaID'] = downID
                                self.cur.arraysize = 100000
                                self.cur.execute(sqlDFile+comment,varMap)
                                resDFiles = self.cur.fetchall()
                                for resDFile in resDFiles:
                                    tmpDFile = FileSpec()
                                    tmpDFile.pack(resDFile)
                                    dJob.addFile(tmpDFile)
                                self.propagateResultToJEDI(dJob,self.cur)
                            # set tobeclosed to sub datasets
                            if dJob.jobDefinitionID not in toBeClosedSubList:
                                # init
                                toBeClosedSubList[dJob.jobDefinitionID] = []
                                # get sub datasets
                                varMap = {}
                                varMap[':type'] = 'output'
                                varMap[':PandaID'] = downID
                                self.cur.arraysize = 1000
                                self.cur.execute(sqlGetSub+comment, varMap)
                                resGetSub = self.cur.fetchall()
                                if len(resGetSub) == 0:
                                    if useCommit and (iDownJobs % nDownChunk) == 0:
                                        if not self._commit():
                                            raise RuntimeError('Commit error')
                                        inTransaction = False
                                    continue
                                # loop over all sub datasets
                                for tmpDestinationDBlock, in resGetSub:
                                    if re.search('_sub\d+$',tmpDestinationDBlock) is None:
                                        continue
                                    if tmpDestinationDBlock not in toBeClosedSubList[dJob.jobDefinitionID]:
                                        # set tobeclosed
                                        varMap = {}
                                        varMap[':status'] = 'tobeclosed'
                                        varMap[':name'] = tmpDestinationDBlock
                                        self.cur.execute(sqlCloseSub+comment, varMap)
                                        _logger.debug("set tobeclosed for %s" % tmpDestinationDBlock)
                                        # append
                                        toBeClosedSubList[dJob.jobDefinitionID].append(tmpDestinationDBlock)
                                        # close top-level user dataset
                                        topUserDsName = re.sub('_sub\d+$','',tmpDestinationDBlock)
                                        if not useJEDI and topUserDsName != tmpDestinationDBlock and topUserDsName not in topUserDsList:
                                            # set tobeclosed
                                            varMap = {}
                                            if dJob.processingType.startswith('gangarobot') or \
                                                   dJob.processingType.startswith('hammercloud'):
                                                varMap[':status'] = 'completed'
                                            else:
                                                varMap[':status'] = 'tobeclosed'
                                            varMap[':name'] = topUserDsName
                                            self.cur.execute(sqlCloseSub+comment, varMap)
                                            _logger.debug("set %s for %s" % (varMap[':status'],topUserDsName))
                                            # append
                                            topUserDsList.append(topUserDsName)
                            if useCommit and (iDownJobs % nDownChunk) == 0:
                                if not self._commit():
                                    raise RuntimeError('Commit error')
                                inTransaction = False
                        if useCommit and inTransaction:
                            if not self._commit():
                                raise RuntimeError('Commit error')
                elif job.prodSourceLabel == 'ddm' and job.jobStatus == 'failed' and job.transferType=='dis':
                    if useCommit:
                        self.conn.begin()
                    # get corresponding jobs for production movers
                    vuid = ''
                    # extract vuid
                    match = re.search('--callBack (\S+)',job.jobParameters)
                    if match is not None:
                        try:
                            callbackUrl = urllib.unquote(match.group(1))
                            callbackUrl = re.sub('[&\?]',' ', callbackUrl)
                            # look for vuid=
                            for item in callbackUrl.split():
                                if item.startswith('vuid='):
                                    vuid = item.split('=')[-1]
                                    break
                        except Exception:
                            pass
                        if vuid == '':
                            _logger.error("cannot extract vuid from %s" % job.jobParameters)
                        else:
                            # get name
                            varMap = {}
                            varMap[':vuid'] = vuid
                            varMap[':type'] = 'dispatch'
                            self.cur.arraysize = 10
                            self.cur.execute("SELECT name FROM ATLAS_PANDA.Datasets WHERE vuid=:vuid AND type=:type "+comment, varMap)
                            res = self.cur.fetchall()
                            if len(res) != 0:
                                disName = res[0][0]
                                # check lost files
                                varMap = {}
                                varMap[':status'] = 'lost'
                                varMap[':dispatchDBlock'] = disName
                                sqlLost = "SELECT /*+ index(tab FILESTABLE4_DISPDBLOCK_IDX) */ distinct PandaID FROM ATLAS_PANDA.filesTable4 tab WHERE status=:status AND dispatchDBlock=:dispatchDBlock"
                                self.cur.execute(sqlLost+comment,varMap)
                                resLost = self.cur.fetchall()
                                # fail jobs with lost files
                                sqlDJS = "SELECT %s " % JobSpec.columnNames()
                                sqlDJS+= "FROM ATLAS_PANDA.jobsDefined4 WHERE PandaID=:PandaID"
                                sqlDJD = "DELETE FROM ATLAS_PANDA.jobsDefined4 WHERE PandaID=:PandaID"
                                sqlDJI = "INSERT INTO ATLAS_PANDA.jobsArchived4 (%s) " % JobSpec.columnNames()
                                sqlDJI+= JobSpec.bindValuesExpression()
                                sqlFMod = "UPDATE ATLAS_PANDA.filesTable4 SET modificationTime=:modificationTime WHERE PandaID=:PandaID"
                                sqlMMod = "UPDATE ATLAS_PANDA.metaTable SET modificationTime=:modificationTime WHERE PandaID=:PandaID"
                                sqlPMod = "UPDATE ATLAS_PANDA.jobParamsTable SET modificationTime=:modificationTime WHERE PandaID=:PandaID"
                                lostJobIDs = []
                                for tmpID, in resLost:
                                    _logger.debug("fail due to lost files : %s" % tmpID)
                                    varMap = {}
                                    varMap[':PandaID'] = tmpID
                                    self.cur.arraysize = 10
                                    self.cur.execute(sqlDJS+comment, varMap)
                                    resJob = self.cur.fetchall()
                                    if len(resJob) == 0:
                                        continue
                                    # instantiate JobSpec
                                    dJob = JobSpec()
                                    dJob.pack(resJob[0])
                                    # delete
                                    varMap = {}
                                    varMap[':PandaID'] = tmpID
                                    self.cur.execute(sqlDJD+comment, varMap)
                                    retD = self.cur.rowcount
                                    if retD == 0:
                                        continue
                                    # error code
                                    dJob.jobStatus = 'failed'
                                    dJob.endTime   = datetime.datetime.utcnow()
                                    dJob.ddmErrorCode = 101 #ErrorCode.EC_LostFile
                                    dJob.ddmErrorDiag = 'lost file in SE'
                                    dJob.modificationTime = dJob.endTime
                                    dJob.stateChangeTime  = dJob.endTime
                                    # insert
                                    self.cur.execute(sqlDJI+comment, dJob.valuesMap())
                                    # update files,metadata,parametes
                                    varMap = {}
                                    varMap[':PandaID'] = tmpID
                                    varMap[':modificationTime'] = dJob.modificationTime
                                    self.cur.execute(sqlFMod+comment,varMap)
                                    self.cur.execute(sqlMMod+comment,varMap)
                                    self.cur.execute(sqlPMod+comment,varMap)
                                    # append
                                    lostJobIDs.append(tmpID)
                                    # collect to record state change
                                    updatedJobList.append(dJob)
                                # get PandaIDs
                                varMap = {}
                                varMap[':jobStatus'] = 'assigned'
                                varMap[':dispatchDBlock'] = disName
                                self.cur.execute("SELECT PandaID FROM ATLAS_PANDA.jobsDefined4 WHERE dispatchDBlock=:dispatchDBlock AND jobStatus=:jobStatus "+comment,
                                                 varMap)
                                resDDM = self.cur.fetchall()
                                for tmpID, in resDDM:
                                    if tmpID not in lostJobIDs:
                                        ddmIDs.append(tmpID)
                                # get offset
                                ddmAttempt = job.attemptNr
                                _logger.debug("get PandaID for reassign : %s ddmAttempt=%s" % (str(ddmIDs),ddmAttempt))
                    if useCommit:
                        if not self._commit():
                            raise RuntimeError('Commit error')
                elif job.prodSourceLabel == 'ddm' and job.jobStatus == 'failed' and job.transferType=='ddm' and job.attemptNr<2 \
                         and job.commandToPilot != 'tobekilled':
                    # instantiate new mover to retry subscription
                    newJob = JobSpec()
                    newJob.jobDefinitionID   = job.jobDefinitionID
                    newJob.jobName           = job.jobName
                    newJob.attemptNr         = job.attemptNr + 1
                    newJob.transformation    = job.transformation
                    newJob.destinationDBlock = job.destinationDBlock
                    newJob.destinationSE     = job.destinationSE
                    newJob.currentPriority   = job.currentPriority
                    newJob.prodSourceLabel   = job.prodSourceLabel
                    newJob.prodUserID        = job.prodUserID
                    newJob.computingSite     = job.computingSite
                    newJob.transferType      = job.transferType
                    newJob.sourceSite        = job.sourceSite
                    newJob.destinationSite   = job.destinationSite
                    newJob.jobParameters     = job.jobParameters
                    if job.Files != []:
                        file = job.Files[0]
                        fileOL = FileSpec()
                        # add attempt nr
                        fileOL.lfn = re.sub("\.\d+$","",file.lfn)
                        fileOL.lfn = "%s.%d" % (fileOL.lfn,job.attemptNr)
                        fileOL.destinationDBlock = file.destinationDBlock
                        fileOL.destinationSE     = file.destinationSE
                        fileOL.dataset           = file.dataset
                        fileOL.type              = file.type
                        newJob.addFile(fileOL)

                # main job
                if useCommit:
                    self.conn.begin()
                oldJobSubStatus = None
                # get current status
                currentJobStatus = None
                if fromJobsDefined or fromJobsWaiting:
                    varMap = {}
                    varMap[':PandaID'] = job.PandaID
                    self.cur.execute(sql0+comment, varMap)
                    res0 = self.cur.fetchone()
                    if res0 is not None:
                        currentJobStatus, = res0
                else:
                    # lock job so that events are not dispatched during the processing
                    varMap = {}
                    varMap[':PandaID'] = job.PandaID
                    self.cur.execute(sql0+comment, varMap)
                    res0 = self.cur.fetchone()
                # check input status for ES merge
                if useJEDI and EventServiceUtils.isEventServiceMerge(job) and job.jobStatus == 'finished':
                    retInputStat = self.checkInputFileStatusInJEDI(job, useCommit=False, withLock=True)
                    _logger.debug("archiveJob : {0} checkInput for ES merge -> {1}".format(job.PandaID, retInputStat))
                    if retInputStat is None:
                        raise RuntimeError("archiveJob : {0} failed to check input".format(job.PandaID))
                    if retInputStat is False:
                        _logger.debug("archiveJob : {0} set jobStatus=failed due to inconsisten input".format(job.PandaID))
                        job.jobStatus = 'failed'
                        job.taskBufferErrorCode = ErrorCode.EC_EventServiceInconsistentIn
                        job.taskBufferErrorDiag = "inconsistent file status between Panda and JEDI"
                        for fileSpec in job.Files:
                            if fileSpec.type in ['output','log']:
                                fileSpec.status = 'failed'
                # actions for jobs without tasks
                if not useJEDI:
                    # update HS06sec for non-JEDI jobs (e.g. HC)
                    hs06sec = self.setHS06sec(job.PandaID, inActive=True)
                    _logger.debug("archiveJob : calculated hs06sec {0} for pandaID {1}".format(hs06sec, job.PandaID))
                    if hs06sec is not None:
                        job.hs06sec = hs06sec
                # actions for successful normal ES jobs
                if useJEDI and EventServiceUtils.isEventServiceJob(job) \
                        and not EventServiceUtils.isJobCloningJob(job):
                    # update some job attributes
                    hs06sec = self.setHS06sec(job.PandaID, inActive=True)
                    if hs06sec is not None:
                        job.hs06sec = hs06sec
                    # post processing
                    oldJobSubStatus = job.jobSubStatus
                    if oldJobSubStatus == 'NULL':
                        oldJobSubStatus = None
                    retEvS,retNewPandaID = self.ppEventServiceJob(job,currentJobStatus,False)
                    _logger.debug("archiveJob : {0} ppE -> {1}".format(job.PandaID, retEvS))
                    # DB error
                    if retEvS is None:
                        raise RuntimeError('Faied to retry for Event Service')
                    elif retEvS == 0:
                        # retry event ranges
                        job.jobStatus = 'merging'
                        job.jobSubStatus = 'es_retry'
                        job.taskBufferErrorCode = ErrorCode.EC_EventServiceRetried
                        job.taskBufferErrorDiag = 'closed to retry unprocessed even ranges in PandaID={0}'.format(retNewPandaID)
                    elif retEvS in [2, 10]:
                        # goes to merging
                        if retEvS == 2:
                            job.jobStatus = 'merging'
                        else:
                            job.jobStatus = 'closed'
                        job.jobSubStatus = 'es_merge'
                        job.taskBufferErrorCode = ErrorCode.EC_EventServiceMerge
                        job.taskBufferErrorDiag = 'closed to merge pre-merged files in PandaID={0}'.format(retNewPandaID)
                        # kill unused event service consumers
                        self.killUnusedEventServiceConsumers(job,False,killAll=True)
                    elif retEvS == 3:
                        # maximum attempts reached
                        job.jobStatus = 'failed'
                        job.taskBufferErrorCode = ErrorCode.EC_EventServiceMaxAttempt
                        job.taskBufferErrorDiag = 'maximum event attempts reached'
                        # kill other consumers
                        self.killEventServiceConsumers(job,False,False)
                        self.killUnusedEventServiceConsumers(job, False, killAll=True, checkAttemptNr=True)
                    elif retEvS == 4:
                        # other consumers are running
                        job.jobStatus = 'merging'
                        job.jobSubStatus = 'es_wait'
                        job.taskBufferErrorCode = ErrorCode.EC_EventServiceWaitOthers
                        job.taskBufferErrorDiag = 'no further action since other Event Service consumers were still running'
                    elif retEvS == 5:
                        # didn't process any event ranges
                        job.jobStatus = 'closed'
                        job.jobSubStatus = 'es_inaction'
                        job.taskBufferErrorCode = ErrorCode.EC_EventServiceUnprocessed
                        job.taskBufferErrorDiag = "didn't process any events on WN or reached last job attempt and take no further action"
                    elif retEvS == 6:
                        # didn't process any event ranges and last consumer
                        job.jobStatus = 'failed'
                        job.taskBufferErrorCode = ErrorCode.EC_EventServiceLastUnprocessed
                        job.taskBufferErrorDiag = "didn't process any events on WN and give up since this is the last consumer"
                    elif retEvS == 7:
                        # all event ranges failed
                        job.jobStatus = 'failed'
                        job.taskBufferErrorCode = ErrorCode.EC_EventServiceAllFailed
                        job.taskBufferErrorDiag = "all event ranges failed"
                    elif retEvS == 8:
                        # retry event ranges but no events were processed
                        job.jobStatus = 'closed'
                        job.jobSubStatus = 'es_noevent'
                        job.taskBufferErrorCode = ErrorCode.EC_EventServiceNoEvent
                        job.taskBufferErrorDiag = "didn't process any events on WN and retry unprocessed even ranges in PandaID={0}".format(retNewPandaID)
                    elif retEvS == 9:
                        # closed in bad job status
                        job.jobStatus = 'closed'
                        job.jobSubStatus = 'es_badstatus'
                        job.taskBufferErrorCode = ErrorCode.EC_EventServiceBadStatus
                        job.taskBufferErrorDiag = "cloded in bad jobStatus like defined and pending"
                    # additional actions when retry
                    codeListWithRetry = [0, 4, 5, 8, 9]
                    if retEvS in codeListWithRetry and job.computingSite != EventServiceUtils.siteIdForWaitingCoJumboJobs:
                        # resurrect consumers at other sites
                        """
                        if retEvS != 4 and EventServiceUtils.isResurrectConsumers(job.specialHandling):
                            archivedConsumers = self.getOriginalConsumers(job.jediTaskID, job.jobsetID, job.PandaID)
                            for archivedConsumer in archivedConsumers:
                                archivedConsumer.attemptNr = job.attemptNr
                                archivedConsumer.maxAttempt = job.maxAttempt
                                tmpS,tmpID = self.ppEventServiceJob(archivedConsumer,None,False)
                                _logger.debug('archiveJob : {0} tried to resurrect old consumer {1} ret={2} new={3}'.format(job.PandaID,
                                                                                                                            archivedConsumer.PandaID,
                                                                                                                            tmpS,tmpID))
                                if tmpID is not None:
                                    retNewPandaID = tmpID
                        """
                        # check jumbo flag
                        sqlJumbo = "SELECT useJumbo FROM {0}.JEDI_Tasks ".format(panda_config.schemaJEDI)
                        sqlJumbo += "WHERE jediTaskID=:jediTaskID "
                        varMap = {}
                        varMap[':jediTaskID'] = job.jediTaskID
                        self.cur.execute(sqlJumbo+comment, varMap)
                        resJumbo = self.cur.fetchone()
                        if resJumbo is not None:
                            useJumbo, = resJumbo
                        else:
                            useJumbo = None
                        _logger.debug("archiveJob : {0} useJumbo={1}".format(job.PandaID, useJumbo))
                        # no new jobs
                        if retNewPandaID is None and (retEvS != 4 or EventServiceUtils.isCoJumboJob(job) or useJumbo is not None):
                            nActiveConsumers = self.getActiveConsumers(job.jediTaskID, job.jobsetID, job.PandaID)
                            # create a fake cojumbo
                            if nActiveConsumers == 0 and retEvS in [4, 5] and (EventServiceUtils.isCoJumboJob(job) or useJumbo is not None) \
                                    and job.computingSite != EventServiceUtils.siteIdForWaitingCoJumboJobs:
                                nActiveConsumers = self.makeFakeCoJumbo(job)
                            # no ES queues for retry
                            if nActiveConsumers == 0:
                                job.jobStatus = 'failed'
                                job.taskBufferErrorCode = ErrorCode.EC_EventServiceNoEsQueues
                                job.taskBufferErrorDiag = "no ES queues available for new consumers"
                                _logger.debug('archiveJob : {0} set {1} since {2}'.format(job.PandaID,
                                                                                          job.jobStatus,
                                                                                          job.taskBufferErrorDiag))
                    # kill unused event ranges
                    if job.jobStatus == 'failed':
                        if not job.notDiscardEvents():
                            self.killUnusedEventRanges(job.jediTaskID,job.jobsetID)
                        self.updateRelatedEventServiceJobs(job,True)
                elif useJEDI and EventServiceUtils.isEventServiceJob(job) \
                        and EventServiceUtils.isJobCloningJob(job):
                    # check for cloned jobs
                    retJC = self.checkClonedJob(job,False)
                    # DB error
                    if retJC is None:
                        raise RuntimeError('Faied to take post-action for cloned job')
                    elif retJC['lock'] is True:
                        # kill other clones if the job done after locking semaphore
                        self.killEventServiceConsumers(job,False,False)
                        self.killUnusedEventServiceConsumers(job,False,killAll=True)
                    else:
                        # failed to lock semaphore
                        if retJC['last'] is False:
                            # set closed if it is not the last clone
                            job.jobStatus = 'closed'
                            job.jobSubStatus = 'jc_unlock'
                            job.taskBufferErrorCode = ErrorCode.EC_JobCloningUnlock
                            if retJC['win'] is not None:
                                job.taskBufferErrorDiag = 'closed since another clone PandaID={0} got semaphore'.format(retJC['win'])
                            else:
                                job.taskBufferErrorDiag = 'closed since failed to lock semaphore'
                # release unprocessed samples for HPO
                if job.is_hpo_workflow():
                    self.release_unprocessed_events(job.jediTaskID, job.PandaID)
                # delete from jobsDefined/Active
                varMap = {}
                varMap[':PandaID'] = job.PandaID
                if fromJobsDefined:
                    varMap[':oldJobStatus1'] = 'assigned'
                    varMap[':oldJobStatus2'] = 'defined'
                self.cur.execute(sql1+comment, varMap)
                n = self.cur.rowcount
                if n==0:
                    # already deleted
                    raise RuntimeError('PandaID={0} already deleted'.format(job.PandaID))
                else:
                    # insert
                    job.modificationTime = datetime.datetime.utcnow()
                    job.stateChangeTime  = job.modificationTime
                    if job.endTime == 'NULL':
                        job.endTime = job.modificationTime
                    self.cur.execute(sql2+comment, job.valuesMap())
                    # update files
                    for file in job.Files:
                        sqlF = ("UPDATE ATLAS_PANDA.filesTable4 SET %s" % file.bindUpdateChangesExpression()) + "WHERE row_ID=:row_ID"
                        varMap = file.valuesMap(onlyChanged=True)
                        if varMap != {}:
                            varMap[':row_ID'] = file.row_ID
                            _logger.debug(sqlF+comment+str(varMap))
                            self.cur.execute(sqlF+comment, varMap)
                    # update metadata and parameters
                    sqlFMod = "UPDATE ATLAS_PANDA.filesTable4 SET modificationTime=:modificationTime WHERE PandaID=:PandaID"
                    sqlMMod = "UPDATE ATLAS_PANDA.metaTable SET modificationTime=:modificationTime WHERE PandaID=:PandaID"
                    sqlPMod = "UPDATE ATLAS_PANDA.jobParamsTable SET modificationTime=:modificationTime WHERE PandaID=:PandaID"
                    varMap = {}
                    varMap[':PandaID'] = job.PandaID
                    varMap[':modificationTime'] = job.modificationTime
                    self.cur.execute(sqlFMod+comment,varMap)
                    self.cur.execute(sqlMMod+comment,varMap)
                    self.cur.execute(sqlPMod+comment,varMap)
                    # increment the number of failed jobs in _dis
                    myDisList = []
                    if job.jobStatus == 'failed' and job.prodSourceLabel in ['managed','test']:
                        for tmpFile in job.Files:
                            if tmpFile.type == 'input' and tmpFile.dispatchDBlock not in ['','NULL',None] \
                                   and tmpFile.dispatchDBlock not in myDisList:
                                varMap = {}
                                varMap[':name'] = tmpFile.dispatchDBlock
                                # check currentfiles
                                sqlGetCurFiles  = """SELECT /*+ BEGIN_OUTLINE_DATA """
                                sqlGetCurFiles += """INDEX_RS_ASC(@"SEL$1" "TAB"@"SEL$1" ("DATASETS"."NAME")) """
                                sqlGetCurFiles += """OUTLINE_LEAF(@"SEL$1") ALL_ROWS """
                                sqlGetCurFiles += """IGNORE_OPTIM_EMBEDDED_HINTS """
                                sqlGetCurFiles += """END_OUTLINE_DATA */ """
                                sqlGetCurFiles += "currentfiles,vuid FROM ATLAS_PANDA.Datasets tab WHERE name=:name"
                                self.cur.execute(sqlGetCurFiles+comment,varMap)
                                resCurFiles = self.cur.fetchone()
                                _logger.debug("archiveJob : %s %s" % (job.PandaID,str(resCurFiles)))
                                if resCurFiles is not None:
                                    # increment currentfiles only for the first failed job since that is enough
                                    tmpCurrentFiles,tmpVUID = resCurFiles
                                    _logger.debug("archiveJob : %s %s currentfiles=%s" % (job.PandaID,tmpFile.dispatchDBlock,tmpCurrentFiles))
                                    if tmpCurrentFiles == 0:
                                        _logger.debug("archiveJob : %s %s update currentfiles" % (job.PandaID,tmpFile.dispatchDBlock))
                                        varMap = {}
                                        varMap[':vuid'] = tmpVUID
                                        sqlFailedInDis  = 'UPDATE ATLAS_PANDA.Datasets '
                                        sqlFailedInDis += 'SET currentfiles=currentfiles+1 WHERE vuid=:vuid'
                                        self.cur.execute(sqlFailedInDis+comment,varMap)
                                myDisList.append(tmpFile.dispatchDBlock)
                    # collect to record state change
                    updatedJobList.append(job)
                    # update JEDI tables unless it is an ES consumer job which was successful but waits for merging or other running consumers
                    if useJEDI and not (EventServiceUtils.isEventServiceJob(job) and (job.isCancelled() or job.jobStatus == 'merging')):
                        self.propagateResultToJEDI(job,self.cur,extraInfo=extraInfo)
                    # update related ES jobs when ES-merge job is done
                    if useJEDI and EventServiceUtils.isEventServiceMerge(job) and job.taskBufferErrorCode not in [ErrorCode.EC_PilotRetried] \
                            and not job.isCancelled():
                        if job.jobStatus == 'failed':
                            self.updateRelatedEventServiceJobs(job,True)
                        else:
                            self.updateRelatedEventServiceJobs(job)
                # propagate successful result to unmerge job
                if useJEDI and job.processingType == 'pmerge' and job.jobStatus == 'finished':
                    self.updateUnmergedJobs(job)
                # overwrite job status
                tmpJobStatus = job.jobStatus
                sqlPRE = "SELECT /* use_json_type */ scj.data.pledgedcpu FROM ATLAS_PANDA.schedconfig_json scj WHERE scj.panda_queue=:siteID "

                sqlOJS = "UPDATE ATLAS_PANDA.jobsArchived4 SET jobStatus=:jobStatus,jobSubStatus=:jobSubStatus WHERE PandaID=:PandaID "
                if oldJobSubStatus in ['pilot_failed', 'es_heartbeat'] or \
                        oldJobSubStatus == 'pilot_killed' and job.jobSubStatus in ['es_noevent', 'es_inaction']:
                    # check if preemptable
                    isPreemptable = False
                    varMap = {}
                    varMap[':siteID'] = job.computingSite
                    self.cur.execute(sqlPRE+comment, varMap)
                    resPRE = self.cur.fetchone()
                    if resPRE is not None:
                        try:
                            if int(resPRE[0]) == -1:
                                isPreemptable = True
                        except Exception:
                            pass
                    # overwrite job status
                    varMap = {}
                    varMap[':PandaID'] = job.PandaID
                    if isPreemptable and oldJobSubStatus not in ['pilot_failed']:
                        varMap[':jobStatus'] = 'closed'
                        varMap[':jobSubStatus'] = 'es_preempted'
                    else:
                        varMap[':jobStatus'] = 'failed'
                        varMap[':jobSubStatus'] = oldJobSubStatus
                    self.cur.execute(sqlOJS+comment,varMap)
                    tmpJobStatus = varMap[':jobStatus']
                if EventServiceUtils.isEventServiceJob(job):
                    if job.jobStatus in ['failed', 'closed'] and \
                            job.taskBufferErrorCode in [ErrorCode.EC_EventServiceLastUnprocessed, ErrorCode.EC_EventServiceUnprocessed] and \
                            job.nEvents > 0:
                        varMap = {}
                        varMap[':PandaID'] = job.PandaID
                        varMap[':jobStatus'] = 'merging'
                        if oldJobSubStatus in ['es_toolong']:
                            varMap[':jobSubStatus'] = oldJobSubStatus
                        else:
                            varMap[':jobSubStatus'] = 'es_wait'
                        self.cur.execute(sqlOJS+comment,varMap)
                        tmpJobStatus = varMap[':jobStatus']
                        _logger.debug("archiveJob : %s change failed to merging" % job.PandaID)
                    elif job.jobStatus in ['failed'] and \
                            job.taskBufferErrorCode in [ErrorCode.EC_EventServiceLastUnprocessed, ErrorCode.EC_EventServiceUnprocessed] and \
                            (oldJobSubStatus in ['pilot_noevents'] or \
                                 (job.pilotErrorCode == 0 and job.ddmErrorCode == 0 and job.supErrorCode == 0 and job.jobDispatcherErrorCode == 0)):
                        varMap = {}
                        varMap[':PandaID'] = job.PandaID
                        varMap[':jobStatus'] = 'closed'
                        varMap[':jobSubStatus'] = oldJobSubStatus
                        self.cur.execute(sqlOJS+comment,varMap)
                        tmpJobStatus = varMap[':jobStatus']
                        _logger.debug("archiveJob : {0} change failed to closed for {1}".format(job.PandaID, oldJobSubStatus))
                # commit
                if useCommit:
                    if not self._commit():
                        raise RuntimeError('Commit error')
                # record status change
                try:
                    for tmpJob in updatedJobList:
                        self.recordStatusChange(tmpJob.PandaID,tmpJobStatus,jobInfo=tmpJob,useCommit=useCommit)
                        self.push_job_status_message(tmpJob, tmpJob.PandaID, tmpJobStatus)
                except Exception:
                    _logger.error('recordStatusChange in archiveJob')
                _logger.debug("archiveJob : %s done" % job.PandaID)
                return True,ddmIDs,ddmAttempt,newJob
            except Exception:
                # roll back
                if useCommit:
                    self._rollback(True)
                errtype,errvalue = sys.exc_info()[:2]
                errStr = "archiveJob %s : %s %s" % (job.PandaID,errtype,errvalue)
                errStr.strip()
                errStr += traceback.format_exc()
                _logger.error(errStr)
                if not useCommit:
                    raise RuntimeError('archiveJob failed')
                return False,[],0,None


    # finalize pending jobs
    def finalizePendingJobs(self,prodUserName,jobDefinitionID,waitLock=False):
        comment = ' /* DBProxy.finalizePendingJobs */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        tmpLog = LogWrapper(_logger, methodName + " < user={} jobdefID={} >".format(prodUserName, jobDefinitionID))
        tmpLog.debug("start")
        sql0 = "SELECT PandaID,lockedBy,jediTaskID FROM ATLAS_PANDA.jobsActive4 "
        sql0+= "WHERE prodUserName=:prodUserName AND jobDefinitionID=:jobDefinitionID "
        sql0+= "AND prodSourceLabel=:prodSourceLabel AND jobStatus=:jobStatus "
        sqlU = "UPDATE ATLAS_PANDA.jobsActive4 SET jobStatus=:newJobStatus "
        sqlU+= "WHERE PandaID=:PandaID AND jobStatus=:jobStatus "
        sql1 = "SELECT %s FROM ATLAS_PANDA.jobsActive4 " % JobSpec.columnNames()
        sql1+= "WHERE PandaID=:PandaID AND jobStatus=:jobStatus "
        sql2 = "DELETE FROM ATLAS_PANDA.jobsActive4 WHERE PandaID=:PandaID AND jobStatus=:jobStatus "
        sql3 = "INSERT INTO ATLAS_PANDA.jobsArchived4 (%s) " % JobSpec.columnNames()
        sql3+= JobSpec.bindValuesExpression()
        sqlFMod = "UPDATE ATLAS_PANDA.filesTable4 SET modificationTime=:modificationTime WHERE PandaID=:PandaID"
        sqlMMod = "UPDATE ATLAS_PANDA.metaTable SET modificationTime=:modificationTime WHERE PandaID=:PandaID"
        sqlPMod = "UPDATE ATLAS_PANDA.jobParamsTable SET modificationTime=:modificationTime WHERE PandaID=:PandaID"
        try:
            # begin transaction
            self.conn.begin()
            self.cur.arraysize = 100000
            # select
            varMap = {}
            varMap[':jobStatus']       = 'failed'
            varMap[':prodUserName']    = prodUserName
            varMap[':jobDefinitionID'] = jobDefinitionID
            varMap[':prodSourceLabel'] = 'user'
            self.cur.execute(sql0+comment,varMap)
            resPending = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # lock
            pPandaIDs = []
            lockedBy = None
            jediTaskID = None
            for pandaID,tmpLockedBy,tmpJediTaskID in resPending:
                if lockedBy is None:
                    lockedBy = tmpLockedBy
                if jediTaskID is None:
                    jediTaskID = tmpJediTaskID
                pPandaIDs.append(pandaID)
            # check if JEDI is used
            useJEDI = False
            if hasattr(panda_config,'useJEDI') and panda_config.useJEDI is True and \
                    lockedBy == 'jedi' and self.checkTaskStatusJEDI(jediTaskID,self.cur):
                useJEDI = True
            # loop over all PandaIDs
            for pandaID in pPandaIDs:
                # begin transaction
                self.conn.begin()
                # lock
                varMap = {}
                varMap[':jobStatus']    = 'failed'
                varMap[':newJobStatus'] = 'holding'
                varMap[':PandaID']      = pandaID
                self.cur.execute(sqlU+comment,varMap)
                retU = self.cur.rowcount
                if retU == 0:
                    # commit
                    if not self._commit():
                        raise RuntimeError('Commit error')
                # get job
                varMap = {}
                varMap[':PandaID']   = pandaID
                varMap[':jobStatus'] = 'holding'
                self.cur.arraysize = 10
                self.cur.execute(sql1+comment,varMap)
                res = self.cur.fetchall()
                if len(res) == 0:
                    tmpLog.debug("PandaID %s not found" % pandaID)
                    # commit
                    if not self._commit():
                        raise RuntimeError('Commit error')
                    continue
                job = JobSpec()
                job.pack(res[0])
                job.jobStatus = 'failed'
                job.modificationTime = datetime.datetime.utcnow()
                # delete
                self.cur.execute(sql2+comment,varMap)
                n = self.cur.rowcount
                if n==0:
                    # already killed
                    tmpLog.debug("Not found %s" % pandaID)
                else:
                    tmpLog.debug("finalizing %s" % pandaID)
                    # insert
                    self.cur.execute(sql3+comment,job.valuesMap())
                    # update files,metadata,parametes
                    varMap = {}
                    varMap[':PandaID'] = pandaID
                    varMap[':modificationTime'] = job.modificationTime
                    self.cur.execute(sqlFMod+comment,varMap)
                    self.cur.execute(sqlMMod+comment,varMap)
                    self.cur.execute(sqlPMod+comment,varMap)
                    # update JEDI tables
                    if useJEDI:
                        # read files
                        sqlFile = "SELECT %s FROM ATLAS_PANDA.filesTable4 " % FileSpec.columnNames()
                        sqlFile+= "WHERE PandaID=:PandaID"
                        varMap = {}
                        varMap[':PandaID'] = pandaID
                        self.cur.arraysize = 100000
                        self.cur.execute(sqlFile+comment, varMap)
                        resFs = self.cur.fetchall()
                        for resF in resFs:
                            tmpFile = FileSpec()
                            tmpFile.pack(resF)
                            job.addFile(tmpFile)
                        self.propagateResultToJEDI(job,self.cur,finishPending=True,waitLock=waitLock)
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
            tmpLog.debug("done {} jobs".format(len(pPandaIDs)))
            return True
        except Exception as e:
            # roll back
            self._rollback()
            tmpLog.error("failed with {}".format(str(e)))
            return False


    # delete stalled jobs
    def deleteStalledJobs(self,libFileName):
        comment = ' /* DBProxy.deleteStalledJobs */'
        _logger.debug("deleteStalledJobs : %s" % libFileName)
        sql2 = "INSERT INTO ATLAS_PANDA.jobsArchived4 (%s) " % JobSpec.columnNames()
        sql2+= JobSpec.bindValuesExpression()
        nTry=3
        try:
            # begin transaction
            self.conn.begin()
            # look for downstream jobs
            sqlD   = "SELECT PandaID FROM ATLAS_PANDA.filesTable4 WHERE type=:type AND lfn=:lfn GROUP BY PandaID"
            sqlDJS = "SELECT %s " % JobSpec.columnNames()
            sqlDJS+= "FROM ATLAS_PANDA.jobsDefined4 WHERE PandaID=:PandaID"
            sqlDJD = "DELETE FROM ATLAS_PANDA.jobsDefined4 WHERE PandaID=:PandaID"
            sqlDJI = "INSERT INTO ATLAS_PANDA.jobsArchived4 (%s) " % JobSpec.columnNames()
            sqlDJI+= JobSpec.bindValuesExpression()
            sqlFMod = "UPDATE ATLAS_PANDA.filesTable4 SET modificationTime=:modificationTime WHERE PandaID=:PandaID"
            sqlMMod = "UPDATE ATLAS_PANDA.metaTable SET modificationTime=:modificationTime WHERE PandaID=:PandaID"
            sqlPMod = "UPDATE ATLAS_PANDA.jobParamsTable SET modificationTime=:modificationTime WHERE PandaID=:PandaID"
            _logger.debug("deleteStalledJobs : look for downstream jobs for %s" % libFileName)
            # select PandaID
            varMap = {}
            varMap[':lfn']  = libFileName
            varMap[':type'] = 'input'
            self.cur.arraysize = 100000
            self.cur.execute(sqlD+comment, varMap)
            res = self.cur.fetchall()
            for downID, in res:
                _logger.debug("deleteStalledJobs : delete %s" % downID)
                # select jobs
                varMap = {}
                varMap[':PandaID'] = downID
                self.cur.arraysize = 10
                self.cur.execute(sqlDJS+comment, varMap)
                resJob = self.cur.fetchall()
                if len(resJob) == 0:
                    continue
                # instantiate JobSpec
                dJob = JobSpec()
                dJob.pack(resJob[0])
                # delete
                varMap = {}
                varMap[':PandaID'] = downID
                self.cur.execute(sqlDJD+comment, varMap)
                retD = self.cur.rowcount
                if retD == 0:
                    continue
                # error code
                dJob.jobStatus = 'cancelled'
                dJob.endTime   = datetime.datetime.utcnow()
                dJob.taskBufferErrorCode = ErrorCode.EC_Kill
                dJob.taskBufferErrorDiag = 'killed by Panda server : upstream job failed'
                dJob.modificationTime = dJob.endTime
                dJob.stateChangeTime  = dJob.endTime
                # insert
                self.cur.execute(sqlDJI+comment, dJob.valuesMap())
                # update files,metadata,parametes
                varMap = {}
                varMap[':PandaID'] = downID
                varMap[':modificationTime'] = dJob.modificationTime
                self.cur.execute(sqlFMod+comment,varMap)
                self.cur.execute(sqlMMod+comment,varMap)
                self.cur.execute(sqlPMod+comment,varMap)
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            return True
        except Exception:
            # roll back
            self._rollback(True)
            errtype,errvalue = sys.exc_info()[:2]
            _logger.error("deleteStalledJobs : %s %s" % (errtype,errvalue))
            return False


    # update Job status in jobsActive
    def updateJobStatus(self,pandaID,jobStatus,param,updateStateChange=False,attemptNr=None):
        comment = ' /* DBProxy.updateJobStatus */'
        _logger.debug("updateJobStatus : PandaID=%s attemptNr=%s status=%s" % (pandaID,attemptNr,jobStatus))
        sql0  = "SELECT commandToPilot,endTime,specialHandling,jobStatus,computingSite,cloud,prodSourceLabel,lockedby,jediTaskID,"
        sql0 += "jobsetID,jobDispatcherErrorDiag,supErrorCode,eventService,batchID "
        sql0 += "FROM ATLAS_PANDA.jobsActive4 WHERE PandaID=:PandaID "
        varMap0 = {}
        varMap0[':PandaID'] = pandaID
        sql1 = "UPDATE ATLAS_PANDA.jobsActive4 SET jobStatus=:jobStatus"
        varMap = {}
        presetEndTime = False
        for key in list(param):
            if key in ['corruptedFiles']:
                continue
            if param[key] is not None or key in ['jobDispatcherErrorDiag']:
                param[key] = JobSpec.truncateStringAttr(key,param[key])
                sql1 += ',%s=:%s' % (key,key)
                varMap[':%s' % key] = param[key]
                if key == 'endTime':
                    presetEndTime = True
                try:
                    # store positive error code even for pilot retry
                    if key == 'pilotErrorCode' and param[key].startswith('-'):
                        varMap[':%s' % key] = param[key][1:]
                except Exception:
                    pass
            if key == 'jobMetrics':
                # extract the memory leak from the pilot jobMetrics
                try:
                    tmpM = re.search('leak=(-?\d+\.*\d+)', param[key])
                    if tmpM is not None:
                        memoryLeak = long(float(tmpM.group(1)))
                        tmpKey = 'memory_leak'
                        sql1 += ',{0}=:{0}'.format(tmpKey)
                        varMap[':{0}'.format(tmpKey)] = memoryLeak
                except Exception:
                    pass

                # extract the chi2 measurement for the memory leak fitting
                try:
                    tmpM = re.search('chi2=(-?\d+\.*\d+)', param[key])
                    if tmpM is not None:
                        # keep measurement under 11 digits because of DB declaration
                        memory_leak_x2 = min(float(tmpM.group(1)), 10**11-1)
                        tmpKey = 'memory_leak_x2'
                        sql1 += ',{0}=:{0}'.format(tmpKey)
                        varMap[':{0}'.format(tmpKey)] = memory_leak_x2
                except Exception:
                    pass
        sql1W = " WHERE PandaID=:PandaID "
        varMap[':PandaID'] = pandaID
        if attemptNr is not None:
            sql0  += "AND attemptNr=:attemptNr "
            sql1W += "AND attemptNr=:attemptNr "
            varMap[':attemptNr'] = attemptNr
            varMap0[':attemptNr'] = attemptNr
        # prevent change from holding to transferring which doesn't register files to sub/tid
        if jobStatus == 'transferring':
            sql1W += "AND NOT jobStatus=:ngStatus "
            varMap[':ngStatus'] = 'holding'
        updatedFlag = False
        nTry=1
        for iTry in range(nTry):
            try:
                # begin transaction
                self.conn.begin()
                # select
                self.cur.arraysize = 10
                self.cur.execute (sql0+comment,varMap0)
                res = self.cur.fetchone()
                if res is not None:
                    ret = ''
                    commandToPilot,endTime,specialHandling,oldJobStatus,computingSite,cloud,prodSourceLabel,\
                        lockedby,jediTaskID,jobsetID,jobDispatcherErrorDiag,supErrorCode,eventService,batchID \
                        = res
                    # debug mode
                    if specialHandling not in [None,''] and 'debug' in specialHandling:
                        ret += 'debug,'
                    # FIXME
                    #else:
                    #    ret += 'debugoff,'
                    # kill command
                    if commandToPilot not in [None,'']:
                        # soft kill
                        if supErrorCode in [ErrorCode.EC_EventServicePreemption]:
                            #commandToPilot = 'softkill'
                            pass
                        ret += '%s,' % commandToPilot
                    ret = ret[:-1]
                    # convert empty to NULL
                    if ret == '':
                        ret = 'NULL'
                    if oldJobStatus == 'failed' and jobStatus in ['holding','transferring','starting','running']:
                        _logger.debug("updateJobStatus : PandaID=%s skip to set %s since it is already %s" \
                                          % (pandaID,jobStatus,oldJobStatus))
                        ret = 'alreadydone'
                    elif oldJobStatus == 'transferring' and jobStatus == 'holding' and jobDispatcherErrorDiag in [None,'']:
                        # skip transferring -> holding
                        _logger.debug("updateJobStatus : PandaID=%s skip to set holding since it is alredy in transferring" \
                                          % pandaID)
                        ret = 'alreadydone'
                    elif oldJobStatus == 'holding' and jobStatus == 'holding' and ('jobDispatcherErrorDiag' not in param or param['jobDispatcherErrorDiag'] not in [None,'']):
                        # just ignore hearbeats for job recovery
                        _logger.debug("updateJobStatus : PandaID=%s skip to reset holding" % pandaID)
                    elif oldJobStatus == 'holding' and jobStatus == 'holding' and jobDispatcherErrorDiag in [None,''] \
                            and 'jobDispatcherErrorDiag' in param and param['jobDispatcherErrorDiag'] in [None,'']:
                        # special return to avoid duplicated XMLs
                        _logger.debug("updateJobStatus : PandaID=%s skip to set holding since it was already set to holding by the final heartbeat" \
                                          % pandaID)
                        ret = 'alreadydone'
                    elif oldJobStatus == 'merging':
                        # don't update merging
                        _logger.debug("updateJobStatus : PandaID=%s skip to change from merging" % pandaID)
                    elif oldJobStatus in ['holding', 'transferring'] and jobStatus == 'starting':
                        # don't update holding
                        _logger.debug("updateJobStatus : PandaID={0} skip to change {1} to {2} to avoid inconsistency".format(pandaID, oldJobStatus, jobStatus))
                    elif oldJobStatus == 'holding' and jobStatus == 'running':
                        # don't update holding
                        _logger.debug("updateJobStatus : PandaID={0} skip to change {1} to {2} not to return to active".format(pandaID, oldJobStatus, jobStatus))
                    elif batchID not in ['', None] and 'batchID' in param and param['batchID'] not in ['', None] \
                            and batchID != param['batchID'] and re.search('^\d+\.*\d+$', batchID) is None \
                            and re.search('^\d+\.*\d+$', param['batchID']) is None:
                        # invalid batchID
                        _logger.debug("updateJobStatus : to be killed since PandaID={0} batchID mismatch old {1} in {2} vs new {3} in {4}".format(pandaID,
                                                                                                                                                  batchID.replace('\n',''),
                                                                                                                                                  oldJobStatus,
                                                                                                                                                  param['batchID'].replace('\n',''),
                                                                                                                                                  jobStatus))
                        ret = 'tobekilled'
                        # set supErrorCode and supErrorDiag
                        varMap = {}
                        varMap[':PandaID'] = pandaID
                        varMap[':code'] = SupErrors.error_codes['INVALID_BATCH_ID']
                        varMap[':diag'] = 'got an update request with invalid batchID={0}'.format(param['batchID'].replace('\n',''))
                        varMap[':diag'] = JobSpec.truncateStringAttr('supErrorDiag', varMap[':diag'])
                        sqlSUP = "UPDATE ATLAS_PANDA.jobsActive4 SET supErrorCode=:code,supErrorDiag=:diag "
                        sqlSUP += "WHERE PandaID=:PandaID "
                        self.cur.execute (sqlSUP+comment, varMap)
                    else:
                        # change starting to running
                        if oldJobStatus == 'running' and jobStatus == 'starting':
                            _logger.debug("updateJobStatus : PandaID={0} changed to {1} from {2} to avoid inconsistency".format(pandaID, oldJobStatus, jobStatus))
                            jobStatus = oldJobStatus
                        # update stateChangeTime
                        if updateStateChange or (jobStatus != oldJobStatus):
                            sql1 += ",stateChangeTime=CURRENT_DATE"
                        # set endTime if undefined for holding
                        if (jobStatus == 'holding' or (jobStatus == 'transferring' and oldJobStatus == 'running')) \
                            and endTime is None and not presetEndTime:
                            sql1 += ',endTime=CURRENT_DATE '
                        # update startTime
                        if oldJobStatus in ['sent', 'starting'] and jobStatus == 'running' and \
                                ':startTime' not in varMap:
                            sql1 += ",startTime=CURRENT_DATE"
                        # update modification time
                        sql1 += ",modificationTime=CURRENT_DATE"
                        # update
                        varMap[':jobStatus'] = jobStatus
                        self.cur.execute (sql1+sql1W+comment,varMap)
                        nUp = self.cur.rowcount
                        _logger.debug("updateJobStatus : PandaID={0} attemptNr={1} nUp={2} old={3} new={4}".format(
                            pandaID, attemptNr, nUp, oldJobStatus, jobStatus))
                        if nUp == 1:
                            updatedFlag = True
                        if nUp == 0 and jobStatus == 'transferring':
                            _logger.debug("updateJobStatus : PandaID=%s ignore to update for transferring" % pandaID)
                        # update waiting ES jobs not to get reassigned
                        if updatedFlag and EventServiceUtils.isEventServiceSH(specialHandling):
                            # sql to update ES jobs
                            sqlUEA = "SELECT PandaID FROM ATLAS_PANDA.jobsActive4 "
                            sqlUEA += "WHERE jediTaskID=:jediTaskID AND jobsetID=:jobsetID AND jobStatus=:jobStatus "
                            sqlUEL = "SELECT modificationTime FROM ATLAS_PANDA.jobsActive4 WHERE PandaID=:PandaID "
                            sqlUEL += "FOR UPDATE NOWAIT "
                            sqlUE  = "UPDATE ATLAS_PANDA.jobsActive4 SET modificationTime=CURRENT_DATE "
                            sqlUE += "WHERE PandaID=:PandaID "
                            varMap = {}
                            varMap[':jediTaskID'] = jediTaskID
                            varMap[':jobsetID']   = jobsetID
                            varMap[':jobStatus']  = 'activated'
                            self.cur.execute(sqlUEA+comment,varMap)
                            resUEA = self.cur.fetchall()
                            nUE = 0
                            for ueaPandaID, in resUEA:
                                varMap = {}
                                varMap[':PandaID'] = ueaPandaID
                                try:
                                    # lock with NOWAIT
                                    self.cur.execute(sqlUEL+comment,varMap)
                                    resUEL = self.cur.fetchone()
                                    if resUEL is None:
                                        continue
                                except Exception:
                                    _logger.debug("updateJobStatus : PandaID=%s skip to update associated ES=%s" % (pandaID, ueaPandaID))
                                    continue
                                self.cur.execute(sqlUE+comment,varMap)
                                nUE += self.cur.rowcount
                            _logger.debug("updateJobStatus : PandaID=%s updated %s ES jobs" % (pandaID,nUE))
                        # update fake co-jumbo jobs
                        if updatedFlag and eventService == EventServiceUtils.jumboJobFlagNumber:
                            # sql to update fake co-jumbo
                            sqlIFL = "SELECT PandaID FROM ATLAS_PANDA.jobsWaiting4 "
                            sqlIFL += "WHERE jediTaskID=:jediTaskID AND eventService=:eventService AND jobStatus=:jobStatus "
                            sqlIFL += "FOR UPDATE NOWAIT "
                            sqlIF = "UPDATE ATLAS_PANDA.jobsWaiting4 SET modificationTime=CURRENT_DATE "
                            sqlIF += "WHERE jediTaskID=:jediTaskID AND eventService=:eventService AND jobStatus=:jobStatus "
                            varMap = {}
                            varMap[':jediTaskID'] = jediTaskID
                            varMap[':eventService'] = EventServiceUtils.coJumboJobFlagNumber
                            varMap[':jobStatus']  = 'waiting'
                            try:
                                # lock with NOWAIT
                                self.cur.execute(sqlIFL+comment,varMap)
                                resIFL = self.cur.fetchall()
                                self.cur.execute (sqlIF+comment,varMap)
                                nUE = self.cur.rowcount
                                _logger.debug("updateJobStatus : PandaID=%s updated %s fake co-jumbo jobs" % (pandaID,nUE))
                            except Exception:
                                _logger.debug("updateJobStatus : PandaID=%s skip to update fake co-jumbo jobs" % pandaID)
                        # update nFilesOnHold for JEDI RW calculation
                        if updatedFlag and jobStatus == 'transferring' and oldJobStatus == 'holding' and \
                                hasattr(panda_config,'useJEDI') and panda_config.useJEDI is True and \
                                lockedby == 'jedi' and self.checkTaskStatusJEDI(jediTaskID,self.cur):
                            # SQL to get file list from Panda
                            sqlJediFP  = "SELECT datasetID,fileID,attemptNr FROM ATLAS_PANDA.filesTable4 "
                            sqlJediFP += "WHERE PandaID=:pandaID AND type IN (:type1,:type2) ORDER BY datasetID,fileID "
                            # SQL to check JEDI files
                            sqlJediFJ  = "SELECT /*+ INDEX_RS_ASC(JEDI_DATASET_CONTENTS (JEDI_DATASET_CONTENTS.JEDITASKID JEDI_DATASET_CONTENTS.DATASETID JEDI_DATASET_CONTENTS.FILEID)) */ 1 FROM ATLAS_PANDA.JEDI_Dataset_Contents "
                            sqlJediFJ += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                            sqlJediFJ += "AND attemptNr=:attemptNr AND status=:status AND keepTrack=:keepTrack "
                            # get file list
                            varMap = {}
                            varMap[':pandaID'] = pandaID
                            varMap[':type1'] = 'input'
                            varMap[':type2'] = 'pseudo_input'
                            self.cur.arraysize = 100000
                            self.cur.execute(sqlJediFP+comment, varMap)
                            resJediFile = self.cur.fetchall()
                            datasetContentsStat = {}
                            # loop over all files
                            for tmpDatasetID,tmpFileID,tmpAttemptNr in resJediFile:
                                # check file in JEDI
                                varMap = {}
                                varMap[':jediTaskID'] = jediTaskID
                                varMap[':datasetID']  = tmpDatasetID
                                varMap[':fileID']     = tmpFileID
                                varMap[':attemptNr']  = tmpAttemptNr
                                varMap[':status']     = 'running'
                                varMap[':keepTrack']  = 1
                                self.cur.execute(sqlJediFJ+comment, varMap)
                                res = self.cur.fetchone()
                                if res is not None:
                                    if tmpDatasetID not in datasetContentsStat:
                                        datasetContentsStat[tmpDatasetID] = 0
                                    if jobStatus == 'transferring':
                                        # increment nOnHold
                                        datasetContentsStat[tmpDatasetID] += 1
                                    else:
                                        # decrement nOnHold
                                        datasetContentsStat[tmpDatasetID] -= 1
                            # loop over all datasets
                            tmpDatasetIDs = list(datasetContentsStat)
                            tmpDatasetIDs.sort()
                            for tmpDatasetID in tmpDatasetIDs:
                                diffNum = datasetContentsStat[tmpDatasetID]
                                # no difference
                                if diffNum == 0:
                                    continue
                                # SQL to lock
                                sqlJediDL  = "SELECT nFilesOnHold FROM ATLAS_PANDA.JEDI_Datasets "
                                sqlJediDL += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID "
                                sqlJediDL += "FOR UPDATE NOWAIT "
                                varMap = {}
                                varMap[':jediTaskID'] = jediTaskID
                                varMap[':datasetID']  = tmpDatasetID
                                _logger.debug(sqlJediDL+comment+str(varMap))
                                self.cur.execute(sqlJediDL+comment, varMap)
                                # SQL to update
                                sqlJediDU  = "UPDATE ATLAS_PANDA.JEDI_Datasets SET "
                                if diffNum > 0:
                                    sqlJediDU += "nFilesOnHold=nFilesOnHold+:diffNum "
                                else:
                                    sqlJediDU += "nFilesOnHold=nFilesOnHold-:diffNum "
                                sqlJediDU += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID "
                                sqlJediDU += "AND NOT type IN (:ngType1,:ngType2) "
                                varMap = {}
                                varMap[':jediTaskID'] = jediTaskID
                                varMap[':datasetID']  = tmpDatasetID
                                varMap[':diffNum']    = abs(diffNum)
                                varMap[':ngType1']    = 'trn_log'
                                varMap[':ngType2']    = 'trn_output'
                                _logger.debug(sqlJediDU+comment+str(varMap))
                                self.cur.execute(sqlJediDU+comment, varMap)
                        # update lastStart
                        if oldJobStatus in ('starting','sent') and jobStatus=='running':
                            sqlLS  = "UPDATE ATLAS_PANDAMETA.siteData SET lastStart=CURRENT_DATE "
                            sqlLS += "WHERE site=:site AND hours=:hours AND flag IN (:flag1,:flag2) "
                            varMap = {}
                            varMap[':site'] = computingSite
                            varMap[':hours'] = 3
                            varMap[':flag1'] = 'production'
                            varMap[':flag2'] = 'analysis'
                            self.cur.execute(sqlLS+comment, varMap)
                            _logger.debug("updateJobStatus : PandaID=%s attemptNr=%s updated lastStart" % (pandaID,attemptNr))
                        # update input
                        if updatedFlag and jediTaskID is not None and jobStatus == 'running' and oldJobStatus != jobStatus:
                            self.updateInputStatusJedi(jediTaskID, pandaID, jobStatus)
                        # register corrupted zip files
                        if updatedFlag and 'corruptedFiles' in param and eventService == EventServiceUtils.esMergeJobFlagNumber:
                            # get exiting files
                            sqlCorF = "SELECT lfn FROM ATLAS_PANDA.filesTable4 WHERE PandaID=:PandaID AND type=:type "
                            varMap = {}
                            varMap[':PandaID'] = pandaID
                            varMap[':type'] = "zipinput"
                            self.cur.execute(sqlCorF+comment, varMap)
                            resCorF = self.cur.fetchall()
                            exCorFiles = set()
                            for tmpLFN, in resCorF:
                                exCorFiles.add(tmpLFN)
                            # register files
                            tmpJobSpec = JobSpec()
                            tmpJobSpec.PandaID = pandaID
                            sqlCorIN = "INSERT INTO ATLAS_PANDA.filesTable4 ({0}) ".format(FileSpec.columnNames())
                            sqlCorIN += FileSpec.bindValuesExpression(useSeq=True)
                            for tmpLFN in param['corruptedFiles'].split(','):
                                tmpLFN = tmpLFN.strip()
                                if tmpLFN in exCorFiles or tmpLFN == '':
                                    continue
                                tmpFileSpec = FileSpec()
                                tmpFileSpec.jediTaskID = jediTaskID
                                tmpFileSpec.fsize = 0
                                tmpFileSpec.lfn = tmpLFN
                                tmpFileSpec.type = 'zipinput'
                                tmpFileSpec.status = 'corrupted'
                                tmpJobSpec.addFile(tmpFileSpec)
                                varMap = tmpFileSpec.valuesMap(useSeq=True)
                                self.cur.execute(sqlCorIN+comment, varMap)
                        # try to update the lastupdate column in the harvester_rel_job_worker table to propagate
                        # changes to Elastic Search
                        sqlJWU = "UPDATE ATLAS_PANDA.Harvester_Rel_Jobs_Workers SET lastUpdate=:lastUpdate "
                        sqlJWU += "WHERE PandaID=:PandaID "
                        varMap = {':PandaID': pandaID, ':lastUpdate': datetime.datetime.utcnow()}
                        self.cur.execute(sqlJWU + comment, varMap)
                        nRow = self.cur.rowcount
                        _logger.debug("updateJobStatus : {0} workers updated for pandaID {1}".format(nRow, pandaID))

                        try:
                            # try to update the computing element from the harvester worker table
                            sql_ce = """
                                     UPDATE ATLAS_PANDA.jobsActive4
                                     SET computingelement = (SELECT * FROM (
                                       SELECT computingelement FROM ATLAS_PANDA.harvester_workers hw, ATLAS_PANDA.Harvester_Rel_Jobs_Workers hrjw
                                       WHERE hw.workerid = hrjw.workerid AND hw.harvesterid = hrjw.harvesterid AND hrjw.pandaid = :PandaID ORDER BY hw.workerid DESC
                                       ) WHERE rownum=1)
                                     where PandaID=:PandaID
                                     """
                            varMap = {':PandaID': pandaID}
                            self.cur.execute(sql_ce + comment, varMap)
                            nRow = self.cur.rowcount
                            _logger.debug("updateJobStatus : succeeded to update CE from harvester table for pandaID {0} (rowcount={1})".format(pandaID, nRow))
                        except Exception:
                            _logger.error("updateJobStatus : failed to update CE from harvester table with {0} for PanDAID {1}".format(traceback.format_exc(), pandaID))
                    # push status change
                    self.push_job_status_message(None, pandaID, jobStatus, jediTaskID, specialHandling)
                else:
                    _logger.debug("updateJobStatus : PandaID=%s attemptNr=%s notFound" % (pandaID,attemptNr))
                    # already deleted or bad attempt number
                    ret = 'tobekilled'
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                # record status change
                try:
                    if updatedFlag and oldJobStatus is not None and oldJobStatus != jobStatus:
                        self.recordStatusChange(pandaID,jobStatus,
                                                infoMap={'computingSite':computingSite,
                                                         'cloud':cloud,
                                                         'prodSourceLabel':prodSourceLabel})
                except Exception:
                    _logger.error('recordStatusChange in updateJobStatus')
                _logger.debug("updateJobStatus : PandaID=%s done" % pandaID)
                return ret
            except Exception:
                # roll back
                self._rollback(True)
                if iTry+1 < nTry:
                    _logger.debug("updateJobStatus : %s retry : %s" % (pandaID,iTry))
                    time.sleep(random.randint(10,20))
                    continue
                type, value, traceBack = sys.exc_info()
                _logger.error("updateJobStatus : %s %s %s" % (type, value, traceback.format_exc()))
                _logger.error("updateJobStatus : %s" % pandaID)
                return False


    # update job information in jobsActive or jobsDefined
    def updateJob(self,job,inJobsDefined,oldJobStatus=None,extraInfo=None):
        comment = ' /* DBProxy.updateJob */'
        _logger.debug("updateJob : %s" % job.PandaID)
        updatedFlag = False
        nTry=3
        for iTry in range(nTry):
            try:
                job.modificationTime = datetime.datetime.utcnow()
                # set stateChangeTime for defined->assigned
                if inJobsDefined:
                    job.stateChangeTime = job.modificationTime
                # make SQL
                if inJobsDefined:
                    sql1 = "UPDATE ATLAS_PANDA.jobsDefined4 SET %s " % job.bindUpdateChangesExpression()
                else:
                    sql1 = "UPDATE ATLAS_PANDA.jobsActive4 SET %s " % job.bindUpdateChangesExpression()
                sql1+= "WHERE PandaID=:PandaID "
                if inJobsDefined:
                    sql1+= " AND (jobStatus=:oldJobStatus1 OR jobStatus=:oldJobStatus2) "
                # begin transaction
                self.conn.begin()
                # update
                varMap = job.valuesMap(onlyChanged=True)
                varMap[':PandaID'] = job.PandaID
                if inJobsDefined:
                    varMap[':oldJobStatus1'] = 'assigned'
                    varMap[':oldJobStatus2'] = 'defined'
                _logger.debug(sql1+comment+str(varMap))
                self.cur.execute(sql1+comment, varMap)
                n = self.cur.rowcount
                if n==0:
                    # already killed or activated
                    _logger.debug("updateJob : Not found %s" % job.PandaID)
                else:
                    # check if JEDI is used
                    useJEDI = False
                    if oldJobStatus != job.jobStatus and (job.jobStatus in ['transferring','merging'] or \
                                                              oldJobStatus in ['transferring','merging']) and \
                            hasattr(panda_config,'useJEDI') and panda_config.useJEDI is True and \
                            job.lockedby == 'jedi' and self.checkTaskStatusJEDI(job.jediTaskID,self.cur):
                        useJEDI = True
                    # SQL to check JEDI files
                    sqlJediFJ  = "SELECT /*+ INDEX_RS_ASC(JEDI_DATASET_CONTENTS (JEDI_DATASET_CONTENTS.JEDITASKID JEDI_DATASET_CONTENTS.DATASETID JEDI_DATASET_CONTENTS.FILEID)) */ 1 FROM ATLAS_PANDA.JEDI_Dataset_Contents "
                    sqlJediFJ += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                    sqlJediFJ += "AND attemptNr=:attemptNr AND status=:status AND keepTrack=:keepTrack "
                    datasetContentsStat = {}
                    # loop over all files
                    for file in job.Files:
                        sqlF = ("UPDATE ATLAS_PANDA.filesTable4 SET %s" % file.bindUpdateChangesExpression()) + "WHERE row_ID=:row_ID"
                        varMap = file.valuesMap(onlyChanged=True)
                        if varMap != {}:
                            varMap[':row_ID'] = file.row_ID
                            _logger.debug(sqlF+comment+str(varMap))
                            self.cur.execute(sqlF+comment, varMap)
                        # actions for JEDI
                        if useJEDI and (job.jobStatus == 'transferring' or oldJobStatus == 'transferring') and \
                                file.type in ['input','pseudo_input'] and job.processingType != 'pmerge':
                            # check file in JEDI
                            varMap = {}
                            varMap[':jediTaskID'] = file.jediTaskID
                            varMap[':datasetID']  = file.datasetID
                            varMap[':fileID']     = file.fileID
                            varMap[':attemptNr']  = file.attemptNr
                            varMap[':status']     = 'running'
                            varMap[':keepTrack']  = 1
                            self.cur.execute(sqlJediFJ+comment, varMap)
                            res = self.cur.fetchone()
                            if res is not None:
                                if file.datasetID not in datasetContentsStat:
                                    datasetContentsStat[file.datasetID] = {'diff':0,'cType':'hold'}
                                if job.jobStatus == 'transferring':
                                    # increment nOnHold
                                    datasetContentsStat[file.datasetID]['diff'] += 1
                                else:
                                    # decrement nOnHold
                                    datasetContentsStat[file.datasetID]['diff'] -= 1
                        elif useJEDI and job.jobStatus == 'merging' and file.type in ['log','output'] and \
                                file.status != 'nooutput':
                            # SQL to update JEDI files
                            varMap = {}
                            varMap[':fileID']     = file.fileID
                            varMap[':attemptNr']  = file.attemptNr
                            varMap[':datasetID']  = file.datasetID
                            varMap[':keepTrack']  = 1
                            varMap[':jediTaskID'] = file.jediTaskID
                            varMap[':status']     = 'ready'
                            varMap[':boundaryID'] = job.PandaID
                            varMap[':maxAttempt'] = file.attemptNr + 3
                            sqlJFile =  "UPDATE ATLAS_PANDA.JEDI_Dataset_Contents "
                            sqlJFile += "SET status=:status,boundaryID=:boundaryID,maxAttempt=:maxAttempt"
                            for tmpKey in ['lfn','GUID','fsize','checksum']:
                                tmpVal = getattr(file,tmpKey)
                                if tmpVal == 'NULL':
                                    if tmpKey in file._zeroAttrs:
                                        tmpVal = 0
                                    else:
                                        tmpVal = None
                                tmpMapKey = ':%s' % tmpKey
                                sqlJFile += ",%s=%s" % (tmpKey,tmpMapKey)
                                varMap[tmpMapKey] = tmpVal
                            sqlJFile += " WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                            sqlJFile += "AND attemptNr=:attemptNr AND keepTrack=:keepTrack "
                            # update JEDI file
                            _logger.debug(sqlJFile+comment+str(varMap))
                            self.cur.execute(sqlJFile+comment,varMap)
                            nRow = self.cur.rowcount
                            if nRow == 1:
                                if file.datasetID not in datasetContentsStat:
                                    datasetContentsStat[file.datasetID] = {'diff':0,'cType':'hold'}
                                datasetContentsStat[file.datasetID]['diff'] += 1
                        # update metadata in JEDI
                        if useJEDI and file.type in ['output','log'] and extraInfo is not None:
                            varMap = {}
                            sqlFileMeta = ''
                            if 'nevents' in extraInfo and file.lfn in extraInfo['nevents']:
                                tmpKey = 'nEvents'
                                tmpMapKey = ':%s' % tmpKey
                                sqlFileMeta += "%s=%s," % (tmpKey,tmpMapKey)
                                varMap[tmpMapKey] = extraInfo['nevents'][file.lfn]
                            if 'lbnr' in extraInfo and file.lfn in extraInfo['lbnr']:
                                tmpKey = 'lumiBlockNr'
                                tmpMapKey = ':%s' % tmpKey
                                sqlFileMeta += "%s=%s," % (tmpKey,tmpMapKey)
                                varMap[tmpMapKey] = extraInfo['lbnr'][file.lfn]
                            if varMap != {}:
                                # update
                                varMap[':fileID']     = file.fileID
                                varMap[':attemptNr']  = file.attemptNr
                                varMap[':datasetID']  = file.datasetID
                                varMap[':jediTaskID'] = file.jediTaskID
                                varMap[':keepTrack']  = 1
                                sqlFileMeta = "UPDATE ATLAS_PANDA.JEDI_Dataset_Contents SET " + sqlFileMeta
                                sqlFileMeta = sqlFileMeta[:-1]
                                sqlFileMeta += " "
                                sqlFileMeta += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                                sqlFileMeta += "AND attemptNr=:attemptNr AND keepTrack=:keepTrack "
                                _logger.debug(sqlFileMeta+comment+str(varMap))
                                self.cur.execute(sqlFileMeta+comment,varMap)
                    # loop over all JEDI datasets
                    tmpDatasetIDs = list(datasetContentsStat)
                    tmpDatasetIDs.sort()
                    for tmpDatasetID in tmpDatasetIDs:
                        valMap = datasetContentsStat[tmpDatasetID]
                        diffNum = valMap['diff']
                        cType = valMap['cType']
                        # no difference
                        if diffNum == 0:
                            continue
                        # SQL to check lock
                        varMap = {}
                        varMap[':jediTaskID'] = job.jediTaskID
                        varMap[':datasetID']  = tmpDatasetID
                        sqlJediCL  = "SELECT nFilesTobeUsed,nFilesOnHold,status FROM ATLAS_PANDA.JEDI_Datasets "
                        sqlJediCL += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID "
                        sqlJediCL += "FOR UPDATE NOWAIT "
                        _logger.debug(sqlJediCL+comment+str(varMap))
                        self.cur.execute(sqlJediCL+comment, varMap)
                        # SQL to update dataset
                        varMap = {}
                        varMap[':jediTaskID'] = job.jediTaskID
                        varMap[':datasetID']  = tmpDatasetID
                        varMap[':diffNum']    = abs(diffNum)
                        sqlJediDU  = "UPDATE ATLAS_PANDA.JEDI_Datasets SET "
                        if cType == 'hold':
                            if diffNum > 0:
                                sqlJediDU += "nFilesOnHold=nFilesOnHold+:diffNum "
                            else:
                                sqlJediDU += "nFilesOnHold=nFilesOnHold-:diffNum "
                        elif cType == 'touse':
                            varMap[':status'] = 'ready'
                            sqlJediDU += "nFilesTobeUsed=nFilesTobeUsed+:diffNum,status=:status "
                        sqlJediDU += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID "
                        _logger.debug(sqlJediDU+comment+str(varMap))
                        self.cur.execute(sqlJediDU+comment, varMap)
                    # update job parameters
                    sqlJobP = "UPDATE ATLAS_PANDA.jobParamsTable SET jobParameters=:param WHERE PandaID=:PandaID"
                    varMap = {}
                    varMap[':PandaID'] = job.PandaID
                    varMap[':param']   = job.jobParameters
                    self.cur.execute(sqlJobP+comment, varMap)
                    updatedFlag = True
                    # update input
                    if useJEDI and job.jobStatus in ['transferring']:
                        self.updateInputStatusJedi(job.jediTaskID, job.PandaID, job.jobStatus)
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                # record status change
                try:
                    if updatedFlag:
                        self.recordStatusChange(job.PandaID,job.jobStatus,jobInfo=job)
                        self.push_job_status_message(job, job.PandaID, job.jobStatus)
                except Exception:
                    _logger.error('recordStatusChange in updateJob')
                return True
            except Exception:
                # roll back
                self._rollback(True)
                if iTry+1 < nTry:
                    _logger.debug("updateJob : %s retry : %s" % (job.PandaID,iTry))
                    time.sleep(random.randint(3,10))
                    continue
                type, value, traceBack = sys.exc_info()
                _logger.error("updateJob : %s %s" % (type,value))
                return False

    # update the worker status as seen by the pilot
    def updateWorkerPilotStatus(self, workerID, harvesterID, status):
        comment = ' /* DBProxy.updateWorkerPilotStatus */'
        method_name = comment.split(' ')[-2].split('.')[-1]
        tmp_logger = LogWrapper(_logger, method_name + '< harvesterID={0} workerID={1} >'.format(harvesterID, workerID))

        timestamp_utc = datetime.datetime.utcnow()
        var_map = {':status': status, ':harvesterID': harvesterID, ':workerID': workerID}
        sql = "UPDATE ATLAS_PANDA.harvester_workers SET pilotStatus=:status "

        tmp_logger.debug('Updating to status={0} at {1}'.format(status, timestamp_utc))

        # add the start or end time
        if status == 'started':
            sql += ", pilotStartTime=:now "
            var_map[':now'] = timestamp_utc
        elif status == 'finished':
            sql += ", pilotEndTime=:now "
            var_map[':now'] = timestamp_utc

        sql += "WHERE workerID=:workerID AND harvesterID=:harvesterID "

        try:
            self.conn.begin()
            self.cur.execute(sql+comment, var_map)
            if not self._commit():
                raise RuntimeError('Commit error')
            tmp_logger.debug('Updated successfully')
            return True

        except Exception:
            # roll back
            self._rollback(True)
            err_type, err_value = sys.exc_info()[:2]
            tmp_logger.error("updateWorkerPilotStatus : %s %s" % (err_type, err_value))
            return False


    # retry analysis job
    def retryJob(self,pandaID,param,failedInActive=False,changeJobInMem=False,inMemJob=None,
                 getNewPandaID=False,attemptNr=None,recoverableEsMerge=False):
        comment = ' /* DBProxy.retryJob */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " < PandaID={0} >".format(pandaID)
        tmpLog = LogWrapper(_logger,methodName)
        tmpLog.debug("inActive=%s" % failedInActive)
        sql1 = "SELECT %s FROM ATLAS_PANDA.jobsActive4 " % JobSpec.columnNames()
        sql1+= "WHERE PandaID=:PandaID "
        if failedInActive:
            sql1+= "AND jobStatus=:jobStatus "
        updatedFlag = False
        nTry=3
        for iTry in range(nTry):
            try:
                retValue = False
                if not changeJobInMem:
                    # begin transaction
                    self.conn.begin()
                    # select
                    varMap = {}
                    varMap[':PandaID'] = pandaID
                    if failedInActive:
                        varMap[':jobStatus'] = 'failed'
                    self.cur.arraysize = 10
                    self.cur.execute(sql1+comment, varMap)
                    res = self.cur.fetchall()
                    if len(res) == 0:
                        tmpLog.debug("PandaID not found")
                        self._rollback()
                        return retValue
                    job = JobSpec()
                    job.pack(res[0])
                else:
                    job = inMemJob
                # don't use getNewPandaID for buildJob since the order of PandaIDs is broken
                if getNewPandaID and job.prodSourceLabel in ['panda']:
                    if not changeJobInMem:
                        # commit
                        if not self._commit():
                            raise RuntimeError('Commit error')
                    # return
                    return retValue
                # convert attemptNr to int
                try:
                    attemptNr = int(attemptNr)
                except Exception:
                    tmpLog.debug("attemptNr=%s non-integer" % attemptNr)
                    attemptNr = -999
                # check attemptNr
                if attemptNr is not None:
                    if job.attemptNr != attemptNr:
                        tmpLog.debug("bad attemptNr job.%s != pilot.%s" % (job.attemptNr,attemptNr))
                        if not changeJobInMem:
                            # commit
                            if not self._commit():
                                raise RuntimeError('Commit error')
                        # return
                        return retValue
                # check if already retried
                if job.taskBufferErrorCode in [ErrorCode.EC_Reassigned,ErrorCode.EC_Retried,ErrorCode.EC_PilotRetried]:
                    tmpLog.debug("already retried %s" % job.taskBufferErrorCode)
                    if not changeJobInMem:
                        # commit
                        if not self._commit():
                            raise RuntimeError('Commit error')
                    # return
                    return retValue
                # use JEDI
                useJEDI = False
                if hasattr(panda_config,'useJEDI') and panda_config.useJEDI is True and \
                        job.lockedby == 'jedi' and self.checkTaskStatusJEDI(job.jediTaskID,self.cur):
                    useJEDI = True
                # check pilot retry
                usePilotRetry = False
                if job.prodSourceLabel in ['user','panda'] + JobUtils.list_ptest_prod_sources and \
                   'pilotErrorCode' in param and \
                   param['pilotErrorCode'].startswith('-') and \
                   job.maxAttempt > job.attemptNr and \
                   (not job.processingType.startswith('gangarobot') or job.processingType=='gangarobot-rctest') and \
                   not job.processingType.startswith('hammercloud'):
                    usePilotRetry = True
                # retry for ES merge
                if recoverableEsMerge and EventServiceUtils.isEventServiceMerge(job) and job.maxAttempt > job.attemptNr:
                    usePilotRetry = True
                # check if it's analysis job # FIXME once pilot retry works correctly the conditions below will be cleaned up
                if (((job.prodSourceLabel == 'user' or job.prodSourceLabel == 'panda') \
                     and not job.processingType.startswith('gangarobot') \
                     and not job.processingType.startswith('hammercloud') \
                     and 'pilotErrorCode' in param \
                     and param['pilotErrorCode'] in ['1200','1201','1213'] and (not job.computingSite.startswith('ANALY_LONG_')) \
                     and job.attemptNr < 2) or (job.prodSourceLabel == 'ddm' and job.cloud == 'CA' and job.attemptNr <= 10) \
                     or failedInActive or usePilotRetry) \
                     and job.commandToPilot != 'tobekilled':
                    # check attemptNr for JEDI
                    moreRetryForJEDI = True
                    if useJEDI:
                        moreRetryForJEDI = self.checkMoreRetryJEDI(job)
                    # OK in JEDI
                    if moreRetryForJEDI:
                        tmpLog.debug('reset PandaID:%s #%s' % (job.PandaID,job.attemptNr))
                        if not changeJobInMem:
                            # job parameters
                            sqlJobP = "SELECT jobParameters FROM ATLAS_PANDA.jobParamsTable WHERE PandaID=:PandaID"
                            varMap = {}
                            varMap[':PandaID'] = job.PandaID
                            self.cur.execute(sqlJobP+comment, varMap)
                            for clobJobP, in self.cur:
                                try:
                                    job.jobParameters = clobJobP.read()
                                except AttributeError:
                                    job.jobParameters = str(clobJobP)
                                break
                        # reset job
                        job.jobStatus = 'activated'
                        job.startTime = None
                        job.modificationTime = datetime.datetime.utcnow()
                        job.attemptNr = job.attemptNr + 1
                        if usePilotRetry:
                            job.currentPriority -= 10
                        job.endTime             = None
                        job.transExitCode       = None
                        job.batchID             = None
                        for attr in job._attributes:
                            if attr.endswith('ErrorCode') or attr.endswith('ErrorDiag'):
                                setattr(job,attr,None)
                        # remove flag regarding to pledge-resource handling
                        if job.specialHandling not in [None,'NULL','']:
                            newSpecialHandling = re.sub(',*localpool','',job.specialHandling)
                            if newSpecialHandling == '':
                                job.specialHandling = None
                            else:
                                job.specialHandling = newSpecialHandling
                        # send it to long queue for analysis jobs
                        oldComputingSite = job.computingSite
                        if not changeJobInMem:
                            if job.computingSite.startswith('ANALY'):
                                longSite = None
                                tmpLongSiteList = []
                                tmpLongSite = re.sub('^ANALY_','ANALY_LONG_',job.computingSite)
                                tmpLongSite = re.sub('_\d+$','',tmpLongSite)
                                tmpLongSiteList.append(tmpLongSite)
                                tmpLongSite = job.computingSite + '_LONG'
                                tmpLongSiteList.append(tmpLongSite)
                                tmpLongSite = re.sub('SHORT','LONG',job.computingSite)
                                if tmpLongSite != job.computingSite:
                                    tmpLongSiteList.append(tmpLongSite)
                                # loop over all possible long sitenames
                                for tmpLongSite in tmpLongSiteList:
                                    varMap = {}
                                    varMap[':siteID'] = tmpLongSite
                                    varMap[':status'] = 'online'
                                    sqlSite = "SELECT /* use_json_type */ COUNT(*) FROM ATLAS_PANDA.schedconfig_json scj WHERE scj.panda_queue=:siteID AND scj.data.status=:status"
                                    self.cur.execute(sqlSite+comment, varMap)
                                    resSite = self.cur.fetchone()
                                    if resSite is not None and resSite[0] > 0:
                                        longSite = tmpLongSite
                                        break
                                # use long site if exists
                                if longSite is not None:
                                    tmpLog.debug('sending PandaID:%s to %s' % (job.PandaID,longSite))
                                    job.computingSite = longSite
                                    # set destinationSE if queue is changed
                                    if oldComputingSite == job.destinationSE:
                                        job.destinationSE = job.computingSite
                        if not changeJobInMem:
                            # select files
                            varMap = {}
                            varMap[':PandaID'] = job.PandaID
                            if not getNewPandaID:
                                varMap[':type1'] = 'log'
                                varMap[':type2'] = 'output'
                            sqlFile = "SELECT %s FROM ATLAS_PANDA.filesTable4 " % FileSpec.columnNames()
                            if not getNewPandaID:
                                sqlFile+= "WHERE PandaID=:PandaID AND (type=:type1 OR type=:type2)"
                            else:
                                sqlFile+= "WHERE PandaID=:PandaID"
                            self.cur.arraysize = 100
                            self.cur.execute(sqlFile+comment, varMap)
                            resFs = self.cur.fetchall()
                        else:
                            # get log or output files only
                            resFs = []
                            for tmpFile in job.Files:
                                if tmpFile.type in ['log','output']:
                                    resFs.append(tmpFile)
                        # loop over all files
                        for resF in resFs:
                            if not changeJobInMem:
                                # set PandaID
                                file = FileSpec()
                                file.pack(resF)
                                job.addFile(file)
                            else:
                                file = resF
                            # set new GUID
                            if file.type == 'log':
                                file.GUID = str(uuid.uuid4())
                            # don't change input or lib.tgz, or ES merge output/log since it causes a problem with input name construction
                            if file.type in ['input','pseudo_input'] or (file.type == 'output' and job.prodSourceLabel == 'panda') or \
                                   (file.type == 'output' and file.lfn.endswith('.lib.tgz') and job.prodSourceLabel in JobUtils.list_ptest_prod_sources):
                                continue
                            # append attemptNr to LFN
                            oldName = file.lfn
                            file.lfn = re.sub('\.\d+$','',file.lfn)
                            file.lfn = '%s.%s' % (file.lfn,job.attemptNr)
                            newName = file.lfn
                            # set destinationSE
                            if oldComputingSite == file.destinationSE:
                                file.destinationSE = job.computingSite
                            # modify jobParameters
                            if not recoverableEsMerge:
                                sepPatt = "(\'|\"|%20|:)" + oldName + "(\'|\"|%20| )"
                            else:
                                sepPatt = "(\'|\"| |:|=)" + oldName + "(\'|\"| |<|$)"
                            matches = re.findall(sepPatt,job.jobParameters)
                            for match in matches:
                                oldPatt = match[0]+oldName+match[-1]
                                newPatt = match[0]+newName+match[-1]
                                job.jobParameters = re.sub(oldPatt,newPatt,job.jobParameters)
                            if not changeJobInMem and not getNewPandaID:
                                # reset file status
                                if file.type in ['output','log']:
                                    file.status = 'unknown'
                                # update files
                                sqlFup = ("UPDATE ATLAS_PANDA.filesTable4 SET %s" % file.bindUpdateChangesExpression()) + "WHERE row_ID=:row_ID"
                                varMap = file.valuesMap(onlyChanged=True)
                                if varMap != {}:
                                    varMap[':row_ID'] = file.row_ID
                                    self.cur.execute(sqlFup+comment, varMap)
                        # set site to ES merger job
                        if recoverableEsMerge and EventServiceUtils.isEventServiceMerge(job):
                            self.setSiteForEsMerge(job, False, methodName, comment)
                        if not changeJobInMem:
                            # reuse original PandaID
                            if not getNewPandaID:
                                # update job
                                sql2 = "UPDATE ATLAS_PANDA.jobsActive4 SET %s " % job.bindUpdateChangesExpression()
                                sql2+= "WHERE PandaID=:PandaID "
                                varMap = job.valuesMap(onlyChanged=True)
                                varMap[':PandaID'] = job.PandaID
                                self.cur.execute(sql2+comment, varMap)
                                # update job parameters
                                sqlJobP = "UPDATE ATLAS_PANDA.jobParamsTable SET jobParameters=:param WHERE PandaID=:PandaID"
                                varMap = {}
                                varMap[':PandaID'] = job.PandaID
                                varMap[':param']   = job.jobParameters
                                self.cur.execute(sqlJobP+comment, varMap)
                                updatedFlag = True
                            else:
                                # read metadata
                                sqlMeta = "SELECT metaData FROM ATLAS_PANDA.metaTable WHERE PandaID=:PandaID"
                                varMap = {}
                                varMap[':PandaID'] = job.PandaID
                                self.cur.execute(sqlMeta+comment, varMap)
                                for clobJobP, in self.cur:
                                    try:
                                        job.metadata = clobJobP.read()
                                    except AttributeError:
                                        job.metadata = str(clobJobP)
                                    break
                                # insert job with new PandaID
                                sql1 = "INSERT INTO ATLAS_PANDA.jobsActive4 (%s) " % JobSpec.columnNames()
                                sql1+= JobSpec.bindValuesExpression(useSeq=True)
                                sql1+= " RETURNING PandaID INTO :newPandaID"
                                # set parentID
                                job.parentID = job.PandaID
                                job.creationTime = datetime.datetime.utcnow()
                                job.modificationTime = job.creationTime
                                varMap = job.valuesMap(useSeq=True)
                                varMap[':newPandaID'] = self.cur.var(varNUMBER)
                                # insert
                                retI = self.cur.execute(sql1+comment, varMap)
                                # set PandaID
                                val = self.getvalue_corrector(self.cur.getvalue(varMap[':newPandaID']))
                                job.PandaID = long(val)
                                tmpLog.debug('Generate new PandaID %s -> %s #%s' % (job.parentID,job.PandaID,job.attemptNr))
                                # insert files
                                sqlFile = "INSERT INTO ATLAS_PANDA.filesTable4 (%s) " % FileSpec.columnNames()
                                sqlFile+= FileSpec.bindValuesExpression(useSeq=True)
                                sqlFile+= " RETURNING row_ID INTO :newRowID"
                                for file in job.Files:
                                    # reset rowID
                                    file.row_ID = None
                                    # insert
                                    varMap = file.valuesMap(useSeq=True)
                                    varMap[':newRowID'] = self.cur.var(varNUMBER)
                                    self.cur.execute(sqlFile+comment, varMap)
                                    val = self.getvalue_corrector(self.cur.getvalue(varMap[':newRowID']))
                                    file.row_ID = long(val)
                                # job parameters
                                sqlJob = "INSERT INTO ATLAS_PANDA.jobParamsTable (PandaID,jobParameters) VALUES (:PandaID,:param)"
                                varMap = {}
                                varMap[':PandaID'] = job.PandaID
                                varMap[':param']   = job.jobParameters
                                self.cur.execute(sqlJob+comment, varMap)
                                # set error code to original job to avoid being retried by another process
                                sqlE = "UPDATE ATLAS_PANDA.jobsActive4 SET taskBufferErrorCode=:errCode,taskBufferErrorDiag=:errDiag WHERE PandaID=:PandaID"
                                varMap = {}
                                varMap[':PandaID'] = job.parentID
                                varMap[':errCode'] = ErrorCode.EC_PilotRetried
                                varMap[':errDiag'] = 'retrying at the same site. new PandaID=%s' % job.PandaID
                                self.cur.execute(sqlE+comment, varMap)
                                # propagate change to JEDI
                                if useJEDI:
                                    self.updateForPilotRetryJEDI(job,self.cur)
                        # set return
                        if not getNewPandaID:
                            retValue = True
                if not changeJobInMem:
                    # commit
                    if not self._commit():
                        raise RuntimeError('Commit error')
                    # record status change
                    try:
                        if updatedFlag:
                            self.recordStatusChange(job.PandaID,job.jobStatus,jobInfo=job)
                            self.push_job_status_message(job, job.PandaID, job.jobStatus)
                    except Exception:
                        tmpLog.error('recordStatusChange in retryJob')
                return retValue
            except Exception:
                # roll back
                self._rollback()
                if iTry+1 < nTry:
                    _logger.debug("retryJob : %s retry : %s" % (pandaID,iTry))
                    time.sleep(random.randint(10,20))
                    continue
                # error report
                self.dumpErrorMessage(_logger,methodName)
                return False


    # retry failed analysis jobs in Active4
    def retryJobsInActive(self,prodUserName,jobDefinitionID,isJEDI=False):
        comment = ' /* DBProxy.retryJobsInActive */'
        _logger.debug("retryJobsInActive : start - %s %s" % (prodUserName,jobDefinitionID))
        try:
            # begin transaction
            self.conn.begin()
            # count the number of jobs in Defined
            sqlC  = "SELECT COUNT(*) FROM ATLAS_PANDA.jobsDefined4 "
            sqlC += "WHERE prodUserName=:prodUserName AND jobDefinitionID=:jobDefinitionID "
            sqlC += "AND prodSourceLabel IN (:prodSourceLabel1,:prodSourceLabel2) "
            varMap = {}
            varMap[':prodUserName']    = prodUserName
            varMap[':jobDefinitionID'] = jobDefinitionID
            varMap[':prodSourceLabel1'] = 'user'
            varMap[':prodSourceLabel2'] = 'panda'
            self.cur.arraysize = 10
            self.cur.execute(sqlC+comment,varMap)
            res = self.cur.fetchone()
            # failed to get the number of jobs in Defined
            if res is None:
                _logger.error("retryJobsInActive : %s %s - failed to get num of jobs in Def" % (prodUserName,jobDefinitionID))
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                # return None for DB error
                return None
            nJobsInDef = res[0]
            # get failed PandaIDs in Active
            sql0 = "SELECT PandaID,jobStatus,taskBufferErrorCode,attemptNr FROM ATLAS_PANDA.jobsActive4 "
            sql0+= "WHERE prodUserName=:prodUserName AND jobDefinitionID=:jobDefinitionID "
            sql0+= "AND prodSourceLabel=:prodSourceLabel "
            if isJEDI:
                sql0+= "AND attemptNr<maxAttempt "
            varMap = {}
            varMap[':prodUserName']    = prodUserName
            varMap[':jobDefinitionID'] = jobDefinitionID
            varMap[':prodSourceLabel'] = 'user'
            self.cur.execute(sql0+comment,varMap)
            res = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # the number of jobs in Active
            nJobsInAct = len(res)
            # loop over all PandaID
            failedPandaIDs = []
            for pandaID,tmpJobStatus,tmpTaskBufferErrorCode,tmpAttemptNr in res:
                if tmpJobStatus == 'failed' and tmpTaskBufferErrorCode not in \
                       [ErrorCode.EC_Reassigned,ErrorCode.EC_Retried,ErrorCode.EC_PilotRetried]:
                    failedPandaIDs.append((pandaID,tmpAttemptNr))
            _logger.debug("retryJobsInActive : %s %s - %s failed jobs" % (prodUserName,jobDefinitionID,len(failedPandaIDs)))
            # there are some failed jobs in Active
            if failedPandaIDs != []:
                # get list of sub datasets to lock Closer
                sqlF  = "SELECT DISTINCT destinationDBlock FROM ATLAS_PANDA.filesTable4 "
                sqlF += "WHERE PandaID=:PandaID AND type IN (:type1,:type2) "
                varMap = {}
                varMap[':PandaID'] = failedPandaIDs[0][0]
                varMap[':type1']   = 'log'
                varMap[':type2']   = 'output'
                # begin transaction
                self.conn.begin()
                self.cur.arraysize = 100000
                self.cur.execute(sqlF+comment,varMap)
                res = self.cur.fetchall()
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                subDsList = []
                for tmpDSname, in res:
                    tmpDS = self.queryDatasetWithMap({'name':tmpDSname})
                    if tmpDS is None:
                        _logger.error("retryJobsInActive : %s %s - failed to get DS=%s" % (prodUserName,jobDefinitionID,tmpDSname))
                        # return None for DB error
                        return None
                    # append
                    subDsList.append(tmpDS)
                # lock datasets
                lockedDS = True
                ngStatus = ['closed','tobeclosed','completed','tobemerged','merging','cleanup']
                sqlD = "UPDATE ATLAS_PANDA.Datasets SET status=:status,modificationdate=CURRENT_DATE "
                sqlD+= "WHERE vuid=:vuid AND NOT status IN ("
                for tmpIdx,tmpNgStat in enumerate(ngStatus):
                    sqlD += ':ngSt%s,' % tmpIdx
                sqlD = sqlD[:-1]
                sqlD += ") "
                self.conn.begin()
                self.cur.arraysize = 10
                for tmpDS in subDsList:
                    varMap = {}
                    varMap[':status'] = 'locked'
                    varMap[':vuid'] = tmpDS.vuid
                    for tmpIdx,tmpNgStat in enumerate(ngStatus):
                        tmpKey = ':ngSt%s' % tmpIdx
                        varMap[tmpKey] = tmpNgStat
                    # update
                    self.cur.execute(sqlD+comment,varMap)
                    retD = self.cur.rowcount
                    # datasets already closed
                    if retD == 0:
                        # roll back
                        self._rollback()
                        # failed to lock datasets
                        _logger.debug("retryJobsInActive : %s %s - %s is closed" % (prodUserName,jobDefinitionID,tmpDS.name))
                        lockedDS = False
                        break
                # retry jobs
                if lockedDS:
                    # commit for dataset lock
                    if not self._commit():
                        raise RuntimeError('Commit error')
                    # loop over all PandaIDs
                    for pandaID,tmpAttemptNr in failedPandaIDs:
                        retryRet = self.retryJob(pandaID,{},failedInActive=True,attemptNr=tmpAttemptNr)
                        _logger.debug("retryJobsInActive : %s %s - PandaID=%s %s" % (prodUserName,jobDefinitionID,pandaID,retryRet))
                    # unlock datasets
                    sqlDU = "UPDATE ATLAS_PANDA.Datasets SET status=:nStatus,modificationdate=CURRENT_DATE "
                    sqlDU+= "WHERE vuid=:vuid AND status=:oStatus"
                    self.conn.begin()
                    self.cur.arraysize = 10
                    for tmpDS in subDsList:
                        varMap = {}
                        varMap[':oStatus'] = 'locked'
                        varMap[':nStatus'] = tmpDS.status
                        varMap[':vuid'] = tmpDS.vuid
                        # update
                        self.cur.execute(sqlDU+comment,varMap)
                    # commit
                    if not self._commit():
                        raise RuntimeError('Commit error')
            # return True when job is active
            retVal = False
            if nJobsInAct > 0 or nJobsInDef > 0:
                retVal = True
            _logger.debug("retryJobsInActive : end %s - %s %s" % (retVal,prodUserName,jobDefinitionID))
            return retVal
        except Exception:
            # roll back
            self._rollback()
            # error report
            errType,errValue = sys.exc_info()[:2]
            _logger.error("retryJobsInActive : %s %s" % (errType,errValue))
            return None


    # get jobs
    def getJobs(self, nJobs, siteName, prodSourceLabel, cpu, mem, diskSpace, node, timeout, computingElement,
                atlasRelease, prodUserID, countryGroup, workingGroup, allowOtherCountry, taskID, background,
                resourceType, harvester_id, worker_id, schedulerID, jobType, is_gu, via_topic):
        """
        1. Construct where clause (sql1) based on applicable filters for request
        2. Select n jobs with the highest priorities and the lowest pandaids
        3. Update the jobs to status SENT
        4. Pack the files and if jobs are AES also the event ranges
        """
        comment = ' /* DBProxy.getJobs */'

        # Number of PanDAIDs that will be tried
        if hasattr(panda_config, 'nJobsInGetJob'):
            maxAttemptIDx = panda_config.nJobsInGetJob
        else:
            maxAttemptIDx = 10

        # construct where clause
        getValMap = {}
        getValMap[':oldJobStatus'] = 'activated'
        getValMap[':computingSite'] = siteName

        # sql1 is the WHERE clause with all the applicable filters for the request
        sql1 = "WHERE jobStatus=:oldJobStatus AND computingSite=:computingSite "
        if mem not in [0,'0']:
            sql1+= "AND (minRamCount<=:minRamCount OR minRamCount=0) "
            getValMap[':minRamCount'] = mem
        if diskSpace not in [0,'0']:
            sql1+= "AND (maxDiskCount<=:maxDiskCount OR maxDiskCount=0) "
            getValMap[':maxDiskCount'] = diskSpace
        if background is True:
            sql1+= "AND jobExecutionID=1 "
        if resourceType is not None:
            sql1+= "AND resource_type=:resourceType "
            getValMap[':resourceType'] = resourceType
        if prodSourceLabel == 'user':
            sql1+= "AND prodSourceLabel IN (:prodSourceLabel1,:prodSourceLabel2,:prodSourceLabel3) "
            getValMap[':prodSourceLabel1'] = 'user'
            getValMap[':prodSourceLabel2'] = 'panda'
            getValMap[':prodSourceLabel3'] = 'install'
        elif prodSourceLabel == 'ddm':
            sql1+= "AND prodSourceLabel=:prodSourceLabel "
            getValMap[':prodSourceLabel'] = 'ddm'
        elif prodSourceLabel in [None,'managed']:
            sql1+= "AND prodSourceLabel IN (:prodSourceLabel1,:prodSourceLabel2,:prodSourceLabel3,:prodSourceLabel4) "
            getValMap[':prodSourceLabel1'] = 'managed'
            getValMap[':prodSourceLabel2'] = 'test'
            getValMap[':prodSourceLabel3'] = 'prod_test'
            getValMap[':prodSourceLabel4'] = 'install'
        elif prodSourceLabel == 'software':
            sql1+= "AND prodSourceLabel=:prodSourceLabel "
            getValMap[':prodSourceLabel'] = 'software'
        elif prodSourceLabel == 'test' and computingElement is not None:
            if is_gu and jobType == 'user':
                    sql1+= "AND processingType=:processingType1 "
                    getValMap[':processingType1'] = 'gangarobot'  # analysis HC jobs
            else:
                sql1+= "AND (processingType=:processingType1 "
                sql1+= "OR prodSourceLabel IN (:prodSourceLabel1,:prodSourceLabel2,:prodSourceLabel3)) "
                getValMap[':processingType1'] = 'gangarobot'  # analysis HC jobs
                getValMap[':prodSourceLabel1'] = 'prod_test'  # production HC jobs
                getValMap[':prodSourceLabel2'] = 'install'
                getValMap[':prodSourceLabel3'] = 'test'
        elif prodSourceLabel == 'unified':
            pass
        else:
            sql1+= "AND prodSourceLabel=:prodSourceLabel "
            getValMap[':prodSourceLabel'] = prodSourceLabel
        # user ID
        if prodUserID is not None:
            # get compact DN
            compactDN = self.cleanUserID(prodUserID)
            if compactDN in ['','NULL',None]:
                compactDN = prodUserID
            sql1+= "AND prodUserName=:prodUserName "
            getValMap[':prodUserName'] = compactDN

        # taskID
        if taskID not in [None,'NULL']:
            sql1+= "AND jediTaskID=:taskID "
            getValMap[':taskID'] = taskID

        # get the sorting criteria (global shares, age, etc.)
        sorting_sql, sorting_varmap = self.getSortingCriteria(siteName, maxAttemptIDx)
        if sorting_varmap:  # copy the var map, but not the sql, since it has to be at the very end
            for tmp_key in sorting_varmap:
                getValMap[tmp_key] = sorting_varmap[tmp_key]

        # sql2 is query to get the DB entry for a specific PanDA ID
        sql2 = "SELECT %s FROM ATLAS_PANDA.jobsActive4 " % JobSpec.columnNames()
        sql2+= "WHERE PandaID=:PandaID"
        retJobs = []
        nSent = 0
        getValMapOrig = copy.copy(getValMap)
        tmpLog = None
        try:
            timeLimit = datetime.timedelta(seconds=timeout-10)
            timeStart = datetime.datetime.utcnow()
            tmpLog = LogWrapper(_logger, "getJobs : %s -> " % datetime.datetime.isoformat(timeStart))
            attLimit  = datetime.datetime.utcnow() - datetime.timedelta(minutes=15)
            attSQL    = "AND ((creationTime<:creationTime AND attemptNr>1) OR attemptNr<=1) "
            # get nJobs
            for iJob in range(nJobs):
                getValMap = copy.copy(getValMapOrig)
                pandaID = 0
                fileMapForMem = {}
                # select channel for ddm jobs
                if prodSourceLabel == 'ddm':
                    sqlDDM = "SELECT count(*),jobStatus,sourceSite,destinationSite,transferType FROM ATLAS_PANDA.jobsActive4 WHERE computingSite=:computingSite AND prodSourceLabel=:prodSourceLabel " \
                             + attSQL + "GROUP BY jobStatus,sourceSite,destinationSite,transferType"
                    ddmValMap = {}
                    ddmValMap[':computingSite']   = siteName
                    ddmValMap[':creationTime']    = attLimit
                    ddmValMap[':prodSourceLabel'] = 'ddm'
                    _logger.debug(sqlDDM+comment+str(ddmValMap))
                    # start transaction
                    self.conn.begin()
                    # select
                    self.cur.arraysize = 100
                    self.cur.execute(sqlDDM+comment, ddmValMap)
                    resDDM = self.cur.fetchall()
                    # commit
                    if not self._commit():
                        raise RuntimeError('Commit error')
                    # make a channel map
                    channelMap = {}
                    for tmp_count,tmp_jobStatus,tmp_sourceSite,tmp_destinationSite,tmp_transferType in resDDM:
                        # use source,dest,type as the key
                        channel = (tmp_sourceSite,tmp_destinationSite,tmp_transferType)
                        if channel not in channelMap:
                            channelMap[channel] = {}
                        # ignore holding
                        if tmp_jobStatus == 'holding':
                            continue
                        # distinguish activate from other stats
                        if tmp_jobStatus != 'activated':
                            tmp_jobStatus = 'others'
                        # append
                        if tmp_jobStatus not in channelMap[channel]:
                            channelMap[channel][tmp_jobStatus] = int(tmp_count)
                        else:
                            channelMap[channel][tmp_jobStatus] += int(tmp_count)
                    _logger.debug(channelMap)
                    # choose channel
                    channels = list(channelMap)
                    random.shuffle(channels)
                    foundChannel = False
                    for channel in channels:
                        # no activated jobs
                        if 'activated' not in channelMap[channel] or channelMap[channel]['activated'] == 0:
                            continue
                        maxRunning = 15
                        # prestaging job
                        if channel[0] == channel[1] and channel[2] == 'dis':
                            maxRunning = 50
                        if 'others' not in channelMap[channel] or channelMap[channel]['others'] < maxRunning:
                            # set SQL
                            sql1+= "AND sourceSite=:sourceSite AND destinationSite=:destinationSite AND transferType=:transferType "
                            getValMap[':sourceSite']      = channel[0]
                            getValMap[':destinationSite'] = channel[1]
                            getValMap[':transferType']    = channel[2]
                            foundChannel = True
                            break
                    # no proper channel
                    if not foundChannel:
                        _logger.debug("getJobs : no DDM jobs for Site %s" % siteName)
                        break
                # get job
                if prodSourceLabel in ['ddm']:
                    # to add some delay for attempts
                    sql1 += attSQL
                    getValMap[':creationTime'] = attLimit
                nTry=1
                for iTry in range(nTry):
                    # set siteID
                    tmpSiteID = siteName
                    # get file lock
                    tmpLog.debug("lock")
                    if (datetime.datetime.utcnow() - timeStart) < timeLimit:
                        toGetPandaIDs = True
                        pandaIDs = []
                        specialHandlingMap = {}

                        if toGetPandaIDs:
                            # get PandaIDs
                            sqlP = "SELECT /*+ INDEX_RS_ASC(tab (PRODSOURCELABEL COMPUTINGSITE JOBSTATUS) ) */ PandaID,currentPriority,specialHandling FROM ATLAS_PANDA.jobsActive4 tab "
                            sqlP += sql1

                            if sorting_sql:
                                sqlP = 'SELECT * FROM (' + sqlP
                                sqlP += sorting_sql

                            tmpLog.debug(sqlP+comment+str(getValMap))
                            # start transaction
                            self.conn.begin()
                            # select
                            self.cur.arraysize = 100000
                            self.cur.execute(sqlP+comment, getValMap)
                            resIDs = self.cur.fetchall()
                            # commit
                            if not self._commit():
                                raise RuntimeError('Commit error')

                            for tmpPandaID, tmpCurrentPriority, tmpSpecialHandling in resIDs:
                                pandaIDs.append(tmpPandaID)
                                specialHandlingMap[tmpPandaID] = tmpSpecialHandling

                        if pandaIDs == []:
                            tmpLog.debug("no PandaIDs")
                            retU = 0 # retU: return from update
                        else:
                            # update
                            for indexID,tmpPandaID in enumerate(pandaIDs):
                                # max attempts
                                if indexID > maxAttemptIDx:
                                    break
                                # lock first
                                sqlPL = "SELECT jobStatus FROM ATLAS_PANDA.jobsActive4 "\
                                        "WHERE PandaID=:PandaID FOR UPDATE NOWAIT "
                                # update
                                sqlJ = "UPDATE ATLAS_PANDA.jobsActive4 "
                                sqlJ+= "SET jobStatus=:newJobStatus,modificationTime=CURRENT_DATE,modificationHost=:modificationHost,startTime=CURRENT_DATE"
                                varMap = {}
                                varMap[':PandaID']          = tmpPandaID
                                varMap[':newJobStatus']     = 'sent'
                                varMap[':oldJobStatus']     = 'activated'
                                varMap[':modificationHost'] = node
                                # set CE
                                if computingElement is not None:
                                    sqlJ+= ",computingElement=:computingElement"
                                    varMap[':computingElement'] =  computingElement
                                # set schedulerID
                                if schedulerID is not None:
                                    sqlJ+= ",schedulerID=:schedulerID"
                                    varMap[':schedulerID'] = schedulerID

                                # background flag
                                if background is not True:
                                    sqlJ+= ",jobExecutionID=0"
                                sqlJ+= " WHERE PandaID=:PandaID AND jobStatus=:oldJobStatus"
                                # SQL to get nSent
                                sentLimit = timeStart - datetime.timedelta(seconds=60)
                                sqlSent  = "SELECT count(*) FROM ATLAS_PANDA.jobsActive4 WHERE jobStatus=:jobStatus "
                                sqlSent += "AND prodSourceLabel IN (:prodSourceLabel1,:prodSourceLabel2) "
                                sqlSent += "AND computingSite=:computingSite "
                                sqlSent += "AND modificationTime>:modificationTime "
                                varMapSent = {}
                                varMapSent[':jobStatus'] = 'sent'
                                varMapSent[':computingSite'] = tmpSiteID
                                varMapSent[':modificationTime'] = sentLimit
                                varMapSent[':prodSourceLabel1'] = 'managed'
                                varMapSent[':prodSourceLabel2'] = 'test'
                                # start
                                # start transaction
                                self.conn.begin()
                                # pre-lock
                                prelocked = False
                                try:
                                    varMapPL = {':PandaID': tmpPandaID}
                                    tmpLog.debug(sqlPL + comment + str(varMapPL))
                                    self.cur.execute(sqlPL + comment, varMapPL)
                                    prelocked = True
                                except Exception:
                                    tmpLog.debug("cannot pre-lock")
                                # update
                                if prelocked:
                                    tmpLog.debug(sqlJ + comment + str(varMap))
                                    self.cur.execute(sqlJ+comment, varMap)
                                    retU = self.cur.rowcount
                                    tmpLog.debug("retU=%s" % retU)
                                else:
                                    retU = 0
                                if retU != 0:
                                    # get nSent for production jobs
                                    if prodSourceLabel in [None,'managed']:
                                        tmpLog.debug(sqlSent+comment+str(varMapSent))
                                        self.cur.execute(sqlSent+comment, varMapSent)
                                        resSent = self.cur.fetchone()
                                        if resSent is not None:
                                            nSent, = resSent
                                    # insert job and worker mapping
                                    if harvester_id is not None and worker_id is not None:
                                        # insert worker if missing
                                        self.updateWorkers(harvester_id,
                                                           [{'workerID': worker_id, 'nJobs': 1, 'status': 'running',
                                                             'lastUpdate': datetime.datetime.utcnow()}],
                                                           useCommit=False)
                                        # insert mapping
                                        sqlJWH = "SELECT 1 FROM ATLAS_PANDA.Harvester_Instances WHERE harvester_ID=:harvesterID "
                                        sqlJWC  = "SELECT PandaID FROM ATLAS_PANDA.Harvester_Rel_Jobs_Workers "
                                        sqlJWC += "WHERE harvesterID=:harvesterID AND workerID=:workerID AND PandaID=:PandaID "
                                        sqlJWI  = "INSERT INTO ATLAS_PANDA.Harvester_Rel_Jobs_Workers (harvesterID,workerID,PandaID,lastUpdate) "
                                        sqlJWI += "VALUES (:harvesterID,:workerID,:PandaID,:lastUpdate) "
                                        sqlJWU  = "UPDATE ATLAS_PANDA.Harvester_Rel_Jobs_Workers SET lastUpdate=:lastUpdate "
                                        sqlJWU += "WHERE harvesterID=:harvesterID AND workerID=:workerID AND PandaID=:PandaID "
                                        varMap = dict()
                                        varMap[':harvesterID'] = harvester_id
                                        self.cur.execute(sqlJWH+comment, varMap)
                                        resJWH = self.cur.fetchone()
                                        if resJWH is None:
                                            tmpLog.debug("getJobs : Site {0} harvester_id={1} not found".format(tmpSiteID, harvester_id))
                                        else:
                                            varMap = dict()
                                            varMap[':harvesterID'] = harvester_id
                                            varMap[':workerID'] = worker_id
                                            varMap[':PandaID'] = tmpPandaID
                                            self.cur.execute(sqlJWC+comment, varMap)
                                            resJWC = self.cur.fetchone()
                                            varMap = dict()
                                            varMap[':harvesterID'] = harvester_id
                                            varMap[':workerID'] = worker_id
                                            varMap[':PandaID'] = tmpPandaID
                                            varMap[':lastUpdate'] = datetime.datetime.utcnow()
                                            if resJWC is None:
                                                # insert
                                                self.cur.execute(sqlJWI+comment, varMap)
                                            else:
                                                # update
                                                self.cur.execute(sqlJWU+comment, varMap)
                                # commit
                                if not self._commit():
                                    raise RuntimeError('Commit error')
                                # succeeded
                                if retU != 0:
                                    pandaID = tmpPandaID
                                    break
                    else:
                        tmpLog.debug("do nothing")
                        retU = 0
                    # release file lock
                    tmpLog.debug("unlock")
                    # succeeded
                    if retU != 0:
                        break
                    if iTry+1 < nTry:
                        #time.sleep(0.5)
                        pass
                # failed to UPDATE
                if retU == 0:
                    # reset pandaID
                    pandaID = 0
                tmpLog.debug("Site %s : retU %s : PandaID %s - %s"
                              % (siteName,retU,pandaID,prodSourceLabel))
                if pandaID == 0:
                    break
                # start transaction
                self.conn.begin()
                # select
                varMap = {}
                varMap[':PandaID'] = pandaID
                self.cur.arraysize = 10
                self.cur.execute(sql2+comment, varMap)
                res = self.cur.fetchone()
                if len(res) == 0:
                    # commit
                    if not self._commit():
                        raise RuntimeError('Commit error')
                    break
                # instantiate Job
                job = JobSpec()
                job.pack(res)
                # sql to read range
                sqlRR  = "SELECT /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
                sqlRR += "PandaID,job_processID,attemptNr,objStore_ID,zipRow_ID,path_convention "
                sqlRR += "FROM {0}.JEDI_Events tab ".format(panda_config.schemaJEDI)
                sqlRR += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID AND status=:eventStatus "
                # sql to read log backet IDs
                sqlLBK  = "SELECT jobMetrics FROM ATLAS_PANDA.jobsArchived4 WHERE PandaID=:PandaID "
                sqlLBK += "UNION "
                sqlLBK += "SELECT jobMetrics FROM ATLAS_PANDAARCH.jobsArchived WHERE PandaID=:PandaID AND modificationTime>(CURRENT_DATE-30) "
                # read files
                sqlFile = "SELECT %s FROM ATLAS_PANDA.filesTable4 " % FileSpec.columnNames()
                sqlFile+= "WHERE PandaID=:PandaID "
                # read LFN and dataset name for output files
                sqlFileOut = "SELECT lfn,dataset FROM ATLAS_PANDA.filesTable4 "
                sqlFileOut+= "WHERE PandaID=:PandaID AND type=:type "
                # read files from JEDI for jumbo jobs
                sqlFileJEDI  = "SELECT lfn,GUID,fsize,checksum "
                sqlFileJEDI += "FROM {0}.JEDI_Dataset_Contents ".format(panda_config.schemaJEDI)
                sqlFileJEDI += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID "
                sqlFileJEDI += "ORDER BY lfn "
                # read zip file
                sqlZipFile  = "SELECT lfn,destinationSE,fsize,checksum FROM ATLAS_PANDA.filesTable4 "
                sqlZipFile += "WHERE row_ID=:row_ID "
                sqlZipFile += "UNION "
                sqlZipFile += "SELECT lfn,destinationSE,fsize,checksum FROM ATLAS_PANDAARCH.filesTable_ARCH "
                sqlZipFile += "WHERE row_ID=:row_ID "
                self.cur.arraysize = 10000
                self.cur.execute(sqlFile+comment, varMap)
                resFs = self.cur.fetchall()
                eventRangeIDs = {}
                esDonePandaIDs = []
                esOutputZipMap = {}
                esZipRow_IDs = set()
                esOutputFileMap = {}
                # use new file format for ES
                useNewFileFormatForES = False
                if job.AtlasRelease is not None:
                    try:
                        tmpMajorVer = job.AtlasRelease.split('-')[-1].split('.')[0]
                        if int(tmpMajorVer) == 20:
                            useNewFileFormatForES = True
                    except Exception:
                        pass
                for resF in resFs:
                    file = FileSpec()
                    file.pack(resF)
                    # add files except event service merge or jumbo
                    if (not EventServiceUtils.isEventServiceMerge(job) and not EventServiceUtils.isJumboJob(job)) \
                            or file.type in ['output','log']:
                        job.addFile(file)
                    # read real input files for jumbo jobs
                    elif EventServiceUtils.isJumboJob(job):
                        # get files
                        varMap = {}
                        varMap[':jediTaskID'] = file.jediTaskID
                        varMap[':datasetID']  = file.datasetID
                        self.cur.execute(sqlFileJEDI+comment, varMap)
                        resFileJEDI = self.cur.fetchall()
                        for tmpLFN,tmpGUID,tmpFsize,tmpChecksum in resFileJEDI:
                            newFileSpec = FileSpec()
                            newFileSpec.pack(resF)
                            newFileSpec.lfn = tmpLFN
                            newFileSpec.GUID = tmpGUID
                            newFileSpec.fsize = tmpFsize
                            newFileSpec.checksum = tmpChecksum
                            # add file
                            job.addFile(newFileSpec)
                        continue
                    # construct input files from event ragnes for event service merge
                    if EventServiceUtils.isEventServiceMerge(job):
                        # only for input
                        if file.type not in ['output','log']:
                            # get ranges
                            varMap = {}
                            varMap[':jediTaskID'] = file.jediTaskID
                            varMap[':datasetID']  = file.datasetID
                            varMap[':fileID']     = file.fileID
                            varMap[':eventStatus'] = EventServiceUtils.ST_done
                            self.cur.execute(sqlRR+comment, varMap)
                            resRR = self.cur.fetchall()
                            for esPandaID,job_processID,attemptNr,objStoreID,zipRow_ID,pathConvention in resRR:
                                tmpEventRangeID = self.makeEventRangeID(file.jediTaskID,esPandaID,file.fileID,job_processID,attemptNr)
                                if file.fileID not in eventRangeIDs:
                                    eventRangeIDs[file.fileID] = {}
                                addFlag = False
                                if job_processID not in eventRangeIDs[file.fileID]:
                                    addFlag = True
                                else:
                                    oldEsPandaID = eventRangeIDs[file.fileID][job_processID]['pandaID']
                                    if esPandaID > oldEsPandaID:
                                        addFlag= True
                                        if oldEsPandaID in esDonePandaIDs:
                                            esDonePandaIDs.remove(oldEsPandaID)
                                if addFlag:
                                    # append
                                    if pathConvention is not None:
                                        objStoreID = '{0}/{1}'.format(objStoreID,pathConvention)
                                    eventRangeIDs[file.fileID][job_processID] = {'pandaID':esPandaID,
                                                                                 'eventRangeID':tmpEventRangeID,
                                                                                 'objStoreID':objStoreID}
                                    # zip file in jobMetrics
                                    if esPandaID not in esDonePandaIDs:
                                        esDonePandaIDs.append(esPandaID)
                                        # get jobMetrics
                                        varMap = {}
                                        varMap[':PandaID'] = esPandaID
                                        self.cur.execute(sqlLBK+comment,varMap)
                                        resLBK = self.cur.fetchone()
                                        if resLBK is not None and resLBK[0] is not None:
                                            outputZipBucketID = None
                                            tmpPatch = re.search('outputZipBucketID=(\d+)',resLBK[0])
                                            if tmpPatch is not None:
                                                outputZipBucketID = tmpPatch.group(1)
                                            outputZipName = None
                                            tmpPatch = re.search('outputZipName=([^ ]+)',resLBK[0])
                                            if tmpPatch is not None:
                                                outputZipName = tmpPatch.group(1)
                                            if outputZipBucketID is not None and outputZipName is not None:
                                                if esPandaID not in esOutputZipMap:
                                                    esOutputZipMap[esPandaID] = []
                                                esOutputZipMap[esPandaID].append({'name':outputZipName,
                                                                                  'osid':outputZipBucketID})
                                    # output LFN and dataset
                                    if esPandaID not in esOutputFileMap:
                                        esOutputFileMap[esPandaID] = dict()
                                        varMap = {}
                                        varMap[':PandaID'] = esPandaID
                                        varMap[':type'] = 'output'
                                        self.cur.execute(sqlFileOut+comment, varMap)
                                        resFileOut = self.cur.fetchall()
                                        for tmpOutLFN, tmpOutDataset in resFileOut:
                                             esOutputFileMap[esPandaID][tmpOutDataset] = tmpOutLFN
                                # zip file in fileTable
                                if zipRow_ID is not None and zipRow_ID not in esZipRow_IDs:
                                    esZipRow_IDs.add(zipRow_ID)
                                    varMap = {}
                                    varMap[':row_ID'] = zipRow_ID
                                    self.cur.execute(sqlZipFile+comment,varMap)
                                    resZip = self.cur.fetchone()
                                    if resZip is not None:
                                        outputZipName,outputZipBucketID,outputZipFsize,outputZipChecksum = resZip
                                        if esPandaID not in esOutputZipMap:
                                            esOutputZipMap[esPandaID] = []
                                        esOutputZipMap[esPandaID].append({'name':outputZipName,
                                                                          'osid':outputZipBucketID,
                                                                          'fsize':outputZipFsize,
                                                                          'checksum':outputZipChecksum})
                # make input for event service output merging
                mergeInputOutputMap = {}
                mergeInputFiles = []
                mergeFileObjStoreMap = {}
                mergeZipPandaIDs = []
                mergeZipLFNs = set()
                for tmpFileID in eventRangeIDs:
                    tmpMapEventRangeID = eventRangeIDs[tmpFileID]
                    jobProcessIDs = list(tmpMapEventRangeID)
                    jobProcessIDs.sort()
                    # make input
                    for jobProcessID in jobProcessIDs:
                        for tmpFileSpec in job.Files:
                            if tmpFileSpec.type not in ['output']:
                                continue
                            esPandaID = tmpMapEventRangeID[jobProcessID]['pandaID']
                            tmpInputFileSpec = copy.copy(tmpFileSpec)
                            tmpInputFileSpec.type = 'input'
                            outLFN = tmpInputFileSpec.lfn
                            # change LFN
                            if esPandaID in esOutputFileMap and tmpInputFileSpec.dataset in esOutputFileMap[esPandaID]:
                                tmpInputFileSpec.lfn = esOutputFileMap[esPandaID][tmpInputFileSpec.dataset]
                            # change attemptNr back to the original, which could have been changed by ES merge retry
                            if not useNewFileFormatForES:
                                origLFN = re.sub('\.\d+$','.1',tmpInputFileSpec.lfn)
                                outLFN  = re.sub('\.\d+$','.1',outLFN)
                            else:
                                origLFN = re.sub('\.\d+$','.1_000',tmpInputFileSpec.lfn)
                                outLFN  = re.sub('\.\d+$','.1_000',outLFN)
                            # append eventRangeID as suffix
                            tmpInputFileSpec.lfn = origLFN + '.' + tmpMapEventRangeID[jobProcessID]['eventRangeID']
                            # make input/output map
                            if outLFN not in mergeInputOutputMap:
                                mergeInputOutputMap[outLFN] = []
                            mergeInputOutputMap[outLFN].append(tmpInputFileSpec.lfn)
                            # add file
                            if esPandaID not in esOutputZipMap:
                                # no zip
                                mergeInputFiles.append(tmpInputFileSpec)
                                # mapping for ObjStore
                                mergeFileObjStoreMap[tmpInputFileSpec.lfn] = tmpMapEventRangeID[jobProcessID]['objStoreID']
                            elif esPandaID not in mergeZipPandaIDs:
                                # zip
                                mergeZipPandaIDs.append(esPandaID)
                                for tmpEsOutZipFile in esOutputZipMap[esPandaID]:
                                    # copy for zip
                                    tmpZipInputFileSpec = copy.copy(tmpInputFileSpec)
                                    # add prefix
                                    tmpZipInputFileSpec.lfn = 'zip://'+tmpEsOutZipFile['name']
                                    if 'fsize' in tmpEsOutZipFile:
                                        tmpZipInputFileSpec.fsize = tmpEsOutZipFile['fsize']
                                    if 'checksum' in tmpEsOutZipFile:
                                        tmpZipInputFileSpec.checksum = tmpEsOutZipFile['checksum']
                                    mergeInputFiles.append(tmpZipInputFileSpec)
                                    # mapping for ObjStore
                                    mergeFileObjStoreMap[tmpZipInputFileSpec.lfn] = tmpEsOutZipFile['osid']
                for tmpInputFileSpec in mergeInputFiles:
                    job.addFile(tmpInputFileSpec)
                # job parameters
                sqlJobP = "SELECT jobParameters FROM ATLAS_PANDA.jobParamsTable WHERE PandaID=:PandaID"
                varMap = {}
                varMap[':PandaID'] = job.PandaID
                self.cur.execute(sqlJobP+comment, varMap)
                for clobJobP, in self.cur:
                    try:
                        job.jobParameters = clobJobP.read()
                    except AttributeError:
                        job.jobParameters = str(clobJobP)
                    break
                # remove or extract parameters for merge
                if EventServiceUtils.isEventServiceJob(job) or EventServiceUtils.isJumboJob(job) or \
                   EventServiceUtils.isCoJumboJob(job):
                    try:
                        job.jobParameters = re.sub('<PANDA_ESMERGE_.+>.*</PANDA_ESMERGE_.+>','',job.jobParameters)
                    except Exception:
                        pass
                    # sort files since file order is important for positional event number
                    job.sortFiles()
                elif EventServiceUtils.isEventServiceMerge(job):
                    try:
                        origJobParameters = job.jobParameters
                        tmpMatch = re.search('<PANDA_ESMERGE_JOBP>(.*)</PANDA_ESMERGE_JOBP>',origJobParameters)
                        job.jobParameters = tmpMatch.group(1)
                        tmpMatch = re.search('<PANDA_ESMERGE_TRF>(.*)</PANDA_ESMERGE_TRF>',origJobParameters)
                        job.transformation = tmpMatch.group(1)
                    except Exception:
                        pass
                    # pass in/out map for merging via metadata
                    job.metadata = [mergeInputOutputMap,mergeFileObjStoreMap]
                # read task param
                if job.lockedby == 'jedi':
                    sqlTP = "SELECT ioIntensity,ioIntensityUnit FROM {0}.JEDI_Tasks WHERE jediTaskID=:jediTaskID ".format(panda_config.schemaJEDI)
                    varMap = {}
                    varMap[':jediTaskID'] = job.jediTaskID
                    self.cur.execute(sqlTP+comment, varMap)
                    resTP = self.cur.fetchone()
                    if resTP is not None:
                        ioIntensity, ioIntensityUnit = resTP
                        job.set_task_attribute('ioIntensity', ioIntensity)
                        job.set_task_attribute('ioIntensityUnit', ioIntensityUnit)
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')

                # append
                retJobs.append(job)
                # record status change
                try:
                    self.recordStatusChange(job.PandaID,job.jobStatus,jobInfo=job)
                except Exception:
                    tmpLog.error('recordStatusChange in getJobs')
                self.push_job_status_message(job, job.PandaID, job.jobStatus)
                if via_topic and job.is_push_job():
                    tmpLog.debug('delete job message')
                    mb_proxy_queue = self.get_mb_proxy('panda_pilot_queue')
                    srv_msg_utils.delete_job_message(mb_proxy_queue, job.PandaID)
            return retJobs, nSent
        except Exception as e:
            errStr = "getJobs : {} {}".format(str(e), traceback.format_exc())
            if tmpLog:
                tmpLog.error(errStr)
            else:
                _logger.error(errStr)
            # roll back
            self._rollback()
            return [], 0


    # reset job in jobsActive or jobsWaiting
    def resetJob(self,pandaID,activeTable=True,keepSite=False,getOldSubs=False,forPending=True):
        comment = ' /* DBProxy.resetJob */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += ' <PandaID={0}>'.format(pandaID)
        tmpLog = LogWrapper(_logger, methodName)
        tmpLog.debug("activeTable=%s" % activeTable)
        # select table
        table = 'ATLAS_PANDA.jobsWaiting4'
        if activeTable:
            table = 'ATLAS_PANDA.jobsActive4'
        sql1 = "SELECT %s FROM %s " % (JobSpec.columnNames(),table)
        sql1+= "WHERE PandaID=:PandaID"
        sql2 = "DELETE FROM %s " % table
        sql2+= "WHERE PandaID=:PandaID AND (jobStatus=:oldJobStatus1 OR jobStatus=:oldJobStatus2)"
        sql3 = "INSERT INTO ATLAS_PANDA.jobsDefined4 (%s) " % JobSpec.columnNames()
        sql3+= JobSpec.bindValuesExpression()
        try:
            # transaction causes Request ndbd time-out in ATLAS_PANDA.jobsActive4
            self.conn.begin()
            # select
            varMap = {}
            varMap[':PandaID'] = pandaID
            self.cur.arraysize = 10
            self.cur.execute(sql1+comment,varMap)
            res = self.cur.fetchone()
            # not found
            if res is None:
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                # return
                return None
            # instantiate Job
            job = JobSpec()
            job.pack(res)
            # if already running
            if job.jobStatus != 'waiting' and job.jobStatus != 'activated' \
                   and (forPending and job.jobStatus != 'pending'):
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                # return
                return None
            # do nothing for analysis jobs
            if job.prodSourceLabel in ['user','panda'] and not forPending \
                    and job.jobStatus != 'pending':
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                # return
                return None
            # delete
            varMap = {}
            varMap[':PandaID'] = pandaID
            if not forPending:
                varMap[':oldJobStatus1'] = 'waiting'
            else:
                varMap[':oldJobStatus1'] = 'pending'
            varMap[':oldJobStatus2'] = 'activated'
            self.cur.execute(sql2+comment,varMap)
            retD = self.cur.rowcount
            # delete failed
            tmpLog.debug("retD = %s" % retD)
            if retD != 1:
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                return None
            # delete from jobsDefined4 just in case
            varMap = {}
            varMap[':PandaID'] = pandaID
            sqlD = "DELETE FROM ATLAS_PANDA.jobsDefined4 WHERE PandaID=:PandaID"
            self.cur.execute(sqlD+comment,varMap)
            # increase priority
            if job.jobStatus == 'activated' and job.currentPriority < 100:
                job.currentPriority = 100
            # reset computing site and dispatchDBlocks
            job.jobStatus = 'defined'
            if job.prodSourceLabel not in ['user','panda']:
                job.dispatchDBlock   = None
                # erase old assignment
                if (not keepSite) and job.relocationFlag not in [1,2]:
                    job.computingSite = None
                job.computingElement = None
            # host and time information
            job.modificationHost = self.hostname
            job.modificationTime = datetime.datetime.utcnow()
            job.stateChangeTime  = job.modificationTime
            # reset
            job.brokerageErrorDiag = None
            job.brokerageErrorCode = None
            # insert
            self.cur.execute(sql3+comment, job.valuesMap())
            # job parameters
            sqlJobP = "SELECT jobParameters FROM ATLAS_PANDA.jobParamsTable WHERE PandaID=:PandaID"
            self.cur.execute(sqlJobP+comment, varMap)
            for clobJobP, in self.cur:
                try:
                    job.jobParameters = clobJobP.read()
                except AttributeError:
                    job.jobParameters = str(clobJobP)
                break
            # Files
            oldSubList = []
            sqlFile = "SELECT %s FROM ATLAS_PANDA.filesTable4 " % FileSpec.columnNames()
            sqlFile+= "WHERE PandaID=:PandaID"
            self.cur.arraysize = 10000
            self.cur.execute(sqlFile+comment, varMap)
            resFs = self.cur.fetchall()
            for resF in resFs:
                file = FileSpec()
                file.pack(resF)
                # reset GUID to trigger LRC/LFC scanning
                if file.status == 'missing':
                    file.GUID = None
                # collect old subs
                if job.prodSourceLabel in ['managed','test'] and file.type in ['output','log'] \
                       and re.search('_sub\d+$',file.destinationDBlock) is not None:
                    if file.destinationDBlock not in oldSubList:
                        oldSubList.append(file.destinationDBlock)
                # reset status, destinationDBlock and dispatchDBlock
                if job.lockedby != 'jedi':
                    file.status         ='unknown'
                if job.prodSourceLabel not in ['user','panda']:
                    file.dispatchDBlock = None
                file.destinationDBlock = re.sub('_sub\d+$','',file.destinationDBlock)
                # add file
                job.addFile(file)
                # update files
                sqlF = ("UPDATE ATLAS_PANDA.filesTable4 SET %s" % file.bindUpdateChangesExpression()) + "WHERE row_ID=:row_ID"
                varMap = file.valuesMap(onlyChanged=True)
                if varMap != {}:
                    varMap[':row_ID'] = file.row_ID
                    tmpLog.debug(sqlF+comment+str(varMap))
                    self.cur.execute(sqlF+comment, varMap)
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # record status change
            try:
                self.recordStatusChange(job.PandaID,job.jobStatus,jobInfo=job)
            except Exception:
                _logger.error('recordStatusChange in resetJobs')
            self.push_job_status_message(job, job.PandaID, job.jobStatus)
            if getOldSubs:
                return job,oldSubList
            tmpLog.debug('done')
            return job
        except Exception:
            # roll back
            self._rollback()
            # error report
            type, value, traceBack = sys.exc_info()
            tmpLog.error("%s %s" % (type,value))
            return None


    # reset jobs in jobsDefined
    def resetDefinedJob(self,pandaID,keepSite=False,getOldSubs=False):
        comment = ' /* DBProxy.resetDefinedJob */'
        _logger.debug("resetDefinedJob : %s" % pandaID)
        sql1  = "UPDATE ATLAS_PANDA.jobsDefined4 SET "
        sql1 += "jobStatus=:newJobStatus,"
        sql1 += "modificationTime=CURRENT_DATE,"
        sql1 += "dispatchDBlock=NULL,"
        sql1 += "computingElement=NULL"
        sql1 += " WHERE PandaID=:PandaID AND (jobStatus=:oldJobStatus1 OR jobStatus=:oldJobStatus2)"
        sql2 = "SELECT %s FROM ATLAS_PANDA.jobsDefined4 " % JobSpec.columnNames()
        sql2+= "WHERE PandaID=:PandaID"
        try:
            oldSubList = []
            # begin transaction
            self.conn.begin()
            # update
            varMap = {}
            varMap[':PandaID'] = pandaID
            varMap[':newJobStatus']  = 'defined'
            varMap[':oldJobStatus1'] = 'assigned'
            varMap[':oldJobStatus2'] = 'defined'
            self.cur.execute(sql1+comment,varMap)
            retU = self.cur.rowcount
            # not found
            updatedFlag = False
            job = None
            if retU == 0:
                _logger.debug("resetDefinedJob : Not found %s" % pandaID)
            else:
                # select
                varMap = {}
                varMap[':PandaID'] = pandaID
                self.cur.arraysize = 10
                self.cur.execute(sql2+comment,varMap)
                res = self.cur.fetchone()
                # not found
                if res is None:
                    raise RuntimeError('Could not SELECT : PandaID=%s' % pandaID)
                # instantiate Job
                job = JobSpec()
                job.pack(res)
                # do nothing for analysis jobs
                if job.prodSourceLabel in ['user','panda']:
                    _logger.debug('resetDefinedJob : rollback since PandaID=%s is analysis job' % pandaID)
                    # roll back
                    self._rollback()
                    return None
                job.dispatchDBlock = None
                if (not keepSite) and job.relocationFlag not in [1,2]:
                    # erase old assignment
                    job.computingSite = None
                job.computingElement = None
                # job parameters
                sqlJobP = "SELECT jobParameters FROM ATLAS_PANDA.jobParamsTable WHERE PandaID=:PandaID"
                self.cur.execute(sqlJobP+comment, varMap)
                for clobJobP, in self.cur:
                    try:
                        job.jobParameters = clobJobP.read()
                    except AttributeError:
                        job.jobParameters = str(clobJobP)
                    break
                # Files
                sqlFile = "SELECT %s FROM ATLAS_PANDA.filesTable4 " % FileSpec.columnNames()
                sqlFile+= "WHERE PandaID=:PandaID"
                self.cur.arraysize = 10000
                self.cur.execute(sqlFile+comment, varMap)
                resFs = self.cur.fetchall()
                for resF in resFs:
                    file = FileSpec()
                    file.pack(resF)
                    # collect old subs
                    if job.prodSourceLabel in ['managed','test'] and file.type in ['output','log'] \
                           and re.search('_sub\d+$',file.destinationDBlock) is not None:
                        if file.destinationDBlock not in oldSubList:
                            oldSubList.append(file.destinationDBlock)
                    # reset status, destinationDBlock and dispatchDBlock
                    if job.lockedby != 'jedi':
                        file.status         ='unknown'
                    file.dispatchDBlock = None
                    file.destinationDBlock = re.sub('_sub\d+$','',file.destinationDBlock)
                    # add file
                    job.addFile(file)
                    # update files
                    sqlF = ("UPDATE ATLAS_PANDA.filesTable4 SET %s" % file.bindUpdateChangesExpression()) + "WHERE row_ID=:row_ID"
                    varMap = file.valuesMap(onlyChanged=True)
                    if varMap != {}:
                        varMap[':row_ID'] = file.row_ID
                        _logger.debug(sqlF+comment+str(varMap))
                        self.cur.execute(sqlF+comment, varMap)
                updatedFlag = True
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # record status change
            try:
                if updatedFlag:
                    self.recordStatusChange(job.PandaID,job.jobStatus,jobInfo=job)
                    self.push_job_status_message(job, job.PandaID, job.jobStatus)
            except Exception:
                _logger.error('recordStatusChange in resetDefinedJobs')
            if getOldSubs:
                return job,oldSubList
            return job
        except Exception:
            # error report
            type, value, traceBack = sys.exc_info()
            _logger.error("resetDefinedJobs : %s %s" % (type,value))
            # roll back
            self._rollback()
            return None


    # kill job
    def killJob(self,pandaID,user,code,prodManager,getUserInfo=False,wgProdRole=[],killOpts=[]):
        # code
        # 2  : expire
        # 3  : aborted
        # 4  : expire in waiting
        # 7  : retry by server
        # 8  : rebrokerage
        # 9  : force kill
        # 10 : fast rebrokerage in overloaded PQ
        # 50 : kill by JEDI
        # 51 : reassigned by JEDI
        # 52 : force kill by JEDI
        # 55 : killed since task is (almost) done
        # 60 : workload was terminated by the pilot without actual work
        # 91 : kill user jobs with prod role
        comment = ' /* DBProxy.killJob */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += ' <PandaID={0}>'.format(pandaID)
        tmpLog = LogWrapper(_logger,methodName)
        tmpLog.debug("code=%s role=%s user=%s wg=%s opts=%s" % (code,prodManager,user,wgProdRole,killOpts))
        timeStart = datetime.datetime.utcnow()
        # check PandaID
        try:
            long(pandaID)
        except Exception:
            tmpLog.error("not an integer : %s" % pandaID)
            if getUserInfo:
                return False,{}
            return False
        sql0  = "SELECT prodUserID,prodSourceLabel,jobDefinitionID,jobsetID,workingGroup,specialHandling,jobStatus,taskBufferErrorCode,eventService FROM %s "
        sql0 += "WHERE PandaID=:PandaID "
        sql0 += "FOR UPDATE NOWAIT "
        sql1  = "UPDATE %s SET commandToPilot=:commandToPilot,taskBufferErrorDiag=:taskBufferErrorDiag WHERE PandaID=:PandaID "
        sql1 += "AND (commandToPilot IS NULL OR commandToPilot<>'tobekilled') "
        sql1F = "UPDATE %s SET commandToPilot=:commandToPilot,taskBufferErrorDiag=:taskBufferErrorDiag WHERE PandaID=:PandaID "
        sql2  = "SELECT %s " % JobSpec.columnNames()
        sql2 += "FROM %s WHERE PandaID=:PandaID AND jobStatus<>:jobStatus"
        sql3  = "DELETE FROM %s WHERE PandaID=:PandaID"
        sqlU  = "DELETE FROM ATLAS_PANDA.jobsDefined4 WHERE PandaID=:PandaID AND (jobStatus=:oldJobStatus1 OR jobStatus=:oldJobStatus2)"
        sql4  = "INSERT INTO ATLAS_PANDA.jobsArchived4 (%s) " % JobSpec.columnNames()
        sql4 += JobSpec.bindValuesExpression()
        sqlF  = "UPDATE ATLAS_PANDA.filesTable4 SET status=:status WHERE PandaID=:PandaID AND type IN (:type1,:type2)"
        sqlFMod = "UPDATE ATLAS_PANDA.filesTable4 SET modificationTime=:modificationTime WHERE PandaID=:PandaID"
        sqlMMod = "UPDATE ATLAS_PANDA.metaTable SET modificationTime=:modificationTime WHERE PandaID=:PandaID"
        sqlPMod = "UPDATE ATLAS_PANDA.jobParamsTable SET modificationTime=:modificationTime WHERE PandaID=:PandaID"
        sqlFile = "SELECT %s FROM ATLAS_PANDA.filesTable4 " % FileSpec.columnNames()
        sqlFile+= "WHERE PandaID=:PandaID"
        try:
            flagCommand = False
            flagKilled  = False
            userProdUserID      = ''
            userProdSourceLabel = ''
            userJobDefinitionID = ''
            userJobsetID        = ''
            updatedFlag = False
            # begin transaction
            self.conn.begin()
            for table in ('ATLAS_PANDA.jobsDefined4','ATLAS_PANDA.jobsActive4','ATLAS_PANDA.jobsWaiting4'):
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                # begin transaction
                self.conn.begin()
                # get DN if user is not production DN
                varMap = {}
                varMap[':PandaID'] = pandaID
                self.cur.arraysize = 10
                self.cur.execute((sql0+comment) % table, varMap)
                res = self.cur.fetchone()
                # not found
                if res is None:
                    continue
                # owner?
                def getCN(dn):
                    distinguishedName = ''
                    for line in dn.split('/'):
                        if line.startswith('CN='):
                            distinguishedName = re.sub('^CN=','',line)
                            distinguishedName = re.sub('\d+$','',distinguishedName)
                            distinguishedName = distinguishedName.strip()
                            break
                    if distinguishedName == '':
                        distinguishedName = dn
                    return distinguishedName
                # prevent prod proxy from killing analysis jobs
                userProdUserID,userProdSourceLabel,userJobDefinitionID,userJobsetID,workingGroup,specialHandling,\
                    jobStatusInDB,taskBufferErrorCode,eventService = res
                # check group prod role
                validGroupProdRole = False
                if res[1] in ['managed','test'] and workingGroup != '':
                    for tmpGroupProdRole in wgProdRole:
                        if tmpGroupProdRole == '':
                            continue
                        if re.search('(^|_)'+tmpGroupProdRole+'$',workingGroup,re.I) is not None:
                            validGroupProdRole = True
                            break
                if prodManager:
                    if res[1] in ['user','panda'] and (code not in ['2','4','7','8','9','50','51','52','91',
                                                                    '10']):
                        tmpLog.debug("ignored -> prod proxy tried to kill analysis job type=%s" % res[1])
                        break
                    tmpLog.debug("using prod role")
                elif validGroupProdRole:
                    # WGs with prod role
                    tmpLog.debug("using group prod role for workingGroup=%s" % workingGroup)
                    pass
                else:
                    cn1 = getCN(res[0])
                    cn2 = getCN(user)
                    tmpLog.debug("Owner:%s  - Requester:%s " % (cn1,cn2))
                    if cn1 != cn2:
                        tmpLog.debug("ignored  -> Owner != Requester")
                        break
                # event service
                useEventService =  EventServiceUtils.isEventServiceSH(specialHandling) or \
                    eventService in [EventServiceUtils.jumboJobFlagNumber, EventServiceUtils.coJumboJobFlagNumber]
                useEventServiceMerge = EventServiceUtils.isEventServiceMergeSH(specialHandling)
                # update
                varMap = {}
                varMap[':PandaID'] = pandaID
                varMap[':commandToPilot'] = 'tobekilled'
                varMap[':taskBufferErrorDiag'] = 'killed by %s' % user
                if code in ['2', '9', '10', '52', '51', '60']:
                    # ignore commandToPilot for force kill
                    self.cur.execute((sql1F+comment) % table, varMap)
                elif useEventService or jobStatusInDB in ['merging']:
                    # use force kill for event service or merging
                    self.cur.execute((sql1F+comment) % table, varMap)
                else:
                    self.cur.execute((sql1+comment) % table, varMap)
                retU = self.cur.rowcount
                if retU == 0:
                    continue
                # set flag
                flagCommand = True
                # select
                varMap = {}
                varMap[':PandaID'] = pandaID
                if (userProdSourceLabel in ['managed','test',None] or 'test' in userProdSourceLabel) and code in ['9','52']:
                    # use dummy for force kill
                    varMap[':jobStatus'] = 'dummy'
                elif useEventService or jobStatusInDB in ['merging']:
                    # use dummy for force kill
                    varMap[':jobStatus'] = 'dummy'
                else:
                    varMap[':jobStatus'] = 'running'
                self.cur.arraysize = 10
                self.cur.execute((sql2+comment) % table, varMap)
                res = self.cur.fetchall()
                if len(res) == 0:
                    continue
                # instantiate JobSpec
                job = JobSpec()
                job.pack(res[0])
                # delete
                if table=='ATLAS_PANDA.jobsDefined4':
                    varMap = {}
                    varMap[':PandaID'] = pandaID
                    varMap[':oldJobStatus1'] = 'assigned'
                    varMap[':oldJobStatus2'] = 'defined'
                    self.cur.execute(sqlU+comment, varMap)
                else:
                    varMap = {}
                    varMap[':PandaID'] = pandaID
                    self.cur.execute((sql3+comment) % table, varMap)
                retD = self.cur.rowcount
                if retD == 0:
                    continue
                oldJobStatus = job.jobStatus
                # error code
                if job.jobStatus != 'failed':
                    currentTime = datetime.datetime.utcnow()
                    # set status etc for non-failed jobs
                    if job.endTime in [None,'NULL']:
                        job.endTime = currentTime
                    # reset startTime for aCT where starting jobs don't acutally get started
                    if job.jobStatus == 'starting':
                        job.startTime = job.endTime
                    job.modificationTime = currentTime
                    if code in ['2','4']:
                        # expire
                        job.jobStatus = 'closed'
                        job.jobSubStatus = 'toreassign'
                        job.taskBufferErrorCode = ErrorCode.EC_Expire
                        job.taskBufferErrorDiag = 'expired in {0}. status unchanged since {1}'.format(oldJobStatus,str(job.stateChangeTime))
                    elif code=='3':
                        # aborted
                        job.taskBufferErrorCode = ErrorCode.EC_Aborted
                        job.taskBufferErrorDiag = 'aborted by ExtIF'
                    elif code=='8':
                        # reassigned by rebrokeage
                        job.taskBufferErrorCode = ErrorCode.EC_Reassigned
                        job.taskBufferErrorDiag = 'reassigned to another site by rebrokerage. new %s' % user
                        job.commandToPilot      = None
                    elif code in ['50','52']:
                        # killed by JEDI
                        job.taskBufferErrorCode = ErrorCode.EC_Kill
                        job.taskBufferErrorDiag = user
                    elif code=='51':
                        # reassigned by JEDI
                        job.jobStatus = 'closed'
                        job.jobSubStatus = 'toreassign'
                        job.taskBufferErrorCode = ErrorCode.EC_Kill
                        job.taskBufferErrorDiag = 'reassigned by JEDI'
                    elif code=='55':
                        # killed since task is (almost) done
                        job.jobStatus = 'closed'
                        job.jobSubStatus = 'taskdone'
                        job.taskBufferErrorCode = ErrorCode.EC_Kill
                        job.taskBufferErrorDiag = 'killed since task is (almost) done'
                    elif code=='60':
                        # terminated by the pilot. keep jobSubStatus reported by the pilot
                        job.jobStatus = 'closed'
                        job.taskBufferErrorCode = ErrorCode.EC_Kill
                        job.taskBufferErrorDiag = 'closed by the pilot'
                    elif code == '10':
                        job.jobStatus = 'closed'
                        job.taskBufferErrorCode = ErrorCode.EC_FastRebrokerage
                        job.taskBufferErrorDiag = 'fast rebrokerage due to Nq/Nr overshoot'
                    else:
                        # killed
                        job.taskBufferErrorCode = ErrorCode.EC_Kill
                        job.taskBufferErrorDiag = 'killed by %s' % user
                    # set job status
                    if job.jobStatus != 'closed':
                        job.jobStatus = 'cancelled'
                else:
                    # keep status for failed jobs
                    job.modificationTime = datetime.datetime.utcnow()
                    if code=='7':
                        # retried by server
                        job.taskBufferErrorCode = ErrorCode.EC_Retried
                        job.taskBufferErrorDiag = 'retrying at another site. new %s' % user
                        job.commandToPilot      = None
                job.stateChangeTime = job.modificationTime
                # insert
                self.cur.execute(sql4+comment, job.valuesMap())
                # update file
                varMap = {}
                varMap[':PandaID'] = pandaID
                varMap[':status'] = 'failed'
                varMap[':type1'] = 'output'
                varMap[':type2'] = 'log'
                self.cur.execute(sqlF+comment,varMap)
                # update files,metadata,parametes
                varMap = {}
                varMap[':PandaID'] = pandaID
                varMap[':modificationTime'] = job.modificationTime
                self.cur.execute(sqlFMod+comment,varMap)
                self.cur.execute(sqlMMod+comment,varMap)
                self.cur.execute(sqlPMod+comment,varMap)
                flagKilled = True
                updatedFlag = True
                # update JEDI tables
                if hasattr(panda_config,'useJEDI') and panda_config.useJEDI is True and \
                        job.lockedby == 'jedi' and self.checkTaskStatusJEDI(job.jediTaskID,self.cur):
                    # read files
                    varMap = {}
                    varMap[':PandaID'] = pandaID
                    self.cur.arraysize = 10000
                    self.cur.execute(sqlFile+comment, varMap)
                    resFs = self.cur.fetchall()
                    for resF in resFs:
                        fileSpec = FileSpec()
                        fileSpec.pack(resF)
                        job.addFile(fileSpec)
                    # actions for event service unless it was already retried
                    if taskBufferErrorCode not in [ErrorCode.EC_Reassigned,ErrorCode.EC_Retried,ErrorCode.EC_PilotRetried]:
                        # kill associated consumers for event service
                        if useEventService:
                            self.killEventServiceConsumers(job,True,False)
                            if job.computingSite != EventServiceUtils.siteIdForWaitingCoJumboJobs:
                                self.killUnusedEventServiceConsumers(job, False, killAll=True, checkAttemptNr=True)
                            self.updateRelatedEventServiceJobs(job,True)
                            if not job.notDiscardEvents():
                                self.killUnusedEventRanges(job.jediTaskID,job.jobsetID)
                            if eventService == EventServiceUtils.jumboJobFlagNumber:
                                self.hasDoneEvents(job.jediTaskID, job.PandaID, job, False)
                        elif useEventServiceMerge:
                            self.updateRelatedEventServiceJobs(job,True)
                    # disable reattempt
                    if job.processingType == 'pmerge' and 'keepUnmerged' not in killOpts:
                        self.disableFurtherReattempt(job)
                    # update JEDI
                    self.propagateResultToJEDI(job,self.cur,oldJobStatus)
                break
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            timeDelta = datetime.datetime.utcnow()-timeStart
            tmpLog.debug("com=%s kill=%s time=%s" % (flagCommand,flagKilled,timeDelta.seconds))
            # record status change
            try:
                if updatedFlag:
                    self.recordStatusChange(job.PandaID,job.jobStatus,jobInfo=job)
                    self.push_job_status_message(job, job.PandaID, job.jobStatus)
            except Exception:
                tmpLog.error('recordStatusChange in killJob')
            if getUserInfo:
                return (flagCommand or flagKilled),{'prodUserID':userProdUserID,
                                                    'prodSourceLabel':userProdSourceLabel,
                                                    'jobDefinitionID':userJobDefinitionID,
                                                    'jobsetID':userJobsetID}
            return (flagCommand or flagKilled)
        except Exception:
            self.dumpErrorMessage(_logger,methodName)
            # roll back
            self._rollback()
            timeDelta = datetime.datetime.utcnow()-timeStart
            tmpLog.debug("time=%s" % timeDelta.seconds)
            if getUserInfo:
                return False,{}
            return False


    # peek at job
    def peekJob(self,pandaID,fromDefined,fromActive,fromArchived,fromWaiting,forAnal=False):
        comment = ' /* DBProxy.peekJob */'
        _logger.debug("peekJob : %s" % pandaID)
        # return None for NULL PandaID
        if pandaID in ['NULL','','None',None]:
            return None
        # only int
        try:
            tmpID = int(pandaID)
        except Exception:
            _logger.debug("peekJob : return None for %s:non-integer" % pandaID)
            return None
        sql1_0 = "SELECT %s FROM %s "
        sql1_1 = "WHERE PandaID=:PandaID"
        nTry=3
        for iTry in range(nTry):
            try:
                tables=[]
                if fromDefined:
                    tables.append('ATLAS_PANDA.jobsDefined4')
                if fromActive:
                    tables.append('ATLAS_PANDA.jobsActive4')
                if fromArchived:
                    tables.append('ATLAS_PANDA.jobsArchived4')
                if fromWaiting:
                    tables.append('ATLAS_PANDA.jobsWaiting4')
                if fromDefined:
                    # for jobs which are just reset
                    tables.append('ATLAS_PANDA.jobsDefined4')
                # select
                varMap = {}
                varMap[':PandaID'] = pandaID
                for table in tables:
                    # start transaction
                    self.conn.begin()
                    # select
                    sql = sql1_0 % (JobSpec.columnNames(),table) + sql1_1
                    self.cur.arraysize = 10
                    self.cur.execute(sql+comment, varMap)
                    res = self.cur.fetchall()
                    # commit
                    if not self._commit():
                        raise RuntimeError('Commit error')
                    if len(res) != 0:
                        # Job
                        job = JobSpec()
                        job.pack(res[0])
                        # Files
                        # start transaction
                        self.conn.begin()
                        # select
                        sqlFile = "SELECT %s FROM ATLAS_PANDA.filesTable4 " % FileSpec.columnNames()
                        sqlFile+= "WHERE PandaID=:PandaID"
                        self.cur.arraysize = 10000
                        self.cur.execute(sqlFile+comment, varMap)
                        resFs = self.cur.fetchall()
                        # metadata
                        resMeta = None
                        if table == 'ATLAS_PANDA.jobsArchived4' or forAnal:
                            # read metadata only for finished/failed production jobs
                            sqlMeta = "SELECT metaData FROM ATLAS_PANDA.metaTable WHERE PandaID=:PandaID"
                            self.cur.execute(sqlMeta+comment, varMap)
                            for clobMeta, in self.cur:
                                if clobMeta is not None:
                                    try:
                                        resMeta = clobMeta.read()
                                    except AttributeError:
                                        resMeta = str(clobMeta)
                                break
                        # job parameters
                        job.jobParameters = None
                        sqlJobP = "SELECT jobParameters FROM ATLAS_PANDA.jobParamsTable WHERE PandaID=:PandaID"
                        varMap = {}
                        varMap[':PandaID'] = job.PandaID
                        self.cur.execute(sqlJobP+comment, varMap)
                        for clobJobP, in self.cur:
                            if clobJobP is not None:
                                try:
                                    job.jobParameters = clobJobP.read()
                                except AttributeError:
                                    job.jobParameters = str(clobJobP)
                            break
                        # commit
                        if not self._commit():
                            raise RuntimeError('Commit error')
                        # set files
                        for resF in resFs:
                            file = FileSpec()
                            file.pack(resF)
                            job.addFile(file)
                        # set metadata
                        job.metadata = resMeta
                        return job
                _logger.debug("peekJob() : PandaID %s not found" % pandaID)
                return None
            except Exception:
                # roll back
                self._rollback()
                if iTry+1 < nTry:
                    _logger.debug("peekJob : %s retry : %s" % (pandaID,iTry))
                    time.sleep(random.randint(10,20))
                    continue
                type, value, traceBack = sys.exc_info()
                _logger.error("peekJob : %s %s %s" % (pandaID,type,value))
                # return None for analysis
                if forAnal:
                    return None
                # return 'unknown'
                job = JobSpec()
                job.PandaID = pandaID
                job.jobStatus = 'unknown'
                return job


    # get PandaID with jobexeID
    def getPandaIDwithJobExeID(self,jobexeID):
        comment = ' /* DBProxy.getPandaIDwithJobExeID */'
        _logger.debug("getPandaIDwithJobExeID : %s" % jobexeID)
        failedRetVal = (None,None,'')
        # return for wrong jobexeID
        if jobexeID in ['NULL','','None',None]:
            return failedRetVal
        # SQL
        sql  = "SELECT PandaID,jobDefinitionID,jobName FROM ATLAS_PANDA.jobsWaiting4 "
        sql += "WHERE jobExecutionID=:jobexeID AND prodSourceLabel=:prodSourceLabel "
        sql += "AND jobStatus=:jobStatus "
        varMap = {}
        varMap[':jobexeID'] = jobexeID
        varMap[':jobStatus'] = 'pending'
        varMap[':prodSourceLabel'] = 'managed'
        try:
            # start transaction
            self.conn.begin()
            # select
            self.cur.arraysize = 10
            self.cur.execute(sql+comment,varMap)
            res = self.cur.fetchone()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # not found
            if res is None:
                _logger.debug("getPandaIDwithJobExeID : jobexeID %s not found" % jobexeID)
                return failedRetVal
            _logger.debug("getPandaIDwithJobExeID : %s -> %s" % (jobexeID,str(res)))
            return res
        except Exception:
            # roll back
            self._rollback()
            errtype, errvalue = sys.exc_info()[:2]
            _logger.error("getPandaIDwithJobExeID : %s %s %s" % (jobexeID,errtype, errvalue))
            return failedRetVal


    # get PandaIDs with TaskID
    def getPandaIDsWithTaskID(self,jediTaskID):
        comment = ' /* DBProxy.getPandaIDsWithTaskID */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        tmpLog = LogWrapper(_logger,methodName+" <jediTaskID={0}>".format(jediTaskID))
        tmpLog.debug("start")
        # SQL
        sql  = "SELECT PandaID FROM ATLAS_PANDA.jobsWaiting4 "
        sql += "WHERE jediTaskID=:jediTaskID "
        sql += "UNION "
        sql += "SELECT PandaID FROM ATLAS_PANDA.jobsDefined4 "
        sql += "WHERE jediTaskID=:jediTaskID "
        sql += "UNION "
        sql += "SELECT PandaID FROM ATLAS_PANDA.jobsActive4 "
        sql += "WHERE jediTaskID=:jediTaskID "
        sql += "UNION "
        sql += "SELECT PandaID FROM ATLAS_PANDA.jobsArchived4 "
        sql += "WHERE jediTaskID=:jediTaskID "
        varMap = {}
        varMap[':jediTaskID'] = jediTaskID
        try:
            # start transaction
            self.conn.begin()
            # select
            self.cur.arraysize = 1000000
            self.cur.execute(sql+comment,varMap)
            res = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            retList = []
            for pandaID, in res:
                retList.append(pandaID)

            tmpLog.debug("found {0} IDs".format(len(retList)))
            return retList
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            return []


    # get express jobs
    def getExpressJobs(self,dn):
        comment = ' /* DBProxy.getExpressJobs */'
        _logger.debug("getExpressJobs : %s" % dn)
        sqlX  = "SELECT specialHandling,COUNT(*) FROM %s "
        sqlX += "WHERE prodUserName=:prodUserName AND prodSourceLabel=:prodSourceLabel1 "
        sqlX += "AND specialHandling IS NOT NULL "
        sqlXJob  = "SELECT PandaID,jobStatus,prodSourceLabel,modificationTime,jobDefinitionID,jobsetID,startTime,endTime FROM %s "
        sqlXJob += "WHERE prodUserName=:prodUserName AND prodSourceLabel=:prodSourceLabel1 "
        sqlXJob += "AND specialHandling IS NOT NULL AND specialHandling=:specialHandling "
        sqlQ  = sqlX
        sqlQ += "GROUP BY specialHandling "
        sqlQJob  = sqlXJob
        sqlA  = sqlX
        sqlA += "AND modificationTime>:modificationTime GROUP BY specialHandling "
        sqlAJob  = sqlXJob
        sqlAJob += "AND modificationTime>:modificationTime "
        try:
            # get compact DN
            compactDN = self.cleanUserID(dn)
            if compactDN in ['','NULL',None]:
                compactDN = dn
            expressStr = 'express'
            activeExpressU = []
            timeUsageU  = datetime.timedelta(0)
            executionTimeU = datetime.timedelta(hours=1)
            jobCreditU  = 3
            timeCreditU = executionTimeU * jobCreditU
            timeNow   = datetime.datetime.utcnow()
            timeLimit = timeNow - datetime.timedelta(hours=6)
            # loop over tables
            for table in ['ATLAS_PANDA.jobsDefined4','ATLAS_PANDA.jobsActive4','ATLAS_PANDA.jobsArchived4']:
                varMap = {}
                varMap[':prodUserName'] = compactDN
                varMap[':prodSourceLabel1'] = 'user'
                if table == 'ATLAS_PANDA.jobsArchived4':
                    varMap[':modificationTime'] = timeLimit
                    sql = sqlA % table
                    sqlJob = sqlAJob % table
                else:
                    sql = sqlQ % table
                    sqlJob = sqlQJob % table
                # start transaction
                self.conn.begin()
                # get the number of jobs for each specialHandling
                self.cur.arraysize = 10
                _logger.debug(sql+comment+str(varMap))
                self.cur.execute(sql+comment, varMap)
                res = self.cur.fetchall()
                _logger.debug("getExpressJobs %s" % str(res))
                for specialHandling,countJobs in res:
                    if specialHandling is None:
                        continue
                    # look for express jobs
                    if expressStr in specialHandling:
                        varMap[':specialHandling'] = specialHandling
                        self.cur.arraysize = 1000
                        self.cur.execute(sqlJob+comment, varMap)
                        resJobs = self.cur.fetchall()
                        _logger.debug("getExpressJobs %s" % str(resJobs))
                        for tmp_PandaID,tmp_jobStatus,tmp_prodSourceLabel,tmp_modificationTime,\
                                tmp_jobDefinitionID,tmp_jobsetID,tmp_startTime,tmp_endTime \
                                in resJobs:
                            # collect active jobs
                            if tmp_jobStatus not in ['finished','failed','cancelled','closed']:
                                activeExpressU.append((tmp_PandaID,tmp_jobsetID,tmp_jobDefinitionID))
                            # get time usage
                            if tmp_jobStatus not in ['defined','activated']:
                                # check only jobs which actually use or used CPU on WN
                                if tmp_startTime is not None:
                                    # running or not
                                    if tmp_endTime is None:
                                        # job got started before/after the time limit
                                        if timeLimit > tmp_startTime:
                                            timeDelta = timeNow - timeLimit
                                        else:
                                            timeDelta = timeNow - tmp_startTime
                                    else:
                                        # job got started before/after the time limit
                                        if timeLimit > tmp_startTime:
                                            timeDelta = tmp_endTime - timeLimit
                                        else:
                                            timeDelta = tmp_endTime - tmp_startTime
                                    # add
                                    if timeDelta > datetime.timedelta(0):
                                        timeUsageU += timeDelta
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
            # check quota
            rRet = True
            rRetStr = ''
            rQuota = 0
            if len(activeExpressU) >= jobCreditU:
                rRetStr += "The number of queued runXYZ exceeds the limit = %s. " % jobCreditU
                rRet = False
            if timeUsageU >= timeCreditU:
                rRetStr += "The total execution time for runXYZ exceeds the limit = %s min. " % (timeCreditU.seconds / 60)
                rRet = False
            # calculate available quota
            if rRet:
                tmpQuota = jobCreditU - len(activeExpressU) - timeUsageU.seconds/executionTimeU.seconds
                if tmpQuota < 0:
                    rRetStr += "Quota for runXYZ exceeds. "
                    rRet = False
                else:
                    rQuota = tmpQuota
            # return
            retVal = {'status':rRet,'quota':rQuota,'output':rRetStr,'usage':timeUsageU,'jobs':activeExpressU}
            _logger.debug("getExpressJobs : %s" % str(retVal))
            return retVal
        except Exception:
            # roll back
            self._rollback()
            errtype,errvalue = sys.exc_info()[:2]
            _logger.error("getExpressJobs : %s %s" % (errtype,errvalue))
            return None


    # get active debug jobs
    def getActiveDebugJobs(self,dn=None,workingGroup=None,prodRole=False):
        comment = ' /* DBProxy.getActiveDebugJobs */'
        _logger.debug("getActiveDebugJobs : DN={0} wg={1} prodRole={2}".format(dn,workingGroup,prodRole))
        varMap = {}
        sqlX  = "SELECT PandaID,jobStatus,specialHandling FROM %s "
        sqlX += "WHERE "
        if prodRole:
            pass
        elif workingGroup is not None:
            sqlX += "UPPER(workingGroup) IN (:wg1,:wg2) AND "
            varMap[':wg1'] = 'AP_{0}'.format(workingGroup.upper())
            varMap[':wg2'] = 'GP_{0}'.format(workingGroup.upper())
        else:
            sqlX += "prodUserName=:prodUserName AND "
            # get compact DN
            compactDN = self.cleanUserID(dn)
            if compactDN in ['','NULL',None]:
                compactDN = dn
            varMap[':prodUserName'] = compactDN
        sqlX += "specialHandling IS NOT NULL "
        try:
            debugStr = 'debug'
            activeDebugJobs = []
            # loop over tables
            for table in ['ATLAS_PANDA.jobsDefined4','ATLAS_PANDA.jobsActive4']:
                sql = sqlX % table
                # start transaction
                self.conn.begin()
                # get jobs with specialHandling
                self.cur.arraysize = 100000
                self.cur.execute(sql+comment, varMap)
                res = self.cur.fetchall()
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                # loop over all PandaIDs
                for pandaID,jobStatus,specialHandling in res:
                    if specialHandling is None:
                        continue
                    # only active jobs
                    if jobStatus not in ['defined','activated','running','sent','starting']:
                        continue
                    # look for debug jobs
                    if debugStr in specialHandling and pandaID not in activeDebugJobs:
                        activeDebugJobs.append(pandaID)
            # return
            activeDebugJobs.sort()
            _logger.debug("getActiveDebugJobs : DN=%s -> %s" % (dn,str(activeDebugJobs)))
            return activeDebugJobs
        except Exception:
            # roll back
            self._rollback()
            errtype,errvalue = sys.exc_info()[:2]
            _logger.error("getActiveDebugJobs : %s %s" % (errtype,errvalue))
            return None


    # set debug mode
    def setDebugMode(self,dn,pandaID,prodManager,modeOn,workingGroup):
        comment = ' /* DBProxy.setDebugMode */'
        _logger.debug("turnDebugModeOn : dn=%s id=%s prod=%s wg=%s mode=%s" % (dn,pandaID,prodManager,workingGroup,modeOn))
        sqlX  = "SELECT prodUserName,jobStatus,specialHandling,workingGroup FROM %s "
        sqlX += "WHERE PandaID=:PandaID "
        sqlU  = "UPDATE %s SET specialHandling=:specialHandling "
        sqlU += "WHERE PandaID=:PandaID "
        try:
            # get compact DN
            compactDN = self.cleanUserID(dn)
            if compactDN in ['','NULL',None]:
                compactDN = dn
            debugStr = 'debug'
            retStr = ''
            retCode = False
            # loop over tables
            for table in ['ATLAS_PANDA.jobsDefined4','ATLAS_PANDA.jobsActive4']:
                varMap = {}
                varMap[':PandaID'] = pandaID
                sql = sqlX % table
                # start transaction
                self.conn.begin()
                # get jobs with specialHandling
                self.cur.arraysize = 10
                self.cur.execute(sql+comment, varMap)
                res = self.cur.fetchone()
                # not found
                if res is None:
                    retStr = 'PandaID={0} not found in active DB'.format(pandaID)
                    # commit
                    if not self._commit():
                        raise RuntimeError('Commit error')
                    continue
                prodUserName,jobStatus,specialHandling,wGroup = res
                # not active
                changeableState = ['defined','activated','running','sent','starting','assigned']
                if jobStatus not in changeableState:
                    retStr = 'Cannot set debugMode since the job status is {0} which is not in one of {1}'.format(jobStatus,
                                                                                                                  str(changeableState))
                    # commit
                    if not self._commit():
                        raise RuntimeError('Commit error')
                    break
                # extract workingGroup
                try:
                    wGroup = wGroup.split('_')[-1]
                    wGroup = wGroup.lower()
                except Exception:
                    pass
                # not owner
                notOwner = False
                if not prodManager:
                    if workingGroup is not None:
                        if workingGroup.lower() != wGroup:
                            retStr = 'Permission denied. Not the production manager for workingGroup={0}'.format(wGroup)
                            notOwner = True
                    else:
                        if prodUserName != compactDN:
                            retStr = 'Permission denied. Not the owner or production manager'
                            notOwner = True
                    if notOwner:
                        # commit
                        if not self._commit():
                            raise RuntimeError('Commit error')
                        break
                # set specialHandling
                updateSH = True
                if specialHandling in [None,'']:
                    if modeOn:
                        # set debug mode
                        specialHandling = debugStr
                    else:
                        # already disabled debug mode
                        updateSH = False
                elif debugStr in specialHandling:
                    if modeOn:
                        # already in debug mode
                        updateSH = False
                    else:
                        # disable debug mode
                        specialHandling = re.sub(debugStr,'',specialHandling)
                        specialHandling = re.sub(',,',',',specialHandling)
                        specialHandling = re.sub('^,','',specialHandling)
                        specialHandling = re.sub(',$','',specialHandling)
                else:
                    if modeOn:
                        # set debug mode
                        specialHandling = debugStr
                    else:
                        # already disabled debug mode
                        updateSH = False

                # no update
                if not updateSH:
                    retStr = 'Already set accordingly'
                    # commit
                    if not self._commit():
                        raise RuntimeError('Commit error')
                    break
                # update
                varMap = {}
                varMap[':PandaID'] = pandaID
                varMap[':specialHandling'] = specialHandling
                self.cur.execute((sqlU+comment) % table, varMap)
                retD = self.cur.rowcount
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                if retD == 0:
                    retStr = 'Failed to update DB'
                else:
                    retStr = 'Succeeded'
                    break
            # return
            _logger.debug("setDebugMode : %s %s -> %s" % (dn,pandaID,retStr))
            return retStr
        except Exception:
            # roll back
            self._rollback()
            errtype,errvalue = sys.exc_info()[:2]
            _logger.error("setDebugMode : %s %s" % (errtype,errvalue))
            return None


    # get PandaID with destinationDBlock
    def getPandaIDwithDestDBlock(self,destinationDBlock):
        comment = ' /* DBProxy.getPandaIDwithDestDBlock */'
        _logger.debug("getPandaIDwithDestDBlock : %s" % destinationDBlock)
        try:
            sqlP  = "SELECT /*+ index(tab FILESTABLE4_DESTDBLOCK_IDX) */ PandaID FROM ATLAS_PANDA.filesTable4 tab "
            sqlP += "WHERE type IN (:type1,:type2) AND destinationDBlock=:destinationDBlock AND rownum<=1"
            # start transaction
            self.conn.begin()
            pandaID = None
            varMap = {}
            varMap[':type1'] = 'log'
            varMap[':type2'] = 'output'
            varMap[':destinationDBlock'] = destinationDBlock
            # select
            self.cur.arraysize = 10
            self.cur.execute(sqlP+comment, varMap)
            res = self.cur.fetchone()
            # append
            if res is not None:
                pandaID, = res
            # commit to release tables
            if not self._commit():
                raise RuntimeError('Commit error')
            # return
            return pandaID
        except Exception:
            # roll back
            self._rollback()
            errType,errValue = sys.exc_info()[:2]
            _logger.error("getPandaIDwithDestDBlock : %s %s" % (errType,errValue))
            # return empty list
            return None


    # get destSE with destinationDBlock
    def getDestSEwithDestDBlock(self,destinationDBlock):
        comment = ' /* DBProxy.getDestSEwithDestDBlock */'
        _logger.debug("getDestSEwithDestDBlock : %s" % destinationDBlock)
        try:
            sqlP  = "SELECT /*+ index(tab FILESTABLE4_DESTDBLOCK_IDX) */ destinationSE,destinationDBlockToken FROM ATLAS_PANDA.filesTable4 tab "
            sqlP += "WHERE type IN (:type1,:type2) AND destinationDBlock=:destinationDBlock AND rownum<=1"
            # start transaction
            self.conn.begin()
            varMap = {}
            varMap[':type1'] = 'log'
            varMap[':type2'] = 'output'
            varMap[':destinationDBlock'] = destinationDBlock
            # select
            self.cur.arraysize = 10
            self.cur.execute(sqlP+comment, varMap)
            res = self.cur.fetchone()
            # append
            destinationSE = None
            destinationDBlockToken = None
            if res is not None:
                destinationSE,destinationDBlockToken = res
            # commit to release tables
            if not self._commit():
                raise RuntimeError('Commit error')
            # return
            return destinationSE,destinationDBlockToken
        except Exception:
            # roll back
            self._rollback()
            errType,errValue = sys.exc_info()[:2]
            _logger.error("getDestSEwithDestDBlock : %s %s" % (errType,errValue))
            # return empty list
            return None,None


    # get number of activated/defined jobs with output datasets
    def getNumWaitingJobsWithOutDS(self,outputDSs):
        comment = ' /* DBProxy.getNumWaitingJobsWithOutDS */'
        _logger.debug("getNumWaitingJobsWithOutDS : %s" % str(outputDSs))
        try:
            sqlD  = "SELECT distinct destinationDBlock FROM ATLAS_PANDA.filesTable4 "
            sqlD += "WHERE type IN (:type1,:type2) AND dataset=:dataset AND status IN (:status1,:status2)"
            sqlP  = "SELECT /*+ index(tab FILESTABLE4_DESTDBLOCK_IDX) */ PandaID FROM ATLAS_PANDA.filesTable4 tab "
            sqlP += "WHERE type IN (:type1,:type2) AND destinationDBlock=:destinationDBlock AND status IN (:status1,:status2) AND rownum<=1"
            sqlJ  = "SELECT jobDefinitionID,taskID,prodUserName,jobStatus,prodSourceLabel FROM %s "
            sqlJ += "WHERE PandaID=:PandaID"
            sqlC  = "SELECT count(*) FROM ATLAS_PANDA.jobsActive4 "
            sqlC += "WHERE jobDefinitionID=:jobDefinitionID AND prodUserName=:prodUserName AND jobStatus IN (:jobStatus1)"
            # start transaction
            self.conn.begin()
            # get sub datasets
            subDSList = []
            for outputDS in outputDSs:
                varMap = {}
                varMap[':type1'] = 'log'
                varMap[':type2'] = 'output'
                varMap[':status1'] = 'unknown'
                varMap[':status2'] = 'pending'
                varMap[':dataset'] = outputDS
                # select
                self.cur.arraysize = 1000
                self.cur.execute(sqlD+comment, varMap)
                resList = self.cur.fetchall()
                # append
                for destinationDBlock, in resList:
                    subDSList.append(destinationDBlock)
            # get PandaIDs
            pandaIDs = []
            for subDS in subDSList:
                varMap = {}
                varMap[':type1'] = 'log'
                varMap[':type2'] = 'output'
                varMap[':status1'] = 'unknown'
                varMap[':status2'] = 'pending'
                varMap[':destinationDBlock'] = subDS
                # select
                self.cur.arraysize = 10
                self.cur.execute(sqlP+comment, varMap)
                res = self.cur.fetchone()
                # append
                if res is not None:
                    pandaID, = res
                    pandaIDs.append(pandaID)
            # commit to release tables
            if not self._commit():
                raise RuntimeError('Commit error')
            # loop over all PandaIDs
            jobInfos = []
            for pandaID in pandaIDs:
                varMap = {}
                varMap[':PandaID'] = pandaID
                # start transaction
                self.conn.begin()
                # get jobID,nJobs,jobStatus,userName
                res = None
                for table in ['ATLAS_PANDA.jobsActive4','ATLAS_PANDA.jobsDefined4']:
                    # select
                    self.cur.arraysize = 10
                    self.cur.execute((sqlJ % table)+comment,varMap)
                    res = self.cur.fetchone()
                    if res is not None:
                        break
                # commit to release tables
                if not self._commit():
                    raise RuntimeError('Commit error')
                # not found
                if res is None:
                    continue
                # append
                jobInfos.append(res)
            # no jobs
            if jobInfos == []:
                _logger.error("getNumWaitingJobsWithOutDS : no jobs found")
                return False,{}
            # loop over all jobIDs
            retMap = {}
            for jobID,taskID,prodUserName,jobStatus,prodSourceLabel in jobInfos:
                if jobID in retMap:
                    continue
                retMap[jobID] = {}
                retMap[jobID]['nJobs'] = taskID
                retMap[jobID]['sourceLabel'] = prodSourceLabel
                # don't check # of activated
                if jobStatus in ['defined']:
                    retMap[jobID]['activated'] = False
                    retMap[jobID]['nActs'] = 0
                    continue
                retMap[jobID]['activated'] = True
                # get # of activated jobs
                varMap = {}
                varMap[':prodUserName'] = prodUserName
                varMap[':jobDefinitionID'] = jobID
                varMap[':jobStatus1'] = 'activated'
                # start transaction
                self.conn.begin()
                # select
                self.cur.arraysize = 10
                self.cur.execute(sqlC+comment, varMap)
                res = self.cur.fetchone()
                # commit to release tables
                if not self._commit():
                    raise RuntimeError('Commit error')
                if res is None:
                    _logger.error("getNumWaitingJobsWithOutDS : cannot get # of activated for %s:%s" % \
                                  (jobID,prodUserName))
                    return False,{}
                # set # of activated
                nActs, = res
                retMap[jobID]['nActs'] = nActs
            # return
            _logger.debug("getNumWaitingJobsWithOutDS -> %s" % str(retMap))
            return True,retMap
        except Exception:
            # roll back
            self._rollback()
            errType,errValue = sys.exc_info()[:2]
            _logger.error("getNumWaitingJobsWithOutDS : %s %s" % (errType,errValue))
            # return empty list
            return False,{}


    # get slimmed file info with PandaIDs
    def getSlimmedFileInfoPandaIDs(self,pandaIDs):
        comment = ' /* DBProxy.getSlimmedFileInfoPandaIDs */'
        _logger.debug("getSlimmedFileInfoPandaIDs : %s len=%s" % (pandaIDs[0],len(pandaIDs)))
        try:
            sqlL  = "SELECT lfn,type,dataset FROM ATLAS_PANDA.filesTable4 WHERE PandaID=:PandaID"
            sqlA  = "SELECT /*+ INDEX(tab FILES_ARCH_PANDAID_IDX)*/ lfn,type,dataset FROM ATLAS_PANDAARCH.filesTable_ARCH tab "
            sqlA += "WHERE PandaID=:PandaID AND modificationTime>(CURRENT_DATE-60)"
            retMap = {'inDS':[],'outDS':[]}
            # start transaction
            self.conn.begin()
            # select
            for pandaID in pandaIDs:
                # make sql
                varMap = {}
                varMap[':PandaID'] = pandaID
                # select
                self.cur.arraysize = 10000
                self.cur.execute(sqlL+comment, varMap)
                resList = self.cur.fetchall()
                # try archived if not found in filesTable4
                if len(resList) == 0:
                    self.cur.execute(sqlA+comment, varMap)
                    resList = self.cur.fetchall()
                # append
                for tmp_lfn,tmp_type,tmp_dataset in resList:
                    # skip lib.tgz
                    if tmp_lfn.endswith('.lib.tgz'):
                        continue
                    if tmp_type == 'input':
                        if tmp_dataset not in retMap['inDS']:
                            retMap['inDS'].append(tmp_dataset)
                    elif tmp_type == 'output':
                        if tmp_dataset not in retMap['outDS']:
                            retMap['outDS'].append(tmp_dataset)
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            _logger.debug("getSlimmedFileInfoPandaIDs : %s" % str(retMap))
            return retMap
        except Exception:
            # roll back
            self._rollback()
            type, value, traceBack = sys.exc_info()
            _logger.error("getSlimmedFileInfoPandaIDs : %s %s" % (type,value))
            # return empty list
            return {}


    # get JobIDs in a time range
    def getJobIDsInTimeRange(self,dn,timeRange,retJobIDs):
        comment = ' /* DBProxy.getJobIDsInTimeRange */'
        _logger.debug("getJobIDsInTimeRange : %s %s" % (dn,timeRange.strftime('%Y-%m-%d %H:%M:%S')))
        try:
            # get compact DN
            compactDN = self.cleanUserID(dn)
            if compactDN in ['','NULL',None]:
                compactDN = dn
            tables = ['ATLAS_PANDA.jobsArchived4','ATLAS_PANDA.jobsActive4','ATLAS_PANDA.jobsWaiting4','ATLAS_PANDA.jobsDefined4']
            # select
            for table in tables:
                # make sql
                if table == 'ATLAS_PANDA.jobsArchived4':
                    sql  = 'SELECT /*+ INDEX_RS_ASC(TAB("JOBSARCHIVED4"."PRODUSERNAME")) NO_INDEX(TAB("JOBSARCHIVED4"."MODIFICATIONTIME")) */ jobDefinitionID FROM %s tab ' % table
                elif table == 'ATLAS_PANDA.jobsActive4':
                    sql  = 'SELECT /*+ INDEX_RS_ASC(TAB("JOBSACTIVE4"."PRODUSERNAME")) NO_INDEX(TAB("JOBSACTIVE4"."MODIFICATIONTIME")) */ jobDefinitionID FROM %s tab ' % table
                else:
                    sql  = "SELECT jobDefinitionID FROM %s " % table
                sql += "WHERE prodUserName=:prodUserName AND modificationTime>:modificationTime "
                sql += "AND prodSourceLabel=:prodSourceLabel AND lockedBy<>:ngLock GROUP BY jobDefinitionID"
                varMap = {}
                varMap[':prodUserName'] = compactDN
                varMap[':prodSourceLabel']  = 'user'
                varMap[':ngLock'] = 'jedi'
                varMap[':modificationTime'] = timeRange
                # start transaction
                self.conn.begin()
                # select
                self.cur.arraysize = 10000
                _logger.debug(sql+comment+str(varMap))
                self.cur.execute(sql+comment, varMap)
                resList = self.cur.fetchall()
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                # append
                for tmpID, in resList:
                    if tmpID not in retJobIDs:
                        retJobIDs.append(tmpID)
            _logger.debug("getJobIDsInTimeRange : %s" % str(retJobIDs))
            return retJobIDs
        except Exception:
            # roll back
            self._rollback()
            type, value, traceBack = sys.exc_info()
            _logger.error("getJobIDsInTimeRange : %s %s" % (type,value))
            # return empty list
            return []


    # get PandaIDs for a JobID
    def getPandIDsWithJobID(self,dn,jobID,idStatus,nJobs):
        comment = ' /* DBProxy.getPandIDsWithJobID */'
        _logger.debug("getPandIDsWithJobID : %s %s" % (dn,jobID))
        try:
            # get compact DN
            compactDN = self.cleanUserID(dn)
            if compactDN in ['','NULL',None]:
                compactDN = dn
            tables = ['ATLAS_PANDA.jobsDefined4','ATLAS_PANDA.jobsActive4','ATLAS_PANDA.jobsWaiting4','ATLAS_PANDA.jobsArchived4']
            buildJobID = None
            # select
            for table in tables:
                # skip if all jobs have already been gotten
                if nJobs > 0 and len(idStatus) >= nJobs:
                    continue
                # make sql
                sql  = "SELECT PandaID,jobStatus,commandToPilot,prodSourceLabel,taskBufferErrorCode FROM %s " % table
                sql += "WHERE prodUserName=:prodUserName AND jobDefinitionID=:jobDefinitionID "
                sql += "AND prodSourceLabel in (:prodSourceLabel1,:prodSourceLabel2)"
                varMap = {}
                varMap[':prodUserName'] = compactDN
                varMap[':jobDefinitionID']  = jobID
                varMap[':prodSourceLabel1'] = 'user'
                varMap[':prodSourceLabel2'] = 'panda'
                # start transaction
                self.conn.begin()
                # select
                self.cur.arraysize = 10000
                # select
                _logger.debug(sql+comment+str(varMap))
                self.cur.execute(sql+comment, varMap)
                resList = self.cur.fetchall()
                # append
                for tmpID,tmpStatus,tmpCommand,tmpProdSourceLabel,tmpTaskBufferErrorCode in resList:
                    # ignore jobs retried by pilot since they have new PandaIDs with the same jobsetID/jobdefID
                    if tmpTaskBufferErrorCode in [ErrorCode.EC_PilotRetried]:
                        continue
                    # ignore old buildJob which was replaced by rebrokerage
                    if tmpProdSourceLabel == 'panda':
                        if buildJobID is None:
                            # first buildJob
                            buildJobID = tmpID
                        elif buildJobID >= tmpID:
                            # don't append old one
                            continue
                        else:
                            # delete old one
                            del idStatus[buildJobID]
                            buildJobID = tmpID
                    # append
                    idStatus[tmpID] = (tmpStatus,tmpCommand)
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
            _logger.debug("getPandIDsWithJobID : %s" % str(idStatus))
            return idStatus,buildJobID
        except Exception:
            # roll back
            self._rollback()
            type, value, traceBack = sys.exc_info()
            _logger.error("getPandIDsWithJobID : %s %s" % (type,value))
            # return empty list
            return {},None


    # lock jobs for reassign
    def lockJobsForReassign(self,tableName,timeLimit,statList,labels,processTypes,sites,clouds,
                            useJEDI=False,onlyReassignable=False,useStateChangeTime=False,
                            getEventService=False):
        comment = ' /* DBProxy.lockJobsForReassign */'
        _logger.debug("lockJobsForReassign : %s %s %s %s %s %s %s %s" % \
                      (tableName,timeLimit,statList,labels,processTypes,sites,clouds,useJEDI))
        try:
            # make sql
            if not useJEDI:
                sql  = "SELECT PandaID FROM %s " % tableName
            elif getEventService:
                sql  = "SELECT PandaID,lockedby,eventService,attemptNr,computingSite FROM %s " % tableName
            else:
                sql  = "SELECT PandaID,lockedby FROM %s " % tableName
            if not useStateChangeTime:
                sql += "WHERE modificationTime<:modificationTime "
            else:
                sql += "WHERE stateChangeTime<:modificationTime "
            varMap = {}
            varMap[':modificationTime'] = timeLimit
            if statList != []:
                sql += 'AND jobStatus IN ('
                tmpIdx = 0
                for tmpStat in statList:
                    tmpKey = ':stat%s' % tmpIdx
                    varMap[tmpKey] = tmpStat
                    sql += '%s,' % tmpKey
                    tmpIdx += 1
                sql = sql[:-1]
                sql += ') '
            if labels != []:
                sql += 'AND prodSourceLabel IN ('
                tmpIdx = 0
                for tmpStat in labels:
                    tmpKey = ':label%s' % tmpIdx
                    varMap[tmpKey] = tmpStat
                    sql += '%s,' % tmpKey
                    tmpIdx += 1
                sql = sql[:-1]
                sql += ') '
            if processTypes != []:
                sql += 'AND processingType IN ('
                tmpIdx = 0
                for tmpStat in processTypes:
                    tmpKey = ':processType%s' % tmpIdx
                    varMap[tmpKey] = tmpStat
                    sql += '%s,' % tmpKey
                    tmpIdx += 1
                sql = sql[:-1]
                sql += ') '
            if sites != []:
                sql += 'AND computingSite IN ('
                tmpIdx = 0
                for tmpStat in sites:
                    tmpKey = ':site%s' % tmpIdx
                    varMap[tmpKey] = tmpStat
                    sql += '%s,' % tmpKey
                    tmpIdx += 1
                sql = sql[:-1]
                sql += ') '
            if clouds != []:
                sql += 'AND cloud IN ('
                tmpIdx = 0
                for tmpStat in clouds:
                    tmpKey = ':cloud%s' % tmpIdx
                    varMap[tmpKey] = tmpStat
                    sql += '%s,' % tmpKey
                    tmpIdx += 1
                sql = sql[:-1]
                sql += ') '
            if onlyReassignable:
                sql += "AND (relocationFlag IS NULL OR relocationFlag<>:relocationFlag) "
                varMap[':relocationFlag'] = 2
            # sql for lock
            if not useStateChangeTime:
                sqlLock = 'UPDATE %s SET modificationTime=CURRENT_DATE WHERE PandaID=:PandaID' % tableName
            else:
                sqlLock = 'UPDATE %s SET stateChangeTime=CURRENT_DATE WHERE PandaID=:PandaID' % tableName
            # start transaction
            self.conn.begin()
            # select
            self.cur.arraysize = 1000000
            _logger.debug(sql+comment+str(varMap))
            self.cur.execute(sql+comment,varMap)
            resList = self.cur.fetchall()
            _logger.debug("lockJobsForReassign : found %s" % (len(resList)))
            retList = []
            # lock
            for tmpItem in resList:
                tmpID = tmpItem[0]
                varLock = {':PandaID':tmpID}
                self.cur.execute(sqlLock+comment,varLock)
                retList.append(tmpItem)
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # sort
            retList.sort()
            _logger.debug("lockJobsForReassign : return %s" % (len(retList)))
            return True,retList
        except Exception:
            # roll back
            self._rollback()
            errType,errValue = sys.exc_info()[:2]
            _logger.error("lockJobsForReassign : %s %s" % (errType,errValue))
            # return empty
            return False,[]


    # lock jobs for finisher
    def lockJobsForFinisher(self,timeNow,rownum,highPrio):
        comment = ' /* DBProxy.lockJobsForFinisher */'
        _logger.debug("lockJobsForFinisher : %s %s %s" % (timeNow,rownum,highPrio))
        try:
            varMap = {}
            varMap[':jobStatus'] = 'transferring'
            varMap[':currentPriority'] = 800
            varMap[':pLabel1'] = 'managed'
            varMap[':pLabel2'] = 'test'
            varMap[':esJumbo'] = EventServiceUtils.jumboJobFlagNumber
            # make sql
            sql  = "SELECT PandaID FROM ATLAS_PANDA.jobsActive4 "
            sql += "WHERE jobStatus=:jobStatus AND modificationTime<:modificationTime AND prodSourceLabel IN (:pLabel1,:pLabel2) "
            sql += "AND (eventService IS NULL OR eventService<>:esJumbo) "
            if highPrio:
                varMap[':modificationTime'] = timeNow - datetime.timedelta(hours=1)
                sql += "AND currentPriority>=:currentPriority AND rownum<=%s " % rownum
            else:
                sql += "AND currentPriority<:currentPriority AND rownum<=%s " % rownum
                varMap[':modificationTime'] = timeNow - datetime.timedelta(hours=2)
            sql += "FOR UPDATE "
            # sql for lock
            sqlLock = 'UPDATE ATLAS_PANDA.jobsActive4 SET modificationTime=CURRENT_DATE WHERE PandaID=:PandaID'
            # start transaction
            self.conn.begin()
            # select
            self.cur.arraysize = 1000
            self.cur.execute(sql+comment,varMap)
            resList = self.cur.fetchall()
            retList = []
            # lock
            for tmpID, in resList:
                varLock = {':PandaID':tmpID}
                self.cur.execute(sqlLock+comment,varLock)
                retList.append(tmpID)
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # sort
            retList.sort()
            _logger.debug("lockJobsForFinisher : %s" % (len(retList)))
            return True,retList
        except Exception:
            # roll back
            self._rollback()
            errType,errValue = sys.exc_info()[:2]
            _logger.error("lockJobsForFinisher : %s %s" % (errType,errValue))
            # return empty
            return False,[]


    # lock jobs for activator
    def lockJobsForActivator(self,timeLimit,rownum,prio):
        comment = ' /* DBProxy.lockJobsForActivator */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        tmpLog = LogWrapper(_logger,methodName)
        tmpLog.debug("start")
        try:
            varMap = {}
            varMap[':jobStatus'] = 'assigned'
            if prio > 0:
                varMap[':currentPriority'] = prio
            varMap[':timeLimit'] = timeLimit
            # make sql
            sql  = "SELECT PandaID FROM ATLAS_PANDA.jobsDefined4 "
            sql += "WHERE jobStatus=:jobStatus AND (prodDBUpdateTime IS NULL OR prodDBUpdateTime<:timeLimit) "
            if prio > 0:
                sql += "AND currentPriority>=:currentPriority "
            sql += "AND rownum<=%s " % rownum
            sql += "FOR UPDATE "
            # sql for lock
            sqlLock = 'UPDATE ATLAS_PANDA.jobsDefined4 SET prodDBUpdateTime=CURRENT_DATE WHERE PandaID=:PandaID'
            # start transaction
            self.conn.begin()
            # select
            self.cur.arraysize = 1000
            self.cur.execute(sql+comment,varMap)
            resList = self.cur.fetchall()
            retList = []
            # lock
            for tmpID, in resList:
                varLock = {':PandaID':tmpID}
                self.cur.execute(sqlLock+comment,varLock)
                retList.append(tmpID)
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # sort
            retList.sort()
            tmpLog.debug("locked %s jobs" % (len(retList)))
            return True,retList
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            # return empty
            return False,[]


    # get the number of waiting jobs with a dataset
    def getNumWaitingJobsForPD2P(self,datasetName):
        comment = ' /* DBProxy.getNumWaitingJobsForPD2P */'
        _logger.debug("getNumWaitingJobsForPD2P : %s" % datasetName)
        try:
            tables = ['ATLAS_PANDA.jobsDefined4','ATLAS_PANDA.jobsActive4']
            nJobs = 0
            # select
            for table in tables:
                # make sql
                sql  = "SELECT COUNT(*) FROM %s " % table
                sql += "WHERE prodDBlock=:prodDBlock AND prodSourceLabel IN (:prodSourceLabel1,:prodSourceLabel2) "
                sql += "AND jobStatus IN (:jobStatus1,:jobStatus2) "
                varMap = {}
                varMap[':prodDBlock'] = datasetName
                varMap[':jobStatus1'] = 'defined'
                varMap[':jobStatus2'] = 'activated'
                varMap[':prodSourceLabel1'] = 'user'
                varMap[':prodSourceLabel2'] = 'panda'
                # start transaction
                self.conn.begin()
                # select
                self.cur.arraysize = 10
                self.cur.execute(sql+comment, varMap)
                res = self.cur.fetchone()
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                if res is not None:
                    tmpN, = res
                    nJobs += tmpN
            _logger.debug("getNumWaitingJobsForPD2P : %s -> %s" % (datasetName,nJobs))
            return nJobs
        except Exception:
            # roll back
            self._rollback()
            errType,errValue = sys.exc_info()[:2]
            _logger.error("getNumWaitingJobsForPD2P : %s %s" % (errType,errValue))
            # return 0
            return 0


    # get the number of waiting jobsets with a dataset
    def getNumWaitingJobsetsForPD2P(self,datasetName):
        comment = ' /* DBProxy.getNumWaitingJobsetsForPD2P */'
        _logger.debug("getNumWaitingJobsetsForPD2P : %s" % datasetName)
        try:
            tables = ['ATLAS_PANDA.jobsDefined4','ATLAS_PANDA.jobsActive4']
            jobsetIDuserList = []
            # select
            for table in tables:
                # make sql
                sql  = "SELECT jobsetID,prodUserName FROM %s " % table
                sql += "WHERE prodDBlock=:prodDBlock AND prodSourceLabel IN (:prodSourceLabel1,:prodSourceLabel2) "
                sql += "AND jobStatus IN (:jobStatus1,:jobStatus2) GROUP BY jobsetID,prodUserName"
                varMap = {}
                varMap[':prodDBlock'] = datasetName
                varMap[':jobStatus1'] = 'defined'
                varMap[':jobStatus2'] = 'activated'
                varMap[':prodSourceLabel1'] = 'user'
                varMap[':prodSourceLabel2'] = 'panda'
                # start transaction
                self.conn.begin()
                # select
                self.cur.arraysize = 10000
                self.cur.execute(sql+comment, varMap)
                resList = self.cur.fetchall()
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                for jobsetID,prodUserName in resList:
                    tmpKey = (jobsetID,prodUserName)
                    if tmpKey not in jobsetIDuserList:
                        jobsetIDuserList.append(tmpKey)
            _logger.debug("getNumWaitingJobsetsForPD2P : %s -> %s" % (datasetName,len(jobsetIDuserList)))
            return len(jobsetIDuserList)
        except Exception:
            # roll back
            self._rollback()
            errType,errValue = sys.exc_info()[:2]
            _logger.error("getNumWaitingJobsetsForPD2P : %s %s" % (errType,errValue))
            # return 0
            return 0


    # lock job for re-brokerage
    def lockJobForReBrokerage(self,dn,jobID,simulation,forceOpt,forFailed=False):
        comment = ' /* lockJobForReBrokerage */'
        _logger.debug("lockJobForReBrokerage : %s %s %s %s %s" % (dn,jobID,simulation,forceOpt,forFailed))
        try:
            errMsg = ''
            # get compact DN
            compactDN = self.cleanUserID(dn)
            if compactDN in ['','NULL',None]:
                compactDN = dn
            # start transaction
            self.conn.begin()
            buildJobPandaID   = None
            buildJobStatus    = None
            buildJobDefID     = None
            buildCreationTime = None
            runPandaID        = None
            minPandaIDlibDS   = None
            maxPandaIDlibDS   = None
            # get one runXYZ job
            if errMsg == '':
                for table in ['ATLAS_PANDA.jobsActive4','ATLAS_PANDA.jobsDefined4']:
                    sql  = "SELECT PandaID FROM %s " % table
                    sql += "WHERE prodUserName=:prodUserName AND jobDefinitionID=:jobDefinitionID "
                    sql += "AND prodSourceLabel=:prodSourceLabel1 AND jobStatus IN (:jobStatus1,:jobStatus2) "
                    sql += "AND rownum <= 1"
                    varMap = {}
                    varMap[':prodUserName'] = compactDN
                    varMap[':jobDefinitionID']  = jobID
                    varMap[':prodSourceLabel1'] = 'user'
                    if not forFailed:
                        # lock active jobs for normal rebrokerage
                        varMap[':jobStatus1'] = 'defined'
                        varMap[':jobStatus2'] = 'activated'
                    else:
                        # lock failed jobs for retry
                        varMap[':jobStatus1'] = 'failed'
                        varMap[':jobStatus2'] = 'dummy'
                    # select
                    self.cur.execute(sql+comment, varMap)
                    res = self.cur.fetchone()
                    # not found
                    if res is not None:
                        runPandaID, = res
                        break
                if runPandaID is None:
                    if not forFailed:
                        errMsg = "no defined/activated jobs to reassign. running/finished/failed jobs are not reassigned by rebrokerage "
                    else:
                        errMsg = "could not get failed runXYZ jobs"
            # get libDS
            libDS = ''
            if errMsg == '':
                sql = "SELECT lfn,dataset FROM ATLAS_PANDA.filesTable4 WHERE type=:type AND PandaID=:PandaID"
                varMap = {}
                varMap[':type']    = 'input'
                varMap[':PandaID'] = runPandaID
                # select
                self.cur.arraysize = 10000
                self.cur.execute(sql+comment, varMap)
                resList = self.cur.fetchall()
                for tmpLFN,tmpDS in resList:
                    if tmpLFN.endswith('.lib.tgz'):
                        libDS = tmpDS
                        break
            # check status of corresponding buildJob
            if libDS != '':
                sql  = "SELECT PandaID FROM ATLAS_PANDA.filesTable4 "
                sql += "WHERE type=:type AND dataset=:dataset"
                varMap = {}
                varMap[':type']    = 'output'
                varMap[':dataset'] = libDS
                # select
                self.cur.arraysize = 10
                # select
                self.cur.execute(sql+comment, varMap)
                res = self.cur.fetchone()
                # not found in active table
                if res is None:
                    # look for buildJob in archived table
                    sql  = "SELECT /*+ NO_INDEX(tab JOBS_MODTIME_IDX) INDEX_COMBINE(tab JOBS_PRODSOURCELABEL_IDX JOBS_PRODUSERNAME_IDX) */ "
                    sql += "PandaID,jobStatus,jobDefinitionID,creationTime "
                    sql += "FROM ATLAS_PANDAARCH.jobsArchived tab "
                    sql += "WHERE prodUserName=:prodUserName AND prodSourceLabel=:prodSourceLable1 "
                    sql += "AND modificationTime>(CURRENT_DATE-10) ORDER BY PandaID DESC"
                    varMap = {}
                    varMap[':prodUserName'] = compactDN
                    varMap[':prodSourceLable1'] = 'panda'
                    # select
                    self.cur.arraysize = 10000
                    self.cur.execute(sql+comment, varMap)
                    resList = self.cur.fetchall()
                    # loop over PandaIDs to find corresponding libDS
                    sql  = "SELECT /*+ INDEX(tab FILES_ARCH_PANDAID_IDX)*/ PandaID FROM ATLAS_PANDAARCH.filesTable_ARCH tab "
                    sql += "WHERE PandaID=:PandaID AND type=:type AND dataset=:dataset AND status=:status "
                    sql += "AND modificationTime>(CURRENT_DATE-10)"
                    self.cur.arraysize = 10
                    for tmpID,tmpJobStatus,tmpJobDefID,tmpCreationTime in resList:
                        varMap = {}
                        varMap[':PandaID'] = tmpID
                        varMap[':type']    = 'output'
                        varMap[':status']  = 'ready'
                        varMap[':dataset'] = libDS
                        # select
                        self.cur.execute(sql+comment, varMap)
                        res = self.cur.fetchone()
                        if res is not None:
                            # get PandaID of buildJob
                            buildJobPandaID, = res
                            buildJobStatus = tmpJobStatus
                            buildJobDefID  = tmpJobDefID
                            buildCreationTime = tmpCreationTime
                            break
                    # not found
                    if buildJobPandaID is None:
                        errMsg = "could not find successful buildJob for %s" % libDS
                else:
                    # get PandaID of buildJob
                    buildJobPandaID, = res
                # found buildJob
                if errMsg == '':
                    # get current buildJob status
                    if buildJobStatus is None:
                        for table in ['ATLAS_PANDA.jobsActive4','ATLAS_PANDA.jobsArchived4','ATLAS_PANDA.jobsDefined4']:
                            # make sql
                            sql  = "SELECT jobStatus,jobDefinitionID,creationTime FROM %s " % table
                            sql += "WHERE PandaID=:PandaID "
                            varMap = {}
                            varMap[':PandaID'] = buildJobPandaID
                            # select
                            self.cur.execute(sql+comment, varMap)
                            res = self.cur.fetchone()
                            # found
                            if res is not None:
                                buildJobStatus,buildJobDefID,buildCreationTime = res
                                break
                        # not found
                        if buildJobStatus is None:
                            errMsg = "could not find buildJob=%s in database" % buildJobPandaID
                    # check status
                    if errMsg != '':
                        if buildJobStatus not in ['defined','activated','finished','cancelled','closed']:
                            errMsg = "status of buildJob is '%s' != defined/activated/finished/cancelled so that jobs cannot be reassigned" \
                                     % buildJobStatus
            # get max/min PandaIDs using the libDS
            if errMsg == '':
                sql = "SELECT MAX(PandaID),MIN(PandaID) FROM ATLAS_PANDA.filesTable4 "
                sql += "WHERE type=:type AND dataset=:dataset"
                varMap = {}
                varMap[':type']    = 'input'
                varMap[':dataset'] = libDS
                self.cur.arraysize = 10
                self.cur.execute(sql+comment, varMap)
                res = self.cur.fetchone()
                if res is None:
                    errMsg = "cannot get MAX/MIN PandaID for multiple usage for %s" % libDS
                else:
                    maxPandaIDlibDS,minPandaIDlibDS = res
            # check creationDate of buildJob
            if errMsg == '':
                # buildJob has already finished
                timeLimit = datetime.datetime.utcnow()-datetime.timedelta(days=6)
                if buildJobStatus in ['finished','cancelled','closed'] and buildCreationTime < timeLimit:
                    errMsg = "corresponding buildJob %s is too old %s" % (buildJobPandaID,buildCreationTime.strftime('%Y-%m-%d %H:%M:%S'))
            # check modificationTime
            if errMsg == '':
                # make sql
                tables = ['ATLAS_PANDA.jobsDefined4']
                if buildJobStatus not in ['defined']:
                    tables.append('ATLAS_PANDA.jobsActive4')
                sql  = "SELECT modificationTime FROM %s "
                sql += "WHERE prodUserName=:prodUserName AND jobDefinitionID=:jobDefinitionID "
                sql += "AND prodSourceLabel IN (:prodSourceLabel1,:prodSourceLabel2) AND jobStatus IN (:jobStatus1,:jobStatus2) "
                sql += "FOR UPDATE "
                varMap = {}
                varMap[':prodUserName'] = compactDN
                varMap[':jobDefinitionID']  = jobID
                varMap[':prodSourceLabel1'] = 'user'
                varMap[':prodSourceLabel2'] = 'panda'
                if not forFailed:
                    # normal rebrokerage
                    varMap[':jobStatus1'] = 'defined'
                    varMap[':jobStatus2'] = 'activated'
                else:
                    # retry
                    varMap[':jobStatus1'] = 'failed'
                    varMap[':jobStatus2'] = 'dummy'
                for tableName in tables:
                    # select
                    self.cur.execute((sql % tableName)+comment, varMap)
                    res = self.cur.fetchone()
                    if res is not None:
                        break
                if res is None:
                    if not forFailed:
                        errMsg = "no defined/activated jobs to be reassigned"
                    else:
                        errMsg = "no failed jobs to be retried"
                else:
                    tmpModificationTime, = res
                    # prevent users from rebrokering more than once in one hour
                    timeLimit = datetime.datetime.utcnow()-datetime.timedelta(hours=1)
                    if timeLimit < tmpModificationTime and not forceOpt:
                        errMsg = "last mod time is %s > current-1hour. Cannot run (re)brokerage more than once in one hour" \
                                 % tmpModificationTime.strftime('%Y-%m-%d %H:%M:%S')
                    elif simulation:
                        pass
                    else:
                        # update modificationTime for locking
                        for tableName in tables:
                            sql = 'UPDATE %s ' % tableName
                            sql += 'SET modificationTime=CURRENT_DATE '
                            sql += "WHERE prodUserName=:prodUserName AND jobDefinitionID=:jobDefinitionID "
                            sql += "AND prodSourceLabel IN (:prodSourceLabel1,:prodSourceLabel2) AND jobStatus IN (:jobStatus1,:jobStatus2) "
                            self.cur.execute(sql+comment, varMap)
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # return failure
            if errMsg != '':
                _logger.debug('lockJobForReBrokerage : '+errMsg)
                return False,{'err':errMsg}
            # return
            retMap = {'bPandaID':buildJobPandaID,'bStatus':buildJobStatus,'userName':compactDN,
                      'bJobID':buildJobDefID,'rPandaID':runPandaID,
                      'maxPandaIDlibDS':maxPandaIDlibDS,'minPandaIDlibDS':minPandaIDlibDS}
            _logger.debug("lockJobForReBrokerage %s" % str(retMap))
            return True,retMap
        except Exception:
            # roll back
            self._rollback()
            type, value, traceBack = sys.exc_info()
            _logger.error("lockJobForReBrokerage : %s %s" % (type,value))
            # return empty list
            return False,{'err':'database error'}


    # get input datasets for rebrokerage
    def getInDatasetsForReBrokerage(self,jobID,userName):
        comment = ' /* DBProxy.getInDatasetsForReBrokerage */'
        failedRet = False,{},None
        try:
            _logger.debug("getInDatasetsForReBrokerage(%s,%s)" % (jobID,userName))
            # start transaction
            self.conn.begin()
            # get pandaID
            pandaIDs = []
            maxTotalFileSize = None
            for table in ['jobsActive4','jobsDefined4']:
                sql  = "SELECT PandaID FROM ATLAS_PANDA.%s WHERE prodUserName=:prodUserName AND jobDefinitionID=:jobDefinitionID " % table
                sql += "AND prodSourceLabel=:prodSourceLabel AND jobStatus IN (:jobStatus1,:jobStatus2)"
                varMap = {}
                varMap[':prodUserName'] = userName
                varMap[':jobDefinitionID']  = jobID
                varMap[':prodSourceLabel'] = 'user'
                varMap[':jobStatus1'] = 'defined'
                varMap[':jobStatus2'] = 'activated'
                self.cur.arraysize = 10000
                self.cur.execute(sql+comment, varMap)
                res = self.cur.fetchall()
                if res != []:
                    for tmpItem in res:
                        pandaIDs.append(tmpItem[0])
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # not found
            if pandaIDs == []:
                _logger.debug("getInDatasetsForReBrokerage : PandaIDs not found")
                return failedRet
            # get dataset and lfn
            retMapLFN = {}
            sql  = "SELECT dataset,lfn,fsize FROM ATLAS_PANDA.filesTable4 "
            sql += "WHERE PandaID=:PandaID AND type=:type"
            for pandaID in pandaIDs:
                varMap = {}
                varMap[':PandaID'] = pandaID
                varMap[':type'] = 'input'
                # start transaction
                self.conn.begin()
                self.cur.arraysize = 10000
                self.cur.execute(sql+comment, varMap)
                resL = self.cur.fetchall()
                # append
                tmpTotalFileSize = 0
                for tmpDataset,tmpLFN,tmpFileSize in resL:
                    # ignore lib.tgz
                    if tmpLFN.endswith('.lib.tgz'):
                        continue
                    if tmpDataset not in retMapLFN:
                        retMapLFN[tmpDataset] = []
                    if tmpLFN not in retMapLFN[tmpDataset]:
                        retMapLFN[tmpDataset].append(tmpLFN)
                    try:
                        tmpTotalFileSize += long(tmpFileSize)
                    except Exception:
                        pass
                if maxTotalFileSize is None or maxTotalFileSize < tmpTotalFileSize:
                    maxTotalFileSize = tmpTotalFileSize
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
            _logger.debug("getInDatasetsForReBrokerage : done")
            # max size in MB
            maxTotalFileSize /= (1024*1024)
            # return
            return True,retMapLFN,maxTotalFileSize
        except Exception:
            # roll back
            self._rollback()
            errType,errValue = sys.exc_info()[:2]
            _logger.error("getInDatasetsForReBrokerage(%s,%s) : %s %s" % (jobID,userName,errType,errValue))
            return failedRet


    # move jobs to jobsDefine4 for re-brokerage
    def resetBuildJobForReBrokerage(self,pandaID):
        comment = ' /* resetBuildJobForReBrokerage */'
        _logger.debug("resetBuildJobForReBrokerage : start %s" % pandaID)
        try:
            # make sql to move jobs
            sql1 = "SELECT %s FROM ATLAS_PANDA.jobsActive4 " % JobSpec.columnNames()
            sql1+= "WHERE PandaID=:PandaID AND jobStatus=:jobStatus1"
            sql3 = "INSERT INTO ATLAS_PANDA.jobsDefined4 (%s) " % JobSpec.columnNames()
            sql3+= JobSpec.bindValuesExpression()
            # start transaction
            self.conn.begin()
            # select
            varMap = {}
            varMap[':PandaID'] = pandaID
            varMap[':jobStatus1'] = 'activated'
            self.cur.arraysize = 10
            self.cur.execute(sql1+comment,varMap)
            res = self.cur.fetchone()
            # not found
            if res is None:
                _logger.error("resetBuildJobForReBrokerage : PandaID=%s not found" % pandaID)
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                # return
                return False
            # instantiate Job
            job = JobSpec()
            job.pack(res)
            # delete from jobsDefined4 just in case
            varMap = {}
            varMap[':PandaID'] = pandaID
            sqlD = "DELETE FROM ATLAS_PANDA.jobsDefined4 WHERE PandaID=:PandaID"
            self.cur.execute(sqlD+comment,varMap)
            # reset job status
            job.jobStatus = 'defined'
            # host and time information
            job.modificationHost = self.hostname
            job.modificationTime = datetime.datetime.utcnow()
            # insert to Defined
            self.cur.execute(sql3+comment, job.valuesMap())
            # delete from Active
            varMap = {}
            varMap[':PandaID'] = pandaID
            varMap[':jobStatus1'] = 'activated'
            sql2 = "DELETE FROM ATLAS_PANDA.jobsActive4 WHERE PandaID=:PandaID AND jobStatus=:jobStatus1"
            self.cur.execute(sql2+comment,varMap)
            retD = self.cur.rowcount
            # delete failed
            if retD != 1:
                _logger.error("resetBuildJobForReBrokerage : failed to delete PandaID=%s" % pandaID)
                # rollback
                self._rollback()
                return False
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # return
            _logger.debug("resetBuildJobForReBrokerage : end %s" % pandaID)
            return True
        except Exception:
            # roll back
            self._rollback()
            type, value, traceBack = sys.exc_info()
            _logger.error("resetBuildJobForReBrokerage : %s %s" % (type,value))
            # return empty list
            return False


    # get PandaIDs using userName/jobID for re-brokerage or retry
    def getPandaIDsForReBrokerage(self,userName,jobID,fromActive,forFailed=False):
        comment = ' /* DBProxy.getPandaIDsForReBrokerage */'
        _logger.debug("getPandaIDsForReBrokerage : %s %s %s %s" % (userName,jobID,fromActive,forFailed))
        try:
            returnList = []
            varMap = {}
            varMap[':prodUserName'] = userName
            varMap[':jobDefinitionID'] = jobID
            if not forFailed:
                varMap[':jobStatus1'] = 'activated'
            else:
                varMap[':jobStatus1'] = 'failed'
            sql  = "SELECT PandaID FROM ATLAS_PANDA.jobsActive4 "
            sql += "WHERE prodUserName=:prodUserName AND jobDefinitionID=:jobDefinitionID "
            sql += "AND jobStatus=:jobStatus1"
            # get IDs from Active table
            if fromActive:
                # start transaction
                self.conn.begin()
                # select
                self.cur.arraysize = 20000
                self.cur.execute(sql+comment,varMap)
                resList = self.cur.fetchall()
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                # append
                for tmpID, in resList:
                    if tmpID not in returnList:
                        returnList.append(tmpID)
                # set holding to prevent activated jobs from being picked up
                if not forFailed:
                    sql  = 'UPDATE ATLAS_PANDA.jobsActive4 SET jobStatus=:newStatus '
                    sql += 'WHERE prodUserName=:prodUserName AND jobDefinitionID=:jobDefinitionID '
                    sql += "AND jobStatus=:jobStatus1"
                    varMap[':newStatus'] = 'holding'
                    # start transaction
                    self.conn.begin()
                    # update
                    self.cur.execute(sql+comment,varMap)
                    # commit
                    if not self._commit():
                        raise RuntimeError('Commit error')
            # get IDs from Defined table just in case
            varMap = {}
            varMap[':prodUserName'] = userName
            varMap[':jobDefinitionID'] = jobID
            varMap[':jobStatus1'] = 'defined'
            varMap[':jobStatus2'] = 'assgined'
            sql  = "SELECT PandaID FROM ATLAS_PANDA.jobsDefined4 "
            sql += "WHERE prodUserName=:prodUserName AND jobDefinitionID=:jobDefinitionID "
            sql += "AND jobStatus IN (:jobStatus1,:jobStatus2)"
            # start transaction
            self.conn.begin()
            # select
            self.cur.arraysize = 20000
            self.cur.execute(sql+comment,varMap)
            resList = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # append
            for tmpID, in resList:
                if tmpID not in returnList:
                    returnList.append(tmpID)
            # sort
            returnList.sort()
            # return
            return returnList
        except Exception:
            # roll back
            self._rollback()
            type, value, traceBack = sys.exc_info()
            _logger.error("getPandaIDsForReBrokerage : %s %s" % (type,value))
            # return empty list
            return []


    # get outDSs with userName/jobID
    def getOutDSsForReBrokerage(self,userName,jobID):
        comment = ' /* DBProxy.getOutDSsForReBrokerage */'
        _logger.debug("getOutDSsForReBrokerage : %s %s" % (userName,jobID))
        falseRet = (False,[],None,None)
        try:
            # get one PandaID
            sql  = "SELECT PandaID,computingSite,destinationSE FROM ATLAS_PANDA.jobsActive4 "
            sql += "WHERE prodUserName=:prodUserName AND jobDefinitionID=:jobDefinitionID "
            sql += "AND prodSourceLabel=:prodSourceLabel AND rownum<=1"
            varMap = {}
            varMap[':prodUserName'] = userName
            varMap[':jobDefinitionID'] = jobID
            varMap[':prodSourceLabel'] = 'user'
            # start transaction
            self.conn.begin()
            # select
            self.cur.arraysize = 10
            self.cur.execute(sql+comment, varMap)
            res = self.cur.fetchone()
            # not found
            if res is None:
                _logger.debug("getOutDSsForReBrokerage : failed to get PandaID")
                if not self._commit():
                    raise RuntimeError('Commit error')
                return falseRet
            pandaID,computingSite,destinationSE = res
            # get outDSs
            sql = "SELECT dataset FROM ATLAS_PANDA.filesTable4 WHERE PandaID=:PandaID AND type IN (:type1,:type2)"
            varMap = {}
            varMap[':type1'] = 'output'
            varMap[':type2'] = 'log'
            varMap[':PandaID'] = pandaID
            self.cur.arraysize = 1000
            self.cur.execute(sql+comment, varMap)
            resList = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # append
            returnList = []
            for tmpOutDS, in resList:
                if tmpOutDS not in returnList:
                    returnList.append(tmpOutDS)
            # return
            return True,returnList,computingSite,destinationSE
        except Exception:
            # roll back
            self._rollback()
            type, value, traceBack = sys.exc_info()
            _logger.error("getOutDSsForReBrokerage : %s %s" % (type,value))
            # return empty list
            return falseRet


    # query PandaID
    def queryPandaID(self,jobDefID):
        comment = ' /* DBProxy.queryPandaID */'
        _logger.debug("queryPandaID : %s" % jobDefID)
        sql0 = "SELECT PandaID,attemptNr FROM %s WHERE attemptNr=("
        sql0+= "SELECT MAX(attemptNr) FROM %s"
        sql1= " WHERE prodSourceLabel=:prodSourceLabel AND jobDefinitionID=:jobDefinitionID)"
        sql1+=" AND prodSourceLabel=:prodSourceLabel AND jobDefinitionID=:jobDefinitionID"
        try:
            ids = []
            # select
            varMap = {}
            varMap[':jobDefinitionID'] = jobDefID
            varMap[':prodSourceLabel'] = 'managed'
            for table in ['ATLAS_PANDA.jobsDefined4','ATLAS_PANDA.jobsActive4','ATLAS_PANDA.jobsArchived4','ATLAS_PANDA.jobsWaiting4']:
                # start transaction
                self.conn.begin()
                # select
                sql = sql0 % (table,table) + sql1
                self.cur.arraysize = 10
                self.cur.execute(sql+comment, varMap)
                res = self.cur.fetchall()
                ids += list(res)
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
            # look for the latest attempt
            preAtt =-1
            pandaID=None
            for pID,att in ids:
                if att > preAtt:
                    pandaID = pID
                    preAtt = att
                if att == preAtt:
                    if pandaID < pID:
                        pandaID = pID
            return pandaID
        except Exception:
            type, value, traceBack = sys.exc_info()
            _logger.error("queryPandaID : %s %s" % (type,value))
            # roll back
            self._rollback()
            return None


    # query job info per cloud
    def queryJobInfoPerCloud(self,cloud,schedulerID=None):
        comment = ' /* DBProxy.queryJobInfoPerCloud */'
        _logger.debug("queryJobInfoPerCloud : %s %s" % (cloud,schedulerID))
        attrs = ['PandaID','jobStatus','jobName']
        sql0 = "SELECT "
        for attr in attrs:
            sql0 += "%s," % attr
        sql0 = "%s " % sql0[:-1]
        sql0+= "FROM %s "
        sql0+= "WHERE cloud=:cloud "
        varMap = {}
        varMap[':cloud'] = cloud
        if schedulerID is not None:
            sql0+= "AND schedulerID=:schedulerID "
            varMap[':schedulerID'] = schedulerID
        try:
            ids = []
            returnList = []
            # select
            for table in ['ATLAS_PANDA.jobsActive4','ATLAS_PANDA.jobsWaiting4','ATLAS_PANDA.jobsDefined4']:
                # start transaction
                self.conn.begin()
                # select
                sql = sql0 % table
                self.cur.arraysize = 10000
                self.cur.execute(sql+comment,varMap)
                resList = self.cur.fetchall()
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                # loop over all
                for res in resList:
                    valMap = {}
                    # skip if already in the list
                    PandaID = res[0]
                    if PandaID in ids:
                        continue
                    # convert to map
                    for idx,attr in enumerate(attrs):
                        valMap[attr] = res[idx]
                    # append to list
                    ids.append(PandaID)
                    returnList.append(valMap)
            # return
            return returnList
        except Exception:
            type, value, traceBack = sys.exc_info()
            _logger.error("queryJobInfoPerCloud : %s %s" % (type,value))
            # roll back
            self._rollback()
            return None


    # get PandaIDs at Site
    def getPandaIDsSite(self,site,status,limit):
        comment = ' /* DBProxy.getPandaIDsSite */'
        _logger.debug("getPandaIDsSite : %s %s %s" % (site,status,limit))
        try:
            ids = []
            # find table
            if status in ['defined','assigned']:
                table = 'ATLAS_PANDA.jobsDefined4'
            elif status in ['activated','running','holding','transferring']:
                table = 'ATLAS_PANDA.jobsActive4'
            elif status in ['waiting']:
                table = 'ATLAS_PANDA.jobsWaiting4'
            elif status in ['finished','failed']:
                table = 'ATLAS_PANDA.jobsArchived4'
            else:
                _logger.error("unknown status:%s" % status)
                return ids
            # limit
            limit = int(limit)
            # SQL
            sql  = "SELECT PandaID FROM %s " % table
            sql += "WHERE computingSite=:computingSite AND jobStatus=:jobStatus AND prodSourceLabel=:prodSourceLabel "
            sql += "AND rownum<=:limit"
            # start transaction
            self.conn.begin()
            # select
            varMap = {}
            varMap[':computingSite'] = site
            varMap[':jobStatus'] = status
            varMap[':limit'] = limit
            varMap[':prodSourceLabel'] = 'managed'
            self.cur.arraysize = limit
            self.cur.execute(sql+comment, varMap)
            res = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # convert to list
            for id, in res:
                ids.append(id)
            return ids
        except Exception:
            type, value, traceBack = sys.exc_info()
            _logger.error("getPandaIDsSite : %s %s" % (type,value))
            # roll back
            self._rollback()
            return []


    # get PandaIDs to be updated in prodDB
    def getPandaIDsForProdDB(self,limit,lockedby):
        comment = ' /* DBProxy.getPandaIDsForProdDB */'
        _logger.debug("getPandaIDsForProdDB %s" % limit)
        varMap = {}
        varMap[':lockedby'] = lockedby
        varMap[':limit'] = limit
        varMap[':prodSourceLabel1'] = 'managed'
        sql0 = "PandaID,jobStatus,stateChangeTime,attemptNr,jobDefinitionID,jobExecutionID FROM %s "
        sqlW = "WHERE prodSourceLabel IN (:prodSourceLabel1,"
        for tmpLabel in JobUtils.list_ptest_prod_sources:
            tmpKey = ':prodSourceLabel_{0}'.format(tmpLabel)
            sqlW += tmpKey
            sqlW += ','
            varMap[tmpKey] = tmpLabel
        sqlW = sqlW[:-1]
        sqlW += ") AND lockedby=:lockedby "
        sqlX = "AND stateChangeTime>prodDBUpdateTime "
        sqlA = "AND (CASE WHEN stateChangeTime>prodDBUpdateTime THEN 1 ELSE null END) = 1 "
        sql1 = "AND rownum<=:limit "
        try:
            retMap   = {}
            totalIDs = 0
            # select
            for table in ['ATLAS_PANDA.jobsArchived4','ATLAS_PANDA.jobsActive4','ATLAS_PANDA.jobsWaiting4','ATLAS_PANDA.jobsDefined4']:
                # start transaction
                self.conn.begin()
                # select
                sql = sql0 % table
                if table in ['ATLAS_PANDA.jobsArchived4']:
                    sql = "SELECT /*+ INDEX_RS_ASC(tab JOBSARCHIVED4_CHANGETIME) NO_INDEX(tab(PRODSOURCELABEL))*/ " + sql + " tab " + sqlW + sqlA
                else:
                    sql = "SELECT " + sql + sqlW + sqlX
                sql += sql1
                self.cur.arraysize = limit
                _logger.debug("getPandaIDsForProdDB %s %s" % (sql+comment,str(varMap)))
                self.cur.execute(sql+comment, varMap)
                res = self.cur.fetchall()
                _logger.debug("getPandaIDsForProdDB got %s" % len(res))
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                for PandaID,jobStatus,stateChangeTime,attemptNr,jobDefinitionID,jobExecutionID in res:
                    # ignore dummy jobs in jobsDefined4
                    if table == 'ATLAS_PANDA.jobsDefined4' and (jobStatus not in ['defined','assigned']):
                        continue
                    # add status
                    if jobStatus not in retMap:
                        retMap[jobStatus] = []
                    # append
                    retMap[jobStatus].append({'PandaID':PandaID,'attemptNr':attemptNr,
                                              'stateChangeTime':stateChangeTime.strftime('%Y-%m-%d %H:%M:%S'),
                                              'jobDefinitionID':jobDefinitionID,
                                              'jobExecutionID':jobExecutionID})
                    totalIDs += 1
                    # limit
                    if totalIDs > limit:
                        break
            _logger.debug("getPandaIDsForProdDB %s ret->%s" % (limit,totalIDs))
            return retMap
        except Exception:
            type, value, traceBack = sys.exc_info()
            _logger.error("getPandaIDsForProdDB : %s %s" % (type,value))
            # roll back
            self._rollback()
            return {}


    # update prodDBUpdateTime
    def updateProdDBUpdateTime(self,param):
        comment = ' /* DBProxy.updateProdDBUpdateTime */'
        _logger.debug("updateProdDBUpdateTime %s" % str(param))
        sql0 = "UPDATE %s "
        sql0+= "SET prodDBUpdateTime=TO_TIMESTAMP(:prodDBUpdateTime,'YYYY-MM-DD HH24:MI:SS') "
        sql0+= "WHERE PandaID=:PandaID AND jobStatus=:jobStatus AND stateChangeTime=TO_TIMESTAMP(:stateChangeTime,'YYYY-MM-DD HH24:MI:SS') "
        varMap = {}
        varMap[':prodDBUpdateTime'] = param['stateChangeTime']
        varMap[':PandaID']          = param['PandaID']
        varMap[':jobStatus']        = param['jobStatus']
        varMap[':stateChangeTime']  = param['stateChangeTime']
        try:
            # convert to string
            if isinstance(varMap[':prodDBUpdateTime'],datetime.datetime):
                varMap[':prodDBUpdateTime'] = varMap[':prodDBUpdateTime'].strftime('%Y-%m-%d %H:%M:%S')
            if isinstance(varMap[':stateChangeTime'],datetime.datetime):
                varMap[':stateChangeTime'] = varMap[':stateChangeTime'].strftime('%Y-%m-%d %H:%M:%S')
            # set table
            if param['jobStatus'] in ['defined','assigned']:
                table = 'ATLAS_PANDA.jobsDefined4'
            elif param['jobStatus'] in ['waiting','pending']:
                table = 'ATLAS_PANDA.jobsWaiting4'
            elif param['jobStatus'] in ['activated','sent','starting','running','holding','transferring']:
                table = 'ATLAS_PANDA.jobsActive4'
            elif param['jobStatus'] in ['finished','failed','cancelled','closed']:
                table = 'ATLAS_PANDA.jobsArchived4'
            else:
                _logger.error("invalid status %s" % param['jobStatus'])
                return False
            # set transaction
            self.conn.begin()
            # update
            sql = sql0 % table
            _logger.debug(sql+comment+str(varMap))
            self.cur.execute(sql+comment, varMap)
            retU = self.cur.rowcount
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            _logger.debug("updateProdDBUpdateTime %s ret=%s" % (param['PandaID'],retU))
            if retU == 1:
                return True
            return False
        except Exception:
            type, value, traceBack = sys.exc_info()
            _logger.error("updateProdDBUpdateTime : %s %s" % (type,value))
            # roll back
            self._rollback()
            return False


    # add metadata
    def addMetadata(self,pandaID,metadata,newStatus):
        comment = ' /* DBProxy.addMetaData */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        tmpLog = LogWrapper(_logger,methodName+" <PandaID={0}>".format(pandaID))
        tmpLog.debug("start {0}".format(newStatus))
        # discard metadata for failed jobs
        if newStatus == 'failed':
            tmpLog.debug('skip')
            return True
        sqlJ  = "SELECT jobStatus FROM ATLAS_PANDA.jobsActive4 WHERE PandaID=:PandaID "
        sqlJ += "UNION "
        sqlJ += "SELECT jobStatus FROM ATLAS_PANDA.jobsArchived4 WHERE PandaID=:PandaID "
        sql0 = "SELECT PandaID FROM ATLAS_PANDA.metaTable WHERE PandaID=:PandaID"
        sql1 = "INSERT INTO ATLAS_PANDA.metaTable (PandaID,metaData) VALUES (:PandaID,:metaData)"
        nTry=1
        regStart = datetime.datetime.utcnow()
        for iTry in range(nTry):
            try:
                # autocommit on
                self.conn.begin()
                self.cur.arraysize = 10
                # check job status
                varMap = {}
                varMap[':PandaID'] = pandaID
                self.cur.execute(sqlJ+comment, varMap)
                resJ = self.cur.fetchone()
                if resJ is not None:
                    jobStatus, = resJ
                else:
                    jobStatus = 'unknown'
                if jobStatus in ['unknown']:
                    tmpLog.debug("skip jobStatus={0}".format(jobStatus))
                    if not self._commit():
                        raise RuntimeError('Commit error')
                    return False
                # skip if in final state
                if jobStatus in ['cancelled','closed','finished','failed']:
                    tmpLog.debug("skip jobStatus={0}".format(jobStatus))
                    if not self._commit():
                        raise RuntimeError('Commit error')
                    # return True so that subsequent procedure can keep going
                    return True
                # select
                varMap = {}
                varMap[':PandaID'] = pandaID
                self.cur.arraysize = 10
                self.cur.execute(sql0+comment, varMap)
                res = self.cur.fetchone()
                # already exist
                if res is not None:
                    tmpLog.debug("skip duplicated during jobStatus={0}".format(jobStatus))
                    if not self._commit():
                        raise RuntimeError('Commit error')
                    return True
                # truncate
                if metadata is not None:
                    origSize = len(metadata)
                else:
                    origSize = 0
                maxSize = 1024*1024
                if newStatus in ['failed'] and origSize > maxSize:
                    metadata = metadata[:maxSize]
                # insert
                varMap = {}
                varMap[':PandaID'] = pandaID
                varMap[':metaData'] = metadata
                self.cur.execute(sql1+comment, varMap)
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                regTime = datetime.datetime.utcnow() - regStart
                msgStr = "done in jobStatus={0}->{1} took {2} sec".format(jobStatus,newStatus,regTime.seconds)
                if metadata is not None:
                    msgStr += ' for {0} (orig {1}) bytes'.format(len(metadata),origSize)
                tmpLog.debug(msgStr)
                return True
            except Exception:
                # roll back
                self._rollback()
                if iTry+1 < nTry:
                    tmpLog.debug("retry : %s" % iTry)
                    time.sleep(random.randint(10,20))
                    continue
                self.dumpErrorMessage(_logger,methodName)
                return False


    # add stdout
    def addStdOut(self,pandaID,stdOut):
        comment = ' /* DBProxy.addStdOut */'
        _logger.debug("addStdOut : %s start" % pandaID)
        sqlJ = "SELECT PandaID FROM ATLAS_PANDA.jobsActive4 WHERE PandaID=:PandaID FOR UPDATE "
        sqlC = "SELECT PandaID FROM ATLAS_PANDA.jobsDebug WHERE PandaID=:PandaID "
        sqlI = "INSERT INTO ATLAS_PANDA.jobsDebug (PandaID,stdOut) VALUES (:PandaID,:stdOut) "
        sqlU = "UPDATE ATLAS_PANDA.jobsDebug SET stdOut=:stdOut WHERE PandaID=:PandaID "
        try:
            # autocommit on
            self.conn.begin()
            # select
            varMap = {}
            varMap[':PandaID'] = pandaID
            self.cur.arraysize = 10
            # check job table
            self.cur.execute(sqlJ+comment, varMap)
            res = self.cur.fetchone()
            if res is None:
                _logger.debug("addStdOut : %s non active" % pandaID)
            else:
                # check debug table
                self.cur.execute(sqlC+comment, varMap)
                res = self.cur.fetchone()
                # already exist
                if res is not None:
                    # update
                    sql = sqlU
                else:
                    # insert
                    sql = sqlI
                # write stdout
                varMap = {}
                varMap[':PandaID'] = pandaID
                varMap[':stdOut']  = stdOut
                self.cur.execute(sql+comment, varMap)
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            return True
        except Exception:
            # roll back
            self._rollback()
            errtype,errvalue = sys.exc_info()[:2]
            _logger.error("addStdOut : %s %s" % (errtype,errvalue))
            return False


    # insert sandbox file info
    def insertSandboxFileInfo(self,userName,hostName,fileName,fileSize,checkSum):
        comment = ' /* DBProxy.insertSandboxFileInfo */'
        _logger.debug("insertSandboxFileInfo : %s %s %s %s %s" % (userName,hostName,fileName,fileSize,checkSum))
        sqlC  = "SELECT userName,fileSize,checkSum FROM ATLAS_PANDAMETA.userCacheUsage "
        sqlC += "WHERE hostName=:hostName AND fileName=:fileName FOR UPDATE"
        sql  = "INSERT INTO ATLAS_PANDAMETA.userCacheUsage "
        sql += "(userName,hostName,fileName,fileSize,checkSum,creationTime,modificationTime) "
        sql += "VALUES (:userName,:hostName,:fileName,:fileSize,:checkSum,CURRENT_DATE,CURRENT_DATE) "
        try:
            # begin transaction
            self.conn.begin()
            # check if it already exists
            varMap = {}
            varMap[':hostName'] = hostName
            varMap[':fileName'] = fileName
            self.cur.arraysize = 10
            self.cur.execute(sqlC+comment, varMap)
            res = self.cur.fetchall()
            if len(res) != 0:
                _logger.debug("insertSandboxFileInfo : skip %s %s since already exists" % (hostName,fileName))
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                return "WARNING: file exist"
            # insert
            varMap = {}
            varMap[':userName'] = userName
            varMap[':hostName'] = hostName
            varMap[':fileName'] = fileName
            varMap[':fileSize'] = fileSize
            varMap[':checkSum'] = checkSum
            self.cur.execute(sql+comment, varMap)
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            return "OK"
        except Exception:
            # roll back
            self._rollback()
            # error
            type, value, traceBack = sys.exc_info()
            _logger.error("insertSandboxFileInfo : %s %s" % (type,value))
            return "ERROR: DB failure"


    # get and lock sandbox files
    def getLockSandboxFiles(self, time_limit, n_files):
        comment = ' /* DBProxy.getLockSandboxFiles */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        tmpLog = LogWrapper(_logger, methodName)
        sqlC = "SELECT * FROM ("\
               "SELECT userName,hostName,fileName,creationTime,modificationTime FROM ATLAS_PANDAMETA.userCacheUsage "\
               "WHERE modificationTime<:timeLimit AND (fileName like 'sources%' OR fileName like 'jobO%') ) "\
               "WHERE rownum<:nRows "
        sqlU = "UPDATE ATLAS_PANDAMETA.userCacheUsage SET modificationTime=CURRENT_DATE " \
               "WHERE userName=:userName AND fileName=:fileName "
        try:
            tmpLog.debug('start')
            # begin transaction
            self.conn.begin()
            # check if it already exists
            varMap = {}
            varMap[':timeLimit'] = time_limit
            varMap[':nRows'] = n_files
            self.cur.execute(sqlC+comment, varMap)
            res = self.cur.fetchall()
            retList = []
            for userName, hostName, fileName, creationTime, modificationTime in res:
                retList.append((userName, hostName, fileName, creationTime, modificationTime))
                # lock
                varMap = dict()
                varMap[':userName'] = userName
                varMap[':fileName'] = fileName
                self.cur.execute(sqlU + comment, varMap)
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            tmpLog.debug('locked {0} files'.format(len(retList)))
            return retList
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return None


    # check duplicated sandbox file
    def checkSandboxFile(self,dn,fileSize,checkSum):
        comment = ' /* DBProxy.checkSandboxFile */'
        _logger.debug("checkSandboxFile : %s %s %s" % (dn,fileSize,checkSum))
        sqlC  = "SELECT hostName,fileName FROM ATLAS_PANDAMETA.userCacheUsage "
        sqlC += "WHERE userName=:userName AND fileSize=:fileSize AND checkSum=:checkSum "
        sqlC += "AND hostName<>:ngHostName AND creationTime>CURRENT_DATE-3 "
        sqlC += "AND creationTime>CURRENT_DATE-3 "
        try:
            retStr = 'NOTFOUND'
            # get compact DN
            compactDN = self.cleanUserID(dn)
            if compactDN in ['','NULL',None]:
                compactDN = dn
            # begin transaction
            self.conn.begin()
            # check if it already exists
            varMap = {}
            varMap[':userName'] = compactDN
            varMap[':fileSize'] = fileSize
            varMap[':checkSum'] = str(checkSum)
            varMap[':ngHostName'] = 'localhost.localdomain'
            self.cur.arraysize = 10
            self.cur.execute(sqlC+comment, varMap)
            res = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            if len(res) != 0:
                hostName,fileName = res[0]
                retStr = "FOUND:%s:%s" % (hostName,fileName)
            _logger.debug("checkSandboxFile -> %s" % retStr)
            return retStr
        except Exception:
            # roll back
            self._rollback()
            # error
            type, value, traceBack = sys.exc_info()
            _logger.error("checkSandboxFile : %s %s" % (type,value))
            return "ERROR: DB failure"


    # insert dataset
    def insertDataset(self,dataset,tablename="ATLAS_PANDA.Datasets"):
        comment = ' /* DBProxy.insertDataset */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        tmpLog = LogWrapper(_logger,methodName+" <dataset={0}>".format(dataset.name))
        tmpLog.debug("start")
        sql0 = "SELECT COUNT(*) FROM %s WHERE vuid=:vuid " % tablename
        sql1 = "INSERT INTO %s " % tablename
        sql1+= "(%s) " % DatasetSpec.columnNames()
        sql1+= DatasetSpec.bindValuesExpression()
        sql2 = "SELECT name FROM %s WHERE vuid=:vuid " % tablename
        # time information
        dataset.creationdate = datetime.datetime.utcnow()
        dataset.modificationdate = dataset.creationdate
        try:
            # subtype
            if dataset.subType in ['','NULL',None]:
                # define using name
                if re.search('_dis\d+$',dataset.name) is not None:
                    dataset.subType = 'dis'
                elif re.search('_sub\d+$',dataset.name) is not None:
                    dataset.subType= 'sub'
                else:
                    dataset.subType= 'top'
            # begin transaction
            self.conn.begin()
            # check if it already exists
            varMap = {}
            varMap[':vuid'] = dataset.vuid
            self.cur.execute(sql0+comment, varMap)
            nDS, = self.cur.fetchone()
            tmpLog.debug("nDS=%s with %s" % (nDS,dataset.vuid))
            if nDS == 0:
                # insert
                tmpLog.debug(sql1+comment+str(dataset.valuesMap()))
                self.cur.execute(sql1+comment, dataset.valuesMap())
                # check name in DB
                varMap = {}
                varMap[':vuid'] = dataset.vuid
                self.cur.execute(sql2+comment, varMap)
                nameInDB, = self.cur.fetchone()
                tmpLog.debug("inDB -> %s %s" % (nameInDB,dataset.name==nameInDB))
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            return True
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            return False


    # get and lock dataset with a query
    def getLockDatasets(self,sqlQuery,varMapGet,modTimeOffset='',getVersion=False):
        comment = ' /* DBProxy.getLockDatasets */'
        _logger.debug("getLockDatasets(%s,%s,%s)" % (sqlQuery,str(varMapGet),modTimeOffset))
        sqlGet  = "SELECT /*+ INDEX_RS_ASC(tab(STATUS,TYPE,MODIFICATIONDATE)) */ vuid,name,modificationdate,version,transferStatus FROM ATLAS_PANDA.Datasets tab WHERE " + sqlQuery
        sqlLock = "UPDATE ATLAS_PANDA.Datasets SET modificationdate=CURRENT_DATE"
        if modTimeOffset != '':
            sqlLock += "+%s" % modTimeOffset
        sqlLock += ",transferStatus=MOD(transferStatus+1,10)"
        if getVersion:
            sqlLock += ",version=:version"
        sqlLock += " WHERE vuid=:vuid AND transferStatus=:transferStatus"
        retList = []
        try:
            # begin transaction
            self.conn.begin()
            # get datasets
            self.cur.arraysize = 1000000
            self.cur.execute(sqlGet+comment,varMapGet)
            res = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # loop over all datasets
            if res is not None and len(res) != 0:
                for vuid,name,modificationdate,version,transferStatus in res:
                    # lock
                    varMapLock = {}
                    varMapLock[':vuid'] = vuid
                    varMapLock[':transferStatus'] = transferStatus
                    if getVersion:
                        try:
                            varMapLock[':version'] = str(int(version) + 1)
                        except Exception:
                            varMapLock[':version'] = str(1)
                    # begin transaction
                    self.conn.begin()
                    # update for lock
                    self.cur.execute(sqlLock+comment,varMapLock)
                    retU = self.cur.rowcount
                    # commit
                    if not self._commit():
                        raise RuntimeError('Commit error')
                    if retU > 0:
                        # append
                        if not getVersion:
                            retList.append((vuid,name,modificationdate))
                        else:
                            retList.append((vuid,name,modificationdate,version))
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # retrun
            return retList
        except Exception:
            # roll back
            self._rollback()
            # error
            type, value, traceBack = sys.exc_info()
            _logger.error("getLockDatasets : %s %s" % (type,value))
            return []


    # query dataset with map
    def queryDatasetWithMap(self,map):
        comment = ' /* DBProxy.queryDatasetWithMap */'
        _logger.debug("queryDatasetWithMap(%s)" % map)
        if 'name' in map:
            sql1  = """SELECT /*+ BEGIN_OUTLINE_DATA """
            sql1 += """INDEX_RS_ASC(@"SEL$1" "TAB"@"SEL$1" ("DATASETS"."NAME")) """
            sql1 += """OUTLINE_LEAF(@"SEL$1") ALL_ROWS """
            sql1 += """IGNORE_OPTIM_EMBEDDED_HINTS """
            sql1 += """END_OUTLINE_DATA */ """
            sql1 += "%s FROM ATLAS_PANDA.Datasets tab" % DatasetSpec.columnNames()
        else:
            sql1 = "SELECT %s FROM ATLAS_PANDA.Datasets" % DatasetSpec.columnNames()
        varMap = {}
        for key in map:
            if len(varMap)==0:
                sql1+= " WHERE %s=:%s" % (key,key)
            else:
                sql1+= " AND %s=:%s" % (key,key)
            varMap[':%s' % key] = map[key]
        try:
            # start transaction
            self.conn.begin()
            # select
            self.cur.arraysize = 100
            _logger.debug(sql1+comment+str(varMap))
            self.cur.execute(sql1+comment, varMap)
            res = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # instantiate Dataset
            if res is not None and len(res) != 0:
                dataset = DatasetSpec()
                dataset.pack(res[0])
                return dataset
            _logger.error("queryDatasetWithMap(%s) : dataset not found" % map)
            return None
        except Exception:
            # roll back
            self._rollback()
            type, value, traceBack = sys.exc_info()
            _logger.error("queryDatasetWithMap(%s) : %s %s" % (map,type,value))
            return None


    # update dataset
    def updateDataset(self,datasets,withLock,withCriteria,criteriaMap):
        comment = ' /* DBProxy.updateDataset */'
        _logger.debug("updateDataset()")
        sql1 = "UPDATE ATLAS_PANDA.Datasets SET %s " % DatasetSpec.bindUpdateExpression()
        sql1+= "WHERE vuid=:vuid"
        if withCriteria != "":
            sql1+= " AND %s" % withCriteria
        retList = []
        try:
            # start transaction
            self.conn.begin()
            for dataset in datasets:
                _logger.debug("updateDataset(%s,%s)" % (dataset.name,dataset.status))
                # time information
                dataset.modificationdate = datetime.datetime.utcnow()
                # update
                varMap = dataset.valuesMap()
                varMap[':vuid'] = dataset.vuid
                for cKey in criteriaMap:
                    varMap[cKey] = criteriaMap[cKey]
                _logger.debug(sql1+comment+str(varMap))
                self.cur.execute(sql1+comment, varMap)
                retU = self.cur.rowcount
                if retU != 0 and retU != 1:
                    raise RuntimeError('Invalid retrun %s' % retU)
                retList.append(retU)
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            _logger.debug("updateDataset() ret:%s" % retList)
            return retList
        except Exception:
            # roll back
            self._rollback()
            type, value, traceBack = sys.exc_info()
            _logger.error("updateDataset() : %s %s" % (type,value))
            return []


    # delete dataset
    def deleteDataset(self,name):
        comment = ' /* DBProxy.deleteDataset */'
        sql1 = "DELETE /*+ INDEX(tab DATASETS_NAME_IDX)*/ FROM ATLAS_PANDA.Datasets tab WHERE name=:name"
        try:
            # start transaction
            self.conn.begin()
            # delete
            varMap = {}
            varMap[':name'] = name
            self.cur.execute(sql1+comment, varMap)
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            return True
        except Exception:
            # roll back
            self._rollback()
            # error
            type, value, traceBack = sys.exc_info()
            _logger.error("deleteDataset() : %s %s" % (type,value))
            return False


    # get serial number for dataset, insert dummy datasets to increment SN
    def getSerialNumber(self,datasetname,definedFreshFlag=None):
        comment = ' /* DBProxy.getSerialNumber */'
        try:
            _logger.debug("getSerialNumber(%s,%s)" % (datasetname,definedFreshFlag))
            if isinstance(datasetname,unicode):
                datasetname = datasetname.encode('ascii','ignore')
                _logger.debug("getSerialNumber converted unicode for %s" % datasetname)
            # start transaction
            self.conn.begin()
            # check freashness
            if definedFreshFlag is None:
                # select
                varMap = {}
                varMap[':name'] = datasetname
                varMap[':type'] = 'output'
                sql = "SELECT /*+ INDEX_RS_ASC(TAB (DATASETS.NAME)) */ COUNT(*) FROM ATLAS_PANDA.Datasets tab WHERE type=:type AND name=:name"
                self.cur.arraysize = 100
                self.cur.execute(sql+comment, varMap)
                res = self.cur.fetchone()
                # fresh dataset or not
                if res is not None and len(res) != 0 and res[0] > 0:
                    freshFlag = False
                else:
                    freshFlag = True
            else:
                # use predefined flag
                freshFlag = definedFreshFlag
            # get serial number
            if self.backend == 'oracle':
                sql = "SELECT ATLAS_PANDA.SUBCOUNTER_SUBID_SEQ.nextval FROM dual"
                self.cur.arraysize = 100
                self.cur.execute(sql+comment, {})
                sn, = self.cur.fetchone()
            else:
                # panda_config.backend == 'mysql'
                ### fake sequence
                sql = " INSERT INTO ATLAS_PANDA.SUBCOUNTER_SUBID_SEQ (col) VALUES (NULL) "
                self.cur.arraysize = 100
                self.cur.execute(sql + comment, {})
                sql2 = """ SELECT LAST_INSERT_ID() """
                self.cur.execute(sql2 + comment, {})
                sn, = self.cur.fetchone()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            _logger.debug("getSerialNumber : %s %s" % (sn,freshFlag))
            return (sn,freshFlag)
        except Exception:
            # roll back
            self._rollback()
            # error
            type, value, traceBack = sys.exc_info()
            _logger.error("getSerialNumber() : %s %s" % (type,value))
            return (-1,False)


    # get serial number for group job
    def getSerialNumberForGroupJob(self,name):
        comment = ' /* DBProxy.getSerialNumberForGroupJob */'
        retVal = {'sn':'','status':False}
        try:
            _logger.debug("getSerialNumberForGroupJob(%s)" % name)
            # start transaction
            self.conn.begin()
            # get serial number
            if self.backend == 'oracle':
                sql = "SELECT ATLAS_PANDA.GROUP_JOBID_SEQ.nextval FROM dual"
                self.cur.execute(sql+comment, {})
                sn, = self.cur.fetchone()
            else:
                # panda_config.backend == 'mysql'
                ### fake sequence
                sql = " INSERT INTO ATLAS_PANDA.GROUP_JOBID_SEQ (col) VALUES (NULL) "
                self.cur.arraysize = 100
                self.cur.execute(sql + comment, {})
                sql2 = """ SELECT LAST_INSERT_ID() """
                self.cur.execute(sql2 + comment, {})
                sn, = self.cur.fetchone()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # return
            retVal['sn'] = sn
            retVal['status'] = True
            _logger.debug("getSerialNumberForGroupJob : %s %s" % (name,str(retVal)))
            return retVal
        except Exception:
            # roll back
            self._rollback()
            # error
            errtype,errvalue = sys.exc_info()[:2]
            _logger.error("getSerialNumberForGroupJob : %s %s" % (errtype,errvalue))
            retVal['status'] = False
            return retVal


    # change job priorities
    def changeJobPriorities(self,newPrioMap):
        comment = ' /* DBProxy.changeJobPriorities */'
        try:
            _logger.debug("changeJobPriorities start")
            sql  = "UPDATE %s SET currentPriority=:currentPriority,assignedPriority=:assignedPriority "
            sql += "WHERE PandaID=:PandaID"
            # loop over all PandaIDs
            for pandaID in newPrioMap:
                newPrio = newPrioMap[pandaID]
                varMap = {}
                varMap[':PandaID'] = pandaID
                varMap[':currentPriority']  = newPrio
                varMap[':assignedPriority'] = newPrio
                _logger.debug("changeJobPriorities PandaID=%s -> prio=%s" % (pandaID,newPrio))
                # start transaction
                self.conn.begin()
                # try active tables
                retU = None
                for tableName in ['ATLAS_PANDA.jobsActive4','ATLAS_PANDA.jobsDefined4','ATLAS_PANDA.jobsWaiting4']:
                    # execute
                    self.cur.execute((sql % tableName)+comment,varMap)
                    retU = self.cur.rowcount
                    if retU > 0:
                        break
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                _logger.debug("changeJobPriorities PandaID=%s retU=%s" % (pandaID,retU))
            # return
            _logger.debug("changeJobPriorities done")
            return True,''
        except Exception:
            # roll back
            self._rollback()
            # error
            errtype,errvalue = sys.exc_info()[:2]
            _logger.error("changeJobPriorities : %s %s" % (errtype,errvalue))
            return False,'database error'


    # update transfer status for a dataset
    def updateTransferStatus(self,datasetname,bitMap):
        comment = ' /* DBProxy.updateTransferStatus */'
        try:
            _logger.debug("updateTransferStatus(%s,%s)" % (datasetname,hex(bitMap)))
            # start transaction
            self.conn.begin()
            retTransSt = 0
            # update bitmap
            sqlU = 'UPDATE /*+ INDEX_RS_ASC(TAB("DATASETS"."NAME")) */ ATLAS_PANDA.Datasets tab SET transferStatus=ATLAS_PANDA.BITOR(transferStatus,:bitMap) WHERE name=:name'
            varMap = {}
            varMap[':bitMap'] = bitMap
            varMap[':name'] = datasetname
            retU = self.cur.execute(sqlU+comment, varMap)
            # get transferStatus
            sqlS = 'SELECT /*+ INDEX_RS_ASC(TAB("DATASETS"."NAME")) */ transferStatus FROM ATLAS_PANDA.Datasets tab WHERE name=:name'
            varMap = {}
            varMap[':name'] = datasetname
            self.cur.arraysize = 10
            retS = self.cur.execute(sqlS+comment, varMap)
            resS = self.cur.fetchall()
            if resS is not None and len(resS) != 0:
                retTransSt = resS[0][0]
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            _logger.debug("updateTransferStatus : %s" % hex(retTransSt))
            return retTransSt
        except Exception:
            # roll back
            self._rollback()
            # error
            type, value, traceBack = sys.exc_info()
            _logger.error("updateTransferStatus : %s %s" % (type,value))
            return 0


    # get CloudTask. If not exist, create it
    def getCloudTask(self,tid):
        comment = ' /* getCloudTask */'
        try:
            _logger.debug("getCloudTask(%s)" % tid)
            # check tid
            if tid in [None,'NULL']:
                _logger.error("invalid TID : %s" % tid)
                return None
            # start transaction
            self.conn.begin()
            # get CloudTask
            sql  = "SELECT %s FROM ATLAS_PANDA.cloudtasks " % CloudTaskSpec.columnNames()
            sql += "WHERE taskid=:taskid"
            varMap = {}
            varMap[':taskid'] = tid
            # select
            self.cur.arraysize = 10
            self.cur.execute(sql+comment, varMap)
            res = self.cur.fetchall()
            # already exist
            if res is not None and len(res) != 0:
                # instantiate CloudTask
                cloudTask = CloudTaskSpec()
                cloudTask.pack(res[0])
                # update tmod if status is defined
                if cloudTask.status == 'defined':
                    sql = "UPDATE ATLAS_PANDA.cloudtasks SET tmod=CURRENT_DATE WHERE taskid=:taskid"
                    varMap = {}
                    varMap[':taskid'] = cloudTask.taskid
                    self.cur.execute(sql+comment, varMap)
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                return cloudTask
            # insert new CloudTask
            _logger.debug("insert new CloudTask")
            cloudTask = CloudTaskSpec()
            cloudTask.taskid = tid
            cloudTask.status = 'defined'
            sql = "INSERT INTO ATLAS_PANDA.cloudtasks (id,taskid,status,tmod,tenter) VALUES(ATLAS_PANDA.CLOUDTASKS_ID_SEQ.nextval,:taskid,:status,CURRENT_DATE,CURRENT_DATE)"
            sql+= " RETURNING id INTO :newID"
            varMap = {}
            varMap[':taskid'] = cloudTask.taskid
            varMap[':status'] = cloudTask.status
            varMap[':newID']  = self.cur.var(varNUMBER)
            self.cur.execute(sql+comment, varMap)
            # get id
            val = self.getvalue_corrector(self.cur.getvalue(varMap[':newID']))
            cloudTask.id = long(val)
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            _logger.debug("return new CloudTask")
            return cloudTask
        except Exception:
            # roll back
            self._rollback()
            # error
            type, value, traceBack = sys.exc_info()
            _logger.error("getCloudTask() : %s %s" % (type,value))
            return None


    # set cloud to CloudTask
    def setCloudTask(self,cloudTask):
        comment = ' /* setCloudTask */'
        try:
            _logger.debug("setCloudTask(id=%s,taskid=%s)" % (cloudTask.id,cloudTask.taskid))
            sql  = "UPDATE ATLAS_PANDA.cloudtasks SET cloud=:cloud,status=:newStatus,tmod=CURRENT_DATE WHERE id=:id AND status=:oldStatus"
            # start transaction
            self.conn.begin()
            # update
            varMap = {}
            varMap[':cloud'] = cloudTask.cloud
            varMap[':id'] = cloudTask.id
            varMap[':newStatus'] = 'assigned'
            varMap[':oldStatus'] = 'defined'
            self.cur.execute(sql+comment, varMap)
            retU = self.cur.rowcount
            # succeeded
            if retU == 1:
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                return cloudTask
            # read if it is already set by another thread
            sql  = "SELECT %s FROM ATLAS_PANDA.cloudtasks " % CloudTaskSpec.columnNames()
            sql += "WHERE id=:id"
            varMap = {}
            varMap[':id'] = cloudTask.id
            # select
            self.cur.arraysize = 10
            retS = self.cur.execute(sql+comment, varMap)
            res = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # retrun CloudTask
            if res is not None and len(res) != 0:
                # instantiate CloudTask
                cloudTask = CloudTaskSpec()
                cloudTask.pack(res[0])
                return cloudTask
            _logger.error("setCloudTask() : cannot find CloudTask for %s" % cloudTask.id)
            return None
        except Exception:
            # roll back
            self._rollback()
            # error
            type, value, traceBack = sys.exc_info()
            _logger.error("setCloudTask() : %s %s" % (type,value))
            return None


    # see CloudTask
    def seeCloudTask(self,tid):
        comment = ' /* seeCloudTask */'
        try:
            _logger.debug("seeCloudTask(%s)" % tid)
            # check tid
            if tid in [None,'NULL']:
                _logger.error("invalid TID : %s" % tid)
                return None
            # start transaction
            self.conn.begin()
            # select
            sql  = "SELECT cloud FROM ATLAS_PANDA.cloudtasks WHERE taskid=:taskid"
            varMap = {}
            varMap[':taskid'] = tid
            self.cur.execute(sql+comment, varMap)
            res = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # existing task
            if res is not None and len(res) != 0:
                # return cloud
                return res[0][0]
            else:
                return None
        except Exception:
            # roll back
            self._rollback()
            # error
            type, value, traceBack = sys.exc_info()
            _logger.error("seeCloudTask() : %s %s" % (type,value))
            return None


    # reset modification time of a task to shorten retry interval
    def resetTmodCloudTask(self,tid):
        comment = ' /* resetTmodCloudTask */'
        try:
            _logger.debug("resetTmodCloudTask %s" % tid)
            # check tid
            if tid in [None,'NULL']:
                _logger.error("invalid TID : %s" % tid)
                return None
            # start transaction
            self.conn.begin()
            # update
            sql  = "UPDATE ATLAS_PANDA.cloudtasks SET tmod=:tmod WHERE taskid=:taskid"
            varMap = {}
            varMap[':taskid'] = tid
            varMap[':tmod'] = datetime.datetime.utcnow() - datetime.timedelta(minutes=165)
            self.cur.execute(sql+comment, varMap)
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            return True
        except Exception:
            # roll back
            self._rollback()
            # error
            type, value, traceBack = sys.exc_info()
            _logger.error("resetTmodCloudTask : %s %s" % (type,value))
            return False


    # get assigning task
    def getAssigningTask(self):
        comment = ' /* getAssigningTask */'
        try:
            _logger.debug("getAssigningTask")
            timeLimit  = datetime.datetime.utcnow() - datetime.timedelta(hours=3)
            # start transaction
            self.conn.begin()
            # select
            sql = "SELECT taskid FROM ATLAS_PANDA.cloudtasks WHERE status=:status AND tmod>:tmod"
            varMap = {}
            varMap[':tmod']   = timeLimit
            varMap[':status'] = 'defined'
            self.cur.arraysize = 100
            self.cur.execute(sql+comment, varMap)
            res = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # loop over all taskid
            retList = []
            if res is not None:
                for tid, in res:
                    retList.append(tid)
            # return
            _logger.debug("getAssigningTask ret:%s" % retList)
            return retList
        except Exception:
            # roll back
            self._rollback()
            # error
            type, value, traceBack = sys.exc_info()
            _logger.error("getAssigningTask : %s %s" % (type,value))
            return []


    # set CloudTask by user
    def setCloudTaskByUser(self,user,tid,cloud,status,forceUpdate=False):
        comment = ' /* setCloudTaskByUser */'
        try:
            _logger.debug("setCloudTaskByUser(tid=%s,cloud=%s,status=%s) by %s" % (tid,cloud,status,user))
            # check tid
            if tid in [None,'NULL']:
                tmpMsg = "invalid TID : %s" % tid
                _logger.error(tmpMsg)
                return "ERROR: " + tmpMsg
            # check status
            statusList = ['tobeaborted','assigned']
            if status not in statusList:
                tmpMsg = "invalid status=%s. Must be one of %s" % (status,str(statusList))
                _logger.error(tmpMsg)
                return "ERROR: " + tmpMsg
            # start transaction
            self.conn.begin()
            # get CloudTask
            sql  = "SELECT %s FROM ATLAS_PANDA.cloudtasks " % CloudTaskSpec.columnNames()
            sql += "WHERE taskid=:taskid"
            varMap = {}
            varMap[':taskid'] = tid
            # select
            self.cur.arraysize = 10
            self.cur.execute(sql+comment, varMap)
            res = self.cur.fetchall()
            # already exist
            if res is not None and len(res) != 0:
                # set status
                sql  = "UPDATE ATLAS_PANDA.cloudtasks SET status=:status,tmod=CURRENT_DATE"
                if forceUpdate:
                    sql += ",cloud=:cloud"
                sql += " WHERE taskid=:taskid"
                varMap = {}
                varMap[':taskid'] = tid
                varMap[':status'] = status
                if forceUpdate:
                    varMap[':cloud'] = cloud
                self.cur.execute(sql+comment, varMap)
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                return "SUCCEEDED"
            # insert new CloudTask
            sql = "INSERT INTO ATLAS_PANDA.cloudtasks (id,taskid,status,tmod,tenter) VALUES(ATLAS_PANDA.CLOUDTASKS_ID_SEQ.nextval,:taskid,:status,CURRENT_DATE,CURRENT_DATE)"
            varMap = {}
            varMap[':taskid'] = tid
            varMap[':status'] = status
            self.cur.execute(sql+comment, varMap)
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            return "SUCCEEDED"
        except Exception:
            # roll back
            self._rollback()
            # error
            errType,errValue = sys.exc_info()[:2]
            _logger.error("setCloudTaskByUser() : %s %s" % (errType,errValue))
            return "ERROR: database error"


    # query files with map
    def queryFilesWithMap(self,map):
        comment = ' /* DBProxy.queryFilesWithMap */'
        _logger.debug("queryFilesWithMap()")
        sql1 = "SELECT PandaID,%s FROM ATLAS_PANDA.filesTable4" % FileSpec.columnNames()
        varMap = {}
        for key in map:
            if len(varMap)==0:
                sql1+= " WHERE %s=:%s" % (key,key)
            else:
                sql1+= " AND %s=:%s" % (key,key)
            varMap[':%s' % key] = map[key]
        nTry=3
        for iTry in range(nTry):
            try:
                # start transaction
                self.conn.begin()
                # select
                self.cur.arraysize = 10000
                self.cur.execute(sql1+comment, varMap)
                res = self.cur.fetchall()
                _logger.debug("queryFilesWithMap() : %s" % str(res))
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                # instantiate files
                retList = []
                for item in res:
                    # instantiate dummy JobSpec obj for PandaID
                    job = JobSpec()
                    job.PandaID = item[0]
                    # instantiate file
                    file = FileSpec()
                    file.pack(item[1:])
                    # set owner
                    file.setOwner(job)
                    # append
                    retList.append(file)
                return retList
            except Exception:
                # roll back
                self._rollback()
                if iTry+1 < nTry:
                    _logger.debug("queryFilesWithMap retry : %s" % iTry)
                    time.sleep(random.randint(10,20))
                    continue
                type, value, traceBack = sys.exc_info()
                _logger.error("queryFilesWithMap : %s %s" % (type,value))
                return []


    # count the number of files with map
    def countFilesWithMap(self,map):
        comment = ' /* DBProxy.countFilesWithMap */'
        sql1 = "SELECT /*+ index(tab FILESTABLE4_DESTDBLOCK_IDX) */ COUNT(*) FROM ATLAS_PANDA.filesTable4 tab"
        varMap = {}
        for key in map:
            if len(varMap)==0:
                sql1+= " WHERE %s=:%s" % (key,key)
            else:
                sql1+= " AND %s=:%s" % (key,key)
            varMap[':%s' % key] = map[key]
        nTry=3
        for iTry in range(nTry):
            try:
                # start transaction
                self.conn.begin()
                # select
                _logger.debug("countFilesWithMap() : %s %s" % (sql1,str(map)))
                self.cur.arraysize = 10
                retS = self.cur.execute(sql1+comment, varMap)
                res = self.cur.fetchone()
                _logger.debug("countFilesWithMap() : %s %s" % (retS,str(res)))
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                nFiles=0
                if res is not None:
                    nFiles=res[0]
                return nFiles
            except Exception:
                # roll back
                self._rollback()
                if iTry+1 < nTry:
                    _logger.debug("countFilesWithMap() retry : %s" % iTry)
                    time.sleep(random.randint(10,20))
                    continue
                type, value, traceBack = sys.exc_info()
                _logger.error("countFilesWithMap(%s) : %s %s" % (map,type,value))
                return -1


    # count the number of pending files
    def countPendingFiles(self,pandaID,forInput=True):
        comment = ' /* DBProxy.countPendingFiles */'
        varMap = {}
        varMap[':pandaID'] = pandaID
        varMap[':status'] = 'ready'
        if forInput:
            sql1 = "SELECT COUNT(*) FROM ATLAS_PANDA.filesTable4 WHERE PandaID=:pandaID AND type=:type AND status<>:status "
            varMap[':type'] = 'input'
        else:
            sql1 = "SELECT COUNT(*) FROM ATLAS_PANDA.filesTable4 WHERE PandaID=:pandaID AND type IN (:type1,:type2) AND status<>:status "
            varMap[':type1'] = 'output'
            varMap[':type2'] = 'log'
        try:
            # start transaction
            self.conn.begin()
            # select
            _logger.debug("countPendingFiles : %s start" % pandaID)
            self.cur.arraysize = 10
            retS = self.cur.execute(sql1+comment, varMap)
            res = self.cur.fetchone()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            nFiles = -1
            if res is not None:
                nFiles=res[0]
            _logger.debug("countPendingFiles : %s -> %s" % (pandaID,nFiles))
            return nFiles
        except Exception:
            # roll back
            self._rollback()
            errType,errValue = sys.exc_info()[:2]
            _logger.error("countPendingFiles : %s : %s %s" % (pandaID,errType,errValue))
            return -1


    # get datasets associated with file
    def getDatasetWithFile(self,lfn,jobPrioity=0):
        comment = ' /* DBProxy.getDatasetWithFile */'
        varMap = {}
        varMap[':lfn'] = lfn
        varMap[':status1'] = 'pending'
        varMap[':status2'] = 'transferring'
        sql1  = "SELECT PandaID,status,destinationDBlock,destinationDBlockToken,dispatchDBlock FROM ATLAS_PANDA.filesTable4 "
        sql1 += "WHERE lfn=:lfn AND status IN (:status1,:status2) AND modificationTime<CURRENT_DATE-60 "
        try:
            # start transaction
            self.conn.begin()
            retMap = {}
            # select
            _logger.debug("getDatasetWithFile : %s start" % lfn)
            self.cur.arraysize = 1000
            retS = self.cur.execute(sql1+comment, varMap)
            res = self.cur.fetchall()
            if res is not None and len(res) != 0:
                for pandaID,status,destinationDBlock,destinationDBlockToken,dispatchDBlock in res:
                    varMap = {}
                    varMap[':PandaID'] = pandaID
                    if status == 'pending':
                        # input
                        sqlP = 'SELECT computingSite,prodSourceLabel FROM ATLAS_PANDA.jobsDefined4 '
                        varMap[':jobStatus'] = 'assigned'
                        dsName  = dispatchDBlock
                        dsToken = ''
                    else:
                        # output
                        sqlP = 'SELECT destinationSE,prodSourceLabel FROM ATLAS_PANDA.jobsActive4 '
                        varMap[':jobStatus'] = 'transferring'
                        dsName  = destinationDBlock
                        dsToken = destinationDBlockToken
                    # check duplication
                    if dsName in retMap:
                        continue
                    # get site info
                    sqlP += 'WHERE PandaID=:PandaID AND jobStatus=:jobStatus AND currentPriority>=:currentPriority '
                    varMap[':currentPriority'] = jobPrioity
                    self.cur.execute(sqlP+comment, varMap)
                    resP = self.cur.fetchone()
                    # append
                    if resP is not None and resP[1] in ['managed','test']:
                        retMap[dsName] = (resP[0],dsToken)
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            _logger.debug("getDatasetWithFile : %s -> %s" % (lfn,str(retMap)))
            return retMap
        except Exception:
            # roll back
            self._rollback()
            errType,errValue = sys.exc_info()[:2]
            _logger.error("getDatasetWithFile : %s : %s %s" % (lfn,errType,errValue))
            return {}


    # get input files currently in use for analysis
    def getFilesInUseForAnal(self,outDataset):
        comment = ' /* DBProxy.getFilesInUseForAnal */'
        sqlSub  = "SELECT destinationDBlock,PandaID FROM ATLAS_PANDA.filesTable4 "
        sqlSub += "WHERE dataset=:dataset AND type IN (:type1,:type2) GROUP BY destinationDBlock,PandaID"
        sqlPaA  = "SELECT jobDefinitionID,prodUserName FROM ATLAS_PANDA.jobsDefined4 "
        sqlPaA += "WHERE PandaID=:PandaID "
        sqlPaA += "UNION "
        sqlPaA += "SELECT jobDefinitionID,prodUserName FROM ATLAS_PANDA.jobsActive4 "
        sqlPaA += "WHERE PandaID=:PandaID "
        sqlPan  = "SELECT jobDefinitionID,prodUserName FROM ATLAS_PANDA.jobsArchived4 "
        sqlPan += "WHERE PandaID=:PandaID AND modificationTime<=CURRENT_DATE "
        sqlPan += "UNION "
        sqlPan += "SELECT jobDefinitionID,prodUserName FROM ATLAS_PANDAARCH.jobsArchived "
        sqlPan += "WHERE PandaID=:PandaID AND modificationTime>(CURRENT_DATE-30)"
        sqlIdA  = "SELECT PandaID,jobStatus FROM ATLAS_PANDA.jobsArchived4 "
        sqlIdA += "WHERE prodUserName=:prodUserName AND jobDefinitionID=:jobDefinitionID "
        sqlIdA += "AND prodSourceLabel=:prodSourceLabel1 "
        sqlIdL  = "SELECT /*+ NO_INDEX(tab JOBS_MODTIME_IDX) INDEX_COMBINE(tab JOBS_PRODUSERNAME_IDX JOBS_JOBDEFID_IDX) */ "
        sqlIdL += "PandaID,jobStatus FROM ATLAS_PANDAARCH.jobsArchived tab "
        sqlIdL += "WHERE prodUserName=:prodUserName AND jobDefinitionID=:jobDefinitionID "
        sqlIdL += "AND prodSourceLabel=:prodSourceLabel1 AND modificationTime>(CURRENT_DATE-30) "
        sqlDis  = "SELECT distinct dispatchDBlock FROM ATLAS_PANDA.filesTable4 "
        sqlDis += "WHERE PandaID=:PandaID AND type=:type AND dispatchDBlock IS NOT NULL AND modificationTime <= CURRENT_DATE"
        sqlLfn  = "SELECT /*+ index(tab FILESTABLE4_DISPDBLOCK_IDX) */ lfn,PandaID FROM ATLAS_PANDA.filesTable4 tab "
        sqlLfn += "WHERE dispatchDBlock=:dispatchDBlock AND type=:type "
        sqlLfn += "AND (destinationDBlockToken IS NULL OR destinationDBlockToken<>:noshadow) AND modificationTime<=CURRENT_DATE"
        nTry=3
        for iTry in range(nTry):
            inputFilesList = []
            try:
                # start transaction
                self.conn.begin()
                # get sub datasets
                varMap = {}
                varMap[':dataset'] = outDataset
                varMap[':type1'] = 'output'
                varMap[':type2'] = 'log'
                _logger.debug("getFilesInUseForAnal : %s %s" % (sqlSub,str(varMap)))
                self.cur.arraysize = 100000
                retS = self.cur.execute(sqlSub+comment, varMap)
                res = self.cur.fetchall()
                subDSpandaIDmap = {}
                checkedPandaIDs = {}
                for subDataset,pandaID in res:
                    # avoid redundunt lookup
                    if pandaID in checkedPandaIDs:
                        continue
                    if subDataset in subDSpandaIDmap:
                        # append jobs as running since they are not in archived tables
                        if pandaID not in subDSpandaIDmap[subDataset]:
                            checkedPandaIDs[pandaID] = 'running'
                            subDSpandaIDmap[subDataset].append(pandaID)
                        continue
                    # look for jobdefID and userName
                    varMap = {}
                    varMap[':PandaID'] = pandaID
                    _logger.debug("getFilesInUseForAnal : %s %s" % (sqlPaA,str(varMap)))
                    retP = self.cur.execute(sqlPaA+comment, varMap)
                    resP = self.cur.fetchall()
                    if len(resP) != 0:
                        jobDefinitionID,prodUserName = resP[0]
                    else:
                        _logger.debug("getFilesInUseForAnal : %s %s" % (sqlPan,str(varMap)))
                        retP = self.cur.execute(sqlPan+comment, varMap)
                        resP = self.cur.fetchall()
                        if len(resP) != 0:
                            jobDefinitionID,prodUserName = resP[0]
                        else:
                            continue
                    # get PandaIDs with obdefID and userName
                    tmpPandaIDs = []
                    varMap = {}
                    varMap[':prodUserName']     = prodUserName
                    varMap[':jobDefinitionID']  = jobDefinitionID
                    varMap[':prodSourceLabel1'] = 'user'
                    _logger.debug("getFilesInUseForAnal : %s %s" % (sqlIdA,str(varMap)))
                    retID = self.cur.execute(sqlIdA+comment, varMap)
                    resID = self.cur.fetchall()
                    for tmpPandaID,tmpJobStatus in resID:
                        checkedPandaIDs[tmpPandaID] = tmpJobStatus
                        tmpPandaIDs.append(tmpPandaID)
                    _logger.debug("getFilesInUseForAnal : %s %s" % (sqlIdL,str(varMap)))
                    retID = self.cur.execute(sqlIdL+comment, varMap)
                    resID = self.cur.fetchall()
                    for tmpPandaID,tmpJobStatus in resID:
                        if tmpPandaID not in tmpPandaIDs:
                            checkedPandaIDs[tmpPandaID] = tmpJobStatus
                            tmpPandaIDs.append(tmpPandaID)
                    # append
                    if subDataset not in subDSpandaIDmap:
                        subDSpandaIDmap[subDataset] = []
                    for tmpPandaID in tmpPandaIDs:
                        # reuse failed files if jobs are in Archived since they cannot change back to active
                        if checkedPandaIDs[tmpPandaID] in ['failed','cancelled','closed']:
                            continue
                        # collect PandaIDs
                        subDSpandaIDmap[subDataset].append(tmpPandaID)
                # loop over all sub datasets
                for subDataset in subDSpandaIDmap:
                    activePandaIDs = subDSpandaIDmap[subDataset]
                    # skip empty
                    if activePandaIDs == []:
                        continue
                    # get dispatchDBlocks
                    pandaID = activePandaIDs[0]
                    varMap = {}
                    varMap[':PandaID'] = pandaID
                    varMap[':type'] = 'input'
                    _logger.debug("getFilesInUseForAnal : %s %s" % (sqlDis,str(varMap)))
                    self.cur.arraysize = 10000
                    retD = self.cur.execute(sqlDis+comment, varMap)
                    resD = self.cur.fetchall()
                    # get LFNs
                    for disDataset, in resD:
                        # use new style only
                        if not disDataset.startswith('user_disp.'):
                            continue
                        varMap = {}
                        varMap[':dispatchDBlock'] = disDataset
                        varMap[':type'] = 'input'
                        varMap[':noshadow'] = 'noshadow'
                        _logger.debug("getFilesInUseForAnal : %s %s" % (sqlLfn,str(varMap)))
                        self.cur.arraysize = 100000
                        retL = self.cur.execute(sqlLfn+comment, varMap)
                        resL = self.cur.fetchall()
                        # append
                        for lfn,filePandaID in resL:
                            # skip files used by archived failed or cancelled jobs
                            if filePandaID in activePandaIDs and lfn not in inputFilesList:
                                inputFilesList.append(lfn)
                # commit
                if not self._commit():


                     RuntimeError('Commit error')
                _logger.debug("getFilesInUseForAnal : %s" % len(inputFilesList))
                return inputFilesList
            except Exception:
                # roll back
                self._rollback()
                if iTry+1 < nTry:
                    _logger.debug("inputFilesList retry : %s" % iTry)
                    time.sleep(random.randint(10,20))
                    continue
                type, value, traceBack = sys.exc_info()
                _logger.error("inputFilesList(%s) : %s %s" % (outDataset,type,value))
                return []


    # get list of dis dataset to get input files in shadow
    def getDisInUseForAnal(self,outDataset):
        comment = ' /* DBProxy.getDisInUseForAnal */'
        sqlSub  = "SELECT destinationDBlock,PandaID,status FROM ATLAS_PANDA.filesTable4 "
        sqlSub += "WHERE dataset=:dataset AND type=:type1 GROUP BY destinationDBlock,PandaID,status"
        sqlPaA  = "SELECT jobStatus FROM ATLAS_PANDA.jobsDefined4 "
        sqlPaA += "WHERE PandaID=:PandaID "
        sqlPaA += "UNION "
        sqlPaA += "SELECT jobStatus FROM ATLAS_PANDA.jobsActive4 "
        sqlPaA += "WHERE PandaID=:PandaID "
        sqlPan  = "SELECT jobStatus FROM ATLAS_PANDA.jobsArchived4 "
        sqlPan += "WHERE PandaID=:PandaID AND modificationTime<=CURRENT_DATE "
        sqlPan += "UNION "
        sqlPan += "SELECT jobStatus FROM ATLAS_PANDAARCH.jobsArchived "
        sqlPan += "WHERE PandaID=:PandaID AND modificationTime>(CURRENT_DATE-30)"
        sqlDis  = "SELECT distinct dispatchDBlock FROM ATLAS_PANDA.filesTable4 "
        sqlDis += "WHERE PandaID=:PandaID AND type=:type AND dispatchDBlock IS NOT NULL AND modificationTime <= CURRENT_DATE"
        inputDisList = []
        try:
            timeStart = datetime.datetime.utcnow()
            _logger.debug("getDisInUseForAnal start for %s" % outDataset)
            # start transaction
            self.conn.begin()
            # get sub datasets
            varMap = {}
            varMap[':dataset'] = outDataset
            varMap[':type1'] = 'log'
            _logger.debug("getDisInUseForAnal : %s %s" % (sqlSub,str(varMap)))
            self.cur.arraysize = 100000
            retS = self.cur.execute(sqlSub+comment, varMap)
            res = self.cur.fetchall()
            subDSpandaIDmap = {}
            checkedPandaIDs = {}
            for subDataset,pandaID,fileStatus in res:
                # add map
                if subDataset not in subDSpandaIDmap:
                    subDSpandaIDmap[subDataset] = []
                # check job status
                if fileStatus != 'ready':
                    varMap = {}
                    varMap[':PandaID'] = pandaID
                    _logger.debug("getDisInUseForAnal : %s %s" % (sqlPaA,str(varMap)))
                    retP = self.cur.execute(sqlPaA+comment, varMap)
                    resP = self.cur.fetchall()
                    if len(resP) != 0:
                        # append jobs as running since they are not in archived tables yet
                        checkedPandaIDs[pandaID] = 'running'
                        subDSpandaIDmap[subDataset].append(pandaID)
                    else:
                        _logger.debug("getDisInUseForAnal : %s %s" % (sqlPan,str(varMap)))
                        retP = self.cur.execute(sqlPan+comment, varMap)
                        resP = self.cur.fetchall()
                        if len(resP) != 0:
                            checkedPandaIDs[pandaID], = resP[0]
                            # reuse failed files if jobs are in Archived since they cannot change back to active
                            if checkedPandaIDs[pandaID] in ['failed','cancelled','closed']:
                                continue
                            # collect PandaIDs
                            subDSpandaIDmap[subDataset].append(pandaID)
                        else:
                            # not found
                            continue
                else:
                    # no job lookup since file was sucessfully finished
                    checkedPandaIDs[pandaID] = 'finished'
                    # collect PandaIDs
                    subDSpandaIDmap[subDataset].append(pandaID)
            # loop over all sub datasets
            for subDataset in subDSpandaIDmap:
                activePandaIDs = subDSpandaIDmap[subDataset]
                # skip empty
                if activePandaIDs == []:
                    continue
                resDisList = []
                # get dispatchDBlocks
                pandaID = activePandaIDs[0]
                varMap = {}
                varMap[':PandaID'] = pandaID
                varMap[':type'] = 'input'
                _logger.debug("getDisInUseForAnal : %s %s" % (sqlDis,str(varMap)))
                self.cur.arraysize = 10000
                retD = self.cur.execute(sqlDis+comment, varMap)
                resD = self.cur.fetchall()
                # get shadow dis
                for disDataset, in resD:
                    # use new style only
                    if not disDataset.startswith('user_disp.'):
                        continue
                    if disDataset not in resDisList:
                        resDisList.append(disDataset)
                # append
                if resDisList != []:
                    inputDisList.append((resDisList,activePandaIDs))
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            timeDelta = datetime.datetime.utcnow()-timeStart
            _logger.debug("getDisInUseForAnal end for %s len=%s time=%ssec" % (outDataset,len(inputDisList),timeDelta.seconds))
            return inputDisList
        except Exception:
            # roll back
            self._rollback()
            errtype,errvalue = sys.exc_info()[:2]
            _logger.error("getDisInUseForAnal(%s) : %s %s" % (outDataset,errtype,errvalue))
            return None


    # get input LFNs currently in use for analysis with shadow dis
    def getLFNsInUseForAnal(self,inputDisList):
        comment = ' /* DBProxy.getLFNsInUseForAnal */'
        sqlLfn  = "SELECT /*+ index(tab FILESTABLE4_DISPDBLOCK_IDX) */ lfn,PandaID FROM ATLAS_PANDA.filesTable4 tab "
        sqlLfn += "WHERE dispatchDBlock=:dispatchDBlock AND type=:type "
        sqlLfn += "AND (destinationDBlockToken IS NULL OR destinationDBlockToken<>:noshadow) AND modificationTime<=CURRENT_DATE"
        inputFilesList = []
        try:
            token = datetime.datetime.utcnow().isoformat('/')
            # loop over all shadow dis datasets
            pandaIdLfnMap = {}
            for disDatasetList,activePandaIDs in inputDisList:
                for disDataset in disDatasetList:
                    # use new style only
                    if not disDataset.startswith('user_disp.'):
                        continue
                    # read LFNs and PandaIDs
                    if disDataset not in pandaIdLfnMap:
                        # start transaction
                        self.conn.begin()
                        varMap = {}
                        varMap[':dispatchDBlock'] = disDataset
                        varMap[':type'] = 'input'
                        varMap[':noshadow'] = 'noshadow'
                        _logger.debug("getLFNsInUseForAnal : <%s> %s %s" % (token,sqlLfn,str(varMap)))
                        timeStart = datetime.datetime.utcnow()
                        self.cur.arraysize = 100000
                        retL = self.cur.execute(sqlLfn+comment, varMap)
                        resL = self.cur.fetchall()
                        # commit
                        timeDelta = datetime.datetime.utcnow()-timeStart
                        _logger.debug("getLFNsInUseForAnal : <%s> %s time=%ssec commit" % (token,disDataset,timeDelta.seconds))
                        if not self._commit():
                            raise RuntimeError('Commit error')
                        # make map
                        pandaIdLfnMap[disDataset] = {}
                        for lfn,filePandaID in resL:
                            if filePandaID not in pandaIdLfnMap[disDataset]:
                                pandaIdLfnMap[disDataset][filePandaID] = []
                            pandaIdLfnMap[disDataset][filePandaID].append(lfn)
                        _logger.debug("getLFNsInUseForAnal : <%s> %s map made with len=%s" % \
                                          (token,disDataset,len(resL)))
                # append
                for disDataset in disDatasetList:
                    _logger.debug("getLFNsInUseForAnal : <%s> %s list making pandaIDs=%s fileLen=%s" % \
                                      (token,disDataset,len(activePandaIDs),len(inputFilesList)))
                    for activePandaID in activePandaIDs:
                        # skip files used by archived failed or cancelled jobs
                        if activePandaID in pandaIdLfnMap[disDataset]:
                            inputFilesList += pandaIdLfnMap[disDataset][activePandaID]
                    _logger.debug("getLFNsInUseForAnal : <%s> %s done" % (token,disDataset))
            _logger.debug("getLFNsInUseForAnal : <%s> %s" % (token,len(inputFilesList)))
            return inputFilesList
        except Exception:
            # roll back
            self._rollback()
            errtype,errvalue = sys.exc_info()[:2]
            _logger.error("getLFNsInUseForAnal(%s) : %s %s" % (str(inputDisList),errtype,errvalue))
            return None


    # update input files and return corresponding PandaIDs
    def updateInFilesReturnPandaIDs(self,dataset,status,fileLFN=''):
        comment = ' /* DBProxy.updateInFilesReturnPandaIDs */'
        _logger.debug("updateInFilesReturnPandaIDs(%s,%s)" % (dataset,fileLFN))
        sql0 = "SELECT /*+ index(tab FILESTABLE4_DISPDBLOCK_IDX) */ row_ID,PandaID FROM ATLAS_PANDA.filesTable4 tab WHERE status<>:status AND dispatchDBlock=:dispatchDBlock"
        sql1 = "UPDATE /*+ index(tab FILESTABLE4_DISPDBLOCK_IDX) */ ATLAS_PANDA.filesTable4 tab SET status=:status WHERE status<>:status AND dispatchDBlock=:dispatchDBlock"
        varMap = {}
        varMap[':status'] = status
        varMap[':dispatchDBlock'] = dataset
        if fileLFN != '':
            sql0 += " AND lfn=:lfn"
            sql1 += " AND lfn=:lfn"
            varMap[':lfn'] = fileLFN
        for iTry in range(self.nTry):
            try:
                # start transaction
                self.conn.begin()
                # select
                self.cur.arraysize = 10000
                retS = self.cur.execute(sql0+comment, varMap)
                resS = self.cur.fetchall()
                # update
                retU = self.cur.execute(sql1+comment, varMap)
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                # collect PandaIDs
                retList = []
                for tmpRowID,tmpPandaID in resS:
                    # append
                    if tmpPandaID not in retList:
                        retList.append(tmpPandaID)
                # return
                _logger.debug("updateInFilesReturnPandaIDs : %s" % str(retList))
                return retList
            except Exception:
                # roll back
                self._rollback()
                # error report
                if iTry+1 < self.nTry:
                    _logger.debug("updateInFilesReturnPandaIDs retry : %s" % iTry)
                    time.sleep(random.randint(10,20))
                    continue
                type, value, traceBack = sys.exc_info()
                _logger.error("updateInFilesReturnPandaIDs : %s %s" % (type, value))
        return []


    # update file status in dispatch dataset
    def updateFileStatusInDisp(self,dataset,fileStatusMap):
        comment = ' /* DBProxy.updateFileStatusInDisp */'
        _logger.debug("updateFileStatusInDisp(%s,%s)" % (dataset,fileStatusMap))
        sql1 = "UPDATE /*+ index(tab FILESTABLE4_DISPDBLOCK_IDX) */ ATLAS_PANDA.filesTable4 tab SET status=:status WHERE dispatchDBlock=:dispatchDBlock AND lfn=:lfn"
        nTry = 1
        for iTry in range(nTry):
            try:
                # start transaction
                self.conn.begin()
                # update
                for status in fileStatusMap:
                    lfns = fileStatusMap[status]
                    varMap = {}
                    varMap[':status'] = status
                    varMap[':dispatchDBlock'] = dataset
                    # loop over all files
                    for lfn in lfns:
                        varMap['lfn'] = lfn
                        # update
                        retU = self.cur.execute(sql1+comment, varMap)
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                # return
                _logger.debug("updateFileStatusInDisp : done")
                return True
            except Exception:
                # roll back
                self._rollback()
                # error report
                if iTry+1 < nTry:
                    _logger.debug("updateFileStatusInDisp retry : %s" % iTry)
                    time.sleep(random.randint(5,10))
                    continue
                type, value, traceBack = sys.exc_info()
                _logger.error("updateFileStatusInDisp : %s %s" % (type, value))
        return False


    # update output files and return corresponding PandaIDs
    def updateOutFilesReturnPandaIDs(self,dataset,fileLFN=''):
        comment = ' /* DBProxy.updateOutFilesReturnPandaIDs */'
        _logger.debug("updateOutFilesReturnPandaIDs(%s,%s)" % (dataset,fileLFN))
        sql0 = "SELECT /*+ index(tab FILESTABLE4_DESTDBLOCK_IDX) */ row_ID,PandaID FROM ATLAS_PANDA.filesTable4 tab WHERE destinationDBlock=:destinationDBlock AND status=:status"
        sql1 = "UPDATE /*+ index(tab FILESTABLE4_DESTDBLOCK_IDX) */ ATLAS_PANDA.filesTable4 tab SET status='ready' WHERE destinationDBlock=:destinationDBlock AND status=:status"
        varMap = {}
        varMap[':status'] = 'transferring'
        varMap[':destinationDBlock'] = dataset
        if fileLFN != '':
            sql0 += " AND lfn=:lfn"
            sql1 += " AND lfn=:lfn"
            varMap[':lfn'] = fileLFN
        for iTry in range(self.nTry):
            try:
                # start transaction
                self.conn.begin()
                # select
                self.cur.arraysize = 10000
                retS = self.cur.execute(sql0+comment, varMap)
                resS = self.cur.fetchall()
                # update
                retList = []
                retU = self.cur.execute(sql1+comment, varMap)
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                # collect PandaIDs
                retList = []
                for tmpRowID,tmpPandaID in resS:
                    # append
                    if tmpPandaID not in retList:
                        retList.append(tmpPandaID)
                # return
                _logger.debug("updateOutFilesReturnPandaIDs : %s" % str(retList))
                return retList
            except Exception:
                # roll back
                self._rollback()
                # error report
                if iTry+1 < self.nTry:
                    _logger.debug("updateOutFilesReturnPandaIDs retry : %s" % iTry)
                    time.sleep(random.randint(10,20))
                    continue
                type, value, traceBack = sys.exc_info()
                _logger.error("updateOutFilesReturnPandaIDs : %s %s" % (type, value))
        return []


    # get _dis datasets associated to _sub
    def getAssociatedDisDatasets(self,subDsName):
        comment = ' /* DBProxy.getAssociatedDisDatasets */'
        _logger.debug("getAssociatedDisDatasets(%s)" % subDsName)
        sqlF = "SELECT /*+ index(tab FILESTABLE4_DESTDBLOCK_IDX) */ distinct PandaID FROM ATLAS_PANDA.filesTable4 tab WHERE destinationDBlock=:destinationDBlock"
        sqlJ = "SELECT distinct dispatchDBlock FROM ATLAS_PANDA.filesTable4 WHERE PandaID=:PandaID AND type=:type"
        try:
            # start transaction
            self.conn.begin()
            # get PandaIDs
            varMap = {}
            varMap[':destinationDBlock'] = subDsName
            self.cur.arraysize = 10000
            self.cur.execute(sqlF+comment,varMap)
            resS = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # loop over all PandaIDs
            retList = []
            for pandaID, in resS:
                # start transaction
                self.conn.begin()
                # get _dis name
                varMap = {}
                varMap[':type'] = 'input'
                varMap[':PandaID'] = pandaID
                self.cur.arraysize = 1000
                self.cur.execute(sqlJ+comment,varMap)
                resD = self.cur.fetchall()
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                # append
                for disName, in resD:
                    if disName is not None and disName not in retList:
                        retList.append(disName)
            # return
            _logger.debug("getAssociatedDisDatasets : %s" % str(retList))
            return retList
        except Exception:
            # roll back
            self._rollback()
            errType,errValue = sys.exc_info()[:2]
            _logger.error("getAssociatedDisDatasets : %s : %s %s" % (subDsName,errType,errValue))
            return []


    # set GUIDs
    def setGUIDs(self,files):
        comment = ' /* DBProxy.setGUIDs */'
        _logger.debug("setGUIDs(%s)" % files)
        sql0 = "UPDATE ATLAS_PANDA.filesTable4 SET GUID=:GUID,fsize=:fsize,checksum=:checksum,scope=:scope WHERE lfn=:lfn"
        for iTry in range(self.nTry):
            try:
                # start transaction
                self.conn.begin()
                self.cur.arraysize = 1000000
                # update
                for file in files:
                    varMap = {}
                    varMap[':GUID']     = file['guid']
                    varMap[':lfn']      = file['lfn']
                    if file['checksum'] in ['','NULL']:
                        varMap[':checksum'] = None
                    else:
                        varMap[':checksum'] = file['checksum']
                    varMap[':fsize']    = file['fsize']
                    if 'scope' not in file or file['scope'] in ['','NULL']:
                        varMap[':scope'] = None
                    else:
                        varMap[':scope'] = file['scope']
                    self.cur.execute(sql0+comment, varMap)
                    retU = self.cur.rowcount
                    _logger.debug("setGUIDs : retU %s" % retU)
                    if retU<0:
                        raise RuntimeError('SQL error')
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                return True
            except Exception:
                # roll back
                self._rollback()
                    # error report
                if iTry+1 < self.nTry:
                    _logger.debug("setGUIDs retry : %s" % iTry)
                    time.sleep(random.randint(10,20))
                    continue
                type, value, traceBack = sys.exc_info()
                _logger.error("setGUIDs : %s %s" % (type, value))
        return False


    # query PandaID with Datasets
    def queryPandaIDwithDataset(self,datasets):
        comment = ' /* DBProxy.queryPandaIDwithDataset */'
        _logger.debug("queryPandaIDwithDataset(%s)" % datasets)
        if len(datasets) == 0:
            return []
        # make SQL query
        sql1 = "SELECT /*+ index(tab FILESTABLE4_DESTDBLOCK_IDX) */ PandaID FROM ATLAS_PANDA.filesTable4 tab WHERE destinationDBlock=:destinationDBlock GROUP BY PandaID"
        # execute
        try:
            retList = []
            for dataset in datasets:
                # start transaction
                self.conn.begin()
                # select
                self.cur.arraysize = 10000
                varMap = {}
                varMap[':destinationDBlock'] = dataset
                self.cur.execute(sql1+comment,varMap)
                res = self.cur.fetchall()
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                # get IDs
                for r in res:
                    retList.append(r[0])
            # return
            _logger.debug("queryPandaIDwithDataset : %s" % str(retList))
            return retList
        except Exception:
            # roll back
            self._rollback()
            # error report
            type, value, traceBack = sys.exc_info()
            _logger.error("queryPandaIDwithDataset : %s %s" % (type, value))
            return []


    # query last files in datasets
    def queryLastFilesInDataset(self,datasets):
        comment = ' /* DBProxy.queryLastFilesInDataset */'
        _logger.debug("queryLastFilesInDataset(%s)" % datasets)
        if len(datasets) == 0:
            return []
        # make SQL query
        sql1 = "SELECT lfn,PandaID FROM ATLAS_PANDA.filesTable4 WHERE dataset=:dataset AND type=:type ORDER BY lfn DESC"
        sqlL = "SELECT processingType FROM %s WHERE PandaID=:PandaID "
        sqlA = "UNION SELECT processingType FROM ATLAS_PANDAARCH.jobsArchived WHERE PandaID=:PandaID AND modificationTime>(CURRENT_DATE-30)"
        sql2 = "SELECT lfn FROM ATLAS_PANDA.filesTable4 WHERE PandaID=:PandaID AND type=:type"
        # execute
        try:
            retMap = {}
            for dataset in datasets:
                # start transaction
                self.conn.begin()
                # select max LFN
                varMap = {}
                varMap[':type'] = 'output'
                varMap[':dataset'] = dataset
                self.cur.arraysize = 100000
                self.cur.execute(sql1+comment, varMap)
                res = self.cur.fetchall()
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                # found
                retList = []
                for tmpLFN,pandaID in res:
                    # skip log.tgz
                    if re.search('\.log\.tgz(\.\d+)*$',tmpLFN) is not None:
                        continue
                    # start transaction
                    self.conn.begin()
                    self.cur.arraysize = 10
                    # check processingType
                    processingType = None
                    for tmpTable in ['ATLAS_PANDA.jobsDefined4','ATLAS_PANDA.jobsActive4','ATLAS_PANDA.jobsArchived4']:
                        varMap = {}
                        varMap[':PandaID'] = pandaID
                        if tmpTable == 'ATLAS_PANDA.jobsArchived4':
                            self.cur.execute((sqlL % tmpTable)+sqlA+comment, varMap)
                        else:
                            self.cur.execute((sqlL % tmpTable)+comment, varMap)
                        resP = self.cur.fetchone()
                        if resP is not None:
                            processingType = resP[0]
                            break
                    # commit
                    if not self._commit():
                        raise RuntimeError('Commit error')
                    # job not found
                    if processingType is None:
                        continue
                    # ignore merge jobs
                    if processingType in ['usermerge']:
                        continue
                    # start transaction
                    self.conn.begin()
                    # select LFNs
                    varMap = {}
                    varMap[':PandaID'] = pandaID
                    varMap[':type'] = 'output'
                    self.cur.arraysize = 1000
                    self.cur.execute(sql2+comment, varMap)
                    res = self.cur.fetchall()
                    for r in res:
                        retList.append(r[0])
                    # commit
                    if not self._commit():
                        raise RuntimeError('Commit error')
                    # get only the largest one
                    break
                # append
                retMap[dataset] = retList
            # return
            _logger.debug("queryLastFilesInDataset : %s" % str(retMap))
            return retMap
        except Exception:
            # roll back
            self._rollback()
            type, value, traceBack = sys.exc_info()
            _logger.error("queryLastFilesInDataset : %s %s" % (type, value))
            return {}


    # query PandaID with filenames
    def queryPandaIDwithLFN(self,vlfns):
        comment = ' /* DBProxy.queryPandaIDwithLFN */'
        _logger.debug("queryPandaIDwithLFN(%s)" % vlfns)
        if len(vlfns) == 0:
            return []
        # make SQL query
        sql1 = "SELECT PandaID FROM ATLAS_PANDA.filesTable4 WHERE lfn=:lfn GROUP BY PandaID"
        # execute
        retList = []
        for lfn in vlfns:
            # get generic LFNs
            gLFN = re.sub('\.\d+$','',lfn)
            # try
            try:
                # start transaction
                self.conn.begin()
                # select
                varMap = {}
                varMap[':lfn'] = gLFN
                self.cur.arraysize = 10000
                self.cur.execute(sql1+comment, varMap)
                res = self.cur.fetchall()
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                # append IDs
                for tmpID, in res:
                    if tmpID not in retList:
                        retList.append(tmpID)
            except Exception:
                # roll back
                self._rollback()
                # error report
                type, value, traceBack = sys.exc_info()
                _logger.error("queryPandaIDwithLFN : %s %s" % (type, value))
                return []
        # return
        _logger.debug("queryPandaIDwithLFN : %s" % str(retList))
        return retList


    # get job statistics
    def getJobStatistics(self,archived=False,predefined=False,workingGroup='',countryGroup='',jobType='',forAnal=None,minPriority=None):
        comment = ' /* DBProxy.getJobStatistics */'
        _logger.debug("getJobStatistics(%s,%s,'%s','%s','%s',%s,%s)" % (archived,predefined,workingGroup,countryGroup,jobType,forAnal,minPriority))
        timeLimit = datetime.datetime.utcnow() - datetime.timedelta(hours=12)
        sql0  = "SELECT computingSite,jobStatus,COUNT(*) FROM %s "
        # processingType
        tmpJobTypeMap = {}
        sqlJobType = ''
        useWhereInSQL = True
        if forAnal is None or jobType != "":
            useWhereInSQL = False
        elif forAnal is True:
            tmpJobTypeMap[':prodSourceLabel1'] = 'user'
            tmpJobTypeMap[':prodSourceLabel2'] = 'panda'
            sql0 += "WHERE prodSourceLabel IN ("
            sqlJobType = ":prodSourceLabel1,:prodSourceLabel2) "
        else:
            tmpJobTypeMap[':prodSourceLabel1'] = 'managed'
            sql0 += "WHERE prodSourceLabel IN ("
            sqlJobType = ":prodSourceLabel1) "
        sql0 += sqlJobType
        # predefined
        if predefined:
            if useWhereInSQL:
                sql0 += "AND relocationFlag=1 "
            else:
                sql0 += "WHERE relocationFlag=1 "
                useWhereInSQL = True
        # working group
        tmpGroupMap = {}
        sqlGroups = ''
        if workingGroup != '':
            if useWhereInSQL:
                sqlGroups += "AND workingGroup IN ("
            else:
                sqlGroups += "WHERE workingGroup IN ("
                useWhereInSQL = True
            # loop over all groups
            idxWG = 1
            for tmpWG in workingGroup.split(','):
                tmpWGkey = ':workingGroup%s' % idxWG
                sqlGroups += "%s," % tmpWGkey
                tmpGroupMap[tmpWGkey] = tmpWG
                idxWG += 1
            sqlGroups = sqlGroups[:-1] + ") "

        sql0 += sqlGroups
        # minimum priority
        sqlPrio = ''
        tmpPrioMap = {}
        if minPriority is not None:
            if useWhereInSQL:
                sqlPrio = "AND currentPriority>=:minPriority "
            else:
                sqlPrio = "WHERE currentPriority>=:minPriority "
                useWhereInSQL = True
            tmpPrioMap[':minPriority'] = minPriority
        sql0 += sqlPrio
        sql0 += "GROUP BY computingSite,jobStatus"
        sqlA =  "SELECT /*+ INDEX_RS_ASC(tab (MODIFICATIONTIME PRODSOURCELABEL)) */ computingSite,jobStatus,COUNT(*) FROM ATLAS_PANDA.jobsArchived4 tab WHERE modificationTime>:modificationTime "
        if sqlJobType != "":
            sqlA += "AND prodSourceLabel IN ("
            sqlA += sqlJobType
        if predefined:
            sqlA += "AND relocationFlag=1 "
        sqlA += sqlGroups
        sqlA += sqlPrio
        sqlA += "GROUP BY computingSite,jobStatus"
        tables = ['ATLAS_PANDA.jobsActive4','ATLAS_PANDA.jobsDefined4']
        if archived:
            tables.append('ATLAS_PANDA.jobsArchived4')
        # sql for materialized view
        sqlMV = re.sub('COUNT\(\*\)','SUM(num_of_jobs)',sql0)
        sqlMV = re.sub(':minPriority','TRUNC(:minPriority,-1)',sqlMV)
        sqlMV = re.sub('SELECT ','SELECT /*+ RESULT_CACHE */ ',sqlMV)
        ret = {}
        nTry=3
        for iTry in range(nTry):
            try:
                for table in tables:
                    # start transaction
                    self.conn.begin()
                    # select
                    varMap = {}
                    for tmpJobType in tmpJobTypeMap:
                        varMap[tmpJobType] = tmpJobTypeMap[tmpJobType]
                    for tmpGroup in tmpGroupMap:
                        varMap[tmpGroup] = tmpGroupMap[tmpGroup]
                    for tmpPrio in tmpPrioMap:
                        varMap[tmpPrio] = tmpPrioMap[tmpPrio]
                    if table != 'ATLAS_PANDA.jobsArchived4':
                        self.cur.arraysize = 10000
                        if table == 'ATLAS_PANDA.jobsActive4':
                            sqlExeTmp = (sqlMV+comment) % 'ATLAS_PANDA.MV_JOBSACTIVE4_STATS'
                        else:
                            sqlExeTmp = (sql0+comment) % table
                        _logger.debug("getJobStatistics : %s %s" % (sqlExeTmp,str(varMap)))
                        self.cur.execute(sqlExeTmp, varMap)
                    else:
                        varMap[':modificationTime'] = timeLimit
                        self.cur.arraysize = 10000
                        self.cur.execute(sqlA+comment, varMap)
                    res = self.cur.fetchall()
                    # commit
                    if not self._commit():
                        raise RuntimeError('Commit error')
                    # create map
                    for computingSite,jobStatus,nJobs in res:
                        # FIXME
                        # ignore some job status since they break APF
                        if jobStatus in ['merging']:
                            continue
                        if computingSite not in ret:
                            ret[computingSite] = {}
                        if jobStatus not in ret[computingSite]:
                            ret[computingSite][jobStatus] = 0
                        ret[computingSite][jobStatus] += nJobs
                # for zero
                stateList = ['assigned','activated','running']
                if archived:
                    stateList += ['finished','failed']
                for site in ret:
                    for state in stateList:
                        if state not in ret[site]:
                            ret[site][state] = 0
                # return
                _logger.debug("getJobStatistics -> %s" % str(ret))
                return ret
            except Exception:
                # roll back
                self._rollback()
                if iTry+1 < nTry:
                    _logger.debug("getJobStatistics() retry : %s" % iTry)
                    time.sleep(2)
                    continue
                type, value, traceBack = sys.exc_info()
                _logger.error("getJobStatistics : %s %s" % (type, value))
                return {}


    # get job statistics with label
    def getJobStatisticsWithLabel(self,siteStr=''):
        comment = ' /* DBProxy.getJobStatisticsWithLabel */'
        _logger.debug("getJobStatisticsWithLabel(%s)" % siteStr)
        sql0 = "SELECT computingSite,prodSourceLabel,jobStatus,COUNT(*) FROM %s "
        # site
        tmpSiteMap = {}
        if siteStr != '':
            sql0 += "WHERE computingSite IN ("
            # loop over all sites
            idxSite = 1
            for tmpSite in siteStr.split(','):
                tmpSiteKey = ':site%s' % idxSite
                sql0 += "%s," % tmpSiteKey
                tmpSiteMap[tmpSiteKey] = tmpSite
                idxSite += 1
            sql0 = sql0[:-1] + ") "
        sql0 += "GROUP BY computingSite,prodSourceLabel,jobStatus "
        sqlMV = re.sub('COUNT\(\*\)','SUM(num_of_jobs)',sql0)
        sqlMV = re.sub('SELECT ','SELECT /*+ RESULT_CACHE */ ',sqlMV)
        tables = ['ATLAS_PANDA.jobsActive4','ATLAS_PANDA.jobsDefined4']
        returnMap = {}
        try:
            for table in tables:
                # start transaction
                self.conn.begin()
                # select
                varMap = {}
                self.cur.arraysize = 10000
                if table == 'ATLAS_PANDA.jobsActive4':
                    sqlExeTmp = (sqlMV+comment) % 'ATLAS_PANDA.MV_JOBSACTIVE4_STATS'
                else:
                    sqlExeTmp = (sql0+comment) % table
                self.cur.execute(sqlExeTmp,tmpSiteMap)
                res = self.cur.fetchall()
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                # create map
                for computingSite,prodSourceLabel,jobStatus,nCount in res:
                    # FIXME
                    # ignore some job status since they break APF
                    if jobStatus in ['merging']:
                        continue
                    # add site
                    if computingSite not in returnMap:
                        returnMap[computingSite] = {}
                    # add SourceLabel
                    if prodSourceLabel not in returnMap[computingSite]:
                        returnMap[computingSite][prodSourceLabel] = {}
                    # add jobstatus
                    if jobStatus not in returnMap[computingSite][prodSourceLabel]:
                        returnMap[computingSite][prodSourceLabel][jobStatus] = 0
                    # add
                    returnMap[computingSite][prodSourceLabel][jobStatus] += nCount
            # return
            _logger.debug("getJobStatisticsWithLabel() : %s" % str(returnMap))
            return returnMap
        except Exception:
            # roll back
            self._rollback()
            errType,errValue = sys.exc_info()[:2]
            _logger.error("getJobStatisticsWithLabel : %s %s" % (errType,errValue))
            return {}


    # get job statistics for brokerage
    def getJobStatisticsBrokerage(self,minPriority=None,maxPriority=None):
        comment = ' /* DBProxy.getJobStatisticsBrokerage */'
        _logger.debug("getJobStatisticsBrokerage(min=%s max=%s)" % (minPriority,maxPriority))
        sql0 = "SELECT cloud,computingSite,jobStatus,processingType,COUNT(*) FROM %s WHERE "
        sql0 += "prodSourceLabel IN (:prodSourceLabel1) "
        tmpPrioMap = {}
        if minPriority is not None:
            sql0 += "AND currentPriority>=:minPriority "
            tmpPrioMap[':minPriority'] = minPriority
        sql0 += "GROUP BY cloud,computingSite,jobStatus,processingType"
        # sql for materialized view
        sqlMV = re.sub('COUNT\(\*\)','SUM(num_of_jobs)',sql0)
        sqlMV = re.sub(':minPriority','TRUNC(:minPriority,-1)',sqlMV)
        sqlMV = re.sub('SELECT ','SELECT /*+ RESULT_CACHE */ ',sqlMV)
        tables = ['ATLAS_PANDA.jobsActive4','ATLAS_PANDA.jobsDefined4']
        if minPriority is not None or maxPriority is not None:
            # read the number of running jobs with prio<=MIN
            tables.append('ATLAS_PANDA.jobsActive4')
            sqlMVforRun = re.sub('currentPriority>=','currentPriority<=',sqlMV)
        ret = {}
        nTry=3
        iActive = 0
        for iTry in range(nTry):
            try:
                for table in tables:
                    # start transaction
                    self.conn.begin()
                    # select
                    varMap = {}
                    varMap[':prodSourceLabel1'] = 'managed'
                    for tmpPrio in tmpPrioMap:
                        varMap[tmpPrio] = tmpPrioMap[tmpPrio]
                    self.cur.arraysize = 10000
                    useRunning = None
                    if table == 'ATLAS_PANDA.jobsActive4':
                        # first count non-running and then running if minPriority is specified
                        if minPriority is not None:
                            if iActive == 0:
                                useRunning = False
                            else:
                                useRunning = True
                            iActive += 1
                        if useRunning in [None,False]:
                            self.cur.execute((sqlMV+comment) % 'ATLAS_PANDA.MV_JOBSACTIVE4_STATS', varMap)
                        else:
                            # use maxPriority to avoid underestimation of running jobs
                            if minPriority is not None and maxPriority is not None:
                                varMap[':minPriority'] = maxPriority
                            self.cur.execute((sqlMVforRun+comment) % 'ATLAS_PANDA.MV_JOBSACTIVE4_STATS', varMap)
                    else:
                        self.cur.execute((sql0+comment) % table, varMap)
                    res = self.cur.fetchall()
                    # commit
                    if not self._commit():
                        raise RuntimeError('Commit error')
                    # create map
                    for cloud,computingSite,jobStatus,processingType,count in res:
                        # check jobstatus if minPriority isspecified
                        if minPriority is not None:
                            # count the number of non-running with prio>=MIN
                            if useRunning is True and jobStatus != 'running':
                                continue
                            # count the number of running with prio<=MIN
                            if  useRunning is False and jobStatus == 'running':
                                continue
                        # add cloud
                        ret.setdefault(cloud, {})
                        # add site
                        ret[cloud].setdefault(computingSite, {})
                        # add processingType
                        ret[cloud][computingSite].setdefault(processingType, {})
                        # add jobStatus
                        if jobStatus not in ret[cloud][computingSite][processingType]:
                            ret[cloud][computingSite][processingType][jobStatus] = count
                # for zero
                for cloud in ret:
                    cloudVal = ret[cloud]
                    for site in cloudVal:
                        siteVal = cloudVal[site]
                        for pType in siteVal:
                            typeVal = siteVal[pType]
                            for stateItem in ['assigned','activated','running','transferring']:
                                if stateItem not in typeVal:
                                    typeVal[stateItem] = 0
                # return
                _logger.debug("getJobStatisticsBrokerage -> %s" % str(ret))
                return ret
            except Exception:
                # roll back
                self._rollback()
                if iTry+1 < nTry:
                    _logger.debug("getJobStatisticsBrokerage retry : %s" % iTry)
                    time.sleep(2)
                    continue
                type, value, traceBack = sys.exc_info()
                _logger.error("getJobStatisticsBrokerage : %s %s" % (type, value))
                return {}


    # get job statistics for analysis brokerage
    def getJobStatisticsAnalBrokerage(self,minPriority=None):
        comment = ' /* DBProxy.getJobStatisticsAnalBrokerage */'
        _logger.debug("getJobStatisticsAnalBrokerage(%s)" % minPriority)
        sql0 = "SELECT computingSite,jobStatus,processingType,COUNT(*) FROM %s WHERE "
        sql0 += "prodSourceLabel IN (:prodSourceLabel1,:prodSourceLabel2) "
        if minPriority is not None:
            sql0 += "AND currentPriority>=:minPriority "
        sql0 += "GROUP BY cloud,computingSite,jobStatus,processingType"
        # sql for materialized view
        sqlMV = re.sub('COUNT\(\*\)','SUM(num_of_jobs)',sql0)
        sqlMV = re.sub(':minPriority','TRUNC(:minPriority,-1)',sqlMV)
        sqlMV = re.sub('SELECT ','SELECT /*+ RESULT_CACHE */ ',sqlMV)
        tables = ['ATLAS_PANDA.jobsActive4','ATLAS_PANDA.jobsDefined4']
        ret = {}
        nTry=3
        for iTry in range(nTry):
            try:
                for table in tables:
                    # start transaction
                    self.conn.begin()
                    # select
                    varMap = {}
                    varMap[':prodSourceLabel1'] = 'user'
                    varMap[':prodSourceLabel2'] = 'panda'
                    if minPriority is not None:
                        varMap[':minPriority'] = minPriority
                    self.cur.arraysize = 10000
                    if table == 'ATLAS_PANDA.jobsActive4':
                        self.cur.execute((sqlMV+comment) % 'ATLAS_PANDA.MV_JOBSACTIVE4_STATS', varMap)
                    else:
                        self.cur.execute((sql0+comment) % table, varMap)
                    res = self.cur.fetchall()
                    # commit
                    if not self._commit():
                        raise RuntimeError('Commit error')
                    # create map
                    for computingSite,jobStatus,processingType,count in res:
                        # add site
                        if computingSite not in ret:
                            ret[computingSite] = {}
                        # add processingType
                        if processingType not in ret[computingSite]:
                            ret[computingSite][processingType] = {}
                        # add jobStatus
                        if jobStatus not in ret[computingSite][processingType]:
                            ret[computingSite][processingType][jobStatus] = count
                # for zero
                for site in ret:
                    siteVal = ret[site]
                    for pType in siteVal:
                        typeVal = siteVal[pType]
                        for stateItem in ['defined','assigned','activated','running']:
                            if stateItem not in typeVal:
                                typeVal[stateItem] = 0
                # return
                _logger.debug("getJobStatisticsAnalBrokerage -> %s" % str(ret))
                return ret
            except Exception:
                # roll back
                self._rollback()
                if iTry+1 < nTry:
                    _logger.debug("getJobStatisticsAnalBrokerage retry : %s" % iTry)
                    time.sleep(2)
                    continue
                type, value, traceBack = sys.exc_info()
                _logger.error("getJobStatisticsAnalBrokerage : %s %s" % (type, value))
                return {}


    # get highest prio jobs
    def getHighestPrioJobStat(self):
        comment = ' /* DBProxy.getHighestPrioJobStat */'
        _logger.debug("getHighestPrioJobStat()")
        sql0  = "SELECT cloud,max(currentPriority) FROM %s WHERE "
        sql0 += "prodSourceLabel=:prodSourceLabel AND jobStatus IN (:jobStatus1,:jobStatus2) GROUP BY cloud"
        sqlC  = "SELECT COUNT(*) FROM %s WHERE "
        sqlC += "prodSourceLabel=:prodSourceLabel AND jobStatus IN (:jobStatus1,:jobStatus2) AND "
        sqlC += "cloud=:cloud AND currentPriority=:currentPriority"
        tables = ['ATLAS_PANDA.jobsActive4','ATLAS_PANDA.jobsDefined4']
        ret = {}
        try:
            for table in tables:
                # start transaction
                self.conn.begin()
                # select
                varMap = {}
                varMap[':prodSourceLabel'] = 'managed'
                if table == 'ATLAS_PANDA.jobsActive4':
                    varMap[':jobStatus1'] = 'activated'
                    varMap[':jobStatus2'] = 'dummy'
                else:
                    varMap[':jobStatus1'] = 'defined'
                    varMap[':jobStatus2'] = 'assigned'
                self.cur.arraysize = 100
                _logger.debug((sql0+comment) % table)
                self.cur.execute((sql0+comment) % table, varMap)
                res = self.cur.fetchall()
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                # create map
                for cloud,maxPriority in res:
                    # add cloud
                    if cloud not in ret:
                        ret[cloud] = {}
                    # add max priority
                    prioKey = 'highestPrio'
                    nNotRunKey = 'nNotRun'
                    getNumber = False
                    if prioKey not in ret[cloud]:
                        ret[cloud][prioKey] = maxPriority
                        ret[cloud][nNotRunKey] = 0
                        getNumber = True
                    else:
                        # use highest one
                        if ret[cloud][prioKey] < maxPriority:
                            ret[cloud][prioKey] = maxPriority
                            # reset
                            ret[cloud][nNotRunKey] = 0
                            getNumber = True
                        elif ret[cloud][prioKey] == maxPriority:
                            getNumber = True
                    # get number of jobs with highest prio
                    if getNumber:
                        varMap[':cloud'] = cloud
                        varMap[':currentPriority'] = maxPriority
                        self.cur.arraysize = 10
                        _logger.debug((sqlC+comment) % table)
                        self.cur.execute((sqlC+comment) % table, varMap)
                        resC = self.cur.fetchone()
                        # commit
                        if not self._commit():
                            raise RuntimeError('Commit error')
                        ret[cloud][nNotRunKey] += resC[0]
            # return
            return ret
        except Exception:
            # roll back
            self._rollback()
            type, value, traceBack = sys.exc_info()
            _logger.error("getHighestPrioJobStat : %s %s" % (type, value))
            return {}


    # get highest prio jobs per process group
    def getHighestPrioJobStatPerPG(self,useMorePG=False):
        comment = ' /* DBProxy.getHighestPrioJobStatPerPG */'
        _logger.debug("getHighestPrioJobStatPerPG()")
        if useMorePG is False:
            sql0  = "SELECT cloud,max(currentPriority),processingType FROM %s WHERE "
            sql0 += "prodSourceLabel=:prodSourceLabel AND jobStatus IN (:jobStatus1,:jobStatus2) GROUP BY cloud,processingType"
            sqlC  = "SELECT COUNT(*) FROM %s WHERE "
            sqlC += "prodSourceLabel=:prodSourceLabel AND jobStatus IN (:jobStatus1,:jobStatus2) AND "
            sqlC += "cloud=:cloud AND currentPriority=:currentPriority AND processingType=:processingType"
        else:
            sql0  = "SELECT cloud,max(currentPriority),processingType,coreCount,workingGroup FROM %s WHERE "
            sql0 += "prodSourceLabel=:prodSourceLabel AND jobStatus IN (:jobStatus1,:jobStatus2) "
            sql0 += "GROUP BY cloud,processingType,coreCount,workingGroup"
            sqlC  = "SELECT COUNT(*) FROM %s WHERE "
            sqlC += "prodSourceLabel=:prodSourceLabel AND jobStatus IN (:jobStatus1,:jobStatus2) AND "
            sqlC += "cloud=:cloud AND currentPriority=:currentPriority AND processingType=:processingType AND "
            sqlC += "coreCount=:coreCount AND workingGroup=:workingGroup"
            sqlCN  = "SELECT COUNT(*) FROM %s WHERE "
            sqlCN += "prodSourceLabel=:prodSourceLabel AND jobStatus IN (:jobStatus1,:jobStatus2) AND "
            sqlCN += "cloud=:cloud AND currentPriority=:currentPriority AND processingType=:processingType AND "
            sqlCN += "coreCount IS NULL AND workingGroup=:workingGroup"
        tables = ['ATLAS_PANDA.jobsActive4','ATLAS_PANDA.jobsDefined4']
        ret = {}
        try:
            for table in tables:
                # start transaction
                self.conn.begin()
                # select
                varMap = {}
                varMap[':prodSourceLabel'] = 'managed'
                if table == 'ATLAS_PANDA.jobsActive4':
                    varMap[':jobStatus1'] = 'activated'
                    varMap[':jobStatus2'] = 'dummy'
                else:
                    varMap[':jobStatus1'] = 'defined'
                    varMap[':jobStatus2'] = 'assigned'
                self.cur.arraysize = 100
                _logger.debug((sql0+comment) % table+str(varMap))
                self.cur.execute((sql0+comment) % table, varMap)
                res = self.cur.fetchall()
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                # create map
                for tmpItem in res:
                    if useMorePG is False:
                        cloud,maxPriority,processingType = tmpItem
                        origCloud = cloud
                        origProcessingType = processingType
                    else:
                        origCloud,maxPriority,origProcessingType,coreCount,workingGroup = tmpItem
                        # convert cloud and processingType for extended process group
                        if useMorePG == ProcessGroups.extensionLevel_1:
                            # extension level 1
                            cloud,processingType = ProcessGroups.converCPTforEPG(origCloud,origProcessingType,
                                                                                 coreCount)
                        else:
                            # extension level 2
                            cloud,processingType = ProcessGroups.converCPTforEPG(origCloud,origProcessingType,
                                                                                 coreCount,workingGroup)
                    # add cloud
                    if cloud not in ret:
                        ret[cloud] = {}
                    # get process group
                    processGroup = ProcessGroups.getProcessGroup(processingType)
                    # add process group
                    if processGroup not in ret[cloud]:
                        ret[cloud][processGroup] = {}
                    # add max priority
                    prioKey = 'highestPrio'
                    nNotRunKey = 'nNotRun'
                    getNumber = False
                    if prioKey not in ret[cloud][processGroup]:
                        ret[cloud][processGroup][prioKey] = maxPriority
                        ret[cloud][processGroup][nNotRunKey] = 0
                        getNumber = True
                    else:
                        # use highest one
                        if ret[cloud][processGroup][prioKey] < maxPriority:
                            ret[cloud][processGroup][prioKey] = maxPriority
                            # reset
                            ret[cloud][processGroup][nNotRunKey] = 0
                            getNumber = True
                        elif ret[cloud][processGroup][prioKey] == maxPriority:
                            getNumber = True
                    # get number of jobs with highest prio
                    if getNumber:
                        varMap[':cloud'] = origCloud
                        varMap[':currentPriority'] = maxPriority
                        varMap[':processingType'] = origProcessingType
                        if useMorePG is not False:
                            varMap[':workingGroup'] = workingGroup
                            if coreCount is not None:
                                varMap[':coreCount'] = coreCount
                        self.cur.arraysize = 10
                        _logger.debug((sqlC+comment) % table+str(varMap))
                        self.cur.execute((sqlC+comment) % table, varMap)
                        resC = self.cur.fetchone()
                        # commit
                        if not self._commit():
                            raise RuntimeError('Commit error')
                        ret[cloud][processGroup][nNotRunKey] += resC[0]
            # return
            _logger.debug("getHighestPrioJobStatPerPG -> %s" % ret)
            return ret
        except Exception:
            # roll back
            self._rollback()
            type, value, traceBack = sys.exc_info()
            _logger.error("getHighestPrioJobStatPerPG : %s %s" % (type, value))
            return {}


    # get queued analysis jobs at a site
    def getQueuedAnalJobs(self,site,dn):
        comment = ' /* DBProxy.getQueuedAnalJobs */'
        _logger.debug("getQueuedAnalJobs(%s,%s)" % (site,dn))
        sql0 = "SELECT COUNT(*),jobStatus FROM %s WHERE "
        sql0 += "prodSourceLabel=:prodSourceLabel AND jobStatus IN (:jobStatus1,:jobStatus2) "
        sql0 += "AND computingSite=:computingSite AND prodUserName != :prodUserName "
        sql0 += "GROUP BY jobStatus "
        tables = ['ATLAS_PANDA.jobsActive4','ATLAS_PANDA.jobsDefined4']
        try:
            # get compact DN
            compactDN = self.cleanUserID(dn)
            if compactDN in ['','NULL',None]:
                compactDN = dn
            nQueued  = 0
            nRunning = 0
            # loop over all tables
            for table in tables:
                # start transaction
                self.conn.begin()
                # select
                varMap = {}
                varMap[':prodSourceLabel'] = 'user'
                varMap[':computingSite']   = site
                varMap[':prodUserName']    = compactDN
                if table == 'ATLAS_PANDA.jobsActive4':
                    varMap[':jobStatus1'] = 'activated'
                    varMap[':jobStatus2'] = 'running'
                else:
                    varMap[':jobStatus1'] = 'defined'
                    varMap[':jobStatus2'] = 'assigned'
                self.cur.arraysize = 10
                self.cur.execute((sql0+comment) % table, varMap)
                res = self.cur.fetchall()
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                # sum
                for cnt,jobStatus in res:
                    if jobStatus == 'running':
                        nRunning += cnt
                    else:
                        nQueued += cnt
            # return
            return {'queued':nQueued, 'running':nRunning}
        except Exception:
            # roll back
            self._rollback()
            errType,errValue = sys.exc_info()[:2]
            _logger.error("getQueuedAnalJobs : %s %s" % (errType,errValue))
            return {}


    # get computingSite and destinationSE for a dataset
    def getDestSE(self,dsname,fromArch=False):
        comment = ' /* DBProxy.getDestSE */'
        _logger.debug("getDestSE(%s,%s)" % (dsname,fromArch))
        sql0 = "SELECT /*+ index(tab FILESTABLE4_DESTDBLOCK_IDX) */ PandaID FROM ATLAS_PANDA.filesTable4 tab WHERE destinationDBlock=:destinationDBlock "
        if not fromArch:
            sql0 += "AND status=:status "
        sql0 += "AND rownum=1"
        sql1 = "SELECT computingSite,destinationSE FROM %s WHERE PandaID=:PandaID"
        actTableList = ['ATLAS_PANDA.jobsActive4']
        if fromArch:
            actTableList.append("ATLAS_PANDA.jobsArchived4")
        try:
            # start transaction
            self.conn.begin()
            # select
            varMap = {}
            if not fromArch:
                varMap[':status'] = 'transferring'
            varMap[':destinationDBlock'] = dsname
            self.cur.arraysize = 10
            self.cur.execute(sql0+comment, varMap)
            res = self.cur.fetchall()
            # get PandaID
            pandaID = None
            if len(res) != 0:
                pandaID = res[0][0]
            # get computingSite and destinationSE
            destSE = None,None
            if pandaID is not None:
                varMap = {}
                varMap[':PandaID'] = pandaID
                # loop over all active tables
                foundInActive = False
                for actTable in actTableList:
                    self.cur.execute((sql1 % actTable)+comment, varMap)
                    res = self.cur.fetchall()
                    if len(res) != 0:
                        destSE = res[0]
                        foundInActive = True
                        break
                # look into ARCH table
                if not foundInActive:
                    if fromArch:
                        sqlA  = "SELECT computingSite,destinationSE FROM ATLAS_PANDAARCH.jobsArchived WHERE PandaID=:PandaID "
                        sqlA += "AND modificationTime>(CURRENT_DATE-30) "
                        self.cur.execute(sqlA+comment, varMap)
                        res = self.cur.fetchall()
                        if len(res) != 0:
                            destSE = res[0]
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # return
            _logger.debug("getDestSE(%s) : %s" % (dsname,str(destSE)))
            return destSE
        except Exception:
            # roll back
            self._rollback()
            type, value, traceBack = sys.exc_info()
            _logger.error("getDestSE : %s %s" % (type, value))
            return None,None


    # get destinationDBlockToken for a dataset
    def getDestTokens(self,dsname):
        comment = ' /* DBProxy.getDestTokens */'
        _logger.debug("getDestTokens(%s)" % dsname)
        sql0 = "SELECT /*+ index(tab FILESTABLE4_DESTDBLOCK_IDX) */ destinationDBlockToken FROM ATLAS_PANDA.filesTable4 tab WHERE destinationDBlock=:destinationDBlock AND rownum=1"
        try:
            # start transaction
            self.conn.begin()
            # select
            varMap = {}
            varMap[':destinationDBlock'] = dsname
            self.cur.arraysize = 10
            self.cur.execute(sql0+comment, varMap)
            res = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # create map
            retToken = None
            if len(res) != 0:
                retToken = res[0][0]
                # convert None to NULL
                if retToken is None:
                    retToken = 'NULL'
            # return
            _logger.debug("getDestTokens(%s) : %s" % (dsname,retToken))
            return retToken
        except Exception:
            # roll back
            self._rollback()
            type, value, traceBack = sys.exc_info()
            _logger.error("getDestTokens : %s %s" % (type, value))
            return None


    # get the number of job for a user
    def getNumberJobsUser(self,dn,workingGroup=None):
        comment = ' /* DBProxy.getNumberJobsUser */'
        _logger.debug("getNumberJobsUsers(%s,%s)" % (dn,workingGroup))
        # get compact DN
        compactDN = self.cleanUserID(dn)
        if compactDN in ['','NULL',None]:
            compactDN = dn
        if workingGroup is not None:
            sql0 = "SELECT COUNT(*) FROM %s WHERE prodUserName=:prodUserName AND prodSourceLabel=:prodSourceLabel AND workingGroup=:workingGroup "
        else:
            sql0 = "SELECT COUNT(*) FROM %s WHERE prodUserName=:prodUserName AND prodSourceLabel=:prodSourceLabel AND workingGroup IS NULL "
        sql0 += "AND NOT jobStatus IN (:failed,:merging) "
        nTry = 1
        nJob = 0
        for iTry in range(nTry):
            try:
                for table in ('ATLAS_PANDA.jobsActive4','ATLAS_PANDA.jobsDefined4'):
                    # start transaction
                    self.conn.begin()
                    # select
                    varMap = {}
                    varMap[':prodUserName'] = compactDN
                    varMap[':prodSourceLabel'] = 'user'
                    varMap[':failed'] = 'failed'
                    varMap[':merging'] = 'merging'
                    if workingGroup is not None:
                        varMap[':workingGroup'] = workingGroup
                    self.cur.arraysize = 10
                    self.cur.execute((sql0+comment) % table, varMap)
                    res = self.cur.fetchall()
                    # commit
                    if not self._commit():
                        raise RuntimeError('Commit error')
                    # create map
                    if len(res) != 0:
                        nJob += res[0][0]
                # return
                _logger.debug("getNumberJobsUsers(%s) : %s" % (dn,nJob))
                return nJob
            except Exception:
                # roll back
                self._rollback()
                if iTry+1 < nTry:
                    time.sleep(2)
                    continue
                type, value, traceBack = sys.exc_info()
                _logger.error("getNumberJobsUsers : %s %s" % (type, value))
                return 0


    # get job statistics for ExtIF
    def getJobStatisticsForExtIF(self,sourcetype=None):
        comment = ' /* DBProxy.getJobStatisticsForExtIF */'
        _logger.debug("getJobStatisticsForExtIF()")
        timeLimit = datetime.datetime.utcnow() - datetime.timedelta(hours=12)
        if sourcetype == 'analysis':
            sql0 = "SELECT jobStatus,COUNT(*), cloud FROM %s WHERE prodSourceLabel IN (:prodSourceLabel1, :prodSourceLabel2) GROUP BY jobStatus, cloud"

            sqlA  = "SELECT /* use_json_type */ /*+ INDEX_RS_ASC(tab (MODIFICATIONTIME PRODSOURCELABEL)) */ jobStatus,COUNT(*), tabS.data.cloud FROM %s tab, ATLAS_PANDA.schedconfig_json tabS "
            sqlA += "WHERE prodSourceLabel IN (:prodSourceLabel1,:prodSourceLabel2) AND tab.computingSite=tabS.panda_queue "
        else:
            sql0  = "SELECT /* use_json_type */ tab.jobStatus, COUNT(*), tabS.data.cloud FROM %s tab, ATLAS_PANDA.schedconfig_json tabS "
            sql0 += "WHERE prodSourceLabel IN (:prodSourceLabel1,"
            for tmpLabel in JobUtils.list_ptest_prod_sources:
                tmpKey = ':prodSourceLabel_{0}'.format(tmpLabel)
                sql0 += tmpKey
                sql0 += ','
            sql0 = sql0[:-1]
            sql0 += ") AND tab.computingSite=tabS.panda_queue GROUP BY tab.jobStatus, tabS.data.cloud"
            sqlA  = "SELECT /* use_json_type */ /*+ INDEX_RS_ASC(tab (MODIFICATIONTIME PRODSOURCELABEL)) */ jobStatus, COUNT(*), tabS.data.cloud FROM %s tab, ATLAS_PANDA.schedconfig_json tabS "
            sqlA += "WHERE prodSourceLabel IN (:prodSourceLabel1,"
            for tmpLabel in JobUtils.list_ptest_prod_sources:
                tmpKey = ':prodSourceLabel_{0}'.format(tmpLabel)
                sqlA += tmpKey
                sqlA += ','
            sqlA = sqlA[:-1]
            sqlA += ") AND tab.computingSite=tabS.panda_queue "
        sqlA+= "AND modificationTime>:modificationTime GROUP BY tab.jobStatus,tabS.data.cloud"
        # sql for materialized view
        sqlMV = re.sub('COUNT\(\*\)','SUM(num_of_jobs)', sql0)
        sqlMV = re.sub('SELECT ','SELECT /*+ RESULT_CACHE */ ', sqlMV)
        ret = {}
        try:
            for table in ('ATLAS_PANDA.jobsActive4', 'ATLAS_PANDA.jobsWaiting4',
                          'ATLAS_PANDA.jobsArchived4', 'ATLAS_PANDA.jobsDefined4'):
                # start transaction
                self.conn.begin()
                # select
                varMap = {}
                if sourcetype == 'analysis':
                    varMap[':prodSourceLabel1'] = 'user'
                    varMap[':prodSourceLabel2'] = 'panda'
                else:
                    varMap[':prodSourceLabel1'] = 'managed'
                    for tmpLabel in JobUtils.list_ptest_prod_sources:
                        tmpKey = ':prodSourceLabel_{0}'.format(tmpLabel)
                        varMap[tmpKey] = tmpLabel
                if table != 'ATLAS_PANDA.jobsArchived4':
                    self.cur.arraysize = 10000
                    if table == 'ATLAS_PANDA.jobsActive4':
                        self.cur.execute((sqlMV+comment) % 'ATLAS_PANDA.MV_JOBSACTIVE4_STATS', varMap)
                    else:
                        self.cur.execute((sql0+comment) % table, varMap)
                else:
                    varMap[':modificationTime'] = timeLimit
                    self.cur.arraysize = 10000
                    self.cur.execute((sqlA+comment) % table, varMap)
                res = self.cur.fetchall()
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                # change NULL to US for old jobs
                newRes = []
                usMap = {}
                for jobStatus,count,cloud in res:
                    ret.setdefault(cloud, dict())
                    ret[cloud].setdefault(jobStatus, 0)
                    ret[cloud][jobStatus] += count
            # return
            _logger.debug("getJobStatisticsForExtIF -> %s" % str(ret))
            return ret
        except Exception:
            # roll back
            self._rollback()
            # error
            type, value, traceBack = sys.exc_info()
            _logger.error("getJobStatisticsForExtIF : %s %s" % (type, value))
            return {}


    # get job statistics per processingType
    def getJobStatisticsPerProcessingType(self,useMorePG=False):
        comment = ' /* DBProxy.getJobStatisticsPerProcessingType */'
        timeLimit = datetime.datetime.utcnow() - datetime.timedelta(hours=12)
        _logger.debug("getJobStatisticsPerProcessingType()")
        if useMorePG is False:
            sqlN  = "SELECT /* use_json_type */ jobStatus, COUNT(*), tabS.data.cloud, processingType "
            sqlN += "FROM %s tab, ATLAS_PANDA.schedconfig_json tabS "
            sqlN += "WHERE prodSourceLabel IN (:prodSourceLabel1,"
            for tmpLabel in JobUtils.list_ptest_prod_sources:
                tmpKey = ':prodSourceLabel_{0}'.format(tmpLabel)
                sqlN += tmpKey
                sqlN += ','
            sqlN = sqlN[:-1]
            sqlN += ") AND computingSite=tabS.panda_queue "
            sqlN += "GROUP BY jobStatus,tabS.data.cloud,processingType "

            sqlA  = "SELECT /* use_json_type */ /*+ INDEX_RS_ASC(tab (MODIFICATIONTIME PRODSOURCELABEL)) */ jobStatus, COUNT(*), tabS.data.cloud, processingType "
            sqlA += "FROM %s tab, ATLAS_PANDA.schedconfig_json tabS "
            sqlA += "WHERE prodSourceLabel IN (:prodSourceLabel1,"
            for tmpLabel in JobUtils.list_ptest_prod_sources:
                tmpKey = ':prodSourceLabel_{0}'.format(tmpLabel)
                sqlA += tmpKey
                sqlA += ','
            sqlA = sqlA[:-1]
            sqlA += ") AND modificationTime>:modificationTime "
            sqlA += "AND computingSite=tabS.panda_queue "
            sqlA += "GROUP BY jobStatus,tabS.data.cloud,processingType"
        
        else:
            sqlN  = "SELECT jobStatus,COUNT(*), cloud, processingType, coreCount, workingGroup FROM %s "
            sqlN += "WHERE prodSourceLabel IN (:prodSourceLabel1,"
            for tmpLabel in JobUtils.list_ptest_prod_sources:
                tmpKey = ':prodSourceLabel_{0}'.format(tmpLabel)
                sqlN += tmpKey
                sqlN += ','
            sqlN = sqlN[:-1]
            sqlN += ") "
            sqlN += "GROUP BY jobStatus, cloud, processingType, coreCount, workingGroup"
            
            sqlA  = "SELECT /*+ INDEX_RS_ASC(tab (MODIFICATIONTIME PRODSOURCELABEL)) */ "
            sqlA += "jobStatus, COUNT(*), cloud, processingType, coreCount, workingGroup FROM %s tab "
            sqlA += "WHERE prodSourceLabel IN (:prodSourceLabel1,"
            for tmpLabel in JobUtils.list_ptest_prod_sources:
                tmpKey = ':prodSourceLabel_{0}'.format(tmpLabel)
                sqlA += tmpKey
                sqlA += ','
            sqlA = sqlA[:-1]
            sqlA += ") AND modificationTime>:modificationTime "
            sqlA += "GROUP BY jobStatus, cloud, processingType, coreCount, workingGroup"
        
        # sql for materialized view
        sqlMV = re.sub('COUNT\(\*\)','SUM(num_of_jobs)',sqlN)
        sqlMV = re.sub('SELECT ','SELECT /*+ RESULT_CACHE */ ',sqlMV)
        ret = {}
        try:
            for table in ('ATLAS_PANDA.jobsActive4', 'ATLAS_PANDA.jobsWaiting4',
                          'ATLAS_PANDA.jobsArchived4', 'ATLAS_PANDA.jobsDefined4'):
                # start transaction
                self.conn.begin()
                # select
                self.cur.arraysize = 10000
                # select
                varMap = {}
                varMap[':prodSourceLabel1'] = 'managed'
                for tmpLabel in JobUtils.list_ptest_prod_sources:
                        tmpKey = ':prodSourceLabel_{0}'.format(tmpLabel)
                        varMap[tmpKey] = tmpLabel
                if table == 'ATLAS_PANDA.jobsArchived4':
                    varMap[':modificationTime'] = timeLimit
                    self.cur.execute((sqlA+comment) % table, varMap)
                else:
                    if table == 'ATLAS_PANDA.jobsActive4' and useMorePG is False:
                        self.cur.execute((sqlMV+comment) % 'ATLAS_PANDA.MV_JOBSACTIVE4_STATS', varMap)
                    else:
                        # use real table since coreCount is unavailable in MatView
                        self.cur.execute((sqlN+comment) % table, varMap)
                res = self.cur.fetchall()
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                # create map
                for tmpItem in res:
                    if useMorePG is False:
                        jobStatus,count,cloud,processingType = tmpItem
                    else:
                        jobStatus,count,cloud,processingType,coreCount,workingGroup = tmpItem
                        # convert cloud and processingType for extended process group
                        if useMorePG == ProcessGroups.extensionLevel_1:
                            # extension level 1
                            cloud,processingType = ProcessGroups.converCPTforEPG(cloud,processingType,
                                                                                 coreCount)
                        else:
                            # extension level 2
                            cloud,processingType = ProcessGroups.converCPTforEPG(cloud,processingType,
                                                                                 coreCount,workingGroup)

                    # add cloud
                    if cloud not in ret:
                        ret[cloud] = {}
                    # add processingType
                    if processingType not in ret[cloud]:
                        ret[cloud][processingType] = {}
                    # add status
                    if jobStatus not in ret[cloud][processingType]:
                        ret[cloud][processingType][jobStatus] = 0
                    ret[cloud][processingType][jobStatus] += count
            # return
            _logger.debug("getJobStatisticsPerProcessingType -> %s" % str(ret))
            return ret
        except Exception:
            # roll back
            self._rollback()
            # error
            type, value, traceBack = sys.exc_info()
            _logger.error("getJobStatisticsPerProcessingType : %s %s" % (type, value))
            return {}


    # get the number of waiting jobs per site and user
    def getJobStatisticsPerUserSite(self):
        comment = ' /* DBProxy.getJobStatisticsPerUserSite */'
        _logger.debug("getJobStatisticsPerUserSite()")
        sqlN  = "SELECT COUNT(*),prodUserID,computingSite FROM %s "
        sqlN += "WHERE prodSourceLabel IN (:prodSourceLabel1,:prodSourceLabel2) AND jobStatus=:jobStatus GROUP BY prodUserID,computingSite"
        ret = {}
        try:
            for table in ('ATLAS_PANDA.jobsActive4','ATLAS_PANDA.jobsDefined4'):
                # start transaction
                self.conn.begin()
                # select
                self.cur.arraysize = 100000
                # select
                if table == 'ATLAS_PANDA.jobsActive4':
                    jobStatus = 'activated'
                else:
                    jobStatus = 'assigned'
                varMap = {}
                varMap[':prodSourceLabel1'] = 'user'
                varMap[':prodSourceLabel2'] = 'panda'
                varMap[':jobStatus'] = jobStatus
                self.cur.execute((sqlN+comment) % table, varMap)
                res = self.cur.fetchall()
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                # create map
                for cnt,prodUserName,computingSite in res:
                    # add site
                    if computingSite not in ret:
                        ret[computingSite] = {}
                    # add user
                    if prodUserName not in ret[computingSite]:
                        ret[computingSite][prodUserName] = {'assigned':0,'activated':0}
                    # add info
                    ret[computingSite][prodUserName][jobStatus] = cnt
            # return
            _logger.debug("getJobStatisticsPerUserSite -> %s" % str(ret))
            return ret
        except Exception:
            # roll back
            self._rollback()
            # error
            errtype,errvalue = sys.exc_info()[:2]
            _logger.error("getJobStatisticsPerUserSite : %s %s" % (errtype,errvalue))
            return {}


    # get number of analysis jobs per user
    def getNUserJobs(self,siteName):
        comment = ' /* DBProxy.getNUserJobs */'
        _logger.debug("getNUserJobs(%s)" % siteName)
        sql0  = "SELECT prodUserID,count(*) FROM ATLAS_PANDA.jobsActive4 "
        sql0 += "WHERE jobStatus=:jobStatus AND prodSourceLabel in (:prodSourceLabel1,:prodSourceLabel2) "
        sql0 += "AND computingSite=:computingSite GROUP BY prodUserID "
        varMap = {}
        varMap[':computingSite'] = siteName
        varMap[':jobStatus'] = 'activated'
        varMap[':prodSourceLabel1'] = 'user'
        varMap[':prodSourceLabel2'] = 'panda'
        ret = {}
        try:
            # start transaction
            self.conn.begin()
            # select
            self.cur.arraysize = 10000
            _logger.debug(1)
            self.cur.execute(sql0+comment, varMap)
            res = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # create map
            for prodUserID,nJobs in res:
                ret[prodUserID] = nJobs
            # return
            _logger.debug("getNUserJobs() : %s" % str(ret))
            return ret
        except Exception:
            # roll back
            self._rollback()
            # error
            type, value, traceBack = sys.exc_info()
            _logger.error("getNUserJobs : %s %s" % (type, value))
            return {}


    # get number of activated analysis jobs
    def getNAnalysisJobs(self,nProcesses):
        comment = ' /* DBProxy.getNAnalysisJobs */'
        _logger.debug("getNAnalysisJobs(%s)" % nProcesses)
        sql0 =  "SELECT computingSite,COUNT(*) FROM ATLAS_PANDA.jobsActive4 WHERE jobStatus=:jobStatus "
        sql0 += "AND (prodSourceLabel=:prodSourceLabel1 OR prodSourceLabel=:prodSourceLabel2) GROUP BY computingSite"
        varMap = {}
        varMap[':jobStatus'] = 'activated'
        varMap[':prodSourceLabel1'] = 'user'
        varMap[':prodSourceLabel2'] = 'panda'
        ret = {}
        try:
            # start transaction
            self.conn.begin()
            # select
            self.cur.arraysize = 10000
            self.cur.execute(sql0+comment)
            res = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # create map
            for item in res:
                ret[item[0]] = float(item[1])/nProcesses
            # return
            _logger.debug("getNAnalysisJobs() : %s" % str(ret))
            return ret
        except Exception:
            # roll back
            self._rollback()
            # error
            type, value, traceBack = sys.exc_info()
            _logger.error("getNAnalysisJobs : %s %s" % (type, value))
            return {}


    # generate pilot token
    def genPilotToken(self,schedulerhost,scheduleruser,schedulerid):
        comment = ' /* DBProxy.genPilotToken */'
        try:
            _logger.debug("genPilotToken(%s,%s,%s)" % (schedulerhost,scheduleruser,schedulerid))
            token = str(uuid.uuid4())
            timeNow = datetime.datetime.utcnow()
            timeExp = timeNow + datetime.timedelta(days=4)
            sql  = "INSERT INTO ATLAS_PANDA.pilottoken (token,schedulerhost,scheduleruser,schedulerid,created,expires) "
            sql += "VALUES (:token,:schedulerhost,:scheduleruser,:schedulerid,:created,:expires)"
            # start transaction
            self.conn.begin()
            # execute
            varMap = {':token':token,':schedulerhost':schedulerhost,':scheduleruser':scheduleruser,
                      ':schedulerid':schedulerid,':created':timeNow,':expires':timeExp}
            self.cur.execute(sql+comment,varMap)
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # return
            retVal = "token=%s,created=%s,expires=%s" % (token,timeNow.strftime('%Y-%m-%d %H:%M:%S'),
                                                         timeExp.strftime('%Y-%m-%d %H:%M:%S'))
            _logger.debug("genPilotToken -> %s" % retVal)
            return retVal
        except Exception:
            # roll back
            self._rollback()
            # error
            type, value, traceBack = sys.exc_info()
            _logger.error("genPilotToken : %s %s" % (type, value))
            return None


    # get list of scheduler users
    def getListSchedUsers(self):
        comment = ' /* DBProxy.getListSchedUsers */'
        try:
            _logger.debug("getListSchedUsers")
            sql  = "SELECT token,scheduleruser FROM ATLAS_PANDA.pilottoken WHERE expires>CURRENT_DATE"
            # start transaction
            self.conn.begin()
            # execute
            self.cur.arraysize = 100
            self.cur.execute(sql+comment)
            res = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # return
            retVal = {}
            for token,scheduleruser in res:
                retVal[token] = scheduleruser
            _logger.debug("getListSchedUsers->%s" % str(retVal))
            return retVal
        except Exception:
            # roll back
            self._rollback()
            # error
            type, value, traceBack = sys.exc_info()
            _logger.error("getListSchedUsers : %s %s" % (type, value))
            return {}


    ###########################################################################
    #
    # LogDBProxy stuff

    # update site data
    def updateSiteData(self, hostID, pilotRequests, interval):
        comment = ' /* DBProxy.updateSiteData */'
        _logger.debug("updateSiteData start")

        sqlDel =  "DELETE FROM ATLAS_PANDAMETA.SiteData WHERE LASTMOD<:LASTMOD"

        sqlRst =  "UPDATE ATLAS_PANDAMETA.SiteData "
        sqlRst += "SET GETJOB=:GETJOB,UPDATEJOB=:UPDATEJOB,NOJOB=:NOJOB,GETJOBABS=:GETJOBABS,UPDATEJOBABS=:UPDATEJOBABS,NOJOBABS=:NOJOBABS "
        sqlRst += "WHERE HOURS=:HOURS AND LASTMOD<:LASTMOD"

        sqlCh  =  "SELECT count(*) FROM ATLAS_PANDAMETA.SiteData WHERE FLAG=:FLAG AND HOURS=:HOURS AND SITE=:SITE"

        sqlIn  =  "INSERT INTO ATLAS_PANDAMETA.SiteData (SITE,FLAG,HOURS,GETJOB,UPDATEJOB,NOJOB,GETJOBABS,UPDATEJOBABS,NOJOBABS,"
        sqlIn  += "LASTMOD,NSTART,FINISHED,FAILED,DEFINED,ASSIGNED,WAITING,ACTIVATED,HOLDING,RUNNING,TRANSFERRING) "
        sqlIn  += "VALUES (:SITE,:FLAG,:HOURS,:GETJOB,:UPDATEJOB,:NOJOB,:GETJOBABS,:UPDATEJOBABS,:NOJOBABS,CURRENT_DATE,"
        sqlIn  += "0,0,0,0,0,0,0,0,0,0)"

        sqlUp  =  "UPDATE ATLAS_PANDAMETA.SiteData SET GETJOB=:GETJOB,UPDATEJOB=:UPDATEJOB,NOJOB=:NOJOB,"
        sqlUp  += "GETJOBABS=:GETJOBABS,UPDATEJOBABS=:UPDATEJOBABS,NOJOBABS=:NOJOBABS,LASTMOD=CURRENT_DATE "
        sqlUp  += "WHERE FLAG=:FLAG AND HOURS=:HOURS AND SITE=:SITE"

        sqlAll  = "SELECT getJob,updateJob,noJob,getJobAbs,updateJobAbs,noJobAbs,FLAG "
        sqlAll +=  "FROM ATLAS_PANDAMETA.SiteData WHERE HOURS=:HOURS AND SITE=:SITE"

        try:
            timeNow = datetime.datetime.utcnow()
            self.conn.begin()
            # delete old records
            varMap = {}
            varMap[':LASTMOD'] = timeNow - datetime.timedelta(hours=48)
            self.cur.execute(sqlDel+comment,varMap)
            # set 0 to old records
            varMap = {}
            varMap[':HOURS'] = interval
            varMap[':GETJOB'] = 0
            varMap[':UPDATEJOB'] = 0
            varMap[':NOJOB'] = 0
            varMap[':GETJOBABS'] = 0
            varMap[':UPDATEJOBABS'] = 0
            varMap[':NOJOBABS'] = 0
            varMap[':LASTMOD'] = timeNow - datetime.timedelta(hours=interval)
            self.cur.execute(sqlRst+comment,varMap)
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # shuffle to avoid concatenation
            tmpSiteList = list(pilotRequests)
            random.shuffle(tmpSiteList)
            # loop over all sites
            for tmpSite in tmpSiteList:
                tmpVal = pilotRequests[tmpSite]
                # start transaction
                self.conn.begin()
                # check individual host info first
                varMap = {}
                varMap[':FLAG']  = hostID
                varMap[':SITE']  = tmpSite
                varMap[':HOURS'] = interval
                self.cur.arraysize = 10
                self.cur.execute(sqlCh+comment,varMap)
                res = self.cur.fetchone()
                # row exists or not
                if res[0] == 0:
                    sql = sqlIn
                else:
                    sql = sqlUp

                # getJob, updateJob and noJob entries contain the number of slots/nodes that submitted the request
                # getJobAbs, updateJobAbs and noJobAbs entries contain the absolute number of requests
                if 'getJob' in tmpVal:
                    varMap[':GETJOB'] = len(tmpVal['getJob'])
                    getJobAbs = 0
                    for node in tmpVal['getJob']:
                        getJobAbs += tmpVal['getJob'][node]
                    varMap[':GETJOBABS'] = getJobAbs
                else:
                    varMap[':GETJOB'] = 0
                    varMap[':GETJOBABS'] = 0

                if 'updateJob' in tmpVal:
                    varMap[':UPDATEJOB'] = len(tmpVal['updateJob'])
                    updateJobAbs = 0
                    for node in tmpVal['updateJob']:
                        updateJobAbs += tmpVal['updateJob'][node]
                    varMap[':UPDATEJOBABS'] = updateJobAbs
                else:
                    varMap[':UPDATEJOB'] = 0
                    varMap[':UPDATEJOBABS'] = 0

                if 'noJob' in tmpVal:
                    varMap[':NOJOB'] = len(tmpVal['noJob'])
                    noJobAbs = 0
                    for node in tmpVal['noJob']:
                        noJobAbs += tmpVal['noJob'][node]
                    varMap[':NOJOBABS'] = noJobAbs
                else:
                    varMap[':NOJOB'] = 0
                    varMap[':NOJOBABS'] = 0

                # update
                self.cur.execute(sql+comment,varMap)
                # get all info
                sumExist = False
                varMap = {}
                varMap[':SITE']  = tmpSite
                varMap[':HOURS'] = interval
                self.cur.arraysize = 100
                self.cur.execute(sqlAll+comment,varMap)
                res = self.cur.fetchall()
                # get total getJob/updateJob
                varMap[':GETJOB'] = 0
                varMap[':UPDATEJOB'] = 0
                varMap[':NOJOB'] = 0
                varMap[':GETJOBABS'] = 0
                varMap[':UPDATEJOBABS'] = 0
                varMap[':NOJOBABS'] = 0
                nCol = 0
                for tmpGetJob, tmpUpdateJob, tmpNoJob, tmpGetJobAbs, tmpUpdateJobAbs, tmpNoJobAbs, tmpFlag in res:
                    # don't use summed info
                    if tmpFlag == 'production':
                        sumExist = True
                        continue
                    if tmpFlag == 'analysis':
                        if tmpSite.startswith('ANALY_'):
                            sumExist = True
                        continue
                    if tmpFlag in ['test']:
                        continue

                    if tmpGetJob is None:
                        tmpGetJob = 0
                    if tmpUpdateJob is None:
                        tmpUpdateJob = 0
                    if tmpNoJob is None:
                        tmpNoJob = 0
                    if tmpGetJobAbs is None:
                        tmpGetJobAbs = 0
                    if tmpUpdateJobAbs is None:
                        tmpUpdateJobAbs = 0
                    if tmpNoJobAbs is None:
                        tmpNoJobAbs = 0

                    # sum
                    varMap[':GETJOB'] += tmpGetJob
                    varMap[':UPDATEJOB'] += tmpUpdateJob
                    varMap[':NOJOB'] += tmpNoJob
                    varMap[':GETJOBABS'] += tmpGetJobAbs
                    varMap[':UPDATEJOBABS'] += tmpUpdateJobAbs
                    varMap[':NOJOBABS'] += tmpNoJobAbs
                    nCol += 1
                # get average
                if nCol != 0:
                    if varMap[':GETJOB'] >= nCol:
                        varMap[':GETJOB'] /= nCol
                    if varMap[':UPDATEJOB'] >= nCol:
                        varMap[':UPDATEJOB'] /= nCol
                    if varMap[':NOJOB'] >= nCol:
                        varMap[':NOJOB'] /= nCol
                    if varMap[':GETJOBABS'] >= nCol:
                        varMap[':GETJOBABS'] /= nCol
                    if varMap[':UPDATEJOBABS'] >= nCol:
                        varMap[':UPDATEJOBABS'] /= nCol
                    if varMap[':NOJOBABS'] >= nCol:
                        varMap[':NOJOBABS'] /= nCol

                if tmpSite.startswith('ANALY_'):
                    varMap[':FLAG']  = 'analysis'
                else:
                    varMap[':FLAG']  = 'production'
                # row exists or not
                if sumExist:
                    sql = sqlUp
                else:
                    sql = sqlIn
                # update
                self.cur.execute(sql+comment,varMap)
                _logger.debug('updateSiteData : %s hours=%s getJob=%s updateJob=%s, noJob=%s, getJobAbs=%s updateJobAbs=%s, noJobAbs=%s' %
                              (tmpSite,interval,varMap[':GETJOB'],varMap[':UPDATEJOB'],varMap[':NOJOB'],
                               varMap[':GETJOBABS'], varMap[':UPDATEJOBABS'], varMap[':NOJOBABS']))
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
            _logger.debug("updateSiteData done")
            return True
        except Exception:
            # roll back
            self._rollback()
            type, value, traceBack = sys.exc_info()
            _logger.error("updateSiteData : %s %s" % (type,value))
            return False


    # get site data
    def getCurrentSiteData(self):
        comment = ' /* DBProxy.getCurrentSiteData */'
        _logger.debug("getCurrentSiteData")
        sql = "SELECT SITE,getJob,updateJob,FLAG FROM ATLAS_PANDAMETA.SiteData WHERE FLAG IN (:FLAG1,:FLAG2) and HOURS=3"
        varMap = {}
        varMap[':FLAG1'] = 'production'
        varMap[':FLAG2'] = 'analysis'
        try:
            # set autocommit on
            self.conn.begin()
            # select
            self.cur.arraysize = 10000
            self.cur.execute(sql+comment, varMap)
            res = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            ret = {}
            for site,getJob,updateJob,flag in res:
                if site.startswith('ANALY_'):
                    if flag != 'analysis':
                        continue
                else:
                    if flag != 'production':
                        continue
                ret[site] = {'getJob':getJob,'updateJob':updateJob}
            return ret
        except Exception:
            type, value, traceBack = sys.exc_info()
            _logger.error("getCurrentSiteData : %s %s" % (type,value))
            # roll back
            self._rollback()
            return {}


    # insert nRunning in site data
    def insertnRunningInSiteData(self):
        comment = ' /* DBProxy.insertnRunningInSiteData */'
        _logger.debug("insertnRunningInSiteData start")
        sqlDel =  "DELETE FROM ATLAS_PANDAMETA.SiteData WHERE FLAG IN (:FLAG1,:FLAG2) AND LASTMOD<CURRENT_DATE-1"
        sqlRun =  "SELECT COUNT(*),computingSite FROM ATLAS_PANDA.jobsActive4 "
        sqlRun += "WHERE prodSourceLabel IN (:prodSourceLabel1,:prodSourceLabel2) "
        sqlRun += "AND jobStatus=:jobStatus GROUP BY computingSite"
        sqlCh  =  "SELECT COUNT(*) FROM ATLAS_PANDAMETA.SiteData WHERE FLAG=:FLAG AND HOURS=:HOURS AND SITE=:SITE"
        sqlIn  =  "INSERT INTO ATLAS_PANDAMETA.SiteData (SITE,FLAG,HOURS,GETJOB,UPDATEJOB,LASTMOD,"
        sqlIn  += "NSTART,FINISHED,FAILED,DEFINED,ASSIGNED,WAITING,ACTIVATED,HOLDING,RUNNING,TRANSFERRING) "
        sqlIn  += "VALUES (:SITE,:FLAG,:HOURS,0,0,CURRENT_DATE,"
        sqlIn  += "0,0,0,0,0,0,0,0,:RUNNING,0)"
        sqlUp  =  "UPDATE ATLAS_PANDAMETA.SiteData SET RUNNING=:RUNNING,LASTMOD=CURRENT_DATE "
        sqlUp  += "WHERE FLAG=:FLAG AND HOURS=:HOURS AND SITE=:SITE"
        sqlMax =  "SELECT SITE,MAX(RUNNING) FROM ATLAS_PANDAMETA.SiteData "
        sqlMax += "WHERE FLAG=:FLAG GROUP BY SITE"
        try:
            # use offset(1000)+minites for :HOURS
            timeNow = datetime.datetime.utcnow()
            nHours = 1000 + timeNow.hour*60 + timeNow.minute
            # delete old records
            varMap = {}
            varMap[':FLAG1'] = 'max'
            varMap[':FLAG2'] = 'snapshot'
            self.conn.begin()
            self.cur.execute(sqlDel+comment,varMap)
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # get nRunning
            varMap = {}
            varMap[':jobStatus'] = 'running'
            varMap[':prodSourceLabel1'] = 'user'
            varMap[':prodSourceLabel2'] = 'panda'
            self.conn.begin()
            self.cur.arraysize = 10000
            self.cur.execute(sqlRun+comment,varMap)
            res = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # loop over all sites
            for nRunning,computingSite in res:
                # only ANALY_ sites
                if not computingSite.startswith('ANALY_'):
                    continue
                # check if the row is already there
                varMap = {}
                varMap[':FLAG']  = 'snapshot'
                varMap[':SITE']  = computingSite
                varMap[':HOURS'] = nHours
                # start transaction
                self.conn.begin()
                self.cur.arraysize = 10
                self.cur.execute(sqlCh+comment,varMap)
                res = self.cur.fetchone()
                # row exists or not
                if res[0] == 0:
                    sql = sqlIn
                else:
                    sql = sqlUp
                # set current nRunning
                varMap = {}
                varMap[':FLAG']    = 'snapshot'
                varMap[':SITE']    = computingSite
                varMap[':HOURS']   = nHours
                varMap[':RUNNING'] = nRunning
                # insert or update
                self.cur.execute(sql+comment,varMap)
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
            # get max nRunning
            varMap = {}
            varMap[':FLAG']  = 'snapshot'
            self.conn.begin()
            self.cur.arraysize = 10000
            self.cur.execute(sqlMax+comment,varMap)
            res = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # loop over all sites
            for computingSite,maxnRunning in res:
                # start transaction
                self.conn.begin()
                # check if the row is already there
                varMap = {}
                varMap[':FLAG']  = 'max'
                varMap[':SITE']  = computingSite
                varMap[':HOURS'] = 0
                self.cur.arraysize = 10
                self.cur.execute(sqlCh+comment,varMap)
                res = self.cur.fetchone()
                # row exists or not
                if res[0] == 0:
                    sql = sqlIn
                else:
                    sql = sqlUp
                # set max nRunning
                varMap = {}
                varMap[':FLAG']  = 'max'
                varMap[':SITE']  = computingSite
                varMap[':HOURS'] = 0
                varMap[':RUNNING'] = maxnRunning
                self.cur.execute(sql+comment,varMap)
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
            _logger.debug("insertnRunningInSiteData done")
            return True
        except Exception:
            # roll back
            self._rollback()
            type, value, traceBack = sys.exc_info()
            _logger.error("insertnRunningInSiteData : %s %s" % (type,value))
            return False


    # get nRunning in site data
    def getnRunningInSiteData(self):
        comment = ' /* DBProxy.getnRunningInSiteData */'
        _logger.debug("getnRunningInSiteData start")
        sqlMax =  "SELECT SITE,RUNNING FROM ATLAS_PANDAMETA.SiteData WHERE HOURS=:HOURS AND FLAG=:FLAG"
        try:
            # get nRunning
            varMap = {}
            varMap[':FLAG']  = 'max'
            varMap[':HOURS'] = 0
            # start transaction
            self.conn.begin()
            self.cur.arraysize = 10000
            # get
            self.cur.execute(sqlMax+comment,varMap)
            res = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # loop over all sites
            retMap = {}
            for computingSite,maxnRunning in res:
                retMap[computingSite] = maxnRunning
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            _logger.debug("getnRunningInSiteData done")
            return retMap
        except Exception:
            # roll back
            self._rollback()
            type, value, traceBack = sys.exc_info()
            _logger.error("getnRunningInSiteData : %s %s" % (type,value))
            return {}

    # get site info
    def getSiteInfo(self):
        comment = ' /* DBProxy.getSiteInfo */'
        _logger.debug("getSiteInfo start")
        methodName = comment.split(' ')[-2].split('.')[-1]
        try:
            # set autocommit on
            self.conn.begin()
            # get CVMFS availability
            sqlCVMFS  = "SELECT distinct siteid FROM ATLAS_PANDAMETA.installedSW WHERE `release`=:release"
            self.cur.execute(sqlCVMFS, {':release': 'CVMFS'})
            tmpList = self.cur.fetchall()
            cvmfsSites = []
            for tmpItem, in tmpList:
                if tmpItem not in cvmfsSites:
                    cvmfsSites.append(tmpItem)
            if not self._commit():
                raise RuntimeError('Commit error')

            # get DDM endpoints
            pandaEndpointMap = self.getDdmEndpoints()

            # sql to get site spec
            sql = """
                   SELECT /* use_json_type */ panda_queue, data, b.site_name, c.role
                   FROM (ATLAS_PANDA.schedconfig_json a
                   LEFT JOIN ATLAS_PANDA.panda_site b ON a.panda_queue = b.panda_site_name)
                   LEFT JOIN ATLAS_PANDA.site c ON b.site_name = c.site_name
                   WHERE panda_queue IS NOT NULL
                   """
            self.cur.arraysize = 10000
            # self.cur.execute(sql+comment)
            # resList = self.cur.fetchall()
            ret, resList = self.getClobObj(sql, {})
            if not resList:
                _logger.error('Empty site list!')

            # set autocommit on
            self.conn.begin()
            # sql to get num slots
            sqlSL = "SELECT pandaQueueName, gshare, resourcetype, numslots FROM ATLAS_PANDA.Harvester_Slots "
            sqlSL += "WHERE (expirationTime IS NULL OR expirationTime>CURRENT_DATE) "

            num_slots_by_site = {}
            self.cur.execute(sqlSL + comment)
            resSL = self.cur.fetchall()

            for sl_queuename, sl_gshare, sl_resourcetype, sl_numslots in resSL:
                if sl_numslots < 0:
                    continue
                num_slots_by_site.setdefault(sl_queuename, {})
                num_slots_by_site[sl_queuename].setdefault(sl_gshare, {})
                num_slots_by_site[sl_queuename][sl_gshare][sl_resourcetype] = sl_numslots


            retList = {}
            if resList is not None:
                # loop over all results
                for res in resList:
                    try:   # don't let a problem with one queue break the whole map
                        # change None to ''
                        resTmp = []
                        for tmpItem in res:
                            if tmpItem is None:
                                tmpItem = ''
                            resTmp.append(tmpItem)

                        siteid, queue_data_json, pandasite, role = resTmp
                        try:
                            if isinstance(queue_data_json, dict):
                                queue_data = queue_data_json
                            else:
                                queue_data = json.loads(queue_data_json)
                        except Exception:
                            _logger.error("loading json for queue {0} excepted. json was: {1}".format(siteid, queue_data_json))
                            continue

                        # skip invalid siteid
                        if siteid in [None,''] or not queue_data:
                            _logger.error("siteid {0} had no queue_data {1}".format(siteid, queue_data))
                            continue

                        _logger.debug("processing queue {0}".format(siteid))

                        # instantiate SiteSpec
                        ret = SiteSpec.SiteSpec()
                        ret.sitename = siteid
                        ret.pandasite = pandasite
                        ret.role = role

                        ret.type = queue_data.get('type', 'production')
                        ret.nickname = queue_data.get('nickname')
                        try:
                            ret.ddm = queue_data.get('ddm', '').split(',')[0]
                        except AttributeError:
                            ret.ddm = ''
                        try:
                            ret.cloud = queue_data.get('cloud', '').split(',')[0]
                        except AttributeError:
                            ret.cloud = ''
                        ret.memory = queue_data.get('memory')
                        ret.maxrss = queue_data.get('maxrss')
                        ret.minrss = queue_data.get('minrss')
                        ret.maxtime = queue_data.get('maxtime')
                        ret.status = queue_data.get('status')
                        ret.space = queue_data.get('space')
                        ret.maxinputsize = queue_data.get('maxinputsize')
                        ret.comment = queue_data.get('comment_')
                        ret.statusmodtime = queue_data.get('lastmod')
                        ret.catchall = queue_data.get('catchall')
                        ret.tier = queue_data.get('tier')
                        ret.jobseed = queue_data.get('jobseed')
                        ret.capability = queue_data.get('capability')
                        ret.workflow = queue_data.get('workflow')
                        ret.maxDiskio = queue_data.get('maxdiskio')
                        ret.pandasite_state = 'ACTIVE'
                        ret.fairsharePolicy = queue_data.get('fairsharepolicy')
                        ret.priorityoffset = queue_data.get('priorityoffset')
                        ret.defaulttoken   = queue_data.get('defaulttoken')

                        ret.direct_access_lan = (queue_data.get('direct_access_lan') is True)
                        ret.direct_access_wan = (queue_data.get('direct_access_wan') is True)

                        if queue_data.get('corepower') is None:
                            ret.corepower = 0
                        else:
                            ret.corepower = queue_data.get('corepower')

                        ret.wnconnectivity = queue_data.get('wnconnectivity')
                        if ret.wnconnectivity == '':
                            ret.wnconnectivity = None

                        # maxwdir
                        try:
                            if queue_data.get('maxwdir') is None:
                                ret.maxwdir = 0
                            else:
                                ret.maxwdir = int(queue_data['maxwdir'])
                        except Exception:
                            if ret.maxinputsize in [0, None]:
                                ret.maxwdir = 0
                            else:
                                try:
                                    ret.maxwdir = ret.maxinputsize + 2000
                                except Exception:
                                    ret.maxwdir = 16336

                        # mintime
                        if queue_data.get('mintime') is not None:
                            ret.mintime = queue_data['mintime']
                        else:
                            ret.mintime = 0

                        # reliability
                        ret.reliabilityLevel = None

                        # pledged CPUs
                        ret.pledgedCPU = 0
                        if queue_data.get('pledgedcpu') not in ['', None]:
                            try:
                                ret.pledgedCPU = int(queue_data['pledgedcpu'])
                            except Exception:
                                pass

                        # core count
                        ret.coreCount = 0
                        if queue_data.get('corecount') not in ['', None]:
                            try:
                                ret.coreCount = int(queue_data['corecount'])
                            except Exception:
                                pass

                        # cloud list
                        if queue_data.get('cloud') not in ['', None]:
                            ret.cloudlist = [queue_data['cloud'].split(',')[0]]
                            if queue_data.get('multicloud') not in ['', None, 'None']:
                                ret.cloudlist += queue_data['multicloud'].split(',')
                        else:
                            ret.cloudlist = []

                        # convert releases to list
                        ret.releases = []
                        if queue_data.get('releases'):
                            ret.releases = queue_data['releases']

                        # convert validatedreleases to list
                        ret.validatedreleases = []
                        if queue_data.get('validatedreleases'):
                            for tmpRel in queue_data['validatedreleases'].split('|'):
                                # remove white space
                                tmpRel = tmpRel.strip()
                                if tmpRel != '':
                                    ret.validatedreleases.append(tmpRel)

                        # CVMFS
                        if siteid in cvmfsSites:
                            ret.iscvmfs = True
                        else:
                            ret.iscvmfs = False

                        # limit of the number of transferring jobs
                        ret.transferringlimit = 0
                        if queue_data.get('transferringlimit') not in ['', None]:
                            try:
                                ret.transferringlimit = int(queue_data['transferringlimit'])
                            except Exception:
                                pass

                        # FAX
                        ret.allowfax = False
                        try:
                            if queue_data.get('catchall') is not None and 'allowfax' in queue_data['catchall']:
                                ret.allowfax = True
                            if queue_data.get('allowfax') is True:
                                ret.allowfax = True
                        except Exception:
                            pass

                        # DDM endpoints
                        ret.ddm_endpoints_input = {}
                        ret.ddm_endpoints_output = {}
                        if siteid in pandaEndpointMap:
                            for scope in pandaEndpointMap[siteid]:
                                if 'input' in pandaEndpointMap[siteid][scope]:
                                    ret.ddm_endpoints_input[scope] = pandaEndpointMap[siteid][scope]['input']
                                if 'output' in pandaEndpointMap[siteid][scope]:
                                    ret.ddm_endpoints_output[scope] = pandaEndpointMap[siteid][scope]['output']
                        else:
                            # empty
                            ret.ddm_endpoints_input['default'] = DdmSpec()
                            ret.ddm_endpoints_output['default'] = DdmSpec()

                        # initialize dictionary fields
                        ret.setokens_input = {}
                        ret.setokens_output = {}
                        ret.ddm_input = {}
                        for scope in ret.ddm_endpoints_input:
                            # mapping between token and endpoints
                            ret.setokens_input[scope] = ret.ddm_endpoints_input[scope].getTokenMap('input')
                            # set DDM to the default endpoint
                            ret.ddm_input[scope] = ret.ddm_endpoints_input[scope].getDefaultRead()

                        ret.ddm_output = {}
                        for scope in ret.ddm_endpoints_output:
                            # mapping between token and endpoints
                            ret.setokens_output[scope] = ret.ddm_endpoints_output[scope].getTokenMap('output')
                            # set DDM to the default endpoint
                            ret.ddm_output[scope] = ret.ddm_endpoints_output[scope].getDefaultWrite()

                        # object stores
                        try:
                            ret.objectstores = queue_data['objectstores']
                        except Exception:
                            ret.objectstores = []

                        # default unified flag
                        ret.is_unified = False

                        # num slots
                        ret.num_slots_map = num_slots_by_site.get(siteid, {})

                        # append
                        retList[ret.nickname] = ret
                    except Exception:
                        _logger.error('getSiteInfo exception in queue: {0}'.format(traceback.format_exc()))
                        continue
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            _logger.debug("getSiteInfo done")
            return retList
        except Exception as e:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return {}


    def getDdmEndpoints(self):
        """
        get list of ddm input endpoints
        """
        comment = ' /* DBProxy.getDdmEndpoints */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        _logger.debug("{0} start".format(methodName))

        # get all ddm endpoints
        sql_ddm  = "SELECT * FROM ATLAS_PANDA.ddm_endpoint "
        self.cur.arraysize = 10000
        self.cur.execute('{0}{1}'.format(sql_ddm, comment))
        results_ddm = self.cur.fetchall()

        # extract the column names from the query
        column_names = [i[0].lower() for i in self.cur.description]

        # save the endpoints into a dictionary
        endpoint_dict = {}
        for ddm_endpoint_row in results_ddm:
            tmp_endpoint = {}
            # unzip the ddm_endpoint row into a dictionary
            for column_name, column_val in zip(column_names, ddm_endpoint_row):
                tmp_endpoint[column_name] = column_val

            # ignore TEST
            # if tmp_endpoint['type'] == 'TEST':
            #    continue

            endpoint_dict[tmp_endpoint['ddm_endpoint_name']] = tmp_endpoint

        # get relationship between panda sites and ddm endpoints
        sql_panda_ddm = """
               SELECT pdr.panda_site_name, pdr.ddm_endpoint_name, pdr.is_local, de.ddm_spacetoken_name,
                      de.is_tape, pdr.default_read, pdr.default_write, pdr.roles, pdr.order_read, pdr.order_write,
                      nvl(pdr.scope, 'default') as scope, de.blacklisted_read
               FROM ATLAS_PANDA.panda_ddm_relation pdr, ATLAS_PANDA.ddm_endpoint de
               WHERE pdr.ddm_endpoint_name = de.ddm_endpoint_name
               """
        if self.backend == 'mysql':
            sql_panda_ddm = """
               SELECT pdr.panda_site_name, pdr.ddm_endpoint_name, pdr.is_local, de.ddm_spacetoken_name,
                      de.is_tape, pdr.default_read, pdr.default_write, pdr.roles, pdr.order_read, pdr.order_write,
                      ifnull(pdr.scope, 'default') as scope, de.blacklisted
               FROM ATLAS_PANDA.panda_ddm_relation pdr, ATLAS_PANDA.ddm_endpoint de
               WHERE pdr.ddm_endpoint_name = de.ddm_endpoint_name
               """

        self.cur.execute('{0}{1}'.format(sql_panda_ddm, comment))
        results_panda_ddm = self.cur.fetchall()
        column_names = [i[0].lower() for i in self.cur.description]

        # save the panda ddm relations into a dictionary
        panda_endpoint_map = {}
        for panda_ddm_row in results_panda_ddm:
            tmp_relation = {}
            for column_name, column_val in zip(column_names, panda_ddm_row):
                # Default unavailable endpoint space to 0
                if column_name.startswith('space_') and column_val is None:
                    column_val = 0
                tmp_relation[column_name] = column_val

            # add the relations to the panda endpoint map
            panda_site_name = tmp_relation['panda_site_name']
            scope = tmp_relation['scope']
            panda_endpoint_map.setdefault(panda_site_name, {})
            panda_endpoint_map[panda_site_name].setdefault(scope, {})

            if panda_site_name not in panda_endpoint_map:
                panda_endpoint_map[panda_site_name] = {'input': DdmSpec(), 'output': DdmSpec()}
            if 'read_lan' in tmp_relation['roles'] and tmp_relation['blacklisted_read'] != 'Y':
                panda_endpoint_map[panda_site_name][scope].setdefault('input', DdmSpec())
                panda_endpoint_map[panda_site_name][scope]['input'].add(tmp_relation, endpoint_dict)
            if 'write_lan' in tmp_relation['roles']:
                panda_endpoint_map[panda_site_name][scope].setdefault('output', DdmSpec())
                panda_endpoint_map[panda_site_name][scope]['output'].add(tmp_relation, endpoint_dict)

        _logger.debug("{0} done".format(methodName))
        return panda_endpoint_map


    # get dispatch sorting criteria
    def getSortingCriteria(self, site_name, max_jobs):
        method_name = 'getSortingCriteria'
        # throw the dice to decide the algorithm
        random_number = random.randrange(100)

        sloppy_ratio = self.getConfigValue('jobdispatch', 'SLOPPY_DISPATCH_RATIO')
        if not sloppy_ratio:
            sloppy_ratio = 10

        _logger.debug('{0} random_number: {1} sloppy_ratio: {2}'.format(method_name, random_number, sloppy_ratio))

        if random_number <= sloppy_ratio:
            # generate the age sorting
            _logger.debug('{0} sorting by age'.format(method_name))
            return self.getCriteriaByAge(site_name, max_jobs)
        else:
            # generate the global share sorting
            _logger.debug('{0} sorting by gshare'.format(method_name))
            return self.getCriteriaForGlobalShares(site_name, max_jobs)

    # get selection criteria for share of production activities
    def getCriteriaForGlobalShares(self, site_name, max_jobs):
        method_name = 'getCriteriaForGlobalShare'
        # return for no criteria
        var_map = {}
        ret_empty = '', {}

        try:
            # Get the share leaves sorted by order of under-pledging
            _logger.debug('{0} Going to call get sorted leaves'.format(method_name))
            t_before = time.time()
            sorted_leaves = self.get_sorted_leaves()
            t_after = time.time()
            total = t_after - t_before
            _logger.debug('{0} Sorting leaves took {1}s'.format(method_name, total))

            i = 0
            tmp_list = []
            for leave in sorted_leaves:
                var_map[':leave{0}'.format(i)] = leave.name
                if leave.name == 'Test':
                    # Test share will bypass others for the moment
                    tmp_list.append('WHEN gshare=:leave{0} THEN 0'.format(i))
                else:
                    tmp_list.append('WHEN gshare=:leave{0} THEN {0}'.format(i))
                i += 1

            # Only get max_jobs, to avoid getting all activated jobs from the table
            var_map[':njobs'] = max_jobs

            # We want to sort by global share, highest priority and lowest pandaid
            leave_bindings = ' '.join(tmp_list)
            ret_sql = """
                      ORDER BY (CASE {1} ELSE {2} END), currentpriority desc, pandaid asc)
                      WHERE ROWNUM <= {0}
                      """.format(':njobs', leave_bindings, len(sorted_leaves))

            _logger.debug('{0} ret_sql: {1}'.format(method_name, ret_sql))
            _logger.debug('{0} var_map: {1}'.format(method_name, var_map))
            return ret_sql, var_map

        except Exception:
            err_type, err_value = sys.exc_info()[:2]
            err_str = "{0} {1} : {2} {3}".format(method_name, site_name, err_type, err_value)
            err_str.strip()
            err_str += traceback.format_exc()
            _logger.error(err_str)
            # roll back
            self._rollback()
            return ret_empty

    # get selection criteria for share of production activities
    def getCriteriaByAge(self, site_name, max_jobs):
        comment = ' /* DBProxy.getCriteriaByAge */'
        # return for no criteria
        ret_sql = ''
        var_map = {}
        ret_empty = '', {}

        try:
            # Only get max_jobs, to avoid getting all activated jobs from the table
            var_map[':njobs'] = max_jobs

            # We want to ignore global share and just take the oldest pandaid
            ret_sql = """
                      ORDER BY pandaid asc)
                      WHERE ROWNUM <= :njobs
                      """

            _logger.debug('ret_sql: {0}'.format(ret_sql))
            _logger.debug('var_map: {0}'.format(var_map))
            return ret_sql, var_map

        except Exception:
            err_type, err_value = sys.exc_info()[:2]
            err_str = "getCriteriaByAge {0} : {1} {2}".format(site_name, err_type, err_value)
            err_str.strip()
            err_str += traceback.format_exc()
            _logger.error(err_str)
            # roll back
            self._rollback()
            return ret_empty


    # get fairshare policy
    def getFairsharePolicy(self, getNewMap=False):
        comment = ' /* DBProxy.getFairsharePolicy */'
        # check utime
        if not getNewMap and self.updateTimeForFairsharePolicy is not None and \
               (datetime.datetime.utcnow()-self.updateTimeForFairsharePolicy) < datetime.timedelta(minutes=15):
            return
        if not getNewMap:
            # update utime
            self.updateTimeForFairsharePolicy = datetime.datetime.utcnow()
        _logger.debug("getFairsharePolicy")
        try:
            # set autocommit on
            self.conn.begin()
            # get default share
            cloudShareMap = {}
            cloudTier1Map = {}
            sqlD = "SELECT name, fairshare, tier1 FROM ATLAS_PANDAMETA.cloudconfig"
            self.cur.arraysize = 100000
            self.cur.execute(sqlD+comment)
            res = self.cur.fetchall()
            for cloudName,cloudShare,cloudTier1 in res:
                try:
                    cloudTier1Map[cloudName] = cloudTier1.split(',')
                except Exception:
                    pass
                if cloudShare not in ['',None]:
                    cloudShareMap[cloudName] = cloudShare
            # get share per site
            sql  = "SELECT /* use_json_type */ scj.panda_queue, scj.data.fairsharepolicy, scj.data.cloud "
            sql += "FROM ATLAS_PANDA.schedconfig_json scj "
            sql += "WHERE scj.data.type != 'analysis' "
            sql += "GROUP BY scj.panda_queue, scj.data.fairsharepolicy, scj.data.cloud"

            self.cur.execute(sql+comment)
            res = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # update policy
            fairsharePolicy = {}
            for siteid, fairsharePolicyStr, cloudName in res:
                try:
                    # consolidate to WORLD
                    cloudName = 'WORLD'
                    # share is undefined
                    usingCloudShare = ''
                    if fairsharePolicyStr in ['', None, ' None']:
                        # skip if share is not defined at site or cloud
                        if cloudName not in cloudShareMap:
                            continue
                        # use cloud share
                        fairsharePolicyStr = cloudShareMap[cloudName]
                        usingCloudShare = cloudName
                    # decompose
                    hasNonPrioPolicy = False
                    for tmpItem in fairsharePolicyStr.split(','):
                        # skip empty
                        tmpItem = tmpItem.strip()
                        if tmpItem == '':
                            continue
                        # keep name
                        tmpPolicy = {'name':tmpItem}
                        # group
                        tmpPolicy['group'] = None
                        tmpMatch = re.search('group=([^:]+)',tmpItem)
                        if tmpMatch is not None:
                            if tmpMatch.group(1) in ['','central','*','any']:
                                # use None for catchall
                                pass
                            else:
                                tmpPolicy['group'] = tmpMatch.group(1)
                        # type
                        tmpPolicy['type'] = None
                        tmpMatch = re.search('type=([^:]+)',tmpItem)
                        if tmpMatch is not None:
                            if tmpMatch.group(1) in ['*','any']:
                                # use None for catchall
                                pass
                            else:
                                tmpPolicy['type'] = tmpMatch.group(1)
                        # priority
                        tmpPolicy['priority'] = None
                        tmpPolicy['prioCondition'] = None
                        tmpMatch = re.search('priority([=<>]+)(\d+)',tmpItem)
                        if tmpMatch is not None:
                            tmpPolicy['priority'] = int(tmpMatch.group(2))
                            tmpPolicy['prioCondition'] = tmpMatch.group(1)
                        else:
                            hasNonPrioPolicy = True
                        # workqueue_ID
                        tmpPolicy['id'] = None
                        tmpMatch = re.search('id=([^:]+)',tmpItem)
                        if tmpMatch is not None:
                            if tmpMatch.group(1) in ['*','any']:
                                # use None for catchall
                                pass
                            else:
                                try:
                                    tmpPolicy['id'] = int(tmpMatch.group(1))
                                except Exception:
                                    pass
                        # share
                        tmpPolicy['share'] = tmpItem.split(':')[-1]
                        # append
                        if siteid not in fairsharePolicy:
                            fairsharePolicy[siteid] = {'policyList':[]}
                        fairsharePolicy[siteid]['policyList'].append(tmpPolicy)
                    # add any:any if only priority policies
                    if not hasNonPrioPolicy:
                        tmpPolicy = {'name'          : 'type=any',
                                     'group'         : None,
                                     'type'          : None,
                                     'id'            : None,
                                     'priority'      : None,
                                     'prioCondition' : None,
                                     'share'         : '100%'}
                        fairsharePolicy[siteid]['policyList'].append(tmpPolicy)
                    # some translation
                    fairsharePolicy[siteid]['usingGroup'] = False
                    fairsharePolicy[siteid]['usingType']  = False
                    fairsharePolicy[siteid]['usingID']    = False
                    fairsharePolicy[siteid]['usingPrio']  = False
                    fairsharePolicy[siteid]['usingCloud'] = usingCloudShare
                    fairsharePolicy[siteid]['groupList']  = []
                    fairsharePolicy[siteid]['typeList']   = {}
                    fairsharePolicy[siteid]['idList']     = []
                    fairsharePolicy[siteid]['groupListWithPrio']  = []
                    fairsharePolicy[siteid]['typeListWithPrio']   = {}
                    fairsharePolicy[siteid]['idListWithPrio']     = []
                    for tmpDefItem in fairsharePolicy[siteid]['policyList']:
                        # using WG
                        if tmpDefItem['group'] is not None:
                            fairsharePolicy[siteid]['usingGroup'] = True
                        # using PG
                        if tmpDefItem['type'] is not None:
                            fairsharePolicy[siteid]['usingType'] = True
                        # using workqueue_ID
                        if tmpDefItem['id'] is not None:
                            fairsharePolicy[siteid]['usingID'] = True
                        # using prio
                        if tmpDefItem['priority'] is not None:
                            fairsharePolicy[siteid]['usingPrio'] = True
                        # get list of WG and PG with/without priority
                        if tmpDefItem['priority'] is None:
                            # get list of woringGroups
                            if tmpDefItem['group'] is not None and not tmpDefItem['group'] in fairsharePolicy[siteid]['groupList']:
                                fairsharePolicy[siteid]['groupList'].append(tmpDefItem['group'])
                            # get list of processingGroups
                            if tmpDefItem['group'] not in fairsharePolicy[siteid]['typeList']:
                                fairsharePolicy[siteid]['typeList'][tmpDefItem['group']] = []
                            if tmpDefItem['type'] is not None and not tmpDefItem['type'] in fairsharePolicy[siteid]['typeList'][tmpDefItem['group']]:
                                fairsharePolicy[siteid]['typeList'][tmpDefItem['group']].append(tmpDefItem['type'])
                            # get list of workqueue_ids
                            if tmpDefItem['id'] is not None and not tmpDefItem['id'] in fairsharePolicy[siteid]['idList']:
                                fairsharePolicy[siteid]['idList'].append(tmpDefItem['id'])
                        else:
                            # get list of woringGroups
                            if tmpDefItem['group'] is not None and not tmpDefItem['group'] in fairsharePolicy[siteid]['groupListWithPrio']:
                                fairsharePolicy[siteid]['groupListWithPrio'].append(tmpDefItem['group'])
                            # get list of processingGroups
                            if tmpDefItem['group'] not in fairsharePolicy[siteid]['typeListWithPrio']:
                                fairsharePolicy[siteid]['typeListWithPrio'][tmpDefItem['group']] = []
                            if tmpDefItem['type'] is not None and not tmpDefItem['type'] in fairsharePolicy[siteid]['typeListWithPrio'][tmpDefItem['group']]:
                                fairsharePolicy[siteid]['typeListWithPrio'][tmpDefItem['group']].append(tmpDefItem['type'])
                            # get list of workqueue_ids
                            if tmpDefItem['id'] is not None and not tmpDefItem['id'] in fairsharePolicy[siteid]['idListWithPrio']:
                                fairsharePolicy[siteid]['idListWithPrio'].append(tmpDefItem['id'])
                except Exception:
                    errtype,errvalue = sys.exc_info()[:2]
                    _logger.warning("getFairsharePolicy : wrong definition '%s' for %s : %s %s" % (fairsharePolicy, siteid, errtype, errvalue))
            _logger.debug("getFairsharePolicy -> %s" % str(fairsharePolicy))
            if not getNewMap:
                self.fairsharePolicy = fairsharePolicy
                return
            else:
                return fairsharePolicy
        except Exception:
            errtype,errvalue = sys.exc_info()[:2]
            _logger.error("getFairsharePolicy : %s %s" % (errtype,errvalue))
            # roll back
            self._rollback()
            if not getNewMap:
                return
            else:
                return {}


    # get cloud list
    def getCloudList(self):
        comment = ' /* DBProxy.getCloudList */'
        _logger.debug("getCloudList start")
        try:
            # set autocommit on
            self.conn.begin()
            # select
            sql  = "SELECT name,tier1,tier1SE,relocation,weight,server,status,transtimelo,"
            sql += "transtimehi,waittime,validation,mcshare,countries,fasttrack,nprestage,"
            sql += "pilotowners "
            sql+= "FROM ATLAS_PANDAMETA.cloudconfig"
            self.cur.arraysize = 10000
            self.cur.execute(sql+comment)
            resList = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            ret = {}
            if resList is not None and len(resList) != 0:
                for res in resList:
                    # change None to ''
                    resTmp = []
                    for tmpItem in res:
                        if tmpItem is None:
                            tmpItem = ''
                        resTmp.append(tmpItem)
                    name,tier1,tier1SE,relocation,weight,server,status,transtimelo,transtimehi,\
                        waittime,validation,mcshare,countries,fasttrack,nprestage,pilotowners = resTmp
                    # instantiate CloudSpec
                    tmpC = CloudSpec.CloudSpec()
                    tmpC.name = name
                    tmpC.tier1 = tier1
                    tmpC.tier1SE = re.sub(' ','',tier1SE).split(',')
                    tmpC.relocation  = relocation
                    tmpC.weight      = weight
                    tmpC.server      = server
                    tmpC.status      = status
                    tmpC.transtimelo = transtimelo
                    tmpC.transtimehi = transtimehi
                    tmpC.waittime    = waittime
                    tmpC.validation  = validation
                    tmpC.mcshare     = mcshare
                    tmpC.countries   = countries
                    tmpC.fasttrack   = fasttrack
                    tmpC.nprestage   = nprestage
                    tmpC.pilotowners = pilotowners
                    # append
                    ret[name] = tmpC
            _logger.debug("getCloudList done")
            return ret
        except Exception:
            type, value, traceBack = sys.exc_info()
            _logger.error("getCloudList : %s %s" % (type,value))
            # roll back
            self._rollback()
            return {}


    # check sites with release/cache
    def checkSitesWithRelease(self, sites, releases,caches,
                              cmtConfig=None, onlyCmtConfig=False, cmtConfigPattern=False):
        comment = ' /* DBProxy.checkSitesWithRelease */'
        try:
            relStr = releases
            if releases is not None:
                relStr = releases.replace('\n',' ')
            caStr = caches
            if caches is not None:
                caStr = caches.replace('\n',' ')
            _logger.debug("checkSitesWithRelease(%s,%s,%s,%s,%s)" % (sites,relStr,caStr,cmtConfig,
                                                                     cmtConfigPattern))
            # select
            sql  = "SELECT distinct siteid FROM ATLAS_PANDAMETA.InstalledSW WHERE "
            loopKey2 = None
            loopValues2 = []
            if caches not in ['','NULL',None]:
                loopKey = ':cache'
                loopValues = caches.split('\n')
                sql += "cache=:cache "
                if releases not in ['','NULL',None]:
                    loopKey2 = ':release'
                    loopValues2 = releases.split('\n')
                    sql += "AND `release`=:release "
            elif releases not in ['','NULL',None]:
                loopKey = ':release'
                loopValues = releases.split('\n')
                sql += "`release`=:release AND cache='None' "
            elif onlyCmtConfig:
                loopKey = None
                loopValues = [None]
            else:
                # don't check
                return sites
            checkCMT = False
            if cmtConfig not in ['','NULL',None]:
                if onlyCmtConfig:
                    if not cmtConfigPattern:
                        sql += "cmtConfig=:cmtConfig "
                    else:
                        sql += "cmtConfig LIKE :cmtConfig "
                else:
                    sql += "AND cmtConfig=:cmtConfig "
                checkCMT = True
            sql += "AND siteid IN ("
            # start transaction
            self.conn.begin()
            self.cur.arraysize = 1000
            # loop over all releases/caches
            for loopIdx,loopVal in enumerate(loopValues):
                sqlSite = sql
                varMap = {}
                if loopKey is not None:
                    # remove Atlas-
                    loopVal = re.sub('^Atlas-','',loopVal)
                    varMap[loopKey] = loopVal
                    if loopKey2 is not None:
                        loopVal2 = loopValues2[loopIdx]
                        loopVal2 = re.sub('^Atlas-','',loopVal2)
                        varMap[loopKey2] = loopVal2
                if checkCMT:
                    varMap[':cmtConfig'] = cmtConfig
                tmpRetSites = []
                # loop over sites
                nSites = 10
                iSite = 0
                for siteIndex,site in enumerate(sites):
                    iSite += 1
                    tmpSiteKey = ':siteid%s' % iSite
                    varMap[tmpSiteKey] = site
                    sqlSite += '%s,' % tmpSiteKey
                    if iSite == nSites or (siteIndex+1) == len(sites):
                        iSite = 0
                        # close bracket in SQL
                        sqlSite = sqlSite[:-1]
                        sqlSite += ')'
                        # execute
                        _logger.debug(sqlSite+comment+str(varMap))
                        self.cur.execute(sqlSite+comment, varMap)
                        resList = self.cur.fetchall()
                        # collect candidates
                        if len(resList) > 0:
                            for tmpSite, in resList:
                                # append
                                tmpRetSites.append(tmpSite)
                        # reset
                        sqlSite = sql
                        varMap = {}
                        if loopKey is not None:
                            varMap[loopKey] = loopVal
                            if loopKey2 is not None:
                                varMap[loopKey2] = loopVal2
                        if checkCMT:
                            varMap[':cmtConfig'] = cmtConfig
                # set
                sites = tmpRetSites
                # escape
                if sites == []:
                    break
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            _logger.debug("checkSitesWithRelease -> %s" % sites)
            return sites
        except Exception:
            # roll back
            self._rollback()
            type,value,traceBack = sys.exc_info()
            _logger.error("checkSitesWithRelease : %s %s" % (type,value))
            return []


    # get sites with release/cache in cloud
    def getSitesWithReleaseInCloud(self,cloud,releases,caches,validation):
        comment = ' /* DBProxy.getSitesWithReleaseInCloud */'
        try:
            relStr = releases
            if releases is not None:
                relStr = releases.replace('\n',' ')
            caStr = caches
            if caches is not None:
                caStr = caches.replace('\n',' ')
            _logger.debug("getSitesWithReleaseInCloud(%s,%s,%s,%s)" % (cloud,relStr,caStr,validation))
            # select
            sql  = "SELECT distinct siteid FROM ATLAS_PANDAMETA.InstalledSW WHERE cloud=:cloud AND "
            varMap = {}
            varMap[':cloud'] = cloud
            if caches not in ['','NULL',None]:
                loopKey = ':cache'
                loopValues = caches.split('\n')
                sql += "cache=:cache "
            else:
                loopKey = ':release'
                loopValues = releases.split('\n')
                sql += "`release`=:release AND cache='None' "
            # validation
            if validation:
                sql += "validation=:validation "
                varMap[':validation'] = 'validated'
            # start transaction
            self.conn.begin()
            self.cur.arraysize = 100
            # loop over all releases/caches
            retSites = None
            for loopVal in loopValues:
                # remove Atlas-
                loopVal = re.sub('^Atlas-','',loopVal)
                varMap[loopKey] = loopVal
                # execute
                _logger.debug(sql+comment+str(varMap))
                self.cur.execute(sql+comment, varMap)
                resList = self.cur.fetchall()
                # append
                tmpRetSites = []
                for tmpItem, in resList:
                    if retSites is None or (tmpItem in retSites):
                        tmpRetSites.append(tmpItem)
                # set
                retSites = tmpRetSites
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # append
            retSites = []
            for tmpItem, in resList:
                retSites.append(tmpItem)
            _logger.debug("getSitesWithReleaseInCloud -> %s" % retSites)
            return retSites
        except Exception:
            # roll back
            self._rollback()
            type,value,traceBack = sys.exc_info()
            _logger.error("getSitesWithReleaseInCloud : %s %s" % (type,value))
            return []


    # get list of cache prefix
    def getCachePrefixes(self):
        comment = ' /* DBProxy.getCachePrefixes */'
        try:
            _logger.debug("getCachePrefixes")
            # select
            sql  = "SELECT distinct cache FROM ATLAS_PANDAMETA.installedSW WHERE cache IS NOT NULL"
            # start transaction
            self.conn.begin()
            self.cur.arraysize = 10000
            # execute
            self.cur.execute(sql+comment, {})
            resList = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # append
            tmpList = []
            for tmpItem, in resList:
                match = re.search('^([^-]+)-',tmpItem)
                if match is not None:
                    tmpPrefix = match.group(1)
                    if tmpPrefix not in tmpList:
                        tmpList.append(tmpPrefix)
            _logger.debug("getCachePrefixes -> %s" % tmpList)
            return tmpList
        except Exception:
            # roll back
            self._rollback()
            type,value,traceBack = sys.exc_info()
            _logger.error("getCachePrefixes : %s %s" % (type,value))
            return []


    # get list of cmtConfig
    def getCmtConfigList(self, relaseVer):
        comment = ' /* DBProxy.getCmtConfigList */'
        try:
            methodName = "getCmtConfigList"
            _logger.debug("{0} for {1}".format(methodName,relaseVer))
            # select
            sql  = "SELECT distinct cmtConfig FROM ATLAS_PANDAMETA.installedSW WHERE release=:release"
            # start transaction
            self.conn.begin()
            self.cur.arraysize = 10
            # execute
            varMap = {}
            varMap[':release'] = relaseVer
            self.cur.execute(sql+comment, varMap)
            resList = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # append
            tmpList = []
            for tmpItem, in resList:
                tmpList.append(tmpItem)
            _logger.debug("{0} -> {1}".format(methodName,str(tmpList)))
            return tmpList
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            return []


    # get pilot owners
    def getPilotOwners(self):
        comment = ' /* DBProxy.getPilotOwners */'
        _logger.debug("getPilotOwners")
        try:
            ret = {None:set()}
            # set autocommit on
            self.conn.begin()
            # select
            sql = "SELECT pilotowners FROM ATLAS_PANDAMETA.cloudconfig "
            self.cur.arraysize = 10000
            self.cur.execute(sql+comment)
            resList = self.cur.fetchall()
            for tmpItem, in resList:
                if tmpItem is not None:
                    for tmpOwner in tmpItem.split('|'):
                        if tmpOwner != '':
                            ret[None].add(tmpOwner)

            sql = "SELECT /* use_json_type */ scj.data.siteid, scj.data.dn FROM ATLAS_PANDA.schedconfig_json scj WHERE scj.data.dn IS NOT NULL "
            self.cur.execute(sql+comment)
            resList = self.cur.fetchall()
            for tmpSiteID,tmpItem in resList:
                if tmpItem is not None:
                    tmpItem = tmpItem.strip()
                    for tmpOwner in tmpItem.split('|'):
                        if tmpOwner not in ['','None']:
                            if tmpSiteID not in ret:
                                ret[tmpSiteID] = set()
                            ret[tmpSiteID].add(tmpOwner)
            _logger.debug("getPilotOwners -> %s" % str(ret))
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            return ret
        except Exception:
            # roll back
            self._rollback()
            type,value,traceBack = sys.exc_info()
            _logger.error("getPilotOwners : %s %s" % (type,value))
            return ret


    # get special dispatcher parameters
    def getSpecialDispatchParams(self):
        comment = ' /* DBProxy.getSpecialDispatchParams */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        _logger.debug("{0} start".format(methodName))
        try:
            retMap = {}
            # set autocommit on
            self.conn.begin()
            # select to get the list of authorized users
            allowKey = []
            allowProxy = []
            sql = "SELECT DISTINCT name, gridpref FROM ATLAS_PANDAMETA.users " \
                  "WHERE (status IS NULL OR status<>:ngStatus) AND gridpref IS NOT NULL "
            varMap = {}
            varMap[':ngStatus'] = 'disabled'
            self.cur.arraysize = 100000
            self.cur.execute(sql+comment, varMap)
            resList = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            for compactDN,gridpref in resList:
                # users authorized for proxy retrieval
                if PrioUtil.PERMISSION_PROXY in gridpref:
                    if compactDN not in allowProxy:
                        allowProxy.append(compactDN)
                # users authorized for key-pair retrieval
                if PrioUtil.PERMISSION_KEY in gridpref:
                    if compactDN not in allowKey:
                        allowKey.append(compactDN)
            retMap['allowKey'] = allowKey
            retMap['allowProxy'] = allowProxy
            _logger.debug("{0} got {1} users for key {2} users for proxy".format(methodName,
                                                                                 len(retMap['allowKey']),
                                                                                 len(retMap['allowProxy'])))
            # read key pairs
            keyPair = {}
            try:
                keyFileNames = glob.glob(panda_config.keyDir+'/*')
                for keyName in keyFileNames:
                    tmpF = open(keyName)
                    keyPair[os.path.basename(keyName)] = tmpF.read()
                    tmpF.close()
            except Exception:
                self.dumpErrorMessage(_logger,methodName)
            retMap['keyPair'] = keyPair
            _logger.debug("{0} got {1} key files".format(methodName,len(retMap['keyPair'])))
            _logger.debug("{0} done".format(methodName))
            return retMap
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            return {}

    # extract name from DN
    def cleanUserID(self, id):
        return CoreUtils.clean_user_id(id)

    # extract scope from dataset name
    def extractScope(self,name):
        try:
            if name.lower().startswith('user') or \
                   name.lower().startswith('group'):
                # return None if there are not enough fields
                if len(name.split('.')) < 2:
                    return None
                return name.lower().split('.')[0] + '.' + name.lower().split('.')[1]
            return name.split('.')[0]
        except Exception:
            return None


    # check quota
    def checkQuota(self,dn):
        comment = ' /* DBProxy.checkQuota */'
        _logger.debug("checkQuota %s" % dn)
        try:
            # set autocommit on
            self.conn.begin()
            # select
            name = self.cleanUserID(dn)
            sql = "SELECT cpua1, cpua7, cpua30, quotaa1, quotaa7, quotaa30 FROM ATLAS_PANDAMETA.users WHERE name=:name"
            varMap = {}
            varMap[':name'] = name
            self.cur.arraysize = 10
            self.cur.execute(sql+comment,varMap)
            res = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            weight = 0.0
            if res is not None and len(res) != 0:
                item = res[0]
                # cpu and quota
                cpu1    = item[0]
                cpu7    = item[1]
                cpu30   = item[2]
                if item[3] in [0,None]:
                    quota1 = 0
                else:
                    quota1 = item[3] * 3600
                if item[4] in [0,None]:
                    quota7 = 0
                else:
                    quota7 = item[4] * 3600
                if item[5] in [0,None]:
                    quota30 = 0
                else:
                    quota30 = item[5] * 3600
                # CPU usage
                if cpu1 is None:
                    cpu1 = 0.0
                # weight
                if quota1 > 0:
                    weight = float(cpu1) / float(quota1)
                # not exceeded the limit
                weight = 0.0
                _logger.debug("checkQuota %s Weight:%s Quota:%s CPU:%s" % (dn,weight,quota1,cpu1))
            else:
                _logger.debug("checkQuota cannot found %s" % dn)
            return weight
        except Exception:
            type, value, traceBack = sys.exc_info()
            _logger.error("checkQuota : %s %s" % (type,value))
            # roll back
            self._rollback()
            return 0.0


    # check if super user
    def isSuperUser(self,userName):
        comment = ' /* DBProxy.isSuperUser */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        tmpLog = LogWrapper(_logger,methodName+" <{0}>".format(userName))
        tmpLog.debug("start")
        try:
            isSU = False
            isSG = False
            # start transaction
            self.conn.begin()
            # check gridpref
            name = self.cleanUserID(userName)
            sql = "SELECT gridpref FROM ATLAS_PANDAMETA.users WHERE name=:name"
            varMap = {}
            varMap[':name'] = name
            self.cur.arraysize = 10
            self.cur.execute(sql+comment,varMap)
            res = self.cur.fetchone()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # check if s in gridpref
            if res is not None:
                gridpref, = res
                if gridpref is not None:
                    if PrioUtil.PERMISSION_SUPER_USER in gridpref:
                        isSU = True
                    if PrioUtil.PERMISSION_SUPER_GROUP in gridpref:
                        isSG = True
            tmpLog.debug("done with superUser={0} superGroup={1}".format(isSU, isSG))
            return isSU, isSG
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            return False, False



    # get serialize JobID and status
    def getUserParameter(self,dn,jobID,jobsetID):
        comment = ' /* DBProxy.getUserParameter */'
        _logger.debug("getUserParameter %s JobID=%s JobsetID=%s" % (dn,jobID,jobsetID))
        try:
            # set initial values
            retStatus = True
            if jobsetID == -1:
                # generate new jobsetID
                retJobsetID = jobID
                # new jobID = 1 + new jobsetID
                retJobID = retJobsetID + 1
            elif jobsetID in ['NULL',None,0]:
                # no jobsetID
                retJobsetID = None
                retJobID = jobID
            else:
                # user specified jobsetID
                retJobsetID = jobsetID
                retJobID = jobID
            # set autocommit on
            self.conn.begin()
            # select
            name = self.cleanUserID(dn)
            sql  = "SELECT jobid,status FROM ATLAS_PANDAMETA.users WHERE name=:name "
            sql += "FOR UPDATE "
            sqlAdd  = "INSERT INTO ATLAS_PANDAMETA.users "
            sqlAdd += "(ID,NAME,LASTMOD,FIRSTJOB,LATESTJOB,CACHETIME,NCURRENT,JOBID) "
            sqlAdd += "VALUES(ATLAS_PANDAMETA.USERS_ID_SEQ.nextval,:name,"
            sqlAdd += "CURRENT_DATE,CURRENT_DATE,CURRENT_DATE,CURRENT_DATE,0,1) "
            varMap = {}
            varMap[':name'] = name
            self.cur.execute(sql+comment,varMap)
            self.cur.arraysize = 10
            res = self.cur.fetchall()
            # insert if no record
            if res is None or len(res) == 0:
                try:
                    self.cur.execute(sqlAdd+comment,varMap)
                    retI = self.cur.rowcount
                    _logger.debug("getUserParameter %s inserted new row with %s" % (dn,retI))
                    # emulate DB response
                    res = [[1,'']]
                except Exception:
                    errType,errValue = sys.exc_info()[:2]
                    _logger.error("getUserParameter %s failed to insert new row with %s:%s" % (dn,errType,errValue))
            if res is not None and len(res) != 0:
                item = res[0]
                # JobID in DB
                dbJobID  = item[0]
                # check status
                if item[1] in ['disabled']:
                    retStatus = False
                # use larger JobID
                if dbJobID >= int(retJobID) or (jobsetID == -1 and dbJobID >= int(retJobsetID)):
                    if jobsetID == -1:
                        # generate new jobsetID = 1 + exsiting jobID
                        retJobsetID = dbJobID+1
                        # new jobID = 1 + new jobsetID
                        retJobID = retJobsetID + 1
                    else:
                        # new jobID = 1 + exsiting jobID
                        retJobID = dbJobID+1
                # update DB
                varMap = {}
                varMap[':name'] = name
                varMap[':jobid'] = retJobID
                sql = "UPDATE ATLAS_PANDAMETA.users SET jobid=:jobid WHERE name=:name"
                self.cur.execute(sql+comment,varMap)
                _logger.debug("getUserParameter set JobID=%s for %s" % (retJobID,dn))
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            _logger.debug("getUserParameter %s return JobID=%s JobsetID=%s Status=%s" % (dn,retJobID,retJobsetID,retStatus))
            return retJobID,retJobsetID,retStatus
        except Exception:
            type, value, traceBack = sys.exc_info()
            _logger.error("getUserParameter : %s %s" % (type,value))
            # roll back
            self._rollback()
            return retJobID,retJobsetID,retStatus


    # get JobID for user
    def getJobIdUser(self,dn):
        comment = ' /* DBProxy.getJobIdUser */'
        _logger.debug("getJobIdUser %s" % dn)
        jobID = 0
        try:
            # set autocommit on
            self.conn.begin()
            # select
            name = self.cleanUserID(dn)
            sql = "SELECT jobid FROM ATLAS_PANDAMETA.users WHERE name=:name"
            varMap = {}
            varMap[':name'] = name
            self.cur.arraysize = 10
            self.cur.execute(sql+comment,varMap)
            res = self.cur.fetchone()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            if res is not None:
                jobID, = res
            _logger.debug("getJobIdUser %s -> %s" % (name,jobID))
        except Exception:
            errType,errValue = sys.exc_info()[:2]
            _logger.error("getJobIdUser : %s %s" % (errType,errValue))
            # roll back
            self._rollback()
        return jobID


    # check ban user
    def checkBanUser(self,dn,sourceLabel,jediCheck=False):
        comment = ' /* DBProxy.checkBanUser */'
        try:
            methodName = "checkBanUser"
            # set initial values
            retStatus = True
            name = self.cleanUserID(dn)
            methodName += ' {0}'.format(name)
            _logger.debug("{0} start dn={1} label={2} jediCheck={3}".format(methodName,dn,sourceLabel,jediCheck))
            # set autocommit on
            self.conn.begin()
            # select
            sql = "SELECT status,dn FROM ATLAS_PANDAMETA.users WHERE name=:name"
            varMap = {}
            varMap[':name'] = name
            self.cur.execute(sql+comment,varMap)
            self.cur.arraysize = 10
            res = self.cur.fetchone()
            if res is not None:
                # check status
                tmpStatus,dnInDB = res
                if tmpStatus in ['disabled']:
                    retStatus = False
                elif jediCheck and (dnInDB in ['',None] or dnInDB != dn):
                    # add DN
                    sqlUp  = "UPDATE ATLAS_PANDAMETA.users SET dn=:dn WHERE name=:name "
                    varMap = {}
                    varMap[':name'] = name
                    varMap[':dn']   = dn
                    self.cur.execute(sqlUp+comment,varMap)
                    retI = self.cur.rowcount
                    _logger.debug("{0} update DN with Status={1}".format(methodName,retI))
                    if retI != 1:
                        retStatus = 1
            else:
                # new user
                if jediCheck:
                    name = self.cleanUserID(dn)
                    sqlAdd  = "INSERT INTO ATLAS_PANDAMETA.users "
                    sqlAdd += "(ID,NAME,DN,LASTMOD,FIRSTJOB,LATESTJOB,CACHETIME,NCURRENT,JOBID) "
                    sqlAdd += "VALUES(ATLAS_PANDAMETA.USERS_ID_SEQ.nextval,:name,:dn,"
                    sqlAdd += "CURRENT_DATE,CURRENT_DATE,CURRENT_DATE,CURRENT_DATE,0,1) "
                    varMap = {}
                    varMap[':name'] = name
                    varMap[':dn']   = dn
                    self.cur.execute(sqlAdd+comment,varMap)
                    retI = self.cur.rowcount
                    _logger.debug("{0} inserted new row with Status={1}".format(methodName,retI))
                    if retI != 1:
                        retStatus = 2
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            _logger.debug("{0} done with Status={1}".format(methodName,retStatus))
            return retStatus
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            return retStatus


    # get email address for a user
    def getEmailAddr(self,name,withDN=False,withUpTime=False):
        comment = ' /* DBProxy.getEmailAddr */'
        _logger.debug("get email for %s" % name)
        # sql
        if withDN:
            failedRet = "","",None
            sql = "SELECT email,dn,location FROM ATLAS_PANDAMETA.users WHERE name=:name"
        elif withUpTime:
            failedRet = "",None
            sql = "SELECT email,location FROM ATLAS_PANDAMETA.users WHERE name=:name"
        else:
            failedRet = ""
            sql = "SELECT email FROM ATLAS_PANDAMETA.users WHERE name=:name"
        try:
            # set autocommit on
            self.conn.begin()
            # select
            varMap = {}
            varMap[':name'] = name
            self.cur.execute(sql+comment,varMap)
            self.cur.arraysize = 10
            res = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            if res is not None and len(res) != 0:
                if withDN or withUpTime:
                    if withDN:
                        email,dn,upTime = res[0]
                    else:
                        email,upTime = res[0]
                    # convert time
                    try:
                        upTime = datetime.datetime.strptime(upTime,'%Y-%m-%d %H:%M:%S')
                    except Exception:
                        upTime = None
                    if withDN:
                        return email,dn,upTime
                    else:
                        return email,upTime
                else:
                    return res[0][0]
            # return empty string
            return failedRet
        except Exception:
            type, value, traceBack = sys.exc_info()
            _logger.error("getEmailAddr : %s %s" % (type,value))
            # roll back
            self._rollback()
            return failedRet


    # set email address for a user
    def setEmailAddr(self,userName,emailAddr):
        comment = ' /* DBProxy.setEmailAddr */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        _logger.debug("{0} {1} to {2}".format(methodName,userName,emailAddr))
        # sql
        sql = "UPDATE ATLAS_PANDAMETA.users SET email=:email,location=:uptime WHERE name=:name "
        try:
            # set autocommit on
            self.conn.begin()
            # set
            varMap = {}
            varMap[':name'] = userName
            varMap[':email'] = emailAddr
            varMap[':uptime'] = datetime.datetime.utcnow().strftime('%Y-%m-%d %H:%M:%S')
            self.cur.execute(sql+comment,varMap)
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            return True
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            return False

    # get ban users
    def get_ban_users(self):
        comment = ' /* DBProxy.get_ban_user */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        tmpLog = LogWrapper(_logger, methodName)
        tmpLog.debug('start')
        # sql
        sql = "SELECT name FROM ATLAS_PANDAMETA.users WHERE status=:status "
        try:
            # set autocommit on
            self.conn.begin()
            varMap = {}
            varMap[':status'] = 'disabled'
            self.cur.execute(sql+comment, varMap)
            self.cur.arraysize = 10
            res = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            retVal = {name: False for name, in res}
            tmpLog.debug("got {}".format(retVal))
            return True, retVal
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return False, None

    # get client version
    def getPandaClientVer(self):
        comment = ' /* DBProxy.getPandaClientVer */'
        _logger.debug("getPandaClientVer")
        try:
            # set autocommit on
            self.conn.begin()
            # select
            sql = "SELECT pathena FROM ATLAS_PANDAMETA.pandaconfig WHERE name=:name"
            varMap = {}
            varMap[':name'] = 'current'
            self.cur.execute(sql+comment,varMap)
            self.cur.arraysize = 10
            res = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            retStr = ''
            if res is not None and len(res) != 0:
                retStr = res[0][0]
            _logger.debug("getPandaClientVer -> %s" % retStr)
            return retStr
        except Exception:
            # roll back
            self._rollback()
            type, value, traceBack = sys.exc_info()
            _logger.error("getPandaClientVer : %s %s" % (type,value))
            return ""


    # register proxy key
    def registerProxyKey(self,params):
        comment = ' /* DBProxy.registerProxyKey */'
        _logger.debug("register ProxyKey %s" % str(params))
        try:
            # set autocommit on
            self.conn.begin()
            # construct SQL
            vals = {}
            sql0 = 'INSERT INTO ATLAS_PANDAMETA.proxykey (id,'
            sql1 = 'VALUES (ATLAS_PANDAMETA.PROXYKEY_ID_SEQ.nextval,'

            for key in params:
                val = params[key]
                sql0 += '%s,'  % key
                sql1 += ':%s,' % key
                vals[':%s' % key] = val
            sql0 = sql0[:-1]
            sql1 = sql1[:-1]
            sql = sql0 + ') ' + sql1 + ') '
            # insert
            self.cur.execute(sql+comment,vals)
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # return True
            return True
        except Exception:
            type, value, traceBack = sys.exc_info()
            _logger.error("registerProxyKey : %s %s" % (type,value))
            # roll back
            self._rollback()
            return ""


    # get proxy key
    def getProxyKey(self,dn):
        comment = ' /* DBProxy.getProxyKey */'
        _logger.debug("get ProxyKey %s" % dn)
        try:
            # set autocommit on
            self.conn.begin()
            # construct SQL
            sql = 'SELECT credname,expires,origin,myproxy FROM ATLAS_PANDAMETA.proxykey WHERE dn=:dn ORDER BY expires DESC'
            varMap = {}
            varMap[':dn'] = dn
            # select
            self.cur.execute(sql+comment,varMap)
            res = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # return
            retMap = {}
            if res is not None and len(res) != 0:
                credname,expires,origin,myproxy = res[0]
                retMap['credname'] = credname
                retMap['expires']  = expires
                retMap['origin']   = origin
                retMap['myproxy']  = myproxy
            _logger.debug(retMap)
            return retMap
        except Exception:
            type, value, traceBack = sys.exc_info()
            _logger.error("getProxyKey : %s %s" % (type,value))
            # roll back
            self._rollback()
            return {}


    # check site access
    def checkSiteAccess(self,siteid,longDN):
        comment = ' /* DBProxy.checkSiteAccess */'
        _logger.debug("checkSiteAccess %s:%s" % (siteid,longDN))
        try:
            # use compact DN
            dn = self.cleanUserID(longDN)
            # construct SQL
            sql = 'SELECT poffset,rights,status,workingGroups FROM ATLAS_PANDAMETA.siteaccess WHERE dn=:dn AND pandasite=:pandasite'
            varMap = {}
            varMap[':dn'] = dn
            varMap[':pandasite'] = siteid
            # set autocommit on
            self.conn.begin()
            # select
            self.cur.execute(sql+comment,varMap)
            self.cur.arraysize = 10
            res = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # return
            retMap = {}
            if res is not None and len(res) != 0:
                poffset,rights,status,workingGroups = res[0]
                retMap['poffset'] = poffset
                retMap['rights']  = rights
                retMap['status']  = status
                if workingGroups in ['',None]:
                    workingGroups = []
                else:
                    workingGroups = workingGroups.split(',')
                retMap['workingGroups'] = workingGroups
            _logger.debug(retMap)
            return retMap
        except Exception:
            type, value, traceBack = sys.exc_info()
            _logger.error("checkSiteAccess : %s %s" % (type,value))
            # roll back
            self._rollback()
            return {}


    # add account to siteaccess
    def addSiteAccess(self,siteID,longDN):
        comment = ' /* DBProxy.addSiteAccess */'
        _logger.debug("addSiteAccess : %s %s" % (siteID,longDN))
        try:
            # use compact DN
            dn = self.cleanUserID(longDN)
            # set autocommit on
            self.conn.begin()
            # select
            sql = 'SELECT status FROM ATLAS_PANDAMETA.siteaccess WHERE dn=:dn AND pandasite=:pandasite'
            varMap = {}
            varMap[':dn'] = dn
            varMap[':pandasite'] = siteID
            self.cur.execute(sql+comment,varMap)
            self.cur.arraysize = 10
            res = self.cur.fetchone()
            if res is not None:
                _logger.debug("account already exists with status=%s" % res[0])
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                return res[0]
            # add
            sql = 'INSERT INTO ATLAS_PANDAMETA.siteaccess (id,dn,pandasite,status,created) VALUES (ATLAS_PANDAMETA.SITEACCESS_ID_SEQ.nextval,:dn,:pandasite,:status,CURRENT_DATE)'

            varMap = {}
            varMap[':dn'] = dn
            varMap[':pandasite'] = siteID
            varMap[':status'] = 'requested'
            self.cur.execute(sql+comment,varMap)
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            _logger.debug("account was added")
            return 0
        except Exception:
            # roll back
            self._rollback()
            type, value, traceBack = sys.exc_info()
            _logger.error("addSiteAccess : %s %s" % (type,value))
            # return None
            return -1


    # list site access
    def listSiteAccess(self,siteid=None,dn=None,longFormat=False):
        comment = ' /* DBProxy.listSiteAccess */'
        _logger.debug("listSiteAccess %s:%s" % (siteid,dn))
        try:
            if siteid is None and dn is None:
                return []
            longAttributes = 'status,poffset,rights,workingGroups,created'
            # set autocommit on
            self.conn.begin()
            # construct SQL
            if siteid is not None:
                varMap = {':pandasite':siteid}
                if not longFormat:
                    sql = 'SELECT dn,status FROM ATLAS_PANDAMETA.siteaccess WHERE pandasite=:pandasite ORDER BY dn'
                else:
                    sql  = 'SELECT dn,%s FROM ATLAS_PANDAMETA.siteaccess ' % longAttributes
                    sql += 'WHERE pandasite=:pandasite ORDER BY dn'
            else:
                shortDN = self.cleanUserID(dn)
                varMap = {':dn':shortDN}
                if not longFormat:
                    sql = 'SELECT pandasite,status FROM ATLAS_PANDAMETA.siteaccess WHERE dn=:dn ORDER BY pandasite'
                else:
                    sql  = 'SELECT pandasite,%s FROM ATLAS_PANDAMETA.siteaccess ' % longAttributes
                    sql += 'WHERE dn=:dn ORDER BY pandasite'
            # select
            self.cur.execute(sql+comment,varMap)
            self.cur.arraysize = 1000
            res = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # return
            ret = []
            if res is not None and len(res) != 0:
                for tmpRes in res:
                    if not longFormat:
                        ret.append(tmpRes)
                    else:
                        # create map for long format
                        tmpRetMap = {}
                        # use first value as a primary key
                        tmpRetMap['primKey'] = tmpRes[0]
                        idxVal = 1
                        for tmpKey in longAttributes.split(','):
                            tmpRetMap[tmpKey] = tmpRes[idxVal]
                            idxVal += 1
                        ret.append(tmpRetMap)
            _logger.debug(ret)
            return ret
        except Exception:
            # roll back
            self._rollback()
            type, value, traceBack = sys.exc_info()
            _logger.error("listSiteAccess : %s %s" % (type,value))
            return []


    # update site access
    def updateSiteAccess(self,method,siteid,requesterDN,userName,attrValue):
        comment = ' /* DBProxy.updateSiteAccess */'
        _logger.debug("updateSiteAccess %s:%s:%s:%s:%s" % (method, siteid, requesterDN, userName, attrValue))
        try:
            # set autocommit on
            self.conn.begin()
            # check existence
            varMap = {}
            varMap[':pandasite'] = siteid
            varMap[':dn'] = userName
            sql = 'SELECT count(*) FROM ATLAS_PANDAMETA.siteaccess WHERE pandasite=:pandasite AND dn=:dn'
            self.cur.execute(sql+comment,varMap)
            self.cur.arraysize = 10
            res = self.cur.fetchall()
            if res is None or res[0][0] == 0:
                _logger.error("updateSiteAccess : No request for %s" % varMap)
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                # return
                return 'No request for %s:%s' % (siteid,userName)
            # get cloud
            varMap = {':pandasite':siteid}
            sql = 'SELECT /* use_json_type */ scj.data.cloud, scj.data.dn FROM ATLAS_PANDA.schedconfig_json scj WHERE scj.panda_queue=:pandasite AND rownum<=1'
            self.cur.execute(sql+comment,varMap)
            res = self.cur.fetchall()
            if res is None or len(res) == 0:
                _logger.error("updateSiteAccess : No cloud in schedconfig for %s" % siteid)
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                # return
                return "No cloud in schedconfig for %s" % siteid
            cloud = res[0][0]
            siteContact = res[0][1]
            # get cloud responsible
            varMap = {':cloud':cloud}
            sql = 'SELECT dn FROM ATLAS_PANDAMETA.cloudconfig WHERE name=:cloud'
            self.cur.execute(sql+comment,varMap)
            res = self.cur.fetchall()
            if res is None or len(res) == 0:
                _logger.error("updateSiteAccess : No contact in cloudconfig for %s" % cloud)
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                # return
                return "No contact in cloudconfig for %s" % cloud
            contactNames = res[0][0]
            if contactNames in [None,'']:
                contactNames = []
            else:
                contactNames = contactNames.split(',')
            # get site responsible
            if siteContact not in [None,'']:
                contactNames += siteContact.split(',')
            # check privilege
            if self.cleanUserID(requesterDN) not in contactNames:
                _logger.error("updateSiteAccess : %s is not one of contacts %s" % (requesterDN,str(contactNames)))
                # return
                return "Insufficient privilege"
            # update
            varMap = {}
            varMap[':pandasite'] = siteid
            varMap[':dn'] = userName
            if method in ['approve','reject']:
                # update status
                sql = 'UPDATE ATLAS_PANDAMETA.siteaccess SET status=:newStatus WHERE pandasite=:pandasite AND dn=:dn'
                if method == 'approve':
                    varMap[':newStatus'] = 'tobeapproved'
                else:
                    varMap[':newStatus'] = 'toberejected'
            elif method == 'delete':
                # delete
                sql = 'DELETE FROM ATLAS_PANDAMETA.siteaccess WHERE pandasite=:pandasite AND dn=:dn'
            elif method == 'set':
                # check value
                if re.search('^[a-z,A-Z]+:[a-z,A-Z,0-9,\,_\-]+$',attrValue) is None:
                    errStr = "Invalid argument for set : %s. Must be key:value" % attrValue
                    _logger.error("updateSiteAccess : %s" % errStr)
                    # retrun
                    return errStr
                # decompose to key and value
                tmpKey = attrValue.split(':')[0].lower()
                tmpVal = attrValue.split(':')[-1]
                # check key
                changeableKeys = ['poffset','workinggroups','rights']
                if tmpKey not in changeableKeys:
                    errStr = "%s cannot be set. Only %s are allowed" % (tmpKey,str(changeableKeys))
                    _logger.error("updateSiteAccess : %s" % errStr)
                    # retrun
                    return errStr
                # set value map
                varMap[':%s' % tmpKey] = tmpVal
                sql = 'UPDATE ATLAS_PANDAMETA.siteaccess SET %s=:%s WHERE pandasite=:pandasite AND dn=:dn' % (tmpKey,tmpKey)
            else:
                _logger.error("updateSiteAccess : Unknown method '%s'" % method)
                # return
                return "Unknown method '%s'" % method
            # execute
            _logger.debug(sql+comment+str(varMap))
            self.cur.execute(sql+comment, varMap)
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            _logger.debug("updateSiteAccess : completed")
            return True
        except Exception:
            # roll back
            self._rollback()
            type, value, traceBack = sys.exc_info()
            _logger.error("updateSiteAccess : %s %s" % (type,value))
            return 'DB error %s %s' % (type,value)


    # get list of archived tables
    def getArchiveTables(self):
        # return
        return ['ATLAS_PANDAARCH.jobsArchived']


    # get JobIDs in a time range
    def getJobIDsInTimeRangeLog(self,dn,timeRange,retJobIDs):
        comment = ' /* DBProxy.getJobIDsInTimeRangeLog */'
        _logger.debug("getJobIDsInTimeRangeLog : %s %s" % (dn,timeRange.strftime('%Y-%m-%d %H:%M:%S')))
        try:
            # get compact DN
            compactDN = self.cleanUserID(dn)
            if compactDN in ['','NULL',None]:
                compactDN = dn
            # get list of archived tables
            tables = self.getArchiveTables()
            # select
            for table in tables:
                # make sql
                sql  = "SELECT /*+ NO_INDEX(tab JOBS_MODTIME_IDX) INDEX_COMBINE(tab JOBS_PRODSOURCELABEL_IDX JOBS_PRODUSERNAME_IDX) */ "
                sql += "jobDefinitionID FROM %s tab " % table
                sql += "WHERE prodUserName=:prodUserName AND modificationTime>:modificationTime "
                sql += "AND prodSourceLabel=:prodSourceLabel AND lockedBy<>:ngLock GROUP BY jobDefinitionID"
                varMap = {}
                varMap[':prodUserName'] = compactDN
                varMap[':prodSourceLabel'] = 'user'
                varMap[':ngLock'] = 'jedi'
                varMap[':modificationTime'] = timeRange
                # start transaction
                self.conn.begin()
                # select
                self.cur.arraysize = 10000
                _logger.debug(sql+comment+str(varMap))
                self.cur.execute(sql+comment, varMap)
                resList = self.cur.fetchall()
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                # append
                for tmpID, in resList:
                    if tmpID not in retJobIDs:
                        retJobIDs.append(tmpID)
            _logger.debug("getJobIDsInTimeRangeLog : %s" % str(retJobIDs))
            return retJobIDs
        except Exception:
            # roll back
            self._rollback()
            type, value, traceBack = sys.exc_info()
            _logger.error("getJobIDsInTimeRangeLog : %s %s" % (type,value))
            # return empty list
            return retJobIDs


    # get PandaIDs for a JobID
    def getPandIDsWithJobIDLog(self,dn,jobID,idStatus,nJobs,buildJobID=None):
        comment = ' /* Proxy.getPandIDsWithJobIDLog */'
        _logger.debug("getPandIDsWithJobIDLog : %s %s" % (dn,jobID))
        try:
            # get compact DN
            compactDN = self.cleanUserID(dn)
            if compactDN in ['','NULL',None]:
                compactDN = dn
            # get list of archived tables
            tables = self.getArchiveTables()
            # select
            for table in tables:
                # skip if all jobs have already been gotten
                if nJobs > 0 and len(idStatus) >= nJobs:
                    continue
                # make sql
                sql  = "SELECT /*+ NO_INDEX(tab JOBS_MODTIME_IDX) INDEX_COMBINE(tab JOBS_PRODUSERNAME_IDX JOBS_JOBDEFID_IDX) */ "
                sql += "PandaID,jobStatus,commandToPilot,prodSourceLabel,taskBufferErrorCode FROM %s tab " % table
                sql += "WHERE prodUserName=:prodUserName AND jobDefinitionID=:jobDefinitionID "
                sql += "AND prodSourceLabel IN (:prodSourceLabel1,:prodSourceLabel2) AND modificationTime>(CURRENT_DATE-30) "
                varMap = {}
                varMap[':prodUserName'] = compactDN
                varMap[':jobDefinitionID'] = jobID
                varMap[':prodSourceLabel1'] = 'user'
                varMap[':prodSourceLabel2'] = 'panda'
                # start transaction
                self.conn.begin()
                # select
                self.cur.arraysize = 10000
                # select
                _logger.debug(sql+comment+str(varMap))
                self.cur.execute(sql+comment, varMap)
                resList = self.cur.fetchall()
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                # append
                for tmpID,tmpStatus,tmpCommand,tmpProdSourceLabel,tmpTaskBufferErrorCode in resList:
                    # ignore jobs retried by pilot since they have new PandaIDs with the same jobsetID/jobdefID
                    if tmpTaskBufferErrorCode in [ErrorCode.EC_PilotRetried]:
                        continue
                    # ignore old buildJob which was replaced by rebrokerage
                    if tmpProdSourceLabel == 'panda':
                        if buildJobID is None:
                            # first buildJob
                            buildJobID = tmpID
                        elif buildJobID >= tmpID:
                            # don't append old one
                            continue
                        else:
                            # delete old one
                            del idStatus[buildJobID]
                            buildJobID = tmpID
                    # append
                    if tmpID not in idStatus:
                        idStatus[tmpID] = (tmpStatus,tmpCommand)
            _logger.debug("getPandIDsWithJobIDLog : %s" % str(idStatus))
            return idStatus
        except Exception:
            # roll back
            self._rollback()
            type, value, traceBack = sys.exc_info()
            _logger.error("getPandIDsWithJobIDLog : %s %s" % (type,value))
            # return empty list
            return {}


    # get PandaIDs for a JobsetID or JobdefID in jobsArchived
    def getPandIDsWithIdInArch(self,prodUserName,id,isJobset):
        comment = ' /* Proxy.getPandIDsWithIdInArch */'
        _logger.debug("getPandIDsWithIdInArch : %s %s %s" % (prodUserName,id,isJobset))
        try:
            # make sql
            if isJobset:
                sql  = "SELECT /*+ NO_INDEX(tab JOBS_MODTIME_IDX) INDEX_COMBINE(tab JOBS_PRODUSERNAME_IDX JOBS_JOBSETID_IDX) */ "
            else:
                sql  = "SELECT /*+ NO_INDEX(tab JOBS_MODTIME_IDX) INDEX_COMBINE(tab JOBS_PRODUSERNAME_IDX JOBS_JOBDEFID_IDX) */ "
            sql += "PandaID FROM ATLAS_PANDAARCH.jobsArchived tab "
            sql += "WHERE prodUserName=:prodUserName "
            sql += "AND prodSourceLabel IN (:prodSourceLabel1,:prodSourceLabel2) AND modificationTime>(CURRENT_DATE-30) "
            if isJobset:
                sql += "AND jobsetID=:jobID "
            else:
                sql += "AND jobDefinitionID=:jobID "
            varMap = {}
            varMap[':prodUserName'] = prodUserName
            varMap[':jobID'] = id
            varMap[':prodSourceLabel1'] = 'user'
            varMap[':prodSourceLabel2'] = 'panda'
            # start transaction
            self.conn.begin()
            # select
            self.cur.arraysize = 1000000
            # select
            _logger.debug(sql+comment+str(varMap))
            self.cur.execute(sql+comment, varMap)
            resList = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # append
            pandaIDs = []
            for tmpID, in resList:
                pandaIDs.append(tmpID)
            _logger.debug("getPandIDsWithIdInArch : %s %s -> %s" % (prodUserName,id,str(pandaIDs)))
            return pandaIDs
        except Exception:
            # roll back
            self._rollback()
            errType,errValue = sys.exc_info()[:2]
            _logger.error("getPandIDsWithIdInArch : %s %s" % (errType,errValue))
            # return empty list
            return []


    # peek at job
    def peekJobLog(self,pandaID,days=None):
        comment = ' /* DBProxy.peekJobLog */'
        _logger.debug("peekJobLog : %s days=%s" % (pandaID,days))
        # return None for NULL PandaID
        if pandaID in ['NULL','','None',None]:
            return None
        sql1_0 = "SELECT %s FROM %s "
        sql1_1 = "WHERE PandaID=:PandaID AND modificationTime>(CURRENT_DATE-:days) "
        # select
        varMap = {}
        varMap[':PandaID'] = pandaID
        if days is None:
            days = 30
        varMap[':days'] = days
        nTry=1
        for iTry in range(nTry):
            try:
                # get list of archived tables
                tables = self.getArchiveTables()
                # select
                for table in tables:
                    # start transaction
                    self.conn.begin()
                    # select
                    sql = sql1_0 % (JobSpec.columnNames(),table) + sql1_1
                    self.cur.arraysize = 10
                    self.cur.execute(sql+comment, varMap)
                    res = self.cur.fetchall()
                    # commit
                    if not self._commit():
                        raise RuntimeError('Commit error')
                    if len(res) != 0:
                        # Job
                        job = JobSpec()
                        job.pack(res[0])
                        # Files
                        # start transaction
                        self.conn.begin()
                        # select
                        fileTableName = re.sub('jobsArchived','filesTable_ARCH',table)
                        sqlFile = "SELECT /*+ INDEX(tab FILES_ARCH_PANDAID_IDX)*/ %s " % FileSpec.columnNames()
                        sqlFile+= "FROM %s tab " % fileTableName
                        # put constraint on modificationTime to avoid full table scan
                        sqlFile+= "WHERE PandaID=:PandaID AND modificationTime>(CURRENT_DATE-:days)"
                        self.cur.arraysize = 10000
                        self.cur.execute(sqlFile+comment, varMap)
                        resFs = self.cur.fetchall()
                        # metadata
                        varMap = {}
                        varMap[':PandaID'] = job.PandaID
                        job.metadata = None
                        metaTableName = re.sub('jobsArchived','metaTable_ARCH',table)
                        sqlMeta = "SELECT metaData FROM %s WHERE PandaID=:PandaID" % metaTableName
                        self.cur.execute(sqlMeta+comment, varMap)
                        for clobMeta, in self.cur:
                            if clobMeta is not None:
                                try:
                                    job.metadata = clobMeta.read()
                                except AttributeError:
                                    job.metadata = str(clobMeta)
                            break
                        # job parameters
                        job.jobParameters = None
                        jobParamTableName = re.sub('jobsArchived','jobParamsTable_ARCH',table)
                        sqlJobP = "SELECT jobParameters FROM %s WHERE PandaID=:PandaID" % jobParamTableName
                        varMap = {}
                        varMap[':PandaID'] = job.PandaID
                        self.cur.execute(sqlJobP+comment, varMap)
                        for clobJobP, in self.cur:
                            if clobJobP is not None:
                                try:
                                    job.jobParameters = clobJobP.read()
                                except AttributeError:
                                    job.jobParameters = str(clobJobP)
                            break
                        # commit
                        if not self._commit():
                            raise RuntimeError('Commit error')
                        # set files
                        for resF in resFs:
                            file = FileSpec()
                            file.pack(resF)
                            # remove redundant white spaces
                            try:
                                file.md5sum = file.md5sum.strip()
                            except Exception:
                                pass
                            try:
                                file.checksum = file.checksum.strip()
                            except Exception:
                                pass
                            job.addFile(file)
                        return job
                _logger.debug("peekJobLog() : PandaID %s not found" % pandaID)
                return None
            except Exception:
                # roll back
                self._rollback()
                if iTry+1 < nTry:
                    _logger.error("peekJobLog : %s" % pandaID)
                    time.sleep(random.randint(10,20))
                    continue
                type, value, traceBack = sys.exc_info()
                _logger.error("peekJobLog : %s %s" % (type,value))
                # return None
                return None


    # get user subscriptions
    def getUserSubscriptions(self,datasetName,timeRange):
        comment = ' /* DBProxy.getUserSubscriptions */'
        _logger.debug("getUserSubscriptions(%s,%s)" % (datasetName,timeRange))
        sql0  = "SELECT site FROM ATLAS_PANDAMETA.UserSubs "
        sql0 += "WHERE datasetName=:datasetName and modificationDate>CURRENT_DATE-:timeRange"
        varMap = {}
        varMap[':datasetName'] = datasetName
        varMap[':timeRange']   = timeRange
        try:
            # start transaction
            self.conn.begin()
            # select
            self.cur.execute(sql0+comment, varMap)
            resSs = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            retList = []
            for tmpSite, in resSs:
                retList.append(tmpSite)
            return retList
        except Exception:
            # roll back
            self._rollback()
            errType,errValue = sys.exc_info()[:2]
            _logger.error("getUserSubscriptions : %s %s" % (errType,errValue))
            return []


    # get the number of user subscriptions
    def getNumUserSubscriptions(self):
        comment = ' /* DBProxy.getNumUserSubscriptions */'
        _logger.debug("getNumUserSubscriptions")
        sql0  = "SELECT site,COUNT(*) FROM ATLAS_PANDAMETA.UserSubs "
        sql0 += "WHERE creationDate>CURRENT_DATE-2 GROUP BY site"
        try:
            # start transaction
            self.conn.begin()
            # select
            self.cur.execute(sql0+comment,{})
            resSs = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            retList = {}
            for tmpSite,countNum in resSs:
                retList[tmpSite] = countNum
            return retList
        except Exception:
            # roll back
            self._rollback()
            errType,errValue = sys.exc_info()[:2]
            _logger.error("getNumUserSubscriptions : %s %s" % (errType,errValue))
            return []


    # add user subscriptions
    def addUserSubscription(self,datasetName,dq2IDs):
        comment = ' /* DBProxy.addUserSubscription */'
        _logger.debug("addUserSubscription(%s,%s)" % (datasetName,dq2IDs))
        sql0  = "INSERT INTO ATLAS_PANDAMETA.UserSubs "
        sql0 += "(datasetName,site,creationDate,modificationDate,nUsed) "
        sql0 += "VALUES (:datasetName,:site,CURRENT_DATE,CURRENT_DATE,:nUsed)"
        try:
            # start transaction
            self.conn.begin()
            for site in dq2IDs:
                varMap = {}
                varMap[':datasetName'] = datasetName
                varMap[':site']        = site
                varMap[':nUsed']       = 0
                # insert
                self.cur.execute(sql0+comment, varMap)
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            return True
        except Exception:
            # roll back
            self._rollback()
            errType,errValue = sys.exc_info()[:2]
            _logger.error("addUserSubscription : %s %s" % (errType,errValue))
            return False


    # increment counter for subscription
    def incrementUsedCounterSubscription(self,datasetName):
        comment = ' /* DBProxy.incrementUsedCounterSubscription */'
        _logger.debug("incrementUsedCounterSubscription(%s)" % datasetName)
        sql0  = "UPDATE ATLAS_PANDAMETA.UserSubs SET nUsed=nUsed+1 "
        sql0 += "WHERE datasetName=:datasetName AND nUsed IS NOT NULL"
        sqlU  = "SELECT MAX(nUsed) FROM ATLAS_PANDAMETA.UserSubs "
        sqlU += "WHERE datasetName=:datasetName"
        try:
            # start transaction
            self.conn.begin()
            varMap = {}
            varMap[':datasetName'] = datasetName
            # update
            self.cur.execute(sql0+comment,varMap)
            # get nUsed
            nUsed = 0
            retU = self.cur.rowcount
            if retU > 0:
                # get nUsed
                self.cur.execute(sqlU+comment,varMap)
                self.cur.arraysize = 10
                res = self.cur.fetchone()
                if res is not None:
                    nUsed = res[0]
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            return nUsed
        except Exception:
            # roll back
            self._rollback()
            errType,errValue = sys.exc_info()[:2]
            _logger.error("incrementUsedCounterSubscription : %s %s" % (errType,errValue))
            return -1


    # get active datasets
    def getActiveDatasets(self,computingSite,prodSourceLabel):
        comment = ' /* DBProxy.getActiveDatasets */'
        _logger.debug("getActiveDatasets(%s,%s)" % (computingSite,prodSourceLabel))
        varMap = {}
        varMap[':computingSite']   = computingSite
        varMap[':jobStatus1']      = 'assigned'
        varMap[':jobStatus2']      = 'activated'
        varMap[':jobStatus3']      = 'waiting'
        varMap[':prodSourceLabel'] = prodSourceLabel
        try:
            retList = []
            for table in ['jobsActive4','jobsDefined4','jobsWaiting4']:
                if table == 'jobsActive4':
                    sql0  = "SELECT distinct prodDBlock FROM ATLAS_PANDA.%s " % table
                else:
                    sql0  = "SELECT distinct prodDBlock FROM ATLAS_PANDA.%s " % table
                sql0 += "WHERE computingSite=:computingSite AND jobStatus IN (:jobStatus1,:jobStatus2,:jobStatus3) "
                sql0 += "AND prodSourceLabel=:prodSourceLabel"
                # start transaction
                self.conn.begin()
                # select
                self.cur.execute(sql0+comment, varMap)
                resSs = self.cur.fetchall()
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                # append
                for prodDBlock, in resSs:
                    if prodDBlock not in retList:
                        retList.append(prodDBlock)
            # make string
            retStr = ''
            for tmpItem in retList:
                retStr += '%s,' % tmpItem
            retStr = retStr[:-1]
            return retStr
        except Exception:
            # roll back
            self._rollback()
            errType,errValue = sys.exc_info()[:2]
            _logger.error("getActiveDatasets : %s %s" % (errType,errValue))
            return ""


    # check status of all sub datasets to trigger Notifier
    def checkDatasetStatusForNotifier(self,jobsetID,jobDefinitionID,prodUserName):
        comment = ' /* DBProxy.checkDatasetStatusForNotifier */'
        _logger.debug("checkDatasetStatusForNotifier(%s,%s,%s)" % (jobsetID,jobDefinitionID,prodUserName))
        try:
            # get PandaIDs to get all associated destinationDBlocks
            varMap = {}
            varMap[':jobsetID']     = jobsetID
            varMap[':prodUserName'] = prodUserName
            sql = "SELECT MAX(PandaID),jobDefinitionID FROM %s WHERE prodUserName=:prodUserName AND jobsetID=:jobsetID GROUP BY jobDefinitionID"
            pandaIDs = {}
            for table in ['ATLAS_PANDA.jobsArchived4','ATLAS_PANDA.jobsActive4','ATLAS_PANDA.jobsDefined4','ATLAS_PANDA.jobsWaiting4']:
                # start transaction
                self.conn.begin()
                # select
                self.cur.arraysize = 1000
                self.cur.execute((sql % table)+comment, varMap)
                resSs = self.cur.fetchall()
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                # get PandaIDs
                for tmpPandaID,tmpJobDefID in resSs:
                    if tmpJobDefID not in pandaIDs or tmpPandaID > pandaIDs[tmpJobDefID]:
                        pandaIDs[tmpJobDefID] = tmpPandaID
            # get all destinationDBlocks
            varMap = {}
            varMap[':type1'] = 'log'
            varMap[':type2'] = 'output'
            sql = 'SELECT DISTINCT destinationDBlock FROM ATLAS_PANDA.filesTable4 WHERE PandaID=:PandaID AND type IN (:type1,:type2)'
            datasetMap = {}
            # start transaction
            self.conn.begin()
            self.cur.arraysize = 1000
            for tmpJobDefID in pandaIDs:
                tmpPandaID = pandaIDs[tmpJobDefID]
                varMap[':PandaID'] = tmpPandaID
                # select
                self.cur.execute(sql+comment, varMap)
                resSs = self.cur.fetchall()
                # get destinationDBlock
                for tmpDestDBlock, in resSs:
                    if tmpJobDefID not in datasetMap:
                        datasetMap[tmpJobDefID] = []
                    if tmpDestDBlock not in datasetMap[tmpJobDefID]:
                        datasetMap[tmpJobDefID].append(tmpDestDBlock)
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # check dataset status
            allClosed = True
            retInfo = {}
            latestUpdate   = None
            latestJobDefID = None
            varMap = {}
            varMap[':type1'] = 'log'
            varMap[':type2'] = 'output'
            sql = 'SELECT status,modificationDate FROM ATLAS_PANDA.Datasets WHERE name=:name AND type IN (:type1,:type2)'
            sqlJ =  "SELECT MAX(modificationTime) FROM ATLAS_PANDA.jobsArchived4 "
            sqlJ += "WHERE prodUserName=:prodUserName AND jobDefinitionID=:jobDefinitionID"
            # start transaction
            self.conn.begin()
            self.cur.arraysize = 1000
            for tmpJobDefID in datasetMap:
                tmpDatasets = datasetMap[tmpJobDefID]
                retInfo[tmpJobDefID] = []
                for tmpDataset in tmpDatasets:
                    if tmpDataset not in retInfo[tmpJobDefID]:
                        retInfo[tmpJobDefID].append(tmpDataset)
                    varMap[':name'] = tmpDataset
                    # select
                    self.cur.execute(sql+comment, varMap)
                    resSs = self.cur.fetchall()
                    # check status and mod time
                    for tmpStatus,tmpModificationDate in resSs:
                        _logger.debug("checkDatasetStatusForNotifier(%s,%s) %s has %s with %s at %s" % \
                                      (jobsetID,jobDefinitionID,tmpJobDefID,tmpDataset,tmpStatus,tmpModificationDate))
                        if tmpStatus not in ['closed','tobeclosed','completed']:
                            # some datasets are still active
                            allClosed = False
                            _logger.debug("checkDatasetStatusForNotifier(%s,%s) wait due to %s %s %s" % \
                                          (jobsetID,jobDefinitionID,tmpJobDefID,tmpDataset,tmpStatus))
                            break
                        elif tmpStatus == 'tobeclosed':
                            # select latest modificationTime in job table
                            varMapJ = {}
                            varMapJ[':prodUserName'] = prodUserName
                            varMapJ[':jobDefinitionID'] = tmpJobDefID
                            self.cur.execute(sqlJ+comment, varMapJ)
                            resJ = self.cur.fetchone()
                            if resJ is None:
                                # error
                                allClosed = False
                                _logger.error("checkDatasetStatusForNotifier(%s,%s) %s cannot find job" % \
                                              (jobsetID,jobDefinitionID,tmpJobDefID))
                                break
                            tmpModificationTime, = resJ
                            _logger.debug("checkDatasetStatusForNotifier(%s,%s) %s modtime:%s" % \
                                          (jobsetID,jobDefinitionID,tmpJobDefID,tmpModificationTime))
                            if latestUpdate is None or latestUpdate < tmpModificationTime:
                                # use the latest updated jobDefID
                                latestUpdate   = tmpModificationTime
                                latestJobDefID = tmpJobDefID
                            elif latestUpdate == tmpModificationTime and latestJobDefID < tmpJobDefID:
                                # use larger jobDefID when datasets are closed at the same time
                                latestJobDefID = tmpJobDefID
                    # escape
                    if not allClosed:
                        break
                # escape
                if not allClosed:
                    break
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            _logger.debug("checkDatasetStatusForNotifier(%s,%s) -> all:%s %s latest:%s" % \
                          (jobsetID,jobDefinitionID,allClosed,latestJobDefID,
                           jobDefinitionID == latestJobDefID))
            # return
            if not allClosed or jobDefinitionID != latestJobDefID:
                return False,{}
            return True,retInfo
        except Exception:
            # roll back
            self._rollback()
            errType,errValue = sys.exc_info()[:2]
            _logger.error("checkDatasetStatusForNotifier : %s %s" % (errType,errValue))
            return False,{}


    # get MoU share for T2 PD2P
    def getMouShareForT2PD2P(self):
        comment = ' /* DBProxy.getMouShareForT2PD2P */'
        _logger.debug("getMouShareForT2PD2P start")
        sqlG  = "SELECT gid,ntup_share FROM ATLAS_GRISLI.t_tier2_groups "
        sqlT  = "SELECT tier2,t2group,status FROM ATLAS_GRISLI.t_m4regions_replication"
        try:
            # start transaction
            self.conn.begin()
            self.cur.arraysize = 100000
            # get weight for each group
            self.cur.execute(sqlG+comment)
            resG = self.cur.fetchall()
            gidShareMap = {}
            for gid,ntup_share in resG:
                gidShareMap[gid] = {'ntup_share':ntup_share,'nSites':0}
            # get group for each site
            self.cur.execute(sqlT+comment)
            resT = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            siteGroupMap = {}
            # loop over all sites
            for tier2,t2group,t2status in resT:
                # unknown group
                if t2group not in gidShareMap:
                    _logger.error("getMouShareForT2PD2P unknown group %s for %s" % (t2group,tier2))
                    continue
                # use only DATADISK
                if not tier2.endswith('_DATADISK'):
                    continue
                # count the number of ready sites per group
                if t2status in ['ready']:
                    gidShareMap[t2group]['nSites'] += 1
                # append
                siteGroupMap[tier2] = {'group':t2group,'status':t2status}
            # normalize
            _logger.debug("getMouShareForT2PD2P normalize factor = %s" % str(gidShareMap))
            weightsMap = {}
            for tier2 in siteGroupMap:
                t2Val = siteGroupMap[tier2]
                t2group  = t2Val['group']
                t2status = t2Val['status']
                if gidShareMap[t2group]['ntup_share'] == 0:
                    # set 0 to be skipped in the brokerage
                  tmpWeight = 0
                elif gidShareMap[t2group]['nSites'] > 0:
                    # normalize
                    tmpWeight = float(gidShareMap[t2group]['ntup_share']) / float(gidShareMap[t2group]['nSites'])
                else:
                    # no site is ready in this group
                    tmpWeight = 0
                weightsMap[tier2] = {'weight':tmpWeight,'status':t2status}
            _logger.debug("getMouShareForT2PD2P -> %s" % str(weightsMap))
            return weightsMap
        except Exception:
            # roll back
            self._rollback()
            errType,errValue = sys.exc_info()[:2]
            _logger.error("getMouShareForT2PD2P : %s %s" % (errType,errValue))
            return {}


    # record status change
    def recordStatusChange(self,pandaID,jobStatus,jobInfo=None,infoMap={},useCommit=True):
        comment = ' /* DBProxy.recordStatusChange */'
        # check config
        if not hasattr(panda_config,'record_statuschange') or panda_config.record_statuschange is not True:
            return
        # get job info
        varMap = {}
        varMap[':PandaID']          = pandaID
        varMap[':jobStatus']        = jobStatus
        varMap[':modificationHost'] = self.myHostName
        if jobInfo is not None:
            varMap[':computingSite']   = jobInfo.computingSite
            varMap[':cloud']           = jobInfo.cloud
            varMap[':prodSourceLabel'] = jobInfo.prodSourceLabel
        elif infoMap is not None:
            varMap[':computingSite']   = infoMap['computingSite']
            varMap[':cloud']           = infoMap['cloud']
            varMap[':prodSourceLabel'] = infoMap['prodSourceLabel']
        else:
            # no info
            return
        # convert NULL to None
        for tmpKey in varMap:
            if varMap[tmpKey] == 'NULL':
                varMap[tmpKey] = None
        # insert
        sql  = "INSERT INTO ATLAS_PANDA.jobs_StatusLog "
        sql += "(PandaID,modificationTime,jobStatus,prodSourceLabel,cloud,computingSite,modificationHost,modiftime_extended) "
        sql += "VALUES (:PandaID,CURRENT_DATE,:jobStatus,:prodSourceLabel,:cloud,:computingSite,:modificationHost,CURRENT_TIMESTAMP) "
        try:
            # start transaction
            if useCommit:
                self.conn.begin()
            self.cur.execute(sql+comment,varMap)
            # commit
            if useCommit:
                if not self._commit():
                    raise RuntimeError('Commit error')
        except Exception:
            # roll back
            if useCommit:
                self._rollback()
            errType,errValue = sys.exc_info()[:2]
            _logger.error("recordStatusChange %s %s: %s %s" % (pandaID,jobStatus,errType,errValue))
            if not useCommit:
                raise RuntimeError('recordStatusChange failed')
        return

    def push_job_status_message(self, job_spec, panda_id, status, jedi_task_id=None, special_handling=None):
        if not (hasattr(panda_config, 'mbproxy_configFile') and panda_config.mbproxy_configFile):
            # skip if not configured
            return
        to_push = False
        if special_handling is not None:
            to_push = push_status_changes(special_handling)
        elif job_spec is not None:
            to_push = job_spec.push_status_changes()
        # only run if to push status change
        if not to_push:
            return
        # skip statuses unnecessary to push
        if status in ['sent', 'holding', 'merging']:
            return
        # skip if no mb to push to
        mb_proxy = self.get_mb_proxy('panda_jobstatus')
        if not mb_proxy:
            return
        if to_push:
            # push job status change
            try:
                now_time = datetime.datetime.utcnow()
                now_ts = int(now_time.timestamp())
                # task id
                if jedi_task_id is None and job_spec is not None:
                    jedi_task_id = job_spec.jediTaskID
                # inputs
                inputs = []
                if job_spec is not None and job_spec.Files is not None:
                    for file_spec in job_spec.Files:
                        if file_spec.type in ['input', 'pseudo_input']:
                            inputs.append(file_spec.lfn)
                # message
                msg_dict = {
                        'msg_type': 'job_status',
                        'jobid': panda_id,
                        'taskid': jedi_task_id,
                        'status': status,
                        'inputs': inputs if inputs else None,
                        'timestamp': now_ts,
                    }
                msg = json.dumps(msg_dict)
                if mb_proxy.got_disconnected:
                    mb_proxy.restart()
                mb_proxy.send(msg)
                _logger.debug('push_job_status_message: sent message: {0}'.format(msg))
            except Exception:
                errType, errValue = sys.exc_info()[:2]
                _logger.error("push_job_status_message %s %s: %s %s" % (panda_id, status, errType, errValue))


    # propagate result to JEDI
    def propagateResultToJEDI(self,jobSpec,cur,oldJobStatus=None,extraInfo=None,finishPending=False,waitLock=False):
        comment = ' /* DBProxy.propagateResultToJEDI */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " <PandaID={0}>".format(jobSpec.PandaID)
        tmpLog = LogWrapper(_logger,methodName)
        datasetContentsStat = {}
        # loop over all files
        finishUnmerge = set()
        hasInput = False
        tmpLog.debug('waitLock={0}'.format(waitLock))
        # make pseudo files for dynamic number of events
        pseudoFiles = []
        if EventServiceUtils.isDynNumEventsSH(jobSpec.specialHandling):
            # make row_ID and fileSpec map
            rowIdSpecMap = {}
            for fileSpec in jobSpec.Files:
                rowIdSpecMap[fileSpec.row_ID] = fileSpec
            # get pseudo files
            varMap = {}
            varMap[':PandaID']    = jobSpec.PandaID
            varMap[':jediTaskID'] = jobSpec.jediTaskID
            varMap[':eventID']    = -1
            sqlPF  = "SELECT fileID,attemptNr,job_processID "
            sqlPF += "FROM {0}.JEDI_Events ".format(panda_config.schemaJEDI)
            sqlPF += "WHERE jediTaskID=:jediTaskID AND PandaID=:PandaID AND processed_upto_eventID=:eventID "
            cur.execute(sqlPF+comment,varMap)
            resPF = self.cur.fetchall()
            for tmpFileID,tmpAttemptNr,tmpRow_ID in resPF:
                tmpFileSpec = copy.copy(rowIdSpecMap[tmpRow_ID])
                tmpFileSpec.fileID = tmpFileID
                tmpFileSpec.attemptNr = tmpAttemptNr-1
                pseudoFiles.append(tmpFileSpec)
            tmpLog.debug('{0} pseudo files'.format(len(pseudoFiles)))
        # flag for job cloning
        useJobCloning = False
        if EventServiceUtils.isEventServiceJob(jobSpec) and EventServiceUtils.isJobCloningJob(jobSpec):
            useJobCloning = True
        # set delete flag to events
        if (EventServiceUtils.isEventServiceJob(jobSpec) or EventServiceUtils.isEventServiceMerge(jobSpec)) and \
                jobSpec.jobStatus in ['finished','failed','cancelled']:
            # sql to check attemptNr
            sqlDelC  = "SELECT attemptNr FROM {0}.JEDI_Dataset_Contents ".format(panda_config.schemaJEDI)
            sqlDelC += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
            # sql to set delete flag
            sqlDelE  = "UPDATE /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
            sqlDelE += "{0}.JEDI_Events tab ".format(panda_config.schemaJEDI)
            sqlDelE += "SET file_not_deleted=CASE WHEN objStore_ID IS NULL THEN NULL ELSE :delFlag END "
            if jobSpec.jobStatus == 'finished':
                sqlDelE += ",status=CASE WHEN status=:st_done THEN :st_merged ELSE status END "
            sqlDelE += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
            for fileSpec in jobSpec.Files:
                if fileSpec.type not in ['input','pseudo_input']:
                    continue
                # check attemptNr
                varMap = {}
                varMap[':jediTaskID'] = fileSpec.jediTaskID
                varMap[':datasetID']  = fileSpec.datasetID
                varMap[':fileID']     = fileSpec.fileID
                self.cur.execute(sqlDelC+comment,varMap)
                tmpAttemptNr, = self.cur.fetchone()
                if fileSpec.attemptNr != tmpAttemptNr:
                    tmpLog.debug('skip to set Y for datasetID={0} fileID={1} due to attemprNr mismatch'.format(fileSpec.datasetID,fileSpec.fileID))
                    continue
                # set del flag
                varMap = {}
                varMap[':jediTaskID'] = fileSpec.jediTaskID
                varMap[':datasetID']  = fileSpec.datasetID
                varMap[':fileID']     = fileSpec.fileID
                varMap[':delFlag']    = 'Y'
                if jobSpec.jobStatus == 'finished':
                    varMap[':st_done']    = EventServiceUtils.ST_done
                    varMap[':st_merged']  = EventServiceUtils.ST_merged
                tmpLog.debug(sqlDelE+comment+str(varMap))
                self.cur.execute(sqlDelE+comment,varMap)
                retDelE = self.cur.rowcount
                tmpLog.debug('set Y to {0} event ranges'.format(retDelE))
        # loop over all files to update file or dataset attributes
        for fileSpec in jobSpec.Files+pseudoFiles:
            # skip if no JEDI
            if fileSpec.fileID == 'NULL':
                continue
            # do nothing for unmerged output/log files when merged job successfully finishes,
            # since they were already updated by merged job
            if jobSpec.jobStatus == 'finished' and fileSpec.isUnMergedOutput():
                continue
            # check file status
            varMap = {}
            varMap[':fileID']     = fileSpec.fileID
            varMap[':datasetID']  = fileSpec.datasetID
            varMap[':jediTaskID'] = jobSpec.jediTaskID
            # no attemptNr check for premerge since attemptNr can be incremented by pmerge
            if not (jobSpec.isCancelled() and fileSpec.isUnMergedOutput()):
                varMap[':attemptNr']  = fileSpec.attemptNr
            sqlFileStat  = "SELECT status,is_waiting FROM ATLAS_PANDA.JEDI_Dataset_Contents "
            sqlFileStat += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
            if not (jobSpec.isCancelled() and fileSpec.isUnMergedOutput()):
                sqlFileStat += "AND attemptNr=:attemptNr "
            sqlFileStat += "FOR UPDATE "
            if not waitLock:
                sqlFileStat += "NOWAIT "
            cur.execute(sqlFileStat+comment,varMap)
            resFileStat = self.cur.fetchone()
            if resFileStat is not None:
                oldFileStatus,oldIsWaiting = resFileStat
            else:
                oldFileStatus,oldIsWaiting = None,None
            # skip if already cancelled
            if oldFileStatus in ['cancelled']:
                continue
            # update Dataset Contents table
            updateMetadata  = False
            updateAttemptNr = False
            updateNumEvents = False
            updateFailedAttempt = False
            varMap = {}
            varMap[':fileID']     = fileSpec.fileID
            varMap[':datasetID']  = fileSpec.datasetID
            varMap[':keepTrack']  = 1
            varMap[':jediTaskID'] = jobSpec.jediTaskID
            if not (jobSpec.isCancelled() and fileSpec.isUnMergedOutput()):
                varMap[':attemptNr']  = fileSpec.attemptNr
            # set file status
            if fileSpec.type in ['input','pseudo_input']:
                hasInput = True
                updateAttemptNr = True
                if jobSpec.jobStatus == 'finished' and not jobSpec.is_hpo_workflow():
                    varMap[':status'] = 'finished'
                    if fileSpec.type in ['input','pseudo_input']:
                         updateNumEvents = True
                else:
                    # set ready for next attempt
                    varMap[':status'] = 'ready'
                    if jobSpec.jobStatus == 'failed' and not jobSpec.is_hpo_workflow():
                        updateFailedAttempt = True
            else:
                if jobSpec.isCancelled():
                    # use only cancelled for all flavors
                    varMap[':status'] = 'cancelled'
                else:
                    varMap[':status'] = jobSpec.jobStatus
                if fileSpec.status == 'nooutput':
                    varMap[':status'] = fileSpec.status
                elif jobSpec.jobStatus == 'finished':
                    varMap[':status'] = 'finished'
                    # update metadata
                    updateMetadata = True
                    # update nEvents
                    updateNumEvents = True
                elif fileSpec.status == 'merging':
                    # set ready to merge files for failed jobs
                    varMap[':status'] = 'ready'
                     # update metadata
                    updateMetadata = True
            sqlFile = "UPDATE /*+ INDEX_RS_ASC(JEDI_DATASET_CONTENTS (JEDI_DATASET_CONTENTS.JEDITASKID JEDI_DATASET_CONTENTS.DATASETID JEDI_DATASET_CONTENTS.FILEID)) */ ATLAS_PANDA.JEDI_Dataset_Contents SET status=:status,is_waiting=NULL"
            # attempt number
            if updateAttemptNr is True:
                # increment attemptNr for next attempt
                if not jobSpec.is_hpo_workflow():
                    sqlFile += ",attemptNr=attemptNr+1"
                else:
                    sqlFile += ",attemptNr=MOD(attemptNr+1,maxAttempt)"
            # failed attempts
            if updateFailedAttempt is True:
                sqlFile += ",failedAttempt=failedAttempt+1"
            # set correct PandaID for job cloning
            if useJobCloning:
                varMap[':PandaID'] = jobSpec.PandaID
                if fileSpec.type in ['log','output']:
                    sqlFile += ",outPandaID=:PandaID,PandaID=:PandaID"
                else:
                    sqlFile += ",PandaID=:PandaID"
            # metadata
            if updateMetadata:
                # set file metadata
                for tmpKey in ['lfn','GUID','fsize','checksum']:
                    tmpVal = getattr(fileSpec,tmpKey)
                    if tmpVal == 'NULL':
                        if tmpKey in fileSpec._zeroAttrs:
                            tmpVal = 0
                        else:
                            tmpVal = None
                    tmpMapKey = ':%s' % tmpKey
                    sqlFile += ",%s=%s" % (tmpKey,tmpMapKey)
                    varMap[tmpMapKey] = tmpVal
                # extra metadata
                if extraInfo is not None:
                    # nevents
                    if 'nevents' in extraInfo and fileSpec.lfn in extraInfo['nevents']:
                        tmpKey = 'nEvents'
                        tmpMapKey = ':%s' % tmpKey
                        sqlFile += ",%s=%s" % (tmpKey,tmpMapKey)
                        varMap[tmpMapKey] = extraInfo['nevents'][fileSpec.lfn]
                    # LB number
                    if 'lbnr' in extraInfo and fileSpec.lfn in extraInfo['lbnr']:
                        tmpKey = 'lumiBlockNr'
                        tmpMapKey = ':%s' % tmpKey
                        sqlFile += ",%s=%s" % (tmpKey,tmpMapKey)
                        varMap[tmpMapKey] = extraInfo['lbnr'][fileSpec.lfn]
                # reset keepTrack unless merging
                if fileSpec.status != 'merging':
                    sqlFile += ",keepTrack=NULL"
                else:
                    # set boundaryID for merging
                    sqlFile += ",boundaryID=:boundaryID"
                    varMap[':boundaryID'] = jobSpec.PandaID
                    # set max attempt
                    sqlFile += ",maxAttempt=attemptNr+3"
            sqlFile += " WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
            sqlFile += "AND keepTrack=:keepTrack "
            if not (jobSpec.isCancelled() and fileSpec.isUnMergedOutput()):
                sqlFile += "AND attemptNr=:attemptNr "
            tmpLog.debug(sqlFile+comment+str(varMap))
            cur.execute(sqlFile+comment,varMap)
            nRow = cur.rowcount
            if nRow == 1 and fileSpec.status not in ['nooutput']:
                datasetID = fileSpec.datasetID
                fileStatus = varMap[':status']
                if datasetID not in datasetContentsStat:
                    datasetContentsStat[datasetID] = {'nFilesUsed':0,'nFilesFinished':0,
                                                      'nFilesFailed':0,'nFilesOnHold':0,
                                                      'nFilesTobeUsed':0,'nEvents':0,
                                                      'nEventsUsed':0,'nFilesWaiting':0}
                # read nEvents
                if updateNumEvents:
                    sqlEVT = "SELECT nEvents,startEvent,endEvent,keepTrack FROM ATLAS_PANDA.JEDI_Dataset_Contents "
                    sqlEVT += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                    if not waitLock:
                        sqlEVT += "FOR UPDATE NOWAIT "
                    varMap = {}
                    varMap[':fileID']     = fileSpec.fileID
                    varMap[':datasetID']  = fileSpec.datasetID
                    varMap[':jediTaskID'] = jobSpec.jediTaskID
                    tmpLog.debug(sqlEVT+comment+str(varMap))
                    cur.execute(sqlEVT+comment,varMap)
                    resEVT = self.cur.fetchone()
                    if resEVT is not None:
                        tmpNumEvents,tmpStartEvent,tmpEndEvent,tmpKeepTrack = resEVT
                        if tmpNumEvents is not None:
                            try:
                                if fileSpec.type in ['input','pseudo_input']:
                                    if tmpKeepTrack == 1:
                                        # keep track on how many events successfully used
                                        if tmpStartEvent is not None and tmpEndEvent is not None:
                                            datasetContentsStat[datasetID]['nEventsUsed'] += (tmpEndEvent-tmpStartEvent+1)
                                        else:
                                            datasetContentsStat[datasetID]['nEventsUsed'] += tmpNumEvents
                                else:
                                    datasetContentsStat[datasetID]['nEvents'] += tmpNumEvents
                            except Exception:
                                pass
                # update file counts
                isDone = False
                if fileSpec.status == 'merging' and \
                        (finishPending or jobSpec.prodSourceLabel not in ['user','panda']):
                    # files to be merged for pending failed jobs
                    datasetContentsStat[datasetID]['nFilesOnHold'] += 1
                elif fileStatus == 'ready':
                    # check attemptNr and maxAttempt when the file failed (ready = input failed)
                    # skip secondary datasets which have maxAttempt=None
                    sqlAttNr  = "SELECT attemptNr,maxAttempt,failedAttempt,maxFailure FROM ATLAS_PANDA.JEDI_Dataset_Contents "
                    sqlAttNr += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                    varMap = {}
                    varMap[':fileID']     = fileSpec.fileID
                    varMap[':datasetID']  = fileSpec.datasetID
                    varMap[':jediTaskID'] = jobSpec.jediTaskID
                    tmpLog.debug(sqlAttNr+comment+str(varMap))
                    cur.execute(sqlAttNr+comment,varMap)
                    resAttNr = self.cur.fetchone()
                    if resAttNr is not None:
                        newAttemptNr,maxAttempt,failedAttempt,maxFailure = resAttNr
                        if maxAttempt is not None:
                            if maxAttempt > newAttemptNr and (maxFailure is None or maxFailure > failedAttempt):
                                if oldFileStatus == 'ready':
                                    # don't change nFilesUsed when fake co-jumbo is done
                                    pass
                                elif fileSpec.status != 'merging':
                                    # decrement nUsed to trigger reattempt
                                    datasetContentsStat[datasetID]['nFilesUsed'] -= 1
                                else:
                                    # increment nTobeUsed to trigger merging
                                    datasetContentsStat[datasetID]['nFilesTobeUsed'] += 1
                            else:
                                # no more reattempt
                                datasetContentsStat[datasetID]['nFilesFailed'] += 1
                                isDone = True
                                # merge job failed
                                if jobSpec.processingType == 'pmerge':
                                    # update unmerged file
                                    sqlUmFile  = "UPDATE ATLAS_PANDA.JEDI_Dataset_Contents SET status=:status,keepTrack=NULL "
                                    sqlUmFile += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                                    varMap = {}
                                    varMap[':fileID']     = fileSpec.fileID
                                    varMap[':datasetID']  = fileSpec.datasetID
                                    varMap[':jediTaskID'] = jobSpec.jediTaskID
                                    varMap[':status']     = 'notmerged'
                                    tmpLog.debug(sqlUmFile+comment+str(varMap))
                                    cur.execute(sqlUmFile+comment,varMap)
                                    # set flag to update unmerged jobs
                                    finishUnmerge.add(fileSpec.fileID)
                elif fileStatus in ['finished','lost']:
                    # successfully used or produced, or lost
                    datasetContentsStat[datasetID]['nFilesFinished'] += 1
                    isDone = True
                else:
                    # failed to produce the file
                    datasetContentsStat[datasetID]['nFilesFailed'] += 1
                # changed from transferring
                if fileSpec.type in ['input','pseudo_input']:
                    if oldJobStatus == 'transferring':
                        datasetContentsStat[datasetID]['nFilesOnHold'] -= 1
                # reset is_waiting
                if oldIsWaiting is not None:
                    datasetContentsStat[datasetID]['nFilesWaiting'] -= 1
                    if isDone:
                        datasetContentsStat[datasetID]['nFilesUsed'] += 1
                # killed during merging
                if jobSpec.isCancelled() and oldJobStatus == 'merging' and fileSpec.isUnMergedOutput():
                    # get corresponding sub
                    varMap = {}
                    varMap[':pandaID']    = jobSpec.PandaID
                    varMap[':fileID']     = fileSpec.fileID
                    varMap[':datasetID']  = fileSpec.datasetID
                    varMap[':jediTaskID'] = jobSpec.jediTaskID
                    sqlGetDest  = "SELECT destinationDBlock FROM ATLAS_PANDA.filesTable4 "
                    sqlGetDest += "WHERE pandaID=:pandaID AND jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                    tmpLog.debug(sqlGetDest+comment+str(varMap))
                    cur.execute(sqlGetDest+comment,varMap)
                    preMergedDest, = self.cur.fetchone()
                    # check if corresponding sub is closed
                    varMap = {}
                    varMap[':name']    = preMergedDest
                    varMap[':subtype'] = 'sub'
                    sqlCheckDest  = "SELECT status FROM ATLAS_PANDA.Datasets "
                    sqlCheckDest += "WHERE name=:name AND subtype=:subtype "
                    tmpLog.debug(sqlCheckDest+comment+str(varMap))
                    cur.execute(sqlCheckDest+comment,varMap)
                    tmpResDestStat = self.cur.fetchone()
                    if tmpResDestStat is not None:
                        preMergedDestStat, = tmpResDestStat
                    else:
                        preMergedDestStat = 'notfound'
                        tmpLog.debug('{0} not found for datasetID={1}'.format(preMergedDest,datasetID))
                    if preMergedDestStat not in ['tobeclosed','completed']:
                        datasetContentsStat[datasetID]['nFilesOnHold'] -= 1
                    else:
                        tmpLog.debug('not change nFilesOnHold for datasetID={0} since sub is in {1}'.format(datasetID,
                                                                                                            preMergedDestStat))
                        # increment nUsed when mergeing is killed before merge job is generated
                        if oldFileStatus == 'ready':
                            datasetContentsStat[datasetID]['nFilesUsed'] += 1
        # update JEDI_Datasets table
        nOutEvents = 0
        if datasetContentsStat != {}:
            tmpDatasetIDs = list(datasetContentsStat)
            tmpDatasetIDs.sort()
            for tmpDatasetID in tmpDatasetIDs:
                tmpContentsStat = datasetContentsStat[tmpDatasetID]
                sqlJediDL = "SELECT nFilesUsed,nFilesFailed,nFilesTobeUsed,nFilesFinished,nFilesOnHold,type,masterID FROM ATLAS_PANDA.JEDI_Datasets "
                sqlJediDL += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID FOR UPDATE "
                if not waitLock:
                    sqlJediDL += "NOWAIT "
                varMap = {}
                varMap[':jediTaskID'] = jobSpec.jediTaskID
                varMap[':datasetID']  = tmpDatasetID
                cur.execute(sqlJediDL+comment,varMap)
                tmpResJediDL = self.cur.fetchone()
                t_nFilesUsed, t_nFilesFailed, t_nFilesTobeUsed, t_nFilesFinished, t_nFilesOnHold, t_type, t_masterID = tmpResJediDL
                tmpLog.debug('datasetID={0} had nFilesTobeUsed={1} nFilesUsed={2} nFilesFinished={3} nFilesFailed={4}'.format(tmpDatasetID, t_nFilesTobeUsed,
                                                                                                                              t_nFilesUsed, t_nFilesFinished,
                                                                                                                              t_nFilesFailed))
                # sql to update nFiles info
                toUpdateFlag = False
                eventsToRead = False
                sqlJediDS = "UPDATE ATLAS_PANDA.JEDI_Datasets SET "
                for tmpStatKey in tmpContentsStat:
                    tmpStatVal = tmpContentsStat[tmpStatKey]
                    if tmpStatVal == 0:
                        continue
                    if tmpStatVal > 0:
                        sqlJediDS += '%s=%s+%s,' % (tmpStatKey,tmpStatKey,tmpStatVal)
                    else:
                        sqlJediDS += '%s=%s-%s,' % (tmpStatKey,tmpStatKey,abs(tmpStatVal))
                    toUpdateFlag = True
                    if tmpStatKey == 'nEvents' and tmpStatVal > nOutEvents:
                        nOutEvents = tmpStatVal
                sqlJediDS  = sqlJediDS[:-1]
                sqlJediDS += " WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID "
                varMap = {}
                varMap[':jediTaskID'] = jobSpec.jediTaskID
                varMap[':datasetID']  = tmpDatasetID
                # update
                if toUpdateFlag:
                    tmpLog.debug(sqlJediDS+comment+str(varMap))
                    cur.execute(sqlJediDS+comment,varMap)
                    # update events in corrupted input files
                    if EventServiceUtils.isEventServiceMerge(jobSpec) and jobSpec.jobStatus == 'failed' \
                            and jobSpec.pilotErrorCode in \
                            EventServiceUtils.PEC_corruptedInputFiles+EventServiceUtils.PEC_corruptedInputFilesTmp \
                            and t_type in ['input', 'pseudo_input'] and t_masterID is None \
                            and (tmpContentsStat['nFilesUsed'] < 0 or tmpContentsStat['nFilesFailed'] > 0):
                        toSet = True
                        if jobSpec.pilotErrorCode in EventServiceUtils.PEC_corruptedInputFilesTmp:
                            # check failure count for temporary errors
                            toSet = self.checkFailureCountWithCorruptedFiles(jobSpec.jediTaskID, jobSpec.PandaID)
                        if toSet:
                            self.setCorruptedEventRanges(jobSpec.jediTaskID, jobSpec.PandaID)
        # add jobset info for job cloning
        if useJobCloning:
            self.recordRetryHistoryJEDI(jobSpec.jediTaskID,jobSpec.PandaID,[jobSpec.jobsetID],EventServiceUtils.relationTypeJS_ID)
        # update jumbo flag
        if jobSpec.eventService == EventServiceUtils.jumboJobFlagNumber:
            # check site
            varMap = {}
            varMap[':jediTaskID'] = jobSpec.jediTaskID
            sqlJumboS = "SELECT site FROM {0}.JEDI_Tasks WHERE jediTaskID=:jediTaskID ".format(panda_config.schemaJEDI)
            cur.execute(sqlJumboS+comment,varMap)
            tmpResS = self.cur.fetchone()
            jumboSite, = tmpResS
            # count number of events for jumbo
            newUseJumbo = 'L'
            """
            varMap = {}
            varMap[':jediTaskID'] = jobSpec.jediTaskID
            varMap[':eventStatus']  = EventServiceUtils.ST_ready
            varMap[':minAttemptNr'] = 0
            sqlJumboC = "SELECT COUNT(*) FROM {0}.JEDI_Events ".format(panda_config.schemaJEDI)
            sqlJumboC += "WHERE jediTaskID=:jediTaskID AND status=:eventStatus AND attemptNr>:minAttemptNr ".format(panda_config.schemaJEDI)
            cur.execute(sqlJumboC+comment,varMap)
            tmpResC = self.cur.fetchone()
            if tmpResC is not None:
                nEventsJumbo, = tmpResC
                tmpLog.debug('{0} event ranges available for jumbo'.format(nEventsJumbo))
                # no more events
                if nEventsJumbo == 0 and jumboSite is None:
                    newUseJumbo = 'D'
            """
            # update flag
            varMap = {}
            varMap[':jediTaskID'] = jobSpec.jediTaskID
            varMap[':newJumbo'] = newUseJumbo
            varMap[':notUseJumbo'] = 'D'
            sqlJumboF = "UPDATE {0}.JEDI_Tasks ".format(panda_config.schemaJEDI)
            sqlJumboF += "SET useJumbo=:newJumbo WHERE jediTaskID=:jediTaskID "
            sqlJumboF += "AND useJumbo IS NOT NULL AND useJumbo<>:notUseJumbo "
            cur.execute(sqlJumboF+comment,varMap)
            nRow = cur.rowcount
            tmpLog.debug('set task.useJumbo={0} with {1}'.format(varMap[':newJumbo'], nRow))
        # update input
        if not EventServiceUtils.isJumboJob(jobSpec) and \
                not (jobSpec.computingSite == EventServiceUtils.siteIdForWaitingCoJumboJobs and not jobSpec.isCancelled()) \
                and jobSpec.taskBufferErrorCode not in [ErrorCode.EC_PilotRetried]:
            self.updateInputStatusJedi(jobSpec.jediTaskID, jobSpec.PandaID, jobSpec.jobStatus)
        # update t_task
        if jobSpec.jobStatus == 'finished' and jobSpec.prodSourceLabel not in ['panda']:
            varMap = {}
            varMap[':jediTaskID'] = jobSpec.jediTaskID
            varMap[':noutevents']  = nOutEvents
            schemaDEFT = self.getSchemaDEFT()
            sqlTtask  = "UPDATE {0}.T_TASK ".format(schemaDEFT)
            if jobSpec.processingType != 'pmerge':
                updateNumDone = True
                sqlTtask += "SET total_done_jobs=total_done_jobs+1,timestamp=CURRENT_DATE,total_events=LEAST(9999999999,total_events+:noutevents) "
            else:
                updateNumDone = False
                sqlTtask += "SET timestamp=CURRENT_DATE,total_events=LEAST(9999999999,total_events+:noutevents) "
            sqlTtask += "WHERE taskid=:jediTaskID "
            tmpLog.debug(sqlTtask+comment+str(varMap))
            cur.execute(sqlTtask+comment,varMap)
            nRow = cur.rowcount
            # get total_done_jobs
            if updateNumDone and nRow == 1:
                varMap = {}
                varMap[':jediTaskID'] = jobSpec.jediTaskID
                sqlNumDone  = "SELECT total_done_jobs FROM {0}.T_TASK ".format(schemaDEFT)
                sqlNumDone += "WHERE taskid=:jediTaskID "
                cur.execute(sqlNumDone+comment,varMap)
                tmpResNumDone = self.cur.fetchone()
                if tmpResNumDone is not None:
                    numDone, = tmpResNumDone
                    if numDone in [100]:
                        # reset walltimeUnit to recalcurate task parameters
                        varMap = {}
                        varMap[':jediTaskID'] = jobSpec.jediTaskID
                        sqlRecal  = "UPDATE ATLAS_PANDA.JEDI_Tasks SET walltimeUnit=NULL WHERE jediTaskId=:jediTaskID "
                        msgStr  = "trigger recalcuration of task parameters "
                        msgStr += "with nDoneJobs={0} for jediTaskID={1}".format(numDone,jobSpec.jediTaskID)
                        tmpLog.debug(msgStr)
                        cur.execute(sqlRecal+comment,varMap)
        # propagate failed result to unmerge job
        if len(finishUnmerge) > 0:
            self.updateUnmergedJobs(jobSpec,finishUnmerge)
        # update some job attributes
        self.setHS06sec(jobSpec.PandaID)
        # return
        return True



    # check if task is active
    def checkTaskStatusJEDI(self,jediTaskID,cur):
        comment = ' /* DBProxy.checkTaskStatusJEDI */'
        retVal = False
        curStat = None
        if jediTaskID not in ['NULL',None]:
            sql = "SELECT status FROM ATLAS_PANDA.JEDI_Tasks WHERE jediTaskID=:jediTaskID "
            varMap = {}
            varMap[':jediTaskID'] = jediTaskID
            cur.execute(sql+comment,varMap)
            res = cur.fetchone()
            if res is not None:
                curStat = res[0]
                if curStat not in ['done','finished','failed',
                                   'broken','aborted','prepared']:
                    retVal = True
        _logger.debug('checkTaskStatusJEDI jediTaskID=%s in %s with %s' % (jediTaskID,curStat,retVal))
        return retVal



    # update JEDI for pilot retry
    def updateForPilotRetryJEDI(self,job,cur,onlyHistory=False,relationType=None):
        comment = ' /* DBProxy.updateForPilotRetryJEDI */'
        # sql to update file
        sqlFJI  = "UPDATE {0}.JEDI_Dataset_Contents ".format(panda_config.schemaJEDI)
        sqlFJI += "SET attemptNr=attemptNr+1,failedAttempt=failedAttempt+1,PandaID=:PandaID "
        sqlFJI += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
        sqlFJI += "AND attemptNr=:attemptNr AND keepTrack=:keepTrack "
        sqlFJO  = "UPDATE {0}.JEDI_Dataset_Contents ".format(panda_config.schemaJEDI)
        sqlFJO += "SET attemptNr=attemptNr+1,failedAttempt=failedAttempt+1,PandaID=:PandaID,outPandaID=:PandaID "
        sqlFJO += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
        sqlFJO += "AND attemptNr=:attemptNr AND keepTrack=:keepTrack "
        sqlFP  = "UPDATE ATLAS_PANDA.filesTable4 SET attemptNr=attemptNr+1 "
        sqlFP += "WHERE row_ID=:row_ID "
        if not onlyHistory:
            for tmpFile in job.Files:
                # skip if no JEDI
                if tmpFile.fileID == 'NULL':
                    continue
                # update JEDI contents
                varMap = {}
                varMap[':jediTaskID'] = tmpFile.jediTaskID
                varMap[':datasetID']  = tmpFile.datasetID
                varMap[':fileID']     = tmpFile.fileID
                varMap[':attemptNr']  = tmpFile.attemptNr
                varMap[':PandaID']    = tmpFile.PandaID
                varMap[':keepTrack']  = 1
                if tmpFile.type in ['output','log']:
                    sqlFJ = sqlFJO
                else:
                    sqlFJ = sqlFJI
                _logger.debug(sqlFJ+comment+str(varMap))
                cur.execute(sqlFJ+comment,varMap)
                nRow = cur.rowcount
                if nRow == 1:
                    # update fileTable if JEDI contents was updated
                    varMap = {}
                    varMap[':row_ID'] = tmpFile.row_ID
                    _logger.debug(sqlFP+comment+str(varMap))
                    cur.execute(sqlFP+comment,varMap)
        # get origin
        originIDs = self.getOriginPandaIDsJEDI(job.parentID,job.jediTaskID,cur)
        # sql to record retry history
        sqlRH  = "INSERT INTO {0}.JEDI_Job_Retry_History ".format(panda_config.schemaJEDI)
        sqlRH += "(jediTaskID,oldPandaID,newPandaID,originPandaID,relationType) "
        sqlRH += "VALUES(:jediTaskID,:oldPandaID,:newPandaID,:originPandaID,:relationType) "
        # record retry history
        for originID in originIDs:
            varMap = {}
            varMap[':jediTaskID'] = job.jediTaskID
            varMap[':oldPandaID'] = job.parentID
            varMap[':newPandaID'] = job.PandaID
            varMap[':originPandaID'] = originID
            if relationType is None:
                varMap[':relationType'] = 'retry'
            else:
                varMap[':relationType'] = relationType
            cur.execute(sqlRH+comment,varMap)
        # record jobset
        if EventServiceUtils.isEventServiceMerge(job) and relationType is None:
            varMap = {}
            varMap[':jediTaskID'] = job.jediTaskID
            varMap[':oldPandaID'] = job.jobsetID
            varMap[':newPandaID'] = job.PandaID
            varMap[':originPandaID'] = job.jobsetID
            varMap[':relationType'] = EventServiceUtils.relationTypeJS_ID
            cur.execute(sqlRH+comment,varMap)
        return



    # get origin PandaIDs
    def getOriginPandaIDsJEDI(self,pandaID,jediTaskID,cur):
        comment = ' /* DBProxy.getOriginPandaIDsJEDI */'
        # get parent IDs
        varMap = {}
        varMap[':jediTaskID'] = jediTaskID
        varMap[':newPandaID'] = pandaID
        sqlFJ  = "SELECT MIN(originPandaID) FROM {0}.JEDI_Job_Retry_History ".format(panda_config.schemaJEDI)
        sqlFJ += "WHERE jediTaskID=:jediTaskID AND newPandaID=:newPandaID "
        sqlFJ += "AND (relationType IS NULL OR NOT relationType IN ("
        for tmpType in EventServiceUtils.relationTypesForJS:
            tmpKey = ':{0}'.format(tmpType)
            sqlFJ += '{0},'.format(tmpKey)
            varMap[tmpKey] = tmpType
        sqlFJ = sqlFJ[:-1]
        sqlFJ  += ')) '
        cur.execute(sqlFJ+comment,varMap)
        resT = cur.fetchone()
        retList = []
        if resT is None:
            # origin
            retList.append(pandaID)
        else:
            # use only one origin since tracking the whole tree brings too many origins
            originPandaID, = resT
            if originPandaID is None:
                # origin
                retList.append(pandaID)
            else:
                retList.append(originPandaID)
        # return
        return retList



    # get jobsetID for PandaID
    def getJobsetIDforPandaID(self,pandaID,jediTaskID):
        comment = ' /* DBProxy. */'
        # get parent IDs
        varMap = {}
        varMap[':jediTaskID']   = jediTaskID
        varMap[':newPandaID']   = pandaID
        varMap[':relationType'] = EventServiceUtils.relationTypeJS_ID
        sqlFJ  = "SELECT oldPandaID FROM {0}.JEDI_Job_Retry_History ".format(panda_config.schemaJEDI)
        sqlFJ += "WHERE jediTaskID=:jediTaskID AND newPandaID=:newPandaID "
        sqlFJ += "AND relationType=:relationType "
        self.cur.execute(sqlFJ+comment,varMap)
        resT = self.cur.fetchone()
        if resT is not None:
            return resT[0]
        return None



    # get retry history
    def getRetryHistoryJEDI(self,jediTaskID):
        comment = ' /* DBProxy.getRetryHistoryJEDI */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " <jediTaskID={0}>".format(jediTaskID)
        _logger.debug("{0} start".format(methodName))
        try:
            # set autocommit on
            self.conn.begin()
            self.cur.arraysize = 1000000
            # get
            varMap = {}
            varMap[':jediTaskID'] = jediTaskID
            sql  = "SELECT oldPandaID,newPandaID FROM {0}.JEDI_Job_Retry_History ".format(panda_config.schemaJEDI)
            sql += "WHERE jediTaskID=:jediTaskID GROUP BY oldPandaID,newPandaID "
            for tmpType in EventServiceUtils.relationTypesForJS:
                tmpKey = ':{0}'.format(tmpType)
                sql += '{0},'.format(tmpKey)
                varMap[tmpKey] = tmpType
            sql  = sql[:-1]
            sql += ')) '
            self.cur.execute(sql+comment,varMap)
            resG = self.cur.fetchall()
            retMap = {}
            for oldPandaID,newPandaID in resG:
                if oldPandaID not in retMap:
                    retMap[oldPandaID] = []
                retMap[oldPandaID].append(newPandaID)
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            _logger.debug("{0} return len={1}".format(methodName,len(retMap)))
            return retMap
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            return None



    # get original consumers
    def getOriginalConsumers(self, jediTaskID, jobsetID, pandaID):
        comment = ' /* DBProxy.getOriginalConsumers */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " < jediTaskID={0} jobsetID={1} PandaID={2} >".format(jediTaskID, jobsetID, pandaID)
        tmpLog = LogWrapper(_logger,methodName)
        tmpLog.debug("start")
        try:
            # sql to get sites where consumers are active
            sqlA = "SELECT computingSite FROM ATLAS_PANDA.jobsActive4 WHERE jediTaskID=:jediTaskID AND jobsetID=:jobsetID "
            sqlA += "UNION "
            sqlA += "SELECT computingSite FROM ATLAS_PANDA.jobsDefined4 WHERE jediTaskID=:jediTaskID AND jobsetID=:jobsetID "
            # sql to get original IDs
            sqlG  = "SELECT oldPandaID FROM {0}.JEDI_Job_Retry_History ".format(panda_config.schemaJEDI)
            sqlG += "WHERE jediTaskID=:jediTaskID AND newPandaID=:jobsetID AND relationType=:relationType "
            # sql to check computingSite
            sqlC1 = "SELECT computingSite FROM ATLAS_PANDA.jobsArchived4 WHERE PandaID=:PandaID "
            sqlC2 = "SELECT computingSite FROM ATLAS_PANDAARCH.jobsArchived WHERE PandaID=:PandaID AND modificationTime>(CURRENT_DATE-30) "
            # sql to get job info
            sqlJ = "SELECT {0} ".format(JobSpec.columnNames())
            sqlJ += "FROM {0} "
            sqlJ += "WHERE PandaID=:PandaID AND modificationTime>(CURRENT_DATE-30) "
            sqlF = "SELECT {0} FROM ATLAS_PANDA.filesTable4 ".format(FileSpec.columnNames())
            sqlF += "WHERE PandaID=:PandaID "
            sqlP = "SELECT jobParameters FROM {0} WHERE PandaID=:PandaID "
            # get sites
            aSites = set()
            varMap = dict()
            varMap[':jediTaskID'] = jediTaskID
            varMap[':jobsetID'] = jobsetID
            self.cur.execute(sqlA+comment, varMap)
            resA = self.cur.fetchall()
            for computingSite, in resA:
                aSites.add(computingSite)
            # get original IDs
            varMap = dict()
            varMap[':jediTaskID'] = jediTaskID
            varMap[':jobsetID'] = jobsetID
            varMap[':relationType'] = EventServiceUtils.relationTypeJS_Map
            self.cur.execute(sqlG+comment, varMap)
            resG = self.cur.fetchall()
            jobList = []
            for pandaID, in resG:
                # check computingSite
                varMap = dict()
                varMap[':PandaID'] = pandaID
                self.cur.execute(sqlC1+comment, varMap)
                resC = self.cur.fetchone()
                if resC is None:
                    # try archived
                    self.cur.execute(sqlC2+comment, varMap)
                    resC = self.cur.fetchone()
                    inArchived = True
                else:
                    inArchived = False
                # skip since it is not yet archived and thus is still active
                if resC is None:
                    continue
                computingSite, = resC
                # skip since there is an active consumer at the site
                if computingSite in aSites:
                    continue
                # get job
                if inArchived:
                    self.cur.execute(sqlJ.format('ATLAS_PANDAARCH.jobsArchived')+comment, varMap)
                else:
                    self.cur.execute(sqlJ.format('ATLAS_PANDA.jobsArchived4')+comment, varMap)
                resJ = self.cur.fetchone()
                if resJ is not None:
                    jobSpec = JobSpec()
                    jobSpec.pack(resJ)
                    # get files
                    self.cur.execute(sqlF+comment, varMap)
                    resFs = self.cur.fetchall()
                    if len(resFs) == 0:
                        continue
                    for resF in resFs:
                        fileSpec = FileSpec()
                        fileSpec.pack(resF)
                        jobSpec.addFile(fileSpec)
                    # get job params
                    if inArchived:
                        self.cur.execute(sqlP.format('ATLAS_PANDAARCH.jobParamsTable_ARCH')+comment, varMap)
                    else:
                        self.cur.execute(sqlP.format('ATLAS_PANDA.jobParamsTable')+comment, varMap)
                    for clobJobP, in self.cur:
                        if clobJobP is not None:
                            try:
                                jobSpec.jobParameters = clobJobP.read()
                            except AttributeError:
                                jobSpec.jobParameters = str(clobJobP)
                        break
                    # add
                    jobList.append(jobSpec)
                    aSites.add(computingSite)
            tmpLog.debug("got {0} consumers".format(len(jobList)))
            return jobList
        except Exception:
            # error
            self.dumpErrorMessage(_logger,methodName)
            return []



    # wake up connection
    def wakeUp(self):
        for iTry in range(5):
            try:
                # check if the connection is working
                self.conn.ping()
                return
            except Exception:
                type, value, traceBack = sys.exc_info()
                _logger.debug("wakeUp %d : %s %s" % (iTry,type,value))
                # wait for reconnection
                time.sleep(1)
                self.connect(reconnect=True)


    # commit
    def _commit(self):
        try:
            self.conn.commit()
            return True
        except Exception:
            _logger.error("commit error")
            return False


    # rollback
    def _rollback(self,useOtherError=False):
        retVal = True
        # rollback
        _logger.debug("rollback")
        try:
            self.conn.rollback()
        except Exception:
            _logger.error("rollback error")
            retVal = False
        # reconnect if needed
        try:
            # get ORA ErrorCode
            errType,errValue = sys.exc_info()[:2]
            oraErrCode = str(errValue).split()[0]
            oraErrCode = oraErrCode[:-1]
            errMsg = "rollback EC:%s %s" % (oraErrCode,errValue)
            _logger.debug(errMsg)
            # error codes for connection error
            if self.backend == 'oracle':
                error_Codes  = ['ORA-01012','ORA-01033','ORA-01034','ORA-01089',
                                'ORA-03113','ORA-03114','ORA-12203','ORA-12500',
                                'ORA-12571','ORA-03135','ORA-25402']
                # other errors are apperantly given when connection lost contact
                if useOtherError:
                    error_Codes += ['ORA-01861','ORA-01008']
            else:
                # mysql error codes for connection error
                from MySQLdb.constants.ER import ACCESS_DENIED_ERROR, DBACCESS_DENIED_ERROR, \
                    SERVER_SHUTDOWN, ILLEGAL_VALUE_FOR_TYPE
                from MySQLdb.constants.CR import CONNECTION_ERROR, CONN_HOST_ERROR, \
                    LOCALHOST_CONNECTION, SERVER_LOST
                error_Codes = [
                    ACCESS_DENIED_ERROR, DBACCESS_DENIED_ERROR,
                    SERVER_SHUTDOWN,
                    CONNECTION_ERROR, CONN_HOST_ERROR, LOCALHOST_CONNECTION,
                    SERVER_LOST
                    ]
                # other errors are apperantly given when connection lost contact
                if useOtherError:
                    error_Codes += [
                        ILLEGAL_VALUE_FOR_TYPE
                        ]
            if oraErrCode in error_Codes:
                # reconnect
                retFlag = self.connect(reconnect=True)
                _logger.debug("rollback reconnected %s" % retFlag)
        except Exception:
            pass
        # return
        return retVal



    # dump error message
    def dumpErrorMessage(self,tmpLog,methodName):
        # error
        errtype,errvalue = sys.exc_info()[:2]
        errStr = "{0}: {1} {2}".format(methodName,errtype.__name__,errvalue)
        errStr.strip()
        errStr += ' '
        errStr += traceback.format_exc()
        tmpLog.error(errStr)



    # get DEFT schema
    def getSchemaDEFT(self):
        if not hasattr(panda_config,'schemaDEFT'):
            return "ATLAS_DEFT"
        return panda_config.schemaDEFT



    # get working group with production role
    def getWorkingGroup(self,fqans):
        for fqan in fqans:
            # check production role
            match = re.search('/[^/]+/([^/]+)/Role=production',fqan)
            if match is not None:
                return match.group(1)
        return None


    # insert TaskParams
    def insertTaskParamsPanda(self,taskParams,dn,prodRole,fqans,parent_tid,properErrorCode=False,
                              allowActiveTask=False):
        comment = ' /* JediDBProxy.insertTaskParamsPanda */'
        try:
            methodName = "insertTaskParamsPanda"
            # get compact DN
            compactDN = self.cleanUserID(dn)
            if compactDN in ['','NULL',None]:
                compactDN = dn
            methodName += ' <{0}>'.format(compactDN)
            _logger.debug('{0} start'.format(methodName))
            # decode json
            taskParamsJson = PrioUtil.decodeJSON(taskParams)
            # set user name
            if not prodRole or 'userName' not in taskParamsJson:
                taskParamsJson['userName'] = compactDN
            # identify parent
            if 'parentTaskName' in taskParamsJson:
                parent_tid = self.get_parent_task_id_with_name(taskParamsJson['userName'],
                                                               taskParamsJson['parentTaskName'])
                if not parent_tid:
                    tmpMsg = 'failed to find parent with user="{}" name={}'.format(taskParamsJson['userName'],
                                                                                   taskParamsJson['parentTaskName'])
                    _logger.debug('{} {}'.format(methodName, tmpMsg))
                    return 11, tmpMsg
                else:
                    _logger.debug('{} found parent {} with user="{}" name={}'.format(methodName, parent_tid,
                                                                                     taskParamsJson['userName'],
                                                                                     taskParamsJson['parentTaskName']))
            # set task type
            if not prodRole or 'taskType' not in taskParamsJson:
                taskParamsJson['taskType']   = 'anal'
                taskParamsJson['taskPriority'] = 1000
                # extract working group
                if 'official' in taskParamsJson and taskParamsJson['official'] is True:
                    workingGroup = self.getWorkingGroup(fqans)
                    if workingGroup is not None:
                        taskParamsJson['workingGroup'] = workingGroup

            _logger.debug('{0} taskName={1}'.format(methodName,taskParamsJson['taskName']))
            schemaDEFT = self.getSchemaDEFT()
            # sql to check task duplication for user
            sqlTDU  = "SELECT jediTaskID,status FROM {0}.JEDI_Tasks ".format(panda_config.schemaJEDI)
            sqlTDU += "WHERE vo=:vo AND prodSourceLabel=:prodSourceLabel AND userName=:userName AND taskName=:taskName "
            sqlTDU += "ORDER BY jediTaskID DESC FOR UPDATE "
            # sql to check task duplication for group
            sqlTDW  = "SELECT jediTaskID,status FROM {0}.JEDI_Tasks ".format(panda_config.schemaJEDI)
            sqlTDW += "WHERE vo=:vo AND prodSourceLabel=:prodSourceLabel AND taskName=:taskName "
            sqlTDW += "ORDER BY jediTaskID DESC FOR UPDATE "
            # sql to check DEFT table for user
            sqlCDU  = "SELECT taskid FROM {0}.T_TASK ".format(schemaDEFT)
            sqlCDU += "WHERE vo=:vo AND prodSourceLabel=:prodSourceLabel AND userName=:userName AND taskName=:taskName "
            sqlCDU += "ORDER BY taskid DESC FOR UPDATE "
            # sql to check DEFT table for group
            sqlCDW  = "SELECT taskid FROM {0}.T_TASK ".format(schemaDEFT)
            sqlCDW += "WHERE vo=:vo AND prodSourceLabel=:prodSourceLabel AND taskName=:taskName "
            sqlCDW += "ORDER BY taskid DESC FOR UPDATE "
            # sql to insert task parameters
            sqlT  = "INSERT INTO {0}.T_TASK ".format(schemaDEFT)
            sqlT += "(taskid,status,submit_time,vo,prodSourceLabel,userName,taskName,jedi_task_parameters,priority,current_priority,parent_tid) VALUES "
            varMap = {}
            if self.backend in ['oracle', 'postgres']:
                sqlT += "({0}.PRODSYS2_TASK_ID_SEQ.nextval,".format(schemaDEFT)
            else:
                #panda_config.backend == 'mysql':
                ### fake sequence
                sql = " INSERT INTO PRODSYS2_TASK_ID_SEQ (col) VALUES (NULL) "
                self.cur.arraysize = 100
                self.cur.execute(sql + comment, {})
                sql2 = """ SELECT LAST_INSERT_ID() """
                self.cur.execute(sql2 + comment, {})
                nextval, = self.cur.fetchone()
                sqlT += "( :nextval ,".format(schemaDEFT)
                varMap[':nextval'] = nextval
            sqlT += ":status,CURRENT_DATE,:vo,:prodSourceLabel,:userName,:taskName,:param,:priority,:current_priority,"
            if parent_tid is None:
                if self.backend in ['oracle', 'postgres']:
                    sqlT += "{0}.PRODSYS2_TASK_ID_SEQ.currval) ".format(schemaDEFT)
                else:
                    #panda_config.backend == 'mysql':
                    ### fake sequence
                    sql = " SELECT MAX(COL) FROM PRODSYS2_TASK_ID_SEQ "
                    self.cur.arraysize = 100
                    self.cur.execute(sql + comment, {})
                    currval, = self.cur.fetchone()
                    sqlT += " :currval ) "
                    varMap[':currval'] = currval
            else:
                sqlT += ":parent_tid) "
            sqlT += "RETURNING TASKID INTO :jediTaskID"
            # sql to delete command
            sqlDC  = "DELETE FROM {0}.PRODSYS_COMM ".format(schemaDEFT)
            sqlDC += "WHERE COMM_TASK=:jediTaskID "
            # sql to insert command
            sqlIC  = "INSERT INTO {0}.PRODSYS_COMM (COMM_TASK,COMM_OWNER,COMM_CMD,COMM_PARAMETERS) ".format(schemaDEFT)
            sqlIC += "VALUES (:jediTaskID,:comm_owner,:comm_cmd,:comm_parameters) "
            max_n_tasks = self.getConfigValue('dbproxy',
                                              'MAX_ACTIVE_TASKS_PER_USER_{}'.format(taskParamsJson['prodSourceLabel']))
            # begin transaction
            self.conn.begin()
            # check max
            if max_n_tasks is not None:
                sqlTOT = ("SELECT COUNT(*) "
                          "FROM {0}.JEDI_Tasks tabT,{0}.JEDI_AUX_Status_MinTaskID tabA "
                          "WHERE tabT.status=tabA.status AND tabT.jediTaskID>=tabA.min_jediTaskID "
                          "AND tabT.prodSourceLabel=:prodSourceLabel AND tabT.userName=:userName "
                          "AND tabT.status IN (").format(panda_config.schemaJEDI)
                varMapTot = {}
                varMapTot[':prodSourceLabel'] = taskParamsJson['prodSourceLabel']
                varMapTot[':userName'] = taskParamsJson['userName']
                for st in ['registered', 'defined', 'ready', 'scouting', 'running', 'paused', 'throttled']:
                    key = ':{}'.format(st)
                    sqlTOT += '{},'.format(key)
                    varMapTot[key] = st
                sqlTOT = sqlTOT[:-1]
                sqlTOT += ') '
                self.cur.execute(sqlTOT + comment, varMapTot)
                resTOT = self.cur.fetchone()
                if resTOT is not None and resTOT[0] > max_n_tasks:
                    # commit
                    if not self._commit():
                        raise RuntimeError('Commit error')
                    tmpMsg = "Too many active tasks for {} {}>{}".format(taskParamsJson['userName'],
                                                                         resTOT[0], max_n_tasks)
                    _logger.debug('{} {}'.format(methodName, tmpMsg))
                    return 10, tmpMsg
            # check duplication
            goForward = True
            retFlag = False
            retVal = None
            errorCode = 0
            if taskParamsJson['taskType'] == 'anal' and \
                    (('uniqueTaskName' in taskParamsJson and taskParamsJson['uniqueTaskName'] is True) or
                     allowActiveTask):
                if 'official' in taskParamsJson and taskParamsJson['official'] is True:
                    isOfficial = True
                else:
                    isOfficial = False
                # check JEDI
                varMap[':vo'] = taskParamsJson['vo']
                if isOfficial:
                    pass
                else:
                    varMap[':userName'] = taskParamsJson['userName']
                varMap[':taskName'] = taskParamsJson['taskName']
                varMap[':prodSourceLabel'] = taskParamsJson['prodSourceLabel']
                if isOfficial:
                    self.cur.execute(sqlTDW+comment,varMap)
                else:
                    self.cur.execute(sqlTDU+comment,varMap)
                resDT = self.cur.fetchone()
                if resDT is None:
                    # check DEFT table
                    varMap = {}
                    varMap[':vo'] = taskParamsJson['vo']
                    if isOfficial:
                        pass
                    else:
                        varMap[':userName'] = taskParamsJson['userName']
                    varMap[':taskName'] = taskParamsJson['taskName']
                    varMap[':prodSourceLabel'] = taskParamsJson['prodSourceLabel']
                    if isOfficial:
                        self.cur.execute(sqlCDW+comment,varMap)
                    else:
                        self.cur.execute(sqlCDU+comment,varMap)
                    resCD = self.cur.fetchone()
                    if resCD is not None:
                        # task is already in DEFT
                        jediTaskID, = resCD
                        _logger.debug('{0} old jediTaskID={1} with taskName={2} in DEFT table'.format(methodName,jediTaskID,
                                                                                                      varMap[':taskName']))
                        goForward = False
                        retVal  = 'jediTaskID={0} is already queued for outDS={1}. '.format(jediTaskID,
                                                                                            taskParamsJson['taskName'])
                        retVal += 'You cannot submit duplicated tasks. '
                        _logger.debug('{0} skip since old task is already queued in DEFT'.format(methodName))
                        errorCode = 1
                else:
                    # task is already in JEDI table
                    jediTaskID,taskStatus = resDT
                    _logger.debug('{0} old jediTaskID={1} with taskName={2} in status={3}'.format(methodName,jediTaskID,
                                                                                                  varMap[':taskName'],taskStatus))
                    # check task status
                    if taskStatus not in ['finished','failed','aborted','done', 'exhausted'] and \
                            not (allowActiveTask and taskStatus in ['running','scouting','pending'] and taskParamsJson['prodSourceLabel'] in ['user']):
                        # still active
                        goForward = False
                        retVal  = 'jediTaskID={0} is in the {1} state for outDS={2}. '.format(jediTaskID,
                                                                                              taskStatus,
                                                                                              taskParamsJson['taskName'])
                        retVal += 'You can re-submit the task with new parameters for the same or another input '
                        retVal += 'once it goes into finished/failed/done. '
                        retVal += 'Or you can retry the task once it goes into running/finished/failed/done. '
                        retVal += 'Note that retry != resubmission accoring to '
                        retVal += 'https://twiki.cern.ch/twiki/bin/view/PanDA/PandaJEDI#Task_retry_and_resubmission '
                        _logger.debug('{0} skip since old task is not yet finalized'.format(methodName))
                        errorCode = 2
                    else:
                        # extract several params for incremental execution
                        newTaskParams = {}
                        newRamCount = None
                        for tmpKey in taskParamsJson:
                            tmpVal = taskParamsJson[tmpKey]
                            # dataset names
                            # site limitation
                            # command line parameters
                            # splitting hints
                            # fixed source code
                            if tmpKey.startswith('dsFor') \
                                    or tmpKey in ['site','cloud','includedSite','excludedSite',
                                                  'cliParams','nFilesPerJob','nFiles','nEvents',
                                                  'nGBPerJob','fixedSandbox','ignoreMissingInDS',
                                                  'currentPriority', 'priority', 'nMaxFilesPerJob',
                                                  'ramCount', 'noLoopingCheck', 'loopingCheck',
                                                  'avoidVP']:
                                if tmpKey == 'priority':
                                    tmpKey = 'currentPriority'
                                if tmpKey == 'loopingCheck':
                                    tmpKey == 'noLoopingCheck'
                                    if tmpVal:
                                        tmpVal = False
                                    else:
                                        tmpVal = True
                                newTaskParams[tmpKey] = tmpVal
                                if tmpKey == 'fixedSandbox' and 'sourceURL' in taskParamsJson:
                                    newTaskParams['sourceURL'] = taskParamsJson['sourceURL']
                                elif tmpKey == 'ramCount':
                                    newRamCount = tmpVal
                        # send command to reactivate the task
                        if not allowActiveTask or taskStatus in ['finished','failed','aborted','done','exhausted']:
                            # set new RAM count
                            if newRamCount is not None:
                                sqlRAM = "UPDATE {0}.JEDI_Dataset_Contents SET ramCount=:ramCount ".format(panda_config.schemaJEDI)
                                sqlRAM += "WHERE jediTaskID=:jediTaskID AND (ramCount IS NOT NULL AND ramCount>:ramCount) "
                                sqlRAM += "AND datasetID IN (SELECT datasetID FROM {0}.JEDI_Datasets ".format(panda_config.schemaJEDI)
                                sqlRAM += "WHERE jediTaskID=:jediTaskID AND type IN (:type1,:type2)) "
                                varMap = {}
                                varMap[':jediTaskID'] = jediTaskID
                                varMap[':type1'] = 'input'
                                varMap[':type2'] = 'pseudo_input'
                                varMap[':ramCount'] = newRamCount
                                self.cur.execute(sqlRAM+comment, varMap)
                                sqlRAMT = "UPDATE {0}.JEDI_Tasks SET ramCount=:ramCount WHERE jediTaskID=:jediTaskID ".format(panda_config.schemaJEDI)
                                varMap = {}
                                varMap[':jediTaskID'] = jediTaskID
                                varMap[':ramCount'] = newRamCount
                                self.cur.execute(sqlRAMT+comment, varMap)
                            # delete command just in case
                            varMap = {}
                            varMap[':jediTaskID'] = jediTaskID
                            self.cur.execute(sqlDC+comment,varMap)
                            # insert command
                            varMap = {}
                            varMap[':jediTaskID'] = jediTaskID
                            varMap[':comm_cmd']  = 'incexec'
                            varMap[':comm_owner']  = 'DEFT'
                            varMap[':comm_parameters'] = json.dumps(newTaskParams)
                            self.cur.execute(sqlIC+comment,varMap)
                            _logger.info('{0} {1} jediTaskID={2} with {3}'.format(methodName,varMap[':comm_cmd'],
                                                                                  jediTaskID,str(newTaskParams)))
                            retVal  = 'reactivation accepted. '
                            retVal += 'jediTaskID={0} (currently in {1} state) will be re-executed with old and/or new input'.format(jediTaskID,
                                                                                                                                    taskStatus)
                            errorCode = 3
                        else:
                            # sql to read task params
                            sqlTP = "SELECT taskParams FROM {0}.JEDI_TaskParams WHERE jediTaskID=:jediTaskID ".format(panda_config.schemaJEDI)
                            varMap = {}
                            varMap[':jediTaskID'] = jediTaskID
                            self.cur.execute(sqlTP+comment,varMap)
                            tmpStr = ''
                            for tmpItem, in self.cur:
                                try:
                                    tmpStr = tmpItem.read()
                                except AttributeError:
                                    tmpStr = str(tmpItem)
                                break
                            # decode json
                            taskParamsJson = json.loads(tmpStr)
                            # just change some params for active task
                            for tmpKey in newTaskParams:
                                tmpVal = newTaskParams[tmpKey]
                                taskParamsJson[tmpKey] = tmpVal
                            # update params
                            sqlTU  = "UPDATE {0}.JEDI_TaskParams SET taskParams=:taskParams ".format(panda_config.schemaJEDI)
                            sqlTU += "WHERE jediTaskID=:jediTaskID "
                            varMap = {}
                            varMap[':jediTaskID'] = jediTaskID
                            varMap[':taskParams'] = json.dumps(taskParamsJson)
                            self.cur.execute(sqlTU+comment,varMap)
                            _logger.debug('{0} add new params for jediTaskID={1} with {2}'.format(methodName,
                                                                                                  jediTaskID,str(newTaskParams)))
                            retVal  = '{0}. new tasks params have been set to jediTaskID={1}. '.format(taskStatus,jediTaskID)
                            errorCode = 5
                        goForward = False
                        retFlag = True
            if goForward:
                # insert task parameters
                taskParams = json.dumps(taskParamsJson)
                varMap = {}
                varMap[':param']  = taskParams
                varMap[':status'] = 'waiting'
                varMap[':vo']       = taskParamsJson['vo']
                varMap[':userName'] = taskParamsJson['userName']
                varMap[':taskName'] = taskParamsJson['taskName']
                if parent_tid is not None:
                    varMap[':parent_tid']  = parent_tid
                varMap[':prodSourceLabel'] = taskParamsJson['prodSourceLabel']
                varMap[':jediTaskID'] = self.cur.var(varNUMBER)
                if 'taskPriority' in taskParamsJson:
                    varMap[':priority'] = taskParamsJson['taskPriority']
                else:
                    varMap[':priority'] = 100
                varMap[':current_priority'] = varMap[':priority']
                self.cur.execute(sqlT+comment,varMap)
                val = self.getvalue_corrector(self.cur.getvalue(varMap[':jediTaskID']))
                jediTaskID = long(val)
                if properErrorCode:
                    retVal = "succeeded. new jediTaskID={0}".format(jediTaskID)
                else:
                    retVal = jediTaskID
                _logger.debug('{0} inserted new jediTaskID={1}'.format(methodName,jediTaskID))
                retFlag = True
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            _logger.debug('{0} done'.format(methodName))
            if properErrorCode:
                return errorCode,retVal
            return retFlag,retVal
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            errorCode = 4
            retVal = 'failed to register task'
            if properErrorCode:
                return errorCode,retVal
            return False,retVal



    # send command to task through DEFT
    def sendCommandTaskPanda(self,jediTaskID,dn,prodRole,comStr,comComment=None,useCommit=True,properErrorCode=False,
                             comQualifier=None, broadcast=False):
        comment = ' /* JediDBProxy.sendCommandTaskPanda */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += ' < jediTaskID={0} >'.format(jediTaskID)
        tmpLog = LogWrapper(_logger, methodName)
        try:
            # get compact DN
            compactDN = self.cleanUserID(dn)
            if compactDN in ['','NULL',None]:
                compactDN = dn
            tmpLog.debug("start com={} DN={} prod={} comment={} qualifier={} broadcast={}".format(
                comStr, compactDN, prodRole, comComment, comQualifier, broadcast))
            # sql to check status and owner
            sqlTC  = "SELECT status,userName,prodSourceLabel FROM {0}.JEDI_Tasks ".format(panda_config.schemaJEDI)
            sqlTC += "WHERE jediTaskID=:jediTaskID FOR UPDATE "
            # sql to delete command
            schemaDEFT = self.getSchemaDEFT()
            sqlT  = "DELETE FROM {0}.PRODSYS_COMM ".format(schemaDEFT)
            sqlT += "WHERE COMM_TASK=:jediTaskID "
            # sql to insert command
            sqlC  = "INSERT INTO {0}.PRODSYS_COMM (COMM_TASK,COMM_OWNER,COMM_CMD,COMM_COMMENT) ".format(schemaDEFT)
            sqlC += "VALUES (:jediTaskID,:comm_owner,:comm_cmd,:comm_comment) "
            goForward = True
            retStr = ''
            retCode = 0
            sendMsgToPilot = False
            # begin transaction
            if useCommit:
                self.conn.begin()
            # get task status and owner
            varMap = {}
            varMap[':jediTaskID'] = jediTaskID
            self.cur.execute(sqlTC+comment,varMap)
            resTC = self.cur.fetchone()
            if resTC is None:
                # task not found
                retStr = 'jediTaskID={0} not found'.format(jediTaskID)
                _logger.debug("{0} : {1}".format(methodName,retStr))
                goForward = False
                retCode = 2
            else:
                taskStatus,userName,prodSourceLabel = resTC
            # check owner
            if goForward:
                if not prodRole and compactDN != userName:
                    retStr = 'Permission denied: not the task owner or no production role'
                    _logger.debug("{0} : {1}".format(methodName,retStr))
                    goForward = False
                    retCode = 3
            # check task status
            if goForward:
                if comStr in ['kill','finish']:
                    sendMsgToPilot = broadcast
                    if taskStatus in ['finished','done','prepared','broken','aborted','aborted','toabort','aborting','failed','finishing']:
                        goForward = False
                if comStr == 'retry':
                    if taskStatus not in ['finished','failed','aborted','exhausted']:
                        goForward = False
                if comStr == 'incexec':
                    if taskStatus not in ['finished','failed','done','aborted','exhausted']:
                        goForward = False
                if comStr == 'reassign':
                    if taskStatus not in ['registered','defined','ready','running','scouting','scouted','pending','assigning','exhausted']:
                        goForward = False
                if comStr == 'pause':
                    if taskStatus in ['finished','failed','done','aborted','broken','paused']:
                        goForward = False
                if comStr == 'resume':
                    if taskStatus not in ['paused','throttled', 'staging']:
                        goForward = False
                if comStr == 'avalanche':
                    if taskStatus not in ['scouting']:
                        goForward = False
                if not goForward:
                    retStr = 'Command rejected: the {0} command is not accepted if the task is in {1} status'.format(comStr,taskStatus)
                    _logger.debug("{0} : {1}".format(methodName,retStr))
                    retCode = 4
                    # retry for failed analysis jobs
                    if comStr == 'retry' and properErrorCode and taskStatus in ['running','scouting','pending'] and prodSourceLabel in ['user']:
                        retCode = 5
                        retStr = taskStatus
            if goForward:
                # delete command just in case
                varMap = {}
                varMap[':jediTaskID'] = jediTaskID
                self.cur.execute(sqlT+comment,varMap)
                # insert command
                varMap = {}
                varMap[':jediTaskID'] = jediTaskID
                varMap[':comm_cmd']  = comStr
                varMap[':comm_owner']  = 'DEFT'
                if comComment is None:
                    tmpStr = ''
                    if comQualifier not in ['',None]:
                        tmpStr += '{0} '.format(comQualifier)
                    tmpStr += '{0} by {1}'.format(comStr,compactDN)
                    varMap[':comm_comment'] = tmpStr
                else:
                    varMap[':comm_comment'] = comComment
                self.cur.execute(sqlC+comment,varMap)
                retStr = 'command={0} is registered. will be executed in a few minutes'.format(comStr)
                tmpLog.info('{}'.format(retStr))
            # commit
            if useCommit:
                if not self._commit():
                    raise RuntimeError('Commit error')
            # send command to the pilot
            if sendMsgToPilot:
                mb_proxy_topic = self.get_mb_proxy('panda_pilot_topic')
                if mb_proxy_topic:
                    tmpLog.debug('push {}'.format(comStr))
                    srv_msg_utils.send_task_message(mb_proxy_topic, comStr, jediTaskID)
                else:
                    tmpLog.debug('message topic not configured')
            if properErrorCode:
                return retCode,retStr
            else:
                if retCode == 0:
                    return True,retStr
                else:
                    return False,retStr
        except Exception:
            # roll back
            if useCommit:
                self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            if properErrorCode:
                return 1,'failed to register command'
            else:
                return False,'failed to register command'



    # update unmerged jobs
    def updateUnmergedJobs(self,job,fileIDs=None):
        comment = ' /* JediDBProxy.updateUnmergedJobs */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " <PandaID={0}>".format(job.PandaID)
        tmpLog = LogWrapper(_logger,methodName)
        # get PandaID which produced unmerged files
        umPandaIDs = []
        umCheckedIDs = []
        if fileIDs is None:
            fileIDs = set()
        # sql to get PandaIDs
        sqlUMP  = "SELECT PandaID,attemptNr FROM ATLAS_PANDA.filesTable4 "
        sqlUMP += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
        sqlUMP += "AND type IN (:type1,:type2) ORDER BY attemptNr DESC "
        # sql to check job status
        sqlUMS  = "SELECT jobStatus FROM ATLAS_PANDA.jobsActive4 WHERE PandaID=:PandaID "
        # look for unmerged files
        for tmpFile in job.Files:
            if tmpFile.isUnMergedInput():
                # only fileIDs which reach max attempt
                if len(fileIDs) > 0 and tmpFile.fileID not in fileIDs:
                    continue
                varMap = {}
                varMap[':jediTaskID'] = tmpFile.jediTaskID
                varMap[':datasetID']  = tmpFile.datasetID
                varMap[':fileID']     = tmpFile.fileID
                varMap[':type1']  = 'output'
                varMap[':type2']  = 'log'
                self.cur.arraysize = 100
                self.cur.execute(sqlUMP+comment, varMap)
                resUMP = self.cur.fetchall()
                # loop for job in merging state
                for tmpPandaID,tmpAttemptNr in resUMP:
                    # skip checked PandaIDs
                    if tmpPandaID in umCheckedIDs:
                        continue
                    # append to avoid redundant check
                    umCheckedIDs.append(tmpPandaID)
                    # check job status
                    varMap = {}
                    varMap[':PandaID'] = tmpPandaID
                    self.cur.execute(sqlUMS+comment, varMap)
                    resUMS = self.cur.fetchone()
                    # unmerged job should be in merging state
                    if resUMS is not None and resUMS[0] == 'merging':
                        # append
                        umPandaIDs.append(tmpPandaID)
                        break
        # finish unmerge jobs
        sqlJFJ  = "SELECT %s " % JobSpec.columnNames()
        sqlJFJ += "FROM ATLAS_PANDA.jobsActive4 WHERE PandaID=:PandaID"
        sqlJFF  = "SELECT %s FROM ATLAS_PANDA.filesTable4 " % FileSpec.columnNames()
        sqlJFF += "WHERE PandaID=:PandaID"
        for tmpPandaID in umPandaIDs:
            # read job
            varMap = {}
            varMap[':PandaID'] = tmpPandaID
            self.cur.arraysize = 10
            self.cur.execute(sqlJFJ+comment, varMap)
            resJFJ = self.cur.fetchone()
            umJob = JobSpec()
            umJob.pack(resJFJ)
            umJob.jobStatus = job.jobStatus
            if umJob.jobStatus in ['failed'] or umJob.isCancelled():
                umJob.taskBufferErrorCode = ErrorCode.EC_MergeFailed
                umJob.taskBufferErrorDiag = "merge job {0}".format(umJob.jobStatus)
                umJob.jobSubStatus = 'merge_{0}'.format(umJob.jobStatus)
            # read files
            self.cur.arraysize = 10000
            self.cur.execute(sqlJFF+comment, varMap)
            resJFFs = self.cur.fetchall()
            for resJFF in resJFFs:
                umFile = FileSpec()
                umFile.pack(resJFF)
                if umFile.status not in ['nooutput']:
                    umFile.status = umJob.jobStatus
                umJob.addFile(umFile)
            # finish
            tmpLog.debug('update unmerged PandaID={0}'.format(umJob.PandaID))
            self.archiveJob(umJob,False,useCommit=False)
        return



    # disable further reattempt for pmerge
    def disableFurtherReattempt(self,jobSpec):
        comment = ' /* JediDBProxy.disableFurtherReattempt */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " <PandaID={0}>".format(jobSpec.PandaID)
        # sql to update file
        sqlFJ  = "UPDATE {0}.JEDI_Dataset_Contents ".format(panda_config.schemaJEDI)
        sqlFJ += "SET maxAttempt=attemptNr-1 "
        sqlFJ += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
        sqlFJ += "AND attemptNr=:attemptNr AND keepTrack=:keepTrack "
        nRow = 0
        for tmpFile in jobSpec.Files:
            # skip if no JEDI
            if tmpFile.fileID == 'NULL':
                continue
            # only input
            if tmpFile.type not in ['input','pseudo_input']:
                continue
            # update JEDI contents
            varMap = {}
            varMap[':jediTaskID'] = tmpFile.jediTaskID
            varMap[':datasetID']  = tmpFile.datasetID
            varMap[':fileID']     = tmpFile.fileID
            varMap[':attemptNr']  = tmpFile.attemptNr
            varMap[':keepTrack']  = 1
            self.cur.execute(sqlFJ+comment,varMap)
            nRow += self.cur.rowcount
        # finish
        _logger.debug('{0} done with nRows={1}'.format(methodName,nRow))
        return



    # update unmerged datasets to trigger merging
    def updateUnmergedDatasets(self,job,finalStatusDS,updateCompleted=False):
        comment = ' /* JediDBProxy.updateUnmergedDatasets */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " <PandaID={0}>".format(job.PandaID)
        # get PandaID which produced unmerged files
        umPandaIDs = []
        umCheckedIDs = []
        # sql to get file counts
        sqlGFC  = "SELECT status,PandaID,outPandaID FROM ATLAS_PANDA.JEDI_Dataset_Contents "
        sqlGFC += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND PandaID IS NOT NULL "
        # sql to update nFiles in JEDI datasets
        sqlUNF  = "UPDATE ATLAS_PANDA.JEDI_Datasets "
        sqlUNF += "SET nFilesOnHold=0,nFiles=:nFiles,"
        sqlUNF += "nFilesUsed=:nFilesUsed,nFilesTobeUsed=:nFilesTobeUsed "
        sqlUNF += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID "
        # sql to check nFiles
        sqlUCF  = "SELECT nFilesTobeUsed,nFilesUsed FROM ATLAS_PANDA.JEDI_Datasets "
        sqlUCF += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID "
        # sql to update dataset status
        sqlUDS  = "UPDATE ATLAS_PANDA.JEDI_Datasets "
        sqlUDS += "SET status=:status,modificationTime=CURRENT_DATE "
        sqlUDS += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID "
        # sql to update dataset status in panda
        sqlUDP  = "UPDATE ATLAS_PANDA.Datasets "
        sqlUDP += "SET status=:status "
        sqlUDP += "WHERE vuid=:vuid AND NOT status IN (:statusR,:statusD) "
        try:
            _logger.debug('{0} start'.format(methodName))
            # begin transaction
            self.conn.begin()
            # update dataset in panda
            toSkip = False
            for datasetSpec in finalStatusDS:
                varMap = {}
                varMap[':vuid'] = datasetSpec.vuid
                varMap[':status'] = 'tobeclosed'
                varMap[':statusR'] = 'tobeclosed'
                if not updateCompleted:
                    varMap[':statusD'] = 'completed'
                else:
                    varMap[':statusD'] = 'dummy'
                _logger.debug(methodName+' '+sqlUDP+comment+str(varMap))
                self.cur.execute(sqlUDP+comment, varMap)
                nRow = self.cur.rowcount
                if nRow != 1:
                    toSkip = True
                    _logger.debug('{0} failed to lock {1}'.format(methodName,
                                                                  datasetSpec.name))
            # look for unmerged files
            if not toSkip:
                updatedDS = []
                for tmpFile in job.Files:
                    if tmpFile.isUnMergedOutput():
                        if tmpFile.datasetID in updatedDS:
                            continue
                        updatedDS.append(tmpFile.datasetID)
                        # get file counts
                        varMap = {}
                        varMap[':jediTaskID'] = tmpFile.jediTaskID
                        varMap[':datasetID']  = tmpFile.datasetID
                        self.cur.arraysize = 100000
                        _logger.debug(sqlGFC+comment+str(varMap))
                        self.cur.execute(sqlGFC+comment, varMap)
                        resListGFC = self.cur.fetchall()
                        varMap = {}
                        tmpNumFiles = 0
                        tmpNumReady = 0
                        for tmpFileStatus,tmpPandaID,tmpOutPandaID in resListGFC:
                            if tmpFileStatus in ['finished','failed','cancelled','notmerged',
                                                 'ready','lost','broken','picked','nooutput']:
                                pass
                            elif tmpFileStatus == 'running' and tmpPandaID != tmpOutPandaID:
                                pass
                            else:
                                continue
                            tmpNumFiles += 1
                            if tmpFileStatus in ['ready']:
                                tmpNumReady += 1
                        # update nFiles
                        varMap = {}
                        varMap[':jediTaskID'] = tmpFile.jediTaskID
                        varMap[':datasetID']  = tmpFile.datasetID
                        varMap[':nFiles'] = tmpNumFiles
                        varMap[':nFilesTobeUsed'] = tmpNumFiles
                        varMap[':nFilesUsed'] = tmpNumFiles-tmpNumReady
                        self.cur.arraysize = 10
                        _logger.debug(sqlUNF+comment+str(varMap))
                        self.cur.execute(sqlUNF+comment, varMap)
                        nRow = self.cur.rowcount
                        if nRow == 1:
                            # check nFilesTobeUsed
                            varMap = {}
                            varMap[':jediTaskID'] = tmpFile.jediTaskID
                            varMap[':datasetID']  = tmpFile.datasetID
                            self.cur.execute(sqlUCF+comment, varMap)
                            resUCF = self.cur.fetchone()
                            if resUCF is not None:
                                nFilesTobeUsed,nFilesUsed = resUCF
                                varMap = {}
                                varMap[':jediTaskID'] = tmpFile.jediTaskID
                                varMap[':datasetID']  = tmpFile.datasetID
                                if nFilesTobeUsed-nFilesUsed > 0:
                                    varMap[':status'] = 'ready'
                                else:
                                    varMap[':status'] = 'done'
                                # update dataset status
                                _logger.debug(methodName+' '+sqlUDS+comment+str(varMap))
                                self.cur.execute(sqlUDS+comment, varMap)
                        else:
                            _logger.debug('{0} skip jediTaskID={1} datasetID={2}'.format(methodName,
                                                                                         tmpFile.jediTaskID,
                                                                                         tmpFile.datasetID))
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            _logger.debug('{0} done'.format(methodName))
            return True
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            return False


    # get active JediTasks in a time range
    def getJediTasksInTimeRange(self, dn, timeRange, fullFlag=False, minTaskID=None, task_type='user'):
        comment = ' /* DBProxy.getJediTasksInTimeRange */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        _logger.debug("{0} : DN={1} range={2} full={3}".format(methodName, dn, timeRange.strftime('%Y-%m-%d %H:%M:%S'), fullFlag))
        try:
            # get compact DN
            compactDN = self.cleanUserID(dn)
            if compactDN in ['','NULL',None]:
                compactDN = dn
            # make sql
            attrList = ['jediTaskID','modificationTime','status','processingType',
                        'transUses','transHome','architecture','reqID','creationDate',
                        'site','cloud','taskName']
            sql  = 'SELECT '
            if fullFlag:
                sql += '* FROM (SELECT '
            for tmpAttr in attrList:
                sql += '{0},'.format(tmpAttr)
            sql  = sql[:-1]
            sql += " FROM {0}.JEDI_Tasks ".format(panda_config.schemaJEDI)
            sql += "WHERE userName=:userName AND modificationTime>=:modificationTime AND prodSourceLabel=:prodSourceLabel "
            varMap = {}
            varMap[':userName'] = compactDN
            varMap[':prodSourceLabel']  = task_type
            varMap[':modificationTime'] = timeRange
            if minTaskID is not None:
                sql += "AND jediTaskID>:minTaskID "
                varMap[':minTaskID'] = minTaskID
            if fullFlag:
                sql += "ORDER BY jediTaskID) WHERE rownum<=500 "
            # start transaction
            self.conn.begin()
            # select
            self.cur.arraysize = 10000
            _logger.debug(sql+comment+str(varMap))
            self.cur.execute(sql+comment, varMap)
            resList = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # append
            retTasks = {}
            for tmpRes in resList:
                tmpDict = {}
                for tmpIdx,tmpAttr in enumerate(attrList):
                    tmpDict[tmpAttr] = tmpRes[tmpIdx]
                if fullFlag:
                    # additional info
                    addInfo = self.getJediTaskDigest(tmpDict['jediTaskID'])
                    for k in addInfo:
                        v = addInfo[k]
                        tmpDict[k] = v
                retTasks[tmpDict['reqID']] = tmpDict
            _logger.debug("{0} : {1}".format(methodName,str(retTasks)))
            return retTasks
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            return {}


    # get details of JediTask
    def getJediTaskDetails(self,jediTaskID,fullFlag,withTaskInfo):
        comment = ' /* DBProxy.getJediTaskDetails */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        _logger.debug("{0} : jediTaskID={1} full={2}".format(methodName,jediTaskID,fullFlag))
        try:
            retDict = {'inDS':'','outDS':'','statistics':'','PandaID':set(),
                       'mergeStatus':None,'mergePandaID':set()}
            # sql to get task status
            sqlT  = 'SELECT status FROM {0}.JEDI_Tasks WHERE jediTaskID=:jediTaskID '.format(panda_config.schemaJEDI)
            # sql to get datasets
            sqlD  = 'SELECT datasetID,datasetName,containerName,type,nFiles,nFilesTobeUsed,nFilesFinished,nFilesFailed,masterID,nFilesUsed,nFilesOnHold '
            sqlD += 'FROM {0}.JEDI_Datasets '.format(panda_config.schemaJEDI)
            sqlD += "WHERE jediTaskID=:jediTaskID "
            # sql to get PandaIDs
            sqlP  = "SELECT PandaID,COUNT(*) FROM {0}.JEDI_Dataset_Contents ".format(panda_config.schemaJEDI)
            sqlP += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND PandaID IS NOT NULL "
            sqlP += "GROUP BY PandaID "
            # sql to get job status
            sqlJS  = "SELECT PandaID,jobStatus,processingType FROM ATLAS_PANDA.jobsDefined4 "
            sqlJS += "WHERE jediTaskID=:jediTaskID AND prodSourceLabel=:prodSourceLabel "
            sqlJS += "UNION "
            sqlJS  = "SELECT PandaID,jobStatus,processingType FROM ATLAS_PANDA.jobsActive4 "
            sqlJS += "WHERE jediTaskID=:jediTaskID AND prodSourceLabel=:prodSourceLabel "
            varMap = {}
            varMap[':jediTaskID'] = jediTaskID
            # start transaction
            self.conn.begin()
            # select
            self.cur.arraysize = 100000
            # get task status
            if withTaskInfo:
                self.cur.execute(sqlT+comment, varMap)
                resT = self.cur.fetchone()
                if resT is None:
                    raise RuntimeError('No task info')
                retDict['status'] = resT[0]
            # get datasets
            self.cur.execute(sqlD+comment, varMap)
            resList = self.cur.fetchall()
            # get job status
            varMap = {}
            varMap[':jediTaskID'] = jediTaskID
            varMap[':prodSourceLabel'] = 'user'
            self.cur.execute(sqlJS+comment, varMap)
            resJS = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # make jobstatus map
            jobStatPandaIDs = {}
            for tmpPandaID,tmpJobStatus,tmpProcessingType in resJS:
                # ignore merge jobs
                if tmpProcessingType == 'pmerge':
                    continue
                jobStatPandaIDs[tmpPandaID] = tmpJobStatus
            # append
            inDSs = []
            outDSs = []
            totalNumFiles = 0
            totalTobeDone = 0
            totalFinished = 0
            totalFailed = 0
            totalStatMap = {}
            for datasetID,datasetName,containerName,datasetType,nFiles,nFilesTobeUsed,nFilesFinished,nFilesFailed,masterID,nFilesUsed,nFilesOnHold in resList:
                # primay input
                if datasetType in ['input','pseudo_input','trn_log'] and masterID is None:
                    # unmerge dataset
                    if datasetType == 'trn_log':
                        unmergeFlag = True
                    else:
                        unmergeFlag = False
                    # collect input dataset names
                    if datasetType == 'input':
                        # use container name if not empty
                        if containerName not in [None,'']:
                            targetName = containerName
                        else:
                            targetName = datasetName
                        if targetName not in inDSs:
                            inDSs.append(targetName)
                            retDict['inDS'] += '{0},'.format(targetName)
                    # statistics
                    if datasetType in ['input','pseudo_input']:
                        totalNumFiles += nFiles
                        totalFinished += nFilesFinished
                        totalFailed   += nFilesFailed
                        totalTobeDone += (nFiles-nFilesUsed)
                    # collect PandaIDs
                    self.conn.begin()
                    varMap = {}
                    varMap[':jediTaskID'] = jediTaskID
                    varMap[':datasetID'] = datasetID
                    self.cur.execute(sqlP+comment, varMap)
                    resP = self.cur.fetchall()
                    # commit
                    if not self._commit():
                        raise RuntimeError('Commit error')
                    for tmpPandaID,tmpNumFiles in resP:
                        if not unmergeFlag:
                            retDict['PandaID'].add(tmpPandaID)
                        else:
                            retDict['mergePandaID'].add(tmpPandaID)
                        # map to job status
                        if datasetType in ['input','pseudo_input']:
                            if tmpPandaID in jobStatPandaIDs:
                                tmpJobStatus = jobStatPandaIDs[tmpPandaID]
                                if tmpJobStatus not in totalStatMap:
                                    totalStatMap[tmpJobStatus] = 0
                                totalStatMap[tmpJobStatus] += tmpNumFiles
                # output
                if datasetType.endswith('output') or datasetType.endswith('log'):
                    # ignore transient datasets
                    if 'trn_' in datasetType:
                        continue
                    # use container name if not empty
                    if containerName not in [None,'']:
                        targetName = containerName
                    else:
                        targetName = datasetName
                    if targetName not in outDSs:
                        outDSs.append(targetName)
                        retDict['outDS'] += '{0},'.format(targetName)
            retDict['inDS'] = retDict['inDS'][:-1]
            retDict['outDS'] = retDict['outDS'][:-1]
            # statistics
            statStr = ''
            nPicked = totalNumFiles
            if totalTobeDone > 0:
                statStr += 'tobedone*{0},'.format(totalTobeDone)
                nPicked -= totalTobeDone
            if totalFinished > 0:
                statStr += 'finished*{0},'.format(totalFinished)
                nPicked -= totalFinished
            if totalFailed > 0:
                statStr += 'failed*{0},'.format(totalFailed)
                nPicked -= totalFailed
            for tmpJobStatus in totalStatMap:
                tmpNumFiles = totalStatMap[tmpJobStatus]
                # skip active failed
                if tmpJobStatus == 'failed':
                    continue
                statStr += '{0}*{1},'.format(tmpJobStatus,tmpNumFiles)
                nPicked -= tmpNumFiles
            if nPicked > 0:
                statStr += 'picked*{0},'.format(nPicked)
            retDict['statistics'] = statStr[:-1]
            # command line parameters
            if fullFlag:
                # sql to read task params
                sql = "SELECT taskParams FROM {0}.JEDI_TaskParams WHERE jediTaskID=:jediTaskID ".format(panda_config.schemaJEDI)
                varMap = {}
                varMap[':jediTaskID'] = jediTaskID
                # begin transaction
                self.conn.begin()
                self.cur.execute(sql+comment,varMap)
                retStr = ''
                for tmpItem, in self.cur:
                    try:
                        retStr = tmpItem.read()
                    except AttributeError:
                        retStr = str(tmpItem)
                    break
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                # decode json
                taskParamsJson = json.loads(retStr)
                if 'cliParams' in taskParamsJson:
                    retDict['cliParams'] = taskParamsJson['cliParams']
                else:
                    retDict['cliParams'] = ''
            retDict['PandaID'] = list(retDict['PandaID'])
            retDict['mergePandaID'] = list(retDict['mergePandaID'])
            _logger.debug("{0} : {1}".format(methodName,str(retDict)))
            return retDict
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            return {}



    # get JediTask digest
    def getJediTaskDigest(self, jediTaskID):
        comment = ' /* DBProxy.getJediTaskDigest */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " < jediTaskID={0} >".format(jediTaskID)
        tmpLog = LogWrapper(_logger,methodName)
        try:
            retDict = {'inDS':'','outDS':'','statistics':'','PandaID': [],
                       'mergeStatus':None,'mergePandaID': []}
            # sql to get datasets
            sqlD  = 'SELECT datasetName,containerName,type '
            sqlD += 'FROM {0}.JEDI_Datasets '.format(panda_config.schemaJEDI)
            sqlD += "WHERE jediTaskID=:jediTaskID AND ((type IN (:in1,:in2) AND masterID IS NULL) OR type IN (:out1,:out2)) "
            sqlD += "GROUP BY datasetName,containerName,type "
            # sql to get job status
            sqlJS  = "SELECT proc_status,COUNT(*) FROM {0}.JEDI_Datasets d,{0}.JEDI_Dataset_Contents c ".format(panda_config.schemaJEDI)
            sqlJS += "WHERE c.jediTaskID=d.jediTaskID AND c.datasetID=d.datasetID AND d.jediTaskID=:jediTaskID "
            sqlJS += "AND d.type IN (:in1,:in2) AND d.masterID IS NULL "
            sqlJS += "GROUP BY proc_status "
            # sql to read task params
            sqlTP = "SELECT taskParams FROM {0}.JEDI_TaskParams WHERE jediTaskID=:jediTaskID ".format(panda_config.schemaJEDI)
            # start transaction
            self.conn.begin()
            self.cur.arraysize = 100000
            # get datasets
            inDSs = set()
            outDSs = set()
            varMap = {}
            varMap[':jediTaskID'] = jediTaskID
            varMap[':in1'] = 'input'
            varMap[':in2'] = 'pseudo_input'
            varMap[':out1'] = 'output'
            varMap[':out2'] = 'tmpl_output'
            self.cur.execute(sqlD+comment, varMap)
            resList = self.cur.fetchall()
            for datasetName, containerName, datasetType in resList:
                # use container name if not empty
                if containerName not in [None,'']:
                    targetName = containerName
                else:
                    targetName = datasetName
                if 'output' in datasetType:
                    outDSs.add(targetName)
                else:
                    inDSs.add(targetName)
            inDSs = list(inDSs)
            inDSs.sort()
            retDict['inDS'] = ','.join(inDSs)
            outDSs = list(outDSs)
            outDSs.sort()
            retDict['outDS'] = ','.join(outDSs)
            # get job status
            varMap = {}
            varMap[':jediTaskID'] = jediTaskID
            varMap[':in1'] = 'input'
            varMap[':in2'] = 'pseudo_input'
            self.cur.execute(sqlJS+comment, varMap)
            resJS = self.cur.fetchall()
            jobStatMap = dict()
            for proc_status, ninputs in resJS:
                jobStatMap[proc_status] = ninputs
            psList = list(jobStatMap)
            psList.sort()
            retDict['statistics'] = ','.join(['{0}*{1}'.format(j, jobStatMap[j]) for j in psList])
            # command line parameters
            varMap = {}
            varMap[':jediTaskID'] = jediTaskID
            self.cur.execute(sqlTP+comment, varMap)
            retStr = ''
            for tmpItem, in self.cur:
                try:
                    retStr = tmpItem.read()
                except AttributeError:
                    retStr = str(tmpItem)
                break
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # decode json
            taskParamsJson = json.loads(retStr)
            if 'cliParams' in taskParamsJson:
                retDict['cliParams'] = taskParamsJson['cliParams']
            else:
                retDict['cliParams'] = ''
            tmpLog.debug("{0}".format(str(retDict)))
            return retDict
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            return {}



    # make event range ID for event service
    def makeEventRangeID(self,jediTaskID,pandaID,fileID,job_processID,attemptNr):
        return '{0}-{1}-{2}-{3}-{4}'.format(jediTaskID,pandaID,
                                            fileID,job_processID,
                                            attemptNr)



    # get a list of even ranges for a PandaID
    def getEventRanges(self, pandaID, jobsetID, jediTaskID, nRanges, acceptJson, scattered, segment_id):
        comment = ' /* DBProxy.getEventRanges */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " <PandaID={0} jobsetID={1} jediTaskID={2}>".format(pandaID, jobsetID, jediTaskID)
        tmpLog = LogWrapper(_logger,methodName)
        tmpLog.debug("start nRanges={0} scattered={1} segment={2}".format(nRanges, scattered, segment_id))
        try:
            regStart = datetime.datetime.utcnow()
            # convert to int
            try:
                nRanges = int(nRanges)
            except Exception:
                nRanges = 8
            try:
                pandaID = long(pandaID)
            except Exception:
                pass
            try:
                jobsetID = long(jobsetID)
            except Exception:
                pass
            try:
                jediTaskID = long(jediTaskID)
            except Exception:
                jediTaskID = None
            iRanges = 0
            # sql to get job
            sqlJ  = "SELECT jobStatus,commandToPilot,eventService,jediTaskID FROM {0}.jobsActive4 ".format(panda_config.schemaPANDA)
            sqlJ += "WHERE PandaID=:pandaID FOR UPDATE "
            # sql to find a file to lock
            sqlFF = "SELECT jediTaskID,datasetID,fileID FROM {0}.filesTable4 ".format(panda_config.schemaPANDA)
            sqlFF += "WHERE PandaID=:pandaID AND type IN (:type1,:type2) "
            sqlFF += "ORDER BY fileID "
            # sql to use a dataset as lock
            sqlLD = "SELECT status FROM {0}.JEDI_Datasets ".format(panda_config.schemaJEDI)
            sqlLD += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID "
            sqlLD += "FOR UPDATE "
            # sql to use a file as lock
            sqlLK = "SELECT status FROM {0}.JEDI_Dataset_Contents ".format(panda_config.schemaJEDI)
            sqlLK += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
            sqlLK += "FOR UPDATE "
            # sql to get ranges with jediTaskID
            sqlW  = "UPDATE {0}.JEDI_Events tab ".format(panda_config.schemaJEDI)
            sqlW += "SET PandaID=:pandaID,status=:newEventStatus "
            sqlW += "WHERE (jediTaskID,PandaID,fileID,job_processID,attemptNr) IN ("
            sqlW += "SELECT jediTaskID,PandaID,fileID,job_processID,attemptNr FROM ("
            sqlW += "SELECT jediTaskID,PandaID,fileID,job_processID,attemptNr FROM "
            sqlW += "/* sorted by JEDITASKID, PANDAID, FILEID to take advantage of the IOT table structure*/ "
            sqlW += "{0}.JEDI_Events tab ".format(panda_config.schemaJEDI)
            sqlW += "WHERE jediTaskID=:jediTaskID AND PandaID=:jobsetID AND status=:eventStatus AND attemptNr>:minAttemptNr "
            if segment_id is not None:
                sqlW += "AND datasetID=:datasetID "
            sqlW += "ORDER BY jediTaskID,PandaID,fileID "
            sqlW += ") WHERE rownum<={0}) ".format(nRanges+1)
            # sql to get ranges for jumbo
            sqlJM  = "UPDATE {0}.JEDI_Events tab ".format(panda_config.schemaJEDI)
            sqlJM += "SET PandaID=:pandaID,status=:newEventStatus "
            sqlJM += "WHERE (jediTaskID,PandaID,fileID,job_processID,attemptNr) IN ("
            sqlJM += "SELECT jediTaskID,PandaID,fileID,job_processID,attemptNr FROM ("
            sqlJM += "SELECT jediTaskID,PandaID,fileID,job_processID,attemptNr FROM "
            sqlJM += "/* sorted by JEDITASKID, PANDAID, FILEID to take advantage of the IOT table structure*/ "
            sqlJM += "{0}.JEDI_Events tab ".format(panda_config.schemaJEDI)
            sqlJM += "WHERE jediTaskID=:jediTaskID AND status=:eventStatus AND attemptNr>:minAttemptNr "
            if scattered:
                pass
            else:
                sqlJM += "ORDER BY jediTaskID,PandaID,fileID "
            sqlJM += ") WHERE rownum<={0}) ".format(nRanges+1)
            # sql to get ranges
            sqlRR = 'SELECT jediTaskID,datasetID,fileID,attemptNr,job_processID,def_min_eventID,def_max_eventID,event_offset '
            sqlRR += "FROM {0}.JEDI_Events tab ".format(panda_config.schemaJEDI)
            sqlRR += "WHERE jediTaskID=:jediTaskID AND PandaID=:PandaID AND status=:eventStatus "
            # sql to get datasets
            sqlGD  = "SELECT datasetID FROM {0}.JEDI_Datasets ".format(panda_config.schemaJEDI)
            sqlGD += "WHERE jediTaskID=:jediTaskID AND type IN (:type1,:type2) "
            # sql to update files in the jobset
            sqlJS  = "UPDATE {0}.JEDI_Dataset_Contents ".format(panda_config.schemaJEDI)
            sqlJS += "SET status=:newStatus,is_waiting=NULL "
            sqlJS += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID "
            sqlJS += "AND status=:oldStatus AND keepTrack=:keepTrack AND PandaID IN ("
            # sql to update dataset
            sqlUD  = "UPDATE {0}.JEDI_Datasets ".format(panda_config.schemaJEDI)
            sqlUD += "SET nFilesUsed=nFilesUsed+:nDiff,nFilesWaiting=nFilesWaiting-:nDiff "
            sqlUD += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID "
            # sql to get file info
            sqlF  = "SELECT lfn,GUID,scope FROM {0}.JEDI_Dataset_Contents ".format(panda_config.schemaJEDI)
            sqlF += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
            # sql to lock range
            sqlU  = "UPDATE {0}.JEDI_Events ".format(panda_config.schemaJEDI)
            sqlU += "SET status=:eventStatus,is_jumbo=:isJumbo "
            sqlU += "WHERE jediTaskID=:jediTaskID AND PandaID=:pandaID "
            sqlU += "AND status=:oldEventStatus "
            # sql to release range
            sqlRS  = "UPDATE {0}.JEDI_Events ".format(panda_config.schemaJEDI)
            sqlRS += "SET PandaID=event_offset,status=:eventStatus "
            sqlRS += "WHERE jediTaskID=:jediTaskID AND fileID=:fileID AND PandaID=:pandaID "
            sqlRS += "AND job_processID=:job_processID AND attemptNr=:attemptNr "
            sqlRS += "AND status=:oldEventStatus "
            # start transaction
            self.conn.begin()
            self.cur.arraysize = 100000
            # get job
            varMap = {}
            varMap[':pandaID'] = pandaID
            self.cur.execute(sqlJ+comment,varMap)
            resJ = self.cur.fetchone()
            toSkip = True
            retRanges = []
            noMoreEvents = False
            if resJ is None:
                # job not found
                tmpLog.debug("skip job not found")
            elif resJ[0] not in ['sent','running','starting']:
                # wrong job status
                tmpLog.debug("skip wrong job status in {0}".format(resJ[0]))
            elif resJ[1] == 'tobekilled':
                # job is being killed
                tmpLog.debug("skip job is being killed")
            else:
                toSkip = False
                # jumbo
                if resJ[2] == EventServiceUtils.jumboJobFlagNumber:
                    isJumbo = True
                else:
                    isJumbo = False
                # get jediTaskID
                if jediTaskID is None:
                    jediTaskID = resJ[3]
                # find a file to lock
                varMap = dict()
                varMap[':pandaID'] = pandaID
                varMap[':type1'] = 'input'
                varMap[':type2'] = 'pseudo_input'
                self.cur.execute(sqlFF+comment,varMap)
                resFF = self.cur.fetchone()
                if resFF is not None:
                    ffJediTask, ffDatasetID, ffFileID = resFF
                    varMap = dict()
                    varMap[':jediTaskID'] = ffJediTask
                    varMap[':datasetID'] = ffDatasetID
                    if isJumbo:
                        self.cur.execute(sqlLD+comment, varMap)
                        tmpLog.debug("locked datasetID={0}".format(ffDatasetID))
                # prelock event ranges
                varMap = {}
                varMap[':eventStatus']  = EventServiceUtils.ST_ready
                varMap[':minAttemptNr'] = 0
                varMap[':jediTaskID'] = jediTaskID
                varMap[':pandaID'] = pandaID
                varMap[':eventStatus'] = EventServiceUtils.ST_ready
                varMap[':newEventStatus'] = EventServiceUtils.ST_reserved_get
                if segment_id is not None:
                    varMap[':datasetID'] = segment_id
                if not isJumbo:
                    varMap[':jobsetID'] = jobsetID
                if isJumbo:
                    tmpLog.debug(sqlJM+comment+str(varMap))
                    self.cur.execute(sqlJM+comment, varMap)
                else:
                    self.cur.execute(sqlW+comment, varMap)
                nRow = self.cur.rowcount
                tmpLog.debug("pre-locked {0} events".format(nRow))
                # get event ranges
                varMap = dict()
                varMap[':jediTaskID'] = jediTaskID
                varMap[':PandaID'] = pandaID
                varMap[':eventStatus'] = EventServiceUtils.ST_reserved_get
                tmpLog.debug(sqlRR+comment+str(varMap))
                self.cur.execute(sqlRR+comment, varMap)
                resList = self.cur.fetchall()
                if len(resList) > nRanges:
                    # release the last event range
                    tmpJediTaskID,datasetID,fileID,attemptNr,job_processID,startEvent,lastEvent,tmpJobsetID = resList[-1]
                    varMap = {}
                    varMap[':jediTaskID'] = tmpJediTaskID
                    varMap[':fileID'] = fileID
                    varMap[':job_processID'] = job_processID
                    varMap[':pandaID'] = pandaID
                    varMap[':attemptNr'] = attemptNr
                    varMap[':eventStatus'] = EventServiceUtils.ST_ready
                    varMap[':oldEventStatus'] = EventServiceUtils.ST_reserved_get
                    self.cur.execute(sqlRS+comment, varMap)
                    resList = resList[:nRanges]
                else:
                    noMoreEvents = True
                # make dict
                fileInfo = {}
                jobsetList = {}
                for tmpJediTaskID,datasetID,fileID,attemptNr,job_processID,startEvent,lastEvent,tmpJobsetID in resList:
                    # get file info
                    if fileID not in fileInfo:
                        varMap = {}
                        varMap[':jediTaskID'] = tmpJediTaskID
                        varMap[':datasetID'] = datasetID
                        varMap[':fileID'] = fileID
                        self.cur.execute(sqlF+comment, varMap)
                        resF = self.cur.fetchone()
                        # not found
                        if resF is None:
                            resF = (None, None, None)
                            tmpLog.warning("file info is not found for fileID={0}".format(fileID))
                        fileInfo[fileID] = resF
                    # get LFN and GUID
                    tmpLFN,tmpGUID,tmpScope = fileInfo[fileID]
                    # make dict
                    tmpDict = {'eventRangeID':self.makeEventRangeID(tmpJediTaskID,pandaID,
                                                                    fileID,job_processID,
                                                                    attemptNr),
                               'startEvent':startEvent,
                               'lastEvent':lastEvent,
                               'LFN':tmpLFN,
                               'GUID':tmpGUID,
                               'scope':tmpScope}
                    # append
                    retRanges.append(tmpDict)
                    iRanges += 1
                    if tmpJediTaskID not in jobsetList:
                        jobsetList[tmpJediTaskID] = []
                    jobsetList[tmpJediTaskID].append(tmpJobsetID)
                tmpLog.debug("got {0} events".format(len(retRanges)))
                # lock events
                varMap = {}
                varMap[':jediTaskID'] = jediTaskID
                varMap[':pandaID'] = pandaID
                varMap[':eventStatus'] = EventServiceUtils.ST_sent
                varMap[':oldEventStatus'] = EventServiceUtils.ST_reserved_get
                if isJumbo:
                    varMap[':isJumbo'] = EventServiceUtils.eventTableIsJumbo
                else:
                    varMap[':isJumbo'] = None
                self.cur.execute(sqlU+comment, varMap)
                nRow = self.cur.rowcount
                tmpLog.debug("locked {0} events".format(nRow))
                # kill unused consumers
                if not isJumbo and not toSkip and (retRanges == [] or noMoreEvents) and jediTaskID is not None\
                        and segment_id is None:
                    tmpLog.debug("kill unused consumers")
                    tmpJobSpec = JobSpec()
                    tmpJobSpec.PandaID = pandaID
                    tmpJobSpec.jobsetID = jobsetID
                    tmpJobSpec.jediTaskID = jediTaskID
                    self.killUnusedEventServiceConsumers(tmpJobSpec, False, checkAttemptNr=True)
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            regTime = datetime.datetime.utcnow() - regStart
            tmpLog.debug("done with {0} event ranges. took {1} sec".format(iRanges, regTime.seconds))
            if not acceptJson:
                return json.dumps(retRanges)
            return retRanges
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            return None



    # update even ranges
    def updateEventRanges(self,eventDictParam,version=0):
        comment = ' /* DBProxy.updateEventRanges */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += ' <{0}>'.format(datetime.datetime.utcnow().isoformat('/'))
        tmpLog = LogWrapper(_logger,methodName)
        try:
            regStart = datetime.datetime.utcnow()
            retList = []
            jobAttrs = {}
            commandMap = {}
            # sql to update status
            sqlU  = "UPDATE {0}.JEDI_Events ".format(panda_config.schemaJEDI)
            sqlU += "SET status=:eventStatus,objstore_ID=:objstoreID,error_code=:errorCode,path_convention=:pathConvention"
            if version != 0:
                sqlU += ",zipRow_ID=:zipRow_ID"
            sqlU += " WHERE jediTaskID=:jediTaskID AND pandaID=:pandaID AND fileID=:fileID "
            sqlU += "AND job_processID=:job_processID AND attemptNr=:attemptNr "
            sqlU += "AND status IN (:esSent, :esRunning) "
            # sql to get event range
            sqlC  = "SELECT splitRule FROM {0}.JEDI_Tasks ".format(panda_config.schemaJEDI)
            sqlC += "WHERE jediTaskID=:jediTaskID "
            # sql to get nEvents
            sqlE  = "SELECT jobStatus,nEvents,commandToPilot,supErrorCode,specialHandling FROM ATLAS_PANDA.jobsActive4 "
            sqlE += "WHERE PandaID=:pandaID "
            # sql to set nEvents
            sqlS  = "UPDATE ATLAS_PANDA.jobsActive4 "
            sqlS += "SET nEvents=(SELECT COUNT(1) FROM {0}.JEDI_Events ".format(panda_config.schemaJEDI)
            sqlS += "WHERE jediTaskID=:jediTaskID AND PandaID=:PandaID AND status IN (:esFinished,:esDone,:esMerged))*:nEvents "
            sqlS += "WHERE PandaID=:pandaID "
            # sql to check zip file
            sqlFC = "SELECT row_ID FROM ATLAS_PANDA.filesTable4 "
            sqlFC += "WHERE PandaID=:pandaID AND lfn=:lfn "
            # sql to insert zip file
            sqlF  = "INSERT INTO ATLAS_PANDA.filesTable4 (%s) " % FileSpec.columnNames()
            sqlF += FileSpec.bindValuesExpression(useSeq=True)
            sqlF += " RETURNING row_ID INTO :newRowID"
            # sql for fatal events
            sqlFA  = "UPDATE {0}.JEDI_Events ".format(panda_config.schemaJEDI)
            sqlFA += "SET attemptNr=:newAttemptNr "
            sqlFA += " WHERE jediTaskID=:jediTaskID AND pandaID=:pandaID AND fileID=:fileID "
            sqlFA += "AND job_processID=:job_processID AND attemptNr=:oldAttemptNr "
            sqlFA += "AND status=:eventStatus "
            # params formatting with version
            if version == 0:
                # format without zip
                eventDictList = eventDictParam
            else:
                # format with zip
                eventDictList = []
                for eventDictChunk in eventDictParam:
                    # get zip file if any
                    if 'zipFile' in eventDictChunk:
                        zipFile = eventDictChunk['zipFile']
                    else:
                        zipFile = None
                    # collect all dicts
                    if 'eventRanges' in eventDictChunk:
                        for eventDict in eventDictChunk['eventRanges']:
                            # add zip file
                            eventDict['zipFile'] = zipFile
                            # append
                            eventDictList.append(eventDict)
                    else:
                        eventDictList.append(eventDictChunk)
            # update events
            tmpLog.debug('total {0} events'.format(len(eventDictList)))
            zipRowIdMap = {}
            nEventsMap = dict()
            cpuConsumptionTimeMap = dict()
            iEvents = 0
            maxEvents = 100000
            iSkipped = 0
            # start transaction
            self.conn.begin()
            # loop over all events
            varMapListU = []
            varMapListFA = []
            for eventDict in eventDictList:
                # avoid too many events
                iEvents += 1
                if iEvents > maxEvents:
                    retList.append(None)
                    iSkipped += 1
                    continue
                # get event range ID
                if 'eventRangeID' not in eventDict:
                    tmpLog.error('eventRangeID is missing in {0}'.format(str(eventDict)))
                    retList.append(False)
                    continue
                eventRangeID = eventDict['eventRangeID']
                # decompose eventRangeID
                try:
                    tmpItems = eventRangeID.split('-')
                    jediTaskID,pandaID,fileID,job_processID,attemptNr = tmpItems
                    jediTaskID = long(jediTaskID)
                    pandaID = long(pandaID)
                    fileID = long(fileID)
                    job_processID = long(job_processID)
                    attemptNr = long(attemptNr)
                except Exception:
                    _logger.error("{0} : wrongly formatted eventRangeID".format(methodName))
                    retList.append(False)
                    continue
                # get event status
                if 'eventStatus' not in eventDict:
                    tmpLog.error('<eventRangeID={0}> eventStatus is missing in {1}'.format(eventRangeID,str(eventDict)))
                    retList.append(False)
                    continue
                eventStatus = eventDict['eventStatus']
                # map string status to int
                isFatal = False
                if eventStatus == 'running':
                    intEventStatus = EventServiceUtils.ST_running
                elif eventStatus == 'transferring':
                    intEventStatus = EventServiceUtils.ST_running
                elif eventStatus == 'finished':
                    intEventStatus = EventServiceUtils.ST_finished
                elif eventStatus == 'failed':
                    intEventStatus = EventServiceUtils.ST_failed
                elif eventStatus == 'fatal':
                    intEventStatus = EventServiceUtils.ST_failed
                    isFatal = True
                else:
                    tmpLog.error("<eventRangeID={0}> unknown status {1}".format(eventRangeID,eventStatus))
                    retList.append(False)
                    continue
                # only final status
                if eventStatus not in ['finished', 'failed', 'fatal']:
                    retList.append(None)
                    iSkipped += 1
                    tmpLog.debug("<eventRangeID={0}> eventStatus={1} skipped".format(eventRangeID, eventStatus))
                    continue
                # core count
                if 'coreCount' in eventDict:
                    coreCount = eventDict['coreCount']
                else:
                    coreCount = None
                # CPU consumption
                if 'cpuConsumptionTime' in eventDict:
                    cpuConsumptionTime = eventDict['cpuConsumptionTime']
                else:
                    cpuConsumptionTime = None
                # objectstore ID
                if 'objstoreID' in eventDict:
                    objstoreID = eventDict['objstoreID']
                else:
                    objstoreID = None
                # error code
                if 'errorCode' in eventDict:
                    errorCode = eventDict['errorCode']
                else:
                    errorCode = None
                # path convention
                if 'pathConvention' in eventDict:
                    pathConvention = eventDict['pathConvention']
                else:
                    pathConvention = None
                isOK = True
                # get job attributes
                if pandaID not in jobAttrs:
                    varMap = {}
                    varMap[':pandaID'] = pandaID
                    self.cur.execute(sqlE+comment, varMap)
                    resE = self.cur.fetchone()
                    jobAttrs[pandaID] = resE
                    tmpLog.debug("PandaID={0}".format(pandaID))
                resE = jobAttrs[pandaID]
                if resE is None:
                    tmpLog.error("<eventRangeID={0}> unknown PandaID".format(eventRangeID))
                    retList.append(False)
                    isOK = False
                    commandToPilot = 'tobekilled'
                else:
                    # check job status
                    jobStatus,nEventsOld,commandToPilot,supErrorCode,specialHandling = resE
                    if jobStatus not in ['sent','running','starting','transferring']:
                        tmpLog.error("<eventRangeID={0}> wrong jobStatus={1}".format(eventRangeID,jobStatus))
                        _logger.debug("{0} : wrong jobStatus={1}".format(methodName,jobStatus))
                        retList.append(False)
                        isOK = False
                    else:
                        # insert zip
                        zipRow_ID = None
                        if 'zipFile' in eventDict and eventDict['zipFile'] is not None:
                            if eventDict['zipFile']['lfn'] in zipRowIdMap:
                                zipRow_ID = zipRowIdMap[eventDict['zipFile']['lfn']]
                            else:
                                # check zip
                                varMap = dict()
                                varMap[':pandaID'] = pandaID
                                varMap[':lfn'] = eventDict['zipFile']['lfn']
                                self.cur.execute(sqlFC+comment, varMap)
                                resFC = self.cur.fetchone()
                                if resFC is not None:
                                    zipRow_ID, = resFC
                                else:
                                    # insert new file
                                    zipJobSpec = JobSpec()
                                    zipJobSpec.PandaID = pandaID
                                    zipJobSpec.specialHandling = specialHandling
                                    zipFileSpec = FileSpec()
                                    zipFileSpec.jediTaskID = jediTaskID
                                    zipFileSpec.lfn = eventDict['zipFile']['lfn']
                                    zipFileSpec.GUID = str(uuid.uuid4())
                                    if 'fsize' in eventDict['zipFile']:
                                        zipFileSpec.fsize = long(eventDict['zipFile']['fsize'])
                                    else:
                                        zipFileSpec.fsize = 0
                                    if 'adler32' in eventDict['zipFile']:
                                        zipFileSpec.checksum = 'ad:{0}'.format(eventDict['zipFile']['adler32'])
                                    if 'numEvents' in eventDict['zipFile']:
                                        zipFileSpec.dispatchDBlockToken = eventDict['zipFile']['numEvents']
                                    zipFileSpec.type = 'zipoutput'
                                    zipFileSpec.status = 'ready'
                                    zipFileSpec.destinationSE = eventDict['zipFile']['objstoreID']
                                    if 'pathConvention' in eventDict['zipFile']:
                                        zipFileSpec.destinationSE = '{0}/{1}'.format(zipFileSpec.destinationSE,
                                                                                     eventDict['zipFile']['pathConvention'])
                                    zipJobSpec.addFile(zipFileSpec)
                                    varMap = zipFileSpec.valuesMap(useSeq=True)
                                    varMap[':newRowID'] = self.cur.var(varNUMBER)
                                    self.cur.execute(sqlF+comment, varMap)
                                    val = self.getvalue_corrector(self.cur.getvalue(varMap[':newRowID']))
                                    zipRow_ID = long(val)
                                    zipRowIdMap[eventDict['zipFile']['lfn']] = zipRow_ID
                                    # make an empty file to trigger registration for zip files in Adder
                                    if zipJobSpec.registerEsFiles():
                                        # tmpFileName = '{0}_{1}_{2}'.format(pandaID, EventServiceUtils.esRegStatus,
                                        #                                    uuid.uuid3(uuid.NAMESPACE_DNS,''))
                                        # tmpFileName = os.path.join(panda_config.logdir, tmpFileName)
                                        # try:
                                        #     open(tmpFileName, 'w').close()
                                        # except Exception:
                                        #     pass
                                        # sql to insert
                                        sqlI  = (  'INSERT INTO {0}.Job_Output_Report '
                                                        '(PandaID, prodSourceLabel, jobStatus, attemptNr, data, timeStamp) '
                                                        'VALUES(:PandaID, :prodSourceLabel, :jobStatus, :attemptNr, :data, :timeStamp) '
                                                    ).format(panda_config.schemaPANDA)
                                        # insert
                                        varMap = {}
                                        varMap[':PandaID'] = pandaID
                                        varMap[':prodSourceLabel'] = zipJobSpec.prodSourceLabel
                                        varMap[':jobStatus'] = zipJobSpec.jobStatus
                                        varMap[':attemptNr'] = 0 if zipJobSpec.attemptNr in [None, 'NULL', ''] else zipJobSpec.attemptNr
                                        varMap[':data'] = None
                                        varMap[':timeStamp'] = datetime.datetime.utcnow()
                                        try:
                                            self.cur.execute(sqlI+comment, varMap)
                                        except Exception:
                                            pass
                                        else:
                                            tmpLog.debug('successfully inserted job output report {0}.{1}'.format(pandaID, varMap[':attemptNr']))
                        # update event
                        varMap = {}
                        varMap[':jediTaskID'] = jediTaskID
                        varMap[':pandaID'] = pandaID
                        varMap[':fileID'] = fileID
                        varMap[':job_processID'] = job_processID
                        varMap[':attemptNr'] = attemptNr
                        varMap[':eventStatus'] = intEventStatus
                        varMap[':objstoreID'] = objstoreID
                        varMap[':errorCode'] = errorCode
                        varMap[':pathConvention'] = pathConvention
                        varMap[':esSent'] = EventServiceUtils.ST_sent
                        varMap[':esRunning'] = EventServiceUtils.ST_running
                        if version != 0:
                            varMap[':zipRow_ID'] = zipRow_ID
                        varMapListU.append(varMap)
                        # fatal event
                        if isFatal:
                            varMap = {}
                            varMap[':jediTaskID'] = jediTaskID
                            varMap[':pandaID'] = pandaID
                            varMap[':fileID'] = fileID
                            varMap[':job_processID'] = job_processID
                            varMap[':oldAttemptNr'] = attemptNr
                            varMap[':newAttemptNr'] = 1
                            varMap[':eventStatus'] = EventServiceUtils.ST_failed
                            varMapListFA.append(varMap)
                        # nEvents of finished
                        if eventStatus in ['finished']:
                            # get nEvents
                            if pandaID not in nEventsMap:
                                nEventsDef = 1
                                varMap = {}
                                varMap[':jediTaskID'] = jediTaskID
                                self.cur.execute(sqlC+comment, varMap)
                                resC = self.cur.fetchone()
                                if resC is not None:
                                    splitRule, = resC
                                    tmpM = re.search('ES=(\d+)', splitRule)
                                    if tmpM is not None:
                                        nEventsDef = int(tmpM.group(1))
                                nEventsMap[pandaID] = {'jediTaskID': jediTaskID, 'nEvents': nEventsDef}
                    # soft kill
                    if commandToPilot not in [None,''] and supErrorCode in [ErrorCode.EC_EventServicePreemption]:
                            commandToPilot = 'softkill'
                if isOK:
                    #tmpLog.debug("<eventRangeID={0}> eventStatus={1}".format(eventRangeID, eventStatus))
                    retList.append(True)
                if pandaID not in commandMap:
                    commandMap[pandaID] = commandToPilot
            tmpLog.debug('update {0} events'.format(len(varMapListU)))
            if len(varMapListU) > 0:
                self.cur.executemany(sqlU+comment, varMapListU)
            tmpLog.debug('fatal {0} events'.format(len(varMapListFA)))
            if len(varMapListFA) > 0:
                self.cur.executemany(sqlFA+comment, varMapListFA)
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # update nevents
            for pandaID in nEventsMap:
                data = nEventsMap[pandaID]
                self.conn.begin()
                varMap = {}
                varMap[':pandaID'] = pandaID
                varMap[':jediTaskID'] = data['jediTaskID']
                varMap[':nEvents'] = data['nEvents']
                varMap[':esFinished'] = EventServiceUtils.ST_finished
                varMap[':esDone'] = EventServiceUtils.ST_done
                varMap[':esMerged'] = EventServiceUtils.ST_merged
                self.cur.execute(sqlS+comment, varMap)
                if not self._commit():
                    raise RuntimeError('Commit error')
            regTime = datetime.datetime.utcnow() - regStart
            tmpLog.debug('done. {0}/{1} events skipped. took {2} sec'.format(iSkipped, len(eventDictList), regTime.seconds))
            return retList,commandMap
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            retList.append(False)
            return retList,commandMap



    # post-process for event service job
    def ppEventServiceJob(self,job,currentJobStatus,useCommit=True):
        comment = ' /* DBProxy.ppEventServiceJob */'
        pandaID = job.PandaID
        attemptNr = job.attemptNr
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " < PandaID={0} >".format(pandaID)
        _logger.debug("{0} : start attemptNr={1}".format(methodName,attemptNr))
        try:
            # return values
            # 0 : generated a retry job
            # 1 : not retried due to a harmless reason
            # 2 : generated a merge job
            # 3 : max attempts reached
            # 4 : not generated a merge job since other consumers are still running
            # 5 : didn't process any events on WN
            # 6 : didn't process any events on WN and fail since the last one
            # 7 : all event ranges failed
            # 8 : generated a retry job but no events were processed
            # 9 : closed in bad job status
            # 10 : generated a merge job but didn't process any events by itself
            # None : fatal error
            retValue = 1,None
            # begin transaction
            if useCommit:
                self.conn.begin()
            self.cur.arraysize = 10
            # make job spec to not change the original
            jobSpec = copy.copy(job)
            jobSpec.Files = []
            # check if event service job
            if not EventServiceUtils.isEventServiceJob(jobSpec):
                _logger.debug("{0} : no event service job".format(methodName))
                # commit
                if useCommit:
                    if not self._commit():
                        raise RuntimeError('Commit error')
                return retValue
            # check if already retried or not good for retry
            if jobSpec.taskBufferErrorCode in [ErrorCode.EC_EventServiceRetried,ErrorCode.EC_EventServiceMerge,
                                               ErrorCode.EC_EventServiceInconsistentIn,ErrorCode.EC_EventServiceBadStatus]:
                _logger.debug("{0} : already post-processed for event service with EC={1}".format(methodName,jobSpec.taskBufferErrorCode))
                # commit
                if useCommit:
                    if not self._commit():
                        raise RuntimeError('Commit error')
                return retValue
            # check if JEDI is used
            if hasattr(panda_config,'useJEDI') and panda_config.useJEDI is True and \
                    jobSpec.lockedby == 'jedi' and self.checkTaskStatusJEDI(jobSpec.jediTaskID,self.cur):
                pass
            else:
                _logger.debug("{0} : JEDI is not used".format(methodName))
                # commit
                if useCommit:
                    if not self._commit():
                        raise RuntimeError('Commit error')
                return retValue
            # use an input file as lock since FOR UPDATE doesn't work on JEDI_Events
            lockFileSpec = None
            for fileSpec in job.Files:
                if fileSpec.type in ['input', 'pseudo_input']:
                    if lockFileSpec is None or lockFileSpec.fileID > fileSpec.fileID:
                        lockFileSpec = fileSpec
            if lockFileSpec is not None:
                # sql to lock the file
                sqlLIF = "SELECT status FROM {0}.JEDI_Dataset_Contents ".format(panda_config.schemaJEDI)
                sqlLIF += 'WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID '
                sqlLIF += 'FOR UPDATE NOWAIT '
                varMap = dict()
                varMap[':jediTaskID'] = lockFileSpec.jediTaskID
                varMap[':datasetID'] = lockFileSpec.datasetID
                varMap[':fileID'] = lockFileSpec.fileID
                _logger.debug("{0} : locking {1}".format(methodName, str(varMap)))
                self.cur.execute(sqlLIF+comment, varMap)
                _logger.debug("{0} : locked".format(methodName))
            # change event status processed by jumbo jobs
            nRowDoneJumbo = 0
            nRowFailedJumbo = 0
            if EventServiceUtils.isCoJumboJob(jobSpec):
                # sql to change event status
                sqlJE  = "UPDATE /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
                sqlJE += "{0}.JEDI_Events tab ".format(panda_config.schemaJEDI)
                sqlJE += "SET status=:newStatus "
                sqlJE += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                sqlJE += "AND status=:oldStatus AND is_jumbo=:isJumbo "
                # sql to lock failed events
                sqlJFL = sqlJE + "AND processed_upto_eventID IS NOT NULL "
                # sql to copy failed events
                sqlJFC  = "INSERT INTO {0}.JEDI_Events ".format(panda_config.schemaJEDI)
                sqlJFC += "(jediTaskID,datasetID,PandaID,fileID,attemptNr,status,"
                sqlJFC += "job_processID,def_min_eventID,def_max_eventID,processed_upto_eventID,"
                sqlJFC += "event_offset,is_jumbo) "
                sqlJFC += "SELECT /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
                sqlJFC += "jediTaskID,datasetID,event_offset,fileID,attemptNr-1,:newStatus,"
                sqlJFC += "job_processID,def_min_eventID,def_max_eventID,processed_upto_eventID,"
                sqlJFC += "event_offset,NULL "
                sqlJFC += "FROM {0}.JEDI_Events tab ".format(panda_config.schemaJEDI)
                sqlJFC += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                sqlJFC += "AND status=:oldStatus AND processed_upto_eventID IS NOT NULL AND is_jumbo=:isJumbo "
                # sql to release failed events
                sqlJFR  = "UPDATE /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
                sqlJFR += "{0}.JEDI_Events tab ".format(panda_config.schemaJEDI)
                sqlJFR += "SET PandaID=:pandaID,status=:newStatus,processed_upto_eventID=NULL "
                sqlJFR += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                sqlJFR += "AND status=:oldStatus AND is_jumbo=:isJumbo "
                sqlJFR += "AND processed_upto_eventID IS NOT NULL "
                for fileSpec in job.Files:
                    if fileSpec.type != 'input':
                        continue
                    # done events in jumbo jobs
                    varMap = {}
                    varMap[':jediTaskID']  = fileSpec.jediTaskID
                    varMap[':datasetID']   = fileSpec.datasetID
                    varMap[':fileID']      = fileSpec.fileID
                    varMap[':oldStatus']   = EventServiceUtils.ST_finished
                    varMap[':newStatus']   = EventServiceUtils.ST_done
                    varMap[':isJumbo']     = EventServiceUtils.eventTableIsJumbo
                    self.cur.execute(sqlJE+comment, varMap)
                    nRowDoneJumbo += self.cur.rowcount
                    # lock failed events
                    varMap = {}
                    varMap[':jediTaskID']  = fileSpec.jediTaskID
                    varMap[':datasetID']   = fileSpec.datasetID
                    varMap[':fileID']      = fileSpec.fileID
                    varMap[':oldStatus']   = EventServiceUtils.ST_failed
                    varMap[':newStatus']   = EventServiceUtils.ST_reserved_fail
                    varMap[':isJumbo']     = EventServiceUtils.eventTableIsJumbo
                    self.cur.execute(sqlJFL+comment, varMap)
                    tmpNumRow = self.cur.rowcount
                    if tmpNumRow > 0:
                        # copy failed events
                        varMap = {}
                        varMap[':jediTaskID']  = fileSpec.jediTaskID
                        varMap[':datasetID']   = fileSpec.datasetID
                        varMap[':fileID']      = fileSpec.fileID
                        varMap[':oldStatus']   = EventServiceUtils.ST_reserved_fail
                        varMap[':newStatus']   = EventServiceUtils.ST_ready
                        varMap[':isJumbo']     = EventServiceUtils.eventTableIsJumbo
                        self.cur.execute(sqlJFC+comment, varMap)
                        # release failed events. Change PandaID to avoid unique constraint of PK
                        varMap = {}
                        varMap[':jediTaskID']  = fileSpec.jediTaskID
                        varMap[':datasetID']   = fileSpec.datasetID
                        varMap[':fileID']      = fileSpec.fileID
                        varMap[':pandaID']     = pandaID
                        varMap[':oldStatus']   = EventServiceUtils.ST_reserved_fail
                        varMap[':newStatus']   = EventServiceUtils.ST_failed
                        varMap[':isJumbo']     = EventServiceUtils.eventTableIsJumbo
                        self.cur.execute(sqlJFR+comment, varMap)
                        nRowFailedJumbo += tmpNumRow
                _logger.debug("{0} : set done for jumbo to {1} event ranges".format(methodName,nRowDoneJumbo))
                _logger.debug("{0} : copied {1} failed event ranges in jumbo".format(methodName,nRowFailedJumbo))
            # change status to done
            sqlED  = "UPDATE {0}.JEDI_Events SET status=:newStatus ".format(panda_config.schemaJEDI)
            sqlED += "WHERE jediTaskID=:jediTaskID AND pandaID=:pandaID AND status=:oldStatus "
            varMap = {}
            varMap[':jediTaskID'] = jobSpec.jediTaskID
            varMap[':pandaID']    = pandaID
            varMap[':oldStatus']  = EventServiceUtils.ST_finished
            varMap[':newStatus']  = EventServiceUtils.ST_done
            self.cur.execute(sqlED+comment, varMap)
            nRowDone = self.cur.rowcount
            _logger.info("{0} : set done to n_er_done={1} event ranges".format(methodName,nRowDone))
            # release unprocessed event ranges
            sqlEC  = "UPDATE {0}.JEDI_Events ".format(panda_config.schemaJEDI)
            if jobSpec.decAttOnFailedES():
                sqlEC += "SET status=:newStatus,pandaID=:jobsetID "
            else:
                sqlEC += "SET status=:newStatus,attemptNr=attemptNr-1,pandaID=:jobsetID "
            sqlEC += "WHERE jediTaskID=:jediTaskID AND pandaID=:pandaID AND NOT status IN (:esDone,:esFailed,:esDiscarded,:esCancelled) "
            varMap = {}
            varMap[':jediTaskID']  = jobSpec.jediTaskID
            varMap[':pandaID']     = pandaID
            varMap[':jobsetID']    = jobSpec.jobsetID
            varMap[':esDone']      = EventServiceUtils.ST_done
            varMap[':esFailed']    = EventServiceUtils.ST_failed
            varMap[':esDiscarded'] = EventServiceUtils.ST_discarded
            varMap[':esCancelled'] = EventServiceUtils.ST_cancelled
            varMap[':newStatus']   = EventServiceUtils.ST_ready
            self.cur.execute(sqlEC+comment, varMap)
            nRowReleased = self.cur.rowcount
            _logger.info("{0} : released n_er_released={1} event ranges".format(methodName,nRowReleased))
            # copy failed event ranges
            varMap = {}
            varMap[':jediTaskID']  = jobSpec.jediTaskID
            varMap[':pandaID']     = pandaID
            varMap[':jobsetID']    = jobSpec.jobsetID
            varMap[':esFailed']    = EventServiceUtils.ST_failed
            varMap[':newStatus']   = EventServiceUtils.ST_ready
            sqlEF  = "INSERT INTO {0}.JEDI_Events ".format(panda_config.schemaJEDI)
            sqlEF += "(jediTaskID,datasetID,PandaID,fileID,attemptNr,status,"
            sqlEF += "job_processID,def_min_eventID,def_max_eventID,processed_upto_eventID,event_offset) "
            sqlEF += "SELECT jediTaskID,datasetID,:jobsetID,fileID,attemptNr-1,:newStatus,"
            sqlEF += "job_processID,def_min_eventID,def_max_eventID,processed_upto_eventID,event_offset "
            sqlEF += "FROM {0}.JEDI_Events ".format(panda_config.schemaJEDI)
            sqlEF += "WHERE jediTaskID=:jediTaskID AND pandaID=:pandaID AND status=:esFailed "
            sqlEF += "AND processed_upto_eventID IS NOT NULL "
            self.cur.execute(sqlEF+comment, varMap)
            nRowCopied = self.cur.rowcount
            _logger.debug("{0} : copied {1} failed event ranges".format(methodName,nRowCopied))
            # unset processed_upto for failed events
            sqlUP  = "UPDATE {0}.JEDI_Events SET processed_upto_eventID=NULL ".format(panda_config.schemaJEDI)
            sqlUP += "WHERE jediTaskID=:jediTaskID AND pandaID=:pandaID AND status=:esFailed "
            varMap = {}
            varMap[':jediTaskID']  = jobSpec.jediTaskID
            varMap[':pandaID']     = pandaID
            varMap[':esFailed']    = EventServiceUtils.ST_failed
            self.cur.execute(sqlUP+comment, varMap)
            nRowFailed = self.cur.rowcount
            _logger.info("{0} : failed n_er_failed={1} event ranges".format(methodName,nRowFailed))
            sqlEU  = "SELECT /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
            sqlEU += "COUNT(*) FROM {0}.JEDI_Events ".format(panda_config.schemaJEDI)
            sqlEU += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID AND attemptNr=:minAttempt "
            sqlEU += "AND NOT status IN (:esDiscarded,:esCancelled) "
            # look for hopeless event ranges
            nRowFatal = 0
            for fileSpec in job.Files:
                if fileSpec.type != 'input':
                    continue
                varMap = {}
                varMap[':jediTaskID'] = fileSpec.jediTaskID
                varMap[':datasetID']  = fileSpec.datasetID
                varMap[':fileID']     = fileSpec.fileID
                varMap[':minAttempt'] = 0
                varMap[':esDiscarded'] = EventServiceUtils.ST_discarded
                varMap[':esCancelled'] = EventServiceUtils.ST_cancelled
                self.cur.execute(sqlEU+comment, varMap)
                resEU = self.cur.fetchone()
                if resEU is not None:
                    nRowFatal += resEU[0]
            # there is hopeless event ranges
            _logger.info("{0} : has n_hopeless={1} hopeless event ranges".format(methodName, nRowFatal))
            if nRowFatal != 0:
                if jobSpec.acceptPartialFinish():
                    # set fatal to hopeless event ranges
                    sqlFH  = "UPDATE {0}.JEDI_Events SET status=:esFatal ".format(panda_config.schemaJEDI)
                    sqlFH += "WHERE jediTaskID=:jediTaskID AND pandaID=:jobsetID AND attemptNr=:minAttempt AND status<>:esFatal "
                    varMap = {}
                    varMap[':jediTaskID']  = jobSpec.jediTaskID
                    varMap[':jobsetID']    = jobSpec.jobsetID
                    varMap[':esFatal']     = EventServiceUtils.ST_fatal
                    varMap[':minAttempt']  = 0
                    self.cur.execute(sqlFH+comment, varMap)
            # look for event ranges to process
            sqlERP  = "SELECT job_processID FROM {0}.JEDI_Events ".format(panda_config.schemaJEDI)
            sqlERP += "WHERE jediTaskID=:jediTaskID AND pandaID=:jobsetID AND status=:esReady "
            sqlERP += "AND attemptNr>:minAttempt "
            varMap = {}
            varMap[':jediTaskID']  = jobSpec.jediTaskID
            varMap[':jobsetID']    = jobSpec.jobsetID
            varMap[':esReady']     = EventServiceUtils.ST_ready
            varMap[':minAttempt']  = 0
            self.cur.execute(sqlERP+comment, varMap)
            resERP = self.cur.fetchall()
            nRow = len(resERP)
            _logger.info("{0} : left n_er_unprocessed={1} unprocessed event ranges".format(methodName,nRow))
            otherRunning = False
            hasDoneRange = False
            if nRow == 0:
                # check if other consumers finished
                sqlEOC  = "SELECT /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
                sqlEOC += "job_processID,attemptNr,status,processed_upto_eventID,PandaID FROM {0}.JEDI_Events tab ".format(panda_config.schemaJEDI)
                sqlEOC += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                sqlEOC += "AND ((NOT status IN (:esDone,:esDiscarded,:esCancelled,:esFatal,:esFailed,:esCorrupted) AND attemptNr>:minAttempt) "
                sqlEOC += "OR (status=:esFailed AND processed_upto_eventID IS NOT NULL)) "
                # count the number of done ranges
                sqlCDO  = "SELECT /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
                sqlCDO += "COUNT(*) FROM {0}.JEDI_Events tab ".format(panda_config.schemaJEDI)
                sqlCDO += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                sqlCDO += "AND status=:esDone AND rownum=1 "
                for fileSpec in job.Files:
                    if fileSpec.type == 'input':
                        varMap = {}
                        varMap[':jediTaskID']  = fileSpec.jediTaskID
                        varMap[':datasetID']   = fileSpec.datasetID
                        varMap[':fileID']      = fileSpec.fileID
                        varMap[':esDone']      = EventServiceUtils.ST_done
                        varMap[':esDiscarded'] = EventServiceUtils.ST_discarded
                        varMap[':esCancelled'] = EventServiceUtils.ST_cancelled
                        varMap[':esCorrupted'] = EventServiceUtils.ST_corrupted
                        varMap[':esFatal']     = EventServiceUtils.ST_fatal
                        varMap[':esFailed']    = EventServiceUtils.ST_failed
                        varMap[':minAttempt']  = 0
                        self.cur.execute(sqlEOC+comment, varMap)
                        resEOC = self.cur.fetchone()
                        if resEOC is not None:
                            # there are unprocessed ranges
                            otherRunning = True
                            eocDump = dict()
                            eocDump['jediTaskID'] = fileSpec.jediTaskID
                            eocDump['datasetID'] = fileSpec.datasetID
                            eocDump['fileID'] = fileSpec.fileID
                            eocDump['job_processID'] = resEOC[0]
                            eocDump['attemptNr'] = resEOC[1]
                            eocDump['status'] = resEOC[2]
                            eocDump['processed_upto_eventID'] = resEOC[3]
                            eocDump['PandaID'] = resEOC[4]
                            _logger.debug("{0} : some event ranges still running like {1}".format(methodName,str(eocDump)))
                            break
                        # check if there are done ranges
                        if not hasDoneRange:
                            varMap = {}
                            varMap[':jediTaskID']  = fileSpec.jediTaskID
                            varMap[':datasetID']   = fileSpec.datasetID
                            varMap[':fileID']      = fileSpec.fileID
                            varMap[':esDone']      = EventServiceUtils.ST_done
                            self.cur.execute(sqlCDO+comment, varMap)
                            resCDO = self.cur.fetchone()
                            nCDORow, = resCDO
                            if nCDORow != 0:
                                hasDoneRange = True
                # do merging since all ranges were done
                if not otherRunning:
                    doMerging = True
            else:
                doMerging = False
            # do nothing since other consumers are still running
            if otherRunning:
                _logger.debug("{0} : do nothing as other consumers are still running".format(methodName))
                # commit
                if useCommit:
                    if not self._commit():
                        raise RuntimeError('Commit error')
                if nRowDone == 0:
                    # didn't process any events
                    retValue = 5,None
                else:
                    # processed some events
                    retValue = 4,None
                return retValue
            # all failed
            if doMerging and not hasDoneRange:
                # fail immediately
                _logger.debug("{0} : all event ranges failed".format(methodName))
                # commit
                if useCommit:
                    if not self._commit():
                        raise RuntimeError('Commit error')
                retValue = 7,None
                return retValue
            # fail immediately if not all events were done in the largest attemptNr
            if (jobSpec.attemptNr >= jobSpec.maxAttempt and not (doMerging and hasDoneRange)) or (doMerging and nRowFatal > 0):
                _logger.debug("{0} : no more retry since not all events were done in the largest attemptNr".format(methodName))
                # check if there is active consumer
                sqlAC  = "SELECT COUNT(*) FROM ("
                sqlAC += "SELECT PandaID FROM ATLAS_PANDA.jobsDefined4 "
                sqlAC += "WHERE jediTaskID=:jediTaskID AND jobsetID=:jobsetID "
                sqlAC += "UNION "
                sqlAC += "SELECT PandaID FROM ATLAS_PANDA.jobsActive4 "
                sqlAC += "WHERE jediTaskID=:jediTaskID AND jobsetID=:jobsetID "
                sqlAC += ") "
                varMap = {}
                varMap[':jediTaskID'] = jobSpec.jediTaskID
                varMap[':jobsetID']   = jobSpec.jobsetID
                self.cur.execute(sqlAC+comment, varMap)
                resAC = self.cur.fetchone()
                numActiveEC, = resAC
                _logger.debug("{0} : num of active consumers = {1}".format(methodName,numActiveEC))
                # commit
                if useCommit:
                    if not self._commit():
                        raise RuntimeError('Commit error')
                if numActiveEC <= 1:
                    # last one
                    retValue = 6,None
                else:
                    # there are active consumers
                    retValue = 5,None
                return retValue
            # no merging for inaction ES jobs
            if doMerging and nRowDoneJumbo == 0 and nRowDone == 0 and not job.allOkEvents():
                _logger.debug("{0} : skip merge generation since nDone=0".format(methodName))
                retValue = 5,None
                return retValue
            # change waiting file status
            if doMerging and EventServiceUtils.isCoJumboJob(jobSpec):
                # update file
                sqlUWF  = "UPDATE {0}.JEDI_Dataset_Contents ".format(panda_config.schemaJEDI)
                sqlUWF += "SET status=:newStatus,is_waiting=NULL "
                sqlUWF += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                sqlUWF += "AND attemptNr=:attemptNr AND status=:oldStatus AND keepTrack=:keepTrack "
                # update dataset
                sqlUWD  = "UPDATE {0}.JEDI_Datasets ".format(panda_config.schemaJEDI)
                sqlUWD += "SET nFilesUsed=nFilesUsed+:nDiff,nFilesWaiting=nFilesWaiting-:nDiff "
                sqlUWD += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID "
                nFilesUsedMap = {}
                for fileSpec in job.Files:
                    if fileSpec.type not in ['input', 'pseudo_input']:
                        continue
                    varMap = {}
                    varMap[':jediTaskID']  = fileSpec.jediTaskID
                    varMap[':datasetID']   = fileSpec.datasetID
                    varMap[':fileID']      = fileSpec.fileID
                    varMap[':attemptNr']   = fileSpec.attemptNr
                    varMap[':newStatus']   = 'running'
                    varMap[':oldStatus']   = 'ready'
                    varMap[':keepTrack']   = 1
                    self.cur.execute(sqlUWF+comment, varMap)
                    nDiff = self.cur.rowcount
                    if nDiff > 0:
                        nFilesUsedMap.setdefault(fileSpec.datasetID, 0)
                        nFilesUsedMap[fileSpec.datasetID] += nDiff
                for datasetID in nFilesUsedMap:
                    nDiff = nFilesUsedMap[datasetID]
                    varMap = {}
                    varMap[':jediTaskID'] = jobSpec.jediTaskID
                    varMap[':datasetID'] = datasetID
                    varMap[':nDiff'] = nDiff
                    self.cur.execute(sqlUWD+comment, varMap)
            # check if there is fatal range
            hasFatalRange = False
            if doMerging:
                sqlCFE  = "SELECT COUNT(*) FROM {0}.JEDI_Events ".format(panda_config.schemaJEDI)
                sqlCFE += "WHERE jediTaskID=:jediTaskID AND pandaID=:jobsetID AND "
                sqlCFE += "status=:esFatal AND rownum=1 "
                varMap = {}
                varMap[':jediTaskID']  = jobSpec.jediTaskID
                varMap[':jobsetID']    = jobSpec.jobsetID
                varMap[':esFatal']     = EventServiceUtils.ST_fatal
                self.cur.execute(sqlCFE+comment, varMap)
                resCFE = self.cur.fetchone()
                nRowCEF, = resCFE
                _logger.debug("{0} : {1} fatal even ranges ".format(methodName,nRowCEF))
                if nRowCEF > 0:
                    hasFatalRange = True
            # reset job attributes
            jobSpec.startTime        = None
            jobSpec.creationTime     = datetime.datetime.utcnow()
            jobSpec.modificationTime = jobSpec.creationTime
            jobSpec.stateChangeTime  = jobSpec.creationTime
            jobSpec.prodDBUpdateTime = jobSpec.creationTime
            jobSpec.attemptNr       += 1
            jobSpec.batchID          = None
            jobSpec.schedulerID = None
            jobSpec.pilotID = None
            if doMerging:
                jobSpec.maxAttempt = jobSpec.attemptNr
                jobSpec.currentPriority = 5000
            else:
                jobSpec.currentPriority += 1
            jobSpec.endTime          = None
            jobSpec.transExitCode    = None
            jobSpec.jobMetrics       = None
            jobSpec.jobSubStatus     = None
            jobSpec.actualCoreCount  = None
            jobSpec.hs06sec          = None
            jobSpec.nEvents          = None
            jobSpec.cpuConsumptionTime = None
            # disable background flag
            jobSpec.jobExecutionID = 0
            if hasFatalRange:
                jobSpec.jobSubStatus = 'partial'
            for attr in jobSpec._attributes:
                for patt in ['ErrorCode', 'ErrorDiag', 'CHAR', 'BYTES', 'RSS', 'PSS', 'VMEM', 'SWAP']:
                    if attr.endswith(patt):
                        setattr(jobSpec,attr,None)
                        break
            # read files
            varMap = {}
            varMap[':PandaID'] = pandaID
            sqlFile  = "SELECT {0} FROM ATLAS_PANDA.filesTable4 ".format(FileSpec.columnNames())
            sqlFile += "WHERE PandaID=:PandaID"
            self.cur.arraysize = 100000
            self.cur.execute(sqlFile+comment, varMap)
            resFs = self.cur.fetchall()
            # loop over all files
            for resF in resFs:
                # add
                fileSpec = FileSpec()
                fileSpec.pack(resF)
                jobSpec.addFile(fileSpec)
                # reset file status
                if fileSpec.type in ['output','log']:
                    fileSpec.status = 'unknown'
            # set current status if unspecified
            if currentJobStatus is None:
                currentJobStatus = 'activated'
                for fileSpec in jobSpec.Files:
                    if fileSpec.type == 'input' and fileSpec.status != 'ready':
                        currentJobStatus = 'assigned'
                        break
            if doMerging and currentJobStatus == 'assigned':
                # send merge jobs to activated since input data don't have to move
                _logger.debug("{0} : sending to activated".format(methodName))
                jobSpec.jobStatus = 'activated'
            elif currentJobStatus in ['defined','assigned','waiting','pending']:
                jobSpec.jobStatus = currentJobStatus
            else:
                jobSpec.jobStatus = 'activated'
            # read job parameters
            sqlJobP = "SELECT jobParameters FROM ATLAS_PANDA.jobParamsTable WHERE PandaID=:PandaID"
            varMap = {}
            varMap[':PandaID'] = jobSpec.PandaID
            self.cur.execute(sqlJobP+comment, varMap)
            for clobJobP, in self.cur:
                try:
                    jobSpec.jobParameters = clobJobP.read()
                except AttributeError:
                    jobSpec.jobParameters = str(clobJobP)
                break
            # changes some attributes
            noNewJob = False
            closedInBadStatus = False
            if not doMerging:
                minUnprocessed = self.getConfigValue('dbproxy', 'AES_MINEVENTSFORMCORE')
                
                sqlCore = "SELECT /* use_json_type */ scj.data.corecount, scj.data.status, scj.data.jobseed " \
                          "FROM ATLAS_PANDA.schedconfig_json scj " \
                          "WHERE scj.panda_queue=:siteid "

                varMap = {}
                varMap[':siteid'] = jobSpec.computingSite
                self.cur.execute(sqlCore+comment, varMap)
                resCore = self.cur.fetchone()
                if resCore is not None:
                    coreCount, tmpState, tmpJobSeed = resCore
                    if coreCount is not None:
                        coreCount = int(coreCount)
                        if minUnprocessed is None:
                            minUnprocessed = coreCount
                        else:
                            minUnprocessed = max(minUnprocessed, coreCount)

                    if tmpState not in ['online', 'brokeroff'] or tmpJobSeed == 'std':
                        noNewJob = True
                if jobSpec.coreCount > 1 and minUnprocessed is not None and minUnprocessed > nRow:
                    self.setScoreSiteToEs(jobSpec, methodName, comment)
                # not to repeat useless consumers
                if currentJobStatus in ['defined', 'pending']:
                    noNewJob = True
                    closedInBadStatus = True
            else:
                # extract parameters for merge
                try:
                    tmpMatch = re.search('<PANDA_ESMERGE_TRF>(.*)</PANDA_ESMERGE_TRF>',jobSpec.jobParameters)
                    jobSpec.transformation = tmpMatch.group(1)
                except Exception:
                    pass
                try:
                    tmpMatch = re.search('<PANDA_EVSMERGE>(.*)</PANDA_EVSMERGE>',jobSpec.jobParameters)
                    jobSpec.jobParameters = tmpMatch.group(1)
                except Exception:
                    pass
                # use siteid of jumbo jobs to generate merge jobs for fake co-jumbo
                isFakeCJ = False
                if jobSpec.computingSite == EventServiceUtils.siteIdForWaitingCoJumboJobs:
                    isFakeCJ = True
                    # sql to get PandaIDs of jumbo jobs
                    sqlJJ  = "SELECT /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
                    sqlJJ += "DISTINCT PandaID FROM {0}.JEDI_Events tab ".format(panda_config.schemaJEDI)
                    sqlJJ += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                    sqlJJ += "AND status=:esDone AND is_jumbo=:isJumbo "
                    # sql to get siteid of jumbo job
                    sqlJJS  = "SELECT computingSite FROM ATLAS_PANDA.jobsActive4 "
                    sqlJJS += "WHERE PandaID=:PandaID "
                    sqlJJS += "UNION "
                    sqlJJS += "SELECT computingSite FROM ATLAS_PANDA.jobsArchived4 "
                    sqlJJS += "WHERE PandaID=:PandaID "
                    sqlJJS += "UNION "
                    sqlJJS += "SELECT computingSite FROM ATLAS_PANDAARCH.jobsArchived "
                    sqlJJS += "WHERE PandaID=:PandaID AND modificationTime>CURRENT_DATE-30 "
                    # look for jumbo jobs
                    toEscape = False
                    for fileSpec in job.Files:
                        if fileSpec.type != 'input':
                            continue
                        # get PandaIDs of jumbo jobs
                        varMap = {}
                        varMap[':jediTaskID']  = fileSpec.jediTaskID
                        varMap[':datasetID']   = fileSpec.datasetID
                        varMap[':fileID']      = fileSpec.fileID
                        varMap[':esDone']      = EventServiceUtils.ST_done
                        varMap[':isJumbo']     = EventServiceUtils.eventTableIsJumbo
                        self.cur.execute(sqlJJ+comment, varMap)
                        resJJList = self.cur.fetchall()
                        for jPandaID, in resJJList:
                            # get siteid of jumbo job
                            varMap = {}
                            varMap[':PandaID'] = jPandaID
                            self.cur.execute(sqlJJS+comment, varMap)
                            resJJS = self.cur.fetchone()
                            if resJJS is not None:
                                tmpStr = '{0} changed co-jumbo site {1} '.format(methodName,jobSpec.computingSite)
                                jobSpec.computingSite = resJJS[0]
                                tmpStr += 'to {0}'.format(jobSpec.computingSite)
                                toEscape = True
                                _logger.debug(tmpStr)
                                break
                        if toEscape:
                            break
                # change special handling and set the share to express for merge jobs
                EventServiceUtils.setEventServiceMerge(jobSpec)
                # set site
                self.setSiteForEsMerge(jobSpec, isFakeCJ, methodName, comment)
                jobSpec.coreCount = None
                jobSpec.minRamCount = 2000

            # reset resource type
            jobSpec.resource_type = self.get_resource_type_job(jobSpec)

            # no new job since ES is disabled
            if noNewJob:
                jobSpec.PandaID = None
                msgStr = '{0} No new job since event service is disabled or queue is offline or old job status {1} is not active'.format(methodName, currentJobStatus)
                _logger.debug(msgStr)
            else:
                # update input
                if doMerging:
                    self.updateInputStatusJedi(jobSpec.jediTaskID, jobSpec.PandaID, 'merging')
                else:
                    self.updateInputStatusJedi(jobSpec.jediTaskID, jobSpec.PandaID, 'queued', checkOthers=True)
                # insert job with new PandaID
                if jobSpec.jobStatus in ['defined','assigned']:
                    sql1  = "INSERT INTO ATLAS_PANDA.jobsDefined4 ({0}) ".format(JobSpec.columnNames())
                elif jobSpec.jobStatus in ['waiting','pending']:
                    sql1  = "INSERT INTO ATLAS_PANDA.jobsWaiting4 ({0}) ".format(JobSpec.columnNames())
                else:
                    sql1  = "INSERT INTO ATLAS_PANDA.jobsActive4 ({0}) ".format(JobSpec.columnNames())
                sql1 += JobSpec.bindValuesExpression(useSeq=True)
                sql1 += " RETURNING PandaID INTO :newPandaID"
                # set parentID
                jobSpec.parentID = jobSpec.PandaID
                varMap = jobSpec.valuesMap(useSeq=True)
                varMap[':newPandaID'] = self.cur.var(varNUMBER)
                # insert
                if not noNewJob:
                    retI = self.cur.execute(sql1+comment, varMap)
                    # set PandaID
                    val = self.getvalue_corrector(self.cur.getvalue(varMap[':newPandaID']))
                    jobSpec.PandaID = long(val)
                else:
                    jobSpec.PandaID = None
                msgStr = '{0} Generate new PandaID -> {1}#{2} at {3} '.format(methodName,jobSpec.PandaID,jobSpec.attemptNr,
                                                                              jobSpec.computingSite)
                if doMerging:
                    msgStr += "for merge"
                else:
                    msgStr += "for retry"
                _logger.debug(msgStr)
                # insert files
                sqlFile = "INSERT INTO ATLAS_PANDA.filesTable4 ({0}) ".format(FileSpec.columnNames())
                sqlFile+= FileSpec.bindValuesExpression(useSeq=True)
                sqlFile+= " RETURNING row_ID INTO :newRowID"
                sqlMaxFail = "UPDATE {0}.JEDI_Dataset_Contents ".format(panda_config.schemaJEDI)
                sqlMaxFail += "SET maxFailure=(CASE "
                sqlMaxFail += "WHEN maxFailure IS NULL THEN failedAttempt+:increase "
                sqlMaxFail += "WHEN maxFailure>failedAttempt+:increase THEN failedAttempt+:increase "
                sqlMaxFail += "ELSE maxFailure "
                sqlMaxFail += "END) "
                sqlMaxFail += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                sqlMaxFail += "AND keepTrack=:keepTrack "
                for fileSpec in jobSpec.Files:
                    # skip zip
                    if fileSpec.type.startswith('zip'):
                        continue
                    # reset rowID
                    fileSpec.row_ID = None
                    # change GUID and LFN for log
                    if fileSpec.type == 'log':
                        fileSpec.GUID = str(uuid.uuid4())
                        if doMerging:
                            fileSpec.lfn = re.sub('\.{0}$'.format(pandaID),''.format(jobSpec.PandaID),fileSpec.lfn)
                        else:
                            fileSpec.lfn = re.sub('\.{0}$'.format(pandaID),'.{0}'.format(jobSpec.PandaID),fileSpec.lfn)
                    # insert
                    varMap = fileSpec.valuesMap(useSeq=True)
                    varMap[':newRowID'] = self.cur.var(varNUMBER)
                    self.cur.execute(sqlFile+comment, varMap)
                    val = self.getvalue_corrector(self.cur.getvalue(varMap[':newRowID']))
                    fileSpec.row_ID = long(val)
                    # change max failure for esmerge
                    if doMerging and fileSpec.type in ['input', 'pseudo_input']:
                         varMap = {}
                         varMap[':jediTaskID'] = fileSpec.jediTaskID
                         varMap[':datasetID']  = fileSpec.datasetID
                         varMap[':fileID']     = fileSpec.fileID
                         varMap[':increase']   = 5
                         varMap[':keepTrack']  = 1
                         self.cur.execute(sqlMaxFail+comment, varMap)
                # insert job parameters
                sqlJob = "INSERT INTO ATLAS_PANDA.jobParamsTable (PandaID,jobParameters) VALUES (:PandaID,:param)"
                varMap = {}
                varMap[':PandaID'] = jobSpec.PandaID
                varMap[':param']   = jobSpec.jobParameters
                self.cur.execute(sqlJob+comment, varMap)
                # propagate change to JEDI
                if doMerging:
                    relationType = 'es_merge'
                else:
                    relationType = None
                self.updateForPilotRetryJEDI(jobSpec,self.cur,onlyHistory=True,relationType=relationType)
            # commit
            if useCommit:
                if not self._commit():
                    raise RuntimeError('Commit error')
            # set return
            if not doMerging:
                if closedInBadStatus:
                    # closed in bad status
                    retValue = 9,jobSpec.PandaID
                elif nRowDone == 0:
                    # didn't process any events
                    retValue = 8,jobSpec.PandaID
                else:
                    # processed some events
                    retValue = 0,jobSpec.PandaID
            else:
                if nRowDone == 0:
                    retValue = 10,jobSpec.PandaID
                else:
                    retValue = 2,jobSpec.PandaID
            # record status change
            try:
                if not noNewJob:
                    self.recordStatusChange(jobSpec.PandaID,jobSpec.jobStatus,jobInfo=jobSpec,useCommit=useCommit)
                    self.push_job_status_message(jobSpec, jobSpec.PandaID, jobSpec.jobStatus)
            except Exception:
                _logger.error('recordStatusChange in ppEventServiceJob')
            _logger.debug('{0} done for doMergeing={1}'.format(methodName,doMerging))
            if retValue[-1] == 'NULL':
                retValue = retValue[0],None
            return retValue
        except Exception:
            # roll back
            if useCommit:
                self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            return None,None



    # set site for ES merge
    def setSiteForEsMerge(self, jobSpec, isFakeCJ, methodName, comment):
        _logger.debug('{0} looking for ES merge site'.format(methodName))
        # merge on OS
        isMergeAtOS = EventServiceUtils.isMergeAtOS(jobSpec.specialHandling)
        # check where merge is done
        lookForMergeSite = True
        sqlWM  = "SELECT /* use_json_type */ scj.data.catchall, scj.data.objectstores " \
                 "FROM ATLAS_PANDA.schedconfig_json scj " \
                 "WHERE scj.panda_queue=:siteid "
        
        varMap = {}
        varMap[':siteid'] = jobSpec.computingSite
        self.cur.execute(sqlWM+comment, varMap)
        resWM = self.cur.fetchone()
        resSN = []
        resSN_back = []
        catchAll, objectstores = None, None
        if resWM is not None:
            catchAll,objectstores = resWM
        if catchAll is None:
            catchAll = ''
        try:
            if isFakeCJ:
                objectstores = []
            else:
                objectstores = json.loads(objectstores)
        except Exception:
            objectstores = []
        # get objstoreIDs
        sqlZIP = "SELECT /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
        sqlZIP += "DISTINCT zipRow_ID FROM {0}.JEDI_Events ".format(panda_config.schemaJEDI)
        sqlZIP += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
        sqlZIP += "AND status=:esDone "
        sqlOST = "SELECT fsize,destinationSE FROM {0}.filesTable4 ".format(panda_config.schemaPANDA)
        sqlOST += "WHERE row_ID=:row_ID "
        sqlOST += "UNION "
        sqlOST += "SELECT fsize,destinationSE FROM {0}.filesTable_ARCH ".format(panda_config.schemaPANDAARCH)
        sqlOST += "WHERE row_ID=:row_ID "
        objStoreZipMap = dict()
        storageZipMap = dict()
        zipRowIDs = set()
        totalZipSize = 0
        for tmpFileSpec in jobSpec.Files:
            if tmpFileSpec.type in ['input', 'pseudo_input']:
                varMap = dict()
                varMap[':jediTaskID'] = tmpFileSpec.jediTaskID
                varMap[':datasetID']  = tmpFileSpec.datasetID
                varMap[':fileID']     = tmpFileSpec.fileID
                varMap[':esDone']     = EventServiceUtils.ST_done
                self.cur.execute(sqlZIP+comment,varMap)
                resZIP = self.cur.fetchall()
                for zipRowID, in resZIP:
                    if zipRowID is None:
                        continue
                    if zipRowID in zipRowIDs:
                        continue
                    zipRowIDs.add(zipRowID)
                    # get file info
                    varMap = dict()
                    varMap[':row_ID'] = zipRowID
                    self.cur.execute(sqlOST+comment, varMap)
                    resOST = self.cur.fetchone()
                    tmpFsize, tmpDestSE = resOST
                    totalZipSize += tmpFsize
                    tmpRSE = self.convertObjIDtoEndPoint(panda_config.endpoint_mapfile, int(tmpDestSE.split('/')[0]))
                    if tmpRSE is not None:
                        objStoreZipMap.setdefault(tmpRSE['name'], 0)
                        objStoreZipMap[tmpRSE['name']] += tmpFsize
                        if tmpRSE['type'].endswith('DISK'):
                            storageZipMap.setdefault(tmpRSE['name'], 0)
                            storageZipMap[tmpRSE['name']] += tmpFsize
        if len(storageZipMap) > 0:
            sortedOST = sorted(storageZipMap.items(), key=operator.itemgetter(1))
        else:
            sortedOST = sorted(objStoreZipMap.items(), key=operator.itemgetter(1))
        sortedOST.reverse()
        if len(sortedOST) > 0:
            _logger.debug('{0} old objectstores {1}'.format(methodName, str(objectstores)))
            objectstores = [{'ddmendpoint': sortedOST[0][0]}]
            _logger.debug('{0} new objectstores {1}'.format(methodName, str(objectstores)))
        if isFakeCJ:
            # use nucleus for fake co-jumbo since they don't have sub datasets
            pass
        elif 'localEsMergeNC' in catchAll:
            # no site change
            lookForMergeSite = False
        else:
            # get sites in the nucleus associated to the site to run merge jobs in the same nucleus
            sqlSN  = "SELECT /* use_json_type */ dr.panda_site_name, dr.ddm_endpoint_name "
            sqlSN += "FROM ATLAS_PANDA.panda_site ps1, ATLAS_PANDA.panda_site ps2, ATLAS_PANDA.schedconfig_json sc, ATLAS_PANDA.panda_ddm_relation dr "
            sqlSN += "WHERE ps1.panda_site_name=:site AND ps1.site_name=ps2.site_name AND sc.panda_queue=ps2.panda_site_name "
            sqlSN += "AND dr.panda_site_name=ps2.panda_site_name "
            sqlSN += "AND (sc.data.corecount IS NULL OR sc.data.corecount=1 OR sc.data.capability=:capability) "
            sqlSN += "AND (sc.data.maxtime=0 OR sc.data.maxtime>=86400) "
            sqlSN += "AND (sc.data.maxrss IS NULL OR sc.data.minrss=0) "
            sqlSN += "AND (sc.data.jobseed IS NULL OR sc.data.jobseed<>'es') "
            sqlSN += "AND sc.data.type != 'analysis' "
            
            if 'localEsMerge' in catchAll and 'useBrokerOff' in catchAll:
                sqlSN += "AND sc.data.status IN (:siteStatus1,:siteStatus2) "
            else:
                sqlSN += "AND sc.data.status=:siteStatus "
            
            sqlSN += "AND dr.default_write ='Y' "
            sqlSN += "AND (scope = 'default' OR scope IS NULL) "  # skip endpoints with analysis roles
            sqlSN += "AND (sc.data.wnconnectivity IS NULL OR sc.data.wnconnectivity LIKE :wc1) "
            
            varMap = {}
            varMap[':site'] = jobSpec.computingSite
            if 'localEsMerge' in catchAll and 'useBrokerOff' in catchAll:
                varMap[':siteStatus1'] = 'online'
                varMap[':siteStatus2'] = 'brokeroff'
            else:
                varMap[':siteStatus'] = 'online'
            varMap[':wc1'] = 'full%'
            varMap[':capability'] = 'ucore'
            # get sites
            self.cur.execute(sqlSN+comment,varMap)
            if 'localEsMerge' in catchAll:
                resSN = self.cur.fetchall()
            else:
                resSN_back = self.cur.fetchall()
        if len(resSN) == 0 and lookForMergeSite:
            # run merge jobs at destination
            if not jobSpec.destinationSE.startswith('nucleus:'):
                jobSpec.computingSite = jobSpec.destinationSE
                lookForMergeSite = False
            else:
                # use nucleus close to OS
                tmpNucleus = None
                if isMergeAtOS and len(objectstores) > 0:
                    osEndpoint = objectstores[0]['ddmendpoint']
                    sqlCO = "SELECT site_name FROM ATLAS_PANDA.ddm_endpoint WHERE ddm_endpoint_name=:osEndpoint "
                    varMap = dict()
                    varMap[':osEndpoint'] = osEndpoint
                    self.cur.execute(sqlCO+comment,varMap)
                    resCO = self.cur.fetchone()
                    if resCO is not None:
                        tmpNucleus, = resCO
                        _logger.info('{0} look for merge sites in nucleus:{1} close to pre-merged files'.format(methodName,tmpNucleus))
                # use nucleus
                if tmpNucleus is None:
                    tmpNucleus = jobSpec.destinationSE.split(':')[-1]
                    _logger.info('{0} look for merge sites in destination nucleus:{1}'.format(methodName,tmpNucleus))
                # get sites in a nucleus
                sqlSN  = "SELECT /* use_json_type */ dr.panda_site_name, dr.ddm_endpoint_name "
                sqlSN += "FROM ATLAS_PANDA.panda_site ps, ATLAS_PANDA.schedconfig_json sc, ATLAS_PANDA.panda_ddm_relation dr "
                sqlSN += "WHERE site_name=:nucleus AND sc.panda_queue=ps.panda_site_name "
                sqlSN += "AND dr.panda_site_name=ps.panda_site_name "
                sqlSN += "AND (sc.data.corecount IS NULL OR sc.data.corecount=1 OR sc.data.capability=:capability) "
                sqlSN += "AND (sc.maxtime=0 OR sc.maxtime>=86400) "
                sqlSN += "AND (sc.maxrss IS NULL OR sc.minrss=0) "
                sqlSN += "AND (sc.jobseed IS NULL OR sc.jobseed<>'es') "
                sqlSN += "AND sc.data.type != 'analysis' "
                sqlSN += "AND sc.data.status=:siteStatus "
                sqlSN += "AND dr.default_write='Y' "
                sqlSN += "AND (dr.scope = 'default' OR dr.scope IS NULL) "  # skip endpoints with analysis roles
                sqlSN += "AND (sc.data.wnconnectivity IS NULL OR sc.data.wnconnectivity LIKE :wc1) "
                
                varMap = {}
                varMap[':nucleus'] = tmpNucleus
                varMap[':siteStatus'] = 'online'
                varMap[':wc1'] = 'full%'
                varMap[':capability'] = 'ucore'
                # get sites
                self.cur.execute(sqlSN+comment,varMap)
                resSN = self.cur.fetchall()
        
        # last resort for jumbo
        resSN_all = []
        if lookForMergeSite and (isFakeCJ or 'useJumboJobs' in catchAll or len(resSN + resSN_back) == 0):
            sqlSN  = "SELECT /* use_json_type */ dr.panda_site_name, dr.ddm_endpoint_name "
            sqlSN += "FROM ATLAS_PANDA.panda_site ps, ATLAS_PANDA.schedconfig_json sc, ATLAS_PANDA.panda_ddm_relation dr "
            sqlSN += "WHERE sc.panda_queue=ps.panda_site_name "
            sqlSN += "AND dr.panda_site_name=ps.panda_site_name "
            sqlSN += "AND (sc.data.corecount IS NULL OR sc.data.corecount=1 OR sc.data.capability=:capability) "
            sqlSN += "AND (sc.data.maxtime=0 OR sc.data.maxtime>=86400) "
            sqlSN += "AND (sc.data.maxrss IS NULL OR sc.data.minrss=0) "
            sqlSN += "AND (sc.data.jobseed IS NULL OR sc.data.jobseed<>'es') "
            sqlSN += "AND sc.data.type != 'analysis' "
            sqlSN += "AND sc.data.status=:siteStatus "
            sqlSN += "AND dr.default_write='Y' "
            sqlSN += "AND (dr.scope = 'default' OR dr.scope IS NULL) " # skip endpoints with analysis roles
            sqlSN += "AND (sc.data.wnconnectivity IS NULL OR sc.data.wnconnectivity LIKE :wc1) "
            
            varMap = {}
            varMap[':siteStatus'] = 'online'
            varMap[':wc1'] = 'full%'
            varMap[':capability'] = 'ucore'
            
            # get sites
            self.cur.execute(sqlSN+comment,varMap)
            resSN_all = self.cur.fetchall()
        
        # look for a site for merging
        if lookForMergeSite:
            # compare number of pilot requests
            maxNumPilot = 0
            sqlUG  = "SELECT updateJob+getJob FROM ATLAS_PANDAMETA.sitedata "
            sqlUG += "WHERE site=:panda_site AND HOURS=:hours AND FLAG=:flag "
            
            sqlRJ  = "SELECT SUM(num_of_jobs) FROM ATLAS_PANDA.MV_JOBSACTIVE4_STATS "
            sqlRJ += "WHERE computingSite=:panda_site AND jobStatus=:jobStatus "
            
            newSiteName = None
            for resItem in [resSN, resSN_back, resSN_all]:
                for tmp_panda_site_name,tmp_ddm_endpoint in resItem:
                    # get nPilot
                    varMap = {}
                    varMap[':panda_site'] = tmp_panda_site_name
                    varMap[':hours'] = 3
                    varMap[':flag'] = 'production'
                    self.cur.execute(sqlUG+comment,varMap)
                    resUG = self.cur.fetchone()
                    if resUG is None:
                        nPilots = 0
                    else:
                        nPilots, = resUG
                    # get nRunning
                    varMap = {}
                    varMap[':panda_site'] = tmp_panda_site_name
                    varMap[':jobStatus'] = 'running'
                    self.cur.execute(sqlRJ+comment,varMap)
                    resRJ = self.cur.fetchone()
                    if resRJ is None:
                        nRunning = 0
                    else:
                        nRunning, = resRJ
                    tmpStr = 'site={0} nPilot={1} nRunning={2}'.format(tmp_panda_site_name, nPilots, nRunning)
                    _logger.info('{0} {1}'.format(methodName,tmpStr))
                    # use larger
                    if maxNumPilot < nPilots:
                        maxNumPilot = nPilots
                        jobSpec.computingSite = tmp_panda_site_name
                        newSiteName = jobSpec.computingSite
                        for tmpFileSpec in jobSpec.Files:
                            if tmpFileSpec.destinationDBlockToken.startswith('ddd:'):
                                tmpFileSpec.destinationDBlockToken = 'ddd:{0}'.format(tmp_ddm_endpoint)
                                tmpFileSpec.destinationSE = jobSpec.computingSite
                if newSiteName is not None:
                    _logger.info('{0} set merge site to {1}'.format(methodName,newSiteName))
                    break
        # return
        return



    # set score site to ES job
    def setScoreSiteToEs(self, jobSpec, methodName, comment):
        _logger.debug('{0} looking for SCORE site'.format(methodName))

        # get score PQ in the nucleus associated to the site to run the small ES job
        sqlSN  = "SELECT /* use_json_type */ ps2.panda_site_name "
        sqlSN += "FROM ATLAS_PANDA.panda_site ps1, ATLAS_PANDA.panda_site ps2, ATLAS_PANDA.schedconfig_json sc "
        sqlSN += "WHERE ps1.panda_site_name=:site AND ps1.site_name=ps2.site_name AND sc.panda_queue=ps2.panda_site_name "
        sqlSN += "AND (sc.data.corecount IS NULL OR sc.data.corecount=1 OR sc.data.capability=:capability) "
        sqlSN += "AND (sc.data.jobseed IS NULL OR sc.data.jobseed<>'std') "
        sqlSN += "AND sc.data.status=:siteStatus "

        varMap = {}
        varMap[':site'] = jobSpec.computingSite
        varMap[':siteStatus'] = 'online'
        varMap[':capability'] = 'ucore'

        # get sites
        self.cur.execute(sqlSN+comment,varMap)
        resSN = self.cur.fetchall()
        # compare number of pilot requests
        maxNumPilot = 0
        sqlUG  = "SELECT updateJob+getJob FROM ATLAS_PANDAMETA.sitedata "
        sqlUG += "WHERE site=:panda_site AND HOURS=:hours AND FLAG=:flag "
        sqlRJ  = "SELECT SUM(num_of_jobs) FROM ATLAS_PANDA.MV_JOBSACTIVE4_STATS "
        sqlRJ += "WHERE computingSite=:panda_site AND jobStatus=:jobStatus "
        newSiteName = None
        for tmp_panda_site_name, in resSN:
            # get nPilot
            varMap = {}
            varMap[':panda_site'] = tmp_panda_site_name
            varMap[':hours'] = 3
            varMap[':flag'] = 'production'
            self.cur.execute(sqlUG+comment,varMap)
            resUG = self.cur.fetchone()
            if resUG is None:
                nPilots = 0
            else:
                nPilots, = resUG
            # get nRunning
            varMap = {}
            varMap[':panda_site'] = tmp_panda_site_name
            varMap[':jobStatus'] = 'running'
            self.cur.execute(sqlRJ+comment,varMap)
            resRJ = self.cur.fetchone()
            if resRJ is None:
                nRunning = 0
            else:
                nRunning, = resRJ
            tmpStr = 'site={0} nPilot={1} nRunning={2}'.format(tmp_panda_site_name, nPilots, nRunning)
            _logger.info('{0} {1}'.format(methodName,tmpStr))
            # use larger
            if maxNumPilot < nPilots:
                maxNumPilot = nPilots
                jobSpec.computingSite = tmp_panda_site_name
                jobSpec.coreCount = 1
                jobSpec.minRamCount = 0
                jobSpec.resource_type = self.get_resource_type_job(jobSpec)
                newSiteName = jobSpec.computingSite
        if newSiteName is not None:
            _logger.info('{0} set SCORE site to {1}'.format(methodName, newSiteName))
        else:
            _logger.info('{0} no SCORE site for {1}'.format(methodName, jobSpec.computingSite))
        # return
        return



    # change task priority
    def changeTaskPriorityPanda(self,jediTaskID,newPriority):
        comment = ' /* DBProxy.changeTaskPriorityPanda */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " <jediTaskID={0}>".format(jediTaskID)
        _logger.debug("{0} newPrio={1}".format(methodName,newPriority))
        try:
            # sql to update JEDI task table
            sqlT  = 'UPDATE {0}.JEDI_Tasks SET currentPriority=:newPriority WHERE jediTaskID=:jediTaskID '.format(panda_config.schemaJEDI)
            # sql to update DEFT task table
            schemaDEFT = self.getSchemaDEFT()
            sqlD  = 'UPDATE {0}.T_TASK SET current_priority=:newPriority,timestamp=CURRENT_DATE WHERE taskid=:jediTaskID '.format(schemaDEFT)
            # update job priorities
            sqlJ  = 'UPDATE ATLAS_PANDA.{0} SET currentPriority=:newPriority WHERE jediTaskID=:jediTaskID '
            # start transaction
            self.conn.begin()
            # select
            self.cur.arraysize = 10
            varMap = {}
            varMap[':jediTaskID']  = jediTaskID
            varMap[':newPriority'] = newPriority
            # update JEDI
            self.cur.execute(sqlT+comment, varMap)
            nRow = self.cur.rowcount
            if nRow == 1:
                # update jobs
                for tableName in ['jobsActive4','jobsDefined4','jobsWaiting4']:
                    self.cur.execute(sqlJ.format(tableName)+comment, varMap)
            # update DEFT
            self.cur.execute(sqlD+comment, varMap)
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            _logger.debug("{0} done with {1}".format(methodName,nRow))
            return nRow
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            return None



    # get WAN data flow matrix
    def getWanDataFlowMaxtrix(self):
        comment = ' /* DBProxy.getWanDataFlowMaxtrix */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        _logger.debug("{0} start".format(methodName))
        try:
            # sql to get data flow
            sqlT  = 'SELECT PandaID,jobStatus,prodUserName,computingSite,sourceSite,maxCpuCount,inputFileBytes,currentPriority '
            sqlT += 'FROM ATLAS_PANDA.jobsActive4 '
            sqlT += 'WHERE prodSourceLabel=:prodSourceLabel '
            sqlT += 'AND jobStatus IN (:jobStatus1,:jobStatus2,:jobStatus3,:jobStatus4,:jobStatus5) '
            sqlT += 'AND transferType=:transferType AND relocationFlag=:relocationFlag '
            # start transaction
            self.conn.begin()
            # select
            self.cur.arraysize = 100000
            varMap = {}
            varMap[':jobStatus1']   = 'activated'
            varMap[':jobStatus2']   = 'running'
            varMap[':jobStatus3']   = 'throttled'
            varMap[':jobStatus4']   = 'sent'
            varMap[':jobStatus5']   = 'starting'
            varMap[':transferType'] = 'fax'
            varMap[':prodSourceLabel'] = 'user'
            varMap[':relocationFlag'] = 1
            # get data
            self.cur.execute(sqlT+comment,varMap)
            resFs = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # loop over all data
            retMap = {}
            for pandaID,jobStatus,prodUserName,computingSite,sourceSite,maxCpuCount,inputFileBytes,currentPriority in resFs:
                # add sink
                if computingSite not in retMap:
                    retMap[computingSite] = {}
                # add source
                if sourceSite not in retMap[computingSite]:
                    retMap[computingSite][sourceSite] = {'flow':0,'user':{}}
                # add user
                if prodUserName not in retMap[computingSite][sourceSite]['user']:
                    retMap[computingSite][sourceSite]['user'][prodUserName] = {'activated':{'nJobs':0,'jobList':{}},
                                                                               'throttled':{'nJobs':0,'jobList':{}}
                                                                               }
                # action for each jobStatus
                if jobStatus in ['activated','throttled']:
                    # collect PandaIDs
                    if currentPriority not in retMap[computingSite][sourceSite]['user'][prodUserName][jobStatus]['jobList']:
                       retMap[computingSite][sourceSite]['user'][prodUserName][jobStatus]['jobList'][currentPriority] = []
                    retMap[computingSite][sourceSite]['user'][prodUserName][jobStatus]['jobList'][currentPriority].append(pandaID)
                    # number of jobs
                    retMap[computingSite][sourceSite]['user'][prodUserName][jobStatus]['nJobs'] += 1
                else:
                    # calcurate total flow in bps
                    if maxCpuCount in [0,None]:
                        # use the default data flow
                        dataFlow = 1024*8
                    else:
                        dataFlow = inputFileBytes*8/maxCpuCount
                    retMap[computingSite][sourceSite]['flow'] += dataFlow
            _logger.debug("{0} done".format(methodName))
            return retMap
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            return {}



    # throttle job
    def throttleJob(self,pandaID):
        comment = ' /* DBProxy.throttleJob */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " <PandaID={0}>".format(pandaID)
        _logger.debug("{0} start".format(methodName))
        try:
            # sql to update job
            sqlT  = 'UPDATE ATLAS_PANDA.jobsActive4 SET currentPriority=assignedPriority,jobStatus=:newJobStatus '
            sqlT += 'WHERE PandaID=:PandaID AND jobStatus=:oldJobStatus '
            # start transaction
            self.conn.begin()
            # select
            self.cur.arraysize = 10
            varMap = {}
            varMap[':PandaID']      = pandaID
            varMap[':newJobStatus'] = 'throttled'
            varMap[':oldJobStatus'] = 'activated'
            # get datasets
            self.cur.execute(sqlT+comment, varMap)
            nRow = self.cur.rowcount
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            try:
                self.recordStatusChange(pandaID,varMap[':newJobStatus'])
            except Exception:
                _logger.error('recordStatusChange in throttleJob')
            _logger.debug("{0} done with {1}".format(methodName,nRow))
            return nRow
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            return None



    # throttle user jobs
    def throttleUserJobs(self, prodUserName, workingGroup, get_dict):
        comment = ' /* DBProxy.throttleUserJobs */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " <user={0} group={1}>".format(prodUserName, workingGroup)
        tmpLog = LogWrapper(_logger,methodName)
        tmpLog.debug("start")
        try:
            # sql to get tasks
            sqlT  = 'SELECT /*+ INDEX_RS_ASC(tab JOBSACTIVE4_PRODUSERNAMEST_IDX) */ '
            sqlT += 'distinct jediTaskID '
            sqlT += 'FROM ATLAS_PANDA.jobsActive4 tab '
            sqlT += 'WHERE prodSourceLabel=:prodSourceLabel AND prodUserName=:prodUserName '
            sqlT += 'AND jobStatus=:oldJobStatus AND relocationFlag=:oldRelFlag '
            sqlT += 'AND maxCpuCount>:maxTime '
            if workingGroup is not None:
                sqlT += 'AND workingGroup=:workingGroup '
            else:
                sqlT += 'AND workingGroup IS NULL '
            # sql to get jobs
            sqlJ  = 'SELECT '\
                    'PandaID, jediTaskID, cloud, computingSite, prodSourceLabel '\
                    'FROM ATLAS_PANDA.jobsActive4 '\
                    'WHERE jediTaskID=:jediTaskID '\
                    'AND jobStatus=:oldJobStatus AND relocationFlag=:oldRelFlag '\
                    'AND maxCpuCount>:maxTime '
            # sql to update job
            sqlU = ("UPDATE {0}.jobsActive4 SET jobStatus=:newJobStatus,relocationFlag=:newRelFlag "
                    "WHERE jediTaskID=:jediTaskID AND jobStatus=:oldJobStatus AND maxCpuCount>:maxTime").format(
                panda_config.schemaPANDA)
            # start transaction
            self.conn.begin()
            # select
            self.cur.arraysize = 10
            varMap = {}
            varMap[':prodSourceLabel'] = 'user'
            varMap[':oldRelFlag'] = 1
            varMap[':prodUserName'] = prodUserName
            varMap[':oldJobStatus'] = 'activated'
            varMap[':maxTime'] = 6 * 60 * 60
            if workingGroup is not None:
                varMap[':workingGroup'] = workingGroup
            # get tasks
            self.cur.execute(sqlT+comment, varMap)
            resT = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # loop over all tasks
            tasks = [jediTaskID for jediTaskID, in resT]
            random.shuffle(tasks)
            nRow = 0
            nRows = {}
            for jediTaskID in tasks:
                tmpLog.debug("reset jediTaskID={0}".format(jediTaskID))
                # start transaction
                self.conn.begin()
                # get jobs
                varMap = {}
                varMap[':jediTaskID'] = jediTaskID
                varMap[':oldRelFlag'] = 1
                varMap[':oldJobStatus'] = 'activated'
                varMap[':maxTime'] = 6 * 60 * 60
                self.cur.execute(sqlJ + comment, varMap)
                resJ = self.cur.fetchall()
                infoMapDict = {pandaID: {'computingSite': computingSite,
                                         'cloud': cloud,
                                         'prodSourceLabel': prodSourceLabel}
                               for pandaID, taskID, cloud, computingSite, prodSourceLabel in resJ}
                # update jobs
                varMap = {}
                varMap[':jediTaskID'] = jediTaskID
                varMap[':newRelFlag'] = 3
                varMap[':newJobStatus'] = 'throttled'
                varMap[':oldJobStatus'] = 'activated'
                varMap[':maxTime'] = 6 * 60 * 60
                self.cur.execute(sqlU + comment, varMap)
                nTmp = self.cur.rowcount
                tmpLog.debug("reset {0} jobs".format(nTmp))
                if nTmp > 0:
                    nRow += nTmp
                    nRows[jediTaskID] = nTmp
                for pandaID in infoMapDict:
                    infoMap = infoMapDict[pandaID]
                    self.recordStatusChange(pandaID, varMap[':newJobStatus'], infoMap=infoMap, useCommit=False)
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
            if get_dict:
                tmpLog.debug("done with {0}".format(nRows))
                return nRows
            tmpLog.debug("done with {0}".format(nRow))
            return nRow
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            return None



    # unthrottle job
    def unThrottleJob(self,pandaID):
        comment = ' /* DBProxy.unThrottleJob */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " <PandaID={0}>".format(pandaID)
        _logger.debug("{0} start".format(methodName))
        try:
            # sql to update job
            sqlT  = 'UPDATE ATLAS_PANDA.jobsActive4 SET jobStatus=:newJobStatus '
            sqlT += 'WHERE PandaID=:PandaID AND jobStatus=:oldJobStatus '
            # start transaction
            self.conn.begin()
            # select
            self.cur.arraysize = 10
            varMap = {}
            varMap[':PandaID']      = pandaID
            varMap[':newJobStatus'] = 'activated'
            varMap[':oldJobStatus'] = 'throttled'
            # get datasets
            self.cur.execute(sqlT+comment, varMap)
            nRow = self.cur.rowcount
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            try:
                self.recordStatusChange(pandaID,varMap[':newJobStatus'])
            except Exception:
                _logger.error('recordStatusChange in unThrottleJob')
            _logger.debug("{0} done with {1}".format(methodName,nRow))
            return nRow
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            return None



    # unthrottle user jobs
    def unThrottleUserJobs(self, prodUserName, workingGroup, get_dict):
        comment = ' /* DBProxy.unThrottleUserJobs */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " <user={0} group={1}>".format(prodUserName, workingGroup)
        tmpLog = LogWrapper(_logger,methodName)
        tmpLog.debug("start")
        try:
            # sql to get tasks
            sqlT  = 'SELECT /*+ INDEX_RS_ASC(tab JOBSACTIVE4_PRODUSERNAMEST_IDX) */ '
            sqlT += 'distinct jediTaskID '
            sqlT += 'FROM ATLAS_PANDA.jobsActive4 tab '
            sqlT += 'WHERE prodSourceLabel=:prodSourceLabel AND prodUserName=:prodUserName '
            sqlT += 'AND jobStatus=:oldJobStatus AND relocationFlag=:oldRelFlag '
            if workingGroup is not None:
                sqlT += 'AND workingGroup=:workingGroup '
            else:
                sqlT += 'AND workingGroup IS NULL '
            # sql to get jobs
            sqlJ  = 'SELECT '\
                    'PandaID, jediTaskID, cloud, computingSite, prodSourceLabel '\
                    'FROM ATLAS_PANDA.jobsActive4 '\
                    'WHERE jediTaskID=:jediTaskID '\
                    'AND jobStatus=:oldJobStatus AND relocationFlag=:oldRelFlag '
            # sql to update job
            sqlU = ("UPDATE {0}.jobsActive4 SET jobStatus=:newJobStatus,relocationFlag=:newRelFlag "
                    "WHERE jediTaskID=:jediTaskID AND jobStatus=:oldJobStatus ").format(
                panda_config.schemaPANDA)
            # start transaction
            self.conn.begin()
            # select
            self.cur.arraysize = 10
            varMap = {}
            varMap[':prodSourceLabel'] = 'user'
            varMap[':oldRelFlag'] = 3
            varMap[':prodUserName'] = prodUserName
            varMap[':oldJobStatus'] = 'throttled'
            if workingGroup is not None:
                varMap[':workingGroup'] = workingGroup
            # get tasks
            self.cur.execute(sqlT+comment, varMap)
            resT = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # loop over all tasks
            tasks = [jediTaskID for jediTaskID, in resT]
            random.shuffle(tasks)
            nRow = 0
            nRows = {}
            for jediTaskID in tasks:
                tmpLog.debug("reset jediTaskID={0}".format(jediTaskID))
                # start transaction
                self.conn.begin()
                # get jobs
                varMap = {}
                varMap[':jediTaskID'] = jediTaskID
                varMap[':oldRelFlag'] = 3
                varMap[':oldJobStatus'] = 'throttled'
                self.cur.execute(sqlJ + comment, varMap)
                resJ = self.cur.fetchall()
                infoMapDict = {pandaID: {'computingSite': computingSite,
                                         'cloud': cloud,
                                         'prodSourceLabel': prodSourceLabel}
                               for pandaID, taskID, cloud, computingSite, prodSourceLabel in resJ}
                # update jobs
                varMap = {}
                varMap[':jediTaskID'] = jediTaskID
                varMap[':newRelFlag'] = 1
                varMap[':newJobStatus'] = 'activated'
                varMap[':oldJobStatus'] = 'throttled'
                self.cur.execute(sqlU + comment, varMap)
                nTmp = self.cur.rowcount
                tmpLog.debug("reset {0} jobs".format(nTmp))
                if nTmp > 0:
                    nRow += nTmp
                    nRows[jediTaskID] = nTmp
                for pandaID in infoMapDict:
                    infoMap = infoMapDict[pandaID]
                    self.recordStatusChange(pandaID, varMap[':newJobStatus'], infoMap=infoMap, useCommit=False)
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
            if get_dict:
                tmpLog.debug("done with {0}".format(nRows))
                return nRows
            tmpLog.debug("done with {0}".format(nRow))
            return nRow
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            return None



    # get throttled users
    def getThrottledUsers(self):
        comment = ' /* DBProxy.getThrottledUsers */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        tmpLog = LogWrapper(_logger,methodName)
        tmpLog.debug("start")
        retVal = set()
        try:
            # sql to get users
            sqlT  = 'SELECT distinct prodUserName,workingGroup FROM ATLAS_PANDA.jobsActive4 '
            sqlT += 'WHERE prodSourceLabel=:prodSourceLabel AND jobStatus=:jobStatus AND relocationFlag=:relocationFlag '
            # start transaction
            self.conn.begin()
            # select
            self.cur.arraysize = 10
            varMap = {}
            varMap[':prodSourceLabel'] = 'user'
            varMap[':relocationFlag'] = 3
            varMap[':jobStatus'] = 'throttled'
            # get datasets
            self.cur.execute(sqlT+comment, varMap)
            resPs = self.cur.fetchall()
            for prodUserName, workingGroup in resPs:
                retVal.add((prodUserName, workingGroup))
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            tmpLog.debug("done with {0}".format(str(retVal)))
            return retVal
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            return []



    # kill active consumers related to an ES job
    def killEventServiceConsumers(self,job,killedFlag,useCommit=True):
        comment = ' /* DBProxy.killEventServiceConsumers */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " <PandaID={0}>".format(job.PandaID)
        _logger.debug("{0} : start".format(methodName))
        try:
            # begin transaction
            if useCommit:
                self.conn.begin()
            # sql to get consumers
            sqlCP  = "SELECT /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
            sqlCP += "distinct PandaID FROM {0}.JEDI_Events tab ".format(panda_config.schemaJEDI)
            sqlCP += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
            sqlCP += "AND NOT status IN (:esDiscarded,:esCancelled) "
            # sql to discard or cancel event ranges
            sqlDE  = "UPDATE /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
            sqlDE += "{0}.JEDI_Events tab ".format(panda_config.schemaJEDI)
            sqlDE += "SET status=:status "
            sqlDE += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID AND PandaID=:PandaID "
            sqlDE += "AND status IN (:esFinished,:esDone) "
            sqlCE  = "UPDATE /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
            sqlCE += "{0}.JEDI_Events tab ".format(panda_config.schemaJEDI)
            sqlCE += "SET status=:status "
            sqlCE += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID AND PandaID=:PandaID "
            sqlCE += "AND NOT status IN (:esFinished,:esDone,:esDiscarded,:esCancelled,:esFailed,:esFatal,:esCorrupted) "
            # look for consumers for each input
            killPandaIDs = {}
            for fileSpec in job.Files:
                if fileSpec.type not in ['input', 'pseudo_input']:
                    continue
                if fileSpec.fileID in ['NULL', None]:
                    continue
                # get PandaIDs
                varMap = {}
                varMap[':jediTaskID']  = fileSpec.jediTaskID
                varMap[':datasetID']   = fileSpec.datasetID
                varMap[':fileID']      = fileSpec.fileID
                varMap[':esDiscarded'] = EventServiceUtils.ST_discarded
                varMap[':esCancelled'] = EventServiceUtils.ST_cancelled
                self.cur.arraysize = 100000
                self.cur.execute(sqlCP+comment, varMap)
                resPs = self.cur.fetchall()
                for esPandaID, in resPs:
                    if esPandaID not in killPandaIDs:
                        killPandaIDs[esPandaID] = set()
                    killPandaIDs[esPandaID].add((fileSpec.jediTaskID,fileSpec.datasetID,fileSpec.fileID))
            # kill consumers
            sqlDJS = "SELECT %s " % JobSpec.columnNames()
            sqlDJS+= "FROM ATLAS_PANDA.jobsActive4 WHERE PandaID=:PandaID "
            sqlDJS+= "FOR UPDATE NOWAIT "
            sqlDJD = "DELETE FROM ATLAS_PANDA.jobsActive4 WHERE PandaID=:PandaID"
            sqlDJI = "INSERT INTO ATLAS_PANDA.jobsArchived4 (%s) " % JobSpec.columnNames()
            sqlDJI+= JobSpec.bindValuesExpression()
            sqlFSF  = "UPDATE ATLAS_PANDA.filesTable4 SET status=:newStatus "
            sqlFSF += "WHERE PandaID=:PandaID AND type IN (:type1,:type2) "
            sqlFMod = "UPDATE ATLAS_PANDA.filesTable4 SET modificationTime=:modificationTime WHERE PandaID=:PandaID"
            sqlMMod = "UPDATE ATLAS_PANDA.metaTable SET modificationTime=:modificationTime WHERE PandaID=:PandaID"
            sqlPMod = "UPDATE ATLAS_PANDA.jobParamsTable SET modificationTime=:modificationTime WHERE PandaID=:PandaID"
            nKilled = 0
            killPandaIDsList = list(killPandaIDs)
            killPandaIDsList.sort()
            for pandaID in killPandaIDsList:
                # ignore original PandaID since it will be killed by caller
                if pandaID == job.PandaID:
                    continue
                # skip jobsetID
                if pandaID == job.jobsetID:
                    continue
                # read job
                varMap = {}
                varMap[':PandaID'] = pandaID
                self.cur.arraysize = 10
                self.cur.execute(sqlDJS+comment, varMap)
                resJob = self.cur.fetchall()
                if len(resJob) == 0:
                    continue
                # instantiate JobSpec
                dJob = JobSpec()
                dJob.pack(resJob[0])
                # skip if jobset different
                if dJob.jobsetID != job.jobsetID:
                    _logger.debug("{0} : skip consumer {1} since jobsetID is different".format(methodName,pandaID))
                    continue
                # skip jumbo
                if EventServiceUtils.isJumboJob(dJob):
                    _logger.debug("{0} : skip jumbo {1}".format(methodName,pandaID))
                    continue
                _logger.debug("{0} : kill associated consumer {1}".format(methodName,pandaID))
                # delete
                varMap = {}
                varMap[':PandaID'] = pandaID
                self.cur.execute(sqlDJD+comment, varMap)
                retD = self.cur.rowcount
                if retD == 0:
                    continue
                # set error code
                dJob.endTime   = datetime.datetime.utcnow()
                if EventServiceUtils.isJobCloningJob(dJob):
                    dJob.jobStatus = 'closed'
                    dJob.jobSubStatus = 'jc_unlock'
                    dJob.taskBufferErrorCode = ErrorCode.EC_JobCloningUnlock
                    dJob.taskBufferErrorDiag = 'closed since another clone PandaID={0} got semaphore'.format(job.PandaID)
                elif killedFlag:
                    dJob.jobStatus = 'cancelled'
                    dJob.jobSubStatus = 'es_killed'
                    dJob.taskBufferErrorCode = ErrorCode.EC_EventServiceKillOK
                    dJob.taskBufferErrorDiag = 'killed since an associated consumer PandaID={0} was killed'.format(job.PandaID)
                else:
                    dJob.jobStatus = 'failed'
                    dJob.jobSubStatus = 'es_aborted'
                    dJob.taskBufferErrorCode = ErrorCode.EC_EventServiceKillNG
                    dJob.taskBufferErrorDiag = 'killed since an associated consumer PandaID={0} failed'.format(job.PandaID)
                dJob.modificationTime = dJob.endTime
                dJob.stateChangeTime  = dJob.endTime
                # insert
                self.cur.execute(sqlDJI+comment, dJob.valuesMap())
                # set file status
                varMap = {}
                varMap[':PandaID']   = pandaID
                varMap[':type1']     = 'output'
                varMap[':type2']     = 'log'
                varMap[':newStatus'] = 'failed'
                self.cur.execute(sqlFSF+comment,varMap)
                # update files,metadata,parametes
                varMap = {}
                varMap[':PandaID'] = pandaID
                varMap[':modificationTime'] = dJob.modificationTime
                self.cur.execute(sqlFMod+comment,varMap)
                self.cur.execute(sqlMMod+comment,varMap)
                self.cur.execute(sqlPMod+comment,varMap)
                nKilled += 1
                # discard event ranges
                nRowsDis = 0
                nRowsCan = 0
                for jediTaskID,datasetID,fileID in killPandaIDs[pandaID]:
                    varMap = {}
                    varMap[':jediTaskID'] = jediTaskID
                    varMap[':datasetID']  = datasetID
                    varMap[':fileID']     = fileID
                    varMap[':PandaID']    = pandaID
                    varMap[':status']     = EventServiceUtils.ST_discarded
                    varMap[':esFinished'] = EventServiceUtils.ST_finished
                    varMap[':esDone']     = EventServiceUtils.ST_done
                    if not job.notDiscardEvents():
                        self.cur.execute(sqlDE+comment, varMap)
                        nRowsDis += self.cur.rowcount
                    varMap[':status']      = EventServiceUtils.ST_cancelled
                    varMap[':esDiscarded'] = EventServiceUtils.ST_discarded
                    varMap[':esCancelled'] = EventServiceUtils.ST_cancelled
                    varMap[':esCorrupted'] = EventServiceUtils.ST_corrupted
                    varMap[':esFatal']     = EventServiceUtils.ST_fatal
                    varMap[':esFailed']    = EventServiceUtils.ST_failed
                    self.cur.execute(sqlCE+comment, varMap)
                    nRowsCan += self.cur.rowcount
                _logger.debug("{0} : {1} discarded {2} events".format(methodName,pandaID,nRowsDis))
                _logger.debug("{0} : {1} cancelled {2} events".format(methodName,pandaID,nRowsCan))
            # commit
            if useCommit:
                if not self._commit():
                    raise RuntimeError('Commit error')
            _logger.debug('{0} killed {1} jobs'.format(methodName,nKilled))
            return True
        except Exception:
            # roll back
            if useCommit:
                self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            if not useCommit:
                raise RuntimeError("{0} failed".format(methodName))
            return False



    # kill unused consumers related to an ES job
    def killUnusedEventServiceConsumers(self,job,useCommit=True,killAll=False, checkAttemptNr=False):
        comment = ' /* DBProxy.killUnusedEventServiceConsumers */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " <PandaID={0}>".format(job.PandaID)
        _logger.debug("{0} : start".format(methodName))
        try:
            # begin transaction
            if useCommit:
                self.conn.begin()
            self.cur.arraysize = 100000
            # get dataset
            sqlPD = "SELECT f.datasetID,f.fileID FROM ATLAS_PANDA.JEDI_Datasets d,ATLAS_PANDA.filesTable4 f "
            sqlPD += "WHERE d.jediTaskID=:jediTaskID AND d.type IN (:type1,:type2) AND d.masterID IS NULL "
            sqlPD += "AND f.PandaID=:PandaID AND f.jeditaskID=f.jediTaskID AND f.datasetID=d.datasetID "
            varMap = {}
            varMap[':jediTaskID']  = job.jediTaskID
            varMap[':PandaID'] = job.PandaID
            varMap[':type1'] = 'input'
            varMap[':type2'] = 'pseudo_input'
            self.cur.execute(sqlPD+comment, varMap)
            resPD = self.cur.fetchall()
            # get PandaIDs
            killPandaIDs = set()
            myAttemptNr = None
            sqlCP = "SELECT PandaID,attemptNr FROM ATLAS_PANDA.filesTable4 WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
            for datasetID, fileID in resPD:
                if fileID is None:
                    continue
                varMap = {}
                varMap[':jediTaskID']  = job.jediTaskID
                varMap[':datasetID']   = datasetID
                varMap[':fileID']      = fileID
                self.cur.execute(sqlCP+comment, varMap)
                resCP = self.cur.fetchall()
                for esPandaID, esAttemptNr in resCP:
                    if esPandaID == job.PandaID:
                        myAttemptNr = esAttemptNr
                        continue
                    killPandaIDs.add((esPandaID, esAttemptNr))
            # kill consumers
            nKilled = 0
            sqlDJS = "SELECT %s " % JobSpec.columnNames()
            sqlDJS+= "FROM ATLAS_PANDA.{0} WHERE PandaID=:PandaID"
            sqlDJD = "DELETE FROM ATLAS_PANDA.{0} WHERE PandaID=:PandaID"
            sqlDJI = "INSERT INTO ATLAS_PANDA.jobsArchived4 (%s) " % JobSpec.columnNames()
            sqlDJI+= JobSpec.bindValuesExpression()
            sqlFSF  = "UPDATE ATLAS_PANDA.filesTable4 SET status=:newStatus "
            sqlFSF += "WHERE PandaID=:PandaID AND type IN (:type1,:type2) "
            sqlFMod = "UPDATE ATLAS_PANDA.filesTable4 SET modificationTime=:modificationTime WHERE PandaID=:PandaID"
            sqlMMod = "UPDATE ATLAS_PANDA.metaTable SET modificationTime=:modificationTime WHERE PandaID=:PandaID"
            sqlPMod = "UPDATE ATLAS_PANDA.jobParamsTable SET modificationTime=:modificationTime WHERE PandaID=:PandaID"
            for pandaID, attemptNr in killPandaIDs:
                # read job
                varMap = {}
                varMap[':PandaID'] = pandaID
                self.cur.arraysize = 10
                deletedFlag = False
                notToDelete = False
                for tableName in ['jobsActive4','jobsDefined4','jobsWaiting4']:
                    # check attemptNr
                    if checkAttemptNr and attemptNr != myAttemptNr:
                        _logger.debug("{0} : skip to kill {1} since attemptNr:{2} is different from mine={3}".format(methodName, pandaID, attemptNr, myAttemptNr))
                        notToDelete = True
                        break
                    self.cur.execute(sqlDJS.format(tableName)+comment, varMap)
                    resJob = self.cur.fetchall()
                    if len(resJob) == 0:
                        continue
                    # instantiate JobSpec
                    dJob = JobSpec()
                    dJob.pack(resJob[0])
                    # not kill all status
                    if not killAll:
                        if dJob.jobStatus not in ['activated', 'assigned', 'throttled']:
                            _logger.debug("{0} : skip to kill unused consumer {1} since status={2}".format(methodName, pandaID, dJob.jobStatus))
                            notToDelete = True
                            break
                    # skip merge
                    if EventServiceUtils.isEventServiceMerge(dJob):
                        _logger.debug("{0} : skip to kill merge {1}".format(methodName, pandaID))
                        notToDelete = True
                        break
                    # skip jumbo
                    if EventServiceUtils.isJumboJob(dJob):
                        _logger.debug("{0} : skip to kill jumbo {1}".format(methodName, pandaID))
                        notToDelete = True
                        break
                    # delete
                    varMap = {}
                    varMap[':PandaID'] = pandaID
                    self.cur.execute(sqlDJD.format(tableName)+comment, varMap)
                    retD = self.cur.rowcount
                    if retD != 0:
                        deletedFlag = True
                        break
                # not to be deleted
                if notToDelete:
                    continue
                # not found
                if not deletedFlag:
                    _logger.debug("{0} : skip to kill {1} as already deleted".format(methodName, pandaID))
                    continue
                _logger.debug("{0} : kill unused consumer {1}".format(methodName,pandaID))
                # set error code
                dJob.jobStatus = 'closed'
                dJob.endTime   = datetime.datetime.utcnow()
                if EventServiceUtils.isJobCloningJob(dJob):
                    dJob.jobSubStatus = 'jc_unlock'
                    dJob.taskBufferErrorCode = ErrorCode.EC_JobCloningUnlock
                    dJob.taskBufferErrorDiag = 'closed since another clone PandaID={0} got semaphore while waiting in the queue'.format(job.PandaID)
                else:
                    dJob.jobSubStatus = 'es_unused'
                    dJob.taskBufferErrorCode = ErrorCode.EC_EventServiceUnused
                    dJob.taskBufferErrorDiag = 'killed since all event ranges were processed by other consumers while waiting in the queue'
                dJob.modificationTime = dJob.endTime
                dJob.stateChangeTime  = dJob.endTime
                # insert
                self.cur.execute(sqlDJI+comment, dJob.valuesMap())
                # set file status
                varMap = {}
                varMap[':PandaID']   = pandaID
                varMap[':type1']     = 'output'
                varMap[':type2']     = 'log'
                varMap[':newStatus'] = 'failed'
                self.cur.execute(sqlFSF+comment,varMap)
                # update files,metadata,parametes
                varMap = {}
                varMap[':PandaID'] = pandaID
                varMap[':modificationTime'] = dJob.modificationTime
                self.cur.execute(sqlFMod+comment,varMap)
                self.cur.execute(sqlMMod+comment,varMap)
                self.cur.execute(sqlPMod+comment,varMap)
                nKilled += 1
            # commit
            if useCommit:
                if not self._commit():
                    raise RuntimeError('Commit error')
            _logger.debug('{0} killed {1} jobs'.format(methodName,nKilled))
            return True
        except Exception:
            # roll back
            if useCommit:
                self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            if not useCommit:
                raise RuntimeError("{0} failed".format(methodName))
            return False



    # kill unused event ranges
    def killUnusedEventRanges(self,jediTaskID,jobsetID):
        comment = ' /* DBProxy.killUnusedEventRanges */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " <jediTaskID={0} jobsetID={1}>".format(jediTaskID,jobsetID)
        tmpLog = LogWrapper(_logger,methodName)
        # sql to kill event ranges
        varMap = {}
        varMap[':jediTaskID']  = jediTaskID
        varMap[':jobsetID']    = jobsetID
        varMap[':esReady']     = EventServiceUtils.ST_ready
        varMap[':esCancelled'] = EventServiceUtils.ST_cancelled
        sqlCE  = "UPDATE {0}.JEDI_Events ".format(panda_config.schemaJEDI)
        sqlCE += "SET status=:esCancelled "
        sqlCE += "WHERE jediTaskID=:jediTaskID AND pandaID=:jobsetID "
        sqlCE += "AND status=:esReady "
        self.cur.execute(sqlCE, varMap)
        nRowsCan = self.cur.rowcount
        tmpLog.debug("cancelled {0} events".format(nRowsCan))

    # release unprocessed events
    def release_unprocessed_events(self, jedi_task_id, panda_id):
        comment = ' /* DBProxy.release_unprocessed_events */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " <jediTaskID={0} PandaID={1}>".format(jedi_task_id, panda_id)
        tmpLog = LogWrapper(_logger, methodName)
        # look for hopeless events
        varMap = {}
        varMap[':jediTaskID'] = jedi_task_id
        varMap[':PandaID']    = panda_id
        varMap[':esReady']    = EventServiceUtils.ST_ready
        varMap[':esFinished'] = EventServiceUtils.ST_finished
        varMap[':esFailed']   = EventServiceUtils.ST_failed
        sqlBE = ("SELECT job_processID FROM {0}.JEDI_Events "
                 "WHERE jediTaskID=:jediTaskID AND pandaID=:PandaID "
                 "AND status NOT IN (:esReady,:esFinished,:esFailed) "
                 "AND attemptNr=1 ").format(panda_config.schemaJEDI)
        self.cur.execute(sqlBE, varMap)
        resBD = self.cur.fetchall()
        if len(resBD) > 0:
            # report very large loss
            c = iDDS_Client(idds.common.utils.get_rest_host())
            for sample_id, in resBD:
                tmpLog.debug('reporting large loss for id={0}'.format(sample_id))
                c.update_hyperparameter(workload_id=jedi_task_id, request_id=None, id=sample_id, loss=1e5)
        # release
        sqlCE  = "UPDATE {0}.JEDI_Events ".format(panda_config.schemaJEDI)
        sqlCE += "SET status=(CASE WHEN attemptNr>1 THEN :esReady ELSE :esFailed END),"\
                 "pandaID=(CASE WHEN attemptNr>1 THEN 0 ELSE pandaID END),"\
                 "attemptNr=attemptNr-1 "\
                 "WHERE jediTaskID=:jediTaskID AND pandaID=:PandaID "\
                 "AND status NOT IN (:esReady,:esFinished,:esFailed) "
        self.cur.execute(sqlCE, varMap)
        nRowsCan = self.cur.rowcount
        tmpLog.debug("released {0} events".format(nRowsCan))

    # kill used event ranges
    def killUsedEventRanges(self, jediTaskID, pandaID, notDiscardEvents=False):
        comment = ' /* DBProxy.killUsedEventRanges */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " <jediTaskID={0} pandaID={1}>".format(jediTaskID,pandaID)
        tmpLog = LogWrapper(_logger,methodName)
        # sql to discard or cancel event ranges
        sqlDE  = "UPDATE "
        sqlDE += "{0}.JEDI_Events tab ".format(panda_config.schemaJEDI)
        sqlDE += "SET status=:status "
        sqlDE += "WHERE jediTaskID=:jediTaskID AND PandaID=:PandaID "
        sqlDE += "AND status IN (:esFinished,:esDone) "
        sqlCE  = "UPDATE "
        sqlCE += "{0}.JEDI_Events tab ".format(panda_config.schemaJEDI)
        sqlCE += "SET status=:status "
        sqlCE += "WHERE jediTaskID=:jediTaskID AND PandaID=:PandaID "
        sqlCE += "AND NOT status IN (:esFinished,:esDone,:esDiscarded,:esCancelled,:esFailed,:esFatal,:esCorrupted) "
        varMap = {}
        varMap[':jediTaskID'] = jediTaskID
        varMap[':PandaID']    = pandaID
        varMap[':status']     = EventServiceUtils.ST_discarded
        varMap[':esFinished'] = EventServiceUtils.ST_finished
        varMap[':esDone']     = EventServiceUtils.ST_done
        if not notDiscardEvents:
            self.cur.execute(sqlDE+comment, varMap)
            nRowsDis = self.cur.rowcount
        else:
            nRowsDis = 0
        varMap[':status']      = EventServiceUtils.ST_cancelled
        varMap[':esDiscarded'] = EventServiceUtils.ST_discarded
        varMap[':esCancelled'] = EventServiceUtils.ST_cancelled
        varMap[':esCorrupted'] = EventServiceUtils.ST_corrupted
        varMap[':esFatal']     = EventServiceUtils.ST_fatal
        varMap[':esFailed']    = EventServiceUtils.ST_failed
        self.cur.execute(sqlCE+comment, varMap)
        nRowsCan = self.cur.rowcount
        tmpLog.debug("discarded {0} events".format(nRowsDis))
        tmpLog.debug("cancelled {0} events".format(nRowsCan))



    # set corrupted events
    def setCorruptedEventRanges(self, jediTaskID, pandaID):
        comment = ' /* DBProxy.setCorruptedEventRanges */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " <jediTaskID={0} pandaID={1}>".format(jediTaskID, pandaID)
        tmpLog = LogWrapper(_logger,methodName)
        # sql to get bad files
        sqlBD  = "SELECT lfn FROM ATLAS_PANDA.filesTable4 WHERE PandaID=:PandaID AND type=:type AND status=:status "
        # sql to get PandaID produced the bad file
        sqlPP  = "SELECT row_ID,PandaID FROM ATLAS_PANDA.filesTable4 WHERE lfn=:lfn AND type=:type "
        # sql to get PandaIDs with jobMetrics
        sqlJJ = "SELECT /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
        sqlJJ += "DISTINCT e.PandaID FROM ATLAS_PANDA.filesTable4 f,ATLAS_PANDA.JEDI_Events e "
        sqlJJ += "WHERE f.PandaID=:PandaID AND f.type IN (:type1,:type2) "
        sqlJJ += "AND e.jediTaskID=f.jediTaskID AND e.datasetID=f.datasetID AND e.fileID=f.fileID "
        # sql to get jobMetrics
        sqlJM = "SELECT jobMetrics FROM ATLAS_PANDA.jobsArchived4 WHERE PandaID=:PandaID "
        sqlJM += "UNION "
        sqlJM += "SELECT jobMetrics FROM ATLAS_PANDAARCH.jobsArchived WHERE PandaID=:PandaID AND modificationTime=CURRENT_DATE-90 "
        # sql to get dataset and file IDs
        sqlGI  = "SELECT datasetID,fileID FROM ATLAS_PANDA.filesTable4 "
        sqlGI += "WHERE PandaID=:PandaID AND type IN (:t1,:t2) "
        # sql to update event ranges
        sqlCE  = "UPDATE "
        sqlCE += "{0}.JEDI_Events tab ".format(panda_config.schemaJEDI)
        sqlCE += "SET status=:esCorrupted "
        sqlCE += "WHERE jediTaskID=:jediTaskID AND PandaID=:PandaID AND zipRow_ID=:row_ID "
        sqlCE += "AND datasetID=:datasetID AND fileID=:fileID AND status=:esDone "
        # sql to update event ranges with jobMetrics
        sqlJE  = "UPDATE "
        sqlJE += "{0}.JEDI_Events tab ".format(panda_config.schemaJEDI)
        sqlJE += "SET status=:esCorrupted "
        sqlJE += "WHERE jediTaskID=:jediTaskID AND PandaID=:PandaID "
        sqlJE += "AND datasetID=:datasetID AND fileID=:fileID AND status=:esDone "
        # get bad files
        varMap = {}
        varMap[':PandaID'] = pandaID
        varMap[':status']  = 'corrupted'
        varMap[':type']    = 'zipinput'
        self.cur.execute(sqlBD+comment, varMap)
        resBD = self.cur.fetchall()
        for lfn, in resBD:
            # get origon PandaID
            nCor = 0
            varMap = {}
            varMap[':lfn'] = lfn
            varMap[':type'] = 'zipoutput'
            self.cur.execute(sqlPP+comment, varMap)
            resPP = self.cur.fetchall()
            if len(resPP) > 0:
                # with zipoutput
                for zipRow_ID, oPandaID in resPP:
                    # get dataset and file IDs
                    varMap = {}
                    varMap[':PandaID'] = oPandaID
                    varMap[':t1'] = 'input'
                    varMap[':t2'] = 'pseudo_input'
                    self.cur.execute(sqlGI+comment, varMap)
                    resGI = self.cur.fetchall()
                    # set corrupted
                    for datasetID, fileID in resGI:
                        varMap = {}
                        varMap[':PandaID'] = oPandaID
                        varMap[':row_ID'] = zipRow_ID
                        varMap[':jediTaskID'] = jediTaskID
                        varMap[':datasetID'] = datasetID
                        varMap[':fileID'] = fileID
                        varMap[':esDone'] = EventServiceUtils.ST_done
                        varMap[':esCorrupted'] = EventServiceUtils.ST_corrupted
                        self.cur.execute(sqlCE+comment, varMap)
                        nCor += self.cur.rowcount
            else:
                # check jobMetrics
                varMap = dict()
                varMap[':PandaID'] = pandaID
                varMap[':type1'] = 'input'
                varMap[':type2'] = 'pseudo_input'
                self.cur.execute(sqlJJ + comment, varMap)
                resJJ = self.cur.fetchall()
                # get jobMetrics
                for oPandaID, in resJJ:
                    varMap = dict()
                    varMap[':PandaID'] = oPandaID
                    self.cur.execute(sqlJM + comment, varMap)
                    resJM = self.cur.fetchone()
                    if resJM is not None:
                        jobMetrics, = resJM
                        if jobMetrics is not None and 'outputZipName={0}'.format(lfn) in jobMetrics:
                            # get dataset and file IDs
                            varMap = {}
                            varMap[':PandaID'] = oPandaID
                            varMap[':t1'] = 'input'
                            varMap[':t2'] = 'pseudo_input'
                            self.cur.execute(sqlGI+comment, varMap)
                            resGI = self.cur.fetchall()
                            # set corrupted
                            for datasetID, fileID in resGI:
                                varMap = {}
                                varMap[':PandaID'] = oPandaID
                                varMap[':jediTaskID'] = jediTaskID
                                varMap[':datasetID'] = datasetID
                                varMap[':fileID'] = fileID
                                varMap[':esDone'] = EventServiceUtils.ST_done
                                varMap[':esCorrupted'] = EventServiceUtils.ST_corrupted
                                self.cur.execute(sqlJE+comment, varMap)
                                nCor += self.cur.rowcount
                            break
            tmpLog.debug("{0} corrupted events in {1}".format(nCor, lfn))



    # check failure count due to corrupted files
    def checkFailureCountWithCorruptedFiles(self, jediTaskID, pandaID):
        comment = ' /* DBProxy.checkFailureCountWithCorruptedFiles */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " <jediTaskID={0} pandaID={1}>".format(jediTaskID, pandaID)
        tmpLog = LogWrapper(_logger,methodName)
        # sql to failure counts
        sqlBD  = "SELECT f2.lfn,COUNT(*) FROM ATLAS_PANDA.filesTable4 f1, ATLAS_PANDA.filesTable4 f2 "
        sqlBD += "WHERE f1.PandaID=:PandaID AND f1.type=:type AND f1.status=:status "
        sqlBD += "AND f2.lfn=f1.lfn AND f2.type=:type AND f2.status=:status AND f2.jediTaskID=:jediTaskID "
        sqlBD += "GROUP BY f2.lfn "
        varMap = {}
        varMap[':jediTaskID'] = jediTaskID
        varMap[':PandaID'] = pandaID
        varMap[':status']  = 'corrupted'
        varMap[':type']    = 'zipinput'
        self.cur.execute(sqlBD+comment, varMap)
        resBD = self.cur.fetchall()
        tooMany = False
        for lfn, nFailed in resBD:
            tmpLog.debug("{0} failures with {1}".format(nFailed, lfn))
            if nFailed >= 3:
                tooMany = True
        tmpLog.debug("too many failures : {0}".format(tooMany))
        return tooMany



    # check attemptNr for more retry
    def checkMoreRetryJEDI(self,job):
        comment = ' /* DBProxy.self.checkMoreRetryJEDI */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " <PandaID={0}>".format(job.PandaID)
        _logger.debug("{0} : start".format(methodName))
        # sql to get files
        sqlGF  = "SELECT datasetID,fileID,attemptNr FROM ATLAS_PANDA.filesTable4 "
        sqlGF += "WHERE PandaID=:PandaID AND type IN (:type1,:type2) "
        # sql to check file
        sqlFJ  = "SELECT attemptNr,maxAttempt,failedAttempt,maxFailure FROM {0}.JEDI_Dataset_Contents ".format(panda_config.schemaJEDI)
        sqlFJ += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
        sqlFJ += "AND attemptNr=:attemptNr AND keepTrack=:keepTrack AND PandaID=:PandaID "
        # get files
        varMap = {}
        varMap[':PandaID'] = job.PandaID
        varMap[':type1'] = 'input'
        varMap[':type2'] = 'pseudo_input'
        self.cur.execute(sqlGF+comment,varMap)
        resGF = self.cur.fetchall()
        for datasetID,fileID,attemptNr in resGF:
            # check JEDI contents
            varMap = {}
            varMap[':jediTaskID'] = job.jediTaskID
            varMap[':datasetID']  = datasetID
            varMap[':fileID']     = fileID
            varMap[':attemptNr']  = attemptNr
            varMap[':PandaID']    = job.PandaID
            varMap[':keepTrack']  = 1
            self.cur.execute(sqlFJ+comment,varMap)
            resFJ = self.cur.fetchone()
            if resFJ is None:
                continue
            attemptNr,maxAttempt,failedAttempt,maxFailure = resFJ
            if maxAttempt is None:
                continue
            if attemptNr+1 >= maxAttempt:
                # hit the limit
                _logger.debug("{0} : NG - fileID={1} no more attempt attemptNr({2})+1>=maxAttempt({3})".format(methodName,
                                                                                                               fileID,
                                                                                                               attemptNr,
                                                                                                               maxAttempt))
                return False
            if maxFailure is not None and failedAttempt is not None and failedAttempt+1 >= maxFailure:
                # hit the limit
                _logger.debug("{0} : NG - fileID={1} no more attempt failedAttempt({2})+1>=maxFailure({3})".format(methodName,
                                                                                                                   fileID,
                                                                                                                   failedAttempt,
                                                                                                                   maxFailure))
                return False
        _logger.debug("{0} : OK".format(methodName))
        return True



    # get the list of jobdefIDs for failed jobs in a task
    def getJobdefIDsForFailedJob(self,jediTaskID):
        comment = ' /* DBProxy.getJobdefIDsForFailedJob */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " <jediTaskID={0}>".format(jediTaskID)
        _logger.debug("{0} : start".format(methodName))
        try:
            # begin transaction
            self.conn.begin()
            # dql to get jobDefIDs
            sqlGF  = "SELECT distinct jobDefinitionID FROM ATLAS_PANDA.jobsActive4 "
            sqlGF += "WHERE jediTaskID=:jediTaskID AND jobStatus=:jobStatus "
            sqlGF += "AND attemptNr<maxAttempt "
            varMap = {}
            varMap[':jediTaskID'] = jediTaskID
            varMap[':jobStatus']  = 'failed'
            self.cur.execute(sqlGF+comment,varMap)
            resGF = self.cur.fetchall()
            retList = []
            for jobDefinitionID, in resGF:
                retList.append(jobDefinitionID)
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            _logger.debug("{0} : {1}".format(methodName,str(retList)))
            return retList
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            return []



    # change task attribute
    def changeTaskAttributePanda(self,jediTaskID,attrName,attrValue):
        comment = ' /* DBProxy.changeTaskAttributePanda */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " <jediTaskID={0}>".format(jediTaskID)
        _logger.debug("{0} name={1} value={2}".format(methodName,attrName,attrValue))
        try:
            # sql to update JEDI task table
            sqlT  = 'UPDATE {0}.JEDI_Tasks SET '.format(panda_config.schemaJEDI)
            sqlT += '{0}=:{0} WHERE jediTaskID=:jediTaskID '.format(attrName)
            # start transaction
            self.conn.begin()
            # select
            self.cur.arraysize = 10
            varMap = {}
            varMap[':jediTaskID']  = jediTaskID
            keyName = ':{0}'.format(attrName)
            varMap[keyName] = attrValue
            # update JEDI
            self.cur.execute(sqlT+comment, varMap)
            nRow = self.cur.rowcount
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            _logger.debug("{0} done with {1}".format(methodName,nRow))
            return nRow
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            return None



    # change split rule for task
    def changeTaskSplitRulePanda(self,jediTaskID,attrName,attrValue,useCommit=True,sendLog=True):
        comment = ' /* DBProxy.changeTaskSplitRulePanda */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " <jediTaskID={0}>".format(jediTaskID)
        tmpLog = LogWrapper(_logger,methodName,monToken="<jediTaskID={0}>".format(jediTaskID))
        tmpLog.debug("changing {0}={1}".format(attrName,attrValue))
        try:
            # sql to get split rule
            sqlS  = 'SELECT splitRule FROM {0}.JEDI_Tasks '.format(panda_config.schemaJEDI)
            sqlS += 'WHERE jediTaskID=:jediTaskID '
            # sql to update JEDI task table
            sqlT  = 'UPDATE {0}.JEDI_Tasks SET '.format(panda_config.schemaJEDI)
            sqlT += 'splitRule=:splitRule WHERE jediTaskID=:jediTaskID '
            # start transaction
            if useCommit:
                self.conn.begin()
            # select
            self.cur.arraysize = 10
            varMap = {}
            varMap[':jediTaskID']  = jediTaskID
            # get split rule
            self.cur.execute(sqlS+comment, varMap)
            resS = self.cur.fetchone()
            if resS is None:
                retVal = 0
            else:
                splitRule = resS[0]
                if splitRule is None:
                    items = []
                else:
                    items = splitRule.split(',')
                # remove old
                newItems = []
                for tmpItem in items:
                    if tmpItem.startswith('{0}='.format(attrName)):
                        continue
                    newItems.append(tmpItem)
                # add new
                if attrValue not in [None, '', 'None']:
                    newItems.append('{0}={1}'.format(attrName,attrValue))
                # update
                varMap = {}
                varMap[':jediTaskID']  = jediTaskID
                varMap[':splitRule']   = ','.join(newItems)
                self.cur.execute(sqlT+comment, varMap)
                retVal = 1
            # commit
            if useCommit:
                if not self._commit():
                    raise RuntimeError('Commit error')
            tmpLog.debug("done with {0}".format(retVal))
            if sendLog:
                tmpLog.sendMsg('set {0}={1} to splitRule'.format(attrName,attrValue),'jedi','pandasrv')
            return retVal
        except Exception:
            # roll back
            if useCommit:
                self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            return None



    # increase attempt number for unprocessed files
    def increaseAttemptNrPanda(self,jediTaskID,increasedNr):
        comment = ' /* DBProxy.increaseAttemptNrPanda */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " <jediTaskID={0}>".format(jediTaskID)
        tmpLog = LogWrapper(_logger,methodName,monToken="<jediTaskID={0}>".format(jediTaskID))
        tmpLog.debug("increasedNr={0}".format(increasedNr))
        try:
            # sql to check task status
            sqlT  = 'SELECT status,oldStatus FROM {0}.JEDI_Tasks '.format(panda_config.schemaJEDI)
            sqlT += 'WHERE jediTaskID=:jediTaskID FOR UPDATE '
            # start transaction
            self.conn.begin()
            # select
            self.cur.arraysize = 10
            varMap = {}
            varMap[':jediTaskID']  = jediTaskID
            # get task status
            self.cur.execute(sqlT+comment, varMap)
            resT = self.cur.fetchone()
            if resT is None:
                tmpMsg = "jediTaskID={0} not found".format(jediTaskID)
                tmpLog.debug(tmpMsg)
                retVal = 1,tmpMsg
            else:
                taskStatus,oldStatus = resT
                # check task status
                okStatusList = ['running','scouting','ready']
                if taskStatus not in okStatusList and oldStatus not in okStatusList:
                    tmpMsg = "command rejected since status={0} or oldStatus={1} not in {2}".format(taskStatus,
                                                                                                    oldStatus,
                                                                                                    str(okStatusList))
                    tmpLog.debug(tmpMsg)
                    retVal = 2,tmpMsg
                else:
                    # sql to get datasetIDs for master
                    sqlM  = 'SELECT datasetID FROM {0}.JEDI_Datasets '.format(panda_config.schemaJEDI)
                    sqlM += 'WHERE jediTaskID=:jediTaskID AND type IN (:type1,:type2) '
                    # sql to increase attempt numbers
                    sqlAB  = "UPDATE {0}.JEDI_Dataset_Contents ".format(panda_config.schemaJEDI)
                    sqlAB += "SET maxAttempt=CASE WHEN maxAttempt > attemptNr THEN maxAttempt+:increasedNr ELSE attemptNr+:increasedNr END "
                    sqlAB += ",proc_status=CASE WHEN maxAttempt > attemptNr AND maxFailure > failedAttempt THEN proc_status ELSE :proc_status END "
                    sqlAB += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND status=:status AND keepTrack=:keepTrack "
                    # sql to increase attempt numbers and failure counts
                    sqlAF  = "UPDATE {0}.JEDI_Dataset_Contents ".format(panda_config.schemaJEDI)
                    sqlAF += "SET maxAttempt=CASE WHEN maxAttempt > attemptNr THEN maxAttempt+:increasedNr ELSE attemptNr+:increasedNr END "
                    sqlAF += ",maxFailure=maxFailure+:increasedNr "
                    sqlAF += ",proc_status=CASE WHEN maxAttempt > attemptNr AND maxFailure > failedAttempt THEN proc_status ELSE :proc_status END "
                    sqlAF += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND status=:status AND keepTrack=:keepTrack "
                    # sql to update datasets
                    sqlD  = "UPDATE {0}.JEDI_Datasets ".format(panda_config.schemaJEDI)
                    sqlD += "SET nFilesUsed=nFilesUsed-:nFilesReset,nFilesFailed=nFilesFailed-:nFilesReset "
                    sqlD += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID "
                    # get datasetIDs for master
                    varMap = {}
                    varMap[':jediTaskID'] = jediTaskID
                    varMap[':type1'] = 'input'
                    varMap[':type2'] = 'pseudo_input'
                    self.cur.execute(sqlM+comment, varMap)
                    resM = self.cur.fetchall()
                    total_nFilesIncreased = 0
                    total_nFilesReset = 0
                    for datasetID, in resM:
                        # increase attempt numbers
                        varMap = {}
                        varMap[':jediTaskID'] = jediTaskID
                        varMap[':datasetID'] = datasetID
                        varMap[':status'] = 'ready'
                        varMap[':proc_status'] = 'ready'
                        varMap[':keepTrack']  = 1
                        varMap[':increasedNr'] = increasedNr
                        nFilesIncreased = 0
                        nFilesReset = 0
                        # still active and maxFailure is undefined
                        sqlA = sqlAB + "AND maxAttempt>attemptNr AND maxFailure IS NULL "
                        self.cur.execute(sqlA+comment, varMap)
                        nRow = self.cur.rowcount
                        nFilesIncreased += nRow
                        # still active and maxFailure is defined
                        sqlA = sqlAF + "AND maxAttempt>attemptNr AND (maxFailure IS NOT NULL AND maxFailure>failedAttempt) "
                        self.cur.execute(sqlA+comment, varMap)
                        nRow = self.cur.rowcount
                        nFilesIncreased += nRow
                        # already done and maxFailure is undefined
                        sqlA = sqlAB + "AND maxAttempt<=attemptNr AND maxFailure IS NULL "
                        self.cur.execute(sqlA+comment, varMap)
                        nRow = self.cur.rowcount
                        nFilesReset += nRow
                        nFilesIncreased += nRow
                        # already done and maxFailure is defined
                        sqlA = sqlAF + "AND (maxAttempt<=attemptNr OR (maxFailure IS NOT NULL AND maxFailure=failedAttempt)) "
                        self.cur.execute(sqlA+comment, varMap)
                        nRow = self.cur.rowcount
                        nFilesReset += nRow
                        nFilesIncreased += nRow
                        # update dataset
                        if nFilesReset > 0:
                            varMap = {}
                            varMap[':jediTaskID'] = jediTaskID
                            varMap[':datasetID'] = datasetID
                            varMap[':nFilesReset'] = nFilesReset
                            tmpLog.debug(sqlD+comment+str(varMap))
                            self.cur.execute(sqlD+comment, varMap)
                        total_nFilesIncreased += nFilesIncreased
                        total_nFilesReset += nFilesReset
                    tmpMsg = "increased attemptNr for {0} inputs ({1} reactivated)".format(total_nFilesIncreased,
                                                                                           total_nFilesReset)
                    tmpLog.debug(tmpMsg)
                    tmpLog.sendMsg(tmpMsg,'jedi','pandasrv')
                    retVal = 0,tmpMsg
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            tmpLog.debug("done")
            return retVal
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(tmpLog,methodName)
            return None,"DB error"



    # get jediTaskID from taskName
    def getTaskIDwithTaskNameJEDI(self,userName,taskName):
        comment = ' /* DBProxy.getTaskIDwithTaskNameJEDI */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " <userName={0} taskName={1}>".format(userName,taskName)
        _logger.debug("{0} : start".format(methodName))
        try:
            # begin transaction
            self.conn.begin()
            # sql to get jediTaskID
            sqlGF  = "SELECT MAX(jediTaskID) FROM {0}.JEDI_Tasks ".format(panda_config.schemaJEDI)
            sqlGF += "WHERE userName=:userName AND taskName=:taskName "
            varMap = {}
            varMap[':userName'] = userName
            varMap[':taskName'] = taskName
            self.cur.execute(sqlGF+comment,varMap)
            resFJ = self.cur.fetchone()
            if resFJ is not None:
                jediTaskID, = resFJ
            else:
                jediTaskID = None
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            _logger.debug("{0} : jediTaskID={1}".format(methodName,jediTaskID))
            return jediTaskID
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            return None


    # get prodSourceLabel from TaskID
    def getProdSourceLabelwithTaskID(self, jediTaskID):
        comment = ' /* DBProxy.getProdSourceLabelwithTaskID */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " <jediTaskID={0}>".format(jediTaskID)
        _logger.debug("{0} : start".format(methodName))
        try:
            # begin transaction
            self.conn.begin()
            # sql to get jediTaskID
            sqlGF  = "SELECT prodSourceLabel, tasktype FROM {0}.JEDI_Tasks ".format(panda_config.schemaJEDI)
            sqlGF += "WHERE jediTaskID=:jediTaskID "
            varMap = {}
            varMap[':jediTaskID'] = jediTaskID
            self.cur.execute(sqlGF+comment,varMap)
            resFJ = self.cur.fetchone()
            if resFJ is not None:
                prodSourceLabel, task_type = resFJ
            else:
                prodSourceLabel, task_type = None, None

            if task_type:
                job_label = JobUtils.translate_tasktype_to_jobtype(task_type)
            else:
                job_label = None

            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            _logger.debug("{0} : jediTaskID={1} prodSourceLabel={2} job_label={3}".format(methodName, jediTaskID,
                                                                                          prodSourceLabel, job_label))
            return prodSourceLabel, job_label
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            return None, None



    # update error dialog for a jediTaskID
    def updateTaskErrorDialogJEDI(self,jediTaskID,msg):
        comment = ' /* DBProxy.updateTaskErrorDialogJEDI */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " <jediTaskID={0}>".format(jediTaskID)
        _logger.debug("{0} : start".format(methodName))
        try:
            # begin transaction
            self.conn.begin()
            # get existing dialog
            sqlGF  = "SELECT errorDialog FROM {0}.JEDI_Tasks ".format(panda_config.schemaJEDI)
            sqlGF += "WHERE jediTaskID=:jediTaskID "
            varMap = {}
            varMap[':jediTaskID'] = jediTaskID
            self.cur.execute(sqlGF+comment,varMap)
            resFJ = self.cur.fetchone()
            if resFJ is not None:
                # update existing dialog
                errorDialog, = resFJ
                errorDialog = msg
                varMap = {}
                varMap[':jediTaskID'] = jediTaskID
                varMap[':errorDialog'] = errorDialog
                sqlUE  = "UPDATE {0}.JEDI_Tasks SET errorDialog=:errorDialog,modificationTime=CURRENT_DATE ".format(panda_config.schemaJEDI)
                sqlUE += "WHERE jediTaskID=:jediTaskID "
                self.cur.execute(sqlUE+comment,varMap)
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            _logger.debug("{0} : done".format(methodName))
            return True
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            return False



    # update modificationtime for a jediTaskID to trigger subsequent process
    def updateTaskModTimeJEDI(self,jediTaskID,newStatus):
        comment = ' /* DBProxy.updateTaskErrorDialogJEDI */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " <jediTaskID={0}>".format(jediTaskID)
        _logger.debug("{0} : start".format(methodName))
        try:
            # begin transaction
            self.conn.begin()
            # update mod time
            varMap = {}
            varMap[':jediTaskID'] = jediTaskID
            if newStatus is not None:
                varMap[':newStatus'] = newStatus
            sqlUE  = "UPDATE {0}.JEDI_Tasks SET ".format(panda_config.schemaJEDI)
            sqlUE += "modificationTime=CURRENT_DATE-1,"
            if newStatus is not None:
                sqlUE += "status=:newStatus,oldStatus=NULL,"
            sqlUE = sqlUE[:-1]
            sqlUE += " WHERE jediTaskID=:jediTaskID "
            self.cur.execute(sqlUE+comment,varMap)
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            _logger.debug("{0} : done".format(methodName))
            return True
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            return False



    # check input file status
    def checkInputFileStatusInJEDI(self, jobSpec, useCommit=True, withLock=False):
        comment = ' /* DBProxy.checkInputFileStatusInJEDI */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " <PandaID={0}>".format(jobSpec.PandaID)
        tmpLog = LogWrapper(_logger,methodName,monToken="<PandaID={0}>".format(jobSpec.PandaID))
        tmpLog.debug("start")
        try:
            # only JEDI
            if jobSpec.lockedby != 'jedi':
                return True
            # sql to check file status
            sqlFileStat  = "SELECT status,attemptNr,keepTrack,is_waiting FROM ATLAS_PANDA.JEDI_Dataset_Contents "
            sqlFileStat += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
            if withLock:
                sqlFileStat += "FOR UPDATE NOWAIT "
            # begin transaction
            if useCommit:
                self.conn.begin()
            # get dataset
            sqlPD = "SELECT datasetID FROM ATLAS_PANDA.JEDI_Datasets "
            sqlPD += "WHERE jediTaskID=:jediTaskID AND type IN (:type1,:type2) AND masterID IS NULL "
            varMap = {}
            varMap[':jediTaskID']  = jobSpec.jediTaskID
            varMap[':type1'] = 'input'
            varMap[':type2'] = 'pseudo_input'
            self.cur.execute(sqlPD+comment, varMap)
            resPD = self.cur.fetchone()
            if resPD is not None:
                datasetID, = resPD
            else:
                datasetID = None
            # loop over all input files
            allOK = True
            for fileSpec in jobSpec.Files:
                if datasetID is None:
                    continue
                # only input file
                if fileSpec.datasetID != datasetID:
                    continue
                # skip if not normal JEDI files
                if fileSpec.fileID == 'NULL':
                    continue
                varMap = {}
                varMap[':jediTaskID'] = fileSpec.jediTaskID
                varMap[':datasetID']  = fileSpec.datasetID
                varMap[':fileID']     = fileSpec.fileID
                self.cur.execute(sqlFileStat+comment,varMap)
                resFileStat = self.cur.fetchone()
                if resFileStat is None:
                    tmpLog.debug("jediTaskID={0} datasetID={1} fileID={2} is not found".format(fileSpec.jediTaskID,
                                                                                               fileSpec.datasetID,
                                                                                               fileSpec.fileID))
                    allOK = False
                    break
                else:
                    fileStatus,attemptNr,keepTrack,is_waiting = resFileStat
                    if attemptNr is None:
                        continue
                    if keepTrack != 1:
                        continue
                    if attemptNr != fileSpec.attemptNr:
                        tmpLog.debug("jediTaskID={0} datasetID={1} fileID={2} attemptNr={3} is inconsitent with attemptNr={4} in JEDI".format(fileSpec.jediTaskID,
                                                                                                                                              fileSpec.datasetID,
                                                                                                                                              fileSpec.fileID,
                                                                                                                                              fileSpec.attemptNr,
                                                                                                                                              attemptNr))
                        allOK = False
                        break
                    if fileStatus in ['finished'] or \
                            (fileStatus not in ['running'] and jobSpec.computingSite != EventServiceUtils.siteIdForWaitingCoJumboJobs and is_waiting is None):
                        tmpLog.debug("jediTaskID={0} datasetID={1} fileID={2} attemptNr={3} is in wrong status ({4}) in JEDI".format(fileSpec.jediTaskID,
                                                                                                                                     fileSpec.datasetID,
                                                                                                                                     fileSpec.fileID,
                                                                                                                                     fileSpec.attemptNr,
                                                                                                                                     fileStatus))
                        allOK = False
                        break
            # commit
            if useCommit:
                if not self._commit():
                    raise RuntimeError('Commit error')
            tmpLog.debug("done with {0}".format(allOK))
            return allOK
        except Exception:
            if useCommit:
                # roll back
                self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            return None


    # increase memory limit
    def increaseRamLimitJEDI(self, jediTaskID, jobRamCount):
        comment = ' /* DBProxy.increaseRamLimitJEDI */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " <jediTaskID={0}>".format(jediTaskID)
        _logger.debug("{0} : start".format(methodName))
        try:
            # RAM limit
            limitList = [1000,2000,3000,4000,6000,8000]
            # begin transaction
            self.conn.begin()
            # get current limit
            varMap = {}
            varMap[':jediTaskID'] = jediTaskID
            sqlUE  = "SELECT ramCount FROM {0}.JEDI_Tasks ".format(panda_config.schemaJEDI)
            sqlUE += "WHERE jediTaskID=:jediTaskID "
            self.cur.execute(sqlUE+comment,varMap)
            taskRamCount, = self.cur.fetchone()
            _logger.debug("{0} : RAM limit task={1} job={2}".format(methodName,taskRamCount,jobRamCount))

            increased = False

            # skip if already increased or largest limit
            if taskRamCount > jobRamCount:
                dbgStr = "no change since task RAM limit ({0}) is larger than job limit ({1})".format(taskRamCount,
                                                                                                      jobRamCount)
                _logger.debug("{0} : {1}".format(methodName,dbgStr))
            elif taskRamCount >= limitList[-1]:
                dbgStr  = "no change "
                dbgStr += "since task RAM limit ({0}) is larger than or equal to the highest limit ({1})".format(taskRamCount,
                                                                                                                 limitList[-1])
                _logger.debug("{0} : {1}".format(methodName,dbgStr))
            else:
                increased = True
                limit = max(taskRamCount, jobRamCount)
                for nextLimit in limitList:
                    if limit < nextLimit:
                        break
                # update RAM limit
                varMap = {}
                varMap[':jediTaskID'] = jediTaskID
                varMap[':ramCount'] = nextLimit
                sqlRL  = "UPDATE {0}.JEDI_Tasks ".format(panda_config.schemaJEDI)
                sqlRL += "SET ramCount=:ramCount "
                sqlRL += "WHERE jediTaskID=:jediTaskID "
                self.cur.execute(sqlRL+comment,varMap)
                _logger.debug("{0} : increased RAM limit to {1} from {2}".format(methodName,nextLimit,taskRamCount))
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')

            # reset the tasks resource type, since it could have jumped to HIMEM
            if increased:
                try:
                    self.reset_resource_type(jediTaskID)
                except Exception:
                    _logger.error("reset_resource_type excepted with {0}".format(traceback.format_exc()))

            _logger.debug("{0} : done".format(methodName))
            return True
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            return False


    # increase memory limit
    def increaseRamLimitJobJEDI(self, job, jobRamCount, jediTaskID):
        """Note that this function only increases the min RAM count for the job,
        not for the entire task (for the latter use increaseRamLimitJEDI)
        """
        comment = ' /* DBProxy.increaseRamLimitJobJEDI */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " <PanDAID={0}>".format(job.PandaID)
        _logger.debug("{0} : start".format(methodName))

        # RAM limit
        limitList = [1000, 2000, 3000, 4000, 6000, 8000]
        # Files defined as input types
        input_types = ('input', 'pseudo_input', 'pp_input', 'trn_log', 'trn_output')

        try:
            # If no task associated to job don't take any action
            if job.jediTaskID in [None, 0, 'NULL']:
                _logger.debug("No task({0}) associated to job({1}). Skipping increase of RAM limit"
                              .format(job.jediTaskID, job.PandaID))
            else:
                # get current task Ram info
                varMap = {}
                varMap[':jediTaskID'] = jediTaskID
                sqlUE  = "SELECT ramCount, ramUnit, baseRamCount FROM {0}.JEDI_Tasks ".format(panda_config.schemaJEDI)
                sqlUE += "WHERE jediTaskID=:jediTaskID "
                self.cur.execute(sqlUE+comment, varMap)
                taskRamCount, taskRamUnit, taskBaseRamCount = self.cur.fetchone()

                if taskBaseRamCount in [0, None, 'NULL']:
                    taskBaseRamCount = 0

                coreCount = job.coreCount

                if coreCount in [0, None, 'NULL']:
                    coreCount = 1

                _logger.debug("{0} : RAM limit task={1}{2} cores={3} baseRamCount={4} job={5}{6} jobPSS={7}kB"
                              .format(methodName, taskRamCount, taskRamUnit, coreCount, taskBaseRamCount,
                                      jobRamCount, job.minRamUnit, job.maxPSS))

                # If more than x% of the task's jobs needed a memory increase, increase the task's memory instead
                varMap = {}
                varMap[':jediTaskID'] = jediTaskID
                i = 0
                for input_type in input_types:
                    varMap[':type{0}'.format(i)] = input_type
                    i += 1
                input_type_bindings = ','.join(':type{0}'.format(i) for i in range(len(input_types)))

                sqlMS  = """
                         SELECT ramCount, count(*)
                         FROM {0}.JEDI_Datasets tabD,{0}.JEDI_Dataset_Contents tabC
                         WHERE tabD.jediTaskID=tabC.jediTaskID
                         AND tabD.datasetID=tabC.datasetID
                         AND tabD.jediTaskID=:jediTaskID
                         AND tabD.type IN ({1})
                         AND tabD.masterID IS NULL
                         GROUP BY ramCount
                         """.format(panda_config.schemaJEDI, input_type_bindings)

                self.cur.execute(sqlMS+comment,varMap)
                memory_stats = self.cur.fetchall()
                total = sum([entry[1] for entry in memory_stats])
                above_task = sum(tuple[1] for tuple in filter(lambda entry: entry[0] > taskRamCount, memory_stats))
                max_task = max([entry[0] for entry in memory_stats])
                _logger.debug("{0} : #increased_files: {1}; #total_files: {2}".format(methodName, above_task, total))

                # normalize the job ramcounts by base ram count and number of cores
                try:
                    normalizedJobRamCount = (jobRamCount - taskBaseRamCount) * 1.0
                    if taskRamUnit in ['MBPerCore', 'MBPerCoreFixed'] and job.minRamUnit in ('MB', None, 'NULL'):
                        normalizedJobRamCount  = normalizedJobRamCount / coreCount
                except TypeError:
                    normalizedJobRamCount = 0

                try:
                    normalizedMaxPSS = (job.maxPSS - taskBaseRamCount) / 1024.0
                    if taskRamUnit in ['MBPerCore', 'MBPerCoreFixed']:
                        normalizedMaxPSS  = normalizedMaxPSS / coreCount
                except TypeError:
                    normalizedMaxPSS = 0

                # increase task limit in case >30% of the jobs were increased and the task is not fixed
                if taskRamUnit != 'MBPerCoreFixed' and (1.0 * above_task) / total > 0.3:
                    if normalizedMaxPSS:
                        minimumRam = normalizedMaxPSS
                    if normalizedJobRamCount and normalizedJobRamCount > minimumRam:
                        minimumRam = normalizedJobRamCount
                    if max_task > minimumRam:
                        minimumRam = max_task - 1  # otherwise we go over the max_task step
                    if minimumRam:
                        _logger.debug("{0} : calling increaseRamLimitJEDI with minimumRam {1}"
                                      .format(methodName, minimumRam))
                        return self.increaseRamLimitJEDI(jediTaskID, minimumRam)


                #Ops could have increased task RamCount through direct DB access. In this case don't do anything
                if (taskRamCount > normalizedJobRamCount) and (normalizedMaxPSS not in [None, 0, 'NULL']) and (taskRamCount > normalizedMaxPSS):
                    _logger.debug("{0} : task ramcount has already been increased and is higher than maxPSS. Skipping".
                                  format(methodName))
                    return True

                # skip if already at largest limit
                if normalizedJobRamCount >= limitList[-1]:
                    dbgStr  = "no change "
                    dbgStr += "since job RAM limit ({0}) is larger than or equal to the highest limit ({1})".\
                        format(normalizedJobRamCount, limitList[-1])
                    _logger.debug("{0} : {1}".format(methodName, dbgStr))
                else:
                    #If maxPSS is present, then jump all the levels until the one above
                    if normalizedMaxPSS > normalizedJobRamCount:
                        minimumRam = normalizedMaxPSS
                    else:
                        minimumRam = normalizedJobRamCount

                    for nextLimit in limitList:
                        if minimumRam < nextLimit:
                            break

                    # update RAM limit
                    varMap = {}
                    varMap[':jediTaskID'] = job.jediTaskID
                    varMap[':ramCount'] = nextLimit
                    input_files = filter(lambda pandafile: pandafile.type in input_types, job.Files)
                    input_tuples = [(input_file.datasetID, input_file.fileID, input_file.attemptNr) for input_file in input_files]

                    for entry in input_tuples:
                        datasetID, fileId, attemptNr = entry
                        varMap[':datasetID'] = datasetID
                        varMap[':fileID'] = fileId

                        sqlRL  = "UPDATE {0}.JEDI_Dataset_Contents ".format(panda_config.schemaJEDI)
                        sqlRL += "SET ramCount=:ramCount "
                        sqlRL += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                        sqlRL += "AND ramCount<:ramCount "

                        self.cur.execute(sqlRL+comment,varMap)
                        _logger.debug("{0} : increased RAM limit to {1} from {2} for PandaID {3} fileID {4} attemptNr {5} jediTaskID {6} datasetID {7}"
                            .format(methodName, nextLimit, normalizedJobRamCount, job.PandaID, fileId, attemptNr, job.jediTaskID, datasetID))
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')

            _logger.debug("{0} : done".format(methodName))
            return True
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger, methodName)
            return False


    # reset files in JEDI
    def resetFileStatusInJEDI(self,dn,prodManager,datasetName,lostFiles,recoverParent,simul):
        comment = ' /* DBProxy.resetFileStatusInJEDI */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " <datasetName={0}>".format(datasetName)
        tmpLog = LogWrapper(_logger,methodName)
        tmpLog.debug("start")
        try:
            # list of lost input files
            lostInputFiles = {}
            # get compact DN
            compactDN = self.cleanUserID(dn)
            if compactDN in ['','NULL',None]:
                compactDN = dn
            tmpLog.debug("userName={0}".format(compactDN))
            toSkip = False
            # begin transaction
            self.conn.begin()
            # get jediTaskID
            varMap = {}
            varMap[':type1'] = 'log'
            varMap[':type2'] = 'output'
            varMap[':name1'] = datasetName
            varMap[':name2'] = datasetName.split(':')[-1]
            sqlGI  = 'SELECT jediTaskID,datasetID FROM {0}.JEDI_Datasets '.format(panda_config.schemaJEDI)
            sqlGI += 'WHERE type IN (:type1,:type2) AND datasetName IN (:name1,:name2) '
            self.cur.execute(sqlGI+comment,varMap)
            resGI = self.cur.fetchall()
            # use the largest datasetID since broken tasks might have been retried
            jediTaskID = None
            datasetID = None
            for tmpJediTaskID,tmpDatasetID in resGI:
                if jediTaskID is None or jediTaskID < tmpJediTaskID:
                    jediTaskID = tmpJediTaskID
                    datasetID = tmpDatasetID
                elif datasetID < tmpDatasetID:
                    datasetID =tmpDatasetID
            if jediTaskID is None:
                tmpLog.debug("jediTaskID not found")
                toSkip = True
            if not toSkip:
                # get task status and owner
                tmpLog.debug("jediTaskID={0} datasetID={1}".format(jediTaskID,datasetID))
                varMap = {}
                varMap[':jediTaskID'] = jediTaskID
                sqlOW  = 'SELECT status,userName,useJumbo FROM {0}.JEDI_Tasks '.format(panda_config.schemaJEDI)
                sqlOW += 'WHERE jediTaskID=:jediTaskID '
                self.cur.execute(sqlOW+comment,varMap)
                resOW = self.cur.fetchone()
                taskStatus,ownerName,useJumbo = resOW
                # check ownership
                if not prodManager and ownerName != compactDN:
                    tmpLog.debug("not the owner = {0}".format(ownerName))
                    toSkip = True
            if not toSkip:
                # get affected PandaIDs
                sqlLP  = 'SELECT pandaID FROM {0}.JEDI_Dataset_Contents '.format(panda_config.schemaJEDI)
                sqlLP += 'WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND lfn=:lfn '
                # sql to update file status
                sqlUFO  = 'UPDATE {0}.JEDI_Dataset_Contents '.format(panda_config.schemaJEDI)
                sqlUFO += 'SET status=:newStatus '
                sqlUFO += 'WHERE jediTaskID=:jediTaskID AND type=:type AND status=:oldStatus AND PandaID=:PandaID '
                # sql to cancel events
                sqlCE  = "UPDATE /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
                sqlCE += "{0}.JEDI_Events tab ".format(panda_config.schemaJEDI)
                sqlCE += "SET status=:status "
                sqlCE += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                sqlCE += "AND status IN (:esFinished,:esDone,:esMerged) "
                # get affected PandaIDs
                lostPandaIDs = set([])
                nDiff = 0
                for lostFile in lostFiles:
                    varMap = {}
                    varMap[':jediTaskID'] = jediTaskID
                    varMap[':datasetID'] = datasetID
                    varMap[':lfn'] = lostFile
                    self.cur.execute(sqlLP+comment,varMap)
                    resLP = self.cur.fetchone()
                    if resLP is not None:
                        pandaID, = resLP
                        lostPandaIDs.add(pandaID)
                        # update the file and coproduced files to lost
                        varMap = {}
                        varMap[':jediTaskID'] = jediTaskID
                        varMap[':PandaID'] = pandaID
                        varMap[':type'] = 'output'
                        varMap[':newStatus'] = 'lost'
                        varMap[':oldStatus'] = 'finished'
                        if not simul:
                            self.cur.execute(sqlUFO+comment,varMap)
                            nRow = self.cur.rowcount
                            if nRow > 0:
                                nDiff += 1
                        else:
                            tmpLog.debug(sqlUFO+comment+str(varMap))
                            nDiff += 1
                # update output dataset statistics
                sqlUDO  = 'UPDATE {0}.JEDI_Datasets '.format(panda_config.schemaJEDI)
                sqlUDO += 'SET nFilesFinished=nFilesFinished-:nDiff '
                sqlUDO += 'WHERE jediTaskID=:jediTaskID AND type=:type '
                varMap = {}
                varMap[':jediTaskID'] = jediTaskID
                varMap[':type'] = 'output'
                varMap[':nDiff'] = nDiff
                tmpLog.debug(sqlUDO+comment+str(varMap))
                if not simul:
                    self.cur.execute(sqlUDO+comment,varMap)
                # get nEvents
                sqlGNE  = 'SELECT SUM(c.nEvents),c.datasetID '
                sqlGNE += 'FROM {0}.JEDI_Datasets d,{0}.JEDI_Dataset_Contents c '.format(panda_config.schemaJEDI)
                sqlGNE += 'WHERE c.jediTaskID=d.jediTaskID AND c.datasetID=d.datasetID '
                sqlGNE += 'AND d.jediTaskID=:jediTaskID AND d.type=:type AND c.status=:status '
                sqlGNE += 'GROUP BY c.datasetID '
                varMap = {}
                varMap[':jediTaskID'] = jediTaskID
                varMap[':type'] = 'output'
                varMap[':status'] = 'finished'
                self.cur.execute(sqlGNE+comment,varMap)
                resGNE = self.cur.fetchall()
                # update nEvents
                sqlUNE  = 'UPDATE {0}.JEDI_Datasets '.format(panda_config.schemaJEDI)
                sqlUNE += 'SET nEvents=:nEvents '
                sqlUNE += 'WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID '
                for tmpCount,tmpDatasetID in resGNE:
                    varMap = {}
                    varMap[':jediTaskID'] = jediTaskID
                    varMap[':datasetID'] = tmpDatasetID
                    varMap[':nEvents'] = tmpCount
                    if not simul:
                        self.cur.execute(sqlUNE+comment,varMap)
                        tmpLog.debug(sqlUNE+comment+str(varMap))
                # get input datasets
                sqlID  = 'SELECT datasetID,datasetName,masterID FROM {0}.JEDI_Datasets '.format(panda_config.schemaJEDI)
                sqlID += 'WHERE jediTaskID=:jediTaskID AND type=:type '
                varMap = {}
                varMap[':jediTaskID'] = jediTaskID
                varMap[':type'] = 'input'
                self.cur.execute(sqlID+comment,varMap)
                resID = self.cur.fetchall()
                inputDatasets = {}
                masterID = None
                for tmpDatasetID,tmpDatasetName,tmpMasterID in resID:
                    inputDatasets[tmpDatasetID] = tmpDatasetName
                    if tmpMasterID is None:
                        masterID = tmpDatasetID
                # sql to get affected inputs
                if useJumbo is None:
                    sqlAI  = 'SELECT fileID,datasetID,lfn,outPandaID FROM {0}.JEDI_Dataset_Contents '.format(panda_config.schemaJEDI)
                    sqlAI += 'WHERE jediTaskID=:jediTaskID AND type IN (:type1,:type2,:type3) AND PandaID=:PandaID '
                else:
                    sqlAI = "SELECT fileID,datasetID,lfn,NULL FROM {0}.filesTable4 ".format(panda_config.schemaPANDA)
                    sqlAI += "WHERE PandaID=:PandaID AND type IN (:type1,:type2) "
                    sqlAI += "UNION "
                    sqlAI = "SELECT fileID,datasetID,lfn,NULL FROM {0}.filesTable4 ".format(panda_config.schemaPANDA)
                    sqlAI += "WHERE PandaID=:PandaID AND type IN (:type1,:type2) AND modificationTime>CURRENT_TIMESTAMP-365 "
                # sql to update input file status
                sqlUFI  = 'UPDATE {0}.JEDI_Dataset_Contents '.format(panda_config.schemaJEDI)
                sqlUFI += 'SET status=:newStatus,attemptNr=attemptNr+1 '
                sqlUFI += 'WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID AND status=:oldStatus '
                # get affected inputs
                datasetCountMap = {}
                for lostPandaID in lostPandaIDs:
                    varMap = {}
                    if useJumbo is None:
                        varMap[':jediTaskID'] = jediTaskID
                        varMap[':PandaID'] = lostPandaID
                        varMap[':type1'] = 'input'
                        varMap[':type2'] = 'pseudo_input'
                        varMap[':type3'] = 'output'
                    else:
                        varMap[':PandaID'] = lostPandaID
                        varMap[':type1'] = 'input'
                        varMap[':type2'] = 'pseudo_input'
                    self.cur.execute(sqlAI+comment,varMap)
                    resAI = self.cur.fetchall()
                    newResAI = []
                    for tmpItem in resAI:
                        tmpFileID,tmpDatasetID,tmpLFN,tmpOutPandaID = tmpItem
                        # skip output file
                        if lostPandaID == tmpOutPandaID:
                            continue
                        # input for merged files
                        if tmpOutPandaID is not None:
                            varMap = {}
                            varMap[':jediTaskID'] = jediTaskID
                            varMap[':PandaID'] = tmpOutPandaID
                            varMap[':type1'] = 'input'
                            varMap[':type2'] = 'pseudo_input'
                            varMap[':type3'] = 'dummy'
                            self.cur.execute(sqlAI+comment,varMap)
                            resAI2 = self.cur.fetchall()
                            for tmpItem in resAI2:
                                newResAI.append(tmpItem)
                        else:
                            newResAI.append(tmpItem)
                    for tmpFileID,tmpDatasetID,tmpLFN,tmpOutPandaID in newResAI:
                        # collect if dataset was already deleted
                        is_lost = False
                        if recoverParent and tmpDatasetID == masterID:
                                lostInputFiles.setdefault(inputDatasets[tmpDatasetID], [])
                                lostInputFiles[inputDatasets[tmpDatasetID]].append(tmpLFN)
                                is_lost = True
                        # reset file status
                        if tmpDatasetID not in datasetCountMap:
                            datasetCountMap[tmpDatasetID] = 0
                        varMap = {}
                        varMap[':jediTaskID'] = jediTaskID
                        varMap[':datasetID'] = tmpDatasetID
                        varMap[':fileID'] = tmpFileID
                        if is_lost:
                            varMap[':newStatus'] = 'lost'
                        else:
                            varMap[':newStatus'] = 'ready'
                        varMap[':oldStatus'] = 'finished'
                        if not simul:
                            self.cur.execute(sqlUFI+comment,varMap)
                            nRow = self.cur.rowcount
                        else:
                            tmpLog.debug(sqlUFI+comment+str(varMap))
                            nRow = 1
                        if nRow > 0:
                            datasetCountMap[tmpDatasetID] += 1
                            if useJumbo is not None:
                                # cancel events
                                varMap = {}
                                varMap[':jediTaskID'] = jediTaskID
                                varMap[':datasetID'] = tmpDatasetID
                                varMap[':fileID'] = tmpFileID
                                varMap[':status'] = EventServiceUtils.ST_cancelled
                                varMap[':esFinished'] = EventServiceUtils.ST_finished
                                varMap[':esDone'] = EventServiceUtils.ST_done
                                varMap[':esMerged'] = EventServiceUtils.ST_merged
                                if not simul:
                                    self.cur.execute(sqlCE+comment, varMap)
                                else:
                                    tmpLog.debug(sqlCE+comment+str(varMap))
                # update dataset statistics
                sqlUDI  = 'UPDATE {0}.JEDI_Datasets '.format(panda_config.schemaJEDI)
                sqlUDI += 'SET nFilesUsed=nFilesUsed-:nDiff,nFilesFinished=nFilesFinished-:nDiff,'
                sqlUDI += 'nEventsUsed=(SELECT SUM(CASE WHEN startEvent IS NULL THEN nEvents ELSE endEvent-startEvent+1 END) '
                sqlUDI += 'FROM {0}.JEDI_Dataset_Contents '.format(panda_config.schemaJEDI)
                sqlUDI += 'WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND status=:status) '
                sqlUDI += 'WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID '
                for tmpDatasetID in datasetCountMap:
                    nDiff = datasetCountMap[tmpDatasetID]
                    varMap = {}
                    varMap[':jediTaskID'] = jediTaskID
                    varMap[':datasetID'] = tmpDatasetID
                    varMap[':nDiff'] = nDiff
                    varMap[':status'] = 'finished'
                    tmpLog.debug(sqlUDI+comment+str(varMap))
                    if not simul:
                        self.cur.execute(sqlUDI+comment,varMap)
                # update task status
                if taskStatus == 'done':
                    sqlUT  = 'UPDATE {0}.JEDI_Tasks SET status=:newStatus WHERE jediTaskID=:jediTaskID '.format(panda_config.schemaJEDI)
                    varMap = {}
                    varMap[':jediTaskID'] = jediTaskID
                    varMap[':newStatus'] = 'finished'
                    if not simul:
                        self.cur.execute(sqlUT+comment,varMap)
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            tmpLog.debug("done")
            return True, jediTaskID, lostInputFiles
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            return False, None, None



    # get input datasets for output dataset
    def getInputDatasetsForOutputDatasetJEDI(self,datasetName):
        comment = ' /* DBProxy.getInputDatasetsForOutputDatasetJEDI */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " <datasetName={0}>".format(datasetName)
        tmpLog = LogWrapper(_logger,methodName)
        tmpLog.debug("start")
        try:
            toSkip = False
            inputDatasets = {}
            # begin transaction
            self.conn.begin()
            # get jediTaskID
            varMap = {}
            varMap[':type1'] = 'log'
            varMap[':type2'] = 'output'
            varMap[':name1'] = datasetName
            varMap[':name2'] = datasetName.split(':')[-1]
            sqlGI  = 'SELECT jediTaskID,datasetID FROM {0}.JEDI_Datasets '.format(panda_config.schemaJEDI)
            sqlGI += 'WHERE type IN (:type1,:type2) AND datasetName IN (:name1,:name2) '
            self.cur.execute(sqlGI+comment,varMap)
            resGI = self.cur.fetchall()
            # use the largest datasetID since broken tasks might have been retried
            jediTaskID = None
            datasetID = None
            for tmpJediTaskID,tmpDatasetID in resGI:
                if jediTaskID is None or jediTaskID < tmpJediTaskID:
                    jediTaskID = tmpJediTaskID
                    datasetID = tmpDatasetID
                elif datasetID < tmpDatasetID:
                    datasetID =tmpDatasetID
            if jediTaskID is None:
                tmpLog.debug("jediTaskID not found")
                toSkip = True
            if not toSkip:
                # get input datasets
                sqlID  = 'SELECT datasetID,datasetName,masterID FROM {0}.JEDI_Datasets '.format(panda_config.schemaJEDI)
                sqlID += 'WHERE jediTaskID=:jediTaskID AND type=:type '
                varMap = {}
                varMap[':jediTaskID'] = jediTaskID
                varMap[':type'] = 'input'
                self.cur.execute(sqlID+comment,varMap)
                resID = self.cur.fetchall()
                for tmpDatasetID,tmpDatasetName,tmpMasterID in resID:
                    inputDatasets[tmpDatasetID] = tmpDatasetName
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            tmpLog.debug("done")
            return True,inputDatasets
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            return False,None



    # record retry history
    def recordRetryHistoryJEDI(self,jediTaskID,newPandaID,oldPandaIDs,relationType):
        comment = ' /* DBProxy.recordRetryHistoryJEDI */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " <PandaID={0}>".format(newPandaID)
        tmpLog = LogWrapper(_logger,methodName)
        tmpLog.debug("start")
        # sql to check record
        sqlCK  = "SELECT jediTaskID FROM {0}.JEDI_Job_Retry_History ".format(panda_config.schemaJEDI)
        sqlCK += "WHERE jediTaskID=:jediTaskID AND oldPandaID=:oldPandaID AND newPandaID=:newPandaID AND originPandaID=:originPandaID "
        # sql to insert record
        sqlIN = "INSERT INTO {0}.JEDI_Job_Retry_History ".format(panda_config.schemaJEDI)
        if relationType is None:
            sqlIN += "(jediTaskID,oldPandaID,newPandaID,originPandaID) "
            sqlIN += "VALUES(:jediTaskID,:oldPandaID,:newPandaID,:originPandaID) "
        else:
            sqlIN += "(jediTaskID,oldPandaID,newPandaID,originPandaID,relationType) "
            sqlIN += "VALUES(:jediTaskID,:oldPandaID,:newPandaID,:originPandaID,:relationType) "
        for oldPandaID in oldPandaIDs:
            # get origin
            originIDs = self.getOriginPandaIDsJEDI(oldPandaID,jediTaskID,self.cur)
            for originID in originIDs:
                # check
                varMap = {}
                varMap[':jediTaskID'] = jediTaskID
                varMap[':oldPandaID'] = oldPandaID
                varMap[':newPandaID'] = newPandaID
                varMap[':originPandaID'] = originID
                self.cur.execute(sqlCK+comment,varMap)
                resCK = self.cur.fetchone()
                # insert
                if resCK is None:
                    varMap = {}
                    varMap[':jediTaskID'] = jediTaskID
                    varMap[':oldPandaID'] = oldPandaID
                    varMap[':newPandaID'] = newPandaID
                    varMap[':originPandaID'] = originID
                    if relationType is not None:
                        varMap[':relationType'] = relationType
                    self.cur.execute(sqlIN+comment,varMap)
        # return
        tmpLog.debug("done")



    # copy file record
    def copyFileRecord(self,newLFN,fileSpec,updateOrig):
        comment = ' /* DBProxy.copyFileRecord */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " <PandaID={0} oldLFN={1} newLFN={2} updateOrig={3}>".format(fileSpec.PandaID, fileSpec.lfn, newLFN, updateOrig)
        tmpLog = LogWrapper(_logger,methodName)
        tmpLog.debug("start")
        try:
            # reset rowID
            tmpFileSpec = copy.copy(fileSpec)
            tmpFileSpec.lfn = newLFN
            if not updateOrig:
                tmpFileSpec.row_ID = None
            # begin transaction
            self.conn.begin()
            # insert file in JEDI
            if not updateOrig and tmpFileSpec.jediTaskID not in [None,'NULL'] and tmpFileSpec.fileID not in ['', 'NULL', None]:
                # get fileID
                sqlFileID = "SELECT ATLAS_PANDA.JEDI_DATASET_CONT_FILEID_SEQ.nextval FROM dual "
                self.cur.execute(sqlFileID+comment)
                newFileID, = self.cur.fetchone()
                # read file in JEDI
                varMap = {}
                varMap[':jediTaskID'] = tmpFileSpec.jediTaskID
                varMap[':datasetID']  = tmpFileSpec.datasetID
                varMap[':fileID']     = tmpFileSpec.fileID
                sqlGI  = 'SELECT * FROM {0}.JEDI_Dataset_Contents '.format(panda_config.schemaJEDI)
                sqlGI += 'WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID '
                self.cur.execute(sqlGI+comment,varMap)
                resGI = self.cur.fetchone()
                tmpFileSpec.fileID = newFileID
                if resGI is not None:
                    # make sql and map
                    sqlJI  = "INSERT INTO {0}.JEDI_Dataset_Contents ".format(panda_config.schemaJEDI)
                    sqlJI += "VALUES ("
                    varMap = {}
                    for columDesc,columVal in zip(self.cur.description,resGI):
                        columName = columDesc[0]
                        # overwrite fileID
                        if columName == 'FILEID':
                            columVal = tmpFileSpec.fileID
                        keyName = ':{0}'.format(columName)
                        varMap[keyName] = columVal
                        sqlJI += '{0},'.format(keyName)
                    sqlJI = sqlJI[:-1]
                    sqlJI += ") "
                    # insert file in JEDI
                    self.cur.execute(sqlJI+comment,varMap)
            if not updateOrig:
                # insert file in Panda
                sqlFile = "INSERT INTO ATLAS_PANDA.filesTable4 ({0}) ".format(FileSpec.columnNames())
                sqlFile+= FileSpec.bindValuesExpression(useSeq=True)
                varMap = tmpFileSpec.valuesMap(useSeq=True)
                self.cur.execute(sqlFile+comment, varMap)
            else:
                # update LFN
                sqlFSF  = "UPDATE ATLAS_PANDA.filesTable4 SET lfn=:lfn "
                sqlFSF += "WHERE row_ID=:row_ID "
                varMap = {}
                varMap[':lfn']    = tmpFileSpec.lfn
                varMap[':row_ID'] = tmpFileSpec.row_ID
                self.cur.execute(sqlFSF+comment,varMap)
            # update LFN in JEDI
            if tmpFileSpec.fileID not in ['', 'NULL', None]:
                sqlJF  = "UPDATE {0}.JEDI_Dataset_Contents ".format(panda_config.schemaJEDI)
                sqlJF += "SET lfn=:lfn "
                sqlJF += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                varMap = {}
                varMap[':lfn']    = tmpFileSpec.lfn
                varMap[':jediTaskID'] = tmpFileSpec.jediTaskID
                varMap[':datasetID']  = tmpFileSpec.datasetID
                varMap[':fileID']     = tmpFileSpec.fileID
                self.cur.execute(sqlJF+comment,varMap)
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            tmpLog.debug("done")
            return True
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            return False

    # get error definitions from DB (values cached for 1 hour)
    @memoize
    def getRetrialRules(self):
        #Logging
        comment = ' /* DBProxy.getRetrialRules */'
        method_name = comment.split(' ')[-2].split('.')[-1]

        tmp_logger = LogWrapper(_logger, method_name)
        tmp_logger.debug("start")

        # SQL to extract the error definitions
        sql  = """
        SELECT re.retryerror_id, re.errorsource, re.errorcode, re.errorDiag, re.parameters, re.architecture, re.release, re.workqueue_id, ra.retry_action, re.active, ra.active
        FROM ATLAS_PANDA.RETRYERRORS re, ATLAS_PANDA.RETRYACTIONS ra
        WHERE re.retryaction=ra.retryaction_id
        AND (CURRENT_TIMESTAMP > re.expiration_date or re.expiration_date IS NULL)
        """
        self.cur.execute(sql+comment, {})
        definitions = self.cur.fetchall()  # example of output: [('pilotErrorCode', 1, None, None, None, None, 'no_retry', 'Y', 'Y'),...]

        # commit
        if not self._commit():
            raise RuntimeError('Commit error')

        tmp_logger.debug("definitions %s"%(definitions))

        retrial_rules = {}
        for definition in definitions:
            retryerror_id, error_source, error_code, error_diag, parameters, architecture, release, wqid, action, e_active, a_active = definition

            #Convert the parameter string into a dictionary
            try:
                # 1. Convert a string like "key1=value1&key2=value2" into [[key1, value1],[key2,value2]]
                params_list = map(lambda key_value_pair: key_value_pair.split("="), parameters.split("&"))
                # 2. Convert a list [[key1, value1],[key2,value2]] into {key1: value1, key2: value2}
                params_dict = dict((key, value) for (key, value) in params_list)
            except AttributeError:
                params_dict = {}
            except ValueError:
                params_dict = {}

            # Calculate if action and error combination should be active
            if e_active == 'Y' and a_active == 'Y':
                active = True  # Apply the action for this error
            else:
                active = False  # Do not apply the action for this error, only log

            retrial_rules.setdefault(error_source,{})
            retrial_rules[error_source].setdefault(error_code,[])
            retrial_rules[error_source][error_code].append({'error_id': retryerror_id,
                                                            'error_diag': error_diag,
                                                            'action': action,
                                                            'params': params_dict,
                                                            'architecture': architecture,
                                                            'release': release,
                                                            'wqid': wqid,
                                                            'active': active})
        tmp_logger.debug("Loaded retrial rules from DB: %s" %retrial_rules)
        return retrial_rules


    def setMaxAttempt(self, jobID, taskID, files, maxAttempt):
        #Logging
        comment = ' /* DBProxy.setMaxAttempt */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        tmpLog = LogWrapper(_logger,methodName)
        tmpLog.debug("start")

        #Update the file entries to avoid JEDI generating new jobs
        input_types = ('input', 'pseudo_input', 'pp_input', 'trn_log','trn_output')
        input_files = list(filter(lambda pandafile: pandafile.type in input_types and re.search('DBRelease', pandafile.lfn) is None, files))
        input_fileIDs = [input_file.fileID for input_file in input_files]
        input_datasetIDs = [input_file.datasetID for input_file in input_files]

        if input_fileIDs:
            try:
                #Start transaction
                self.conn.begin()

                varMap = {}
                varMap[':taskID'] = taskID
                varMap[':pandaID'] = jobID

                #Bind the files
                f = 0
                for fileID in input_fileIDs:
                    varMap[':file{0}'.format(f)] = fileID
                    f+=1
                file_bindings = ','.join(':file{0}'.format(i) for i in range(len(input_fileIDs)))

                #Bind the datasets
                d = 0
                for datasetID in input_datasetIDs:
                    varMap[':dataset{0}'.format(d)] = datasetID
                    d+=1
                dataset_bindings = ','.join(':dataset{0}'.format(i) for i in range(len(input_fileIDs)))

                #Get the minimum maxAttempt value of the files
                sql_select = """
                select min(maxattempt) from ATLAS_PANDA.JEDI_Dataset_Contents
                WHERE JEDITaskID = :taskID
                AND datasetID IN ({0})
                AND fileID IN ({1})
                AND pandaID = :pandaID
                """.format(dataset_bindings, file_bindings)
                self.cur.execute(sql_select+comment, varMap)
                try:
                    maxAttempt_select = self.cur.fetchone()[0]
                except (TypeError, IndexError):
                    maxAttempt_select = None

                # Don't update the maxAttempt if the new value is higher than the old value
                if maxAttempt_select and maxAttempt_select > maxAttempt:
                    varMap[':maxAttempt'] = min(maxAttempt, maxAttempt_select)

                    sql_update  = """
                    UPDATE ATLAS_PANDA.JEDI_Dataset_Contents
                    SET maxAttempt=:maxAttempt
                    WHERE JEDITaskID = :taskID
                    AND datasetID IN ({0})
                    AND fileID IN ({1})
                    AND pandaID = :pandaID
                    """.format(dataset_bindings, file_bindings)

                    self.cur.execute(sql_update+comment, varMap)

                #Commit updates
                if not self._commit():
                    raise RuntimeError('Commit error')
            except Exception:
                # roll back
                self._rollback()
                # error
                self.dumpErrorMessage(_logger, methodName)
                return False


        tmpLog.debug("done")
        return True


    def setNoRetry(self, jobID, taskID, files):
        # Logging
        comment = ' /* DBProxy.setNoRetry */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " <PandaID={0} jediTaskID={1}>".format(jobID, taskID)
        tmpLog = LogWrapper(_logger, methodName)
        tmpLog.debug("start")

        # Update the file entries to avoid JEDI generating new jobs
        input_types = ('input', 'pseudo_input', 'pp_input', 'trn_log', 'trn_output')
        input_files = list(filter(lambda pandafile: pandafile.type in input_types and re.search('DBRelease', pandafile.lfn) is None, files))
        input_fileIDs = [input_file.fileID for input_file in input_files]
        input_datasetIDs = [input_file.datasetID for input_file in input_files]

        if input_fileIDs:
            try:
                # Start transaction
                self.conn.begin()

                # loop over all datasets
                for datasetID in input_datasetIDs:
                    varMap = {}
                    varMap[':taskID'] = taskID
                    varMap[':datasetID'] = datasetID
                    varMap[':keepTrack'] = 1

                    # Bind the files
                    f = 0
                    for fileID in input_fileIDs:
                        varMap[':file{0}'.format(f)] = fileID
                        f += 1
                    file_bindings = ','.join(':file{0}'.format(i) for i in range(len(input_fileIDs)))

                    sql_update = """
                    UPDATE ATLAS_PANDA.JEDI_Dataset_Contents
                    SET maxAttempt=attemptNr
                    WHERE JEDITaskID = :taskID
                    AND datasetID=:datasetID
                    AND fileID IN ({0})
                    AND maxAttempt IS NOT NULL AND attemptNr IS NOT NULL
                    AND maxAttempt > attemptNr
                    AND (maxFailure IS NULL OR failedAttempt IS NULL OR maxFailure > failedAttempt)
                    AND keepTrack=:keepTrack
                    AND status=:status
                    """.format(file_bindings)

                    # update files in 'running' status. These files do NOT need to be counted for the nFiles*
                    varMap[':status'] = 'running'
                    self.cur.execute(sql_update + comment, varMap)

                    # update files in 'ready' status. These files need to be counted for the nFiles*
                    varMap[':status'] = 'ready'
                    self.cur.execute(sql_update + comment, varMap)
                    rowcount = self.cur.rowcount

                    # update datasets
                    if rowcount > 0:
                        sql_dataset = "UPDATE ATLAS_PANDA.JEDI_Datasets "
                        sql_dataset += "SET nFilesUsed=nFilesUsed+:nDiff,nFilesFailed=nFilesFailed+:nDiff "
                        sql_dataset += "WHERE jediTaskID=:taskID AND datasetID=:datasetID "
                        varMap = dict()
                        varMap[':taskID'] = taskID
                        varMap[':datasetID'] = datasetID
                        varMap[':nDiff'] = rowcount
                        self.cur.execute(sql_dataset + comment, varMap)

                # Commit updates
                if not self._commit():
                    raise RuntimeError('Commit error')
            except Exception:
                # roll back
                self._rollback()
                # error
                self.dumpErrorMessage(_logger, methodName)
                return False

        tmpLog.debug("done")
        return True


    def increaseCpuTimeTask(self, jobID, taskID, siteid, files, active):
        """
        Increases the CPU time of a task
        walltime = basewalltime + cpuefficiency*CPUTime*nEvents/Corepower/Corecount

        CPU time: execution time per event
        Walltime: time for a job
        Corepower: HS06 score
        Basewalltime: Setup time, time to download, etc. taken by the pilot
        """
        comment = ' /* DBProxy.increaseCpuTimeTask */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " <PandaID={0}; TaskID={1}>".format(jobID, taskID)
        tmpLog = LogWrapper(_logger,methodName)
        tmpLog.debug("start")

        #1. Get the site information from schedconfig
        sql = """
        SELECT /* use_json_type */ sc.data.maxtime, sc.data.corepower,
            CASE
                WHEN sc.data.corecount IS NULL THEN 1
                ELSE sc.data.corecount
            END as corecount
        FROM ATLAS_PANDA.schedconfig_json sc
        WHERE sc.panda_queue=:siteid
        """
        varMap={"siteid": siteid}
        self.cur.execute(sql+comment, varMap)
        siteParameters = self.cur.fetchone()   #example of output: [('pilotErrorCode', 1, None, None, None, None, 'no_retry', 'Y', 'Y'),...]

        if not siteParameters:
            tmpLog.debug("No site parameters retrieved for {0}".format(siteid))

        (maxtime, corepower, corecount) = siteParameters
        tmpLog.debug("siteid {0} has parameters: maxtime {1}, corepower {2}, corecount {3}".format(siteid, maxtime, corepower, corecount))
        if (not maxtime) or (not corepower) or (not corecount):
            tmpLog.debug("One or more site parameters are not defined for {0}... nothing to do".format(siteid))
            return None
        else:
            (maxtime, corepower, corecount) = (int(maxtime), float(corepower), int(corecount))

        #2. Get the task information
        sql = """
        SELECT jt.cputime, jt.walltime, jt.basewalltime, jt.cpuefficiency, jt.cputimeunit
        FROM ATLAS_PANDA.jedi_tasks jt
        WHERE jt.jeditaskid=:jeditaskid
        """
        varMap={"jeditaskid": taskID}
        self.cur.execute(sql+comment, varMap)
        taskParameters = self.cur.fetchone()

        if not taskParameters:
            tmpLog.debug("No task parameters retrieved for jeditaskid {0}... nothing to do".format(taskID))
            return None

        (cputime, walltime, basewalltime, cpuefficiency, cputimeunit) = taskParameters
        if not cpuefficiency or not basewalltime:
            tmpLog.debug("CPU efficiency and/or basewalltime are not defined for task {0}... nothing to do".format(taskID))
            return None

        tmpLog.debug("task {0} has parameters: cputime {1}, walltime {2}, basewalltime {3}, cpuefficiency {4}, cputimeunit {5}".\
                     format(taskID, cputime, walltime, basewalltime, cpuefficiency, cputimeunit))

        #2. Get the file information
        input_types = ('input', 'pseudo_input', 'pp_input', 'trn_log', 'trn_output')
        input_files = list(filter(lambda pandafile: pandafile.type in input_types
                                               and re.search('DBRelease', pandafile.lfn) is None, files))
        input_fileIDs = [input_file.fileID for input_file in input_files]
        input_datasetIDs = [input_file.datasetID for input_file in input_files]

        if input_fileIDs:
            varMap = {}
            varMap[':taskID'] = taskID
            varMap[':pandaID'] = jobID

            #Bind the files
            f = 0
            for fileID in input_fileIDs:
                varMap[':file{0}'.format(f)] = fileID
                f+=1
            file_bindings = ','.join(':file{0}'.format(i) for i in range(len(input_fileIDs)))

            #Bind the datasets
            d = 0
            for datasetID in input_datasetIDs:
                varMap[':dataset{0}'.format(d)] = datasetID
                d+=1
            dataset_bindings = ','.join(':dataset{0}'.format(i) for i in range(len(input_fileIDs)))

            sql_select = """
            SELECT jdc.fileid, jdc.nevents, jdc.startevent, jdc.endevent
            FROM ATLAS_PANDA.JEDI_Dataset_Contents jdc, ATLAS_PANDA.JEDI_Datasets jd
            WHERE jdc.JEDITaskID = :taskID
            AND jdc.datasetID IN ({0})
            AND jdc.fileID IN ({1})
            AND jd.datasetID = jdc.datasetID
            AND jd.masterID IS NULL
            AND jdc.pandaID = :pandaID
            """.format(dataset_bindings, file_bindings)
            self.cur.execute(sql_select+comment, varMap)

            resList = self.cur.fetchall()
            nevents_total = 0
            for fileid, nevents, startevent, endevent in resList:
                tmpLog.debug("event information: fileid {0}, nevents {1}, startevent {2}, endevent {3}".format(fileid, nevents, startevent, endevent))

                if endevent is not None and startevent is not None:
                    nevents_total += endevent - startevent
                elif nevents:
                    nevents_total += nevents

            if not nevents_total:
                tmpLog.debug("nevents could not be calculated for job {0}... nothing to do".format(jobID))
                return None
        else:
            tmpLog.debug("No input files for job {0}, so could not update CPU time for task {1}".format(jobID, taskID))
            return None

        try:
            new_cputime = (maxtime - basewalltime) * corepower * corecount * 1.1 / (cpuefficiency/100.0) / nevents_total

            if cputime > new_cputime:
                tmpLog.debug("Skipping CPU time increase since old CPU time {0} > new CPU time {1}".format(cputime, new_cputime))
                return None

            if active: # only run the update if active mode. Otherwise return what would have been done
                sql_update_cputime = """
                UPDATE ATLAS_PANDA.jedi_tasks SET cputime=:cputime
                WHERE jeditaskid=:jeditaskid
                """
                varMap = {}
                varMap[':cputime'] = new_cputime
                varMap[':jeditaskid'] = taskID
                self.conn.begin()
                self.cur.execute(sql_update_cputime+comment, varMap)
                if not self._commit():
                    raise RuntimeError('Commit error')

                tmpLog.debug("Successfully updated the task CPU time from {0} to {1}".format(cputime, new_cputime))
            return new_cputime

        except (ZeroDivisionError, TypeError):
            return None


    def requestTaskParameterRecalculation(self, taskID):
        """
        Requests the recalculation of the CPU time of a task:
         1. set the walltimeUnit to NULL and the modificationTime to Now
         2. AtlasProdWatchDog > JediDBProxy.setScoutJobDataToTasks will pick up tasks with walltimeUnit == NULL
            and modificationTime > Now - 24h. This will trigger a recalculation of the task parameters (outDiskCount,
            outDiskUnit, outDiskCount, walltime, walltimeUnit, cpuTime, ioIntensity, ioIntensityUnit, ramCount, ramUnit,
            workDiskCount, workDiskUnit, workDiskCount)
        """
        comment = ' /* DBProxy.requestTaskParameterRecalculation */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " TaskID={0}>".format(taskID)
        tmpLog = LogWrapper(_logger, methodName)
        tmpLog.debug("start")

        timeNow = datetime.datetime.utcnow()
        timeLimit = timeNow -  datetime.timedelta(minutes=30)

        # update the task if it was not already updated in the last 30 minutes (avoid continuous recalculation)
        sql  = """
               UPDATE ATLAS_PANDA.jedi_tasks
               SET walltimeUnit=NULL, modificationTime=:timeNow
               WHERE jediTaskId=:taskID AND modificationTime < :timeLimit
               """
        varMap={"taskID": taskID,
                "timeNow": timeNow,
                "timeLimit": timeLimit}
        self.conn.begin()
        self.cur.execute(sql, varMap)

        rowcount = self.cur.rowcount

        if not self._commit():
            raise RuntimeError('Commit error')

        tmpLog.debug("Forced recalculation of CPUTime")
        return rowcount


    # throttle jobs for resource shares
    def throttleJobsForResourceShare(self,site):
        comment = ' /* DBProxy.throttleJobsForResourceShare */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " <site={0}>".format(site)
        tmpLog = LogWrapper(_logger,methodName)
        tmpLog.debug("start")
        try:
            # sql to throttle jobs
            sql  = "UPDATE ATLAS_PANDA.jobsActive4 SET jobStatus=:newStatus "
            sql += "WHERE computingSite=:site AND jobStatus=:oldStatus AND lockedby=:lockedby "
            varMap = {}
            varMap[':site'] = site
            varMap[':lockedby'] = 'jedi'
            varMap[':newStatus'] = 'throttled'
            varMap[':oldStatus'] = 'activated'
            # begin transaction
            self.conn.begin()
            self.cur.execute(sql+comment,varMap)
            nRow = self.cur.rowcount
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            tmpLog.debug("throttled {0} jobs".format(nRow))
            return nRow
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            return None



    # activate jobs for resource shares
    def activateJobsForResourceShare(self,site,nJobsPerQueue):
        comment = ' /* DBProxy.activateJobsForResourceShare */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " <site={0} nJobsPerQueue={1}>".format(site,nJobsPerQueue)
        tmpLog = LogWrapper(_logger,methodName)
        tmpLog.debug("start")
        try:
            # sql to get jobs
            sqlJ  = "SELECT PandaID,jobStatus FROM ("
            sqlJ += "SELECT PandaID,jobStatus,"
            sqlJ += "ROW_NUMBER() OVER(PARTITION BY workqueue_id ORDER BY currentPriority DESC,PandaID) AS row_number "
            sqlJ += "FROM ATLAS_PANDA.jobsActive4 "
            sqlJ += "WHERE computingSite=:site AND lockedby=:lockedby AND jobStatus IN (:st1,:st2) "
            sqlJ += ") "
            sqlJ += "WHERE row_number<={0} ".format(nJobsPerQueue)
            # sql to activate jobs
            sqlA  = "UPDATE ATLAS_PANDA.jobsActive4 SET jobStatus=:newStatus "
            sqlA += "WHERE PandaID=:PandaID AND jobStatus=:oldStatus "
            varMap = {}
            varMap[':site'] = site
            varMap[':lockedby'] = 'jedi'
            varMap[':st1'] = 'throttled'
            varMap[':st2'] = 'activated'
            # begin transaction
            self.conn.begin()
            self.cur.execute(sqlJ+comment,varMap)
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            resList = self.cur.fetchall()
            nRow = 0
            for pandaID,jobStatus in resList:
                if jobStatus == 'activated':
                    continue
                # activate job
                varMap = {}
                varMap[':PandaID'] = pandaID
                varMap[':newStatus'] = 'activated'
                varMap[':oldStatus'] = 'throttled'
                self.conn.begin()
                self.cur.execute(sqlA+comment,varMap)
                # commit
                if not self._commit():
                    raise RuntimeError('Commit error')
                nRow += self.cur.rowcount
            tmpLog.debug("activated {0} jobs".format(nRow))
            return nRow
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            return None



    # add associate sub datasets for single consumer job
    def getDestDBlocksWithSingleConsumer(self,jediTaskID,PandaID,ngDatasets):
        comment = ' /* DBProxy.getDestDBlocksWithSingleConsumer */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " <jediTaskID={0},PandaID={1}>".format(jediTaskID,PandaID)
        tmpLog = LogWrapper(_logger,methodName)
        tmpLog.debug("start")
        try:
            retMap = {}
            checkedDS = set()
            # sql to get files
            sqlF  = "SELECT datasetID,fileID FROM ATLAS_PANDA.JEDI_Events "
            sqlF += "WHERE jediTaskID=:jediTaskID AND PandaID=:PandaID "
            # sql to get PandaIDs
            sqlP  = "SELECT distinct PandaID FROM ATLAS_PANDA.filesTable4 "
            sqlP += "WHERE jediTaskID=:jediTaskID ANd datasetID=:datasetID AND fileID=:fileID "
            # sql to get sub datasets
            sqlD  = "SELECT destinationDBlock,datasetID FROM ATLAS_PANDA.filesTable4 "
            sqlD += "WHERE PandaID=:PandaID AND type IN (:type1,:type2) "
            # sql to get PandaIDs in merging
            sqlM  = "SELECT distinct PandaID FROM ATLAS_PANDA.filesTable4 "
            sqlM += "WHERE jediTaskID=:jediTaskID ANd datasetID=:datasetID AND status=:status "
            varMap = {}
            varMap[':jediTaskID'] = jediTaskID
            varMap[':PandaID']    = PandaID
            # begin transaction
            self.conn.begin()
            # get files
            self.cur.execute(sqlF+comment,varMap)
            resF = self.cur.fetchall()
            for datasetID,fileID in resF:
                # get parallel jobs
                varMap = {}
                varMap[':jediTaskID'] = jediTaskID
                varMap[':datasetID']  = datasetID
                varMap[':fileID']     = fileID
                self.cur.execute(sqlP+comment,varMap)
                resP = self.cur.fetchall()
                for sPandaID, in resP:
                    if sPandaID == PandaID:
                        continue
                    # get sub datasets of parallel jobs
                    varMap = {}
                    varMap[':PandaID']    = sPandaID
                    varMap[':type1'] = 'output'
                    varMap[':type2'] = 'log'
                    self.cur.execute(sqlD+comment,varMap)
                    resD = self.cur.fetchall()
                    subDatasets = []
                    subDatasetID = None
                    for destinationDBlock,datasetID in resD:
                        if destinationDBlock in ngDatasets:
                            continue
                        if destinationDBlock in checkedDS:
                            continue
                        checkedDS.add(destinationDBlock)
                        subDatasets.append(destinationDBlock)
                        subDatasetID = datasetID
                    if subDatasets == []:
                        continue
                    # get merging PandaID which uses sub dataset
                    varMap = {}
                    varMap[':jediTaskID'] = jediTaskID
                    varMap[':datasetID']  = datasetID
                    varMap[':status']     = 'merging'
                    self.cur.execute(sqlM+comment,varMap)
                    resM = self.cur.fetchone()
                    if resM is not None:
                        mPandaID, = resM
                        retMap[mPandaID] = subDatasets
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            tmpLog.debug("got {0} jobs".format(len(retMap)))
            return retMap
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            return {}



    # check validity of merge job
    def isValidMergeJob(self,pandaID,jediTaskID):
        comment = ' /* DBProxy.isValidMergeJob */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " <Panda={0}>".format(pandaID)
        tmpLog = LogWrapper(_logger,methodName)
        tmpLog.debug("start")
        try:
            retVal = True
            retMsg = ''
            # sql to check if merge job is active
            sqlJ  = "SELECT jobStatus FROM ATLAS_PANDA.jobsDefined4 WHERE PandaID=:PandaID "
            sqlJ += "UNION "
            sqlJ += "SELECT jobStatus FROM ATLAS_PANDA.jobsActive4 WHERE PandaID=:PandaID "
            # sql to get input files
            sqlF  = "SELECT datasetID,fileID FROM ATLAS_PANDA.filesTable4 "
            sqlF += "WHERE PandaID=:PandaID AND type IN (:type1,:type2) "
            # sql to get PandaIDs for pre-merged jobs
            sqlP  = "SELECT outPandaID FROM ATLAS_PANDA.JEDI_Dataset_Contents "
            sqlP += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID AND type<>:type1"
            # sql to check if pre-merge job is active
            sqlC  = "SELECT jobStatus FROM ATLAS_PANDA.jobsActive4 WHERE PandaID=:PandaID "
            # begin transaction
            self.conn.begin()
            # check if merge job is active
            varMap = {}
            varMap[':PandaID'] = pandaID
            self.cur.execute(sqlJ+comment,varMap)
            resJ = self.cur.fetchone()
            if resJ is None:
                tmpLog.debug("merge job not found")
            else:
                # get input files
                varMap = {}
                varMap[':PandaID'] = pandaID
                varMap[':type1'] = 'input'
                varMap[':type2'] = 'pseudo_input'
                self.cur.execute(sqlF+comment,varMap)
                resF = self.cur.fetchall()
                firstDatasetID = None
                fileIDsMap = {}
                for datasetID,fileID in resF:
                    if datasetID not in fileIDsMap:
                        fileIDsMap[datasetID] = set()
                    fileIDsMap[datasetID].add(fileID)
                # get PandaIDs for pre-merged jobs
                pandaIDs = set()
                for datasetID in fileIDsMap:
                    fileIDs = fileIDsMap[datasetID]
                    for fileID in fileIDs:
                        varMap = {}
                        varMap[':jediTaskID'] = jediTaskID
                        varMap[':datasetID']  = datasetID
                        varMap[':fileID']     = fileID
                        varMap[':type1']      = 'lib'
                        self.cur.execute(sqlP+comment,varMap)
                        resP = self.cur.fetchone()
                        if resP is not None and resP[0] is not None:
                            pandaIDs.add(resP[0])
                    # only files in the first dataset are enough
                    if len(pandaIDs) > 0:
                        break
                # check pre-merge job
                for tmpPandaID in pandaIDs:
                    varMap = {}
                    varMap[':PandaID'] = tmpPandaID
                    self.cur.execute(sqlC+comment,varMap)
                    resC = self.cur.fetchone()
                    if resC is None:
                        # not found
                        tmpLog.debug("pre-merge job {0} not found".format(tmpPandaID))
                        retVal = False
                        retMsg = tmpPandaID
                        break
                    elif resC[0] != 'merging':
                        # not in merging
                        tmpLog.debug("pre-merge job in {0} != merging".format(tmpPandaID,resC[0]))
                        retVal = False
                        retMsg = tmpPandaID
                        break
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            tmpLog.debug("ret={0}".format(retVal))
            return retVal,retMsg
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            return None,""



    # update related ES jobs when ES-merge job is done
    def updateRelatedEventServiceJobs(self,job,killEvents=False, forceFailed=False):
        comment = ' /* DBProxy.updateRelatedEventServiceJobs */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " <PandaID={0}>".format(job.PandaID)
        tmpLog = LogWrapper(_logger,methodName)
        if forceFailed:
            jobStatus = 'failed'
        else:
            jobStatus = job.jobStatus
        if not forceFailed and jobStatus not in ['finished'] and not (killEvents and not job.notDiscardEvents()):
            tmpLog.debug("skip jobStatus={0} killEvents={1} discard={2}".format(jobStatus, killEvents, job.notDiscardEvents()))
            return True
        tmpLog.debug("start jobStatus={0} killEvents={1} discard={2}".format(jobStatus, killEvents, job.notDiscardEvents()))
        try:
            # sql to read range
            sqlRR  = "SELECT /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
            sqlRR += "distinct PandaID "
            sqlRR += "FROM {0}.JEDI_Events tab ".format(panda_config.schemaJEDI)
            sqlRR += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID AND status IN (:es_done,:es_finished,:es_merged) "
            # loop over all files
            esPandaIDs = set()
            for tmpFile in job.Files:
                # only for input
                if tmpFile.type in ['input','pseudo_input']:
                    # get ranges
                    if tmpFile.fileID is [None, 'NULL']:
                        continue
                    varMap = {}
                    varMap[':jediTaskID']  = tmpFile.jediTaskID
                    varMap[':datasetID']   = tmpFile.datasetID
                    varMap[':fileID']      = tmpFile.fileID
                    varMap[':es_done']   = EventServiceUtils.ST_done
                    varMap[':es_finished'] = EventServiceUtils.ST_finished
                    varMap[':es_merged'] = EventServiceUtils.ST_merged
                    self.cur.execute(sqlRR+comment,varMap)
                    resRR = self.cur.fetchall()
                    for tmpPandaID, in resRR:
                        esPandaIDs.add(tmpPandaID)
            # sql to update ES job
            sqlUE  = "UPDATE {0} SET jobStatus=:newStatus,stateChangeTime=CURRENT_DATE,taskBufferErrorDiag=:errDiag "
            if jobStatus in ['failed']:
                updateSubStatus = True
                sqlUE += ',jobSubStatus=:jobSubStatus '
            else:
                updateSubStatus = False
            sqlUE += "WHERE PandaID=:PandaID AND jobStatus in (:oldStatus1,:oldStatus2,:oldStatus3) AND modificationTime>(CURRENT_DATE-90) "
            sqlUE += "AND NOT eventService IN (:esJumbo) "
            for tmpPandaID in esPandaIDs:
                varMap = {}
                varMap[':PandaID']   = tmpPandaID
                varMap[':newStatus'] = jobStatus
                varMap[':oldStatus1'] = 'closed'
                varMap[':oldStatus2'] = 'merging'
                varMap[':oldStatus3'] = 'failed'
                varMap[':esJumbo'] = EventServiceUtils.jumboJobFlagNumber
                if updateSubStatus is True:
                    if forceFailed:
                        varMap[':jobSubStatus'] = 'es_discard'
                    elif EventServiceUtils.isEventServiceMerge(job):
                        varMap[':jobSubStatus'] = 'es_merge_{0}'.format(jobStatus)
                    else:
                        varMap[':jobSubStatus'] = 'es_ass_{0}'.format(jobStatus)
                if forceFailed:
                    varMap[':errDiag'] = '{0} to discard old events to rety in PandaID={1}'.format(jobStatus,job.PandaID)
                else:
                    varMap[':errDiag'] = '{0} since an associated ES or merge job PandaID={1} {2}'.format(jobStatus,job.PandaID,jobStatus)
                isUpdated = False
                for tableName in ['ATLAS_PANDA.jobsArchived4','ATLAS_PANDAARCH.jobsArchived']:
                    self.cur.execute(sqlUE.format(tableName)+comment,varMap)
                    nRow = self.cur.rowcount
                    if nRow > 0:
                        tmpLog.debug('change PandaID={0} to {1}'.format(tmpPandaID,jobStatus))
                        isUpdated = True
                # kill processed events if necessary
                if killEvents and isUpdated:
                    self.killUsedEventRanges(job.jediTaskID, tmpPandaID, job.notDiscardEvents())
            tmpLog.debug("done")
            return True
        except Exception:
            # error
            self.dumpErrorMessage(_logger,methodName)
            return False


    # Configurator function: inserts data into the network matrix
    def insertNetworkMatrixData(self, data):
        comment = ' /* DBProxy.insertNetworkMatrixData */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        tmpLog = LogWrapper(_logger,methodName)
        tmpLog.debug("start")

        # For performance reasons we will insert the data into a temporary table
        # and then merge this data into the permanent table.

        sql_insert = """
        INSERT INTO ATLAS_PANDA.network_matrix_kv_temp (src, dst, key, value, ts)
        VALUES (:src, :dst, :key, :value, :ts)
        """

        if self.backend == 'postgres':
            sql_merge = "INSERT INTO ATLAS_PANDA.network_matrix_kv "\
                        "(src, dst, key, value, ts) "\
                        "SELECT  src, dst, key, value, ts FROM ATLAS_PANDA.NETWORK_MATRIX_KV_TEMP "\
                        "ON CONFLICT (src, dst, key) "\
                        "DO UPDATE SET value=EXCLUDED.value, ts=EXCLUDED.ts "
        else:
            sql_merge = """
            MERGE /*+ FULL(nm_kv) */ INTO ATLAS_PANDA.network_matrix_kv nm_kv USING
                (SELECT  src, dst, key, value, ts FROM ATLAS_PANDA.NETWORK_MATRIX_KV_TEMP) input
                ON (nm_kv.src = input.src AND nm_kv.dst= input.dst AND nm_kv.key = input.key)
            WHEN NOT MATCHED THEN
                INSERT (src, dst, key, value, ts)
                VALUES (input.src, input.dst, input.key, input.value, input.ts)
            WHEN MATCHED THEN
                UPDATE SET nm_kv.value = input.value, nm_kv.ts = input.ts
            """
        try:
            self.conn.begin()
            for shard in create_shards(data, 100):
                time1 = time.time()
                var_maps = []
                for entry in shard:
                    var_map = {
                        ':src': entry[0],
                        ':dst': entry[1],
                        ':key': entry[2],
                        ':value': entry[3],
                        ':ts': entry[4]
                    }
                    var_maps.append(var_map)

                time2 = time.time()
                self.cur.executemany(sql_insert+comment, var_maps)
                time3 = time.time()
                tmpLog.debug("Processing a shard took: {0}s of data preparation and {1}s of insertion = {2}".format(time2 - time1, time3 - time2, time3 - time1))

            time4 = time.time()
            self.cur.execute(sql_merge+comment)
            time5 = time.time()
            tmpLog.debug("Final merge took: {0}s".format(time5 - time4))
            if self.backend == 'postgres':
                # cleanup since ON CONFLICT DO UPDATE doesn't work with duplicated entries
                self.cur.execute("DELETE FROM ATLAS_PANDA.NETWORK_MATRIX_KV_TEMP " + comment)
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')

        except Exception:
            # roll back
            self._rollback()
            # error
            exc_type, exc_value, exc_traceback = sys.exc_info()
            tmpLog.debug("Failed to commit bulk with exception {0}".format(repr(traceback.format_exception(exc_type, exc_value, exc_traceback))))
            self.dumpErrorMessage(_logger, methodName)
            return None,""



    # Configurator function: delete old network data
    def deleteOldNetworkData(self):
        comment = ' /* DBProxy.deleteOldNetworkData */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        tmpLog = LogWrapper(_logger,methodName)
        tmpLog.debug("start")

        # delete any data older than a week
        sql_delete = """
        DELETE FROM ATLAS_PANDA.network_matrix_kv
        WHERE ts < (current_date - 7)
        """
        try:
            self.conn.begin()
            time1 = time.time()
            self.cur.execute(sql_delete + comment)
            time2 = time.time()
            tmpLog.debug("Deletion of old network data took: {0}s".format(time2 - time1))

            # commit
            if not self._commit():
                raise RuntimeError('Commit error')

        except Exception:
            # roll back
            self._rollback()
            # error
            exc_type, exc_value, exc_traceback = sys.exc_info()
            tmpLog.debug("Failed to commit: {0}".format(repr(traceback.format_exception(exc_type, exc_value, exc_traceback))))
            self.dumpErrorMessage(_logger, methodName)
            return None,""



    # get dispatch datasets per user
    def getDispatchDatasetsPerUser(self,vo,prodSourceLabel,onlyActive,withSize):
        comment = ' /* DBProxy.getDispatchDatasetsPerUser */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " <vo={0} label={1}>".format(vo,prodSourceLabel)
        tmpLog = LogWrapper(_logger,methodName)
        tmpLog.debug("start")
        # mapping for table and job status
        tableStatMap = {'jobsDefined4':['defined','assigned']}
        if not onlyActive:
            tableStatMap['jobsActive4'] = None
            tableStatMap['jobsArchived4'] = None
        try:
            userDispMap = {}
            for tableName in tableStatMap:
                statusList = tableStatMap[tableName]
                # make sql to get dispatch datasets
                varMap = {}
                varMap[':vo'] = vo
                varMap[':label'] = prodSourceLabel
                varMap[':dType'] = 'dispatch'
                sqlJ  = "SELECT distinct prodUserName,dispatchDBlock,jediTaskID,currentFiles "
                sqlJ += "FROM {0}.{1} j, {0}.Datasets d ".format(panda_config.schemaPANDA,tableName)
                sqlJ += "WHERE vo=:vo AND prodSourceLabel=:label "
                if statusList is not None:
                    sqlJ += "AND jobStatus IN ("
                    for tmpStat in statusList:
                        tmpKey = ':jobStat_{0}'.format(tmpStat)
                        sqlJ += '{0},'.format(tmpKey)
                        varMap[tmpKey] = tmpStat
                    sqlJ = sqlJ[:-1]
                    sqlJ += ") "
                sqlJ += "AND dispatchDBlock IS NOT NULL "
                sqlJ += "AND d.name=j.dispatchDBlock AND d.modificationDate>CURRENT_DATE-14 "
                sqlJ += 'AND d.type=:dType '
                # begin transaction
                self.conn.begin()
                # get dispatch datasets
                self.cur.execute(sqlJ+comment,varMap)
                resJ = self.cur.fetchall()
                if not self._commit():
                    raise RuntimeError('Commit error')
                # make map
                for prodUserName,dispatchDBlock,jediTaskID,dsSize in resJ:
                    transferType = 'transfer'
                    try:
                        if dispatchDBlock.split('.')[4] == 'prestaging':
                            transferType = 'prestaging'
                    except Exception:
                        pass
                    userDispMap.setdefault(prodUserName, {})
                    userDispMap[prodUserName].setdefault(transferType,
                                                         {'datasets':set(),
                                                         'size':0,
                                                         'tasks':set()})
                    if dispatchDBlock not in userDispMap[prodUserName][transferType]['datasets']:
                        userDispMap[prodUserName][transferType]['datasets'].add(dispatchDBlock)
                        userDispMap[prodUserName][transferType]['tasks'].add(jediTaskID)
                        userDispMap[prodUserName][transferType]['size'] += dsSize
            tmpLog.debug("done")
            return userDispMap
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            return {}



    # get task parameters
    def getTaskPramsPanda(self,jediTaskID):
        comment = ' /* DBProxy.getTaskPramsPanda */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " <jediTaskID={0}>".format(jediTaskID)
        tmpLog = LogWrapper(_logger,methodName)
        tmpLog.debug("start")
        try:
            # sql to get task parameters
            sqlRR  = "SELECT jedi_task_parameters FROM {0}.T_TASK ".format(panda_config.schemaDEFT)
            sqlRR += "WHERE taskid=:jediTaskID "
            varMap = {}
            varMap[':jediTaskID'] = jediTaskID
            # start transaction
            self.conn.begin()
            self.cur.execute(sqlRR+comment,varMap)
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # read clob
            taskParams = ''
            for clobJobP, in self.cur:
                if clobJobP is not None:
                    try:
                        taskParams= clobJobP.read()
                    except AttributeError:
                        taskParams = str(clobJobP)
                break
            tmpLog.debug("done")
            return taskParams
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            return ''



    # get task attributes
    def getTaskAttributesPanda(self,jediTaskID,attrs):
        comment = ' /* DBProxy.getTaskAttributesPanda */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        methodName += " <jediTaskID={0}>".format(jediTaskID)
        tmpLog = LogWrapper(_logger,methodName)
        tmpLog.debug("start")
        try:
            # sql to get task attributes
            sqlRR  = "SELECT "
            for attr in attrs:
                sqlRR += '{0},'.format(attr)
            sqlRR = sqlRR[:-1]
            sqlRR += ' FROM {0}.JEDI_Tasks '.format(panda_config.schemaJEDI)
            sqlRR += "WHERE jediTaskID=:jediTaskID "
            varMap = {}
            varMap[':jediTaskID'] = jediTaskID
            # start transaction
            self.conn.begin()
            self.cur.execute(sqlRR+comment,varMap)
            resRR = self.cur.fetchone()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            retVal = {}
            if resRR is not None:
                for idx,attr in enumerate(attrs):
                    retVal[attr] = resRR[idx]
            tmpLog.debug("done {0}".format(str(retVal)))
            return retVal
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            return {}



    # check for cloned jobs
    def checkClonedJob(self,jobSpec,useCommit=True):
        comment = ' /* DBProxy.checkClonedJob */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        tmpLog = LogWrapper(_logger,methodName+" <PandaID={0}>".format(jobSpec.PandaID))
        tmpLog.debug("start")
        try:
            # return value {'lock': True if the job locked the semaphore,
            #               'last': True if the job is the last clone
            #               'win': PandaID of winner
            # None : fatal error
            retValue = {'lock':False,
                        'last':False,
                        'win':None}
            # begin transaction
            if useCommit:
                self.conn.begin()
            self.cur.arraysize = 10000
            # check if semaphore is locked
            sqlED  = "SELECT COUNT(*) FROM {0}.JEDI_Events ".format(panda_config.schemaJEDI)
            sqlED += "WHERE jediTaskID=:jediTaskID AND pandaID=:pandaID "
            varMap = {}
            varMap[':jediTaskID'] = jobSpec.jediTaskID
            varMap[':pandaID']    = jobSpec.PandaID
            self.cur.execute(sqlED+comment, varMap)
            resEU = self.cur.fetchone()
            nRowEU, = resEU
            if nRowEU > 0:
                retValue['lock'] = True
            else:
                # get PandaID of the winner
                sqlWP  = "SELECT /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
                sqlWP += "distinct PandaID "
                sqlWP += "FROM {0}.JEDI_Events tab ".format(panda_config.schemaJEDI)
                sqlWP += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
                for tmpFileSpec in jobSpec.Files:
                    if tmpFileSpec.type == 'input':
                        varMap = {}
                        varMap[':jediTaskID'] = tmpFileSpec.jediTaskID
                        varMap[':datasetID']  = tmpFileSpec.datasetID
                        varMap[':fileID']     = tmpFileSpec.fileID
                        self.cur.execute(sqlWP+comment, varMap)
                        resWP = self.cur.fetchone()
                        if resWP is not None:
                            retValue['win'] = resWP[0]
                            break
            # get PandaIDs of clones
            sqlCP  = "SELECT PandaID FROM ATLAS_PANDA.jobsActive4 "
            sqlCP += "WHERE jediTaskID=:jediTaskID AND jobsetID=:jobsetID "
            sqlCP += "UNION "
            sqlCP += "SELECT PandaID FROM ATLAS_PANDA.jobsDefined4 "
            sqlCP += "WHERE jediTaskID=:jediTaskID AND jobsetID=:jobsetID "
            sqlCP += "UNION "
            sqlCP += "SELECT PandaID FROM ATLAS_PANDA.jobsWaiting4 "
            sqlCP += "WHERE jediTaskID=:jediTaskID AND jobsetID=:jobsetID "
            varMap = {}
            varMap[':jediTaskID']  = jobSpec.jediTaskID
            varMap[':jobsetID']    = jobSpec.jobsetID
            self.cur.execute(sqlCP+comment, varMap)
            resCP = self.cur.fetchall()
            pandaIDsList = set()
            for pandaID, in resCP:
                if pandaID != jobSpec.PandaID:
                    pandaIDsList.add(pandaID)
            if len(pandaIDsList) == 0:
                retValue['last'] = True
            # commit
            if useCommit:
                if not self._commit():
                    raise RuntimeError('Commit error')
            tmpLog.debug(retValue)
            return retValue
        except Exception:
            # roll back
            if useCommit:
                self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            return None



    # set HS06sec
    def setHS06sec(self, pandaID, inActive=False):
        comment = ' /* DBProxy.setHS06sec */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        tmpLog = LogWrapper(_logger,methodName+" <PandaID={0}>".format(pandaID))
        tmpLog.debug("start")
        hs06sec = None

        # sql to get job attributes
        sqlJ  = "SELECT jediTaskID,startTime,endTime,actualCoreCount,coreCount,jobMetrics,computingSite "
        if inActive:
            sqlJ += "FROM ATLAS_PANDA.jobsActive4 WHERE PandaID=:PandaID "
        else:
            sqlJ += "FROM ATLAS_PANDA.jobsArchived4 WHERE PandaID=:PandaID "

        # sql to update HS06sec
        if inActive:
            sqlU  = "UPDATE ATLAS_PANDA.jobsActive4 "
        else:
            sqlU  = "UPDATE ATLAS_PANDA.jobsArchived4 "
        sqlU += "SET hs06sec=:hs06sec WHERE PandaID=:PandaID "

        # get job attributes
        varMap = {}
        varMap[':PandaID'] = pandaID
        self.cur.execute(sqlJ+comment,varMap)
        resJ = self.cur.fetchone()
        if resJ is None:
            tmpLog.debug('skip since job not found')
        else:
            jediTaskID, startTime, endTime, actualCoreCount, defCoreCount, jobMetrics, computingSite = resJ
            # get corePower
            corePower, tmpMsg = self.get_core_power(computingSite)
            if corePower is None:
                tmpLog.debug('skip since corePower is undefined for site={0}'.format(computingSite))
            else:
                # get core count
                coreCount = JobUtils.getCoreCount(actualCoreCount, defCoreCount, jobMetrics)
                # get HS06sec
                hs06sec = JobUtils.getHS06sec(startTime, endTime, corePower, coreCount)
                if hs06sec is None:
                    tmpLog.debug('skip since HS06sec is None')
                else:
                    # cap
                    hs06sec = long(hs06sec)
                    maxHS06sec = 999999999
                    if hs06sec > maxHS06sec:
                        hs06sec = maxHS06sec
                    # update HS06sec
                    varMap = {}
                    varMap[':PandaID'] = pandaID
                    varMap[':hs06sec'] = hs06sec
                    self.cur.execute(sqlU + comment, varMap)
                    tmpLog.debug('set HS06sec={0}'.format(hs06sec))
        # return
        return hs06sec


    # check if all events are done
    def checkAllEventsDone(self,job,pandaID,useCommit=False,dumpLog=True,getProcStatus=False):
        comment = ' /* DBProxy.checkAllEventsDone */'
        if job is not None:
            pandaID = job.PandaID
        methodName = comment.split(' ')[-2].split('.')[-1]
        tmpLog = LogWrapper(_logger,methodName+" <PandaID={0}>".format(pandaID))
        if dumpLog:
            tmpLog.debug("start")
        try:
            # get files
            sqlF  = "SELECT type,jediTaskID,datasetID,fileID FROM {0}.filesTable4 ".format(panda_config.schemaPANDA)
            sqlF += "WHERE PandaID=:PandaID AND type=:type "
            # check if all events are done
            sqlEOC  = "SELECT /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
            sqlEOC += "distinct PandaID,status FROM {0}.JEDI_Events tab ".format(panda_config.schemaJEDI)
            sqlEOC += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
            sqlEOC += "AND NOT status IN (:esDone,:esDiscarded,:esCancelled,:esFatal,:esCorrupted,:esFailed,:esFinished) "
            sqlEOC += "AND NOT (status=:esReady AND attemptNr=0) "
            # get jumbo jobs
            sqlGJ  = "SELECT /*+ INDEX_RS_ASC(tab JEDI_EVENTS_FILEID_IDX) NO_INDEX_FFS(tab JEDI_EVENTS_PK) NO_INDEX_SS(tab JEDI_EVENTS_PK) */ "
            sqlGJ += "distinct PandaID FROM {0}.JEDI_Events tab ".format(panda_config.schemaJEDI)
            sqlGJ += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
            sqlGJ += "AND status IN (:esRunning,:esSent,:esFinished,:esDone) "
            # check if job is still alive
            sqlJAL  = "SELECT jobStatus,eventService FROM {0}.jobsActive4 ".format(panda_config.schemaPANDA)
            sqlJAL += "WHERE PandaID=:PandaID "
            # begin transaction
            if useCommit:
                self.conn.begin()
            self.cur.arraysize = 1000000
            # get files if needed
            if job is not None:
                fileList = job.Files
            else:
                varMap = {}
                varMap[':PandaID'] = pandaID
                varMap[':type'] = 'input'
                self.cur.execute(sqlF+comment, varMap)
                resF = self.cur.fetchall()
                fileList = []
                for tmpType,tmpJediTaskID,tmpDatasetID,tmpFileID in resF:
                    fileSpec = FileSpec()
                    fileSpec.type = tmpType
                    fileSpec.jediTaskID = tmpJediTaskID
                    fileSpec.datasetID = tmpDatasetID
                    fileSpec.fileID = tmpFileID
                    fileList.append(fileSpec)
            # check all inputs
            allDone = True
            proc_status = None
            checkedPandaIDs = set()
            jobStatusMap = dict()
            for fileSpec in fileList:
                if fileSpec.type == 'input':
                    varMap = {}
                    varMap[':jediTaskID']  = fileSpec.jediTaskID
                    varMap[':datasetID']   = fileSpec.datasetID
                    varMap[':fileID']      = fileSpec.fileID
                    varMap[':esDone']      = EventServiceUtils.ST_done
                    varMap[':esFinished']  = EventServiceUtils.ST_finished
                    varMap[':esDiscarded'] = EventServiceUtils.ST_discarded
                    varMap[':esCancelled'] = EventServiceUtils.ST_cancelled
                    varMap[':esCorrupted'] = EventServiceUtils.ST_corrupted
                    varMap[':esFatal']     = EventServiceUtils.ST_fatal
                    varMap[':esFailed']    = EventServiceUtils.ST_failed
                    varMap[':esReady']     = EventServiceUtils.ST_ready
                    self.cur.execute(sqlEOC+comment, varMap)
                    resEOC = self.cur.fetchall()
                    for pandaID,esStatus in resEOC:
                        # skip redundant lookup
                        if pandaID in checkedPandaIDs:
                            continue
                        checkedPandaIDs.add(pandaID)
                        # not yet dispatched
                        if esStatus == EventServiceUtils.ST_ready:
                            tmpStr  = "some events are not yet dispatched "
                            tmpStr += "for jediTaskID={0} datasetID={1} fileID={2}".format(fileSpec.jediTaskID,
                                                                                           fileSpec.datasetID,
                                                                                           fileSpec.fileID)
                            if dumpLog:
                                tmpLog.debug(tmpStr)
                            allDone = False
                            break
                        # check job
                        varMap = {}
                        varMap[':PandaID'] = pandaID
                        self.cur.execute(sqlJAL+comment, varMap)
                        resJAL = self.cur.fetchone()
                        if resJAL is None:
                            # no active job
                            tmpStr  = "no assiciated job is in active "
                            tmpStr += "for jediTaskID={0} datasetID={1} fileID={2}".format(fileSpec.jediTaskID,
                                                                                           fileSpec.datasetID,
                                                                                           fileSpec.fileID)
                            if dumpLog:
                                tmpLog.debug(tmpStr)
                            jobStatusMap[pandaID] = None
                        else:
                            # still active
                            tmpStr  = "PandaID={0} is associated in {1} ".format(pandaID,resJAL[0])
                            tmpStr += "for jediTaskID={0} datasetID={1} fileID={2}".format(fileSpec.jediTaskID,
                                                                                           fileSpec.datasetID,
                                                                                           fileSpec.fileID)
                            if dumpLog:
                                tmpLog.debug(tmpStr)
                            allDone = False
                            if resJAL[1] == EventServiceUtils.jumboJobFlagNumber:
                                jobStatusMap[pandaID] = resJAL[0]
                            else:
                                jobStatusMap[pandaID] = None
                            break
                        # escape
                        if not allDone:
                            break
                # escape
                if not allDone:
                    break
            # get proc_status
            if not allDone and getProcStatus:
                proc_status = 'queued'
                to_escape = False
                is_starting = False
                for fileSpec in fileList:
                    if fileSpec.type == 'input':
                        varMap = {}
                        varMap[':jediTaskID']  = fileSpec.jediTaskID
                        varMap[':datasetID']   = fileSpec.datasetID
                        varMap[':fileID']      = fileSpec.fileID
                        varMap[':esDone']      = EventServiceUtils.ST_done
                        varMap[':esFinished']  = EventServiceUtils.ST_finished
                        varMap[':esRunning']   = EventServiceUtils.ST_running
                        varMap[':esSent']   = EventServiceUtils.ST_sent
                        self.cur.execute(sqlGJ+comment, varMap)
                        resGJ = self.cur.fetchall()
                        for pandaID, in resGJ:
                            if pandaID not in jobStatusMap:
                                # get job
                                varMap = {}
                                varMap[':PandaID'] = pandaID
                                self.cur.execute(sqlJAL+comment, varMap)
                                resJAL = self.cur.fetchone()
                                if resJAL is None:
                                    jobStatusMap[pandaID] = None
                                else:
                                    if resJAL[1] == EventServiceUtils.jumboJobFlagNumber:
                                        jobStatusMap[pandaID] = resJAL[0]
                                    else:
                                        jobStatusMap[pandaID] = None
                            # check status
                            if jobStatusMap[pandaID] == 'running':
                                proc_status = 'running'
                                to_escape = True
                                break
                            elif jobStatusMap[pandaID] == 'starting':
                                is_starting = True
                        if to_escape:
                            break
                if proc_status == 'queued' and is_starting:
                    proc_status = 'starting'
            # commit
            if useCommit:
                if not self._commit():
                    raise RuntimeError('Commit error')
            if dumpLog:
                tmpLog.debug("done with {0} {1}".format(allDone, proc_status))
            if getProcStatus:
                return (allDone, proc_status)
            return allDone
        except Exception:
            # roll back
            if useCommit:
                self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            if getProcStatus:
                return (None, None)
            return None



    # get co-jumbo jobs to be finished
    def getCoJumboJobsToBeFinished(self, timeLimit, minPriority, maxJobs):
        comment = ' /* DBProxy.getCoJumboJobsToBeFinished */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        tmpLog = LogWrapper(_logger,methodName)
        tmpLog.debug("start for minPriority={0} timeLimit={1}".format(minPriority, timeLimit))
        try:
            # get co-jumbo jobs
            sqlEOD  = "SELECT PandaID,jediTaskID,jobStatus,computingSite,creationTime FROM ATLAS_PANDA.{0} "
            sqlEOD += "WHERE eventService=:eventService "
            sqlEOD += "AND (prodDBUpdateTime IS NULL OR prodDBUpdateTime<:timeLimit) "
            sqlEOD += "AND currentPriority>=:minPriority "
            # lock job
            sqlPL  = "SELECT 1 FROM ATLAS_PANDA.{0} "
            sqlPL += "WHERE PandaID=:PandaID "
            sqlPL += "AND (prodDBUpdateTime IS NULL OR prodDBUpdateTime<:timeLimit) "
            sqlPL += "FOR UPDATE NOWAIT "
            sqlLK  = "UPDATE ATLAS_PANDA.{0} "
            sqlLK += "SET prodDBUpdateTime=CURRENT_DATE "
            sqlLK += "WHERE PandaID=:PandaID "
            sqlLK += "AND (prodDBUpdateTime IS NULL OR prodDBUpdateTime<:timeLimit) "
            # get useJumbo
            sqlJM = "SELECT useJumbo FROM {0}.JEDI_Tasks ".format(panda_config.schemaJEDI)
            sqlJM += "WHERE jediTaskID=:jediTaskID "
            # get datasetID and fileID of the primary input
            sqlID = "SELECT f.datasetID,f.fileID,c.status,c.proc_status FROM {0}.JEDI_Datasets d,{0}.JEDI_Dataset_Contents c,{1}.filesTable4 f ".format(panda_config.schemaJEDI, panda_config.schemaPANDA)
            sqlID += "WHERE d.jediTaskID=:jediTaskID AND d.type IN (:t1,:t2) AND d.masterID IS NULL "
            sqlID += "AND f.jediTaskID=d.jediTaskID AND f.datasetID=d.datasetID AND f.PandaID=:PandaID "
            sqlID += "AND c.jediTaskID=d.jediTaskID AND c.datasetID=d.datasetID AND c.fileID=f.fileID "
            # get PandaIDs
            sqlCP = "SELECT PandaID FROM ATLAS_PANDA.filesTable4 "
            sqlCP += "WHERE jediTaskID=:jediTaskID AND datasetID=:datasetID AND fileID=:fileID "
            # check jobs
            sqlWP = "SELECT 1 FROM ATLAS_PANDA.jobsDefined4 WHERE PandaID=:PandaID "
            sqlWP += "UNION "
            sqlWP += "SELECT 1 FROM ATLAS_PANDA.jobsActive4 WHERE PandaID=:PandaID "
            self.cur.arraysize = 1000000
            timeLimit = datetime.datetime.utcnow() - datetime.timedelta(minutes=timeLimit)
            timeLimitWaiting = datetime.datetime.utcnow() - datetime.timedelta(hours=6)
            retList = []
            # get jobs
            coJumboTobeKilled = set()
            useJumbos = dict()
            for tableName in ['jobsActive4','jobsDefined4','jobsWaiting4']:
                self.conn.begin()
                varMap = {}
                varMap[':eventService']  = EventServiceUtils.coJumboJobFlagNumber
                varMap[':timeLimit']     = timeLimit
                varMap[':minPriority']   = minPriority
                self.cur.execute(sqlEOD.format(tableName)+comment, varMap)
                tmpRes = self.cur.fetchall()
                if not self._commit():
                    raise RuntimeError('Commit error')
                tmpLog.debug('checking {0} co-jumbo jobs in {1}'.format(len(tmpRes),tableName))
                checkedPandaIDs = set()
                iJobs = 0
                # scan all jobs
                for pandaID, jediTaskID, jobStatus, computingSite, creationTime in tmpRes:
                    # lock job
                    self.conn.begin()
                    varMap = {}
                    varMap[':PandaID'] = pandaID
                    varMap[':timeLimit'] = timeLimit
                    toSkip = False
                    resPL = None
                    try:
                        # lock with NOWAIT
                        self.cur.execute(sqlPL.format(tableName) + comment, varMap)
                        resPL = self.cur.fetchone()
                    except Exception:
                        toSkip = True
                    if resPL is None:
                        toSkip = True
                    if toSkip:
                        tmpLog.debug('skipped PandaID={0} jediTaskID={1} in {2} since locked by another'.format(pandaID,jediTaskID,tableName))
                    else:
                        # lock
                        self.cur.execute(sqlLK.format(tableName)+comment, varMap)
                        nRow = self.cur.rowcount
                        if nRow > 0:
                            iJobs += 1
                            # check if all events are done
                            allDone, proc_status = self.checkAllEventsDone(None, pandaID, False, True, True)
                            if allDone is True:
                                tmpLog.debug('locked co-jumbo PandaID={0} jediTaskID={1} to finish in {2}'.format(pandaID,jediTaskID,tableName))
                                checkedPandaIDs.add(pandaID)
                            elif jobStatus == 'waiting' and computingSite == EventServiceUtils.siteIdForWaitingCoJumboJobs and \
                                    proc_status == 'queued':
                                # check if jumbo is disabled
                                if jediTaskID not in useJumbos:
                                    varMap = {}
                                    varMap[':jediTaskID'] = jediTaskID
                                    self.cur.execute(sqlJM+comment, varMap)
                                    resJM = self.cur.fetchone()
                                    useJumbos[jediTaskID], = resJM
                                if useJumbos[jediTaskID] == 'D' or creationTime < timeLimitWaiting:
                                    # get info of the primary input
                                    varMap = {}
                                    varMap[':jediTaskID'] = jediTaskID
                                    varMap[':PandaID'] = pandaID
                                    varMap[':t1'] = 'input'
                                    varMap[':t2'] = 'pseudo_input'
                                    self.cur.execute(sqlID+comment, varMap)
                                    resID = self.cur.fetchone()
                                    datasetID, fileID, fileStatus, fileProcStatus = resID
                                    if fileStatus == 'running' and fileProcStatus == 'queued':
                                        # count # of active consumers
                                        nAct = 0
                                        varMap = {}
                                        varMap[':jediTaskID'] = jediTaskID
                                        varMap[':datasetID']  = datasetID
                                        varMap[':fileID']     = fileID
                                        self.cur.execute(sqlCP+comment, varMap)
                                        resCP = self.cur.fetchall()
                                        for tmpPandaID, in resCP:
                                            varMap = {}
                                            varMap[':PandaID'] = tmpPandaID
                                            self.cur.execute(sqlWP+comment, varMap)
                                            resWP = self.cur.fetchone()
                                            if resWP is not None:
                                                nAct += 1
                                        if nAct > 0:
                                            tmpLog.debug('skip to kill PandaID={0} jediTaskID={1} due to {2} active consumers'.format(pandaID, jediTaskID, nAct))
                                        else:
                                            tmpLog.debug('locked co-jumbo PandaID={0} jediTaskID={1} to kill'.format(pandaID, jediTaskID))
                                            coJumboTobeKilled.add(pandaID)
                            if proc_status is not None:
                                self.updateInputStatusJedi(jediTaskID, pandaID, 'queued', checkOthers=True)
                    if not self._commit():
                        raise RuntimeError('Commit error')
                    if iJobs >= maxJobs:
                        break
                retList.append(checkedPandaIDs)
            totJobs = 0
            for tmpList in retList:
                totJobs += len(tmpList)
            tmpLog.debug("got {0} jobs to finish and {1} co-jumbo jobs to kill".format(totJobs, len(coJumboTobeKilled)))
            retList.append(coJumboTobeKilled)
            return retList
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            return None



    # check if there are done events
    def hasDoneEvents(self, jediTaskID, pandaID, jobSpec, useCommit=True):
        comment = ' /* DBProxy.hasDoneEvents */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        tmpLog = LogWrapper(_logger,methodName+" <PandaID={0} jediTaskID={1}>".format(pandaID,jediTaskID))
        tmpLog.debug("start")
        retVal = False
        try:
            # sql to release events
            sqlR  = "UPDATE {0}.JEDI_Events ".format(panda_config.schemaJEDI)
            if jobSpec.decAttOnFailedES():
                sqlR += "SET status=:newStatus,pandaID=event_offset,is_jumbo=NULL "
            else:
                sqlR += "SET status=:newStatus,attemptNr=attemptNr-1,pandaID=event_offset,is_jumbo=NULL "
            sqlR += "WHERE jediTaskID=:jediTaskID AND pandaID=:pandaID AND status IN (:esSent,:esRunning) "
            # sql to check event
            sqlF  = "SELECT COUNT(*) FROM {0}.JEDI_Events ".format(panda_config.schemaJEDI)
            sqlF += "WHERE jediTaskID=:jediTaskID AND PandaID=:pandaID AND status IN (:esDone,:esFinished) "
            # begin transaction
            if useCommit:
                self.conn.begin()
            # release events
            varMap = {}
            varMap[':pandaID'] = pandaID
            varMap[':jediTaskID'] = jediTaskID
            varMap[':esSent']      = EventServiceUtils.ST_sent
            varMap[':esRunning']   = EventServiceUtils.ST_running
            varMap[':newStatus']   = EventServiceUtils.ST_ready
            self.cur.execute(sqlR+comment, varMap)
            resR = self.cur.rowcount
            tmpLog.debug("released {0} event ranges".format(resR))
            # check event
            varMap = {}
            varMap[':pandaID'] = pandaID
            varMap[':jediTaskID'] = jediTaskID
            varMap[':esDone']      = EventServiceUtils.ST_done
            varMap[':esFinished']  = EventServiceUtils.ST_finished
            self.cur.execute(sqlF+comment, varMap)
            resF = self.cur.fetchone()
            # commit
            if useCommit:
                if not self._commit():
                    raise RuntimeError('Commit error')
            nFinished = 0
            if resF is not None:
                nFinished, = resF
            if nFinished > 0:
                retVal = True
            else:
                retVal = False
            tmpLog.debug("finished {0} event ranges. ret={1}".format(nFinished, retVal))
            return retVal
        except Exception:
            # roll back
            if useCommit:
                self._rollback()
            # error
            self.dumpErrorMessage(tmpLog,methodName)
            return retVal



    # check if there are events to be processed
    def hasReadyEvents(self, jediTaskID):
        comment = ' /* DBProxy.hasReadyEvents */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        tmpLog = LogWrapper(_logger,methodName+" <jediTaskID={0}>".format(jediTaskID))
        tmpLog.debug("start")
        retVal = None
        try:
            # sql to check event
            sqlF  = "SELECT COUNT(*) FROM {0}.JEDI_Events ".format(panda_config.schemaJEDI)
            sqlF += "WHERE jediTaskID=:jediTaskID AND status=:esReady AND attemptNr>:minAttemptNr "
            # check event
            varMap = {}
            varMap[':jediTaskID'] = jediTaskID
            varMap[':esReady'] = EventServiceUtils.ST_ready
            varMap[':minAttemptNr'] = 0
            # begin transaction
            self.conn.begin()
            self.cur.execute(sqlF+comment, varMap)
            resF = self.cur.fetchone()
            nReady = None
            if resF is not None:
                nReady, = resF
                retVal = (nReady > 0)
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            tmpLog.debug("{0} ready events. ret={1}".format(nReady, retVal))
            return retVal
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(tmpLog,methodName)
            return None



    # get number of events to be processed
    def getNumReadyEvents(self, jediTaskID):
        comment = ' /* DBProxy.getNumReadyEvents */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        tmpLog = LogWrapper(_logger,methodName+" <jediTaskID={0}>".format(jediTaskID))
        tmpLog.debug("start")
        nReady = None
        try:
            # sql to count event
            sqlF  = "SELECT COUNT(*) FROM {0}.JEDI_Events ".format(panda_config.schemaJEDI)
            sqlF += "WHERE jediTaskID=:jediTaskID AND status=:esReady AND attemptNr>:minAttemptNr "
            # count event
            varMap = {}
            varMap[':jediTaskID'] = jediTaskID
            varMap[':esReady'] = EventServiceUtils.ST_ready
            varMap[':minAttemptNr'] = 0
            # begin transaction
            self.conn.begin()
            self.cur.execute(sqlF+comment, varMap)
            resF = self.cur.fetchone()
            nReady = None
            if resF is not None:
                nReady, = resF
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            tmpLog.debug("{0} ready events".format(nReady))
            return nReady
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(tmpLog,methodName)
            return None



    # check if task is applicable for jumbo jobs
    def isApplicableTaskForJumbo(self,jediTaskID):
        comment = ' /* DBProxy.isApplicableTaskForJumbo */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        tmpLog = LogWrapper(_logger,methodName+" <jediTaskID={0}>".format(jediTaskID))
        tmpLog.debug("start")
        retVal = True
        try:
            # sql to check event
            sqlF  = "SELECT SUM(nFiles),SUM(nFilesFinished),SUM(nFilesFailed) "
            sqlF += "FROM {0}.JEDI_Datasets ".format(panda_config.schemaJEDI)
            sqlF += "WHERE jediTaskID=:jediTaskID AND type IN (:type1,:type2) "
            sqlF += "AND masterID IS NULL "
            # begin transaction
            self.conn.begin()
            # check task status
            if not self.checkTaskStatusJEDI(jediTaskID,self.cur):
                # task is in a final status
                retVal = False
            else:
                # threshold in % to stop jumbo jobs
                threshold = 100
                # check percentage
                varMap = {}
                varMap[':jediTaskID'] = jediTaskID
                varMap[':type1'] = 'input'
                varMap[':type2'] = 'pseudo_input'
                self.cur.execute(sqlF+comment, varMap)
                resF = self.cur.fetchone()
                nFiles,nFilesFinished,nFilesFailed = resF
                if (nFilesFinished+nFilesFailed)*100 >= nFiles*threshold:
                    retVal = False
                    tmpLog.debug("nFilesFinished({0}) + nFilesFailed({1}) >= nFiles({2})*{3}%".format(nFilesFinished,
                                                                                                      nFilesFailed,
                                                                                                      nFiles,
                                                                                                      threshold))
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            tmpLog.debug("done with {0}".format(retVal))
            return retVal
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            return retVal



    # cleanup jumbo jobs
    def cleanupJumboJobs(self,jediTaskID=None):
        comment = ' /* DBProxy.cleanupJumboJobs */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        tmpLog = LogWrapper(_logger,methodName+" <jediTaskID={0}>".format(jediTaskID))
        tmpLog.debug("start")
        try:
            # sql to get jumbo jobs
            sql  = "SELECT PandaID,jediTaskID,jobStatus FROM ATLAS_PANDA.jobsDefined4 WHERE eventService=:eventService "
            if jediTaskID is not None:
                sql += "AND jediTaskID=:jediTaskID "
            sql += "UNION "
            sql += "SELECT PandaID,jediTaskID,jobStatus FROM ATLAS_PANDA.jobsActive4 WHERE eventService=:eventService "
            if jediTaskID is not None:
                sql += "AND jediTaskID=:jediTaskID "
            sql += "UNION "
            sql += "SELECT PandaID,jediTaskID,jobStatus FROM ATLAS_PANDA.jobsWaiting4 WHERE eventService=:eventService "
            if jediTaskID is not None:
                sql += "AND jediTaskID=:jediTaskID "
            # begin transaction
            self.conn.begin()
            # get jobs
            varMap = {}
            varMap[':eventService'] = EventServiceUtils.jumboJobFlagNumber
            self.cur.execute(sql+comment, varMap)
            resF = self.cur.fetchall()
            # commit
            if not self._commit():
                raise RuntimeError('Commit error')
            # get ID mapping
            idMap = {}
            for pandaID, tmpJediTaskID, jobStatus in resF:
                if jobStatus in ['transferring', 'running', 'holding']:
                    continue
                if tmpJediTaskID not in idMap:
                    idMap[tmpJediTaskID] = set()
                idMap[tmpJediTaskID].add(pandaID)
            tmpLog.debug("got {0} taks".format(len(idMap)))
            # sql to check useJumbo
            sqlJ = "SELECT useJumbo FROM ATLAS_PANDA.JEDI_Tasks WHERE jediTaskID=:jediTaskID "
            # loop over all tasks
            for tmpJediTaskID in idMap:
                pandaIDs = idMap[tmpJediTaskID]
                # check useJumbo
                self.conn.begin()
                varMap = {}
                varMap[':jediTaskID'] = tmpJediTaskID
                self.cur.execute(sqlJ+comment, varMap)
                resJ = self.cur.fetchone()
                if resJ is not None and resJ[0] == 'D':
                    disabledFlag = True
                    tmpLog.debug("kill disabled jumbo jobs for jediTaskID={0}".format(tmpJediTaskID))
                else:
                    disabledFlag = False
                if not self._commit():
                    raise RuntimeError('Commit error')
                if jediTaskID is not None or not self.isApplicableTaskForJumbo(tmpJediTaskID) or disabledFlag:
                    for pandaID in pandaIDs:
                        self.killJob(pandaID,'','55',True)
                    tmpLog.debug("killed {0} jobs for jediTaskID={1}".format(len(pandaIDs),tmpJediTaskID))
            tmpLog.debug("done")
            return True
        except Exception:
            # roll back
            self._rollback()
            # error
            self.dumpErrorMessage(_logger,methodName)
            return False

    # get core power
    @memoize
    def get_core_power(self, site_id):
        comment = ' /* DBProxy.get_core_power */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        tmpLog = LogWrapper(_logger, methodName + " <siteid={}>".format(site_id))
        tmpLog.debug("start")
        
        sqlS = "SELECT /* use_json_type */ scj.data.corepower FROM ATLAS_PANDA.schedconfig_json scj "
        sqlS += "WHERE panda_queue=:siteid "

        varMap = {':siteid': site_id}

        try:
            self.cur.execute(sqlS + comment, varMap)
            resS = self.cur.fetchone()
            core_power = None
            if resS is not None:
                core_power, = resS
                core_power = float(core_power)
            tmpLog.debug('got {}'.format(core_power))
            return core_power, None
        
        except Exception:
            # error
            self.dumpErrorMessage(_logger, methodName)
            return None, 'failed to get corePower'

    # convert ObjID to endpoint
    @memoize
    def convertObjIDtoEndPoint(self,srcFileName,objID):
        comment = ' /* DBProxy.convertObjIDtoEndPoint */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        tmpLog = LogWrapper(_logger,methodName+" <ID={0}>".format(objID))
        tmpLog.debug("start")
        try:
            for srcFile in srcFileName.split(','):
                if not os.path.exists(srcFile):
                    continue
                with open(srcFile) as f:
                    data = json.load(f)
                    for rseName in data:
                        rseData = data[rseName]
                        if objID in [rseData['id'], rseName]:
                            retMap = {'name': rseName,
                                      "is_deterministic": rseData['is_deterministic'],
                                      'type': rseData['type']
                                      }
                            tmpLog.debug("got {0}".format(str(retMap)))
                            return retMap
            tmpLog.debug('not found')
        except Exception:
            # error
            self.dumpErrorMessage(_logger,methodName)
            return None



    # get OS IDs
    def getObjIDs(self,jediTaskID,pandaID):
        comment = ' /* DBProxy.getObjIDs */'
        methodName = comment.split(' ')[-2].split('.')[-1]
        tmpLog = LogWrapper(_logger,methodName+" <jediTaskID={0} PandaID={1}>".format(jedi