import * as ec2 from '@aws-cdk/aws-ec2';
import * as iam from '@aws-cdk/aws-iam';
import * as kms from '@aws-cdk/aws-kms';
import * as secretsmanager from '@aws-cdk/aws-secretsmanager';
import { Resource, Duration, RemovalPolicy, IResource } from '@aws-cdk/core';
import { Construct } from 'constructs';
import { IClusterEngine } from './cluster-engine';
import { Endpoint } from './endpoint';
import { IParameterGroup } from './parameter-group';
import { Credentials, RotationMultiUserOptions, RotationSingleUserOptions } from './props';
import { ISubnetGroup } from './subnet-group';
/**
 * Interface representing a serverless database cluster.
 *
 * @stability stable
 */
export interface IServerlessCluster extends IResource, ec2.IConnectable, secretsmanager.ISecretAttachmentTarget {
    /**
     * Identifier of the cluster.
     *
     * @stability stable
     */
    readonly clusterIdentifier: string;
    /**
     * The ARN of the cluster.
     *
     * @stability stable
     */
    readonly clusterArn: string;
    /**
     * The endpoint to use for read/write operations.
     *
     * @stability stable
     * @attribute EndpointAddress,EndpointPort
     */
    readonly clusterEndpoint: Endpoint;
    /**
     * Endpoint to use for load-balanced read-only operations.
     *
     * @stability stable
     * @attribute ReadEndpointAddress
     */
    readonly clusterReadEndpoint: Endpoint;
    /**
     * Grant the given identity to access to the Data API.
     *
     * @param grantee The principal to grant access to.
     * @stability stable
     */
    grantDataApiAccess(grantee: iam.IGrantable): iam.Grant;
}
/**
 * Properties to configure an Aurora Serverless Cluster.
 *
 * @stability stable
 */
export interface ServerlessClusterProps {
    /**
     * What kind of database to start.
     *
     * @stability stable
     */
    readonly engine: IClusterEngine;
    /**
     * Credentials for the administrative user.
     *
     * @default - A username of 'admin' and SecretsManager-generated password
     * @stability stable
     */
    readonly credentials?: Credentials;
    /**
     * An optional identifier for the cluster.
     *
     * @default - A name is automatically generated.
     * @stability stable
     */
    readonly clusterIdentifier?: string;
    /**
     * The number of days during which automatic DB snapshots are retained.
     *
     * Automatic backup retention cannot be disabled on serverless clusters.
     * Must be a value from 1 day to 35 days.
     *
     * @default Duration.days(1)
     * @stability stable
     */
    readonly backupRetention?: Duration;
    /**
     * Name of a database which is automatically created inside the cluster.
     *
     * @default - Database is not created in cluster.
     * @stability stable
     */
    readonly defaultDatabaseName?: string;
    /**
     * Indicates whether the DB cluster should have deletion protection enabled.
     *
     * @default - true if removalPolicy is RETAIN, false otherwise
     * @stability stable
     */
    readonly deletionProtection?: boolean;
    /**
     * Whether to enable the Data API.
     *
     * @default false
     * @see https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/data-api.html
     * @stability stable
     */
    readonly enableDataApi?: boolean;
    /**
     * The VPC that this Aurora Serverless cluster has been created in.
     *
     * @stability stable
     */
    readonly vpc: ec2.IVpc;
    /**
     * Where to place the instances within the VPC.
     *
     * @default - the VPC default strategy if not specified.
     * @stability stable
     */
    readonly vpcSubnets?: ec2.SubnetSelection;
    /**
     * Scaling configuration of an Aurora Serverless database cluster.
     *
     * @default - Serverless cluster is automatically paused after 5 minutes of being idle.
     * minimum capacity: 2 ACU
     * maximum capacity: 16 ACU
     * @stability stable
     */
    readonly scaling?: ServerlessScalingOptions;
    /**
     * The removal policy to apply when the cluster and its instances are removed from the stack or replaced during an update.
     *
     * @default - RemovalPolicy.SNAPSHOT (remove the cluster and instances, but retain a snapshot of the data)
     * @stability stable
     */
    readonly removalPolicy?: RemovalPolicy;
    /**
     * Security group.
     *
     * @default - a new security group is created.
     * @stability stable
     */
    readonly securityGroups?: ec2.ISecurityGroup[];
    /**
     * The KMS key for storage encryption.
     *
     * @default - the default master key will be used for storage encryption
     * @stability stable
     */
    readonly storageEncryptionKey?: kms.IKey;
    /**
     * Additional parameters to pass to the database engine.
     *
     * @default - no parameter group.
     * @stability stable
     */
    readonly parameterGroup?: IParameterGroup;
    /**
     * Existing subnet group for the cluster.
     *
     * @default - a new subnet group will be created.
     * @stability stable
     */
    readonly subnetGroup?: ISubnetGroup;
}
/**
 * Properties that describe an existing cluster instance.
 *
 * @stability stable
 */
export interface ServerlessClusterAttributes {
    /**
     * Identifier for the cluster.
     *
     * @stability stable
     */
    readonly clusterIdentifier: string;
    /**
     * The database port.
     *
     * @default - none
     * @stability stable
     */
    readonly port?: number;
    /**
     * The security groups of the database cluster.
     *
     * @default - no security groups
     * @stability stable
     */
    readonly securityGroups?: ec2.ISecurityGroup[];
    /**
     * Cluster endpoint address.
     *
     * @default - no endpoint address
     * @stability stable
     */
    readonly clusterEndpointAddress?: string;
    /**
     * Reader endpoint address.
     *
     * @default - no reader address
     * @stability stable
     */
    readonly readerEndpointAddress?: string;
    /**
     * The secret attached to the database cluster.
     *
     * @default - no secret
     * @stability stable
     */
    readonly secret?: secretsmanager.ISecret;
}
/**
 * Aurora capacity units (ACUs).
 *
 * Each ACU is a combination of processing and memory capacity.
 *
 * @see https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.how-it-works.html#aurora-serverless.architecture
 * @stability stable
 */
export declare enum AuroraCapacityUnit {
    /**
     * 1 Aurora Capacity Unit.
     *
     * @stability stable
     */
    ACU_1 = 1,
    /**
     * 2 Aurora Capacity Units.
     *
     * @stability stable
     */
    ACU_2 = 2,
    /**
     * 4 Aurora Capacity Units.
     *
     * @stability stable
     */
    ACU_4 = 4,
    /**
     * 8 Aurora Capacity Units.
     *
     * @stability stable
     */
    ACU_8 = 8,
    /**
     * 16 Aurora Capacity Units.
     *
     * @stability stable
     */
    ACU_16 = 16,
    /**
     * 32 Aurora Capacity Units.
     *
     * @stability stable
     */
    ACU_32 = 32,
    /**
     * 64 Aurora Capacity Units.
     *
     * @stability stable
     */
    ACU_64 = 64,
    /**
     * 128 Aurora Capacity Units.
     *
     * @stability stable
     */
    ACU_128 = 128,
    /**
     * 192 Aurora Capacity Units.
     *
     * @stability stable
     */
    ACU_192 = 192,
    /**
     * 256 Aurora Capacity Units.
     *
     * @stability stable
     */
    ACU_256 = 256,
    /**
     * 384 Aurora Capacity Units.
     *
     * @stability stable
     */
    ACU_384 = 384
}
/**
 * Options for configuring scaling on an Aurora Serverless cluster.
 *
 * @stability stable
 */
export interface ServerlessScalingOptions {
    /**
     * The minimum capacity for an Aurora Serverless database cluster.
     *
     * @default - determined by Aurora based on database engine
     * @stability stable
     */
    readonly minCapacity?: AuroraCapacityUnit;
    /**
     * The maximum capacity for an Aurora Serverless database cluster.
     *
     * @default - determined by Aurora based on database engine
     * @stability stable
     */
    readonly maxCapacity?: AuroraCapacityUnit;
    /**
     * The time before an Aurora Serverless database cluster is paused.
     *
     * A database cluster can be paused only when it is idle (it has no connections).
     * Auto pause time must be between 5 minutes and 1 day.
     *
     * If a DB cluster is paused for more than seven days, the DB cluster might be
     * backed up with a snapshot. In this case, the DB cluster is restored when there
     * is a request to connect to it.
     *
     * Set to 0 to disable
     *
     * @default - automatic pause enabled after 5 minutes
     * @stability stable
     */
    readonly autoPause?: Duration;
}
/**
 * New or imported Serverless Cluster
 */
declare abstract class ServerlessClusterBase extends Resource implements IServerlessCluster {
    /**
     * Identifier of the cluster
     */
    abstract readonly clusterIdentifier: string;
    /**
     * The endpoint to use for read/write operations
     */
    abstract readonly clusterEndpoint: Endpoint;
    /**
     * The endpoint to use for read/write operations
     */
    abstract readonly clusterReadEndpoint: Endpoint;
    /**
     * Access to the network connections
     */
    abstract readonly connections: ec2.Connections;
    /**
     * The secret attached to this cluster
     */
    abstract readonly secret?: secretsmanager.ISecret;
    protected abstract enableDataApi?: boolean;
    /**
     * The ARN of the cluster.
     *
     * @stability stable
     */
    get clusterArn(): string;
    /**
     * Grant the given identity to access to the Data API, including read access to the secret attached to the cluster if present.
     *
     * @param grantee The principal to grant access to.
     * @stability stable
     */
    grantDataApiAccess(grantee: iam.IGrantable): iam.Grant;
    /**
     * Renders the secret attachment target specifications.
     *
     * @stability stable
     */
    asSecretAttachmentTarget(): secretsmanager.SecretAttachmentTargetProps;
}
/**
 * Create an Aurora Serverless Cluster.
 *
 * @stability stable
 * @resource AWS::RDS::DBCluster
 */
export declare class ServerlessCluster extends ServerlessClusterBase {
    /**
     * Import an existing DatabaseCluster from properties.
     *
     * @stability stable
     */
    static fromServerlessClusterAttributes(scope: Construct, id: string, attrs: ServerlessClusterAttributes): IServerlessCluster;
    /**
     * Identifier of the cluster.
     *
     * @stability stable
     */
    readonly clusterIdentifier: string;
    /**
     * The endpoint to use for read/write operations.
     *
     * @stability stable
     */
    readonly clusterEndpoint: Endpoint;
    /**
     * The endpoint to use for read/write operations.
     *
     * @stability stable
     */
    readonly clusterReadEndpoint: Endpoint;
    /**
     * Access to the network connections.
     *
     * @stability stable
     */
    readonly connections: ec2.Connections;
    /**
     * The secret attached to this cluster.
     *
     * @stability stable
     */
    readonly secret?: secretsmanager.ISecret;
    /**
     * @stability stable
     */
    protected enableDataApi?: boolean;
    private readonly subnetGroup;
    private readonly vpc;
    private readonly vpcSubnets?;
    private readonly singleUserRotationApplication;
    private readonly multiUserRotationApplication;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: ServerlessClusterProps);
    /**
     * Adds the single user rotation of the master password to this cluster.
     *
     * @stability stable
     */
    addRotationSingleUser(options?: RotationSingleUserOptions): secretsmanager.SecretRotation;
    /**
     * Adds the multi user rotation to this cluster.
     *
     * @stability stable
     */
    addRotationMultiUser(id: string, options: RotationMultiUserOptions): secretsmanager.SecretRotation;
    private renderScalingConfiguration;
}
export {};
