"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const targets = require("../../lib");
describe('CodePipeline event target', () => {
    let stack;
    let pipeline;
    let pipelineArn;
    beforeEach(() => {
        stack = new core_1.Stack();
        pipeline = new codepipeline.Pipeline(stack, 'Pipeline');
        const srcArtifact = new codepipeline.Artifact('Src');
        const buildArtifact = new codepipeline.Artifact('Bld');
        pipeline.addStage({
            stageName: 'Source',
            actions: [new TestAction({
                    actionName: 'Hello',
                    category: codepipeline.ActionCategory.SOURCE,
                    provider: 'x',
                    artifactBounds: { minInputs: 0, maxInputs: 0, minOutputs: 1, maxOutputs: 1, },
                    outputs: [srcArtifact]
                })]
        });
        pipeline.addStage({
            stageName: 'Build',
            actions: [new TestAction({
                    actionName: 'Hello',
                    category: codepipeline.ActionCategory.BUILD,
                    provider: 'y',
                    inputs: [srcArtifact],
                    outputs: [buildArtifact],
                    artifactBounds: { minInputs: 1, maxInputs: 1, minOutputs: 1, maxOutputs: 1, }
                })]
        });
        pipelineArn = {
            'Fn::Join': ['', [
                    'arn:',
                    { Ref: 'AWS::Partition' },
                    ':codepipeline:',
                    { Ref: 'AWS::Region' },
                    ':',
                    { Ref: 'AWS::AccountId' },
                    ':',
                    { Ref: 'PipelineC660917D' }
                ]
            ]
        };
    });
    describe('when added to an event rule as a target', () => {
        let rule;
        beforeEach(() => {
            rule = new events.Rule(stack, 'rule', {
                schedule: events.Schedule.expression('rate(1 minute)'),
            });
        });
        describe('with default settings', () => {
            beforeEach(() => {
                rule.addTarget(new targets.CodePipeline(pipeline));
            });
            test("adds the pipeline's ARN and role to the targets of the rule", () => {
                assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
                    Targets: [
                        {
                            Arn: pipelineArn,
                            Id: 'Target0',
                            RoleArn: { 'Fn::GetAtt': ['PipelineEventsRole46BEEA7C', 'Arn'] },
                        },
                    ],
                }));
            });
            test("creates a policy that has StartPipeline permissions on the pipeline's ARN", () => {
                assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                    PolicyDocument: {
                        Statement: [
                            {
                                Action: 'codepipeline:StartPipelineExecution',
                                Effect: 'Allow',
                                Resource: pipelineArn,
                            }
                        ],
                        Version: '2012-10-17'
                    }
                }));
            });
        });
        describe('with an explicit event role', () => {
            beforeEach(() => {
                const role = new iam.Role(stack, 'MyExampleRole', {
                    assumedBy: new iam.AnyPrincipal(),
                });
                const roleResource = role.node.defaultChild;
                roleResource.overrideLogicalId('MyRole'); // to make it deterministic in the assertion below
                rule.addTarget(new targets.CodePipeline(pipeline, {
                    eventRole: role,
                }));
            });
            test("points at the given event role in the rule's targets", () => {
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                    Targets: [
                        {
                            Arn: pipelineArn,
                            RoleArn: { 'Fn::GetAtt': ['MyRole', 'Arn'] },
                        },
                    ],
                }));
            });
        });
    });
});
class TestAction {
    constructor(actionProperties) {
        this.actionProperties = actionProperties;
        // nothing to do
    }
    bind(_scope, _stage, _options) {
        return {};
    }
    onStateChange(_name, _target, _options) {
        throw new Error('onStateChange() is not available on MockAction');
    }
}
//# sourceMappingURL=data:application/json;base64,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