#!/usr/bin/env node
"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const codebuild = require("@aws-cdk/aws-codebuild");
const codecommit = require("@aws-cdk/aws-codecommit");
const events = require("@aws-cdk/aws-events");
const sns = require("@aws-cdk/aws-sns");
const subs = require("@aws-cdk/aws-sns-subscriptions");
const sqs = require("@aws-cdk/aws-sqs");
const cdk = require("@aws-cdk/core");
const targets = require("../../lib");
const app = new cdk.App();
const stack = new cdk.Stack(app, 'aws-cdk-codebuild-events');
const repo = new codecommit.Repository(stack, 'MyRepo', {
    repositoryName: 'aws-cdk-codebuild-events',
});
const project = new codebuild.Project(stack, 'MyProject', {
    source: codebuild.Source.codeCommit({ repository: repo }),
});
const queue = new sqs.Queue(stack, 'MyQueue');
const topic = new sns.Topic(stack, 'MyTopic');
topic.addSubscription(new subs.SqsSubscription(queue));
// this will send an email with the JSON event for every state change of this
// build project.
project.onStateChange('StateChange', { target: new targets.SnsTopic(topic) });
// this will send an email with the message "Build phase changed to <phase>".
// The phase will be extracted from the "completed-phase" field of the event
// details.
project.onPhaseChange('PhaseChange', {
    target: new targets.SnsTopic(topic, {
        message: events.RuleTargetInput.fromText(`Build phase changed to ${codebuild.PhaseChangeEvent.completedPhase}`)
    })
});
// trigger a build when a commit is pushed to the repo
const onCommitRule = repo.onCommit('OnCommit', {
    target: new targets.CodeBuildProject(project),
    branches: ['master']
});
onCommitRule.addTarget(new targets.SnsTopic(topic, {
    message: events.RuleTargetInput.fromText(`A commit was pushed to the repository ${codecommit.ReferenceEvent.repositoryName} on branch ${codecommit.ReferenceEvent.referenceName}`)
}));
app.synth();
//# sourceMappingURL=data:application/json;base64,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