"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const path = require("path");
const metadata = require("./sdk-api-metadata.json");
const util_1 = require("./util");
const awsSdkMetadata = metadata;
/**
 * Use an AWS Lambda function that makes API calls as an event rule target.
 */
class AwsApi {
    constructor(props) {
        this.props = props;
    }
    /**
     * Returns a RuleTarget that can be used to trigger this AwsApi as a
     * result from a CloudWatch event.
     */
    bind(rule, id) {
        const handler = new lambda.SingletonFunction(rule, `${rule.node.id}${id}Handler`, {
            code: lambda.Code.fromAsset(path.join(__dirname, 'aws-api-handler')),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler',
            uuid: 'b4cf1abd-4e4f-4bc6-9944-1af7ccd9ec37',
            lambdaPurpose: 'AWS',
        });
        if (this.props.policyStatement) {
            handler.addToRolePolicy(this.props.policyStatement);
        }
        else {
            handler.addToRolePolicy(new iam.PolicyStatement({
                actions: [awsSdkToIamAction(this.props.service, this.props.action)],
                resources: ['*']
            }));
        }
        // Allow handler to be called from rule
        util_1.addLambdaPermission(rule, handler);
        const input = {
            service: this.props.service,
            action: this.props.action,
            parameters: this.props.parameters,
            catchErrorPattern: this.props.catchErrorPattern,
            apiVersion: this.props.apiVersion,
        };
        return {
            id: '',
            arn: handler.functionArn,
            input: events.RuleTargetInput.fromObject(input),
            targetResource: handler,
        };
    }
}
exports.AwsApi = AwsApi;
/**
 * Transform SDK service/action to IAM action using metadata from aws-sdk module.
 */
function awsSdkToIamAction(service, action) {
    const srv = service.toLowerCase();
    const iamService = awsSdkMetadata[srv].prefix || srv;
    const iamAction = action.charAt(0).toUpperCase() + action.slice(1);
    return `${iamService}:${iamAction}`;
}
//# sourceMappingURL=data:application/json;base64,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