"""
Resource class for Vulnerabilities
"""

from __future__ import annotations

from datetime import date, datetime
from typing import Iterator, List, Optional

from weasel_client.resources.resource import Resource


class Vulnerability(Resource):
    """
    Resource class for Vulnerabilities
    """

    _cve_number: str
    _publish_date: str
    _last_modified: str
    _title: Optional[str]
    _vuln_type: Optional[str]
    _references: Optional[str]
    _fishy: bool

    def __init__(
        self,
        APIClient,
        primary_key: int,
        url: str,
        cve_number: str,
        publish_date: str,
        last_modified: str,
        fishy: bool,
        title: Optional[str] = None,
        vuln_type: Optional[str] = None,
        references: Optional[str] = None,
    ):
        super().__init__(APIClient=APIClient, primary_key=primary_key, url=url)
        self._cve_number = cve_number
        self._publish_date = publish_date
        self._last_modified = last_modified
        self._title = title
        self._vuln_type = vuln_type
        self._references = references
        self._fishy = fishy

    def cve_number(self) -> str:
        """
        Returns the CVE Number
        """
        return self._cve_number

    def publish_date(self) -> date:
        """
        Returns the publish date
        """
        return date.fromisoformat(self._publish_date)

    def last_modified(self) -> datetime:
        """
        Returns the time the vulnerability was last modified
        """
        return datetime.fromisoformat(self._last_modified)

    def title(self) -> Optional[str]:
        """
        Returns the title
        """
        return self._title

    def type(self) -> Optional[str]:
        """
        Returns the vulnerability type
        """
        return self._vuln_type

    def references(self) -> Optional[List[str]]:
        """
        # Todo
        Should return the vulnerability references
        """

    def fishy(self) -> bool:
        """
        Returns the fishy-flag
        """
        return self._fishy

    def found_in(self) -> Iterator[Resource]:
        """
        Returns the associated releases the vulnerability was found in
        """
        return self._client.vulnerability_found_in(primary_key=self._primary_key)

    def fixed_by(self) -> Iterator[Resource]:
        """
        Returns the associated releases the vulnerability was fixed by
        """
        return self._client.vulnerability_fixed_by(primary_key=self._primary_key)

    @staticmethod
    def from_id(api_client, primary_key: int) -> Optional[Vulnerability]:
        """
        Fetches a vulnerability-object with a given ID
        :param api_client: client for API-requests
        :param id: primary_key of the object to fetch
        """
        return api_client.vulnerability_detail(primary_key=primary_key)
