#!/usr/bin/env python

"""
This script takes a path to a matrix table, and then:
 - exports a sites-only VCF using Hail Query,
 - run a pipeline on Cromwell
 - Collect the results
"""

import logging
import click
import hail as hl

from joint_calling import _version
from joint_calling.utils import get_validation_callback, init_hail, file_exists
from joint_calling import utils
from joint_calling.mt_to_vcf import mt_to_sites_only_mt

logger = logging.getLogger('vqsr_qc')
logging.basicConfig(
    format='%(asctime)s (%(name)s %(lineno)s): %(message)s',
    datefmt='%m/%d/%Y %I:%M:%S %p',
)
logger.setLevel(logging.INFO)


@click.command()
@click.version_option(_version.__version__)
@click.option(
    '--mt',
    'mt_path',
    required=True,
    callback=get_validation_callback(ext='mt', must_exist=True),
    help='path to the input MatrixTable',
)
@click.option(
    '--hard-filtered-samples-ht',
    'hard_filtered_samples_ht_path',
    required=True,
    help='Path to a table with only samples that passed filters '
    '(generated by sample QC)',
)
@click.option(
    '--meta-ht',
    'meta_ht_path',
    required=True,
    help='',
)
@click.option(
    '-o',
    'output_path',
    required=True,
)
@click.option(
    '--local-tmp-dir',
    'local_tmp_dir',
    help='local directory for temporary files and Hail logs (must be local).',
)
@click.option(
    '--overwrite/--reuse',
    'overwrite',
    is_flag=True,
    help='if an intermediate or a final file exists, skip running the code '
    'that generates it.',
)
@click.option(
    '--hail-billing',
    'hail_billing',
    help='Hail billing account ID.',
)
@click.option(
    '--n-partitions',
    'n_partitions',
    type=click.INT,
    default=5000,
    help='Desired base number of partitions for the output matrix table',
)
def main(
    mt_path: str,
    hard_filtered_samples_ht_path: str,
    meta_ht_path: str,
    output_path: str,
    local_tmp_dir: str,
    overwrite: bool,
    hail_billing: str,  # pylint: disable=unused-argument
    n_partitions: int,
):
    """
    Expects hail service to already be initialised
    """
    init_hail('variant_qc', local_tmp_dir)

    logger.info(f'Loading matrix table from "{mt_path}"')
    mt = utils.get_mt(
        mt_path,
        hard_filtered_samples_to_remove_ht=hl.read_table(hard_filtered_samples_ht_path),
        meta_ht=hl.read_table(meta_ht_path),
        add_meta=True,
    )
    mt = mt.filter_cols(~mt.meta.related)

    if file_exists(output_path):
        if overwrite:
            logger.info(f'Output file {output_path} exists and will be overwritten')
        else:
            logger.info(
                f'Output file {output_path} exists, use --overwrite to overwrite'
            )
            return
    export_sites_only_vcf(mt=mt, output_path=output_path, n_partitions=n_partitions)


def export_sites_only_vcf(
    mt: hl.MatrixTable, output_path: str, n_partitions: int = 5000
):
    """
    Take initial matrix table, convert to sites-only matrix table, then export to vcf
    """
    logger.info('Converting matrix table to sites-only matrix table')
    final_mt = mt_to_sites_only_mt(mt, n_partitions)

    # export vcf, and return the path

    logger.info(
        f"Exporting sites-only VCF to '{output_path}' to run in the VQSR pipeline"
    )
    hl.export_vcf(final_mt, output_path)
    logger.info('Successfully exported sites-only VCF')

    return output_path


if __name__ == '__main__':
    main()  # pylint: disable=E1120
