"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFormationProduct = exports.Product = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const association_manager_1 = require("./private/association-manager");
const validation_1 = require("./private/validation");
const servicecatalog_generated_1 = require("./servicecatalog.generated");
class ProductBase extends core_1.Resource {
    associateTagOptions(tagOptions) {
        association_manager_1.AssociationManager.associateTagOptions(this, this.productId, tagOptions);
    }
}
/**
 * Abstract class for Service Catalog Product.
 */
class Product extends ProductBase {
    /**
     * Creates a Product construct that represents an external product.
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param productArn Product Arn
     */
    static fromProductArn(scope, id, productArn) {
        const arn = core_1.Stack.of(scope).splitArn(productArn, core_1.ArnFormat.SLASH_RESOURCE_NAME);
        const productId = arn.resourceName;
        if (!productId) {
            throw new Error('Missing required Portfolio ID from Portfolio ARN: ' + productArn);
        }
        return new class extends ProductBase {
            constructor() {
                super(...arguments);
                this.productId = productId;
                this.productArn = productArn;
            }
        }(scope, id);
    }
}
exports.Product = Product;
_a = JSII_RTTI_SYMBOL_1;
Product[_a] = { fqn: "@aws-cdk/aws-servicecatalog.Product", version: "1.168.0" };
/**
 * A Service Catalog Cloudformation Product.
 */
class CloudFormationProduct extends Product {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_servicecatalog_CloudFormationProductProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, CloudFormationProduct);
            }
            throw error;
        }
        this.validateProductProps(props);
        const product = new servicecatalog_generated_1.CfnCloudFormationProduct(this, 'Resource', {
            acceptLanguage: props.messageLanguage,
            description: props.description,
            distributor: props.distributor,
            name: props.productName,
            owner: props.owner,
            provisioningArtifactParameters: this.renderProvisioningArtifacts(props),
            replaceProvisioningArtifacts: props.replaceProductVersionIds,
            supportDescription: props.supportDescription,
            supportEmail: props.supportEmail,
            supportUrl: props.supportUrl,
        });
        this.productId = product.ref;
        this.productArn = core_1.Stack.of(this).formatArn({
            service: 'catalog',
            resource: 'product',
            resourceName: product.ref,
        });
        if (props.tagOptions !== undefined) {
            this.associateTagOptions(props.tagOptions);
        }
    }
    renderProvisioningArtifacts(props) {
        return props.productVersions.map(productVersion => {
            const template = productVersion.cloudFormationTemplate.bind(this);
            validation_1.InputValidator.validateUrl(this.node.path, 'provisioning template url', template.httpUrl);
            return {
                name: productVersion.productVersionName,
                description: productVersion.description,
                disableTemplateValidation: productVersion.validateTemplate === false ? true : false,
                info: {
                    LoadTemplateFromURL: template.httpUrl,
                },
            };
        });
    }
    ;
    validateProductProps(props) {
        validation_1.InputValidator.validateLength(this.node.path, 'product product name', 1, 100, props.productName);
        validation_1.InputValidator.validateLength(this.node.path, 'product owner', 1, 8191, props.owner);
        validation_1.InputValidator.validateLength(this.node.path, 'product description', 0, 8191, props.description);
        validation_1.InputValidator.validateLength(this.node.path, 'product distributor', 0, 8191, props.distributor);
        validation_1.InputValidator.validateEmail(this.node.path, 'support email', props.supportEmail);
        validation_1.InputValidator.validateUrl(this.node.path, 'support url', props.supportUrl);
        validation_1.InputValidator.validateLength(this.node.path, 'support description', 0, 8191, props.supportDescription);
        if (props.productVersions.length == 0) {
            throw new Error(`Invalid product versions for resource ${this.node.path}, must contain at least 1 product version`);
        }
        props.productVersions.forEach(productVersion => {
            validation_1.InputValidator.validateLength(this.node.path, 'provisioning artifact name', 0, 100, productVersion.productVersionName);
            validation_1.InputValidator.validateLength(this.node.path, 'provisioning artifact description', 0, 8191, productVersion.description);
        });
    }
}
exports.CloudFormationProduct = CloudFormationProduct;
_b = JSII_RTTI_SYMBOL_1;
CloudFormationProduct[_b] = { fqn: "@aws-cdk/aws-servicecatalog.CloudFormationProduct", version: "1.168.0" };
//# sourceMappingURL=data:application/json;base64,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