"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProductStack = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const fs = require("fs");
const path = require("path");
const cdk = require("@aws-cdk/core");
const product_stack_synthesizer_1 = require("./private/product-stack-synthesizer");
/**
 * A Service Catalog product stack, which is similar in form to a Cloudformation nested stack.
 * You can add the resources to this stack that you want to define for your service catalog product.
 *
 * This stack will not be treated as an independent deployment
 * artifact (won't be listed in "cdk list" or deployable through "cdk deploy"),
 * but rather only synthesized as a template and uploaded as an asset to S3.
 *
 */
class ProductStack extends cdk.Stack {
    constructor(scope, id) {
        super(scope, id, {
            synthesizer: new product_stack_synthesizer_1.ProductStackSynthesizer(),
        });
        this._parentStack = findParentStack(scope);
        // this is the file name of the synthesized template file within the cloud assembly
        this.templateFile = `${cdk.Names.uniqueId(this)}.product.template.json`;
    }
    /**
     * Set the parent product stack history
     *
     * @internal
     */
    _setParentProductStackHistory(parentProductStackHistory) {
        return this._parentProductStackHistory = parentProductStackHistory;
    }
    /**
     * Fetch the template URL.
     *
     * @internal
     */
    _getTemplateUrl() {
        return cdk.Lazy.uncachedString({ produce: () => this._templateUrl });
    }
    /**
     * Synthesize the product stack template, overrides the `super` class method.
     *
     * Defines an asset at the parent stack which represents the template of this
     * product stack.
     *
     * @internal
     */
    _synthesizeTemplate(session) {
        const cfn = JSON.stringify(this._toCloudFormation(), undefined, 2);
        const templateHash = crypto.createHash('sha256').update(cfn).digest('hex');
        this._templateUrl = this._parentStack.synthesizer.addFileAsset({
            packaging: cdk.FileAssetPackaging.FILE,
            sourceHash: templateHash,
            fileName: this.templateFile,
        }).httpUrl;
        if (this._parentProductStackHistory) {
            this._parentProductStackHistory._writeTemplateToSnapshot(cfn);
        }
        fs.writeFileSync(path.join(session.assembly.outdir, this.templateFile), cfn);
    }
}
exports.ProductStack = ProductStack;
_a = JSII_RTTI_SYMBOL_1;
ProductStack[_a] = { fqn: "@aws-cdk/aws-servicecatalog.ProductStack", version: "1.168.0" };
/**
 * Validates the scope for a product stack, which must be defined within the scope of another `Stack`.
 */
function findParentStack(scope) {
    try {
        const parentStack = cdk.Stack.of(scope);
        return parentStack;
    }
    catch (e) {
        throw new Error('Product stacks must be defined within scope of another non-product stack');
    }
}
//# sourceMappingURL=data:application/json;base64,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