"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BootstraplessStackSynthesizer = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const cxschema = require("aws-cdk-lib/cloud-assembly-schema");
const core_1 = require("aws-cdk-lib/core");
const cxapi = require("aws-cdk-lib/cx-api");
const REGION_PLACEHOLDER = '${AWS::Region}';
const ERR_MSG_CALL_BIND_FIRST = 'You must call bind() first';
/**
 * A Bootstrapless stack synthesizer that is designated to generate templates that can be directly used by Cloudformation.
 *
 * @stability stable
 */
class BootstraplessStackSynthesizer extends core_1.StackSynthesizer {
    /**
     * @stability stable
     */
    constructor(props = {}) {
        var _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m;
        super();
        this.files = {};
        this.dockerImages = {};
        const { BSS_FILE_ASSET_BUCKET_NAME, BSS_IMAGE_ASSET_REPOSITORY_NAME, BSS_FILE_ASSET_PUBLISHING_ROLE_ARN, BSS_IMAGE_ASSET_PUBLISHING_ROLE_ARN, BSS_FILE_ASSET_PREFIX, BSS_FILE_ASSET_REGION_SET, BSS_TEMPLATE_BUCKET_NAME, BSS_IMAGE_ASSET_TAG_PREFIX, BSS_IMAGE_ASSET_REGION_SET, BSS_IMAGE_ASSET_ACCOUNT_ID, } = process.env;
        this.bucketName = (_b = props.fileAssetBucketName) !== null && _b !== void 0 ? _b : BSS_FILE_ASSET_BUCKET_NAME;
        this.repositoryName = (_c = props.imageAssetRepositoryName) !== null && _c !== void 0 ? _c : BSS_IMAGE_ASSET_REPOSITORY_NAME;
        this.fileAssetPublishingRoleArn = (_d = props.fileAssetPublishingRoleArn) !== null && _d !== void 0 ? _d : BSS_FILE_ASSET_PUBLISHING_ROLE_ARN;
        this.imageAssetPublishingRoleArn = (_e = props.imageAssetPublishingRoleArn) !== null && _e !== void 0 ? _e : BSS_IMAGE_ASSET_PUBLISHING_ROLE_ARN;
        this.fileAssetPrefix = (_f = props.fileAssetPrefix) !== null && _f !== void 0 ? _f : BSS_FILE_ASSET_PREFIX;
        this.fileAssetRegionSet = (_g = props.fileAssetRegionSet) !== null && _g !== void 0 ? _g : commaSplit(BSS_FILE_ASSET_REGION_SET);
        this.templateBucketName = (_h = props.templateBucketName) !== null && _h !== void 0 ? _h : BSS_TEMPLATE_BUCKET_NAME;
        this.imageAssetTagPrefix = (_k = ((_j = props.imageAssetTagPrefix) !== null && _j !== void 0 ? _j : BSS_IMAGE_ASSET_TAG_PREFIX)) !== null && _k !== void 0 ? _k : '';
        this.imageAssetRegionSet = (_l = props.imageAssetRegionSet) !== null && _l !== void 0 ? _l : commaSplit(BSS_IMAGE_ASSET_REGION_SET);
        this.imageAssetAccountId = (_m = props.imageAssetAccountId) !== null && _m !== void 0 ? _m : BSS_IMAGE_ASSET_ACCOUNT_ID;
    }
    /**
     * Bind to the stack this environment is going to be used on.
     *
     * Must be called before any of the other methods are called.
     *
     * @stability stable
     */
    bind(stack) {
        var _b;
        if (this._stack !== undefined) {
            throw new Error('A StackSynthesizer can only be used for one Stack: create a new instance to use with a different Stack');
        }
        this._stack = stack;
        // Function to replace placeholders in the input string as much as possible
        //
        // We replace:
        // - ${AWS::AccountId}, ${AWS::Region}: only if we have the actual values available
        // - ${AWS::Partition}: never, since we never have the actual partition value.
        const specialize = (s) => {
            if (s === undefined) {
                return undefined;
            }
            return cxapi.EnvironmentPlaceholders.replace(s, {
                region: resolvedOr(stack.region, cxapi.EnvironmentPlaceholders.CURRENT_REGION),
                accountId: resolvedOr(stack.account, cxapi.EnvironmentPlaceholders.CURRENT_ACCOUNT),
                partition: cxapi.EnvironmentPlaceholders.CURRENT_PARTITION,
            });
        };
        /* eslint-disable max-len */
        this.bucketName = specialize(this.bucketName);
        this.repositoryName = specialize(this.repositoryName);
        this.fileAssetPublishingRoleArn = specialize(this.fileAssetPublishingRoleArn);
        this.imageAssetPublishingRoleArn = specialize(this.imageAssetPublishingRoleArn);
        this.fileAssetPrefix = specialize((_b = this.fileAssetPrefix) !== null && _b !== void 0 ? _b : '');
        /* eslint-enable max-len */
    }
    /**
     * Register a File Asset.
     *
     * Returns the parameters that can be used to refer to the asset inside the template.
     *
     * @stability stable
     */
    addFileAsset(asset) {
        return this._addFileAsset(asset);
    }
    _addFileAsset(asset, overrideBucketname) {
        var _b;
        assertNotNull(this.stack, ERR_MSG_CALL_BIND_FIRST);
        assertNotNull(this.bucketName, 'The bucketName is null');
        validateFileAssetSource(asset);
        const bucketName = overrideBucketname !== null && overrideBucketname !== void 0 ? overrideBucketname : this.bucketName;
        const objectKey = this.fileAssetPrefix + asset.sourceHash + (asset.packaging === core_1.FileAssetPackaging.ZIP_DIRECTORY ? '.zip' : '');
        const destinations = {};
        if (((_b = this.fileAssetRegionSet) === null || _b === void 0 ? void 0 : _b.length) && bucketName.includes(REGION_PLACEHOLDER)) {
            for (const region of this.fileAssetRegionSet.map(r => r.trim())) {
                if (!region) {
                    continue;
                }
                destinations[region] = {
                    bucketName: replaceAll(bucketName, REGION_PLACEHOLDER, region),
                    objectKey,
                    region,
                    assumeRoleArn: this.fileAssetPublishingRoleArn,
                };
            }
        }
        else {
            destinations[this.manifestEnvName] = {
                bucketName,
                objectKey,
                region: resolvedOr(this.stack.region, undefined),
                assumeRoleArn: this.fileAssetPublishingRoleArn,
            };
        }
        // Add to manifest
        this.files[asset.sourceHash] = {
            source: {
                path: asset.fileName,
                packaging: asset.packaging,
            },
            destinations,
        };
        const { region, urlSuffix } = stackLocationOrInstrinsics(this.stack);
        const httpUrl = cfnify(`https://s3.${region}.${urlSuffix}/${bucketName}/${objectKey}`);
        const s3ObjectUrl = cfnify(`s3://${bucketName}/${objectKey}`);
        // Return CFN expression
        return {
            bucketName: cfnify(bucketName),
            objectKey,
            httpUrl,
            s3ObjectUrl,
        };
    }
    /**
     * Register a Docker Image Asset.
     *
     * Returns the parameters that can be used to refer to the asset inside the template.
     *
     * @stability stable
     */
    addDockerImageAsset(asset) {
        var _b, _c;
        assertNotNull(this.stack, ERR_MSG_CALL_BIND_FIRST);
        assertNotNull(this.repositoryName, 'The repositoryName is null');
        validateDockerImageAssetSource(asset);
        const imageTag = this.imageAssetTagPrefix + asset.sourceHash;
        const destinations = {};
        if ((_b = this.imageAssetRegionSet) === null || _b === void 0 ? void 0 : _b.length) {
            for (const region of this.imageAssetRegionSet.map(r => r.trim())) {
                if (!region) {
                    continue;
                }
                destinations[region] = {
                    repositoryName: this.repositoryName,
                    imageTag,
                    region,
                    assumeRoleArn: this.fileAssetPublishingRoleArn,
                };
            }
        }
        else {
            destinations[this.manifestEnvName] = {
                repositoryName: this.repositoryName,
                imageTag,
                region: resolvedOr(this.stack.region, undefined),
                assumeRoleArn: this.imageAssetPublishingRoleArn,
            };
        }
        // Add to manifest
        this.dockerImages[asset.sourceHash] = {
            source: {
                directory: asset.directoryName,
                dockerBuildArgs: asset.dockerBuildArgs,
                dockerBuildTarget: asset.dockerBuildTarget,
                dockerFile: asset.dockerFile,
            },
            destinations,
        };
        let { account, urlSuffix } = stackLocationOrInstrinsics(this.stack);
        account = (_c = this.imageAssetAccountId) !== null && _c !== void 0 ? _c : account;
        return {
            repositoryName: cfnify(this.repositoryName),
            imageUri: cfnify(`${account}.dkr.ecr.${REGION_PLACEHOLDER}.${urlSuffix}/${this.repositoryName}:${imageTag}`),
        };
    }
    /**
     * Dumps current manifest into JSON format.
     *
     * @stability stable
     */
    dumps() {
        const manifest = {
            version: cxschema.Manifest.version(),
            files: this.files,
            dockerImages: this.dockerImages,
        };
        return JSON.stringify(manifest, undefined, 2);
    }
    /**
     * Synthesize the associated stack to the session.
     *
     * @stability stable
     */
    synthesize(session) {
        assertNotNull(this.stack, ERR_MSG_CALL_BIND_FIRST);
        this.synthesizeStackTemplate(this.stack, session);
        // Add the stack's template to the artifact manifest
        const templateManifestUrl = this.addStackTemplateToAssetManifest(session);
        const artifactId = this.writeAssetManifest(session);
        this.emitStackArtifact(this.stack, session, {
            stackTemplateAssetObjectUrl: templateManifestUrl,
            additionalDependencies: [artifactId],
        });
    }
    /**
     * @stability stable
     */
    get stack() {
        return this._stack;
    }
    /**
     * Add the stack's template as one of the manifest assets
     *
     * This will make it get uploaded to S3 automatically by S3-assets. Return
     * the manifest URL.
     *
     * (We can't return the location returned from `addFileAsset`, as that
     * contains CloudFormation intrinsics which can't go into the manifest).
     */
    addStackTemplateToAssetManifest(_) {
        assertNotNull(this.stack, ERR_MSG_CALL_BIND_FIRST);
        const sourceHash = this.stack.templateFile;
        this._addFileAsset({
            fileName: this.stack.templateFile,
            packaging: core_1.FileAssetPackaging.FILE,
            sourceHash,
        }, this.templateBucketName);
        // We should technically return an 'https://s3.REGION.amazonaws.com[.cn]/name/hash' URL here,
        // because that is what CloudFormation expects to see.
        //
        // However, there's no way for us to actually know the UrlSuffix a priori, so we can't construct it here.
        //
        // Instead, we'll have a protocol with the CLI that we put an 's3://.../...' URL here, and the CLI
        // is going to resolve it to the correct 'https://.../' URL before it gives it to CloudFormation.
        return `s3://${this.bucketName}/${sourceHash}`;
    }
    /**
     * Write an asset manifest to the Cloud Assembly, return the artifact IDs written
     */
    writeAssetManifest(session) {
        assertNotNull(this.stack, ERR_MSG_CALL_BIND_FIRST);
        const artifactId = `${this.stack.artifactId}.assets`;
        const manifestFile = `${artifactId}.json`;
        const outPath = path.join(session.assembly.outdir, manifestFile);
        fs.writeFileSync(outPath, this.dumps());
        session.assembly.addArtifact(artifactId, {
            type: cxschema.ArtifactType.ASSET_MANIFEST,
            properties: {
                file: manifestFile,
            },
        });
        return artifactId;
    }
    get manifestEnvName() {
        assertNotNull(this.stack, ERR_MSG_CALL_BIND_FIRST);
        return [
            resolvedOr(this.stack.account, 'current_account'),
            resolvedOr(this.stack.region, 'current_region'),
        ].join('-');
    }
}
exports.BootstraplessStackSynthesizer = BootstraplessStackSynthesizer;
_a = JSII_RTTI_SYMBOL_1;
BootstraplessStackSynthesizer[_a] = { fqn: "cdk-bootstrapless-synthesizer.BootstraplessStackSynthesizer", version: "2.0.0" };
/**
 * Return the given value if resolved or fall back to a default
 */
function resolvedOr(x, def) {
    return core_1.Token.isUnresolved(x) ? def : x;
}
/**
 * A "replace-all" function that doesn't require us escaping a literal string to a regex
 */
function replaceAll(s, search, replace) {
    return s.split(search).join(replace);
}
/**
 * If the string still contains placeholders, wrap it in a Fn::Sub so they will be substituted at CFN deployment time
 *
 * (This happens to work because the placeholders we picked map directly onto CFN
 * placeholders. If they didn't we'd have to do a transformation here).
 */
function cfnify(s) {
    return s.indexOf('${') > -1 ? core_1.Fn.sub(s) : s;
}
/**
 * Return the stack locations if they're concrete, or the original CFN intrisics otherwise
 *
 * We need to return these instead of the tokenized versions of the strings,
 * since we must accept those same ${AWS::AccountId}/${AWS::Region} placeholders
 * in bucket names and role names (in order to allow environment-agnostic stacks).
 *
 * We'll wrap a single {Fn::Sub} around the final string in order to replace everything,
 * but we can't have the token system render part of the string to {Fn::Join} because
 * the CFN specification doesn't allow the {Fn::Sub} template string to be an arbitrary
 * expression--it must be a string literal.
 */
function stackLocationOrInstrinsics(stack) {
    return {
        account: resolvedOr(stack.account, '${AWS::AccountId}'),
        region: resolvedOr(stack.region, '${AWS::Region}'),
        urlSuffix: resolvedOr(stack.urlSuffix, '${AWS::URLSuffix}'),
    };
}
// function range(startIncl: number, endExcl: number) {
//     const ret = new Array<number>();
//     for (let i = startIncl; i < endExcl; i++) {
//     ret.push(i);
//     }
//     return ret;
// }
function assertNotNull(x, msg = 'Null value error') {
    if (x === null || x === undefined) {
        throw new Error(msg);
    }
}
function commaSplit(v) {
    if (v) {
        return v.split(',');
    }
    return undefined;
}
function validateFileAssetSource(asset) {
    if (!!asset.executable === !!asset.fileName) {
        throw new Error(`Exactly one of 'fileName' or 'executable' is required, got: ${JSON.stringify(asset)}`);
    }
    if (!!asset.packaging !== !!asset.fileName) {
        throw new Error(`'packaging' is expected in combination with 'fileName', got: ${JSON.stringify(asset)}`);
    }
}
function validateDockerImageAssetSource(asset) {
    if (!!asset.executable === !!asset.directoryName) {
        throw new Error(`Exactly one of 'directoryName' or 'executable' is required, got: ${JSON.stringify(asset)}`);
    }
    check('dockerBuildArgs');
    check('dockerBuildTarget');
    check('dockerFile');
    function check(key) {
        if (asset[key] && !asset.directoryName) {
            throw new Error(`'${key}' is only allowed in combination with 'directoryName', got: ${JSON.stringify(asset)}`);
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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