"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_s3_1 = require("monocdk/aws-s3");
const monocdk_1 = require("monocdk");
const nag_rules_1 = require("../../nag-rules");
/**
 * S3 Buckets have default server-side encryption enabled
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_s3_1.CfnBucket) {
        if (node.bucketEncryption == undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        const encryption = monocdk_1.Stack.of(node).resolve(node.bucketEncryption);
        if (encryption.serverSideEncryptionConfiguration == undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        const sse = monocdk_1.Stack.of(node).resolve(encryption.serverSideEncryptionConfiguration);
        for (const rule of sse) {
            const defaultEncryption = monocdk_1.Stack.of(node).resolve(rule.serverSideEncryptionByDefault);
            if (defaultEncryption == undefined) {
                return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
            }
            const sseAlgorithm = nag_rules_1.NagRules.resolveIfPrimitive(node, defaultEncryption.sseAlgorithm);
            if (sseAlgorithm.toLowerCase() != 'aes256' &&
                sseAlgorithm.toLowerCase() != 'aws:kms') {
                return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
            }
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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