"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_elasticbeanstalk_1 = require("monocdk/aws-elasticbeanstalk");
const monocdk_1 = require("monocdk");
const nag_rules_1 = require("../../nag-rules");
/**
 * Elastic Beanstalk environments are configured to use a specific VPC
 * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/command-options-general.html#command-options-general-ec2vpc
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_elasticbeanstalk_1.CfnEnvironment) {
        const optionSettings = monocdk_1.Stack.of(node).resolve(node.optionSettings);
        if (optionSettings == undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        let foundEnabled = false;
        for (const optionSetting of optionSettings) {
            const resolvedOptionSetting = monocdk_1.Stack.of(node).resolve(optionSetting);
            const namespace = resolvedOptionSetting.namespace;
            const optionName = resolvedOptionSetting.optionName;
            const value = resolvedOptionSetting.value;
            if (namespace === 'aws:ec2:vpc' &&
                optionName === 'VPCId' &&
                value !== undefined) {
                foundEnabled = true;
                break;
            }
        }
        if (!foundEnabled) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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