"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_ecs_1 = require("monocdk/aws-ecs");
const monocdk_1 = require("monocdk");
const nag_rules_1 = require("../../nag-rules");
/**
 * Containers in ECS task definitions configured for host networking have 'privileged' set to true and a non-empty non-root 'user'
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_ecs_1.CfnTaskDefinition) {
        if (node.networkMode === aws_ecs_1.NetworkMode.HOST) {
            const containerDefinitions = monocdk_1.Stack.of(node).resolve(node.containerDefinitions);
            if (containerDefinitions !== undefined) {
                for (const containerDefinition of containerDefinitions) {
                    const resolvedDefinition = monocdk_1.Stack.of(node).resolve(containerDefinition);
                    const privileged = nag_rules_1.NagRules.resolveIfPrimitive(node, resolvedDefinition.privileged);
                    const user = nag_rules_1.NagRules.resolveIfPrimitive(node, resolvedDefinition.user);
                    if (privileged !== true || user === undefined) {
                        return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                    }
                    const rootIdentifiers = ['root', '0'];
                    const userParts = user.split(':');
                    for (const userPart of userParts) {
                        if (rootIdentifiers.includes(userPart.toLowerCase())) {
                            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                        }
                    }
                }
            }
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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