import os
from functools import lru_cache
from os.path import exists
from typing import List, Tuple, Union

import geopandas as gpd
import pandas as pd

from ziptool import load, shp_dir
from ziptool.utils import (
    cast_zipcode,
    get_fips_code_from_abbr,
    puma_shapefile_name,
    tract_shapefile_name,
)


@lru_cache(maxsize=100)
def get_state_intersections(state_fips_code: str) -> gpd.GeoDataFrame:
    """
    For a given state, computes the intersections between Census tracts and PUMAs.
    Note that you most run fetch_data.get_shape_files() before using this function.

    Args:
        state_fips_code: a string representing the FIPS code of the state of interest.

    Returns:
        a geopandas dataframe detailing the intersections of tracts and PUMAS for a state
    """

    puma_name = puma_shapefile_name(state_fips_code)
    tract_name = tract_shapefile_name(state_fips_code)

    if not exists(os.path.join(shp_dir.name, puma_name)):
        from ziptool.fetch_data import get_shape_files

        get_shape_files(state_fips_code)

    puma = gpd.read_file(os.path.join(shp_dir.name, puma_name)).to_crs(
        epsg=ALBERS_EPSG_ID
    )
    tract = gpd.read_file(os.path.join(shp_dir.name, tract_name)).to_crs(
        epsg=ALBERS_EPSG_ID
    )

    return gpd.overlay(puma, tract, how="intersection", keep_geom_type=False)


def zip_to_tract(zipcode: Union[str, int]) -> Tuple[Tuple[List[str], List[float]], str]:
    """
    For a given ZIP code, uses HUD Crosswalk data
    (https://www.huduser.gov/portal/datasets/usps_crosswalk.html)
    to find the ratio of persons in each census tract for the given ZIP code.

    Args:
        zip: the five-digit ZIP code of interest, written as a string

    Returns:
        List containing the same number of entries as census tracts within the ZIP code. Each entry is a list,
        entry 0 is the census tract and entry 1 is the residential ratio of the census tract within that ZIP.
    """
    zipcode = cast_zipcode(zipcode)
    hud_crosswalk = load.load_crosswalk()

    zipint = cast_zipcode(zipcode)

    contained = hud_crosswalk[hud_crosswalk["zip"] == zipint][["tract", "res_ratio"]]
    try:
        state = hud_crosswalk[hud_crosswalk["zip"] == zipint][
            "usps_zip_pref_state"
        ].iloc[0]
    except IndexError:
        raise KeyError(f"{zipcode} is not a valid ZIP code!")
    return contained.values.tolist(), get_fips_code_from_abbr(state)


def tracts_to_puma(tracts, state_fips_code: str):
    """
    Takes in a list of tracts and ratios for a given zip code (in a given state) and returns the PUMAs
    composing the ZIP code with ratios (i.e. 88% in PUMA 00101 and 12% in PUMA 00102).

    Args:
        tracts: a 2D list generated by zip_to_tract containing census tracts and weighted_ratios
        state_fips_code: string representing state of interest

    Returns:
        series containing ratio of population for each PUMA
    """
    global ALBERS_EPSG_ID
    ALBERS_EPSG_ID = 5070

    intersection_gdf = get_state_intersections(state_fips_code)
    intersection_gdf["shape_area"] = intersection_gdf.area
    intersection_gdf["GEOID"] = intersection_gdf["GEOID"].astype("int")

    intersection_gdf = intersection_gdf[["GEOID", "PUMACE10", "shape_area"]]
    tract_areas = (
        intersection_gdf[["GEOID", "PUMACE10", "shape_area"]].groupby("GEOID").sum()
    )

    intersection_gdf = intersection_gdf.set_index("GEOID")

    joined = intersection_gdf.join(tract_areas, rsuffix="_total", how="inner")
    joined["ratio"] = joined["shape_area"] / joined["shape_area_total"]

    tracts_of_interest = [int(x[0]) for x in tracts]

    sorted = joined[["PUMACE10", "ratio"]].loc[tracts_of_interest]
    sorted["rounded_ratio"] = sorted["ratio"].apply(lambda x: 1 if x > 0.99 else x)
    rounded = sorted[sorted["rounded_ratio"] > (1 - 0.99)]
    summary = rounded.join(
        pd.DataFrame(tracts, columns=["GEOID", "PUMARAT"])
        .astype({"GEOID": "int", "PUMARAT": "float32"})
        .set_index("GEOID")
    )
    summary["weighted_ratios"] = summary["rounded_ratio"] * summary["PUMARAT"]
    summary_top = summary.groupby("PUMACE10").sum()["weighted_ratios"]

    filtered = summary_top[summary_top > 0.01]
    rounded = filtered.apply(lambda x: 1.0 if x > 0.99 else x)

    return rounded
