# Copyright 2020-2021 Cambridge Quantum Computing
#
# You may not use this file except in compliance with the Licence.
# You may obtain a copy of the Licence in the LICENCE file accompanying
# these documents or at:
#
#     https://cqcl.github.io/pytket/build/html/licence.html

from typing import TYPE_CHECKING, MutableMapping, Optional, Union

from pytket.circuit import Circuit, OpType  # type: ignore
from pytket.passes import BasePass, RebaseCustom  # type: ignore

from .common import _QsharpSimBaseBackend, BackendResult
from pytket.utils.outcomearray import OutcomeArray

if TYPE_CHECKING:
    from qsharp.loader import QSharpCallable  # type: ignore


def is_approx_0_mod_2(x: float, tol: float = 1e-10) -> bool:
    x %= 2  # --> [0,2)
    return min(x, 2 - x) < tol


def toffoli_from_tk1(a: float, b: float, c: float) -> Circuit:
    """ Only accept operations equivalent to I or X. """
    circ = Circuit(1)
    if is_approx_0_mod_2(b) and is_approx_0_mod_2(a + c):
        # identity
        pass
    elif is_approx_0_mod_2(b + 1) and is_approx_0_mod_2(a - c):
        # X
        circ.X()
    else:
        raise RuntimeError(
            "Cannot compile to Toffoli gate set: TK1({}, {}, {}) ∉ {{I, X}}".format(
                a, b, c
            )
        )
    return circ


class QsharpToffoliSimulatorBackend(_QsharpSimBaseBackend):
    """Backend for simulating a Toffoli circuit using the QDK."""

    def default_compilation_pass(self, optimisation_level: int = 1) -> BasePass:
        assert optimisation_level in range(3)
        return RebaseCustom(
            {OpType.CX, OpType.CCX, OpType.CnX, OpType.SWAP},  # multiqs
            Circuit(),  # cx_replacement (irrelevant)
            {OpType.X},  # singleqs
            toffoli_from_tk1,
        )  # tk1_replacement

    def _calculate_results(
        self, qscall: "QSharpCallable", n_shots: Optional[int] = None
    ) -> Union[BackendResult, "MutableMapping"]:
        if n_shots:
            shots = OutcomeArray.from_readouts(
                [qscall.toffoli_simulate() for _ in range(n_shots)]
            )
            return BackendResult(shots=shots)
        raise ValueError("Parameter n_shots is required for this backend")
