"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = void 0;
const os = require("os");
const path = require("path");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const util_1 = require("./util");
/**
 * Bundling
 */
class Bundling {
    constructor(props) {
        var _a, _b, _c, _d;
        this.props = props;
        Bundling.runsLocally = (_b = (_a = Bundling.runsLocally) !== null && _a !== void 0 ? _a : util_1.getGoBuildVersion()) !== null && _b !== void 0 ? _b : false;
        const projectRoot = path.dirname(props.moduleDir);
        this.relativeEntryPath = `./${path.relative(projectRoot, path.resolve(props.entry))}`;
        const cgoEnabled = props.cgoEnabled ? '1' : '0';
        const environment = {
            CGO_ENABLED: cgoEnabled,
            GO111MODULE: 'on',
            GOARCH: 'amd64',
            GOOS: 'linux',
            ...props.environment,
        };
        // Docker bundling
        const shouldBuildImage = props.forcedDockerBundling || !Bundling.runsLocally;
        this.image = shouldBuildImage
            ? (_c = props.dockerImage) !== null && _c !== void 0 ? _c : cdk.DockerImage.fromBuild(path.join(__dirname, '../lib'), {
                buildArgs: {
                    ...(_d = props.buildArgs) !== null && _d !== void 0 ? _d : {},
                    IMAGE: aws_lambda_1.Runtime.GO_1_X.bundlingImage.image,
                },
            }) : cdk.DockerImage.fromRegistry('dummy'); // Do not build if we don't need to
        const bundlingCommand = this.createBundlingCommand(cdk.AssetStaging.BUNDLING_INPUT_DIR, cdk.AssetStaging.BUNDLING_OUTPUT_DIR);
        this.command = ['bash', '-c', bundlingCommand];
        this.environment = environment;
        // Local bundling
        if (!props.forcedDockerBundling) { // only if Docker is not forced
            const osPlatform = os.platform();
            const createLocalCommand = (outputDir) => this.createBundlingCommand(projectRoot, outputDir, osPlatform);
            this.local = {
                tryBundle(outputDir) {
                    if (Bundling.runsLocally == false) {
                        process.stderr.write('go build cannot run locally. Switching to Docker bundling.\n');
                        return false;
                    }
                    const localCommand = createLocalCommand(outputDir);
                    util_1.exec(osPlatform === 'win32' ? 'cmd' : 'bash', [
                        osPlatform === 'win32' ? '/c' : '-c',
                        localCommand,
                    ], {
                        env: { ...process.env, ...environment !== null && environment !== void 0 ? environment : {} },
                        stdio: [
                            'ignore',
                            process.stderr,
                            'inherit',
                        ],
                        cwd: path.dirname(props.moduleDir),
                        windowsVerbatimArguments: osPlatform === 'win32',
                    });
                    return true;
                },
            };
        }
    }
    static bundle(options) {
        var _a;
        const bundling = new Bundling(options);
        return aws_lambda_1.Code.fromAsset(path.dirname(options.moduleDir), {
            assetHashType: (_a = options.assetHashType) !== null && _a !== void 0 ? _a : cdk.AssetHashType.OUTPUT,
            assetHash: options.assetHash,
            bundling: {
                image: bundling.image,
                command: bundling.command,
                environment: bundling.environment,
                local: bundling.local,
            },
        });
    }
    static clearRunsLocallyCache() {
        this.runsLocally = undefined;
    }
    createBundlingCommand(inputDir, outputDir, osPlatform = 'linux') {
        var _a, _b, _c, _d;
        const pathJoin = osPathJoin(osPlatform);
        const hasVendor = util_1.findUp('vendor', path.dirname(this.props.entry));
        const goBuildCommand = [
            'go', 'build',
            hasVendor ? '-mod=vendor' : '',
            '-o', `${pathJoin(outputDir, 'bootstrap')}`,
            `${this.props.goBuildFlags ? this.props.goBuildFlags.join(' ') : ''}`,
            `${this.relativeEntryPath.replace(/\\/g, '/')}`,
        ].filter(c => !!c).join(' ');
        return chain([
            ...(_b = (_a = this.props.commandHooks) === null || _a === void 0 ? void 0 : _a.beforeBundling(inputDir, outputDir)) !== null && _b !== void 0 ? _b : [],
            goBuildCommand,
            ...(_d = (_c = this.props.commandHooks) === null || _c === void 0 ? void 0 : _c.afterBundling(inputDir, outputDir)) !== null && _d !== void 0 ? _d : [],
        ]);
    }
}
exports.Bundling = Bundling;
/**
 * Platform specific path join
 */
function osPathJoin(platform) {
    return function (...paths) {
        const joined = path.join(...paths);
        // If we are on win32 but need posix style paths
        if (os.platform() === 'win32' && platform !== 'win32') {
            return joined.replace(/\\/g, '/');
        }
        return joined;
    };
}
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
//# sourceMappingURL=data:application/json;base64,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