# The module records any regexes used by python code for later inspection.
# Import this hook and install() before loading other modules or start python with `python -i /path/to/hook.py`
import re
import traceback

# By default, the re and traceback modules will not be hooked

regexes = set()


class CompiledRegex:
    def __init__(self, pattern, flags, traceback):
        self.pattern = pattern
        self.flags = flags
        self.traceback = traceback

    def __hash__(self) -> int:
        # ignore the traceback for now
        return hash((self.pattern, self.flags))

    def __repr__(self) -> str:
        return f"({self.pattern} at {self.last_tb.filename})"

    @property
    def last_tb(self):
        return self.traceback[-1]


class WrappedRegex:
    def __init__(self, regex):
        self.regex = regex

    def run_and_log(self, method, args, kwargs):
        print("Pattern:", repr(self.regex.pattern[:200]))
        print(f"{method}()", *(repr(a) for a in args))
        print(*traceback.format_stack()[2:-4])
        return getattr(self.regex, method)(*args, **kwargs)

    def search(self, *args, **kwargs):
        return self.run_and_log("search", args, kwargs)

    def match(self, *args, **kwargs):
        return self.run_and_log("match", args, kwargs)

    def fullmatch(self, *args, **kwargs):
        return self.run_and_log("fullmatch", args, kwargs)

    def sub(self, *args, **kwargs):
        return self.run_and_log("sub", args, kwargs)

    def subn(self, *args, **kwargs):
        return self.run_and_log("subn", args, kwargs)

    def split(self, *args, **kwargs):
        return self.run_and_log("split", args, kwargs)

    def findall(self, *args, **kwargs):
        return self.run_and_log("findall", args, kwargs)

    def finditer(self, *args, **kwargs):
        return self.run_and_log("finditer", args, kwargs)


def get_and_clear_regexes():
    """
    Retrieves regexes that have been `re.compile`-ed and removes them from the `regexes` set.
    """
    while True:
        try:
            yield regexes.pop()
        except KeyError:
            return


def install(log_all_uses: bool = False):
    """
    Activate the hook.
    """
    if not hasattr(re.compile, "_is_hook"):
        old_compile = re.compile

        def compile(pattern, flags=0):
            tb = traceback.extract_stack()[:-1]  # Ignore our hook
            regexes.add(CompiledRegex(pattern, flags, tb))
            regex = old_compile(pattern, flags)
            if log_all_uses:
                return WrappedRegex(regex)
            else:
                return regex

        compile._is_hook = True
        re.compile = compile


if __name__ == "__main__":
    install()
