#!/usr/bin/env python
import argparse
import logging
import re
import traceback
import warnings

from regexploit.ast.sre import SreOpParser
from regexploit.bin.files import file_generator
from regexploit.languages.csharp_string_extractor import find_regexes
from regexploit.languages.javascript import fix_js_regex
from regexploit.output.text import TextOutput
from regexploit.redos import find


def handle_file(filename: str, output: TextOutput):
    with open(filename, "rb") as f:
        code = f.read()
    for regex in find_regexes(code):
        pattern = regex.pattern
        if len(pattern) < 5:
            continue  # (.+)+
        if pattern.count("*") + pattern.count("+") + pattern.count(",}") < 2:
            continue  # no ReDoS possible
        try:
            logging.debug("%s#%s: %s", filename, regex.lineno, pattern)
            parsed = SreOpParser().parse_sre(pattern, regex.flags)
        except:
            try:
                fixed = fix_js_regex(pattern)
                re.compile(fixed, regex.flags)
            except:
                if regex.definitely_regex:
                    print(
                        f"Error parsing: {pattern} from {filename} line {regex.lineno}\n"
                    )
                continue
            try:
                parsed = SreOpParser().parse_sre(fixed, regex.flags)
            except:
                print(f"Error in regexploit parsing: {pattern} from {filename}")
                print(traceback.format_exc())
                continue
        try:
            output.next()
            for redos in find(parsed):
                if redos.starriness > 2:
                    context = None
                    try:
                        context = code.splitlines()[regex.lineno - 1].decode().strip()
                    except UnicodeDecodeError:
                        pass
                    output.record(
                        redos,
                        pattern,
                        filename=filename,
                        lineno=regex.lineno,
                        context=context,
                    )
        except Exception:
            print(f"Error finding ReDoS: {pattern} from {filename} #{regex.lineno}")
            print(traceback.format_exc())


def main():
    with warnings.catch_warnings():
        warnings.simplefilter(
            "ignore", category=FutureWarning
        )  # Some csharp/js regexes are weird
        parser = argparse.ArgumentParser(
            description="Parse regexes out of C# files and scan them for ReDoS"
        )
        parser.add_argument("files", nargs="+", help="C# files")
        parser.add_argument(
            "--glob", action="store_true", help="Glob the input filenames (**/*)"
        )
        parser.add_argument("--verbose", action="store_true", help="Verbose logging")
        parser.add_argument(
            "--ignore", action="append", help="Paths containing this string are ignored"
        )
        args = parser.parse_args()

        if args.verbose:
            logging.basicConfig(level=logging.DEBUG)

        output = TextOutput(js_flavour=True)
        files = file_generator(args.files, args.glob, ["*.cs"], args.ignore)
        for filename in files:
            logging.debug(filename)
            handle_file(filename, output)
        print(f"Processed {output.regexes} regexes")


if __name__ == "__main__":
    main()
