"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3Origin = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudfront = require("@aws-cdk/aws-cloudfront");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const http_origin_1 = require("./http-origin");
/**
 * An Origin that is backed by an S3 bucket.
 *
 * If the bucket is configured for website hosting, this origin will be configured to use the bucket as an
 * HTTP server origin and will use the bucket's configured website redirects and error handling. Otherwise,
 * the origin is created as a bucket origin and will use CloudFront's redirect and error handling.
 */
class S3Origin {
    constructor(bucket, props = {}) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_cloudfront_origins_S3OriginProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, S3Origin);
            }
            throw error;
        }
        this.origin = bucket.isWebsite ?
            new http_origin_1.HttpOrigin(bucket.bucketWebsiteDomainName, {
                protocolPolicy: cloudfront.OriginProtocolPolicy.HTTP_ONLY,
                ...props,
            }) :
            new S3BucketOrigin(bucket, props);
    }
    bind(scope, options) {
        return this.origin.bind(scope, options);
    }
}
exports.S3Origin = S3Origin;
_a = JSII_RTTI_SYMBOL_1;
S3Origin[_a] = { fqn: "@aws-cdk/aws-cloudfront-origins.S3Origin", version: "1.167.0" };
/**
 * An Origin specific to a S3 bucket (not configured for website hosting).
 *
 * Contains additional logic around bucket permissions and origin access identities.
 */
class S3BucketOrigin extends cloudfront.OriginBase {
    constructor(bucket, { originAccessIdentity, ...props }) {
        super(bucket.bucketRegionalDomainName, props);
        this.bucket = bucket;
        if (originAccessIdentity) {
            this.originAccessIdentity = originAccessIdentity;
        }
    }
    bind(scope, options) {
        if (!this.originAccessIdentity) {
            // Using a bucket from another stack creates a cyclic reference with
            // the bucket taking a dependency on the generated S3CanonicalUserId for the grant principal,
            // and the distribution having a dependency on the bucket's domain name.
            // Fix this by parenting the OAI in the bucket's stack when cross-stack usage is detected.
            const bucketStack = cdk.Stack.of(this.bucket);
            const bucketInDifferentStack = bucketStack !== cdk.Stack.of(scope);
            const oaiScope = bucketInDifferentStack ? bucketStack : scope;
            const oaiId = bucketInDifferentStack ? `${cdk.Names.uniqueId(scope)}S3Origin` : 'S3Origin';
            this.originAccessIdentity = new cloudfront.OriginAccessIdentity(oaiScope, oaiId, {
                comment: `Identity for ${options.originId}`,
            });
        }
        // Used rather than `grantRead` because `grantRead` will grant overly-permissive policies.
        // Only GetObject is needed to retrieve objects for the distribution.
        // This also excludes KMS permissions; currently, OAI only supports SSE-S3 for buckets.
        // Source: https://aws.amazon.com/blogs/networking-and-content-delivery/serving-sse-kms-encrypted-content-from-s3-using-cloudfront/
        this.bucket.addToResourcePolicy(new iam.PolicyStatement({
            resources: [this.bucket.arnForObjects('*')],
            actions: ['s3:GetObject'],
            principals: [this.originAccessIdentity.grantPrincipal],
        }));
        return super.bind(scope, options);
    }
    renderS3OriginConfig() {
        return { originAccessIdentity: `origin-access-identity/cloudfront/${this.originAccessIdentity.originAccessIdentityName}` };
    }
}
//# sourceMappingURL=data:application/json;base64,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