"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_elasticloadbalancing_1 = require("monocdk/aws-elasticloadbalancing");
const aws_elasticloadbalancingv2_1 = require("monocdk/aws-elasticloadbalancingv2");
const monocdk_1 = require("monocdk");
const nag_rules_1 = require("../../nag-rules");
/**
 * ELBs have access logs enabled
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_elasticloadbalancing_1.CfnLoadBalancer) {
        if (node.accessLoggingPolicy == undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        const accessLoggingPolicy = monocdk_1.Stack.of(node).resolve(node.accessLoggingPolicy);
        const enabled = nag_rules_1.NagRules.resolveIfPrimitive(node, accessLoggingPolicy.enabled);
        if (enabled == false) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else if (node instanceof aws_elasticloadbalancingv2_1.CfnLoadBalancer) {
        const attributes = monocdk_1.Stack.of(node).resolve(node.loadBalancerAttributes);
        const reg = /"access_logs\.s3\.enabled","value":"true"/gm;
        if (JSON.stringify(attributes).search(reg) == -1) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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