"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_eks_1 = require("monocdk/aws-eks");
const monocdk_1 = require("monocdk");
const nag_rules_1 = require("../../nag-rules");
/**
 * EKS Clusters publish 'api', 'audit', 'authenticator, 'controllerManager', and 'scheduler' control plane logs
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    var _a, _b, _c, _d;
    if (node instanceof aws_eks_1.CfnCluster) {
        const requiredTypes = new Set([
            'api',
            'audit',
            'authenticator',
            'controllerManager',
            'scheduler',
        ]);
        const logging = monocdk_1.Stack.of(node).resolve(node.logging);
        const clusterLogging = monocdk_1.Stack.of(node).resolve(logging === null || logging === void 0 ? void 0 : logging.clusterLogging);
        const enabledTypes = (_a = monocdk_1.Stack.of(node).resolve(clusterLogging === null || clusterLogging === void 0 ? void 0 : clusterLogging.enabledTypes)) !== null && _a !== void 0 ? _a : [];
        for (const enabled of enabledTypes) {
            requiredTypes.delete(nag_rules_1.NagRules.resolveIfPrimitive(node, enabled.type));
            if (requiredTypes.size === 0) {
                break;
            }
        }
        return requiredTypes.size
            ? [...requiredTypes].map((log) => `LogExport::${log}`)
            : nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else if (node.cfnResourceType === 'Custom::AWSCDK-EKS-Cluster') {
        // The CDK uses a Custom Resource with AWS SDK calls to create EKS Clusters
        const props = monocdk_1.Stack.of(node).resolve(node._cfnProperties);
        const clusterLogging = (_d = monocdk_1.Stack.of(node).resolve((_c = (_b = props === null || props === void 0 ? void 0 : props.Config) === null || _b === void 0 ? void 0 : _b.logging) === null || _c === void 0 ? void 0 : _c.clusterLogging)) !== null && _d !== void 0 ? _d : [];
        const requiredTypes = new Set([
            'api',
            'audit',
            'authenticator',
            'controllerManager',
            'scheduler',
        ]);
        for (const config of clusterLogging) {
            if ((config === null || config === void 0 ? void 0 : config.enabled) === true) {
                for (const type of config === null || config === void 0 ? void 0 : config.types) {
                    requiredTypes.delete(type);
                    if (requiredTypes.size === 0) {
                        break;
                    }
                }
            }
        }
        return requiredTypes.size
            ? [...requiredTypes].map((log) => `LogExport::${log}`)
            : nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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