"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsSolutionsChecks = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../nag-pack");
const apigw_1 = require("../rules/apigw");
const appsync_1 = require("../rules/appsync");
const athena_1 = require("../rules/athena");
const autoscaling_1 = require("../rules/autoscaling");
const cloud9_1 = require("../rules/cloud9");
const cloudfront_1 = require("../rules/cloudfront");
const codebuild_1 = require("../rules/codebuild");
const cognito_1 = require("../rules/cognito");
const documentdb_1 = require("../rules/documentdb");
const dynamodb_1 = require("../rules/dynamodb");
const ec2_1 = require("../rules/ec2");
const ecr_1 = require("../rules/ecr");
const ecs_1 = require("../rules/ecs");
const efs_1 = require("../rules/efs");
const eks_1 = require("../rules/eks");
const elasticache_1 = require("../rules/elasticache");
const elasticbeanstalk_1 = require("../rules/elasticbeanstalk");
const elb_1 = require("../rules/elb");
const emr_1 = require("../rules/emr");
const eventbridge_1 = require("../rules/eventbridge");
const glue_1 = require("../rules/glue");
const iam_1 = require("../rules/iam");
const kinesis_1 = require("../rules/kinesis");
const kms_1 = require("../rules/kms");
const mediastore_1 = require("../rules/mediastore");
const msk_1 = require("../rules/msk");
const neptune_1 = require("../rules/neptune");
const opensearch_1 = require("../rules/opensearch");
const quicksight_1 = require("../rules/quicksight");
const rds_1 = require("../rules/rds");
const redshift_1 = require("../rules/redshift");
const s3_1 = require("../rules/s3");
const sagemaker_1 = require("../rules/sagemaker");
const secretsmanager_1 = require("../rules/secretsmanager");
const sns_1 = require("../rules/sns");
const sqs_1 = require("../rules/sqs");
const stepfunctions_1 = require("../rules/stepfunctions");
const timestream_1 = require("../rules/timestream");
const vpc_1 = require("../rules/vpc");
/**
 * Check Best practices based on AWS Solutions Security Matrix
 *
 */
class AwsSolutionsChecks extends nag_pack_1.NagPack {
    constructor(props) {
        super(props);
        this.packName = 'AwsSolutions';
    }
    visit(node) {
        if (node instanceof monocdk_1.CfnResource) {
            this.checkCompute(node);
            this.checkStorage(node);
            this.checkDatabases(node);
            this.checkNetworkDelivery(node);
            this.checkManagementGovernance(node);
            this.checkMachineLearning(node);
            this.checkAnalytics(node);
            this.checkSecurityCompliance(node);
            this.checkServerless(node);
            this.checkApplicationIntegration(node);
            this.checkMediaServices(node);
            this.checkDeveloperTools(node);
        }
    }
    /**
     * Check Compute Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkCompute(node) {
        this.applyRule({
            ruleSuffixOverride: 'EB1',
            info: 'The Elastic Beanstalk environment is not configured to use a specific VPC.',
            explanation: 'Use a non-default in order to seperate your environment from default resources.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elasticbeanstalk_1.ElasticBeanstalkVPCSpecified,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EB3',
            info: 'The Elastic Beanstalk environment does not have managed updates enabled.',
            explanation: 'Enable managed platform updates for beanstalk environments in order to receive bug fixes, software updates and new features. Managed platform updates perform immutable environment updates.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elasticbeanstalk_1.ElasticBeanstalkManagedUpdatesEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EB4',
            info: 'The Elastic Beanstalk environment does not upload EC2 Instance logs to S3.',
            explanation: 'Beanstalk environment logs should be retained and uploaded to Amazon S3 in order to keep the logging data for future audits, historical purposes or to track and analyze the EB application environment behavior for a long period of time.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: elasticbeanstalk_1.ElasticBeanstalkEC2InstanceLogsToS3,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EC23',
            info: 'The Security Group allows for 0.0.0.0/0 or ::/0 inbound access.',
            explanation: 'Large port ranges, when open, expose instances to unwanted attacks. More than that, they make traceability of vulnerabilities very difficult. For instance, your web servers may only require 80 and 443 ports to be open, but not all. One of the most common mistakes observed is when  all ports for 0.0.0.0/0 range are open in a rush to access the instance. EC2 instances must expose only to those ports enabled on the corresponding security group level.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2RestrictedInbound,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EC26',
            info: 'The EBS volume has encryption disabled.',
            explanation: "With EBS encryption, you aren't required to build, maintain, and secure your own key management infrastructure. EBS encryption uses KMS keys when creating encrypted volumes and snapshots. This helps protect data at rest.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2EBSVolumeEncrypted,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EC27',
            info: 'The Security Group does not have a description.',
            explanation: 'Descriptions help simplify operations and remove any opportunities for operator errors.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2SecurityGroupDescription,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EC28',
            info: 'The EC2 instance/AutoScaling launch configuration does not have detailed monitoring enabled.',
            explanation: 'Monitoring data helps make better decisions on architecting and managing compute resources.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2InstanceDetailedMonitoringEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EC29',
            info: 'The EC2 instance is not part of an ASG and has Termination Protection disabled.',
            explanation: 'Termination Protection safety feature enabled in order to protect the instances from being accidentally terminated.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2InstanceTerminationProtection,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'ECR1',
            info: 'The ECR Repository allows open access.',
            explanation: 'Removing * principals in an ECR Repository helps protect against unauthorized access.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ecr_1.ECROpenAccess,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'ECS2',
            info: 'The ECS Task Definition includes a container definition that directly specifies environment variables.',
            explanation: 'Use secrets to inject environment variables during container startup from AWS Systems Manager Parameter Store or Secrets Manager instead of directly specifying plaintext environment variables. Updates to direct environment variables require operators to change task definitions and perform new deployments.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ecs_1.ECSTaskDefinitionNoEnvironmentVariables,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'ECS4',
            info: 'The ECS Cluster has CloudWatch Container Insights disabled.',
            explanation: 'CloudWatch Container Insights allow operators to gain a better perspective on how the cluster’s applications and microservices are performing.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ecs_1.ECSClusterCloudWatchContainerInsights,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'ECS7',
            info: 'One or more containers in the ECS Task Definition do not have container logging enabled.',
            explanation: "Container logging allows operators to view and aggregate the logs from the container. Containers should use the 'awslogs' driver at a minimum.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: ecs_1.ECSTaskDefinitionContainerLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EKS1',
            info: "The EKS cluster's Kubernetes API server endpoint has public access enabled.",
            explanation: "A cluster's Kubernetes API server endpoint should not be publicly accessible from the Internet in order to avoid exposing private data and minimizing security risks. The API server endpoints should only be accessible from within a AWS Virtual Private Cloud (VPC).",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: eks_1.EKSClusterNoEndpointPublicAccess,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EKS2',
            info: "The EKS Cluster does not publish 'api', 'audit', 'authenticator, 'controllerManager', and 'scheduler' control plane logs.",
            explanation: 'EKS control plane logging provides audit and diagnostic logs directly from the Amazon EKS control plane to CloudWatch Logs in your account. These logs make it easy for you to secure and run your clusters.' +
                "This is a granular rule that returns individual findings that can be suppressed with 'appliesTo'. The findings are in the format 'LogExport::<log>' for exported logs. Example: appliesTo: ['LogExport::api'].",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: eks_1.EKSClusterControlPlaneLogs,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'ELB1',
            info: 'The CLB is used for incoming HTTP/HTTPS traffic. Use ALBs instead.',
            explanation: 'HTTP/HTTPS applications (monolithic or containerized) should use the ALB instead of the CLB for enhanced incoming traffic distribution, better performance and lower costs.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.CLBNoInboundHttpHttps,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'ELB2',
            info: 'The ELB does not have access logs enabled.',
            explanation: 'Access logs allow operators to to analyze traffic patterns and identify and troubleshoot security issues.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBLoggingEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'ELB3',
            info: 'The CLB does not have connection draining enabled.',
            explanation: 'With Connection Draining feature enabled, if an EC2 backend instance fails health checks The CLB will not send any new requests to the unhealthy instance. However, it will still allow existing (in-flight) requests to complete for the duration of the configured timeout.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.CLBConnectionDraining,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'ELB4',
            info: 'The CLB does not use at least two AZs with the Cross-Zone Load Balancing feature enabled.',
            explanation: 'CLBs can distribute the traffic evenly across all backend instances. To use Cross-Zone Load Balancing at optimal level, the system should maintain an equal EC2 capacity distribution in each of the AZs registered with the load balancer.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBCrossZoneLoadBalancingEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'ELB5',
            info: 'The CLB listener is not configured for secure (HTTPs or SSL) protocols for client communication.',
            explanation: 'The HTTPs or SSL protocols enable secure communication by encrypting the communication between the client and the load balancer.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBTlsHttpsListenersOnly,
            node: node,
        });
    }
    /**
     * Check Storage Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkStorage(node) {
        this.applyRule({
            ruleSuffixOverride: 'S1',
            info: 'The S3 Bucket has server access logs disabled.',
            explanation: 'The bucket should have server access logging enabled to provide detailed records for the requests that are made to the bucket.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketLoggingEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'S2',
            info: 'The S3 Bucket does not have public access restricted and blocked.',
            explanation: 'The bucket should have public access restricted and blocked to prevent unauthorized access.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketLevelPublicAccessProhibited,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'S3',
            info: 'The S3 Bucket does not default encryption enabled.',
            explanation: 'The bucket should minimally have SSE enabled to help protect data-at-rest.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketServerSideEncryptionEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'S5',
            info: 'The S3 static website bucket either has an open world bucket policy or does not use a CloudFront Origin Access Identity (OAI) in the bucket policy for limited getObject and/or putObject permissions.',
            explanation: 'An OAI allows you to provide access to content in your S3 static website bucket through CloudFront URLs without enabling public access through an open bucket policy, disabling S3 Block Public Access settings, and/or through object ACLs.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.S3WebBucketOAIAccess,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'S10',
            info: 'The S3 Bucket or bucket policy does not require requests to use SSL.',
            explanation: 'You can use HTTPS (TLS) to help prevent potential attackers from eavesdropping on or manipulating network traffic using person-in-the-middle or similar attacks. You should allow only encrypted connections over HTTPS (TLS) using the aws:SecureTransport condition on Amazon S3 bucket policies.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketSSLRequestsOnly,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EFS1',
            info: 'The EFS is not configured for encryption at rest.',
            explanation: 'By using an encrypted file system, data and metadata are automatically encrypted before being written to the file system. Similarly, as data and metadata are read, they are automatically decrypted before being presented to the application. These processes are handled transparently by EFS without requiring modification of applications.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: efs_1.EFSEncrypted,
            node: node,
        });
    }
    /**
     * Check Database Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkDatabases(node) {
        this.applyRule({
            ruleSuffixOverride: 'RDS2',
            info: 'The RDS instance or Aurora DB cluster does not have storage encryption enabled.',
            explanation: 'Storage encryption helps protect data-at-rest by encrypting the underlying storage, automated backups, read replicas, and snapshots for the database.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSStorageEncrypted,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RDS3',
            info: 'The non-Aurora RDS DB instance does not have multi-AZ support enabled.',
            explanation: 'Use multi-AZ deployment configurations for high availability and automatic failover support fully managed by AWS.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSMultiAZSupport,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RDS6',
            info: 'The RDS Aurora MySQL/PostgresSQL cluster does not have IAM Database Authentication enabled.',
            explanation: "With IAM Database Authentication enabled, the system doesn't have to use a password when connecting to the MySQL/PostgreSQL database instances, instead it uses an authentication token.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.AuroraMySQLPostgresIAMAuth,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RDS8',
            info: 'The RDS DB Security Group allows for 0.0.0.0/0 inbound access.',
            explanation: 'RDS DB security groups should not allow access from 0.0.0.0/0 (i.e. anywhere, every machine that has the ability to establish a connection) in order to reduce the risk of unauthorized access.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSRestrictedInbound,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RDS10',
            info: 'The RDS instance or Aurora DB cluster does not have deletion protection enabled.',
            explanation: 'The deletion protection feature helps protect the database from being accidentally deleted.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSInstanceDeletionProtectionEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RDS11',
            info: 'The RDS instance or Aurora DB cluster uses the default endpoint port.',
            explanation: 'Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. MySQL/Aurora port 3306, SQL Server port 1433, PostgreSQL port 5432, etc).',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSNonDefaultPort,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RDS13',
            info: 'The RDS instance is not configured for automated backups.',
            explanation: 'Automated backups allow for point-in-time recovery.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSInstanceBackupEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RDS14',
            info: 'The RDS Aurora MySQL cluster does not have Backtrack enabled.',
            explanation: 'Backtrack helps order to rewind cluster tables to a specific time, without using backups.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.AuroraMySQLBacktrack,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RDS15',
            info: 'The RDS DB instance or Aurora DB cluster does not have deletion protection enabled.',
            explanation: 'Enabling Deletion Protection at the cluster level for Amazon Aurora databases or instance level for non Aurora instances helps protect from accidental deletion.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSInstanceDeletionProtectionEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RDS16',
            info: 'The RDS Aurora MySQL serverless cluster does not have audit, error, general, and slowquery Log Exports enabled.',
            explanation: 'This allows operators to use CloudWatch to view logs to help diagnose problems in the database.' +
                "This is a granular rule that returns individual findings that can be suppressed with 'appliesTo'. The findings are in the format 'LogExport::<log>' for exported logs. Example: appliesTo: ['LogExport::audit'].",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: rds_1.AuroraMySQLLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'DDB3',
            info: 'The DynamoDB table does not have Point-in-time Recovery enabled.',
            explanation: 'DynamoDB continuous backups represent an additional layer of insurance against accidental loss of data on top of on-demand backups. The DynamoDB service can back up the data with per-second granularity and restore it to any single second from the time PITR was enabled up to the prior 35 days.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: dynamodb_1.DynamoDBPITREnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'DDB4',
            info: 'The DAX cluster does not have server-side encryption enabled.',
            explanation: 'Data in cache, configuration data and log files should be encrypted using Server-Side Encryption in order to protect from unauthorized access to the underlying storage.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: dynamodb_1.DAXEncrypted,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'AEC1',
            info: 'The ElastiCache cluster is not provisioned in a VPC.',
            explanation: 'Provisioning the cluster within a VPC allows for better flexibility and control over the cache clusters security, availability, traffic routing and more.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elasticache_1.ElastiCacheClusterInVPC,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'AEC3',
            info: 'The ElastiCache Redis cluster does not have both encryption in transit and at rest enabled.',
            explanation: 'Encryption in transit helps secure communications to the cluster. Encryption at rest helps protect data at rest from unauthorized access.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elasticache_1.ElastiCacheRedisClusterEncryption,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'AEC4',
            info: 'The ElastiCache Redis cluster is not deployed in a Multi-AZ configuration.',
            explanation: 'The cluster should use a Multi-AZ deployment configuration for high availability.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elasticache_1.ElastiCacheRedisClusterMultiAZ,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'AEC5',
            info: 'The ElastiCache cluster uses the default endpoint port.',
            explanation: 'Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. Redis port 6379 and Memcached port 11211).',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elasticache_1.ElastiCacheClusterNonDefaultPort,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'AEC6',
            info: 'The ElastiCache Redis cluster does not use Redis AUTH for user authentication.',
            explanation: 'Redis authentication tokens enable Redis to require a token (password) before allowing clients to execute commands, thereby improving data security.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: elasticache_1.ElastiCacheRedisClusterRedisAuth,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'N1',
            info: 'The Neptune DB cluster is not deployed in a Multi-AZ configuration.',
            explanation: 'The cluster should use a Multi-AZ deployment configuration for high availability.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: neptune_1.NeptuneClusterMultiAZ,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'N2',
            info: 'The Neptune DB instance does have Auto Minor Version Upgrade enabled.',
            explanation: 'The Neptune service regularly releases engine updates. Enabling Auto Minor Version Upgrade will allow the service to automatically apply these upgrades to DB Instances.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: neptune_1.NeptuneClusterAutomaticMinorVersionUpgrade,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'N3',
            info: 'The Neptune DB cluster does not have a reasonable minimum backup retention period configured.',
            explanation: 'The retention period represents the number of days to retain automated snapshots. A minimum retention period of 7 days is recommended but can be adjust to meet system requirements.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: neptune_1.NeptuneClusterBackupRetentionPeriod,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'N4',
            info: 'The Neptune DB cluster does not have encryption at rest enabled.',
            explanation: 'Encrypting data-at-rest protects data confidentiality and prevents unauthorized users from accessing sensitive information.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: neptune_1.NeptuneClusterEncryptionAtRest,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'N5',
            info: 'The Neptune DB cluster does not have IAM Database Authentication enabled.',
            explanation: "With IAM Database Authentication enabled, the system doesn't have to use a password when connecting to the cluster.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: neptune_1.NeptuneClusterIAMAuth,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS1',
            info: 'The Redshift cluster does not require TLS/SSL encryption.',
            explanation: 'Enabling the "require_ssl" parameter secures data-in-transit by encrypting the connection between the clients and the Redshift clusters.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftRequireTlsSSL,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS2',
            info: 'The Redshift cluster is not provisioned in a VPC.',
            explanation: 'Provisioning the cluster within a VPC allows for better flexibility and control over the Redshift clusters security, availability, traffic routing and more.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterInVPC,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS3',
            info: 'The Redshift cluster uses the default "awsuser" username.',
            explanation: 'Using a custom master user name instead of the default master user name (i.e. "awsuser") provides an additional layer of defense against non-targeted attacks.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterNonDefaultUsername,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS4',
            info: 'The Redshift cluster uses the default endpoint port.',
            explanation: 'Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. Redshift port 5439).',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterNonDefaultPort,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS5',
            info: 'The Redshift cluster does not have audit logging enabled.',
            explanation: 'Audit logging helps operators troubleshoot issues and ensure security.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterAuditLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS6',
            info: 'The Redshift cluster does not have encryption at rest enabled.',
            explanation: 'Encrypting data-at-rest protects data confidentiality.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterEncryptionAtRest,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS8',
            info: 'The Redshift cluster is publicly accessible.',
            explanation: 'Disabling public accessibility helps minimize security risks.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterPublicAccess,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS9',
            info: 'The Redshift cluster does not have version upgrade enabled.',
            explanation: 'Version Upgrade must enabled to enable the cluster to automatically receive upgrades during the maintenance window.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterVersionUpgrade,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS10',
            info: 'The Redshift cluster does not have a retention period for automated snapshots configured.',
            explanation: 'The retention period represents the number of days to retain automated snapshots. A positive retention period should be set to configure this feature.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftBackupEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS11',
            info: 'The Redshift cluster does not have user activity logging enabled.',
            explanation: 'User activity logging logs each query before it is performed on the clusters databse. To enable this feature associate a Resdhsift Cluster Parameter Group with the "enable_user_activity_logging" parameter set to "true".',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterUserActivityLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'DOC1',
            info: 'The Document DB cluster does not have encryption at rest enabled.',
            explanation: 'Encrypting data-at-rest protects data confidentiality and prevents unauthorized users from accessing sensitive information.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: documentdb_1.DocumentDBClusterEncryptionAtRest,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'DOC2',
            info: 'The Document DB cluster uses the default endpoint port.',
            explanation: 'Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. MongoDB port 27017).',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: documentdb_1.DocumentDBClusterNonDefaultPort,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'DOC3',
            info: 'The Document DB cluster does not have the username and password stored in Secrets Manager.',
            explanation: "Secrets Manager enables operators to replace hardcoded credentials in your code, including passwords, with an API call to Secrets Manager to retrieve the secret programmatically. This helps ensure the secret can't be compromised by someone examining system code, because the secret no longer exists in the code. Also, operators can configure Secrets Manager to automatically rotate the secret for you according to a specified schedule. This enables you to replace long-term secrets with short-term ones, significantly reducing the risk of compromise.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: documentdb_1.DocumentDBCredentialsInSecretsManager,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'DOC4',
            info: 'The Document DB cluster does not have a reasonable minimum backup retention period configured.',
            explanation: 'The retention period represents the number of days to retain automated snapshots. A minimum retention period of 7 days is recommended but can be adjust to meet system requirements.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: documentdb_1.DocumentDBClusterBackupRetentionPeriod,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'DOC5',
            info: 'The Document DB cluster does not have authenticate, createIndex, and dropCollection Log Exports enabled.',
            explanation: 'This allows operators to use CloudWatch to view logs to help diagnose problems in the database. The events recorded by the AWS DocumentDB audit logs include successful and failed authentication attempts, creating indexes or dropping a collection in a database within the DocumentDB cluster.' +
                "This is a granular rule that returns individual findings that can be suppressed with 'appliesTo'. The findings are in the format 'LogExport::<log>' for exported logs. Example: appliesTo: ['LogExport::authenticate'].",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: documentdb_1.DocumentDBClusterLogExports,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'TS3',
            info: 'The Timestream database does not use a Customer Managed KMS Key for at rest encryption.',
            explanation: 'All Timestream tables in a database are encrypted at rest by default using AWS Managed Key. These keys are rotated every three years. Data at rest must be encrypted using CMKs if you require more control over the permissions and lifecycle of your keys, including the ability to have them automatically rotated on an annual basis.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: timestream_1.TimestreamDatabaseCustomerManagedKey,
            node: node,
        });
    }
    /**
     * Check Network and Delivery Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkNetworkDelivery(node) {
        this.applyRule({
            ruleSuffixOverride: 'VPC3',
            info: 'A Network ACL or Network ACL entry has been implemented.',
            explanation: 'Network ACLs should be used sparingly for the following reasons: they can be complex to manage, they are stateless, every IP address must be explicitly opened in each (inbound/outbound) direction, and they affect a complete subnet. Use security groups when possible as they are stateful and easier to manage.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: vpc_1.VPCNoNACLs,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'VPC7',
            info: 'The VPC does not have an associated Flow Log.',
            explanation: 'VPC Flow Logs capture network flow information for a VPC, subnet, or network interface and stores it in Amazon CloudWatch Logs. Flow log data can help customers troubleshoot network issues; for example, to diagnose why specific traffic is not reaching an instance, which might be a result of overly restrictive security group rules.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: vpc_1.VPCFlowLogsEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'CFR1',
            info: 'The CloudFront distribution may require Geo restrictions.',
            explanation: 'Geo restriction may need to be enabled for the distribution in order to allow or deny a country in order to allow or restrict users in specific locations from accessing content.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: cloudfront_1.CloudFrontDistributionGeoRestrictions,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'CFR2',
            info: 'The CloudFront distribution may require integration with AWS WAF.',
            explanation: 'The Web Application Firewall can help protect against application-layer attacks that can compromise the security of the system or place unnecessary load on them.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: cloudfront_1.CloudFrontDistributionWAFIntegration,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'CFR3',
            info: 'The CloudFront distribution does not have access logging enabled.',
            explanation: 'Enabling access logs helps operators track all viewer requests for the content delivered through the Content Delivery Network.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudfront_1.CloudFrontDistributionAccessLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'CFR4',
            info: 'The CloudFront distribution allows for SSLv3 or TLSv1 for HTTPS viewer connections.',
            explanation: "Vulnerabilities have been and continue to be discovered in the deprecated SSL and TLS protocols. Help protect viewer connections by specifying a viewer certificate that enforces a minimum of TLSv1.1 or TLSv1.2 in the security policy. Distributions that use that use the default CloudFront viewer certificate or use 'vip' for the 'SslSupportMethod' are non-compliant with this rule, as the minimum security policy is set to TLSv1 regardless of the specified 'MinimumProtocolVersion'.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudfront_1.CloudFrontDistributionHttpsViewerNoOutdatedSSL,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'CFR5',
            info: 'The CloudFront distributions uses SSLv3 or TLSv1 for communication to the origin.',
            explanation: 'Vulnerabilities have been and continue to be discovered in the deprecated SSL and TLS protocols. Using a security policy with minimum TLSv1.1 or TLSv1.2 and appropriate security ciphers for HTTPS helps protect viewer connections.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudfront_1.CloudFrontDistributionNoOutdatedSSL,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'CFR6',
            info: 'The CloudFront distribution does not use an origin access identity an S3 origin.',
            explanation: 'Origin access identities help with security by restricting any direct access to objects through S3 URLs.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloudfront_1.CloudFrontDistributionS3OriginAccessIdentity,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'APIG1',
            info: 'The API does not have access logging enabled.',
            explanation: 'Enabling access logs helps operators view who accessed an API and how the caller accessed the API.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: apigw_1.APIGWAccessLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'APIG2',
            info: 'The REST API does not have request validation enabled.',
            explanation: 'The API should have basic request validation enabled. If the API is integrated with custom source (Lambda, ECS, etc..) in the backend, deeper input validation should be considered for implementation.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: apigw_1.APIGWRequestValidation,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'APIG3',
            info: 'The REST API stage is not associated with AWS WAFv2 web ACL.',
            explanation: 'AWS WAFv2 is a web application firewall that helps protect web applications and APIs from attacks by allowing configured rules to allow, block, or monitor (count) web requests based on customizable rules and conditions that are defined.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: apigw_1.APIGWAssociatedWithWAF,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'APIG4',
            info: 'The API does not implement authorization.',
            explanation: 'In most cases an API needs to have an authentication and authorization implementation strategy. This includes using such approaches as IAM, Cognito User Pools, Custom authorizer, etc.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: apigw_1.APIGWAuthorization,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'APIG6',
            info: 'The REST API Stage does not have CloudWatch logging enabled for all methods.',
            explanation: 'Enabling CloudWatch logs at the stage level helps operators to track and analyze execution behavior at the API stage level.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: apigw_1.APIGWExecutionLoggingEnabled,
            node: node,
        });
    }
    /**
     * Check Management and Governance Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkManagementGovernance(node) {
        this.applyRule({
            ruleSuffixOverride: 'AS1',
            info: 'The Auto Scaling Group does not have a cooldown period.',
            explanation: 'A cooldown period temporarily suspends any scaling activities in order to allow the newly launched EC2 instance(s) some time to start handling the application traffic.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: autoscaling_1.AutoScalingGroupCooldownPeriod,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'AS2',
            info: 'The Auto Scaling Group does not have properly configured health checks.',
            explanation: 'The health check feature enables the service to detect whether its registered EC2 instances are healthy or not.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: autoscaling_1.AutoScalingGroupHealthCheck,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'AS3',
            info: 'The Auto Scaling Group does not have notifications configured for all scaling events.',
            explanation: 'Notifications on EC2 instance launch, launch error, termination, and termination errors allow operators to gain better insights into systems attributes such as activity and health.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: autoscaling_1.AutoScalingGroupScalingNotifications,
            node: node,
        });
    }
    /**
     * Check Machine Learning Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkMachineLearning(node) {
        this.applyRule({
            ruleSuffixOverride: 'SM1',
            info: 'The SageMaker notebook instance is not provisioned inside a VPC.',
            explanation: 'Provisioning the notebook instances inside a VPC enables the notebook to access VPC-only resources such as EFS file systems.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.SageMakerNotebookInVPC,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SM2',
            info: 'The SageMaker notebook instance does not have an encrypted storage volume.',
            explanation: 'Encrypting storage volumes helps protect SageMaker data-at-rest.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.SageMakerNotebookInstanceKMSKeyConfigured,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SM3',
            info: 'The SageMaker notebook instance has direct internet access enabled.',
            explanation: 'Disabling public accessibility helps minimize security risks.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.SageMakerNotebookNoDirectInternetAccess,
            node: node,
        });
    }
    /**
     * Check Analytics Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkAnalytics(node) {
        this.applyRule({
            ruleSuffixOverride: 'ATH1',
            info: 'The Athena workgroup does not encrypt query results.',
            explanation: 'Encrypting query results stored in S3 helps secure data to meet compliance requirements for data-at-rest encryption.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: athena_1.AthenaWorkgroupEncryptedQueryResults,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EMR2',
            info: 'The EMR cluster does not have S3 logging enabled.',
            explanation: 'Uploading logs to S3 enables the system to keep the logging data for historical purposes or to track and analyze the clusters behavior.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: emr_1.EMRS3AccessLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EMR4',
            info: 'The EMR cluster does not use a security configuration with local disk encryption enabled.',
            explanation: 'Local disk encryption uses a combination of open-source HDFS encryption and LUKS encryption to secure data at rest.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: emr_1.EMRLocalDiskEncryption,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EMR5',
            info: 'The EMR cluster does not use a security configuration with encryption in transit enabled and configured.',
            explanation: 'EMR Clusters should have a method for encrypting data in transit using Transport Layer Security (TLS).',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: emr_1.EMREncryptionInTransit,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EMR6',
            info: 'The EMR cluster does not implement authentication via an EC2 Key Pair or Kerberos.',
            explanation: 'SSH clients can use an EC2 key pair to authenticate to cluster instances. Alternatively, with EMR release version 5.10.0 or later, solutions can configure Kerberos to authenticate users and SSH connections to the master node.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: emr_1.EMRAuthEC2KeyPairOrKerberos,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'GL1',
            info: 'The Glue crawler or job does not use a security configuration with CloudWatch Log encryption enabled.',
            explanation: 'Enabling encryption at rest helps prevent unauthorized users from getting access to the logging data published to CloudWatch Logs.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: glue_1.GlueEncryptedCloudWatchLogs,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'GL3',
            info: 'The Glue job does not have use a security configuration with job bookmark encryption enabled.',
            explanation: 'Job bookmark encryption encrypts bookmark data before it is sent to Amazon S3 for storage.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: glue_1.GlueJobBookmarkEncrypted,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'KDA3',
            info: 'The Kinesis Data Analytics Flink Application does not have checkpointing enabled.',
            explanation: 'Checkpoints are backups of application state that KDA automatically creates periodically and uses to restore from faults.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: kinesis_1.KinesisDataAnalyticsFlinkCheckpointing,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'KDF1',
            info: 'The Kinesis Data Firehose delivery stream does have server-side encryption enabled.',
            explanation: 'This allows the system to meet strict regulatory requirements and enhance the security of system data.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: kinesis_1.KinesisDataFirehoseSSE,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'KDS1',
            info: 'The Kinesis Data Stream does not has server-side encryption enabled.',
            explanation: "Data is encrypted before it's written to the Kinesis stream storage layer, and decrypted after it’s retrieved from storage. This allows the system to meet strict regulatory requirements and enhance the security of system data.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: kinesis_1.KinesisDataStreamSSE,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'KDS3',
            info: 'The Kinesis Data Stream specifies server-side encryption and does not use the "aws/kinesis" key.',
            explanation: 'Customer Managed Keys can incur additional costs that scale with the amount of consumers and producers. Ensure that Customer Managed Keys are required for compliance before using them (https://docs.aws.amazon.com/streams/latest/dev/costs-performance.html).',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: kinesis_1.KinesisDataStreamDefaultKeyWhenSSE,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'MSK2',
            info: 'The MSK cluster uses plaintext communication between clients and brokers.',
            explanation: 'TLS only communication secures data-in-transit by encrypting the connection between the clients and brokers.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: msk_1.MSKClientToBrokerTLS,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'MSK3',
            info: 'The MSK cluster uses plaintext communication between brokers.',
            explanation: 'TLS communication secures data-in-transit by encrypting the connection between brokers.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: msk_1.MSKBrokerToBrokerTLS,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'MSK6',
            info: 'The MSK cluster does not send broker logs to a supported destination.',
            explanation: 'Broker logs enable operators to troubleshoot Apache Kafka applications and to analyze their communications with the MSK cluster. The cluster can deliver logs to the following resources: a CloudWatch log group, an S3 bucket, a Kinesis Data Firehose delivery stream.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: msk_1.MSKBrokerLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'OS1',
            info: 'The OpenSearch Service domain is not provisioned inside a VPC.',
            explanation: 'Provisioning the domain within a VPC enables better flexibility and control over the clusters access and security as this feature keeps all traffic between the VPC and OpenSearch domains within the AWS network instead of going over the public Internet.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchInVPCOnly,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'OS2',
            info: 'The OpenSearch Service domain does not have node-to-node encryption enabled.',
            explanation: 'Enabling the node-to-node encryption feature adds an extra layer of data protection on top of the existing ES security features such as HTTPS client to cluster encryption and data-at-rest encryption.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchNodeToNodeEncryption,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'OS3',
            info: 'The OpenSearch Service domain does not only grant access via allowlisted IP addresses.',
            explanation: 'Using allowlisted IP addresses helps protect the domain against unauthorized access.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchAllowlistedIPs,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'OS4',
            info: 'The OpenSearch Service domain does not use dedicated master nodes.',
            explanation: 'Using dedicated master nodes helps improve environmental stability by offloading all the management tasks from the data nodes.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchDedicatedMasterNode,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'OS5',
            info: 'The OpenSearch Service domain does not allow for unsigned requests or anonymous access.',
            explanation: 'Restricting public access helps prevent unauthorized access and prevents any unsigned requests to be made to the resources.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchNoUnsignedOrAnonymousAccess,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'OS7',
            info: 'The OpenSearch Service domain does not have Zone Awareness enabled.',
            explanation: 'Enabling cross-zone replication (Zone Awareness) increases the availability of the OpenSearch Service domain by allocating the nodes and replicate the data across two AZs in the same region in order to prevent data loss and minimize downtime in the event of node or AZ failure.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchZoneAwareness,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'OS8',
            info: 'The OpenSearch Service domain does not have encryption at rest enabled.',
            explanation: 'Encrypting data-at-rest protects data confidentiality and prevents unauthorized users from accessing sensitive information.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchEncryptedAtRest,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'OS9',
            info: 'The OpenSearch Service domain does not minimally publish SEARCH_SLOW_LOGS and INDEX_SLOW_LOGS to CloudWatch Logs.',
            explanation: 'These logs enable operators to gain full insight into the performance of these operations.' +
                "This is a granular rule that returns individual findings that can be suppressed with 'appliesTo'. The findings are in the format 'LogExport::<log>' for exported logs. Example: appliesTo: ['LogExport::SEARCH_SLOW_LOGS'].",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchSlowLogsToCloudWatch,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'QS1',
            info: 'The Quicksight data sources connection is not configured to use SSL.',
            explanation: 'SSL secures communications to data sources, especially when using public networks. Using SSL with QuickSight requires the use of certificates signed by a publicly-recognized certificate authority.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: quicksight_1.QuicksightSSLConnections,
            node: node,
        });
    }
    /**
     * Check Security and Compliance Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkSecurityCompliance(node) {
        this.applyRule({
            ruleSuffixOverride: 'IAM4',
            info: 'The IAM user, role, or group uses AWS managed policies.',
            explanation: 'An AWS managed policy is a standalone policy that is created and administered by AWS. Currently, many AWS managed policies do not restrict resource scope. Replace AWS managed policies with system specific (customer) managed policies.' +
                "This is a granular rule that returns individual findings that can be suppressed with 'appliesTo'. The findings are in the format 'Policy::<policy>' for AWS managed policies. Example: appliesTo: ['Policy::arn:<AWS::Partition>:iam::aws:policy/foo'].",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: iam_1.IAMNoManagedPolicies,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'IAM5',
            info: 'The IAM entity contains wildcard permissions and does not have a cdk_nag rule suppression with evidence for those permission.',
            explanation: 'Metadata explaining the evidence (e.g. via supporting links) for wildcard permissions allows for transparency to operators. ' +
                "This is a granular rule that returns individual findings that can be suppressed with 'appliesTo'. The findings are in the format 'Action::<action>' for policy actions and 'Resource::<resource>' for resources. Example: appliesTo: ['Action::s3:*'].",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: iam_1.IAMNoWildcardPermissions,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'COG1',
            info: 'The Cognito user pool does not have a password policy that minimally specify a password length of at least 8 characters, as well as requiring uppercase, numeric, and special characters.',
            explanation: 'Strong password policies increase system security by encouraging users to create reliable and secure passwords.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cognito_1.CognitoUserPoolStrongPasswordPolicy,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'COG2',
            info: 'The Cognito user pool does not require MFA.',
            explanation: 'Multi-factor authentication (MFA) increases security for the application by adding another authentication method, and not relying solely on user name and password.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: cognito_1.CognitoUserPoolMFA,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'COG3',
            info: 'The Cognito user pool does not have AdvancedSecurityMode set to ENFORCED.',
            explanation: 'Advanced security features enable the system to detect and act upon malicious sign-in attempts.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cognito_1.CognitoUserPoolAdvancedSecurityModeEnforced,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'COG4',
            info: 'The API GW method does not use a Cognito user pool authorizer.',
            explanation: 'API Gateway validates the tokens from a successful user pool authentication, and uses them to grant your users access to resources including Lambda functions, or your own API.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cognito_1.CognitoUserPoolAPIGWAuthorizer,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'COG7',
            info: 'The Cognito identity pool allows for unauthenticated logins and does not have a cdk_nag rule suppression with a reason.',
            explanation: 'In many cases applications do not warrant unauthenticated guest access applications. Metadata explaining the use case allows for transparency to operators.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cognito_1.CognitoUserPoolNoUnauthenticatedLogins,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'KMS5',
            info: 'The KMS Symmetric key does not have automatic key rotation enabled.',
            explanation: 'KMS key rotation allow a system to set an yearly rotation schedule for a KMS key so when a AWS KMS key is required to encrypt new data, the KMS service can automatically use the latest version of the HSA backing key to perform the encryption.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: kms_1.KMSBackingKeyRotationEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SMG4',
            info: 'The secret does not have automatic rotation scheduled.',
            explanation: 'AWS Secrets Manager can be configured to automatically rotate the secret for a secured service or database.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: secretsmanager_1.SecretsManagerRotationEnabled,
            node: node,
        });
    }
    /**
     * Check Serverless Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkServerless(node) {
        this.applyRule({
            ruleSuffixOverride: 'ASC3',
            info: 'The GraphQL API does not have request leveling logging enabled.',
            explanation: 'It is important to use CloudWatch Logs to log metrics such as who has accessed the GraphQL API, how the caller accessed the API, and invalid requests.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: appsync_1.AppSyncGraphQLRequestLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SF1',
            info: 'The Step Function does not log "ALL" events to CloudWatch Logs.',
            explanation: 'Logging "ALL" events to CloudWatch logs help operators troubleshoot and audit systems.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: stepfunctions_1.StepFunctionStateMachineAllLogsToCloudWatch,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SF2',
            info: 'The Step Function does not have X-Ray tracing enabled.',
            explanation: 'X-ray provides an end-to-end view of how an application is performing. This helps operators to discover performance issues, detect permission problems, and track requests made to and from other AWS services.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: stepfunctions_1.StepFunctionStateMachineXray,
            node: node,
        });
    }
    /**
     * Check Application Integration Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkApplicationIntegration(node) {
        this.applyRule({
            ruleSuffixOverride: 'EVB1',
            info: 'The event bus policy allows for open access.',
            explanation: 'An open policy ("*" principal without a condition) grants anonymous access to an event bus. Use a condition to limit the permission to accounts that fulfill a certain requirement, such as being a member of a certain AWS organization.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: eventbridge_1.EventBusOpenAccess,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SNS2',
            info: 'The SNS Topic does not have server-side encryption enabled.',
            explanation: 'Server side encryption adds additional protection of sensitive data delivered as messages to subscribers.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sns_1.SNSEncryptedKMS,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SNS3',
            info: 'The SNS Topic does not require publishers to use SSL.',
            explanation: "Without HTTPS (TLS), a network-based attacker can eavesdrop on network traffic or manipulate it, using an attack such as man-in-the-middle. Allow only encrypted connections over HTTPS (TLS) using the aws:SecureTransport condition and the 'sns: Publish' action in the topic policy to force publishers to use SSL. If SSE is already enabled then this control is auto enforced.",
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sns_1.SNSTopicSSLPublishOnly,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SQS2',
            info: 'The SQS Queue does not have server-side encryption enabled.',
            explanation: 'Server side encryption adds additional protection of sensitive data delivered as messages to subscribers.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sqs_1.SQSQueueSSE,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SQS3',
            info: 'The SQS queue does not have a dead-letter queue (DLQ) enabled or have a cdk_nag rule suppression indicating it is a DLQ.',
            explanation: 'Using a DLQ helps maintain the queue flow and avoid losing data by detecting and mitigating failures and service disruptions on time.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sqs_1.SQSQueueDLQ,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SQS4',
            info: 'The SQS queue does not require requests to use SSL.',
            explanation: 'Without HTTPS (TLS), a network-based attacker can eavesdrop on network traffic or manipulate it, using an attack such as man-in-the-middle. Allow only encrypted connections over HTTPS (TLS) using the aws:SecureTransport condition in the queue policy to force requests to use SSL.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: sqs_1.SQSQueueSSLRequestsOnly,
            node: node,
        });
    }
    /**
     * Check Media Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkMediaServices(node) {
        this.applyRule({
            ruleSuffixOverride: 'MS1',
            info: 'The MediaStore container does not have container access logging enabled.',
            explanation: 'The container should have access logging enabled to provide detailed records for the requests that are made to the container.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: mediastore_1.MediaStoreContainerAccessLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'MS3',
            info: 'The MediaStore container does not require requests to use SSL.',
            explanation: 'You can use HTTPS (TLS) to help prevent potential attackers from eavesdropping on or manipulating network traffic using person-in-the-middle or similar attacks. You should allow only encrypted connections over HTTPS (TLS) using the aws:SecureTransport condition on MediaStore container policies.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: mediastore_1.MediaStoreContainerSSLRequestsOnly,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'MS4',
            info: 'The MediaStore container does not define a metric policy to send metrics to CloudWatch.',
            explanation: 'Using a combination of MediaStore metrics and CloudWatch alarms helps operators gain better insights into container operations.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: mediastore_1.MediaStoreCloudWatchMetricPolicy,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'MS7',
            info: 'The MediaStore container does not define a container policy.',
            explanation: 'Using a container policy helps follow the standard security advice of granting least privilege, or granting only the permissions required to allow needed access to the container.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: mediastore_1.MediaStoreContainerHasContainerPolicy,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'MS8',
            info: 'The MediaStore container does not define a CORS policy.',
            explanation: 'Using a CORS policy helps follow the standard security advice of granting least privilege, or granting only the permissions required to allow needed access to the container.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: mediastore_1.MediaStoreContainerCORSPolicy,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'MS10',
            info: 'The MediaStore container does not define a lifecycle policy.',
            explanation: 'Many use cases warrant the usage of lifecycle configurations to manage container objects during their lifetime.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: mediastore_1.MediaStoreContainerLifecyclePolicy,
            node: node,
        });
    }
    /**
     * Check Developer Tools Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkDeveloperTools(node) {
        this.applyRule({
            ruleSuffixOverride: 'CB3',
            info: 'The CodeBuild project has privileged mode enabled.',
            explanation: 'Privileged grants elevated rights to the system, which introduces additional risk. Privileged mode should only be set to true only if the build project is used to build Docker images. Otherwise, a build that attempts to interact with the Docker daemon fails.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: codebuild_1.CodeBuildProjectPrivilegedModeDisabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'CB4',
            info: 'The CodeBuild project does not use an AWS KMS key for encryption.',
            explanation: 'Using an AWS KMS key helps follow the standard security advice of granting least privilege to objects generated by the project.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: codebuild_1.CodeBuildProjectKMSEncryptedArtifacts,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'CB5',
            info: 'The Codebuild project does not use images provided by the CodeBuild service or have a cdk_nag suppression rule explaining the need for a custom image.',
            explanation: 'Explaining differences/edits to Docker images helps operators better understand system dependencies.',
            level: nag_pack_1.NagMessageLevel.WARN,
            rule: codebuild_1.CodeBuildProjectManagedImages,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'C91',
            info: 'The Cloud9 instance does not use a no-ingress EC2 instance with AWS Systems Manager.',
            explanation: 'SSM adds an additional layer of protection as it allows operators to control access through IAM permissions and does not require opening inbound ports.',
            level: nag_pack_1.NagMessageLevel.ERROR,
            rule: cloud9_1.Cloud9InstanceNoIngressSystemsManager,
            node: node,
        });
    }
}
exports.AwsSolutionsChecks = AwsSolutionsChecks;
_a = JSII_RTTI_SYMBOL_1;
AwsSolutionsChecks[_a] = { fqn: "monocdk-nag.AwsSolutionsChecks", version: "1.12.40" };
//# sourceMappingURL=data:application/json;base64,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