"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseClusterFromSnapshot = exports.DatabaseCluster = exports.DatabaseClusterBase = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const logs = require("@aws-cdk/aws-logs");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
const endpoint_1 = require("./endpoint");
const util_1 = require("./private/util");
const props_1 = require("./props");
const proxy_1 = require("./proxy");
const rds_generated_1 = require("./rds.generated");
const subnet_group_1 = require("./subnet-group");
/**
 * A new or imported clustered database.
 *
 * @stability stable
 */
class DatabaseClusterBase extends core_1.Resource {
    /**
     * Add a new db proxy to this cluster.
     *
     * @stability stable
     */
    addProxy(id, options) {
        return new proxy_1.DatabaseProxy(this, id, {
            proxyTarget: proxy_1.ProxyTarget.fromCluster(this),
            ...options,
        });
    }
    /**
     * Renders the secret attachment target specifications.
     *
     * @stability stable
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.clusterIdentifier,
            targetType: secretsmanager.AttachmentTargetType.RDS_DB_CLUSTER,
        };
    }
}
exports.DatabaseClusterBase = DatabaseClusterBase;
_a = JSII_RTTI_SYMBOL_1;
DatabaseClusterBase[_a] = { fqn: "@aws-cdk/aws-rds.DatabaseClusterBase", version: "1.112.0" };
/**
 * Abstract base for ``DatabaseCluster`` and ``DatabaseClusterFromSnapshot``
 */
class DatabaseClusterNew extends DatabaseClusterBase {
    constructor(scope, id, props) {
        var _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p;
        super(scope, id);
        const { subnetIds } = props.instanceProps.vpc.selectSubnets(props.instanceProps.vpcSubnets);
        // Cannot test whether the subnets are in different AZs, but at least we can test the amount.
        if (subnetIds.length < 2) {
            core_1.Annotations.of(this).addError(`Cluster requires at least 2 subnets, got ${subnetIds.length}`);
        }
        this.subnetGroup = (_d = props.subnetGroup) !== null && _d !== void 0 ? _d : new subnet_group_1.SubnetGroup(this, 'Subnets', {
            description: `Subnets for ${id} database`,
            vpc: props.instanceProps.vpc,
            vpcSubnets: props.instanceProps.vpcSubnets,
            removalPolicy: util_1.renderUnless(util_1.helperRemovalPolicy(props.removalPolicy), core_1.RemovalPolicy.DESTROY),
        });
        this.securityGroups = (_e = props.instanceProps.securityGroups) !== null && _e !== void 0 ? _e : [
            new ec2.SecurityGroup(this, 'SecurityGroup', {
                description: 'RDS security group',
                vpc: props.instanceProps.vpc,
            }),
        ];
        let { s3ImportRole, s3ExportRole } = util_1.setupS3ImportExport(this, props);
        // bind the engine to the Cluster
        const clusterEngineBindConfig = props.engine.bindToCluster(this, {
            s3ImportRole,
            s3ExportRole,
            parameterGroup: props.parameterGroup,
        });
        const clusterAssociatedRoles = [];
        if (s3ImportRole) {
            clusterAssociatedRoles.push({ roleArn: s3ImportRole.roleArn, featureName: (_f = clusterEngineBindConfig.features) === null || _f === void 0 ? void 0 : _f.s3Import });
        }
        if (s3ExportRole) {
            clusterAssociatedRoles.push({ roleArn: s3ExportRole.roleArn, featureName: (_g = clusterEngineBindConfig.features) === null || _g === void 0 ? void 0 : _g.s3Export });
        }
        const clusterParameterGroup = (_h = props.parameterGroup) !== null && _h !== void 0 ? _h : clusterEngineBindConfig.parameterGroup;
        const clusterParameterGroupConfig = clusterParameterGroup === null || clusterParameterGroup === void 0 ? void 0 : clusterParameterGroup.bindToCluster({});
        this.engine = props.engine;
        const clusterIdentifier = core_1.FeatureFlags.of(this).isEnabled(cxapi.RDS_LOWERCASE_DB_IDENTIFIER)
            ? (_j = props.clusterIdentifier) === null || _j === void 0 ? void 0 : _j.toLowerCase() : props.clusterIdentifier;
        this.newCfnProps = {
            // Basic
            engine: props.engine.engineType,
            engineVersion: (_k = props.engine.engineVersion) === null || _k === void 0 ? void 0 : _k.fullVersion,
            dbClusterIdentifier: clusterIdentifier,
            dbSubnetGroupName: this.subnetGroup.subnetGroupName,
            vpcSecurityGroupIds: this.securityGroups.map(sg => sg.securityGroupId),
            port: (_l = props.port) !== null && _l !== void 0 ? _l : clusterEngineBindConfig.port,
            dbClusterParameterGroupName: clusterParameterGroupConfig === null || clusterParameterGroupConfig === void 0 ? void 0 : clusterParameterGroupConfig.parameterGroupName,
            associatedRoles: clusterAssociatedRoles.length > 0 ? clusterAssociatedRoles : undefined,
            deletionProtection: util_1.defaultDeletionProtection(props.deletionProtection, props.removalPolicy),
            enableIamDatabaseAuthentication: props.iamAuthentication,
            // Admin
            backupRetentionPeriod: (_o = (_m = props.backup) === null || _m === void 0 ? void 0 : _m.retention) === null || _o === void 0 ? void 0 : _o.toDays(),
            preferredBackupWindow: (_p = props.backup) === null || _p === void 0 ? void 0 : _p.preferredWindow,
            preferredMaintenanceWindow: props.preferredMaintenanceWindow,
            databaseName: props.defaultDatabaseName,
            enableCloudwatchLogsExports: props.cloudwatchLogsExports,
        };
    }
}
/**
 * Represents an imported database cluster.
 */
class ImportedDatabaseCluster extends DatabaseClusterBase {
    constructor(scope, id, attrs) {
        super(scope, id);
        this.clusterIdentifier = attrs.clusterIdentifier;
        const defaultPort = attrs.port ? ec2.Port.tcp(attrs.port) : undefined;
        this.connections = new ec2.Connections({
            securityGroups: attrs.securityGroups,
            defaultPort,
        });
        this.engine = attrs.engine;
        this._clusterEndpoint = (attrs.clusterEndpointAddress && attrs.port) ? new endpoint_1.Endpoint(attrs.clusterEndpointAddress, attrs.port) : undefined;
        this._clusterReadEndpoint = (attrs.readerEndpointAddress && attrs.port) ? new endpoint_1.Endpoint(attrs.readerEndpointAddress, attrs.port) : undefined;
        this._instanceIdentifiers = attrs.instanceIdentifiers;
        this._instanceEndpoints = (attrs.instanceEndpointAddresses && attrs.port)
            ? attrs.instanceEndpointAddresses.map(addr => new endpoint_1.Endpoint(addr, attrs.port))
            : undefined;
    }
    get clusterEndpoint() {
        if (!this._clusterEndpoint) {
            throw new Error('Cannot access `clusterEndpoint` of an imported cluster without an endpoint address and port');
        }
        return this._clusterEndpoint;
    }
    get clusterReadEndpoint() {
        if (!this._clusterReadEndpoint) {
            throw new Error('Cannot access `clusterReadEndpoint` of an imported cluster without a readerEndpointAddress and port');
        }
        return this._clusterReadEndpoint;
    }
    get instanceIdentifiers() {
        if (!this._instanceIdentifiers) {
            throw new Error('Cannot access `instanceIdentifiers` of an imported cluster without provided instanceIdentifiers');
        }
        return this._instanceIdentifiers;
    }
    get instanceEndpoints() {
        if (!this._instanceEndpoints) {
            throw new Error('Cannot access `instanceEndpoints` of an imported cluster without instanceEndpointAddresses and port');
        }
        return this._instanceEndpoints;
    }
}
/**
 * Create a clustered database with a given number of instances.
 *
 * @stability stable
 * @resource AWS::RDS::DBCluster
 */
class DatabaseCluster extends DatabaseClusterNew {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _d, _e, _f;
        super(scope, id, props);
        this.vpc = props.instanceProps.vpc;
        this.vpcSubnets = props.instanceProps.vpcSubnets;
        this.singleUserRotationApplication = props.engine.singleUserRotationApplication;
        this.multiUserRotationApplication = props.engine.multiUserRotationApplication;
        const credentials = util_1.renderCredentials(this, props.engine, props.credentials);
        const secret = credentials.secret;
        const cluster = new rds_generated_1.CfnDBCluster(this, 'Resource', {
            ...this.newCfnProps,
            // Admin
            masterUsername: credentials.username,
            masterUserPassword: (_d = credentials.password) === null || _d === void 0 ? void 0 : _d.toString(),
            // Encryption
            kmsKeyId: (_e = props.storageEncryptionKey) === null || _e === void 0 ? void 0 : _e.keyArn,
            storageEncrypted: props.storageEncryptionKey ? true : props.storageEncrypted,
        });
        this.clusterIdentifier = cluster.ref;
        // create a number token that represents the port of the cluster
        const portAttribute = core_1.Token.asNumber(cluster.attrEndpointPort);
        this.clusterEndpoint = new endpoint_1.Endpoint(cluster.attrEndpointAddress, portAttribute);
        this.clusterReadEndpoint = new endpoint_1.Endpoint(cluster.attrReadEndpointAddress, portAttribute);
        this.connections = new ec2.Connections({
            securityGroups: this.securityGroups,
            defaultPort: ec2.Port.tcp(this.clusterEndpoint.port),
        });
        cluster.applyRemovalPolicy((_f = props.removalPolicy) !== null && _f !== void 0 ? _f : core_1.RemovalPolicy.SNAPSHOT);
        if (secret) {
            this.secret = secret.attach(this);
        }
        setLogRetention(this, props);
        const createdInstances = createInstances(this, props, this.subnetGroup);
        this.instanceIdentifiers = createdInstances.instanceIdentifiers;
        this.instanceEndpoints = createdInstances.instanceEndpoints;
    }
    /**
     * Import an existing DatabaseCluster from properties.
     *
     * @stability stable
     */
    static fromDatabaseClusterAttributes(scope, id, attrs) {
        return new ImportedDatabaseCluster(scope, id, attrs);
    }
    /**
     * Adds the single user rotation of the master password to this cluster.
     *
     * @stability stable
     */
    addRotationSingleUser(options = {}) {
        var _d;
        if (!this.secret) {
            throw new Error('Cannot add single user rotation for a cluster without secret.');
        }
        const id = 'RotationSingleUser';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('A single user rotation was already added to this cluster.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: this.secret,
            application: this.singleUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
            ...options,
            excludeCharacters: (_d = options.excludeCharacters) !== null && _d !== void 0 ? _d : util_1.DEFAULT_PASSWORD_EXCLUDE_CHARS,
        });
    }
    /**
     * Adds the multi user rotation to this cluster.
     *
     * @stability stable
     */
    addRotationMultiUser(id, options) {
        var _d;
        if (!this.secret) {
            throw new Error('Cannot add multi user rotation for a cluster without secret.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            ...options,
            excludeCharacters: (_d = options.excludeCharacters) !== null && _d !== void 0 ? _d : util_1.DEFAULT_PASSWORD_EXCLUDE_CHARS,
            masterSecret: this.secret,
            application: this.multiUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
        });
    }
}
exports.DatabaseCluster = DatabaseCluster;
_b = JSII_RTTI_SYMBOL_1;
DatabaseCluster[_b] = { fqn: "@aws-cdk/aws-rds.DatabaseCluster", version: "1.112.0" };
/**
 * A database cluster restored from a snapshot.
 *
 * @stability stable
 * @resource AWS::RDS::DBInstance
 */
class DatabaseClusterFromSnapshot extends DatabaseClusterNew {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _d;
        super(scope, id, props);
        const cluster = new rds_generated_1.CfnDBCluster(this, 'Resource', {
            ...this.newCfnProps,
            snapshotIdentifier: props.snapshotIdentifier,
        });
        this.clusterIdentifier = cluster.ref;
        // create a number token that represents the port of the cluster
        const portAttribute = core_1.Token.asNumber(cluster.attrEndpointPort);
        this.clusterEndpoint = new endpoint_1.Endpoint(cluster.attrEndpointAddress, portAttribute);
        this.clusterReadEndpoint = new endpoint_1.Endpoint(cluster.attrReadEndpointAddress, portAttribute);
        this.connections = new ec2.Connections({
            securityGroups: this.securityGroups,
            defaultPort: ec2.Port.tcp(this.clusterEndpoint.port),
        });
        cluster.applyRemovalPolicy((_d = props.removalPolicy) !== null && _d !== void 0 ? _d : core_1.RemovalPolicy.SNAPSHOT);
        setLogRetention(this, props);
        const createdInstances = createInstances(this, props, this.subnetGroup);
        this.instanceIdentifiers = createdInstances.instanceIdentifiers;
        this.instanceEndpoints = createdInstances.instanceEndpoints;
    }
}
exports.DatabaseClusterFromSnapshot = DatabaseClusterFromSnapshot;
_c = JSII_RTTI_SYMBOL_1;
DatabaseClusterFromSnapshot[_c] = { fqn: "@aws-cdk/aws-rds.DatabaseClusterFromSnapshot", version: "1.112.0" };
/**
 * Sets up CloudWatch log retention if configured.
 * A function rather than protected member to prevent exposing ``DatabaseClusterBaseProps``.
 */
function setLogRetention(cluster, props) {
    if (props.cloudwatchLogsExports) {
        const unsupportedLogTypes = props.cloudwatchLogsExports.filter(logType => !props.engine.supportedLogTypes.includes(logType));
        if (unsupportedLogTypes.length > 0) {
            throw new Error(`Unsupported logs for the current engine type: ${unsupportedLogTypes.join(',')}`);
        }
        if (props.cloudwatchLogsRetention) {
            for (const log of props.cloudwatchLogsExports) {
                new logs.LogRetention(cluster, `LogRetention${log}`, {
                    logGroupName: `/aws/rds/cluster/${cluster.clusterIdentifier}/${log}`,
                    retention: props.cloudwatchLogsRetention,
                    role: props.cloudwatchLogsRetentionRole,
                });
            }
        }
    }
}
/**
 * Creates the instances for the cluster.
 * A function rather than a protected method on ``DatabaseClusterNew`` to avoid exposing
 * ``DatabaseClusterNew`` and ``DatabaseClusterBaseProps`` in the API.
 */
function createInstances(cluster, props, subnetGroup) {
    var _d, _e, _f, _g, _h;
    const instanceCount = props.instances != null ? props.instances : 2;
    if (core_1.Token.isUnresolved(instanceCount)) {
        throw new Error('The number of instances an RDS Cluster consists of cannot be provided as a deploy-time only value!');
    }
    if (instanceCount < 1) {
        throw new Error('At least one instance is required');
    }
    const instanceIdentifiers = [];
    const instanceEndpoints = [];
    const portAttribute = cluster.clusterEndpoint.port;
    const instanceProps = props.instanceProps;
    // Get the actual subnet objects so we can depend on internet connectivity.
    const internetConnected = instanceProps.vpc.selectSubnets(instanceProps.vpcSubnets).internetConnectivityEstablished;
    let monitoringRole;
    if (props.monitoringInterval && props.monitoringInterval.toSeconds()) {
        monitoringRole = props.monitoringRole || new aws_iam_1.Role(cluster, 'MonitoringRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('monitoring.rds.amazonaws.com'),
            managedPolicies: [
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonRDSEnhancedMonitoringRole'),
            ],
        });
    }
    const enablePerformanceInsights = instanceProps.enablePerformanceInsights
        || instanceProps.performanceInsightRetention !== undefined || instanceProps.performanceInsightEncryptionKey !== undefined;
    if (enablePerformanceInsights && instanceProps.enablePerformanceInsights === false) {
        throw new Error('`enablePerformanceInsights` disabled, but `performanceInsightRetention` or `performanceInsightEncryptionKey` was set');
    }
    const instanceType = (_d = instanceProps.instanceType) !== null && _d !== void 0 ? _d : ec2.InstanceType.of(ec2.InstanceClass.T3, ec2.InstanceSize.MEDIUM);
    const instanceParameterGroupConfig = (_e = instanceProps.parameterGroup) === null || _e === void 0 ? void 0 : _e.bindToInstance({});
    for (let i = 0; i < instanceCount; i++) {
        const instanceIndex = i + 1;
        const instanceIdentifier = props.instanceIdentifierBase != null ? `${props.instanceIdentifierBase}${instanceIndex}` :
            props.clusterIdentifier != null ? `${props.clusterIdentifier}instance${instanceIndex}` :
                undefined;
        const instance = new rds_generated_1.CfnDBInstance(cluster, `Instance${instanceIndex}`, {
            // Link to cluster
            engine: props.engine.engineType,
            engineVersion: (_f = props.engine.engineVersion) === null || _f === void 0 ? void 0 : _f.fullVersion,
            dbClusterIdentifier: cluster.clusterIdentifier,
            dbInstanceIdentifier: instanceIdentifier,
            // Instance properties
            dbInstanceClass: databaseInstanceType(instanceType),
            publiclyAccessible: (_g = instanceProps.publiclyAccessible) !== null && _g !== void 0 ? _g : (instanceProps.vpcSubnets && instanceProps.vpcSubnets.subnetType === ec2.SubnetType.PUBLIC),
            enablePerformanceInsights: enablePerformanceInsights || instanceProps.enablePerformanceInsights,
            performanceInsightsKmsKeyId: (_h = instanceProps.performanceInsightEncryptionKey) === null || _h === void 0 ? void 0 : _h.keyArn,
            performanceInsightsRetentionPeriod: enablePerformanceInsights
                ? (instanceProps.performanceInsightRetention || props_1.PerformanceInsightRetention.DEFAULT)
                : undefined,
            // This is already set on the Cluster. Unclear to me whether it should be repeated or not. Better yes.
            dbSubnetGroupName: subnetGroup.subnetGroupName,
            dbParameterGroupName: instanceParameterGroupConfig === null || instanceParameterGroupConfig === void 0 ? void 0 : instanceParameterGroupConfig.parameterGroupName,
            monitoringInterval: props.monitoringInterval && props.monitoringInterval.toSeconds(),
            monitoringRoleArn: monitoringRole && monitoringRole.roleArn,
            autoMinorVersionUpgrade: props.instanceProps.autoMinorVersionUpgrade,
            allowMajorVersionUpgrade: props.instanceProps.allowMajorVersionUpgrade,
            deleteAutomatedBackups: props.instanceProps.deleteAutomatedBackups,
        });
        // For instances that are part of a cluster:
        //
        //  Cluster DESTROY or SNAPSHOT -> DESTROY (snapshot is good enough to recreate)
        //  Cluster RETAIN              -> RETAIN (otherwise cluster state will disappear)
        instance.applyRemovalPolicy(util_1.helperRemovalPolicy(props.removalPolicy));
        // We must have a dependency on the NAT gateway provider here to create
        // things in the right order.
        instance.node.addDependency(internetConnected);
        instanceIdentifiers.push(instance.ref);
        instanceEndpoints.push(new endpoint_1.Endpoint(instance.attrEndpointAddress, portAttribute));
    }
    return { instanceEndpoints, instanceIdentifiers };
}
/**
 * Turn a regular instance type into a database instance type
 */
function databaseInstanceType(instanceType) {
    return 'db.' + instanceType.toString();
}
//# sourceMappingURL=data:application/json;base64,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