# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['authup',
 'authup.client',
 'authup.client.resource_clients',
 'authup.plugins',
 'authup.schemas',
 'authup.tests',
 'authup.tests.client',
 'authup.tests.plugins']

package_data = \
{'': ['*']}

install_requires = \
['asgiref>=3.6.0,<4.0.0', 'httpx>=0.23.2,<0.24.0', 'pydantic>=1.10.4,<2.0.0']

extras_require = \
{'all': ['fastapi>=0.87.0', 'requests>=2.26.0,<3.0.0', 'flask>=2.0.0,<3.0.0'],
 'fastapi': ['fastapi>=0.87.0'],
 'flask': ['flask>=2.0.0,<3.0.0'],
 'requests': ['requests>=2.26.0,<3.0.0']}

setup_kwargs = {
    'name': 'authup',
    'version': '0.5.0',
    'description': 'Python plugins for the Authup authentication and authorization framework',
    'long_description': '[![CI](https://github.com/migraf/authup-py/actions/workflows/main.yml/badge.svg)](https://github.com/migraf/authup-py/actions/workflows/main.yml)\n[![codecov](https://codecov.io/gh/migraf/authup-py/branch/main/graph/badge.svg?token=qILJEFdh8I)](https://codecov.io/gh/migraf/authup-py)\n![PyPI - Python Version](https://img.shields.io/pypi/pyversions/authup)\n![PyPI - Downloads](https://img.shields.io/pypi/dw/authup)\n[![Maintainability](https://api.codeclimate.com/v1/badges/520401d6c07170a6e413/maintainability)](https://codeclimate.com/github/migraf/authup-py/maintainability)\n[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)\n\n# Authup Python Plugins\n\nThis repository contains python plugins for using the [Authup](https://authup.org) authentication and authorization\nframework in the python language.\nThe plugins are used to integrate Authup with different python frameworks and libraries.\n\n## Supported Python frameworks\n\n### Client\n| Plugin                                      | Extra        | Sync | Async |\n|---------------------------------------------|--------------|:----:|------:|\n| [httpx](https://github.com/encode/httpx)    |              |  ✅   |     ✅ |\n| [requests](https://github.com/psf/requests) | `[requests]` |  ✅   |     ❌ |\n\n### Server\n\n| Plugin                                                        | Extra       | Sync | Async | Middleware | User |\n|---------------------------------------------------------------|-------------|:----:|------:|------------|------|\n| [FastApi](https://fastapi.tiangolo.com/)                      | `[fastapi]` |  ✅   |     ✅ | ✅          | ✅    |\n| [ASGI](https://asgi.readthedocs.io/en/latest/specs/main.html) | `[asgi]`    |  ❌   |     ✅ | ✅          | ✅    |\n| [Flask](https://flask.palletsprojects.com/en/2.2.x/)          | `[flask]`   |  ⏳   |     ⏳ | ⏳          | ⏳    |\n\nTable of Contents\n=================\n\n* [Authup Python Plugins](#authup-python-plugins)\n   * [Supported Python frameworks](#supported-python-frameworks)\n      * [Client](#client)\n      * [Server](#server)\n   * [Installation](#installation)\n      * [Extra dependencies](#extra-dependencies)\n   * [How to use](#how-to-use)\n      * [httpx](#httpx)\n      * [requests](#requests)\n      * [ASGI Middleware](#asgi-middleware)\n         * [Optional user injection](#optional-user-injection)\n      * [FastAPI Dependency](#fastapi-dependency)\n         * [Basic user dependency](#basic-user-dependency)\n         * [Require permissions](#require-permissions)\n   * [How to develop](#how-to-develop)\n      * [Install](#install)\n      * [Test](#test)\n\n\n## Installation\n\nThe plugins are available via [PyPi](https://pypi.org/project/authup-py/).\n\n```bash\npip install authup-py\n```\n\n### Extra dependencies\nThe plugin for the project\'s base library [httpx](https://github.com/encode/httpx) needs no extra dependencies. To\nuse the additional plugins for other libraries, you need to install with the corresponding extra i.e. for `requests`:\n\n```bash\npip install authup-py[requests]\n```\n\n## How to use\nAll the plugins share the underlying `Authup` class. The class is initialized with the url of the Authup server and\nthe credentials you would like to use (username/password or robot_id/secret).    \nThe class provides both sync and async methods for the different authentication and authorization flows.\n\n```python\n\nfrom authup import Authup\n\nauthup = Authup(\n    url="https://authup.org",\n    username="username",\n    password="password"\n)\n\nauthup_robot = Authup(\n    url="https://authup.org",\n    robot_id="robot",\n    robot_secret="secret"\n)\n\n```\n\nThe following plugins all expect the same arguments as the `Authup` class with the addition of the\napp as a first argument for server side libraries (e.g. FastApi, Flask).\n\n### httpx\nFor synchronously using the plugin with [httpx](https://github.com/encode/httpx) , you can use the `AuthupHttpx` class and pass an instance to your\n`httpx.Client` or a basic `httpx.Request` as the `auth` parameter:\n\n```python\nimport httpx\nfrom authup.plugins.httpx import AuthupHttpx\n\nauthup = AuthupHttpx(\n    url="https://authup.org",\n    username="username",\n    password="password",\n)\n\n# Use the authup instance as the auth parameter for the httpx client\nclient = httpx.Client(auth=authup)\n\nwith client:\n    response = client.get("https://authup.org")\n    print(response.status_code)\n\n\n# Use the authup instance as the auth parameter for a top level request function\nrequest = httpx.get("https://authup.org", auth=authup)\n\n```\n\nIt works the same way for the asynchronous httpx client:\n\n```python\nimport httpx\nfrom authup.plugins.httpx import AuthupHttpxAsync\n\nauthup = AuthupHttpxAsync(\n    url="https://authup.org",\n    username="username",\n    password="password",\n)\n\nasync with httpx.AsyncClient(auth=authup) as client:\n    response = await client.get("https://authup.org")\n    print(response.status_code)\n\n```\n\n### requests\nSince [requests](https://github.com/psf/requests) is a synchronous library, the plugin is also synchronous. You can use the `AuthupRequests` class and\nuse it with the `requests.Session` or the `requests.request` functions:\n> **Note**\n> Requires the `requests` extra to be installed. `pip install authup-py[requests]`\n\n```python\nimport requests\nfrom authup.plugins.requests import AuthupRequests\n\nauthup = AuthupRequests(\n    url="https://authup.org",\n    username="username",\n    password="password",\n)\n\n# Use the authup instance as the auth parameter for the requests session\nwith requests.Session() as session:\n    session.auth = authup\n    response = session.get("https://authup.org")\n    print(response.status_code)\n\n# Use the authup instance as the auth parameter for a top level request function\nresponse = requests.get("https://authup.org", auth=authup)\nprint(response.status_code)\n\n```\n\n### ASGI Middleware\n\nThe `AuthupASGIMiddleware` class can be used as an ASGI middleware for any ASGI framework (i.e. FastAPI, Starlette). \nThe middleware will check the incoming requests for a valid token and otherwise return a 401 response. If you pass the\noptional `user` parameter, the middleware will inject the user object into the request scope (`r.state.user`).\n\nThe first argument is the ASGI application and the second argument is the URL of the authup instance.\n> **Note**\n> Requires the `asgi` extra to be installed. `pip install authup-py[asgi]`\n\nThe following shows a simple example for using the middleware with a FastAPI application but it should work with any ASGI framework.\n\n> **Note**\n> Expects a running authup instance available at the given URL.\n> \n```python\nfrom fastapi import FastAPI\nfrom authup.plugins.asgi import AuthupASGIMiddleware\n\napp = FastAPI()\n\nauthup_url = "https://authup.org"  # change to your authup instance\n@app.get("/test")\nasync def test():\n    return {"message": "Hello World"}\n\n# register the middleware pass the authup url as argument\napp.add_middleware(AuthupASGIMiddleware, authup_url=authup_url)\n\n```\nNow you can access the `/test` endpoint without a token and will receive a 401 response. When using a valid token, you will receive the expected response.\n\n```python\nimport httpx\nfrom authup.plugins.httpx import AuthupHttpx\n\n# no token or invalid token raises 401\nresponse = httpx.get("http://localhost:8000/test") # 401\nprint(response.status_code)\n\n# valid token receives the expected response\nauthup = AuthupHttpx(\n    url="https://authup.org",\n    username="username",\n    password="password",\n)\n\nresponse = httpx.get("http://localhost:8000/test", auth=authup) # 200\nprint(response.status_code)\n\n```\n\n#### Optional user injection\nSet the `user` parameter to `True` when adding the middleware to your ASGI application:\n\n```python\nfrom fastapi import FastAPI, Request\nfrom authup.plugins.asgi import AuthupASGIMiddleware\n\napp = FastAPI()\n\nauthup_url = "https://authup.org"  # change to your authup instance\n@app.get("/test-user")\nasync def test(request: Request):\n    return {"user": request.state.user}\n\n# register the middleware pass the authup url as argument\napp.add_middleware(AuthupASGIMiddleware, authup_url=authup_url, user=True)\n\n```\n\nCalling the `/test-user` endpoint without a token will return a 401 response. When using a valid token, the user object \nwill be injected into the request scope, and you will receive the expected response containing your user.\n\n### FastAPI Dependency\nThe `AuthupUser` class can be used as a FastAPI dependency. \nIt will check the incoming requests for a valid token and otherwise return a 401 response. If the token is valid a user object\nwill be available in the dependency call.\n\n#### Basic user dependency\nThe following shows a simple example for using the dependency with a FastAPI application that will return the user\nobject obtained from the token.\n\n```python\nfrom fastapi import FastAPI, Depends\nfrom authup.plugins.fastapi import AuthupUser\nfrom authup import User\n\n\napp = FastAPI()\n\nuser_dependency = AuthupUser(url="http://localhost:3010")\n\n@app.get("/test")\nasync def user_test(user: User = Depends(user_dependency)):\n    return {"user": user.dict()}\n\n```\n\n#### Require permissions\nYou can also require specific permissions for the user. The following example will only allow users with the \n`client_add` permission and a power level of over `100`. Otherwise, a 401 response will be returned.\n\n```python\nfrom fastapi import FastAPI, Depends\nfrom authup.plugins.fastapi import AuthupUser\nfrom authup import User\nfrom authup.permissions import Permission\n\npermissions = [\n        Permission(name="client_add", inverse=False, power=100),\n    ]\n\nrequired_permissions = AuthupUser(\n    url="http://localhost:3010",\n    permissions=permissions,\n)\n\napp = FastAPI()\n\n@app.get("/test")\nasync def user_test(user: User = Depends(required_permissions)):\n    return {"user": user.dict()}\n\n```\n\n\n\n## How to develop\n\n### Install\n\nRequires [poetry](https://python-poetry.org/) and [pre-commit](https://pre-commit.com/) and python 3.7+.\n\n```shell\npoetry install --with dev --all-extras\n```\n\nInstall pre-commit hooks\n\n```shell\npoetry run pre-commit install\n```\n\n### Test\n\n```shell\npoetry run pytest\n```\n\n',
    'author': 'Michael Graf',
    'author_email': 'michael.graf3110@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://authup.org',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
