"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NIST80053Checks = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const monocdk_1 = require("monocdk");
const common_1 = require("../common");
const apigw_1 = require("./rules/apigw");
const autoscaling_1 = require("./rules/autoscaling");
const cloudtrail_1 = require("./rules/cloudtrail");
const cloudwatch_1 = require("./rules/cloudwatch");
const codebuild_1 = require("./rules/codebuild");
const dms_1 = require("./rules/dms");
const dynamodb_1 = require("./rules/dynamodb");
const ec2_1 = require("./rules/ec2");
const efs_1 = require("./rules/efs");
const elasticache_1 = require("./rules/elasticache");
const elb_1 = require("./rules/elb");
const emr_1 = require("./rules/emr");
const iam_1 = require("./rules/iam");
const lambda_1 = require("./rules/lambda");
const opensearch_1 = require("./rules/opensearch");
const rds_1 = require("./rules/rds");
const redshift_1 = require("./rules/redshift");
const s3_1 = require("./rules/s3");
const sagemaker_1 = require("./rules/sagemaker");
const sns_1 = require("./rules/sns");
/**
 * Check for NIST 800-53 compliance.
 *
 * Based on the NIST 800-53 AWS operational best practices: https://docs.aws.amazon.com/config/latest/developerguide/operational-best-practices-for-nist-800-53_rev_4.html
 *
 * @stability stable
 */
class NIST80053Checks extends common_1.NagPack {
    /**
     * All aspects can visit an IConstruct.
     *
     * @stability stable
     */
    visit(node) {
        var _b;
        if (node instanceof monocdk_1.CfnResource) {
            // Get ignores metadata if it exists
            const ignores = (_b = node.getMetadata('cdk_nag')) === null || _b === void 0 ? void 0 : _b.rules_to_suppress;
            this.checkAPIGW(node, ignores);
            this.checkAutoScaling(node, ignores);
            this.checkCloudTrail(node, ignores);
            this.checkCloudWatch(node, ignores);
            this.checkCodeBuild(node, ignores);
            this.checkDMS(node, ignores);
            this.checkDynamoDB(node, ignores);
            this.checkEC2(node, ignores);
            this.checkEFS(node, ignores);
            this.checkElastiCache(node, ignores);
            this.checkELB(node, ignores);
            this.checkEMR(node, ignores);
            this.checkIAM(node, ignores);
            this.checkLambda(node, ignores);
            this.checkOpenSearch(node, ignores);
            this.checkRDS(node, ignores);
            this.checkRedshift(node, ignores);
            this.checkS3(node, ignores);
            this.checkSageMaker(node, ignores);
            this.checkSNS(node, ignores);
        }
    }
    /**
     * Check API Gateway Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkAPIGW(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-APIGWCacheEnabledAndEncrypted') &&
            !apigw_1.nist80053APIGWCacheEnabledAndEncrypted(node)) {
            const ruleId = 'NIST.800.53-APIGWCacheEnabledAndEncrypted';
            const info = 'The API Gateway stage does not have caching enabled and encrypted for all methods - (Control IDs: SC-13, SC-28).';
            const explanation = "To help protect data at rest, ensure encryption is enabled for your API Gateway stage's cache. Because sensitive data can be captured for the API method, enable encryption at rest to help protect that data.";
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-APIGWExecutionLoggingEnabled') &&
            !apigw_1.nist80053APIGWExecutionLoggingEnabled(node)) {
            const ruleId = 'NIST.800.53-APIGWExecutionLoggingEnabled';
            const info = 'The API Gateway stage does not have execution logging enabled for all methods - (Control IDs: AU-2(a)(d), AU-3, AU-12(a)(c)).';
            const explanation = 'API Gateway logging displays detailed views of users who accessed the API and the way they accessed the API. This insight enables visibility of user activities.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Auto Scaling Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkAutoScaling(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-AutoScalingHealthChecks') &&
            !autoscaling_1.nist80053AutoScalingHealthChecks(node)) {
            const ruleId = 'NIST.800.53-AutoScalingHealthChecks';
            const info = 'The Auto Scaling group utilizes a load balancer and does not have an ELB health check configured - (Control IDs: SC-5).';
            const explanation = 'The Elastic Load Balancer (ELB) health checks for Amazon Elastic Compute Cloud (Amazon EC2) Auto Scaling groups support maintenance of adequate capacity and availability.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check CloudTrail Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkCloudTrail(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-CloudTrailCloudWatchLogsEnabled') &&
            !cloudtrail_1.nist80053CloudTrailCloudWatchLogsEnabled(node)) {
            const ruleId = 'NIST.800.53-CloudTrailCloudWatchLogsEnabled';
            const info = 'The trail does not have CloudWatch logs enabled - (Control IDs: AC-2(4), AC-2(g), AU-2(a)(d), AU-3, AU-6(1)(3), AU-7(1), AU-12(a)(c), CA-7(a)(b), SI-4(2), SI-4(4), SI-4(5), SI-4(a)(b)(c)).';
            const explanation = 'Use Amazon CloudWatch to centrally collect and manage log event activity. Inclusion of AWS CloudTrail data provides details of API call activity within your AWS account.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-CloudTrailEncryptionEnabled') &&
            !cloudtrail_1.nist80053CloudTrailEncryptionEnabled(node)) {
            const ruleId = 'NIST.800.53-CloudTrailEncryptionEnabled';
            const info = 'The trail does not have a KMS key ID or have encryption enabled - (Control ID: AU-9).';
            const explanation = 'Because sensitive data may exist and to help protect data at rest, ensure encryption is enabled for your AWS CloudTrail trails.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-CloudTrailLogFileValidationEnabled') &&
            !cloudtrail_1.nist80053CloudTrailLogFileValidationEnabled(node)) {
            const ruleId = 'NIST.800.53-CloudTrailLogFileValidationEnabled';
            const info = 'The trail does not have log file validation enabled - (Control ID: AC-6).';
            const explanation = 'Utilize AWS CloudTrail log file validation to check the integrity of CloudTrail logs. Log file validation helps determine if a log file was modified or deleted or unchanged after CloudTrail delivered it. This feature is built using industry standard algorithms: SHA-256 for hashing and SHA-256 with RSA for digital signing. This makes it computationally infeasible to modify, delete or forge CloudTrail log files without detection.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check CloudWatch Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkCloudWatch(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-CloudWatchAlarmAction') &&
            !cloudwatch_1.nist80053CloudWatchAlarmAction(node)) {
            const ruleId = 'NIST.800.53-CloudWatchAlarmAction';
            const info = 'The CloudWatch alarm does not have at least one alarm action, one INSUFFICIENT_DATA action, or one OK action enabled - (Control IDs: AC-2(4), AU-6(1)(3), AU-7(1), CA-7(a)(b), IR-4(1), SI-4(2), SI-4(4), SI-4(5), SI-4(a)(b)(c)).';
            const explanation = 'Amazon CloudWatch alarms alert when a metric breaches the threshold for a specified number of evaluation periods. The alarm performs one or more actions based on the value of the metric or expression relative to a threshold over a number of time periods.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-CloudWatchLogGroupEncrypted') &&
            !cloudwatch_1.nist80053CloudWatchLogGroupEncrypted(node)) {
            const ruleId = 'NIST.800.53-CloudWatchLogGroupEncrypted';
            const info = 'The CloudWatch Log Group is not encrypted with an AWS KMS key - (Control IDs: AU-9, SC-13, SC-28).';
            const explanation = 'To help protect sensitive data at rest, ensure encryption is enabled for your Amazon CloudWatch Log Groups.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check CodeBuild Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkCodeBuild(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-CodeBuildCheckEnvVars') &&
            !codebuild_1.nist80053CodeBuildCheckEnvVars(node)) {
            const ruleId = 'NIST.800.53-CodeBuildCheckEnvVars';
            const info = 'The CodeBuild environment stores sensitive credentials (such as AWS_ACCESS_KEY_ID and/or AWS_SECRET_ACCESS_KEY) as plaintext environment variables - (Control IDs: AC-6, IA-5(7), SA-3(a)).';
            const explanation = 'Do not store these variables in clear text. Storing these variables in clear text leads to unintended data exposure and unauthorized access.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-CodeBuildURLCheck') &&
            !codebuild_1.nist80053CodeBuildURLCheck(node)) {
            const ruleId = 'NIST.800.53-CodeBuildURLCheck';
            const info = 'The CodeBuild project which utilizes either a GitHub or BitBucket source repository does not utilize OAUTH - (Control ID: SA-3(a)).';
            const explanation = 'OAUTH is the most secure method of authenticating your CodeBuild application. Use OAuth instead of personal access tokens or a user name and password to grant authorization for accessing GitHub or Bitbucket repositories.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check DMS Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkDMS(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-DMSReplicationNotPublic') &&
            !dms_1.nist80053DMSReplicationNotPublic(node)) {
            const ruleId = 'NIST.800.53-DMSReplicationNotPublic';
            const info = 'The DMS replication instance is public - (Control IDs: AC-3).';
            const explanation = 'DMS replication instances can contain sensitive information and access control is required for such accounts.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check DynamoDB Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkDynamoDB(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-DynamoDBPITREnabled') &&
            !dynamodb_1.nist80053DynamoDBPITREnabled(node)) {
            const ruleId = 'NIST.800.53-DynamoDBPITREnabled';
            const info = 'The DynamoDB table does not have Point-in-time Recovery enabled - (Control IDs: CP-9(b), CP-10, SI-12).';
            const explanation = 'The recovery maintains continuous backups of your table for the last 35 days.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check EC2 Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkEC2(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-EC2CheckDefaultSecurityGroupClosed') &&
            !ec2_1.nist80053EC2CheckDefaultSecurityGroupClosed(node)) {
            const ruleId = 'NIST.800.53-EC2CheckDefaultSecurityGroupClosed';
            const info = "The VPC's default security group allows inbound or outbound traffic - (Control IDs: AC-4, SC-7, SC-7(3)).";
            const explanation = 'When creating a VPC through CloudFormation, the default security group will always be open. Therefore it is important to always close the default security group after stack creation whenever a VPC is created. Restricting all the traffic on the default security group helps in restricting remote access to your AWS resources.';
            monocdk_1.Annotations.of(node).addWarning(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-EC2CheckDetailedMonitoring') &&
            !ec2_1.nist80053EC2CheckDetailedMonitoring(node)) {
            const ruleId = 'NIST.800.53-EC2CheckDetailedMonitoring';
            const info = 'The EC2 instance does not have detailed monitoring enabled - (Control IDs: CA-7(a)(b), SI-4(2), SI-4(a)(b)(c)).';
            const explanation = 'Detailed monitoring provides additional monitoring information (such as 1-minute period graphs) on the AWS console.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-EC2CheckInsideVPC') &&
            !ec2_1.nist80053EC2CheckInsideVPC(node)) {
            const ruleId = 'NIST.800.53-EC2CheckInsideVPC';
            const info = 'The EC2 instance is not within a VPC - (Control IDs: AC-4, SC-7, SC-7(3)).';
            const explanation = 'Because of their logical isolation, domains that reside within an Amazon VPC have an extra layer of security when compared to domains that use public endpoints.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-EC2CheckNoPublicIPs') &&
            !ec2_1.nist80053EC2CheckNoPublicIPs(node)) {
            const ruleId = 'NIST.800.53-EC2CheckNoPublicIPs';
            const info = 'The EC2 instance is associated with a public IP address - (Control IDs: AC-4, AC-6, AC-21(b), SC-7, SC-7(3)). ';
            const explanation = 'Amazon EC2 instances can contain sensitive information and access control is required for such resources.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-EC2CheckSSHRestricted') &&
            !ec2_1.nist80053EC2CheckSSHRestricted(node)) {
            const ruleId = 'NIST.800.53-EC2CheckSSHRestricted';
            const info = 'The Security Group allows unrestricted SSH access - (Control IDs: AC-4, SC-7, SC-7(3)).';
            const explanation = 'Not allowing ingress (or remote) traffic from 0.0.0.0/0 or ::/0 to port 22 on your resources helps to restrict remote access.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-EC2CheckCommonPortsRestricted') &&
            !ec2_1.nist80053EC2CheckCommonPortsRestricted(node)) {
            const ruleId = 'NIST.800.53-EC2CheckCommonPortsRestricted';
            const info = 'The EC2 instance allows unrestricted inbound IPv4 TCP traffic on common ports (20, 21, 3389, 3306, 4333) - (Control IDs: AC-4, CM-2, SC-7, SC-7(3)).';
            const explanation = 'Not restricting access to ports to trusted sources can lead to attacks against the availability, integrity and confidentiality of systems. By default, common ports which should be restricted include port numbers 20, 21, 3389, 3306, and 4333.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check EFS Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkEFS(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-EFSEncrypted') &&
            !efs_1.nist80053EFSEncrypted(node)) {
            const ruleId = 'NIST.800.53-EFSEncrypted';
            const info = 'The EFS does not have encryption at rest enabled - (Control IDs: SC-13, SC-28).';
            const explanation = 'Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon Elastic File System (EFS).';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check ElastiCache Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkElastiCache(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-ElastiCacheRedisClusterAutomaticBackup') &&
            !elasticache_1.nist80053ElastiCacheRedisClusterAutomaticBackup(node)) {
            const ruleId = 'NIST.800.53-ElastiCacheRedisClusterAutomaticBackup';
            const info = 'The ElastiCache Redis cluster does not retain automatic backups for at least 15 days - (Control IDs: CP-9(b), CP-10, SI-12).';
            const explanation = 'Automatic backups can help guard against data loss. If a failure occurs, you can create a new cluster, which restores your data from the most recent backup.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Elastic Load Balancer Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkELB(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-ALBHttpDropInvalidHeaderEnabled') &&
            !elb_1.nist80053ALBHttpDropInvalidHeaderEnabled(node)) {
            const ruleId = 'NIST.800.53-ALBHttpDropInvalidHeaderEnabled';
            const info = 'The ALB does not have invalid HTTP header dropping enabled - (Control ID: AC-17(2)).';
            const explanation = 'Ensure that your Application Load Balancers (ALB) are configured to drop http headers. Because sensitive data can exist, enable encryption in transit to help protect that data.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-ALBHttpToHttpsRedirection') &&
            !elb_1.nist80053ALBHttpToHttpsRedirection(node)) {
            const ruleId = 'NIST.800.53-ALBHttpToHttpsRedirection';
            const info = "The ALB's HTTP listeners are not configured to redirect to HTTPS - (Control IDs: AC-17(2), SC-7, SC-8, SC-8(1), SC-13, SC-23).";
            const explanation = 'To help protect data in transit, ensure that your Application Load Balancer automatically redirects unencrypted HTTP requests to HTTPS. Because sensitive data can exist, enable encryption in transit to help protect that data.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-ELBCrossZoneBalancing') &&
            !elb_1.nist80053ELBCrossZoneBalancing(node)) {
            const ruleId = 'NIST.800.53-ELBCrossZoneBalancing';
            const info = 'The CLB does not balance traffic between at least 2 Availability Zones - (Control IDs: SC-5, CP-10).';
            const explanation = 'The cross-zone load balancing reduces the need to maintain equivalent numbers of instances in each enabled availability zone.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-ELBDeletionProtectionEnabled') &&
            !elb_1.nist80053ELBDeletionProtectionEnabled(node)) {
            const ruleId = 'NIST.800.53-ELBDeletionProtectionEnabled';
            const info = 'The ALB, NLB, or GLB does not have deletion protection enabled - (Control IDs: CM-2, CP-10).';
            const explanation = 'Use this feature to prevent your load balancer from being accidentally or maliciously deleted, which can lead to loss of availability for your applications.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-ELBListenersUseSSLOrHTTPS') &&
            !elb_1.nist80053ELBListenersUseSSLOrHTTPS(node)) {
            const ruleId = 'NIST.800.53-ELBListenersUseSSLOrHTTPS';
            const info = 'The CLB does not restrict its listeners to only the SSL and HTTPS protocols - (Control IDs: AC-17(2), SC-7, SC-8, SC-8(1), SC-23).';
            const explanation = 'Because sensitive data can exist, enable encryption in transit to help protect that data.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-ELBLoggingEnabled') &&
            !elb_1.nist80053ELBLoggingEnabled(node)) {
            const ruleId = 'NIST.800.53-ELBLoggingEnabled';
            const info = 'The ELB does not have logging enabled - (Control ID: AU-2(a)(d)).';
            const explanation = "Elastic Load Balancing activity is a central point of communication within an environment. Ensure ELB logging is enabled. The collected data provides detailed information about requests sent to The ELB. Each log contains information such as the time the request was received, the client's IP address, latencies, request paths, and server responses.";
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-ELBUseACMCerts') &&
            !elb_1.nist80053ELBUseACMCerts(node)) {
            const ruleId = 'NIST.800.53-ELBUseACMCerts';
            const info = 'The CLB does not utilize an SSL certificate provided by ACM (Amazon Certificate Manager) - (Control IDs: AC-17(2), SC-7, SC-8, SC-8(1), SC-13).';
            const explanation = 'Because sensitive data can exist and to help protect data at transit, ensure encryption is enabled for your Elastic Load Balancing. Use AWS Certificate Manager to manage, provision and deploy public and private SSL/TLS certificates with AWS services and internal resources.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check EMR Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkEMR(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-EMRKerberosEnabled') &&
            !emr_1.nist80053EMRKerberosEnabled(node)) {
            const ruleId = 'NIST.800.53-EMRKerberosEnabled';
            const info = 'The EMR cluster does not have Kerberos enabled - (Control IDs: AC-2(j), AC-3, AC-5c, AC-6).';
            const explanation = 'The access permissions and authorizations can be managed and incorporated with the principles of least privilege and separation of duties, by enabling Kerberos for Amazon EMR clusters.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check IAM Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkIAM(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-IAMGroupMembershipCheck') &&
            !iam_1.nist80053IAMGroupMembership(node)) {
            const ruleId = 'NIST.800.53-IAMGroupMembershipCheck';
            const info = 'The IAM user does not belong to any group(s) - (Control IDs: AC-2(1), AC-2(j), AC-3, and AC-6).';
            const explanation = 'AWS Identity and Access Management (IAM) can help you restrict access permissions and authorizations, by ensuring IAM users are members of at least one group. Allowing users more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-IAMNoInlinePolicyCheck') &&
            !iam_1.nist80053IAMNoInlinePolicy(node)) {
            const ruleId = 'NIST.800.53-IAMNoInlinePolicyCheck';
            const info = 'The IAM Group, User, or Role contains an inline policy - (Control ID: AC-6).';
            const explanation = 'AWS recommends to use managed policies instead of inline policies. The managed policies allow reusability, versioning and rolling back, and delegating permissions management.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-IAMPolicyNoStatementsWithAdminAccess') &&
            !iam_1.nist80053IAMPolicyNoStatementsWithAdminAccess(node)) {
            const ruleId = 'NIST.800.53-IAMPolicyNoStatementsWithAdminAccess';
            const info = 'The IAM policy grants admin access - (Control IDs: AC-2(1), AC-2(j), AC-3, AC-6).';
            const explanation = 'AWS Identity and Access Management (IAM) can help you incorporate the principles of least privilege and separation of duties with access permissions and authorizations, restricting policies from containing "Effect": "Allow" with "Action": "*" over "Resource": "*". Allowing users to have more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-IAMUserNoPoliciesCheck') &&
            !iam_1.nist80053IAMUserNoPolicies(node)) {
            const ruleId = 'NIST.800.53-IAMUserNoPoliciesCheck';
            const info = 'The IAM policy is attached at the user level - (Control IDs: AC-2(j), AC-3, AC-5c, AC-6).';
            const explanation = 'Assigning privileges at the group or the role level helps to reduce opportunity for an identity to receive or retain excessive privileges.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Lambda Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkLambda(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-LambdaFunctionsInsideVPC') &&
            !lambda_1.nist80053LambdaFunctionsInsideVPC(node)) {
            const ruleId = 'NIST.800.53-LambdaFunctionsInsideVPC';
            const info = 'The Lambda function is not VPC enabled - (Control IDs: AC-4, SC-7, SC-7(3)).';
            const explanation = 'Because of their logical isolation, domains that reside within an Amazon VPC have an extra layer of security when compared to domains that use public endpoints.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check OpenSearch Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkOpenSearch(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-OpenSearchEncryptedAtRest') &&
            !opensearch_1.nist80053OpenSearchEncryptedAtRest(node)) {
            const ruleId = 'NIST.800.53-OpenSearchEncryptedAtRest';
            const info = 'The OpenSearch Service domain does not have encryption at rest enabled - (Control IDs: SC-13, SC-28).';
            const explanation = 'Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon OpenSearch Service (OpenSearch Service) domains.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-OpenSearchNodeToNodeEncrypted') &&
            !opensearch_1.nist80053OpenSearchNodeToNodeEncrypted(node)) {
            const ruleId = 'NIST.800.53-OpenSearchNodeToNodeEncrypted';
            const info = 'The OpenSearch Service domain does not have node-to-node encryption enabled - (Control IDs: SC-7, SC-8, SC-8(1)).';
            const explanation = 'Because sensitive data can exist, enable encryption in transit to help protect that data within your Amazon OpenSearch Service (OpenSearch Service) domains.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-OpenSearchRunningWithinVPC') &&
            !opensearch_1.nist80053OpenSearchRunningWithinVPC(node)) {
            const ruleId = 'NIST.800.53-OpenSearchRunningWithinVPC';
            const info = 'The OpenSearch Service domain is not running within a VPC - (Control IDs: AC-4, SC-7, SC-7(3)).';
            const explanation = 'VPCs help secure your AWS resources and provide an extra layer of protection.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Amazon RDS Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkRDS(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-RDSEnhancedMonitoringEnabled') &&
            !rds_1.nist80053RDSEnhancedMonitoringEnabled(node)) {
            const ruleId = 'NIST.800.53-RDSEnhancedMonitoringEnabled';
            const info = 'The RDS DB instance does not enhanced monitoring enabled - (Control ID: CA-7(a)(b)).';
            const explanation = 'Enable enhanced monitoring to help monitor Amazon RDS availability. This provides detailed visibility into the health of your Amazon RDS database instances.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-RDSInstanceBackupEnabled') &&
            !rds_1.nist80053RDSInstanceBackupEnabled(node)) {
            const ruleId = 'NIST.800.53-RDSInstanceBackupEnabled';
            const info = 'The RDS DB instance does not have backups enabled - (Control IDs: CP-9(b), CP-10, SI-12).';
            const explanation = 'The backup feature of Amazon RDS creates backups of your databases and transaction logs.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-RDSInstanceDeletionProtectionEnabled') &&
            !rds_1.nist80053RDSInstanceDeletionProtectionEnabled(node)) {
            const ruleId = 'NIST.800.53-RDSInstanceDeletionProtectionEnabled';
            const info = 'The RDS DB instance or Aurora DB cluster does not have deletion protection enabled - (Control ID: SC-5).';
            const explanation = 'Ensure Amazon Relational Database Service (Amazon RDS) instances and clusters have deletion protection enabled. Use deletion protection to prevent your Amazon RDS DB instances and clusters from being accidentally or maliciously deleted, which can lead to loss of availability for your applications.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-RDSInstanceMultiAzSupport') &&
            !rds_1.nist80053RDSInstanceMultiAZSupport(node)) {
            const ruleId = 'NIST.800.53-RDSInstanceMultiAzSupport';
            const info = 'The non-Aurora RDS DB instance does not have multi-AZ support enabled - (Control IDs: CP-10, SC-5, SC-36).';
            const explanation = 'Multi-AZ support in Amazon Relational Database Service (Amazon RDS) provides enhanced availability and durability for database instances. When you provision a Multi-AZ database instance, Amazon RDS automatically creates a primary database instance, and synchronously replicates the data to a standby instance in a different Availability Zone. In case of an infrastructure failure, Amazon RDS performs an automatic failover to the standby so that you can resume database operations as soon as the failover is complete.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-RDSInstancePublicAccess') &&
            !rds_1.nist80053RDSInstancePublicAccess(node)) {
            const ruleId = 'NIST.800.53-RDSInstancePublicAccess';
            const info = 'The RDS DB instance allows public access - (Control IDs: AC-4, AC-6, AC-21(b), SC-7, SC-7(3)).';
            const explanation = 'Amazon RDS database instances can contain sensitive information, and principles and access control is required for such accounts.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-RDSLoggingEnabled') &&
            !rds_1.nist80053RDSLoggingEnabled(node)) {
            const ruleId = 'NIST.800.53-RDSLoggingEnabled';
            const info = 'The RDS DB instance does not have all CloudWatch log types exported - (Control IDs: AC-2(4), AC-2(g), AU-2(a)(d), AU-3, AU-12(a)(c)).';
            const explanation = 'To help with logging and monitoring within your environment, ensure Amazon Relational Database Service (Amazon RDS) logging is enabled. With Amazon RDS logging, you can capture events such as connections, disconnections, queries, or tables queried.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-RDSStorageEncrypted') &&
            !rds_1.nist80053RDSStorageEncrypted(node)) {
            const ruleId = 'NIST.800.53-RDSStorageEncrypted';
            const info = 'The RDS DB instance or Aurora DB cluster does not have storage encrypted - (Control IDs: SC-13, SC-28).';
            const explanation = 'Because sensitive data can exist at rest in Amazon RDS DB instances and clusters, enable encryption at rest to help protect that data.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Redshift Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkRedshift(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-RedshiftClusterConfiguration') &&
            !redshift_1.nist80053RedshiftClusterConfiguration(node)) {
            const ruleId = 'NIST.800.53-RedshiftClusterConfiguration';
            const info = 'The Redshift cluster does not have encryption or audit logging enabled - (Control IDs: AC-2(4), AC-2(g), AU-2(a)(d), AU-3, AU-12(a)(c), SC-13).';
            const explanation = 'To protect data at rest, ensure that encryption is enabled for your Amazon Redshift clusters. You must also ensure that required configurations are deployed on Amazon Redshift clusters. The audit logging should be enabled to provide information about connections and user activities in the database.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-RedshiftClusterPublicAccess') &&
            !redshift_1.nist80053RedshiftClusterPublicAccess(node)) {
            const ruleId = 'NIST.800.53-RedshiftClusterPublicAccess';
            const info = 'The Redshift cluster allows public access - (Control IDs: AC-3, AC-4, AC-6, AC-21(b), SC-7, SC-7(3)).';
            const explanation = 'Amazon Redshift clusters can contain sensitive information and principles and access control is required for such accounts.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Amazon S3 Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkS3(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-S3BucketDefaultLockEnabled') &&
            !s3_1.nist80053S3BucketDefaultLockEnabled(node)) {
            const ruleId = 'NIST.800.53-S3BucketDefaultLockEnabled';
            const info = 'The S3 Bucket does not have object lock enabled - (Control ID: SC-28).';
            const explanation = 'Because sensitive data can exist at rest in S3 buckets, enforce object locks at rest to help protect that data.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-S3BucketLoggingEnabled') &&
            !s3_1.nist80053S3BucketLoggingEnabled(node)) {
            const ruleId = 'NIST.800.53-S3BucketLoggingEnabled';
            const info = 'The S3 Bucket does not have server access logs enabled - (Control IDs: AC-2(g), AU-2(a)(d), AU-3, AU-12(a)(c)).';
            const explanation = 'Amazon Simple Storage Service (Amazon S3) server access logging provides a method to monitor the network for potential cybersecurity events. The events are monitored by capturing detailed records for the requests that are made to an Amazon S3 bucket. Each access log record provides details about a single access request. The details include the requester, bucket name, request time, request action, response status, and an error code, if relevant.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-S3BucketPublicReadProhibited') &&
            !s3_1.nist80053S3BucketPublicReadProhibited(node)) {
            const ruleId = 'NIST.800.53-S3BucketPublicReadProhibited';
            const info = 'The S3 Bucket does not prohibit public read access through its Block Public Access configurations and bucket ACLs - (Control IDs: AC-3, AC-4, AC-6, AC-21(b), SC-7, SC-7(3)).';
            const explanation = 'The management of access should be consistent with the classification of the data.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-S3BucketPublicWriteProhibited') &&
            !s3_1.nist80053S3BucketPublicWriteProhibited(node)) {
            const ruleId = 'NIST.800.53-S3BucketPublicWriteProhibited';
            const info = 'The S3 Bucket does not prohibit public write access through its Block Public Access configurations and bucket ACLs - (Control IDs: AC-3, AC-4, AC-6, AC-21(b), SC-7, SC-7(3)).';
            const explanation = 'The management of access should be consistent with the classification of the data.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-S3BucketReplicationEnabled') &&
            !s3_1.nist80053S3BucketReplicationEnabled(node)) {
            const ruleId = 'NIST.800.53-S3BucketReplicationEnabled';
            const info = 'The S3 Bucket does not have replication enabled - (Control IDs: AU-9(2), CP-9(b), CP-10, SC-5, SC-36).';
            const explanation = 'Amazon Simple Storage Service (Amazon S3) Cross-Region Replication (CRR) supports maintaining adequate capacity and availability. CRR enables automatic, asynchronous copying of objects across Amazon S3 buckets to help ensure that data availability is maintained.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-S3BucketServerSideEncryptionEnabled') &&
            !s3_1.nist80053S3BucketServerSideEncryptionEnabled(node)) {
            const ruleId = 'NIST.800.53-S3BucketServerSideEncryptionEnabled';
            const info = 'The S3 Bucket does not have default server-side encryption enabled - (Control IDs: AU-9(2), CP-9(b), CP-10, SC-5, SC-36).';
            const explanation = 'Because sensitive data can exist at rest in Amazon S3 buckets, enable encryption to help protect that data.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-S3BucketVersioningEnabled') &&
            !s3_1.nist80053S3BucketVersioningEnabled(node)) {
            const ruleId = 'NIST.800.53-S3BucketVersioningEnabled';
            const info = 'The S3 Bucket does not have versioning enabled - (Control IDs: CP-10, SI-12).';
            const explanation = 'Use versioning to preserve, retrieve, and restore every version of every object stored in your Amazon S3 bucket. Versioning helps you to easily recover from unintended user actions and application failures.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check SageMaker Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkSageMaker(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-SageMakerEndpointKMS') &&
            !sagemaker_1.nist80053SageMakerEndpointKMS(node)) {
            const ruleId = 'NIST.800.53-SageMakerEndpointKMS';
            const info = 'The SageMaker endpoint is not encrypted with a KMS key - (Control IDs: SC-13, SC-28).';
            const explanation = 'Because sensitive data can exist at rest in SageMaker endpoint, enable encryption at rest to help protect that data.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-SageMakerNotebookDirectInternetAccessDisabled') &&
            !sagemaker_1.nist80053SageMakerNotebookDirectInternetAccessDisabled(node)) {
            const ruleId = 'NIST.800.53-SageMakerNotebookDirectInternetAccessDisabled';
            const info = 'The SageMaker notebook does not disable direct internet access - (Control IDs: AC-3, AC-4, AC-6, AC-21(b), SC-7, SC-7(3)).';
            const explanation = 'By preventing direct internet access, you can keep sensitive data from being accessed by unauthorized users.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'NIST.800.53-SageMakerNotebookKMS') &&
            !sagemaker_1.nist80053SageMakerNotebookKMS(node)) {
            const ruleId = 'NIST.800.53-SageMakerNotebookKMS';
            const info = 'The SageMaker notebook is not encrypted with a KMS key - (Control IDs: SC-13, SC-28).';
            const explanation = 'Because sensitive data can exist at rest in SageMaker notebook, enable encryption at rest to help protect that data.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Amazon SNS Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkSNS(node, ignores) {
        if (!this.ignoreRule(ignores, 'NIST.800.53-SNSEncryptedKMS') &&
            !sns_1.nist80053SNSEncryptedKMS(node)) {
            const ruleId = 'NIST.800.53-SNSEncryptedKMS';
            const info = 'The SNS topic does not have KMS encryption enabled - (Control IDs: SC-13, SC-28).';
            const explanation = 'Because sensitive data can exist at rest in published messages, enable encryption at rest to help protect that data.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
}
exports.NIST80053Checks = NIST80053Checks;
_a = JSII_RTTI_SYMBOL_1;
NIST80053Checks[_a] = { fqn: "monocdk-nag.NIST80053Checks", version: "0.0.107" };
//# sourceMappingURL=data:application/json;base64,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