"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.HIPAASecurityChecks = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const monocdk_1 = require("monocdk");
const common_1 = require("../common");
const apigw_1 = require("./rules/apigw");
const autoscaling_1 = require("./rules/autoscaling");
const cloudtrail_1 = require("./rules/cloudtrail");
const cloudwatch_1 = require("./rules/cloudwatch");
const codebuild_1 = require("./rules/codebuild");
const dms_1 = require("./rules/dms");
const dynamodb_1 = require("./rules/dynamodb");
const ec2_1 = require("./rules/ec2");
const ecs_1 = require("./rules/ecs");
const efs_1 = require("./rules/efs");
const elasticache_1 = require("./rules/elasticache");
const elasticbeanstalk_1 = require("./rules/elasticbeanstalk");
const elb_1 = require("./rules/elb");
const emr_1 = require("./rules/emr");
const iam_1 = require("./rules/iam");
const lambda_1 = require("./rules/lambda");
const opensearch_1 = require("./rules/opensearch");
const rds_1 = require("./rules/rds");
const redshift_1 = require("./rules/redshift");
const s3_1 = require("./rules/s3");
const sagemaker_1 = require("./rules/sagemaker");
const secretsmanager_1 = require("./rules/secretsmanager");
const sns_1 = require("./rules/sns");
/**
 * Check for HIPAA Security compliance.
 *
 * Based on the HIPAA Security AWS operational best practices: https://docs.aws.amazon.com/config/latest/developerguide/operational-best-practices-for-hipaa_security.html
 *
 * @stability stable
 */
class HIPAASecurityChecks extends common_1.NagPack {
    /**
     * All aspects can visit an IConstruct.
     *
     * @stability stable
     */
    visit(node) {
        var _b;
        if (node instanceof monocdk_1.CfnResource) {
            // Get ignores metadata if it exists
            const ignores = (_b = node.getMetadata('cdk_nag')) === null || _b === void 0 ? void 0 : _b.rules_to_suppress;
            this.checkAPIGW(node, ignores);
            this.checkAutoScaling(node, ignores);
            this.checkCloudTrail(node, ignores);
            this.checkCloudWatch(node, ignores);
            this.checkCodeBuild(node, ignores);
            this.checkDMS(node, ignores);
            this.checkDynamoDB(node, ignores);
            this.checkEC2(node, ignores);
            this.checkECS(node, ignores);
            this.checkEFS(node, ignores);
            this.checkElastiCache(node, ignores);
            this.checkElasticBeanstalk(node, ignores);
            this.checkELB(node, ignores);
            this.checkEMR(node, ignores);
            this.checkIAM(node, ignores);
            this.checkLambda(node, ignores);
            this.checkOpenSearch(node, ignores);
            this.checkRDS(node, ignores);
            this.checkRedshift(node, ignores);
            this.checkS3(node, ignores);
            this.checkSageMaker(node, ignores);
            this.checkSecretsManager(node, ignores);
            this.checkSNS(node, ignores);
            // this.checkVPC(node, ignores);
        }
    }
    /**
     * Check API Gateway Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkAPIGW(node, ignores) {
        if (!this.ignoreRule(ignores, 'HIPAA.Security-APIGWCacheEnabledAndEncrypted') &&
            !apigw_1.hipaaSecurityAPIGWCacheEnabledAndEncrypted(node)) {
            const ruleId = 'HIPAA.Security-APIGWCacheEnabledAndEncrypted';
            const info = 'The API Gateway stage does not have caching enabled and encrypted for all methods - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(2)(ii)).';
            const explanation = "To help protect data at rest, ensure encryption is enabled for your API Gateway stage's cache. Because sensitive data can be captured for the API method, enable encryption at rest to help protect that data.";
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-APIGWExecutionLoggingEnabled') &&
            !apigw_1.hipaaSecurityAPIGWExecutionLoggingEnabled(node)) {
            const ruleId = 'HIPAA.Security-APIGWExecutionLoggingEnabled';
            const info = 'The API Gateway stage does not have execution logging enabled for all methods - (Control ID: 164.312(b)).';
            const explanation = 'API Gateway logging displays detailed views of users who accessed the API and the way they accessed the API. This insight enables visibility of user activities.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-APIGWSSLEnabled') &&
            !apigw_1.hipaaSecurityAPIGWSSLEnabled(node)) {
            const ruleId = 'HIPAA.Security-APIGWSSLEnabled';
            const info = 'The API Gateway REST API stage is not configured with SSL certificates - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(1), 164.312(e)(2)(i), 164.312(e)(2)(ii)).';
            const explanation = 'Ensure Amazon API Gateway REST API stages are configured with SSL certificates to allow backend systems to authenticate that requests originate from API Gateway.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-APIGWXrayEnabled') &&
            !apigw_1.hipaaSecurityAPIGWXrayEnabled(node)) {
            const ruleId = 'HIPAA.Security-APIGWXrayEnabled';
            const info = 'The API Gateway REST API stage does not have X-Ray enabled - (Control ID: 164.312(b)).';
            const explanation = 'AWS X-Ray collects data about requests that your application serves, and provides tools you can use to view, filter, and gain insights into that data to identify issues and opportunities for optimization. Ensure X-Ray is enables so you can see detailed information not only about the request and response, but also about calls that your application makes to downstream AWS resources, microservices, databases and HTTP web APIs.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Auto Scaling Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkAutoScaling(node, ignores) {
        if (!this.ignoreRule(ignores, 'HIPAA.Security-AutoscalingGroupELBHealthCheckRequired') &&
            !autoscaling_1.hipaaSecurityAutoscalingGroupELBHealthCheckRequired(node)) {
            const ruleId = 'HIPAA.Security-AutoscalingGroupELBHealthCheckRequired';
            const info = 'The Auto Scaling group utilizes a load balancer and does not have an ELB health check configured - (Control ID: 164.312(b)).';
            const explanation = 'The Elastic Load Balancer (ELB) health checks for Amazon Elastic Compute Cloud (Amazon EC2) Auto Scaling groups support maintenance of adequate capacity and availability. The load balancer periodically sends pings, attempts connections, or sends requests to test Amazon EC2 instances health in an auto-scaling group. If an instance is not reporting back, traffic is sent to a new Amazon EC2 instance.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-AutoscalingLaunchConfigPublicIpDisabled') &&
            !autoscaling_1.hipaaSecurityAutoscalingLaunchConfigPublicIpDisabled(node)) {
            const ruleId = 'HIPAA.Security-AutoscalingLaunchConfigPublicIpDisabled';
            const info = 'The Auto Scaling launch configuration does not have public IP addresses disabled - (Control IDs: 164.308(a)(3)(i), 164.308(a)(3)(ii)(B), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).';
            const explanation = 'If you configure your Network Interfaces with a public IP address, then the associated resources to those Network Interfaces are reachable from the internet. EC2 resources should not be publicly accessible, as this may allow unintended access to your applications or servers.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check CloudTrail Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkCloudTrail(node, ignores) {
        if (!this.ignoreRule(ignores, 'HIPAA.Security-CloudTrailCloudWatchLogsEnabled') &&
            !cloudtrail_1.hipaaSecurityCloudTrailCloudWatchLogsEnabled(node)) {
            const ruleId = 'HIPAA.Security-CloudTrailCloudWatchLogsEnabled';
            const info = 'The trail does not have CloudWatch logs enabled - (Control IDs: 164.308(a)(3)(ii)(A), 164.312(b)).';
            const explanation = 'Use Amazon CloudWatch to centrally collect and manage log event activity. Inclusion of AWS CloudTrail data provides details of API call activity within your AWS account.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-CloudTrailEncryptionEnabled') &&
            !cloudtrail_1.hipaaSecurityCloudTrailEncryptionEnabled(node)) {
            const ruleId = 'HIPAA.Security-CloudTrailEncryptionEnabled';
            const info = 'The trail does not have a KMS key ID or have encryption enabled - (Control ID: 164.312(a)(2)(iv), 164.312(e)(2)(ii)).';
            const explanation = 'Because sensitive data may exist and to help protect data at rest, ensure encryption is enabled for your AWS CloudTrail trails.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-CloudTrailLogFileValidationEnabled') &&
            !cloudtrail_1.hipaaSecurityCloudTrailLogFileValidationEnabled(node)) {
            const ruleId = 'HIPAA.Security-CloudTrailLogFileValidationEnabled';
            const info = 'The trail does not have log file validation enabled - (Control ID: 164.312(c)(1), 164.312(c)(2)).';
            const explanation = 'Utilize AWS CloudTrail log file validation to check the integrity of CloudTrail logs. Log file validation helps determine if a log file was modified or deleted or unchanged after CloudTrail delivered it. This feature is built using industry standard algorithms: SHA-256 for hashing and SHA-256 with RSA for digital signing. This makes it computationally infeasible to modify, delete or forge CloudTrail log files without detection.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check CloudWatch Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkCloudWatch(node, ignores) {
        if (!this.ignoreRule(ignores, 'HIPAA.Security-CloudWatchAlarmAction') &&
            !cloudwatch_1.hipaaSecurityCloudWatchAlarmAction(node)) {
            const ruleId = 'HIPAA.Security-CloudWatchAlarmAction';
            const info = 'The CloudWatch alarm does not have at least one alarm action, one INSUFFICIENT_DATA action, or one OK action enabled - (Control ID: 164.312(b)).';
            const explanation = 'Amazon CloudWatch alarms alert when a metric breaches the threshold for a specified number of evaluation periods. The alarm performs one or more actions based on the value of the metric or expression relative to a threshold over a number of time periods.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-CloudWatchLogGroupEncrypted') &&
            !cloudwatch_1.hipaaSecurityCloudWatchLogGroupEncrypted(node)) {
            const ruleId = 'HIPAA.Security-CloudWatchLogGroupEncrypted';
            const info = 'The CloudWatch Log Group is not encrypted with an AWS KMS key - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(2)(ii)).';
            const explanation = 'To help protect sensitive data at rest, ensure encryption is enabled for your Amazon CloudWatch Log Groups.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-CloudWatchLogGroupRetentionPeriod') &&
            !cloudwatch_1.hipaaSecurityCloudWatchLogGroupRetentionPeriod(node)) {
            const ruleId = 'HIPAA.Security-CloudWatchLogGroupRetentionPeriod';
            const info = 'The CloudWatch Log Group does not have an explicit retention period configured - (Control ID: 164.312(b)).';
            const explanation = 'Ensure a minimum duration of event log data is retained for your log groups to help with troubleshooting and forensics investigations. The lack of available past event log data makes it difficult to reconstruct and identify potentially malicious events.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check CodeBuild Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkCodeBuild(node, ignores) {
        if (!this.ignoreRule(ignores, 'HIPAA.Security-CodeBuildProjectEnvVarAwsCred') &&
            !codebuild_1.hipaaSecurityCodeBuildProjectEnvVarAwsCred(node)) {
            const ruleId = 'HIPAA.Security-CodeBuildProjectEnvVarAwsCred';
            const info = 'The CodeBuild environment stores sensitive credentials (such as AWS_ACCESS_KEY_ID and/or AWS_SECRET_ACCESS_KEY) as plaintext environment variables - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1)).';
            const explanation = 'Do not store these variables in clear text. Storing these variables in clear text leads to unintended data exposure and unauthorized access.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-CodeBuildProjectSourceRepoUrl') &&
            !codebuild_1.hipaaSecurityCodeBuildProjectSourceRepoUrl(node)) {
            const ruleId = 'HIPAA.Security-CodeBuildProjectSourceRepoUrl';
            const info = 'The CodeBuild project which utilizes either a GitHub or BitBucket source repository does not utilize OAUTH - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1)).';
            const explanation = 'OAUTH is the most secure method of authenticating your CodeBuild application. Use OAuth instead of personal access tokens or a user name and password to grant authorization for accessing GitHub or Bitbucket repositories.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check DMS Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkDMS(node, ignores) {
        if (!this.ignoreRule(ignores, 'HIPAA.Security-DMSReplicationNotPublic') &&
            !dms_1.hipaaSecurityDMSReplicationNotPublic(node)) {
            const ruleId = 'HIPAA.Security-DMSReplicationNotPublic';
            const info = 'The DMS replication instance is public - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).';
            const explanation = 'DMS replication instances can contain sensitive information and access control is required for such accounts.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check DynamoDB Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkDynamoDB(node, ignores) {
        if (!this.ignoreRule(ignores, 'HIPAA.Security-DynamoDBPITREnabled') &&
            !dynamodb_1.hipaaSecurityDynamoDBPITREnabled(node)) {
            const ruleId = 'HIPAA.Security-DynamoDBPITREnabled';
            const info = 'The DynamoDB table does not have Point-in-time Recovery enabled - (Control IDs: 164.308(a)(7)(i), 164.308(a)(7)(ii)(A), 164.308(a)(7)(ii)(B)).';
            const explanation = 'The recovery maintains continuous backups of your table for the last 35 days.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check EC2 Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkEC2(node, ignores) {
        if (!this.ignoreRule(ignores, 'HIPAA.Security-EC2InstanceDetailedMonitoringEnabled') &&
            !ec2_1.hipaaSecurityEC2InstanceDetailedMonitoringEnabled(node)) {
            const ruleId = 'HIPAA.Security-EC2InstanceDetailedMonitoringEnabled';
            const info = 'The EC2 instance does not have detailed monitoring enabled - (Control IDs: 164.312(b)).';
            const explanation = 'Detailed monitoring provides additional monitoring information (such as 1-minute period graphs) on the AWS console.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-EC2InstancesInVPC') &&
            !ec2_1.hipaaSecurityEC2InstancesInVPC(node)) {
            const ruleId = 'HIPAA.Security-EC2InstancesInVPC';
            const info = 'The EC2 instance is not within a VPC - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).';
            const explanation = 'Because of their logical isolation, domains that reside within an Amazon VPC have an extra layer of security when compared to domains that use public endpoints.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-EC2CheckNoPublicIPs') &&
            !ec2_1.hipaaSecurityEC2InstanceNoPublicIp(node)) {
            const ruleId = 'HIPAA.Security-EC2CheckNoPublicIPs';
            const info = 'The EC2 instance is associated with a public IP address - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).';
            const explanation = 'Amazon EC2 instances can contain sensitive information and access control is required for such resources.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check ECS Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkECS(node, ignores) {
        if (!this.ignoreRule(ignores, 'HIPAA.Security-ECSTaskDefinitionUserForHostMode') &&
            !ecs_1.hipaaSecurityECSTaskDefinitionUserForHostMode(node)) {
            const ruleId = 'HIPAA.Security-ECSTaskDefinitionUserForHostMode';
            const info = "The ECS task definition is configured for host networking and has at least one container with definitions with 'privileged' set to false or empty or 'user' set to root or empty - (Control IDs: 164.308(a)(3)(i), 164.308(a)(3)(ii)(A), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1)).";
            const explanation = 'If a task definition has elevated privileges it is because you have specifically opted-in to those configurations. This rule checks for unexpected privilege escalation when a task definition has host networking enabled but the customer has not opted-in to elevated privileges.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check EFS Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkEFS(node, ignores) {
        if (!this.ignoreRule(ignores, 'HIPAA.Security-EFSEncrypted') &&
            !efs_1.hipaaSecurityEFSEncrypted(node)) {
            const ruleId = 'HIPAA.Security-EFSEncrypted';
            const info = 'The EFS does not have encryption at rest enabled - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(2)(ii)).';
            const explanation = 'Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon Elastic File System (EFS).';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check ElastiCache Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkElastiCache(node, ignores) {
        if (!this.ignoreRule(ignores, 'HIPAA.Security-ElasticacheRedisClusterAutomaticBackup') &&
            !elasticache_1.hipaaSecurityElastiCacheRedisClusterAutomaticBackup(node)) {
            const ruleId = 'HIPAA.Security-ElasticacheRedisClusterAutomaticBackup';
            const info = 'The ElastiCache Redis cluster does not retain automatic backups for at least 15 days - (Control IDs: 164.308(a)(7)(i), 164.308(a)(7)(ii)(A), 164.308(a)(7)(ii)(B)).';
            const explanation = 'Automatic backups can help guard against data loss. If a failure occurs, you can create a new cluster, which restores your data from the most recent backup.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Elastic Beanstalk Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkElasticBeanstalk(node, ignores) {
        if (!this.ignoreRule(ignores, 'HIPAA.Security-ElasticBeanstalkEnhancedHealthReportingEnabled') &&
            !elasticbeanstalk_1.hipaaSecurityElasticBeanstalkEnhancedHealthReportingEnabled(node)) {
            const ruleId = 'HIPAA.Security-ElasticBeanstalkEnhancedHealthReportingEnabled';
            const info = 'The Elastic Beanstalk environment does not have enhanced health reporting enabled - (Control ID: 164.312(b)).';
            const explanation = 'AWS Elastic Beanstalk enhanced health reporting enables a more rapid response to changes in the health of the underlying infrastructure. These changes could result in a lack of availability of the application. Elastic Beanstalk enhanced health reporting provides a status descriptor to gauge the severity of the identified issues and identify possible causes to investigate.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-ElasticBeanstalkManagedUpdatesEnabled') &&
            !elasticbeanstalk_1.hipaaSecurityElasticBeanstalkManagedUpdatesEnabled(node)) {
            const ruleId = 'HIPAA.Security-ElasticBeanstalkManagedUpdatesEnabled';
            const info = 'The Elastic Beanstalk environment does not have managed updates enabled - (Control ID: 164.308(a)(5)(ii)(A)).';
            const explanation = 'Enabling managed platform updates for an Amazon Elastic Beanstalk environment ensures that the latest available platform fixes, updates, and features for the environment are installed. Keeping up to date with patch installation is a best practice in securing systems.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Elastic Load Balancer Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkELB(node, ignores) {
        if (!this.ignoreRule(ignores, 'HIPAA.Security-ALBHttpDropInvalidHeaderEnabled') &&
            !elb_1.hipaaSecurityALBHttpDropInvalidHeaderEnabled(node)) {
            const ruleId = 'HIPAA.Security-ALBHttpDropInvalidHeaderEnabled';
            const info = 'The ALB does not have invalid HTTP header dropping enabled - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(1), 164.312(e)(2)(i), 164.312(e)(2)(ii)).';
            const explanation = 'Ensure that your Application Load Balancers (ALB) are configured to drop http headers. Because sensitive data can exist, enable encryption in transit to help protect that data.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-ALBHttpToHttpsRedirection') &&
            !elb_1.hipaaSecurityALBHttpToHttpsRedirection(node)) {
            const ruleId = 'HIPAA.Security-ALBHttpToHttpsRedirection';
            const info = "The ALB's HTTP listeners are not configured to redirect to HTTPS - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(1), 164.312(e)(2)(i), 164.312(e)(2)(ii)).";
            const explanation = 'To help protect data in transit, ensure that your Application Load Balancer automatically redirects unencrypted HTTP requests to HTTPS. Because sensitive data can exist, enable encryption in transit to help protect that data.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-ELBACMCertificateRequired') &&
            !elb_1.hipaaSecurityELBACMCertificateRequired(node)) {
            const ruleId = 'HIPAA.Security-ELBACMCertificateRequired';
            const info = 'The CLB does not utilize an SSL certificate provided by ACM (Amazon Certificate Manager) - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(1), 164.312(e)(2)(i), 164.312(e)(2)(ii)).';
            const explanation = 'Because sensitive data can exist and to help protect data at transit, ensure encryption is enabled for your Elastic Load Balancing. Use AWS Certificate Manager to manage, provision and deploy public and private SSL/TLS certificates with AWS services and internal resources.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-ELBCrossZoneBalancingEnabled') &&
            !elb_1.hipaaSecurityELBCrossZoneBalancingEnabled(node)) {
            const ruleId = 'HIPAA.Security-ELBCrossZoneBalancingEnabled';
            const info = 'The CLB does not balance traffic between at least 2 Availability Zones - (Control IDs: 164.308(a)(7)(i), 164.308(a)(7)(ii)(C)).';
            const explanation = 'The cross-zone load balancing reduces the need to maintain equivalent numbers of instances in each enabled availability zone.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-ELBDeletionProtectionEnabled') &&
            !elb_1.hipaaSecurityELBDeletionProtectionEnabled(node)) {
            const ruleId = 'HIPAA.Security-ELBDeletionProtectionEnabled';
            const info = 'The ALB, NLB, or GLB does not have deletion protection enabled - (Control IDs: 164.308(a)(7)(i), 164.308(a)(7)(ii)(C)).';
            const explanation = 'Use this feature to prevent your load balancer from being accidentally or maliciously deleted, which can lead to loss of availability for your applications.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-ELBLoggingEnabled') &&
            !elb_1.hipaaSecurityELBLoggingEnabled(node)) {
            const ruleId = 'HIPAA.Security-ELBLoggingEnabled';
            const info = 'The ELB does not have logging enabled - (Control ID: 164.312(b)).';
            const explanation = "Elastic Load Balancing activity is a central point of communication within an environment. Ensure ELB logging is enabled. The collected data provides detailed information about requests sent to The ELB. Each log contains information such as the time the request was received, the client's IP address, latencies, request paths, and server responses.";
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-ELBTlsHttpsListenersOnly') &&
            !elb_1.hipaaSecurityELBTlsHttpsListenersOnly(node)) {
            const ruleId = 'HIPAA.Security-ELBTlsHttpsListenersOnly';
            const info = 'The CLB does not restrict its listeners to only the SSL and HTTPS protocols - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(1), 164.312(e)(2)(i), 164.312(e)(2)(ii)).';
            const explanation = 'Because sensitive data can exist, enable encryption in transit to help protect that data.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-ELBv2ACMCertificateRequired') &&
            !elb_1.hipaaSecurityELBv2ACMCertificateRequired(node)) {
            const ruleId = 'HIPAA.Security-ELBv2ACMCertificateRequired';
            const info = 'The ALB, NLB, or GLB listener does not utilize an SSL certificate provided by ACM (Amazon Certificate Manager) - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(2)(ii)).';
            const explanation = 'Because sensitive data can exist and to help protect data at transit, ensure encryption is enabled for your Elastic Load Balancing. Use AWS Certificate Manager to manage, provision and deploy public and private SSL/TLS certificates with AWS services and internal resources.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check EMR Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkEMR(node, ignores) {
        if (!this.ignoreRule(ignores, 'HIPAA.Security-EMRKerberosEnabled') &&
            !emr_1.hipaaSecurityEMRKerberosEnabled(node)) {
            const ruleId = 'HIPAA.Security-EMRKerberosEnabled';
            const info = 'The EMR cluster does not have Kerberos enabled - (Control IDs: 164.308(a)(3)(i), 164.308(a)(3)(ii)(A), 164.308(a)(3)(ii)(B), 164.308(a)(4)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(B), 164.308(a)(4)(ii)(C), 164.312(a)(1)).';
            const explanation = 'The access permissions and authorizations can be managed and incorporated with the principles of least privilege and separation of duties, by enabling Kerberos for Amazon EMR clusters.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check IAM Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkIAM(node, ignores) {
        if (!this.ignoreRule(ignores, 'HIPAA.Security-IAMNoInlinePolicy') &&
            !iam_1.hipaaSecurityIAMNoInlinePolicy(node)) {
            const ruleId = 'HIPAA.Security-IAMNoInlinePolicy';
            const info = 'The IAM Group, User, or Role contains an inline policy - (Control IDs: 164.308(a)(3)(i), 164.308(a)(3)(ii)(A), 164.308(a)(3)(ii)(B), 164.308(a)(4)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(B), 164.308(a)(4)(ii)(C), 164.312(a)(1)).';
            const explanation = 'AWS recommends to use managed policies instead of inline policies. The managed policies allow reusability, versioning and rolling back, and delegating permissions management.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-IAMPolicyNoStatementsWithAdminAccess') &&
            !iam_1.hipaaSecurityIAMPolicyNoStatementsWithAdminAccess(node)) {
            const ruleId = 'HIPAA.Security-IAMPolicyNoStatementsWithAdminAccess';
            const info = 'The IAM policy grants admin access - (Control IDs: 164.308(a)(3)(i), 164.308(a)(3)(ii)(A), 164.308(a)(3)(ii)(B), 164.308(a)(4)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(B), 164.308(a)(4)(ii)(C), 164.312(a)(1)).';
            const explanation = 'AWS Identity and Access Management (IAM) can help you incorporate the principles of least privilege and separation of duties with access permissions and authorizations, restricting policies from containing "Effect": "Allow" with "Action": "*" over "Resource": "*". Allowing users to have more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-IAMPolicyNoStatementsWithFullAccess') &&
            !iam_1.hipaaSecurityIAMPolicyNoStatementsWithFullAccess(node)) {
            const ruleId = 'HIPAA.Security-IAMPolicyNoStatementsWithFullAccess';
            const info = 'The IAM policy grants full access - (Control IDs: 164.308(a)(3)(i), 164.308(a)(3)(ii)(A), 164.308(a)(3)(ii)(B), 164.308(a)(4)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(B), 164.308(a)(4)(ii)(C), 164.312(a)(1)).';
            const explanation = 'Ensure IAM Actions are restricted to only those actions that are needed. Allowing users to have more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-IAMUserGroupMembership') &&
            !iam_1.hipaaSecurityIAMUserGroupMembership(node)) {
            const ruleId = 'HIPAA.Security-IAMUserGroupMembership';
            const info = 'The IAM user does not belong to any group(s) - (Control IDs: 164.308(a)(3)(i), 164.308(a)(3)(ii)(A), 164.308(a)(3)(ii)(B), 164.308(a)(4)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(B), 164.308(a)(4)(ii)(C), 164.312(a)(1)).';
            const explanation = 'AWS Identity and Access Management (IAM) can help you restrict access permissions and authorizations, by ensuring IAM users are members of at least one group. Allowing users more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-IAMUserNoPolicies') &&
            !iam_1.hipaaSecurityIAMUserNoPolicies(node)) {
            const ruleId = 'HIPAA.Security-IAMUserNoPolicies';
            const info = 'The IAM policy is attached at the user level - (Control IDs: 164.308(a)(3)(i), 164.308(a)(3)(ii)(A), 164.308(a)(3)(ii)(B), 164.308(a)(4)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(B), 164.308(a)(4)(ii)(C), 164.312(a)(1)).';
            const explanation = 'Assigning privileges at the group or the role level helps to reduce opportunity for an identity to receive or retain excessive privileges.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Lambda Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkLambda(node, ignores) {
        if (!this.ignoreRule(ignores, 'HIPAA.Security-LambdaConcurrency') &&
            !lambda_1.hipaaSecurityLambdaConcurrency(node)) {
            const ruleId = 'HIPAA.Security-LambdaConcurrency';
            const info = 'The Lambda function is not configured with function-level concurrent execution limits - (Control ID: 164.312(b)).';
            const explanation = "Ensure that a Lambda function's concurrency high and low limits are established. This can assist in baselining the number of requests that your function is serving at any given time.";
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-LambdaDlq') &&
            !lambda_1.hipaaSecurityLambdaDlq(node)) {
            const ruleId = 'HIPAA.Security-LambdaDlq';
            const info = 'The Lambda function is not configured with a dead-letter configuration - (Control ID: 164.312(b)).';
            const explanation = 'Notify the appropriate personnel through Amazon Simple Queue Service (Amazon SQS) or Amazon Simple Notification Service (Amazon SNS) when a function has failed.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-LambdaInsideVPC') &&
            !lambda_1.hipaaSecurityLambdaInsideVPC(node)) {
            const ruleId = 'HIPAA.Security-LambdaInsideVPC';
            const info = 'The Lambda function is not VPC enabled - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).';
            const explanation = 'Because of their logical isolation, domains that reside within an Amazon VPC have an extra layer of security when compared to domains that use public endpoints.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check OpenSearch Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkOpenSearch(node, ignores) {
        if (!this.ignoreRule(ignores, 'HIPAA.Security-OpenSearchEncryptedAtRest') &&
            !opensearch_1.hipaaSecurityOpenSearchEncryptedAtRest(node)) {
            const ruleId = 'HIPAA.Security-OpenSearchEncryptedAtRest';
            const info = 'The OpenSearch Service domain does not have encryption at rest enabled - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(2)(ii)).';
            const explanation = 'Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon OpenSearch Service (OpenSearch Service) domains.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-OpenSearchInVPCOnly') &&
            !opensearch_1.hipaaSecurityOpenSearchInVPCOnly(node)) {
            const ruleId = 'HIPAA.Security-OpenSearchInVPCOnly';
            const info = 'The OpenSearch Service domain is not running within a VPC - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).';
            const explanation = 'VPCs help secure your AWS resources and provide an extra layer of protection.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-OpenSearchLogsToCloudWatch') &&
            !opensearch_1.hipaaSecurityOpenSearchLogsToCloudWatch(node)) {
            const ruleId = 'HIPAA.Security-OpenSearchLogsToCloudWatch';
            const info = 'The OpenSearch Service domain does not stream error logs (ES_APPLICATION_LOGS) to CloudWatch Logs - (Control IDs: 164.308(a)(3)(ii)(A), 164.312(b)).';
            const explanation = 'Ensure Amazon OpenSearch Service domains have error logs enabled and streamed to Amazon CloudWatch Logs for retention and response. Domain error logs can assist with security and access audits, and can help to diagnose availability issues.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-OpenSearchNodeToNodeEncryption') &&
            !opensearch_1.hipaaSecurityOpenSearchNodeToNodeEncryption(node)) {
            const ruleId = 'HIPAA.Security-OpenSearchNodeToNodeEncryption';
            const info = 'The OpenSearch Service domain does not have node-to-node encryption enabled - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(1), 164.312(e)(2)(i), 164.312(e)(2)(ii)).';
            const explanation = 'Because sensitive data can exist, enable encryption in transit to help protect that data within your Amazon OpenSearch Service (OpenSearch Service) domains.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check RDS Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkRDS(node, ignores) {
        if (!this.ignoreRule(ignores, 'HIPAA.Security-RDSAutomaticMinorVersionUpgradeEnabled') &&
            !rds_1.hipaaSecurityRDSAutomaticMinorVersionUpgradeEnabled(node)) {
            const ruleId = 'HIPAA.Security-RDSAutomaticMinorVersionUpgradeEnabled';
            const info = 'The RDS DB instance does not have automatic minor version upgrades enabled - (Control ID: 164.308(a)(5)(ii)(A)).';
            const explanation = 'Enable automatic minor version upgrades on your Amazon Relational Database Service (RDS) instances to ensure the latest minor version updates to the Relational Database Management System (RDBMS) are installed, which may include security patches and bug fixes.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-RDSEnhancedMonitoringEnabled') &&
            !rds_1.hipaaSecurityRDSEnhancedMonitoringEnabled(node)) {
            const ruleId = 'HIPAA.Security-RDSEnhancedMonitoringEnabled';
            const info = 'The RDS DB instance does not enhanced monitoring enabled - (Control ID: 164.312(b)).';
            const explanation = 'Enable enhanced monitoring to help monitor Amazon RDS availability. This provides detailed visibility into the health of your Amazon RDS database instances.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-RDSInstanceBackupEnabled') &&
            !rds_1.hipaaSecurityRDSInstanceBackupEnabled(node)) {
            const ruleId = 'HIPAA.Security-RDSInstanceBackupEnabled';
            const info = 'The RDS DB instance does not have backups enabled - (Control IDs: 164.308(a)(7)(i), 164.308(a)(7)(ii)(A), 164.308(a)(7)(ii)(B)).';
            const explanation = 'The backup feature of Amazon RDS creates backups of your databases and transaction logs.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-RDSInstanceDeletionProtectionEnabled') &&
            !rds_1.hipaaSecurityRDSInstanceDeletionProtectionEnabled(node)) {
            const ruleId = 'HIPAA.Security-RDSInstanceDeletionProtectionEnabled';
            const info = 'The RDS DB instance or Aurora DB cluster does not have deletion protection enabled - (Control IDs: 164.308(a)(7)(i), 164.308(a)(7)(ii)(C)).';
            const explanation = 'Ensure Amazon Relational Database Service (Amazon RDS) instances and clusters have deletion protection enabled. Use deletion protection to prevent your Amazon RDS DB instances and clusters from being accidentally or maliciously deleted, which can lead to loss of availability for your applications.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-RDSInstanceMultiAzSupport') &&
            !rds_1.hipaaSecurityRDSInstanceMultiAZSupport(node)) {
            const ruleId = 'HIPAA.Security-RDSInstanceMultiAzSupport';
            const info = 'The non-Aurora RDS DB instance does not have multi-AZ support enabled - (Control IDs: 164.308(a)(7)(i), 164.308(a)(7)(ii)(C)).';
            const explanation = 'Multi-AZ support in Amazon Relational Database Service (Amazon RDS) provides enhanced availability and durability for database instances. When you provision a Multi-AZ database instance, Amazon RDS automatically creates a primary database instance, and synchronously replicates the data to a standby instance in a different Availability Zone. In case of an infrastructure failure, Amazon RDS performs an automatic failover to the standby so that you can resume database operations as soon as the failover is complete.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-RDSInstancePublicAccess') &&
            !rds_1.hipaaSecurityRDSInstancePublicAccess(node)) {
            const ruleId = 'HIPAA.Security-RDSInstancePublicAccess';
            const info = 'The RDS DB instance allows public access - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).';
            const explanation = 'Amazon RDS database instances can contain sensitive information, and principles and access control is required for such accounts.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-RDSLoggingEnabled') &&
            !rds_1.hipaaSecurityRDSLoggingEnabled(node)) {
            const ruleId = 'HIPAA.Security-RDSLoggingEnabled';
            const info = 'The RDS DB instance does not have all CloudWatch log types exported - (Control IDs: 164.308(a)(3)(ii)(A), 164.308(a)(5)(ii)(C)).';
            const explanation = 'To help with logging and monitoring within your environment, ensure Amazon Relational Database Service (Amazon RDS) logging is enabled. With Amazon RDS logging, you can capture events such as connections, disconnections, queries, or tables queried.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-RDSStorageEncrypted') &&
            !rds_1.hipaaSecurityRDSStorageEncrypted(node)) {
            const ruleId = 'HIPAA.Security-RDSStorageEncrypted';
            const info = 'The RDS DB instance or Aurora DB cluster does not have storage encrypted - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(2)(ii)).';
            const explanation = 'Because sensitive data can exist at rest in Amazon RDS DB instances and clusters, enable encryption at rest to help protect that data.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Redshift Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkRedshift(node, ignores) {
        if (!this.ignoreRule(ignores, 'HIPAA.Security-RedshiftBackupEnabled') &&
            !redshift_1.hipaaSecurityRedshiftBackupEnabled(node)) {
            const ruleId = 'HIPAA.Security-RedshiftBackupEnabled';
            const info = 'The Redshift cluster does not have automated snapshots enabled or the retention period is not between 1 and 35 days - (Control IDs: 164.308(a)(7)(i), 164.308(a)(7)(ii)(A), 164.308(a)(7)(ii)(B)).';
            const explanation = 'To help with data back-up processes, ensure your Amazon Redshift clusters have automated snapshots. When automated snapshots are enabled for a cluster, Redshift periodically takes snapshots of that cluster. By default, Redshift takes a snapshot every eight hours or every 5 GB per node of data changes, or whichever comes first.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-RedshiftClusterConfiguration') &&
            !redshift_1.hipaaSecurityRedshiftClusterConfiguration(node)) {
            const ruleId = 'HIPAA.Security-RedshiftClusterConfiguration';
            const info = 'The Redshift cluster does not have encryption or audit logging enabled - (Control IDs: 164.312(a)(2)(iv), 164.312(b), 164.312(e)(2)(ii)).';
            const explanation = 'To protect data at rest, ensure that encryption is enabled for your Amazon Redshift clusters. You must also ensure that required configurations are deployed on Amazon Redshift clusters. The audit logging should be enabled to provide information about connections and user activities in the database.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-RedshiftClusterMaintenanceSettings') &&
            !redshift_1.hipaaSecurityRedshiftClusterMaintenanceSettings(node)) {
            const ruleId = 'HIPAA.Security-RedshiftClusterMaintenanceSettings';
            const info = 'The Redshift cluster has version upgrades enabled, automated snapshot retention periods enabled, and an explicit maintenance window configured - (Control IDs: 164.308(a)(5)(ii)(A), 164.308(a)(7)(ii)(A)).';
            const explanation = 'Ensure that Amazon Redshift clusters have the preferred settings for your organization. Specifically, that they have preferred maintenance windows and automated snapshot retention periods for the database.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-RedshiftClusterPublicAccess') &&
            !redshift_1.hipaaSecurityRedshiftClusterPublicAccess(node)) {
            const ruleId = 'HIPAA.Security-RedshiftClusterPublicAccess';
            const info = 'The Redshift cluster allows public access - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).';
            const explanation = 'Amazon Redshift clusters can contain sensitive information and principles and access control is required for such accounts.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-RedshiftEnhancedVPCRoutingEnabled') &&
            !redshift_1.hipaaSecurityRedshiftEnhancedVPCRoutingEnabled(node)) {
            const ruleId = 'HIPAA.Security-RedshiftEnhancedVPCRoutingEnabled';
            const info = 'The Redshift cluster does not have enhanced VPC routing enabled - (Control IDs: 164.312(e)(1)).';
            const explanation = 'Enhanced VPC routing forces all COPY and UNLOAD traffic between the cluster and data repositories to go through your Amazon VPC. You can then use VPC features such as security groups and network access control lists to secure network traffic. You can also use VPC flow logs to monitor network traffic.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Amazon S3 Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkS3(node, ignores) {
        if (!this.ignoreRule(ignores, 'HIPAA.Security-S3BucketLevelPublicAccessProhibited') &&
            !s3_1.hipaaSecurityS3BucketLevelPublicAccessProhibited(node)) {
            const ruleId = 'HIPAA.Security-S3BucketLevelPublicAccessProhibited';
            const info = 'The S3 bucket does not prohibit public access through bucket level settings - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).';
            const explanation = 'Keep sensitive data safe from unauthorized remote users by preventing public access at the bucket level.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-S3BucketLoggingEnabled') &&
            !s3_1.hipaaSecurityS3BucketLoggingEnabled(node)) {
            const ruleId = 'HIPAA.Security-S3BucketLoggingEnabled';
            const info = 'The S3 Bucket does not have server access logs enabled - (Control IDs: 164.308(a)(3)(ii)(A), 164.312(b)).';
            const explanation = 'Amazon Simple Storage Service (Amazon S3) server access logging provides a method to monitor the network for potential cybersecurity events. The events are monitored by capturing detailed records for the requests that are made to an Amazon S3 bucket. Each access log record provides details about a single access request. The details include the requester, bucket name, request time, request action, response status, and an error code, if relevant.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-S3BucketPublicReadProhibited') &&
            !s3_1.hipaaSecurityS3BucketPublicReadProhibited(node)) {
            const ruleId = 'HIPAA.Security-S3BucketPublicReadProhibited';
            const info = 'The S3 Bucket does not prohibit public read access through its Block Public Access configurations and bucket ACLs - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).';
            const explanation = 'The management of access should be consistent with the classification of the data.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-S3BucketPublicWriteProhibited') &&
            !s3_1.hipaaSecurityS3BucketPublicWriteProhibited(node)) {
            const ruleId = 'HIPAA.Security-S3BucketPublicWriteProhibited';
            const info = 'The S3 Bucket does not prohibit public write access through its Block Public Access configurations and bucket ACLs - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).';
            const explanation = 'The management of access should be consistent with the classification of the data.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-S3BucketReplicationEnabled') &&
            !s3_1.hipaaSecurityS3BucketReplicationEnabled(node)) {
            const ruleId = 'HIPAA.Security-S3BucketReplicationEnabled';
            const info = 'The S3 Bucket does not have replication enabled - (Control IDs: 164.308(a)(7)(i), 164.308(a)(7)(ii)(A), 164.308(a)(7)(ii)(B)).';
            const explanation = 'Amazon Simple Storage Service (Amazon S3) Cross-Region Replication (CRR) supports maintaining adequate capacity and availability. CRR enables automatic, asynchronous copying of objects across Amazon S3 buckets to help ensure that data availability is maintained.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-S3BucketServerSideEncryptionEnabled') &&
            !s3_1.hipaaSecurityS3BucketServerSideEncryptionEnabled(node)) {
            const ruleId = 'HIPAA.Security-S3BucketServerSideEncryptionEnabled';
            const info = 'The S3 Bucket does not have default server-side encryption enabled - (Control IDs: 164.312(a)(2)(iv), 164.312(c)(2), 164.312(e)(2)(ii)).';
            const explanation = 'Because sensitive data can exist at rest in Amazon S3 buckets, enable encryption to help protect that data.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-S3BucketVersioningEnabled') &&
            !s3_1.hipaaSecurityS3BucketVersioningEnabled(node)) {
            const ruleId = 'HIPAA.Security-S3BucketVersioningEnabled';
            const info = 'The S3 Bucket does not have versioning enabled - (Control IDs: 164.308(a)(7)(i), 164.308(a)(7)(ii)(A), 164.308(a)(7)(ii)(B), 164.312(c)(1), 164.312(c)(2)).';
            const explanation = 'Use versioning to preserve, retrieve, and restore every version of every object stored in your Amazon S3 bucket. Versioning helps you to easily recover from unintended user actions and application failures.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-S3DefaultEncryptionKMS') &&
            !s3_1.hipaaSecurityS3DefaultEncryptionKMS(node)) {
            const ruleId = 'HIPAA.Security-S3DefaultEncryptionKMS';
            const info = 'The S3 Bucket is not encrypted with a KMS Key by default - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(2)(ii)).';
            const explanation = 'Ensure that encryption is enabled for your Amazon Simple Storage Service (Amazon S3) buckets. Because sensitive data can exist at rest in an Amazon S3 bucket, enable encryption at rest to help protect that data.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check SageMaker Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkSageMaker(node, ignores) {
        if (!this.ignoreRule(ignores, 'HIPAA.Security-SageMakerEndpointConfigurationKMSKeyConfigured') &&
            !sagemaker_1.hipaaSecuritySageMakerEndpointConfigurationKMSKeyConfigured(node)) {
            const ruleId = 'HIPAA.Security-SageMakerEndpointConfigurationKMSKeyConfigured';
            const info = 'The SageMaker endpoint is not encrypted with a KMS key - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(2)(ii)).';
            const explanation = 'Because sensitive data can exist at rest in SageMaker endpoint, enable encryption at rest to help protect that data.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-SageMakerNotebookInstanceKMSKeyConfigured') &&
            !sagemaker_1.hipaaSecuritySageMakerNotebookInstanceKMSKeyConfigured(node)) {
            const ruleId = 'HIPAA.Security-SageMakerNotebookInstanceKMSKeyConfigured';
            const info = 'The SageMaker notebook is not encrypted with a KMS key - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(2)(ii)).';
            const explanation = 'Because sensitive data can exist at rest in SageMaker notebook, enable encryption at rest to help protect that data.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
        if (!this.ignoreRule(ignores, 'HIPAA.Security-SageMakerNotebookNoDirectInternetAccess') &&
            !sagemaker_1.hipaaSecuritySageMakerNotebookNoDirectInternetAccess(node)) {
            const ruleId = 'HIPAA.Security-SageMakerNotebookNoDirectInternetAccess';
            const info = 'The SageMaker notebook does not disable direct internet access - (Control IDs: 164.308(a)(3)(i), 164.308(a)(4)(ii)(A), 164.308(a)(4)(ii)(C), 164.312(a)(1), 164.312(e)(1)).';
            const explanation = 'By preventing direct internet access, you can keep sensitive data from being accessed by unauthorized users.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Secrets Manager Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkSecretsManager(node, ignores) {
        if (!this.ignoreRule(ignores, 'HIPAA.Security-SecretsManagerUsingKMSKey') &&
            !secretsmanager_1.hipaaSecuritySecretsManagerUsingKMSKey(node)) {
            const ruleId = 'HIPAA.Security-SecretsManagerUsingKMSKey';
            const info = 'The secret is not encrypted with a KMS Customer managed key - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(2)(ii)).';
            const explanation = 'To help protect data at rest, ensure encryption with AWS Key Management Service (AWS KMS) is enabled for AWS Secrets Manager secrets. Because sensitive data can exist at rest in Secrets Manager secrets, enable encryption at rest to help protect that data.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
    /**
     * Check Amazon SNS Resources
     * @param node the IConstruct to evaluate
     * @param ignores list of ignores for the resource
     */
    checkSNS(node, ignores) {
        if (!this.ignoreRule(ignores, 'HIPAA.Security-SNSEncryptedKMS') &&
            !sns_1.hipaaSecuritySNSEncryptedKMS(node)) {
            const ruleId = 'HIPAA.Security-SNSEncryptedKMS';
            const info = 'The SNS topic does not have KMS encryption enabled - (Control IDs: 164.312(a)(2)(iv), 164.312(e)(2)(ii)).';
            const explanation = 'Because sensitive data can exist at rest in published messages, enable encryption at rest to help protect that data.';
            monocdk_1.Annotations.of(node).addError(this.createMessage(ruleId, info, explanation));
        }
    }
}
exports.HIPAASecurityChecks = HIPAASecurityChecks;
_a = JSII_RTTI_SYMBOL_1;
HIPAASecurityChecks[_a] = { fqn: "monocdk-nag.HIPAASecurityChecks", version: "0.0.107" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaGlwYWEtc2VjdXJpdHkuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi9zcmMvSElQQUEtU2VjdXJpdHkvaGlwYWEtc2VjdXJpdHkudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQTs7O0VBR0U7QUFFRixxQ0FBK0Q7QUFDL0Qsc0NBQW9DO0FBQ3BDLHlDQUt1QjtBQUN2QixxREFHNkI7QUFDN0IsbURBSTRCO0FBQzVCLG1EQUk0QjtBQUM1QixpREFHMkI7QUFDM0IscUNBQW1FO0FBQ25FLCtDQUFvRTtBQUNwRSxxQ0FJcUI7QUFDckIscUNBQTRFO0FBQzVFLHFDQUF3RDtBQUN4RCxxREFBMEY7QUFDMUYsK0RBR2tDO0FBQ2xDLHFDQVNxQjtBQUNyQixxQ0FBOEQ7QUFDOUQscUNBTXFCO0FBQ3JCLDJDQUl3QjtBQUN4QixtREFLNEI7QUFDNUIscUNBU3FCO0FBQ3JCLCtDQU0wQjtBQUMxQixtQ0FTb0I7QUFDcEIsaURBSTJCO0FBQzNCLDJEQUFnRjtBQUNoRixxQ0FBMkQ7Ozs7Ozs7O0FBRzNELE1BQWEsbUJBQW9CLFNBQVEsZ0JBQU87Ozs7OztJQUN2QyxLQUFLLENBQUMsSUFBZ0I7O1FBQzNCLElBQUksSUFBSSxZQUFZLHFCQUFXLEVBQUU7WUFDL0Isb0NBQW9DO1lBQ3BDLE1BQU0sT0FBTyxTQUFHLElBQUksQ0FBQyxXQUFXLENBQUMsU0FBUyxDQUFDLDBDQUFFLGlCQUFpQixDQUFDO1lBQy9ELElBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQy9CLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLEVBQUUsT0FBTyxDQUFDLENBQUM7WUFDckMsSUFBSSxDQUFDLGVBQWUsQ0FBQyxJQUFJLEVBQUUsT0FBTyxDQUFDLENBQUM7WUFDcEMsSUFBSSxDQUFDLGVBQWUsQ0FBQyxJQUFJLEVBQUUsT0FBTyxDQUFDLENBQUM7WUFDcEMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxJQUFJLEVBQUUsT0FBTyxDQUFDLENBQUM7WUFDbkMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUsT0FBTyxDQUFDLENBQUM7WUFDN0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLEVBQUUsT0FBTyxDQUFDLENBQUM7WUFDbEMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUsT0FBTyxDQUFDLENBQUM7WUFDN0IsSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUsT0FBTyxDQUFDLENBQUM7WUFDN0IsSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUsT0FBTyxDQUFDLENBQUM7WUFDN0IsSUFBSSxDQUFDLGdCQUFnQixDQUFDLElBQUksRUFBRSxPQUFPLENBQUMsQ0FBQztZQUNyQyxJQUFJLENBQUMscUJBQXFCLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQzFDLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQzdCLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQzdCLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQzdCLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQ2hDLElBQUksQ0FBQyxlQUFlLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQ3BDLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQzdCLElBQUksQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQ2xDLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQzVCLElBQUksQ0FBQyxjQUFjLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQ25DLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLEVBQUUsT0FBTyxDQUFDLENBQUM7WUFDeEMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUsT0FBTyxDQUFDLENBQUM7WUFDN0IsZ0NBQWdDO1NBQ2pDO0lBQ0gsQ0FBQztJQUVEOzs7O09BSUc7SUFDSyxVQUFVLENBQUMsSUFBaUIsRUFBRSxPQUFZO1FBQ2hELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUNkLE9BQU8sRUFDUCw4Q0FBOEMsQ0FDL0M7WUFDRCxDQUFDLGtEQUEwQyxDQUFDLElBQUksQ0FBQyxFQUNqRDtZQUNBLE1BQU0sTUFBTSxHQUFHLDhDQUE4QyxDQUFDO1lBQzlELE1BQU0sSUFBSSxHQUNSLDBJQUEwSSxDQUFDO1lBQzdJLE1BQU0sV0FBVyxHQUNmLGdOQUFnTixDQUFDO1lBQ25OLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FDZCxPQUFPLEVBQ1AsNkNBQTZDLENBQzlDO1lBQ0QsQ0FBQyxpREFBeUMsQ0FBQyxJQUFJLENBQUMsRUFDaEQ7WUFDQSxNQUFNLE1BQU0sR0FBRyw2Q0FBNkMsQ0FBQztZQUM3RCxNQUFNLElBQUksR0FDUiwyR0FBMkcsQ0FBQztZQUM5RyxNQUFNLFdBQVcsR0FDZixrS0FBa0ssQ0FBQztZQUNySyxxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQzNCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO1FBQ0QsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLGdDQUFnQyxDQUFDO1lBQzNELENBQUMsb0NBQTRCLENBQUMsSUFBSSxDQUFDLEVBQ25DO1lBQ0EsTUFBTSxNQUFNLEdBQUcsZ0NBQWdDLENBQUM7WUFDaEQsTUFBTSxJQUFJLEdBQ1IsZ0tBQWdLLENBQUM7WUFDbkssTUFBTSxXQUFXLEdBQ2YsbUtBQW1LLENBQUM7WUFDdEsscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxpQ0FBaUMsQ0FBQztZQUM1RCxDQUFDLHFDQUE2QixDQUFDLElBQUksQ0FBQyxFQUNwQztZQUNBLE1BQU0sTUFBTSxHQUFHLGlDQUFpQyxDQUFDO1lBQ2pELE1BQU0sSUFBSSxHQUNSLHdGQUF3RixDQUFDO1lBQzNGLE1BQU0sV0FBVyxHQUNmLDZhQUE2YSxDQUFDO1lBQ2hiLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7SUFDSCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNLLGdCQUFnQixDQUFDLElBQWlCLEVBQUUsT0FBWTtRQUN0RCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FDZCxPQUFPLEVBQ1AsdURBQXVELENBQ3hEO1lBQ0QsQ0FBQyxpRUFBbUQsQ0FBQyxJQUFJLENBQUMsRUFDMUQ7WUFDQSxNQUFNLE1BQU0sR0FBRyx1REFBdUQsQ0FBQztZQUN2RSxNQUFNLElBQUksR0FDUiw4SEFBOEgsQ0FBQztZQUNqSSxNQUFNLFdBQVcsR0FDZixrWkFBa1osQ0FBQztZQUNyWixxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQzNCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO1FBQ0QsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQ2QsT0FBTyxFQUNQLHdEQUF3RCxDQUN6RDtZQUNELENBQUMsa0VBQW9ELENBQUMsSUFBSSxDQUFDLEVBQzNEO1lBQ0EsTUFBTSxNQUFNLEdBQUcsd0RBQXdELENBQUM7WUFDeEUsTUFBTSxJQUFJLEdBQ1IscU5BQXFOLENBQUM7WUFDeE4sTUFBTSxXQUFXLEdBQ2YscVJBQXFSLENBQUM7WUFDeFIscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtJQUNILENBQUM7SUFFRDs7OztPQUlHO0lBQ0ssZUFBZSxDQUFDLElBQWlCLEVBQUUsT0FBWTtRQUNyRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FDZCxPQUFPLEVBQ1AsZ0RBQWdELENBQ2pEO1lBQ0QsQ0FBQyx5REFBNEMsQ0FBQyxJQUFJLENBQUMsRUFDbkQ7WUFDQSxNQUFNLE1BQU0sR0FBRyxnREFBZ0QsQ0FBQztZQUNoRSxNQUFNLElBQUksR0FDUixvR0FBb0csQ0FBQztZQUN2RyxNQUFNLFdBQVcsR0FDZiwyS0FBMkssQ0FBQztZQUM5SyxxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQzNCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO1FBQ0QsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLDRDQUE0QyxDQUFDO1lBQ3ZFLENBQUMscURBQXdDLENBQUMsSUFBSSxDQUFDLEVBQy9DO1lBQ0EsTUFBTSxNQUFNLEdBQUcsNENBQTRDLENBQUM7WUFDNUQsTUFBTSxJQUFJLEdBQ1IsdUhBQXVILENBQUM7WUFDMUgsTUFBTSxXQUFXLEdBQ2YsaUlBQWlJLENBQUM7WUFDcEkscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUNkLE9BQU8sRUFDUCxtREFBbUQsQ0FDcEQ7WUFDRCxDQUFDLDREQUErQyxDQUFDLElBQUksQ0FBQyxFQUN0RDtZQUNBLE1BQU0sTUFBTSxHQUFHLG1EQUFtRCxDQUFDO1lBQ25FLE1BQU0sSUFBSSxHQUNSLG1HQUFtRyxDQUFDO1lBQ3RHLE1BQU0sV0FBVyxHQUNmLGliQUFpYixDQUFDO1lBQ3BiLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7SUFDSCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNLLGVBQWUsQ0FBQyxJQUFpQixFQUFFLE9BQVk7UUFDckQsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLHNDQUFzQyxDQUFDO1lBQ2pFLENBQUMsK0NBQWtDLENBQUMsSUFBSSxDQUFDLEVBQ3pDO1lBQ0EsTUFBTSxNQUFNLEdBQUcsc0NBQXNDLENBQUM7WUFDdEQsTUFBTSxJQUFJLEdBQ1Isa0pBQWtKLENBQUM7WUFDckosTUFBTSxXQUFXLEdBQ2YsZ1FBQWdRLENBQUM7WUFDblEscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSw0Q0FBNEMsQ0FBQztZQUN2RSxDQUFDLHFEQUF3QyxDQUFDLElBQUksQ0FBQyxFQUMvQztZQUNBLE1BQU0sTUFBTSxHQUFHLDRDQUE0QyxDQUFDO1lBQzVELE1BQU0sSUFBSSxHQUNSLHNIQUFzSCxDQUFDO1lBQ3pILE1BQU0sV0FBVyxHQUNmLDZHQUE2RyxDQUFDO1lBQ2hILHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FDZCxPQUFPLEVBQ1Asa0RBQWtELENBQ25EO1lBQ0QsQ0FBQywyREFBOEMsQ0FBQyxJQUFJLENBQUMsRUFDckQ7WUFDQSxNQUFNLE1BQU0sR0FBRyxrREFBa0QsQ0FBQztZQUNsRSxNQUFNLElBQUksR0FDUiw0R0FBNEcsQ0FBQztZQUMvRyxNQUFNLFdBQVcsR0FDZiwrUEFBK1AsQ0FBQztZQUNsUSxxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQzNCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO0lBQ0gsQ0FBQztJQUVEOzs7O09BSUc7SUFDSyxjQUFjLENBQUMsSUFBaUIsRUFBRSxPQUFZO1FBQ3BELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUNkLE9BQU8sRUFDUCw4Q0FBOEMsQ0FDL0M7WUFDRCxDQUFDLHNEQUEwQyxDQUFDLElBQUksQ0FBQyxFQUNqRDtZQUNBLE1BQU0sTUFBTSxHQUFHLDhDQUE4QyxDQUFDO1lBQzlELE1BQU0sSUFBSSxHQUNSLGtQQUFrUCxDQUFDO1lBQ3JQLE1BQU0sV0FBVyxHQUNmLDhJQUE4SSxDQUFDO1lBQ2pKLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FDZCxPQUFPLEVBQ1AsOENBQThDLENBQy9DO1lBQ0QsQ0FBQyxzREFBMEMsQ0FBQyxJQUFJLENBQUMsRUFDakQ7WUFDQSxNQUFNLE1BQU0sR0FBRyw4Q0FBOEMsQ0FBQztZQUM5RCxNQUFNLElBQUksR0FDUiwwTUFBME0sQ0FBQztZQUM3TSxNQUFNLFdBQVcsR0FDZiw4TkFBOE4sQ0FBQztZQUNqTyxxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQzNCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO0lBQ0gsQ0FBQztJQUVEOzs7O09BSUc7SUFDSyxRQUFRLENBQUMsSUFBaUIsRUFBRSxPQUFZO1FBQzlDLElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSx3Q0FBd0MsQ0FBQztZQUNuRSxDQUFDLDBDQUFvQyxDQUFDLElBQUksQ0FBQyxFQUMzQztZQUNBLE1BQU0sTUFBTSxHQUFHLHdDQUF3QyxDQUFDO1lBQ3hELE1BQU0sSUFBSSxHQUNSLHFKQUFxSixDQUFDO1lBQ3hKLE1BQU0sV0FBVyxHQUNmLCtHQUErRyxDQUFDO1lBQ2xILHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7SUFDSCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNLLGFBQWEsQ0FBQyxJQUFpQixFQUFFLE9BQVk7UUFDbkQsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLG9DQUFvQyxDQUFDO1lBQy9ELENBQUMsMkNBQWdDLENBQUMsSUFBSSxDQUFDLEVBQ3ZDO1lBQ0EsTUFBTSxNQUFNLEdBQUcsb0NBQW9DLENBQUM7WUFDcEQsTUFBTSxJQUFJLEdBQ1IsZ0pBQWdKLENBQUM7WUFDbkosTUFBTSxXQUFXLEdBQ2YsK0VBQStFLENBQUM7WUFDbEYscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtJQUNILENBQUM7SUFFRDs7OztPQUlHO0lBQ0ssUUFBUSxDQUFDLElBQWlCLEVBQUUsT0FBWTtRQUM5QyxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FDZCxPQUFPLEVBQ1AscURBQXFELENBQ3REO1lBQ0QsQ0FBQyx1REFBaUQsQ0FBQyxJQUFJLENBQUMsRUFDeEQ7WUFDQSxNQUFNLE1BQU0sR0FBRyxxREFBcUQsQ0FBQztZQUNyRSxNQUFNLElBQUksR0FDUix5RkFBeUYsQ0FBQztZQUM1RixNQUFNLFdBQVcsR0FDZixxSEFBcUgsQ0FBQztZQUN4SCxxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQzNCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO1FBQ0QsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLGtDQUFrQyxDQUFDO1lBQzdELENBQUMsb0NBQThCLENBQUMsSUFBSSxDQUFDLEVBQ3JDO1lBQ0EsTUFBTSxNQUFNLEdBQUcsa0NBQWtDLENBQUM7WUFDbEQsTUFBTSxJQUFJLEdBQ1IsbUpBQW1KLENBQUM7WUFDdEosTUFBTSxXQUFXLEdBQ2Ysa0tBQWtLLENBQUM7WUFDcksscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxvQ0FBb0MsQ0FBQztZQUMvRCxDQUFDLHdDQUFrQyxDQUFDLElBQUksQ0FBQyxFQUN6QztZQUNBLE1BQU0sTUFBTSxHQUFHLG9DQUFvQyxDQUFDO1lBQ3BELE1BQU0sSUFBSSxHQUNSLHNLQUFzSyxDQUFDO1lBQ3pLLE1BQU0sV0FBVyxHQUNmLDJHQUEyRyxDQUFDO1lBQzlHLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7SUFDSCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNLLFFBQVEsQ0FBQyxJQUFpQixFQUFFLE9BQVk7UUFDOUMsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQ2QsT0FBTyxFQUNQLGlEQUFpRCxDQUNsRDtZQUNELENBQUMsbURBQTZDLENBQUMsSUFBSSxDQUFDLEVBQ3BEO1lBQ0EsTUFBTSxNQUFNLEdBQUcsaURBQWlELENBQUM7WUFDakUsTUFBTSxJQUFJLEdBQ1Isc1NBQXNTLENBQUM7WUFDelMsTUFBTSxXQUFXLEdBQ2Ysc1JBQXNSLENBQUM7WUFDelIscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtJQUNILENBQUM7SUFFRDs7OztPQUlHO0lBQ0ssUUFBUSxDQUFDLElBQWlCLEVBQUUsT0FBWTtRQUM5QyxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsNkJBQTZCLENBQUM7WUFDeEQsQ0FBQywrQkFBeUIsQ0FBQyxJQUFJLENBQUMsRUFDaEM7WUFDQSxNQUFNLE1BQU0sR0FBRyw2QkFBNkIsQ0FBQztZQUM3QyxNQUFNLElBQUksR0FDUix5R0FBeUcsQ0FBQztZQUM1RyxNQUFNLFdBQVcsR0FDZiw0SUFBNEksQ0FBQztZQUMvSSxxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQzNCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO0lBQ0gsQ0FBQztJQUVEOzs7O09BSUc7SUFDSyxnQkFBZ0IsQ0FBQyxJQUFpQixFQUFFLE9BQVk7UUFDdEQsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQ2QsT0FBTyxFQUNQLHVEQUF1RCxDQUN4RDtZQUNELENBQUMsaUVBQW1ELENBQUMsSUFBSSxDQUFDLEVBQzFEO1lBQ0EsTUFBTSxNQUFNLEdBQUcsdURBQXVELENBQUM7WUFDdkUsTUFBTSxJQUFJLEdBQ1IscUtBQXFLLENBQUM7WUFDeEssTUFBTSxXQUFXLEdBQ2YsOEpBQThKLENBQUM7WUFDaksscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtJQUNILENBQUM7SUFFRDs7OztPQUlHO0lBQ0sscUJBQXFCLENBQUMsSUFBaUIsRUFBRSxPQUFZO1FBQzNELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUNkLE9BQU8sRUFDUCwrREFBK0QsQ0FDaEU7WUFDRCxDQUFDLDhFQUEyRCxDQUFDLElBQUksQ0FBQyxFQUNsRTtZQUNBLE1BQU0sTUFBTSxHQUNWLCtEQUErRCxDQUFDO1lBQ2xFLE1BQU0sSUFBSSxHQUNSLCtHQUErRyxDQUFDO1lBQ2xILE1BQU0sV0FBVyxHQUNmLHdYQUF3WCxDQUFDO1lBQzNYLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FDZCxPQUFPLEVBQ1Asc0RBQXNELENBQ3ZEO1lBQ0QsQ0FBQyxxRUFBa0QsQ0FBQyxJQUFJLENBQUMsRUFDekQ7WUFDQSxNQUFNLE1BQU0sR0FBRyxzREFBc0QsQ0FBQztZQUN0RSxNQUFNLElBQUksR0FDUiwrR0FBK0csQ0FBQztZQUNsSCxNQUFNLFdBQVcsR0FDZiw2UUFBNlEsQ0FBQztZQUNoUixxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQzNCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO0lBQ0gsQ0FBQztJQUVEOzs7O09BSUc7SUFDSyxRQUFRLENBQUMsSUFBaUIsRUFBRSxPQUFZO1FBQzlDLElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUNkLE9BQU8sRUFDUCxnREFBZ0QsQ0FDakQ7WUFDRCxDQUFDLGtEQUE0QyxDQUFDLElBQUksQ0FBQyxFQUNuRDtZQUNBLE1BQU0sTUFBTSxHQUFHLGdEQUFnRCxDQUFDO1lBQ2hFLE1BQU0sSUFBSSxHQUNSLG9KQUFvSixDQUFDO1lBQ3ZKLE1BQU0sV0FBVyxHQUNmLGtMQUFrTCxDQUFDO1lBQ3JMLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsMENBQTBDLENBQUM7WUFDckUsQ0FBQyw0Q0FBc0MsQ0FBQyxJQUFJLENBQUMsRUFDN0M7WUFDQSxNQUFNLE1BQU0sR0FBRywwQ0FBMEMsQ0FBQztZQUMxRCxNQUFNLElBQUksR0FDUiwwSkFBMEosQ0FBQztZQUM3SixNQUFNLFdBQVcsR0FDZixtT0FBbU8sQ0FBQztZQUN0TyxxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQzNCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO1FBQ0QsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLDBDQUEwQyxDQUFDO1lBQ3JFLENBQUMsNENBQXNDLENBQUMsSUFBSSxDQUFDLEVBQzdDO1lBQ0EsTUFBTSxNQUFNLEdBQUcsMENBQTBDLENBQUM7WUFDMUQsTUFBTSxJQUFJLEdBQ1Isa0xBQWtMLENBQUM7WUFDckwsTUFBTSxXQUFXLEdBQ2YsbVJBQW1SLENBQUM7WUFDdFIscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUNkLE9BQU8sRUFDUCw2Q0FBNkMsQ0FDOUM7WUFDRCxDQUFDLCtDQUF5QyxDQUFDLElBQUksQ0FBQyxFQUNoRDtZQUNBLE1BQU0sTUFBTSxHQUFHLDZDQUE2QyxDQUFDO1lBQzdELE1BQU0sSUFBSSxHQUNSLGlJQUFpSSxDQUFDO1lBQ3BJLE1BQU0sV0FBVyxHQUNmLCtIQUErSCxDQUFDO1lBQ2xJLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FDZCxPQUFPLEVBQ1AsNkNBQTZDLENBQzlDO1lBQ0QsQ0FBQywrQ0FBeUMsQ0FBQyxJQUFJLENBQUMsRUFDaEQ7WUFDQSxNQUFNLE1BQU0sR0FBRyw2Q0FBNkMsQ0FBQztZQUM3RCxNQUFNLElBQUksR0FDUix5SEFBeUgsQ0FBQztZQUM1SCxNQUFNLFdBQVcsR0FDZiw4SkFBOEosQ0FBQztZQUNqSyxxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQzNCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO1FBQ0QsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLGtDQUFrQyxDQUFDO1lBQzdELENBQUMsb0NBQThCLENBQUMsSUFBSSxDQUFDLEVBQ3JDO1lBQ0EsTUFBTSxNQUFNLEdBQUcsa0NBQWtDLENBQUM7WUFDbEQsTUFBTSxJQUFJLEdBQ1IsbUVBQW1FLENBQUM7WUFDdEUsTUFBTSxXQUFXLEdBQ2YsOFZBQThWLENBQUM7WUFDalcscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSx5Q0FBeUMsQ0FBQztZQUNwRSxDQUFDLDJDQUFxQyxDQUFDLElBQUksQ0FBQyxFQUM1QztZQUNBLE1BQU0sTUFBTSxHQUFHLHlDQUF5QyxDQUFDO1lBQ3pELE1BQU0sSUFBSSxHQUNSLHFLQUFxSyxDQUFDO1lBQ3hLLE1BQU0sV0FBVyxHQUNmLDJGQUEyRixDQUFDO1lBQzlGLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsNENBQTRDLENBQUM7WUFDdkUsQ0FBQyw4Q0FBd0MsQ0FBQyxJQUFJLENBQUMsRUFDL0M7WUFDQSxNQUFNLE1BQU0sR0FBRyw0Q0FBNEMsQ0FBQztZQUM1RCxNQUFNLElBQUksR0FDUix1S0FBdUssQ0FBQztZQUMxSyxNQUFNLFdBQVcsR0FDZixtUkFBbVIsQ0FBQztZQUN0UixxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQzNCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO0lBQ0gsQ0FBQztJQUVEOzs7O09BSUc7SUFDSyxRQUFRLENBQUMsSUFBaUIsRUFBRSxPQUFZO1FBQzlDLElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxtQ0FBbUMsQ0FBQztZQUM5RCxDQUFDLHFDQUErQixDQUFDLElBQUksQ0FBQyxFQUN0QztZQUNBLE1BQU0sTUFBTSxHQUFHLG1DQUFtQyxDQUFDO1lBQ25ELE1BQU0sSUFBSSxHQUNSLGtPQUFrTyxDQUFDO1lBQ3JPLE1BQU0sV0FBVyxHQUNmLDBMQUEwTCxDQUFDO1lBQzdMLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7SUFDSCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNLLFFBQVEsQ0FBQyxJQUFpQixFQUFFLE9BQVk7UUFDOUMsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLGtDQUFrQyxDQUFDO1lBQzdELENBQUMsb0NBQThCLENBQUMsSUFBSSxDQUFDLEVBQ3JDO1lBQ0EsTUFBTSxNQUFNLEdBQUcsa0NBQWtDLENBQUM7WUFDbEQsTUFBTSxJQUFJLEdBQ1IsME9BQTBPLENBQUM7WUFDN08sTUFBTSxXQUFXLEdBQ2YsZ0xBQWdMLENBQUM7WUFDbkwscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUNkLE9BQU8sRUFDUCxxREFBcUQsQ0FDdEQ7WUFDRCxDQUFDLHVEQUFpRCxDQUFDLElBQUksQ0FBQyxFQUN4RDtZQUNBLE1BQU0sTUFBTSxHQUFHLHFEQUFxRCxDQUFDO1lBQ3JFLE1BQU0sSUFBSSxHQUNSLHNOQUFzTixDQUFDO1lBQ3pOLE1BQU0sV0FBVyxHQUNmLHVaQUF1WixDQUFDO1lBQzFaLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFFRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FDZCxPQUFPLEVBQ1Asb0RBQW9ELENBQ3JEO1lBQ0QsQ0FBQyxzREFBZ0QsQ0FBQyxJQUFJLENBQUMsRUFDdkQ7WUFDQSxNQUFNLE1BQU0sR0FBRyxvREFBb0QsQ0FBQztZQUNwRSxNQUFNLElBQUksR0FDUixxTkFBcU4sQ0FBQztZQUN4TixNQUFNLFdBQVcsR0FDZix1TkFBdU4sQ0FBQztZQUMxTixxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQzNCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO1FBQ0QsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLHVDQUF1QyxDQUFDO1lBQ2xFLENBQUMseUNBQW1DLENBQUMsSUFBSSxDQUFDLEVBQzFDO1lBQ0EsTUFBTSxNQUFNLEdBQUcsdUNBQXVDLENBQUM7WUFDdkQsTUFBTSxJQUFJLEdBQ1IsZ09BQWdPLENBQUM7WUFDbk8sTUFBTSxXQUFXLEdBQ2YscVNBQXFTLENBQUM7WUFDeFMscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUVELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxrQ0FBa0MsQ0FBQztZQUM3RCxDQUFDLG9DQUE4QixDQUFDLElBQUksQ0FBQyxFQUNyQztZQUNBLE1BQU0sTUFBTSxHQUFHLGtDQUFrQyxDQUFDO1lBQ2xELE1BQU0sSUFBSSxHQUNSLGdPQUFnTyxDQUFDO1lBQ25PLE1BQU0sV0FBVyxHQUNmLDRJQUE0SSxDQUFDO1lBQy9JLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7SUFDSCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNLLFdBQVcsQ0FBQyxJQUFpQixFQUFFLE9BQVk7UUFDakQsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLGtDQUFrQyxDQUFDO1lBQzdELENBQUMsdUNBQThCLENBQUMsSUFBSSxDQUFDLEVBQ3JDO1lBQ0EsTUFBTSxNQUFNLEdBQUcsa0NBQWtDLENBQUM7WUFDbEQsTUFBTSxJQUFJLEdBQ1IsbUhBQW1ILENBQUM7WUFDdEgsTUFBTSxXQUFXLEdBQ2Ysd0xBQXdMLENBQUM7WUFDM0wscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSwwQkFBMEIsQ0FBQztZQUNyRCxDQUFDLCtCQUFzQixDQUFDLElBQUksQ0FBQyxFQUM3QjtZQUNBLE1BQU0sTUFBTSxHQUFHLDBCQUEwQixDQUFDO1lBQzFDLE1BQU0sSUFBSSxHQUNSLG9HQUFvRyxDQUFDO1lBQ3ZHLE1BQU0sV0FBVyxHQUNmLGtLQUFrSyxDQUFDO1lBQ3JLLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsZ0NBQWdDLENBQUM7WUFDM0QsQ0FBQyxxQ0FBNEIsQ0FBQyxJQUFJLENBQUMsRUFDbkM7WUFDQSxNQUFNLE1BQU0sR0FBRyxnQ0FBZ0MsQ0FBQztZQUNoRCxNQUFNLElBQUksR0FDUixxSkFBcUosQ0FBQztZQUN4SixNQUFNLFdBQVcsR0FDZixrS0FBa0ssQ0FBQztZQUNySyxxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQzNCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO0lBQ0gsQ0FBQztJQUVEOzs7O09BSUc7SUFDSyxlQUFlLENBQUMsSUFBaUIsRUFBRSxPQUFZO1FBQ3JELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSwwQ0FBMEMsQ0FBQztZQUNyRSxDQUFDLG1EQUFzQyxDQUFDLElBQUksQ0FBQyxFQUM3QztZQUNBLE1BQU0sTUFBTSxHQUFHLDBDQUEwQyxDQUFDO1lBQzFELE1BQU0sSUFBSSxHQUNSLCtIQUErSCxDQUFDO1lBQ2xJLE1BQU0sV0FBVyxHQUNmLGtLQUFrSyxDQUFDO1lBQ3JLLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsb0NBQW9DLENBQUM7WUFDL0QsQ0FBQyw2Q0FBZ0MsQ0FBQyxJQUFJLENBQUMsRUFDdkM7WUFDQSxNQUFNLE1BQU0sR0FBRyxvQ0FBb0MsQ0FBQztZQUNwRCxNQUFNLElBQUksR0FDUix3S0FBd0ssQ0FBQztZQUMzSyxNQUFNLFdBQVcsR0FDZiwrRUFBK0UsQ0FBQztZQUNsRixxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQzNCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO1FBQ0QsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLDJDQUEyQyxDQUFDO1lBQ3RFLENBQUMsb0RBQXVDLENBQUMsSUFBSSxDQUFDLEVBQzlDO1lBQ0EsTUFBTSxNQUFNLEdBQUcsMkNBQTJDLENBQUM7WUFDM0QsTUFBTSxJQUFJLEdBQ1Isc0pBQXNKLENBQUM7WUFDekosTUFBTSxXQUFXLEdBQ2YsaVBBQWlQLENBQUM7WUFDcFAscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUNkLE9BQU8sRUFDUCwrQ0FBK0MsQ0FDaEQ7WUFDRCxDQUFDLHdEQUEyQyxDQUFDLElBQUksQ0FBQyxFQUNsRDtZQUNBLE1BQU0sTUFBTSxHQUFHLCtDQUErQyxDQUFDO1lBQy9ELE1BQU0sSUFBSSxHQUNSLHFLQUFxSyxDQUFDO1lBQ3hLLE1BQU0sV0FBVyxHQUNmLDhKQUE4SixDQUFDO1lBQ2pLLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7SUFDSCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNLLFFBQVEsQ0FBQyxJQUFpQixFQUFFLE9BQVk7UUFDOUMsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQ2QsT0FBTyxFQUNQLHVEQUF1RCxDQUN4RDtZQUNELENBQUMseURBQW1ELENBQUMsSUFBSSxDQUFDLEVBQzFEO1lBQ0EsTUFBTSxNQUFNLEdBQUcsdURBQXVELENBQUM7WUFDdkUsTUFBTSxJQUFJLEdBQ1Isa0hBQWtILENBQUM7WUFDckgsTUFBTSxXQUFXLEdBQ2YscVFBQXFRLENBQUM7WUFDeFEscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUNkLE9BQU8sRUFDUCw2Q0FBNkMsQ0FDOUM7WUFDRCxDQUFDLCtDQUF5QyxDQUFDLElBQUksQ0FBQyxFQUNoRDtZQUNBLE1BQU0sTUFBTSxHQUFHLDZDQUE2QyxDQUFDO1lBQzdELE1BQU0sSUFBSSxHQUNSLHNGQUFzRixDQUFDO1lBQ3pGLE1BQU0sV0FBVyxHQUNmLDhKQUE4SixDQUFDO1lBQ2pLLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUseUNBQXlDLENBQUM7WUFDcEUsQ0FBQywyQ0FBcUMsQ0FBQyxJQUFJLENBQUMsRUFDNUM7WUFDQSxNQUFNLE1BQU0sR0FBRyx5Q0FBeUMsQ0FBQztZQUN6RCxNQUFNLElBQUksR0FDUixrSUFBa0ksQ0FBQztZQUNySSxNQUFNLFdBQVcsR0FDZiwwRkFBMEYsQ0FBQztZQUM3RixxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQzNCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO1FBQ0QsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQ2QsT0FBTyxFQUNQLHFEQUFxRCxDQUN0RDtZQUNELENBQUMsdURBQWlELENBQUMsSUFBSSxDQUFDLEVBQ3hEO1lBQ0EsTUFBTSxNQUFNLEdBQUcscURBQXFELENBQUM7WUFDckUsTUFBTSxJQUFJLEdBQ1IsNklBQTZJLENBQUM7WUFDaEosTUFBTSxXQUFXLEdBQ2YsNFNBQTRTLENBQUM7WUFDL1MscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSwwQ0FBMEMsQ0FBQztZQUNyRSxDQUFDLDRDQUFzQyxDQUFDLElBQUksQ0FBQyxFQUM3QztZQUNBLE1BQU0sTUFBTSxHQUFHLDBDQUEwQyxDQUFDO1lBQzFELE1BQU0sSUFBSSxHQUNSLGdJQUFnSSxDQUFDO1lBQ25JLE1BQU0sV0FBVyxHQUNmLHVnQkFBdWdCLENBQUM7WUFDMWdCLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsd0NBQXdDLENBQUM7WUFDbkUsQ0FBQywwQ0FBb0MsQ0FBQyxJQUFJLENBQUMsRUFDM0M7WUFDQSxNQUFNLE1BQU0sR0FBRyx3Q0FBd0MsQ0FBQztZQUN4RCxNQUFNLElBQUksR0FDUix1SkFBdUosQ0FBQztZQUMxSixNQUFNLFdBQVcsR0FDZixtSUFBbUksQ0FBQztZQUN0SSxxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQzNCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO1FBQ0QsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLGtDQUFrQyxDQUFDO1lBQzdELENBQUMsb0NBQThCLENBQUMsSUFBSSxDQUFDLEVBQ3JDO1lBQ0EsTUFBTSxNQUFNLEdBQUcsa0NBQWtDLENBQUM7WUFDbEQsTUFBTSxJQUFJLEdBQ1Isa0lBQWtJLENBQUM7WUFDckksTUFBTSxXQUFXLEdBQ2YsMFBBQTBQLENBQUM7WUFDN1AscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxvQ0FBb0MsQ0FBQztZQUMvRCxDQUFDLHNDQUFnQyxDQUFDLElBQUksQ0FBQyxFQUN2QztZQUNBLE1BQU0sTUFBTSxHQUFHLG9DQUFvQyxDQUFDO1lBQ3BELE1BQU0sSUFBSSxHQUNSLGlJQUFpSSxDQUFDO1lBQ3BJLE1BQU0sV0FBVyxHQUNmLHdJQUF3SSxDQUFDO1lBQzNJLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7SUFDSCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNLLGFBQWEsQ0FBQyxJQUFpQixFQUFFLE9BQVk7UUFDbkQsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLHNDQUFzQyxDQUFDO1lBQ2pFLENBQUMsNkNBQWtDLENBQUMsSUFBSSxDQUFDLEVBQ3pDO1lBQ0EsTUFBTSxNQUFNLEdBQUcsc0NBQXNDLENBQUM7WUFDdEQsTUFBTSxJQUFJLEdBQ1Isb01BQW9NLENBQUM7WUFDdk0sTUFBTSxXQUFXLEdBQ2YsMFVBQTBVLENBQUM7WUFDN1UscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUNkLE9BQU8sRUFDUCw2Q0FBNkMsQ0FDOUM7WUFDRCxDQUFDLG9EQUF5QyxDQUFDLElBQUksQ0FBQyxFQUNoRDtZQUNBLE1BQU0sTUFBTSxHQUFHLDZDQUE2QyxDQUFDO1lBQzdELE1BQU0sSUFBSSxHQUNSLDJJQUEySSxDQUFDO1lBQzlJLE1BQU0sV0FBVyxHQUNmLDZTQUE2UyxDQUFDO1lBQ2hULHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FDZCxPQUFPLEVBQ1AsbURBQW1ELENBQ3BEO1lBQ0QsQ0FBQywwREFBK0MsQ0FBQyxJQUFJLENBQUMsRUFDdEQ7WUFDQSxNQUFNLE1BQU0sR0FBRyxtREFBbUQsQ0FBQztZQUNuRSxNQUFNLElBQUksR0FDUiw2TUFBNk0sQ0FBQztZQUNoTixNQUFNLFdBQVcsR0FDZiwrTUFBK00sQ0FBQztZQUNsTixxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQzNCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO1FBQ0QsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLDRDQUE0QyxDQUFDO1lBQ3ZFLENBQUMsbURBQXdDLENBQUMsSUFBSSxDQUFDLEVBQy9DO1lBQ0EsTUFBTSxNQUFNLEdBQUcsNENBQTRDLENBQUM7WUFDNUQsTUFBTSxJQUFJLEdBQ1Isd0pBQXdKLENBQUM7WUFDM0osTUFBTSxXQUFXLEdBQ2YsNkhBQTZILENBQUM7WUFDaEkscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUNkLE9BQU8sRUFDUCxrREFBa0QsQ0FDbkQ7WUFDRCxDQUFDLHlEQUE4QyxDQUFDLElBQUksQ0FBQyxFQUNyRDtZQUNBLE1BQU0sTUFBTSxHQUFHLGtEQUFrRCxDQUFDO1lBQ2xFLE1BQU0sSUFBSSxHQUNSLGlHQUFpRyxDQUFDO1lBQ3BHLE1BQU0sV0FBVyxHQUNmLCtTQUErUyxDQUFDO1lBQ2xULHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7SUFDSCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNLLE9BQU8sQ0FBQyxJQUFpQixFQUFFLE9BQVk7UUFDN0MsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQ2QsT0FBTyxFQUNQLG9EQUFvRCxDQUNyRDtZQUNELENBQUMscURBQWdELENBQUMsSUFBSSxDQUFDLEVBQ3ZEO1lBQ0EsTUFBTSxNQUFNLEdBQUcsb0RBQW9ELENBQUM7WUFDcEUsTUFBTSxJQUFJLEdBQ1IsMExBQTBMLENBQUM7WUFDN0wsTUFBTSxXQUFXLEdBQ2YsMEdBQTBHLENBQUM7WUFDN0cscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSx1Q0FBdUMsQ0FBQztZQUNsRSxDQUFDLHdDQUFtQyxDQUFDLElBQUksQ0FBQyxFQUMxQztZQUNBLE1BQU0sTUFBTSxHQUFHLHVDQUF1QyxDQUFDO1lBQ3ZELE1BQU0sSUFBSSxHQUNSLDJHQUEyRyxDQUFDO1lBQzlHLE1BQU0sV0FBVyxHQUNmLGtjQUFrYyxDQUFDO1lBQ3JjLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FDZCxPQUFPLEVBQ1AsNkNBQTZDLENBQzlDO1lBQ0QsQ0FBQyw4Q0FBeUMsQ0FBQyxJQUFJLENBQUMsRUFDaEQ7WUFDQSxNQUFNLE1BQU0sR0FBRyw2Q0FBNkMsQ0FBQztZQUM3RCxNQUFNLElBQUksR0FDUixnT0FBZ08sQ0FBQztZQUNuTyxNQUFNLFdBQVcsR0FDZixvRkFBb0YsQ0FBQztZQUN2RixxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQzNCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO1FBQ0QsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQ2QsT0FBTyxFQUNQLDhDQUE4QyxDQUMvQztZQUNELENBQUMsK0NBQTBDLENBQUMsSUFBSSxDQUFDLEVBQ2pEO1lBQ0EsTUFBTSxNQUFNLEdBQUcsOENBQThDLENBQUM7WUFDOUQsTUFBTSxJQUFJLEdBQ1IsaU9BQWlPLENBQUM7WUFDcE8sTUFBTSxXQUFXLEdBQ2Ysb0ZBQW9GLENBQUM7WUFDdkYscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSwyQ0FBMkMsQ0FBQztZQUN0RSxDQUFDLDRDQUF1QyxDQUFDLElBQUksQ0FBQyxFQUM5QztZQUNBLE1BQU0sTUFBTSxHQUFHLDJDQUEyQyxDQUFDO1lBQzNELE1BQU0sSUFBSSxHQUNSLGdJQUFnSSxDQUFDO1lBQ25JLE1BQU0sV0FBVyxHQUNmLHdRQUF3USxDQUFDO1lBQzNRLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FDZCxPQUFPLEVBQ1Asb0RBQW9ELENBQ3JEO1lBQ0QsQ0FBQyxxREFBZ0QsQ0FBQyxJQUFJLENBQUMsRUFDdkQ7WUFDQSxNQUFNLE1BQU0sR0FBRyxvREFBb0QsQ0FBQztZQUNwRSxNQUFNLElBQUksR0FDUiwwSUFBMEksQ0FBQztZQUM3SSxNQUFNLFdBQVcsR0FDZiw2R0FBNkcsQ0FBQztZQUNoSCxxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQzNCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO1FBQ0QsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLDBDQUEwQyxDQUFDO1lBQ3JFLENBQUMsMkNBQXNDLENBQUMsSUFBSSxDQUFDLEVBQzdDO1lBQ0EsTUFBTSxNQUFNLEdBQUcsMENBQTBDLENBQUM7WUFDMUQsTUFBTSxJQUFJLEdBQ1IsNkpBQTZKLENBQUM7WUFDaEssTUFBTSxXQUFXLEdBQ2YsZ05BQWdOLENBQUM7WUFDbk4scUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSx1Q0FBdUMsQ0FBQztZQUNsRSxDQUFDLHdDQUFtQyxDQUFDLElBQUksQ0FBQyxFQUMxQztZQUNBLE1BQU0sTUFBTSxHQUFHLHVDQUF1QyxDQUFDO1lBQ3ZELE1BQU0sSUFBSSxHQUNSLGlIQUFpSCxDQUFDO1lBQ3BILE1BQU0sV0FBVyxHQUNmLHFOQUFxTixDQUFDO1lBQ3hOLHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7SUFDSCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNLLGNBQWMsQ0FBQyxJQUFpQixFQUFFLE9BQVk7UUFDcEQsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQ2QsT0FBTyxFQUNQLCtEQUErRCxDQUNoRTtZQUNELENBQUMsdUVBQTJELENBQUMsSUFBSSxDQUFDLEVBQ2xFO1lBQ0EsTUFBTSxNQUFNLEdBQ1YsK0RBQStELENBQUM7WUFDbEUsTUFBTSxJQUFJLEdBQ1IsK0dBQStHLENBQUM7WUFDbEgsTUFBTSxXQUFXLEdBQ2Ysc0hBQXNILENBQUM7WUFDekgscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtRQUNELElBQ0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUNkLE9BQU8sRUFDUCwwREFBMEQsQ0FDM0Q7WUFDRCxDQUFDLGtFQUFzRCxDQUFDLElBQUksQ0FBQyxFQUM3RDtZQUNBLE1BQU0sTUFBTSxHQUFHLDBEQUEwRCxDQUFDO1lBQzFFLE1BQU0sSUFBSSxHQUNSLCtHQUErRyxDQUFDO1lBQ2xILE1BQU0sV0FBVyxHQUNmLHNIQUFzSCxDQUFDO1lBQ3pILHFCQUFXLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FDM0IsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUM5QyxDQUFDO1NBQ0g7UUFDRCxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FDZCxPQUFPLEVBQ1Asd0RBQXdELENBQ3pEO1lBQ0QsQ0FBQyxnRUFBb0QsQ0FBQyxJQUFJLENBQUMsRUFDM0Q7WUFDQSxNQUFNLE1BQU0sR0FBRyx3REFBd0QsQ0FBQztZQUN4RSxNQUFNLElBQUksR0FDUiw2S0FBNkssQ0FBQztZQUNoTCxNQUFNLFdBQVcsR0FDZiw4R0FBOEcsQ0FBQztZQUNqSCxxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQzNCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO0lBQ0gsQ0FBQztJQUVEOzs7O09BSUc7SUFDSyxtQkFBbUIsQ0FBQyxJQUFpQixFQUFFLE9BQVk7UUFDekQsSUFDRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLDBDQUEwQyxDQUFDO1lBQ3JFLENBQUMsdURBQXNDLENBQUMsSUFBSSxDQUFDLEVBQzdDO1lBQ0EsTUFBTSxNQUFNLEdBQUcsMENBQTBDLENBQUM7WUFDMUQsTUFBTSxJQUFJLEdBQ1Isb0hBQW9ILENBQUM7WUFDdkgsTUFBTSxXQUFXLEdBQ2YsaVFBQWlRLENBQUM7WUFDcFEscUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQzlDLENBQUM7U0FDSDtJQUNILENBQUM7SUFFRDs7OztPQUlHO0lBQ0ssUUFBUSxDQUFDLElBQWlCLEVBQUUsT0FBWTtRQUM5QyxJQUNFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsZ0NBQWdDLENBQUM7WUFDM0QsQ0FBQyxrQ0FBNEIsQ0FBQyxJQUFJLENBQUMsRUFDbkM7WUFDQSxNQUFNLE1BQU0sR0FBRyxnQ0FBZ0MsQ0FBQztZQUNoRCxNQUFNLElBQUksR0FDUiwyR0FBMkcsQ0FBQztZQUM5RyxNQUFNLFdBQVcsR0FDZixzSEFBc0gsQ0FBQztZQUN6SCxxQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQzNCLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FDOUMsQ0FBQztTQUNIO0lBQ0gsQ0FBQzs7QUF4dENILGtEQWd1Q0MiLCJzb3VyY2VzQ29udGVudCI6WyIvKlxuQ29weXJpZ2h0IEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG5TUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogQXBhY2hlLTIuMFxuKi9cblxuaW1wb3J0IHsgQW5ub3RhdGlvbnMsIENmblJlc291cmNlLCBJQ29uc3RydWN0IH0gZnJvbSAnbW9ub2Nkayc7XG5pbXBvcnQgeyBOYWdQYWNrIH0gZnJvbSAnLi4vY29tbW9uJztcbmltcG9ydCB7XG4gIGhpcGFhU2VjdXJpdHlBUElHV0NhY2hlRW5hYmxlZEFuZEVuY3J5cHRlZCxcbiAgaGlwYWFTZWN1cml0eUFQSUdXRXhlY3V0aW9uTG9nZ2luZ0VuYWJsZWQsXG4gIGhpcGFhU2VjdXJpdHlBUElHV1NTTEVuYWJsZWQsXG4gIGhpcGFhU2VjdXJpdHlBUElHV1hyYXlFbmFibGVkLFxufSBmcm9tICcuL3J1bGVzL2FwaWd3JztcbmltcG9ydCB7XG4gIGhpcGFhU2VjdXJpdHlBdXRvc2NhbGluZ0dyb3VwRUxCSGVhbHRoQ2hlY2tSZXF1aXJlZCxcbiAgaGlwYWFTZWN1cml0eUF1dG9zY2FsaW5nTGF1bmNoQ29uZmlnUHVibGljSXBEaXNhYmxlZCxcbn0gZnJvbSAnLi9ydWxlcy9hdXRvc2NhbGluZyc7XG5pbXBvcnQge1xuICBoaXBhYVNlY3VyaXR5Q2xvdWRUcmFpbENsb3VkV2F0Y2hMb2dzRW5hYmxlZCxcbiAgaGlwYWFTZWN1cml0eUNsb3VkVHJhaWxFbmNyeXB0aW9uRW5hYmxlZCxcbiAgaGlwYWFTZWN1cml0eUNsb3VkVHJhaWxMb2dGaWxlVmFsaWRhdGlvbkVuYWJsZWQsXG59IGZyb20gJy4vcnVsZXMvY2xvdWR0cmFpbCc7XG5pbXBvcnQge1xuICBoaXBhYVNlY3VyaXR5Q2xvdWRXYXRjaEFsYXJtQWN0aW9uLFxuICBoaXBhYVNlY3VyaXR5Q2xvdWRXYXRjaExvZ0dyb3VwRW5jcnlwdGVkLFxuICBoaXBhYVNlY3VyaXR5Q2xvdWRXYXRjaExvZ0dyb3VwUmV0ZW50aW9uUGVyaW9kLFxufSBmcm9tICcuL3J1bGVzL2Nsb3Vkd2F0Y2gnO1xuaW1wb3J0IHtcbiAgaGlwYWFTZWN1cml0eUNvZGVCdWlsZFByb2plY3RFbnZWYXJBd3NDcmVkLFxuICBoaXBhYVNlY3VyaXR5Q29kZUJ1aWxkUHJvamVjdFNvdXJjZVJlcG9VcmwsXG59IGZyb20gJy4vcnVsZXMvY29kZWJ1aWxkJztcbmltcG9ydCB7IGhpcGFhU2VjdXJpdHlETVNSZXBsaWNhdGlvbk5vdFB1YmxpYyB9IGZyb20gJy4vcnVsZXMvZG1zJztcbmltcG9ydCB7IGhpcGFhU2VjdXJpdHlEeW5hbW9EQlBJVFJFbmFibGVkIH0gZnJvbSAnLi9ydWxlcy9keW5hbW9kYic7XG5pbXBvcnQge1xuICBoaXBhYVNlY3VyaXR5RUMySW5zdGFuY2VEZXRhaWxlZE1vbml0b3JpbmdFbmFibGVkLFxuICBoaXBhYVNlY3VyaXR5RUMySW5zdGFuY2VzSW5WUEMsXG4gIGhpcGFhU2VjdXJpdHlFQzJJbnN0YW5jZU5vUHVibGljSXAsXG59IGZyb20gJy4vcnVsZXMvZWMyJztcbmltcG9ydCB7IGhpcGFhU2VjdXJpdHlFQ1NUYXNrRGVmaW5pdGlvblVzZXJGb3JIb3N0TW9kZSB9IGZyb20gJy4vcnVsZXMvZWNzJztcbmltcG9ydCB7IGhpcGFhU2VjdXJpdHlFRlNFbmNyeXB0ZWQgfSBmcm9tICcuL3J1bGVzL2Vmcyc7XG5pbXBvcnQgeyBoaXBhYVNlY3VyaXR5RWxhc3RpQ2FjaGVSZWRpc0NsdXN0ZXJBdXRvbWF0aWNCYWNrdXAgfSBmcm9tICcuL3J1bGVzL2VsYXN0aWNhY2hlJztcbmltcG9ydCB7XG4gIGhpcGFhU2VjdXJpdHlFbGFzdGljQmVhbnN0YWxrRW5oYW5jZWRIZWFsdGhSZXBvcnRpbmdFbmFibGVkLFxuICBoaXBhYVNlY3VyaXR5RWxhc3RpY0JlYW5zdGFsa01hbmFnZWRVcGRhdGVzRW5hYmxlZCxcbn0gZnJvbSAnLi9ydWxlcy9lbGFzdGljYmVhbnN0YWxrJztcbmltcG9ydCB7XG4gIGhpcGFhU2VjdXJpdHlBTEJIdHRwRHJvcEludmFsaWRIZWFkZXJFbmFibGVkLFxuICBoaXBhYVNlY3VyaXR5QUxCSHR0cFRvSHR0cHNSZWRpcmVjdGlvbixcbiAgaGlwYWFTZWN1cml0eUVMQkFDTUNlcnRpZmljYXRlUmVxdWlyZWQsXG4gIGhpcGFhU2VjdXJpdHlFTEJDcm9zc1pvbmVCYWxhbmNpbmdFbmFibGVkLFxuICBoaXBhYVNlY3VyaXR5RUxCRGVsZXRpb25Qcm90ZWN0aW9uRW5hYmxlZCxcbiAgaGlwYWFTZWN1cml0eUVMQkxvZ2dpbmdFbmFibGVkLFxuICBoaXBhYVNlY3VyaXR5RUxCVGxzSHR0cHNMaXN0ZW5lcnNPbmx5LFxuICBoaXBhYVNlY3VyaXR5RUxCdjJBQ01DZXJ0aWZpY2F0ZVJlcXVpcmVkLFxufSBmcm9tICcuL3J1bGVzL2VsYic7XG5pbXBvcnQgeyBoaXBhYVNlY3VyaXR5RU1SS2VyYmVyb3NFbmFibGVkIH0gZnJvbSAnLi9ydWxlcy9lbXInO1xuaW1wb3J0IHtcbiAgaGlwYWFTZWN1cml0eUlBTU5vSW5saW5lUG9saWN5LFxuICBoaXBhYVNlY3VyaXR5SUFNUG9saWN5Tm9TdGF0ZW1lbnRzV2l0aEFkbWluQWNjZXNzLFxuICBoaXBhYVNlY3VyaXR5SUFNUG9saWN5Tm9TdGF0ZW1lbnRzV2l0aEZ1bGxBY2Nlc3MsXG4gIGhpcGFhU2VjdXJpdHlJQU1Vc2VyR3JvdXBNZW1iZXJzaGlwLFxuICBoaXBhYVNlY3VyaXR5SUFNVXNlck5vUG9saWNpZXMsXG59IGZyb20gJy4vcnVsZXMvaWFtJztcbmltcG9ydCB7XG4gIGhpcGFhU2VjdXJpdHlMYW1iZGFDb25jdXJyZW5jeSxcbiAgaGlwYWFTZWN1cml0eUxhbWJkYURscSxcbiAgaGlwYWFTZWN1cml0eUxhbWJkYUluc2lkZVZQQyxcbn0gZnJvbSAnLi9ydWxlcy9sYW1iZGEnO1xuaW1wb3J0IHtcbiAgaGlwYWFTZWN1cml0eU9wZW5TZWFyY2hFbmNyeXB0ZWRBdFJlc3QsXG4gIGhpcGFhU2VjdXJpdHlPcGVuU2VhcmNoSW5WUENPbmx5LFxuICBoaXBhYVNlY3VyaXR5T3BlblNlYXJjaExvZ3NUb0Nsb3VkV2F0Y2gsXG4gIGhpcGFhU2VjdXJpdHlPcGVuU2VhcmNoTm9kZVRvTm9kZUVuY3J5cHRpb24sXG59IGZyb20gJy4vcnVsZXMvb3BlbnNlYXJjaCc7XG5pbXBvcnQge1xuICBoaXBhYVNlY3VyaXR5UkRTQXV0b21hdGljTWlub3JWZXJzaW9uVXBncmFkZUVuYWJsZWQsXG4gIGhpcGFhU2VjdXJpdHlSRFNFbmhhbmNlZE1vbml0b3JpbmdFbmFibGVkLFxuICBoaXBhYVNlY3VyaXR5UkRTSW5zdGFuY2VCYWNrdXBFbmFibGVkLFxuICBoaXBhYVNlY3VyaXR5UkRTSW5zdGFuY2VEZWxldGlvblByb3RlY3Rpb25FbmFibGVkLFxuICBoaXBhYVNlY3VyaXR5UkRTSW5zdGFuY2VNdWx0aUFaU3VwcG9ydCxcbiAgaGlwYWFTZWN1cml0eVJEU0luc3RhbmNlUHVibGljQWNjZXNzLFxuICBoaXBhYVNlY3VyaXR5UkRTTG9nZ2luZ0VuYWJsZWQsXG4gIGhpcGFhU2VjdXJpdHlSRFNTdG9yYWdlRW5jcnlwdGVkLFxufSBmcm9tICcuL3J1bGVzL3Jkcyc7XG5pbXBvcnQge1xuICBoaXBhYVNlY3VyaXR5UmVkc2hpZnRCYWNrdXBFbmFibGVkLFxuICBoaXBhYVNlY3VyaXR5UmVkc2hpZnRDbHVzdGVyQ29uZmlndXJhdGlvbixcbiAgaGlwYWFTZWN1cml0eVJlZHNoaWZ0Q2x1c3Rlck1haW50ZW5hbmNlU2V0dGluZ3MsXG4gIGhpcGFhU2VjdXJpdHlSZWRzaGlmdENsdXN0ZXJQdWJsaWNBY2Nlc3MsXG4gIGhpcGFhU2VjdXJpdHlSZWRzaGlmdEVuaGFuY2VkVlBDUm91dGluZ0VuYWJsZWQsXG59IGZyb20gJy4vcnVsZXMvcmVkc2hpZnQnO1xuaW1wb3J0IHtcbiAgaGlwYWFTZWN1cml0eVMzQnVja2V0TGV2ZWxQdWJsaWNBY2Nlc3NQcm9oaWJpdGVkLFxuICBoaXBhYVNlY3VyaXR5UzNCdWNrZXRMb2dnaW5nRW5hYmxlZCxcbiAgaGlwYWFTZWN1cml0eVMzQnVja2V0UHVibGljUmVhZFByb2hpYml0ZWQsXG4gIGhpcGFhU2VjdXJpdHlTM0J1Y2tldFB1YmxpY1dyaXRlUHJvaGliaXRlZCxcbiAgaGlwYWFTZWN1cml0eVMzQnVja2V0UmVwbGljYXRpb25FbmFibGVkLFxuICBoaXBhYVNlY3VyaXR5UzNCdWNrZXRTZXJ2ZXJTaWRlRW5jcnlwdGlvbkVuYWJsZWQsXG4gIGhpcGFhU2VjdXJpdHlTM0J1Y2tldFZlcnNpb25pbmdFbmFibGVkLFxuICBoaXBhYVNlY3VyaXR5UzNEZWZhdWx0RW5jcnlwdGlvbktNUyxcbn0gZnJvbSAnLi9ydWxlcy9zMyc7XG5pbXBvcnQge1xuICBoaXBhYVNlY3VyaXR5U2FnZU1ha2VyRW5kcG9pbnRDb25maWd1cmF0aW9uS01TS2V5Q29uZmlndXJlZCxcbiAgaGlwYWFTZWN1cml0eVNhZ2VNYWtlck5vdGVib29rSW5zdGFuY2VLTVNLZXlDb25maWd1cmVkLFxuICBoaXBhYVNlY3VyaXR5U2FnZU1ha2VyTm90ZWJvb2tOb0RpcmVjdEludGVybmV0QWNjZXNzLFxufSBmcm9tICcuL3J1bGVzL3NhZ2VtYWtlcic7XG5pbXBvcnQgeyBoaXBhYVNlY3VyaXR5U2VjcmV0c01hbmFnZXJVc2luZ0tNU0tleSB9IGZyb20gJy4vcnVsZXMvc2VjcmV0c21hbmFnZXInO1xuaW1wb3J0IHsgaGlwYWFTZWN1cml0eVNOU0VuY3J5cHRlZEtNUyB9IGZyb20gJy4vcnVsZXMvc25zJztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBISVBBQVNlY3VyaXR5Q2hlY2tzIGV4dGVuZHMgTmFnUGFjayB7XG4gIHB1YmxpYyB2aXNpdChub2RlOiBJQ29uc3RydWN0KTogdm9pZCB7XG4gICAgaWYgKG5vZGUgaW5zdGFuY2VvZiBDZm5SZXNvdXJjZSkge1xuICAgICAgLy8gR2V0IGlnbm9yZXMgbWV0YWRhdGEgaWYgaXQgZXhpc3RzXG4gICAgICBjb25zdCBpZ25vcmVzID0gbm9kZS5nZXRNZXRhZGF0YSgnY2RrX25hZycpPy5ydWxlc190b19zdXBwcmVzcztcbiAgICAgIHRoaXMuY2hlY2tBUElHVyhub2RlLCBpZ25vcmVzKTtcbiAgICAgIHRoaXMuY2hlY2tBdXRvU2NhbGluZyhub2RlLCBpZ25vcmVzKTtcbiAgICAgIHRoaXMuY2hlY2tDbG91ZFRyYWlsKG5vZGUsIGlnbm9yZXMpO1xuICAgICAgdGhpcy5jaGVja0Nsb3VkV2F0Y2gobm9kZSwgaWdub3Jlcyk7XG4gICAgICB0aGlzLmNoZWNrQ29kZUJ1aWxkKG5vZGUsIGlnbm9yZXMpO1xuICAgICAgdGhpcy5jaGVja0RNUyhub2RlLCBpZ25vcmVzKTtcbiAgICAgIHRoaXMuY2hlY2tEeW5hbW9EQihub2RlLCBpZ25vcmVzKTtcbiAgICAgIHRoaXMuY2hlY2tFQzIobm9kZSwgaWdub3Jlcyk7XG4gICAgICB0aGlzLmNoZWNrRUNTKG5vZGUsIGlnbm9yZXMpO1xuICAgICAgdGhpcy5jaGVja0VGUyhub2RlLCBpZ25vcmVzKTtcbiAgICAgIHRoaXMuY2hlY2tFbGFzdGlDYWNoZShub2RlLCBpZ25vcmVzKTtcbiAgICAgIHRoaXMuY2hlY2tFbGFzdGljQmVhbnN0YWxrKG5vZGUsIGlnbm9yZXMpO1xuICAgICAgdGhpcy5jaGVja0VMQihub2RlLCBpZ25vcmVzKTtcbiAgICAgIHRoaXMuY2hlY2tFTVIobm9kZSwgaWdub3Jlcyk7XG4gICAgICB0aGlzLmNoZWNrSUFNKG5vZGUsIGlnbm9yZXMpO1xuICAgICAgdGhpcy5jaGVja0xhbWJkYShub2RlLCBpZ25vcmVzKTtcbiAgICAgIHRoaXMuY2hlY2tPcGVuU2VhcmNoKG5vZGUsIGlnbm9yZXMpO1xuICAgICAgdGhpcy5jaGVja1JEUyhub2RlLCBpZ25vcmVzKTtcbiAgICAgIHRoaXMuY2hlY2tSZWRzaGlmdChub2RlLCBpZ25vcmVzKTtcbiAgICAgIHRoaXMuY2hlY2tTMyhub2RlLCBpZ25vcmVzKTtcbiAgICAgIHRoaXMuY2hlY2tTYWdlTWFrZXIobm9kZSwgaWdub3Jlcyk7XG4gICAgICB0aGlzLmNoZWNrU2VjcmV0c01hbmFnZXIobm9kZSwgaWdub3Jlcyk7XG4gICAgICB0aGlzLmNoZWNrU05TKG5vZGUsIGlnbm9yZXMpO1xuICAgICAgLy8gdGhpcy5jaGVja1ZQQyhub2RlLCBpZ25vcmVzKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogQ2hlY2sgQVBJIEdhdGV3YXkgUmVzb3VyY2VzXG4gICAqIEBwYXJhbSBub2RlIHRoZSBJQ29uc3RydWN0IHRvIGV2YWx1YXRlXG4gICAqIEBwYXJhbSBpZ25vcmVzIGxpc3Qgb2YgaWdub3JlcyBmb3IgdGhlIHJlc291cmNlXG4gICAqL1xuICBwcml2YXRlIGNoZWNrQVBJR1cobm9kZTogQ2ZuUmVzb3VyY2UsIGlnbm9yZXM6IGFueSk6IHZvaWQge1xuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoXG4gICAgICAgIGlnbm9yZXMsXG4gICAgICAgICdISVBBQS5TZWN1cml0eS1BUElHV0NhY2hlRW5hYmxlZEFuZEVuY3J5cHRlZCcsXG4gICAgICApICYmXG4gICAgICAhaGlwYWFTZWN1cml0eUFQSUdXQ2FjaGVFbmFibGVkQW5kRW5jcnlwdGVkKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnSElQQUEuU2VjdXJpdHktQVBJR1dDYWNoZUVuYWJsZWRBbmRFbmNyeXB0ZWQnO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgQVBJIEdhdGV3YXkgc3RhZ2UgZG9lcyBub3QgaGF2ZSBjYWNoaW5nIGVuYWJsZWQgYW5kIGVuY3J5cHRlZCBmb3IgYWxsIG1ldGhvZHMgLSAoQ29udHJvbCBJRHM6IDE2NC4zMTIoYSkoMikoaXYpLCAxNjQuMzEyKGUpKDIpKGlpKSkuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgXCJUbyBoZWxwIHByb3RlY3QgZGF0YSBhdCByZXN0LCBlbnN1cmUgZW5jcnlwdGlvbiBpcyBlbmFibGVkIGZvciB5b3VyIEFQSSBHYXRld2F5IHN0YWdlJ3MgY2FjaGUuIEJlY2F1c2Ugc2Vuc2l0aXZlIGRhdGEgY2FuIGJlIGNhcHR1cmVkIGZvciB0aGUgQVBJIG1ldGhvZCwgZW5hYmxlIGVuY3J5cHRpb24gYXQgcmVzdCB0byBoZWxwIHByb3RlY3QgdGhhdCBkYXRhLlwiO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoXG4gICAgICAgIGlnbm9yZXMsXG4gICAgICAgICdISVBBQS5TZWN1cml0eS1BUElHV0V4ZWN1dGlvbkxvZ2dpbmdFbmFibGVkJyxcbiAgICAgICkgJiZcbiAgICAgICFoaXBhYVNlY3VyaXR5QVBJR1dFeGVjdXRpb25Mb2dnaW5nRW5hYmxlZChub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0hJUEFBLlNlY3VyaXR5LUFQSUdXRXhlY3V0aW9uTG9nZ2luZ0VuYWJsZWQnO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgQVBJIEdhdGV3YXkgc3RhZ2UgZG9lcyBub3QgaGF2ZSBleGVjdXRpb24gbG9nZ2luZyBlbmFibGVkIGZvciBhbGwgbWV0aG9kcyAtIChDb250cm9sIElEOiAxNjQuMzEyKGIpKS4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnQVBJIEdhdGV3YXkgbG9nZ2luZyBkaXNwbGF5cyBkZXRhaWxlZCB2aWV3cyBvZiB1c2VycyB3aG8gYWNjZXNzZWQgdGhlIEFQSSBhbmQgdGhlIHdheSB0aGV5IGFjY2Vzc2VkIHRoZSBBUEkuIFRoaXMgaW5zaWdodCBlbmFibGVzIHZpc2liaWxpdHkgb2YgdXNlciBhY3Rpdml0aWVzLic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShpZ25vcmVzLCAnSElQQUEuU2VjdXJpdHktQVBJR1dTU0xFbmFibGVkJykgJiZcbiAgICAgICFoaXBhYVNlY3VyaXR5QVBJR1dTU0xFbmFibGVkKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnSElQQUEuU2VjdXJpdHktQVBJR1dTU0xFbmFibGVkJztcbiAgICAgIGNvbnN0IGluZm8gPVxuICAgICAgICAnVGhlIEFQSSBHYXRld2F5IFJFU1QgQVBJIHN0YWdlIGlzIG5vdCBjb25maWd1cmVkIHdpdGggU1NMIGNlcnRpZmljYXRlcyAtIChDb250cm9sIElEczogMTY0LjMxMihhKSgyKShpdiksIDE2NC4zMTIoZSkoMSksIDE2NC4zMTIoZSkoMikoaSksIDE2NC4zMTIoZSkoMikoaWkpKS4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnRW5zdXJlIEFtYXpvbiBBUEkgR2F0ZXdheSBSRVNUIEFQSSBzdGFnZXMgYXJlIGNvbmZpZ3VyZWQgd2l0aCBTU0wgY2VydGlmaWNhdGVzIHRvIGFsbG93IGJhY2tlbmQgc3lzdGVtcyB0byBhdXRoZW50aWNhdGUgdGhhdCByZXF1ZXN0cyBvcmlnaW5hdGUgZnJvbSBBUEkgR2F0ZXdheS4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoaWdub3JlcywgJ0hJUEFBLlNlY3VyaXR5LUFQSUdXWHJheUVuYWJsZWQnKSAmJlxuICAgICAgIWhpcGFhU2VjdXJpdHlBUElHV1hyYXlFbmFibGVkKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnSElQQUEuU2VjdXJpdHktQVBJR1dYcmF5RW5hYmxlZCc7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSBBUEkgR2F0ZXdheSBSRVNUIEFQSSBzdGFnZSBkb2VzIG5vdCBoYXZlIFgtUmF5IGVuYWJsZWQgLSAoQ29udHJvbCBJRDogMTY0LjMxMihiKSkuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ0FXUyBYLVJheSBjb2xsZWN0cyBkYXRhIGFib3V0IHJlcXVlc3RzIHRoYXQgeW91ciBhcHBsaWNhdGlvbiBzZXJ2ZXMsIGFuZCBwcm92aWRlcyB0b29scyB5b3UgY2FuIHVzZSB0byB2aWV3LCBmaWx0ZXIsIGFuZCBnYWluIGluc2lnaHRzIGludG8gdGhhdCBkYXRhIHRvIGlkZW50aWZ5IGlzc3VlcyBhbmQgb3Bwb3J0dW5pdGllcyBmb3Igb3B0aW1pemF0aW9uLiBFbnN1cmUgWC1SYXkgaXMgZW5hYmxlcyBzbyB5b3UgY2FuIHNlZSBkZXRhaWxlZCBpbmZvcm1hdGlvbiBub3Qgb25seSBhYm91dCB0aGUgcmVxdWVzdCBhbmQgcmVzcG9uc2UsIGJ1dCBhbHNvIGFib3V0IGNhbGxzIHRoYXQgeW91ciBhcHBsaWNhdGlvbiBtYWtlcyB0byBkb3duc3RyZWFtIEFXUyByZXNvdXJjZXMsIG1pY3Jvc2VydmljZXMsIGRhdGFiYXNlcyBhbmQgSFRUUCB3ZWIgQVBJcy4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIENoZWNrIEF1dG8gU2NhbGluZyBSZXNvdXJjZXNcbiAgICogQHBhcmFtIG5vZGUgdGhlIElDb25zdHJ1Y3QgdG8gZXZhbHVhdGVcbiAgICogQHBhcmFtIGlnbm9yZXMgbGlzdCBvZiBpZ25vcmVzIGZvciB0aGUgcmVzb3VyY2VcbiAgICovXG4gIHByaXZhdGUgY2hlY2tBdXRvU2NhbGluZyhub2RlOiBDZm5SZXNvdXJjZSwgaWdub3JlczogYW55KTogdm9pZCB7XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShcbiAgICAgICAgaWdub3JlcyxcbiAgICAgICAgJ0hJUEFBLlNlY3VyaXR5LUF1dG9zY2FsaW5nR3JvdXBFTEJIZWFsdGhDaGVja1JlcXVpcmVkJyxcbiAgICAgICkgJiZcbiAgICAgICFoaXBhYVNlY3VyaXR5QXV0b3NjYWxpbmdHcm91cEVMQkhlYWx0aENoZWNrUmVxdWlyZWQobm9kZSlcbiAgICApIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdISVBBQS5TZWN1cml0eS1BdXRvc2NhbGluZ0dyb3VwRUxCSGVhbHRoQ2hlY2tSZXF1aXJlZCc7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSBBdXRvIFNjYWxpbmcgZ3JvdXAgdXRpbGl6ZXMgYSBsb2FkIGJhbGFuY2VyIGFuZCBkb2VzIG5vdCBoYXZlIGFuIEVMQiBoZWFsdGggY2hlY2sgY29uZmlndXJlZCAtIChDb250cm9sIElEOiAxNjQuMzEyKGIpKS4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnVGhlIEVsYXN0aWMgTG9hZCBCYWxhbmNlciAoRUxCKSBoZWFsdGggY2hlY2tzIGZvciBBbWF6b24gRWxhc3RpYyBDb21wdXRlIENsb3VkIChBbWF6b24gRUMyKSBBdXRvIFNjYWxpbmcgZ3JvdXBzIHN1cHBvcnQgbWFpbnRlbmFuY2Ugb2YgYWRlcXVhdGUgY2FwYWNpdHkgYW5kIGF2YWlsYWJpbGl0eS4gVGhlIGxvYWQgYmFsYW5jZXIgcGVyaW9kaWNhbGx5IHNlbmRzIHBpbmdzLCBhdHRlbXB0cyBjb25uZWN0aW9ucywgb3Igc2VuZHMgcmVxdWVzdHMgdG8gdGVzdCBBbWF6b24gRUMyIGluc3RhbmNlcyBoZWFsdGggaW4gYW4gYXV0by1zY2FsaW5nIGdyb3VwLiBJZiBhbiBpbnN0YW5jZSBpcyBub3QgcmVwb3J0aW5nIGJhY2ssIHRyYWZmaWMgaXMgc2VudCB0byBhIG5ldyBBbWF6b24gRUMyIGluc3RhbmNlLic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShcbiAgICAgICAgaWdub3JlcyxcbiAgICAgICAgJ0hJUEFBLlNlY3VyaXR5LUF1dG9zY2FsaW5nTGF1bmNoQ29uZmlnUHVibGljSXBEaXNhYmxlZCcsXG4gICAgICApICYmXG4gICAgICAhaGlwYWFTZWN1cml0eUF1dG9zY2FsaW5nTGF1bmNoQ29uZmlnUHVibGljSXBEaXNhYmxlZChub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0hJUEFBLlNlY3VyaXR5LUF1dG9zY2FsaW5nTGF1bmNoQ29uZmlnUHVibGljSXBEaXNhYmxlZCc7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSBBdXRvIFNjYWxpbmcgbGF1bmNoIGNvbmZpZ3VyYXRpb24gZG9lcyBub3QgaGF2ZSBwdWJsaWMgSVAgYWRkcmVzc2VzIGRpc2FibGVkIC0gKENvbnRyb2wgSURzOiAxNjQuMzA4KGEpKDMpKGkpLCAxNjQuMzA4KGEpKDMpKGlpKShCKSwgMTY0LjMwOChhKSg0KShpaSkoQSksIDE2NC4zMDgoYSkoNCkoaWkpKEMpLCAxNjQuMzEyKGEpKDEpLCAxNjQuMzEyKGUpKDEpKS4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnSWYgeW91IGNvbmZpZ3VyZSB5b3VyIE5ldHdvcmsgSW50ZXJmYWNlcyB3aXRoIGEgcHVibGljIElQIGFkZHJlc3MsIHRoZW4gdGhlIGFzc29jaWF0ZWQgcmVzb3VyY2VzIHRvIHRob3NlIE5ldHdvcmsgSW50ZXJmYWNlcyBhcmUgcmVhY2hhYmxlIGZyb20gdGhlIGludGVybmV0LiBFQzIgcmVzb3VyY2VzIHNob3VsZCBub3QgYmUgcHVibGljbHkgYWNjZXNzaWJsZSwgYXMgdGhpcyBtYXkgYWxsb3cgdW5pbnRlbmRlZCBhY2Nlc3MgdG8geW91ciBhcHBsaWNhdGlvbnMgb3Igc2VydmVycy4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIENoZWNrIENsb3VkVHJhaWwgUmVzb3VyY2VzXG4gICAqIEBwYXJhbSBub2RlIHRoZSBJQ29uc3RydWN0IHRvIGV2YWx1YXRlXG4gICAqIEBwYXJhbSBpZ25vcmVzIGxpc3Qgb2YgaWdub3JlcyBmb3IgdGhlIHJlc291cmNlXG4gICAqL1xuICBwcml2YXRlIGNoZWNrQ2xvdWRUcmFpbChub2RlOiBDZm5SZXNvdXJjZSwgaWdub3JlczogYW55KTogdm9pZCB7XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShcbiAgICAgICAgaWdub3JlcyxcbiAgICAgICAgJ0hJUEFBLlNlY3VyaXR5LUNsb3VkVHJhaWxDbG91ZFdhdGNoTG9nc0VuYWJsZWQnLFxuICAgICAgKSAmJlxuICAgICAgIWhpcGFhU2VjdXJpdHlDbG91ZFRyYWlsQ2xvdWRXYXRjaExvZ3NFbmFibGVkKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnSElQQUEuU2VjdXJpdHktQ2xvdWRUcmFpbENsb3VkV2F0Y2hMb2dzRW5hYmxlZCc7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSB0cmFpbCBkb2VzIG5vdCBoYXZlIENsb3VkV2F0Y2ggbG9ncyBlbmFibGVkIC0gKENvbnRyb2wgSURzOiAxNjQuMzA4KGEpKDMpKGlpKShBKSwgMTY0LjMxMihiKSkuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ1VzZSBBbWF6b24gQ2xvdWRXYXRjaCB0byBjZW50cmFsbHkgY29sbGVjdCBhbmQgbWFuYWdlIGxvZyBldmVudCBhY3Rpdml0eS4gSW5jbHVzaW9uIG9mIEFXUyBDbG91ZFRyYWlsIGRhdGEgcHJvdmlkZXMgZGV0YWlscyBvZiBBUEkgY2FsbCBhY3Rpdml0eSB3aXRoaW4geW91ciBBV1MgYWNjb3VudC4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoaWdub3JlcywgJ0hJUEFBLlNlY3VyaXR5LUNsb3VkVHJhaWxFbmNyeXB0aW9uRW5hYmxlZCcpICYmXG4gICAgICAhaGlwYWFTZWN1cml0eUNsb3VkVHJhaWxFbmNyeXB0aW9uRW5hYmxlZChub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0hJUEFBLlNlY3VyaXR5LUNsb3VkVHJhaWxFbmNyeXB0aW9uRW5hYmxlZCc7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSB0cmFpbCBkb2VzIG5vdCBoYXZlIGEgS01TIGtleSBJRCBvciBoYXZlIGVuY3J5cHRpb24gZW5hYmxlZCAtIChDb250cm9sIElEOiAxNjQuMzEyKGEpKDIpKGl2KSwgMTY0LjMxMihlKSgyKShpaSkpLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdCZWNhdXNlIHNlbnNpdGl2ZSBkYXRhIG1heSBleGlzdCBhbmQgdG8gaGVscCBwcm90ZWN0IGRhdGEgYXQgcmVzdCwgZW5zdXJlIGVuY3J5cHRpb24gaXMgZW5hYmxlZCBmb3IgeW91ciBBV1MgQ2xvdWRUcmFpbCB0cmFpbHMuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKFxuICAgICAgICBpZ25vcmVzLFxuICAgICAgICAnSElQQUEuU2VjdXJpdHktQ2xvdWRUcmFpbExvZ0ZpbGVWYWxpZGF0aW9uRW5hYmxlZCcsXG4gICAgICApICYmXG4gICAgICAhaGlwYWFTZWN1cml0eUNsb3VkVHJhaWxMb2dGaWxlVmFsaWRhdGlvbkVuYWJsZWQobm9kZSlcbiAgICApIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdISVBBQS5TZWN1cml0eS1DbG91ZFRyYWlsTG9nRmlsZVZhbGlkYXRpb25FbmFibGVkJztcbiAgICAgIGNvbnN0IGluZm8gPVxuICAgICAgICAnVGhlIHRyYWlsIGRvZXMgbm90IGhhdmUgbG9nIGZpbGUgdmFsaWRhdGlvbiBlbmFibGVkIC0gKENvbnRyb2wgSUQ6IDE2NC4zMTIoYykoMSksIDE2NC4zMTIoYykoMikpLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdVdGlsaXplIEFXUyBDbG91ZFRyYWlsIGxvZyBmaWxlIHZhbGlkYXRpb24gdG8gY2hlY2sgdGhlIGludGVncml0eSBvZiBDbG91ZFRyYWlsIGxvZ3MuIExvZyBmaWxlIHZhbGlkYXRpb24gaGVscHMgZGV0ZXJtaW5lIGlmIGEgbG9nIGZpbGUgd2FzIG1vZGlmaWVkIG9yIGRlbGV0ZWQgb3IgdW5jaGFuZ2VkIGFmdGVyIENsb3VkVHJhaWwgZGVsaXZlcmVkIGl0LiBUaGlzIGZlYXR1cmUgaXMgYnVpbHQgdXNpbmcgaW5kdXN0cnkgc3RhbmRhcmQgYWxnb3JpdGhtczogU0hBLTI1NiBmb3IgaGFzaGluZyBhbmQgU0hBLTI1NiB3aXRoIFJTQSBmb3IgZGlnaXRhbCBzaWduaW5nLiBUaGlzIG1ha2VzIGl0IGNvbXB1dGF0aW9uYWxseSBpbmZlYXNpYmxlIHRvIG1vZGlmeSwgZGVsZXRlIG9yIGZvcmdlIENsb3VkVHJhaWwgbG9nIGZpbGVzIHdpdGhvdXQgZGV0ZWN0aW9uLic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogQ2hlY2sgQ2xvdWRXYXRjaCBSZXNvdXJjZXNcbiAgICogQHBhcmFtIG5vZGUgdGhlIElDb25zdHJ1Y3QgdG8gZXZhbHVhdGVcbiAgICogQHBhcmFtIGlnbm9yZXMgbGlzdCBvZiBpZ25vcmVzIGZvciB0aGUgcmVzb3VyY2VcbiAgICovXG4gIHByaXZhdGUgY2hlY2tDbG91ZFdhdGNoKG5vZGU6IENmblJlc291cmNlLCBpZ25vcmVzOiBhbnkpOiB2b2lkIHtcbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdISVBBQS5TZWN1cml0eS1DbG91ZFdhdGNoQWxhcm1BY3Rpb24nKSAmJlxuICAgICAgIWhpcGFhU2VjdXJpdHlDbG91ZFdhdGNoQWxhcm1BY3Rpb24obm9kZSlcbiAgICApIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdISVBBQS5TZWN1cml0eS1DbG91ZFdhdGNoQWxhcm1BY3Rpb24nO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgQ2xvdWRXYXRjaCBhbGFybSBkb2VzIG5vdCBoYXZlIGF0IGxlYXN0IG9uZSBhbGFybSBhY3Rpb24sIG9uZSBJTlNVRkZJQ0lFTlRfREFUQSBhY3Rpb24sIG9yIG9uZSBPSyBhY3Rpb24gZW5hYmxlZCAtIChDb250cm9sIElEOiAxNjQuMzEyKGIpKS4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnQW1hem9uIENsb3VkV2F0Y2ggYWxhcm1zIGFsZXJ0IHdoZW4gYSBtZXRyaWMgYnJlYWNoZXMgdGhlIHRocmVzaG9sZCBmb3IgYSBzcGVjaWZpZWQgbnVtYmVyIG9mIGV2YWx1YXRpb24gcGVyaW9kcy4gVGhlIGFsYXJtIHBlcmZvcm1zIG9uZSBvciBtb3JlIGFjdGlvbnMgYmFzZWQgb24gdGhlIHZhbHVlIG9mIHRoZSBtZXRyaWMgb3IgZXhwcmVzc2lvbiByZWxhdGl2ZSB0byBhIHRocmVzaG9sZCBvdmVyIGEgbnVtYmVyIG9mIHRpbWUgcGVyaW9kcy4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoaWdub3JlcywgJ0hJUEFBLlNlY3VyaXR5LUNsb3VkV2F0Y2hMb2dHcm91cEVuY3J5cHRlZCcpICYmXG4gICAgICAhaGlwYWFTZWN1cml0eUNsb3VkV2F0Y2hMb2dHcm91cEVuY3J5cHRlZChub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0hJUEFBLlNlY3VyaXR5LUNsb3VkV2F0Y2hMb2dHcm91cEVuY3J5cHRlZCc7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSBDbG91ZFdhdGNoIExvZyBHcm91cCBpcyBub3QgZW5jcnlwdGVkIHdpdGggYW4gQVdTIEtNUyBrZXkgLSAoQ29udHJvbCBJRHM6IDE2NC4zMTIoYSkoMikoaXYpLCAxNjQuMzEyKGUpKDIpKGlpKSkuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ1RvIGhlbHAgcHJvdGVjdCBzZW5zaXRpdmUgZGF0YSBhdCByZXN0LCBlbnN1cmUgZW5jcnlwdGlvbiBpcyBlbmFibGVkIGZvciB5b3VyIEFtYXpvbiBDbG91ZFdhdGNoIExvZyBHcm91cHMuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKFxuICAgICAgICBpZ25vcmVzLFxuICAgICAgICAnSElQQUEuU2VjdXJpdHktQ2xvdWRXYXRjaExvZ0dyb3VwUmV0ZW50aW9uUGVyaW9kJyxcbiAgICAgICkgJiZcbiAgICAgICFoaXBhYVNlY3VyaXR5Q2xvdWRXYXRjaExvZ0dyb3VwUmV0ZW50aW9uUGVyaW9kKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnSElQQUEuU2VjdXJpdHktQ2xvdWRXYXRjaExvZ0dyb3VwUmV0ZW50aW9uUGVyaW9kJztcbiAgICAgIGNvbnN0IGluZm8gPVxuICAgICAgICAnVGhlIENsb3VkV2F0Y2ggTG9nIEdyb3VwIGRvZXMgbm90IGhhdmUgYW4gZXhwbGljaXQgcmV0ZW50aW9uIHBlcmlvZCBjb25maWd1cmVkIC0gKENvbnRyb2wgSUQ6IDE2NC4zMTIoYikpLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdFbnN1cmUgYSBtaW5pbXVtIGR1cmF0aW9uIG9mIGV2ZW50IGxvZyBkYXRhIGlzIHJldGFpbmVkIGZvciB5b3VyIGxvZyBncm91cHMgdG8gaGVscCB3aXRoIHRyb3VibGVzaG9vdGluZyBhbmQgZm9yZW5zaWNzIGludmVzdGlnYXRpb25zLiBUaGUgbGFjayBvZiBhdmFpbGFibGUgcGFzdCBldmVudCBsb2cgZGF0YSBtYWtlcyBpdCBkaWZmaWN1bHQgdG8gcmVjb25zdHJ1Y3QgYW5kIGlkZW50aWZ5IHBvdGVudGlhbGx5IG1hbGljaW91cyBldmVudHMuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBDaGVjayBDb2RlQnVpbGQgUmVzb3VyY2VzXG4gICAqIEBwYXJhbSBub2RlIHRoZSBJQ29uc3RydWN0IHRvIGV2YWx1YXRlXG4gICAqIEBwYXJhbSBpZ25vcmVzIGxpc3Qgb2YgaWdub3JlcyBmb3IgdGhlIHJlc291cmNlXG4gICAqL1xuICBwcml2YXRlIGNoZWNrQ29kZUJ1aWxkKG5vZGU6IENmblJlc291cmNlLCBpZ25vcmVzOiBhbnkpOiB2b2lkIHtcbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKFxuICAgICAgICBpZ25vcmVzLFxuICAgICAgICAnSElQQUEuU2VjdXJpdHktQ29kZUJ1aWxkUHJvamVjdEVudlZhckF3c0NyZWQnLFxuICAgICAgKSAmJlxuICAgICAgIWhpcGFhU2VjdXJpdHlDb2RlQnVpbGRQcm9qZWN0RW52VmFyQXdzQ3JlZChub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0hJUEFBLlNlY3VyaXR5LUNvZGVCdWlsZFByb2plY3RFbnZWYXJBd3NDcmVkJztcbiAgICAgIGNvbnN0IGluZm8gPVxuICAgICAgICAnVGhlIENvZGVCdWlsZCBlbnZpcm9ubWVudCBzdG9yZXMgc2Vuc2l0aXZlIGNyZWRlbnRpYWxzIChzdWNoIGFzIEFXU19BQ0NFU1NfS0VZX0lEIGFuZC9vciBBV1NfU0VDUkVUX0FDQ0VTU19LRVkpIGFzIHBsYWludGV4dCBlbnZpcm9ubWVudCB2YXJpYWJsZXMgLSAoQ29udHJvbCBJRHM6IDE2NC4zMDgoYSkoMykoaSksIDE2NC4zMDgoYSkoNCkoaWkpKEEpLCAxNjQuMzA4KGEpKDQpKGlpKShDKSwgMTY0LjMxMihhKSgxKSkuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ0RvIG5vdCBzdG9yZSB0aGVzZSB2YXJpYWJsZXMgaW4gY2xlYXIgdGV4dC4gU3RvcmluZyB0aGVzZSB2YXJpYWJsZXMgaW4gY2xlYXIgdGV4dCBsZWFkcyB0byB1bmludGVuZGVkIGRhdGEgZXhwb3N1cmUgYW5kIHVuYXV0aG9yaXplZCBhY2Nlc3MuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKFxuICAgICAgICBpZ25vcmVzLFxuICAgICAgICAnSElQQUEuU2VjdXJpdHktQ29kZUJ1aWxkUHJvamVjdFNvdXJjZVJlcG9VcmwnLFxuICAgICAgKSAmJlxuICAgICAgIWhpcGFhU2VjdXJpdHlDb2RlQnVpbGRQcm9qZWN0U291cmNlUmVwb1VybChub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0hJUEFBLlNlY3VyaXR5LUNvZGVCdWlsZFByb2plY3RTb3VyY2VSZXBvVXJsJztcbiAgICAgIGNvbnN0IGluZm8gPVxuICAgICAgICAnVGhlIENvZGVCdWlsZCBwcm9qZWN0IHdoaWNoIHV0aWxpemVzIGVpdGhlciBhIEdpdEh1YiBvciBCaXRCdWNrZXQgc291cmNlIHJlcG9zaXRvcnkgZG9lcyBub3QgdXRpbGl6ZSBPQVVUSCAtIChDb250cm9sIElEczogMTY0LjMwOChhKSgzKShpKSwgMTY0LjMwOChhKSg0KShpaSkoQSksIDE2NC4zMDgoYSkoNCkoaWkpKEMpLCAxNjQuMzEyKGEpKDEpKS4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnT0FVVEggaXMgdGhlIG1vc3Qgc2VjdXJlIG1ldGhvZCBvZiBhdXRoZW50aWNhdGluZyB5b3VyIENvZGVCdWlsZCBhcHBsaWNhdGlvbi4gVXNlIE9BdXRoIGluc3RlYWQgb2YgcGVyc29uYWwgYWNjZXNzIHRva2VucyBvciBhIHVzZXIgbmFtZSBhbmQgcGFzc3dvcmQgdG8gZ3JhbnQgYXV0aG9yaXphdGlvbiBmb3IgYWNjZXNzaW5nIEdpdEh1YiBvciBCaXRidWNrZXQgcmVwb3NpdG9yaWVzLic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogQ2hlY2sgRE1TIFJlc291cmNlc1xuICAgKiBAcGFyYW0gbm9kZSB0aGUgSUNvbnN0cnVjdCB0byBldmFsdWF0ZVxuICAgKiBAcGFyYW0gaWdub3JlcyBsaXN0IG9mIGlnbm9yZXMgZm9yIHRoZSByZXNvdXJjZVxuICAgKi9cbiAgcHJpdmF0ZSBjaGVja0RNUyhub2RlOiBDZm5SZXNvdXJjZSwgaWdub3JlczogYW55KSB7XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShpZ25vcmVzLCAnSElQQUEuU2VjdXJpdHktRE1TUmVwbGljYXRpb25Ob3RQdWJsaWMnKSAmJlxuICAgICAgIWhpcGFhU2VjdXJpdHlETVNSZXBsaWNhdGlvbk5vdFB1YmxpYyhub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0hJUEFBLlNlY3VyaXR5LURNU1JlcGxpY2F0aW9uTm90UHVibGljJztcbiAgICAgIGNvbnN0IGluZm8gPVxuICAgICAgICAnVGhlIERNUyByZXBsaWNhdGlvbiBpbnN0YW5jZSBpcyBwdWJsaWMgLSAoQ29udHJvbCBJRHM6IDE2NC4zMDgoYSkoMykoaSksIDE2NC4zMDgoYSkoNCkoaWkpKEEpLCAxNjQuMzA4KGEpKDQpKGlpKShDKSwgMTY0LjMxMihhKSgxKSwgMTY0LjMxMihlKSgxKSkuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ0RNUyByZXBsaWNhdGlvbiBpbnN0YW5jZXMgY2FuIGNvbnRhaW4gc2Vuc2l0aXZlIGluZm9ybWF0aW9uIGFuZCBhY2Nlc3MgY29udHJvbCBpcyByZXF1aXJlZCBmb3Igc3VjaCBhY2NvdW50cy4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIENoZWNrIER5bmFtb0RCIFJlc291cmNlc1xuICAgKiBAcGFyYW0gbm9kZSB0aGUgSUNvbnN0cnVjdCB0byBldmFsdWF0ZVxuICAgKiBAcGFyYW0gaWdub3JlcyBsaXN0IG9mIGlnbm9yZXMgZm9yIHRoZSByZXNvdXJjZVxuICAgKi9cbiAgcHJpdmF0ZSBjaGVja0R5bmFtb0RCKG5vZGU6IENmblJlc291cmNlLCBpZ25vcmVzOiBhbnkpIHtcbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdISVBBQS5TZWN1cml0eS1EeW5hbW9EQlBJVFJFbmFibGVkJykgJiZcbiAgICAgICFoaXBhYVNlY3VyaXR5RHluYW1vREJQSVRSRW5hYmxlZChub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0hJUEFBLlNlY3VyaXR5LUR5bmFtb0RCUElUUkVuYWJsZWQnO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgRHluYW1vREIgdGFibGUgZG9lcyBub3QgaGF2ZSBQb2ludC1pbi10aW1lIFJlY292ZXJ5IGVuYWJsZWQgLSAoQ29udHJvbCBJRHM6IDE2NC4zMDgoYSkoNykoaSksIDE2NC4zMDgoYSkoNykoaWkpKEEpLCAxNjQuMzA4KGEpKDcpKGlpKShCKSkuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ1RoZSByZWNvdmVyeSBtYWludGFpbnMgY29udGludW91cyBiYWNrdXBzIG9mIHlvdXIgdGFibGUgZm9yIHRoZSBsYXN0IDM1IGRheXMuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBDaGVjayBFQzIgUmVzb3VyY2VzXG4gICAqIEBwYXJhbSBub2RlIHRoZSBJQ29uc3RydWN0IHRvIGV2YWx1YXRlXG4gICAqIEBwYXJhbSBpZ25vcmVzIGxpc3Qgb2YgaWdub3JlcyBmb3IgdGhlIHJlc291cmNlXG4gICAqL1xuICBwcml2YXRlIGNoZWNrRUMyKG5vZGU6IENmblJlc291cmNlLCBpZ25vcmVzOiBhbnkpOiB2b2lkIHtcbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKFxuICAgICAgICBpZ25vcmVzLFxuICAgICAgICAnSElQQUEuU2VjdXJpdHktRUMySW5zdGFuY2VEZXRhaWxlZE1vbml0b3JpbmdFbmFibGVkJyxcbiAgICAgICkgJiZcbiAgICAgICFoaXBhYVNlY3VyaXR5RUMySW5zdGFuY2VEZXRhaWxlZE1vbml0b3JpbmdFbmFibGVkKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnSElQQUEuU2VjdXJpdHktRUMySW5zdGFuY2VEZXRhaWxlZE1vbml0b3JpbmdFbmFibGVkJztcbiAgICAgIGNvbnN0IGluZm8gPVxuICAgICAgICAnVGhlIEVDMiBpbnN0YW5jZSBkb2VzIG5vdCBoYXZlIGRldGFpbGVkIG1vbml0b3JpbmcgZW5hYmxlZCAtIChDb250cm9sIElEczogMTY0LjMxMihiKSkuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ0RldGFpbGVkIG1vbml0b3JpbmcgcHJvdmlkZXMgYWRkaXRpb25hbCBtb25pdG9yaW5nIGluZm9ybWF0aW9uIChzdWNoIGFzIDEtbWludXRlIHBlcmlvZCBncmFwaHMpIG9uIHRoZSBBV1MgY29uc29sZS4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoaWdub3JlcywgJ0hJUEFBLlNlY3VyaXR5LUVDMkluc3RhbmNlc0luVlBDJykgJiZcbiAgICAgICFoaXBhYVNlY3VyaXR5RUMySW5zdGFuY2VzSW5WUEMobm9kZSlcbiAgICApIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdISVBBQS5TZWN1cml0eS1FQzJJbnN0YW5jZXNJblZQQyc7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSBFQzIgaW5zdGFuY2UgaXMgbm90IHdpdGhpbiBhIFZQQyAtIChDb250cm9sIElEczogMTY0LjMwOChhKSgzKShpKSwgMTY0LjMwOChhKSg0KShpaSkoQSksIDE2NC4zMDgoYSkoNCkoaWkpKEMpLCAxNjQuMzEyKGEpKDEpLCAxNjQuMzEyKGUpKDEpKS4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnQmVjYXVzZSBvZiB0aGVpciBsb2dpY2FsIGlzb2xhdGlvbiwgZG9tYWlucyB0aGF0IHJlc2lkZSB3aXRoaW4gYW4gQW1hem9uIFZQQyBoYXZlIGFuIGV4dHJhIGxheWVyIG9mIHNlY3VyaXR5IHdoZW4gY29tcGFyZWQgdG8gZG9tYWlucyB0aGF0IHVzZSBwdWJsaWMgZW5kcG9pbnRzLic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShpZ25vcmVzLCAnSElQQUEuU2VjdXJpdHktRUMyQ2hlY2tOb1B1YmxpY0lQcycpICYmXG4gICAgICAhaGlwYWFTZWN1cml0eUVDMkluc3RhbmNlTm9QdWJsaWNJcChub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0hJUEFBLlNlY3VyaXR5LUVDMkNoZWNrTm9QdWJsaWNJUHMnO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgRUMyIGluc3RhbmNlIGlzIGFzc29jaWF0ZWQgd2l0aCBhIHB1YmxpYyBJUCBhZGRyZXNzIC0gKENvbnRyb2wgSURzOiAxNjQuMzA4KGEpKDMpKGkpLCAxNjQuMzA4KGEpKDQpKGlpKShBKSwgMTY0LjMwOChhKSg0KShpaSkoQyksIDE2NC4zMTIoYSkoMSksIDE2NC4zMTIoZSkoMSkpLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdBbWF6b24gRUMyIGluc3RhbmNlcyBjYW4gY29udGFpbiBzZW5zaXRpdmUgaW5mb3JtYXRpb24gYW5kIGFjY2VzcyBjb250cm9sIGlzIHJlcXVpcmVkIGZvciBzdWNoIHJlc291cmNlcy4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIENoZWNrIEVDUyBSZXNvdXJjZXNcbiAgICogQHBhcmFtIG5vZGUgdGhlIElDb25zdHJ1Y3QgdG8gZXZhbHVhdGVcbiAgICogQHBhcmFtIGlnbm9yZXMgbGlzdCBvZiBpZ25vcmVzIGZvciB0aGUgcmVzb3VyY2VcbiAgICovXG4gIHByaXZhdGUgY2hlY2tFQ1Mobm9kZTogQ2ZuUmVzb3VyY2UsIGlnbm9yZXM6IGFueSk6IHZvaWQge1xuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoXG4gICAgICAgIGlnbm9yZXMsXG4gICAgICAgICdISVBBQS5TZWN1cml0eS1FQ1NUYXNrRGVmaW5pdGlvblVzZXJGb3JIb3N0TW9kZScsXG4gICAgICApICYmXG4gICAgICAhaGlwYWFTZWN1cml0eUVDU1Rhc2tEZWZpbml0aW9uVXNlckZvckhvc3RNb2RlKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnSElQQUEuU2VjdXJpdHktRUNTVGFza0RlZmluaXRpb25Vc2VyRm9ySG9zdE1vZGUnO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgIFwiVGhlIEVDUyB0YXNrIGRlZmluaXRpb24gaXMgY29uZmlndXJlZCBmb3IgaG9zdCBuZXR3b3JraW5nIGFuZCBoYXMgYXQgbGVhc3Qgb25lIGNvbnRhaW5lciB3aXRoIGRlZmluaXRpb25zIHdpdGggJ3ByaXZpbGVnZWQnIHNldCB0byBmYWxzZSBvciBlbXB0eSBvciAndXNlcicgc2V0IHRvIHJvb3Qgb3IgZW1wdHkgLSAoQ29udHJvbCBJRHM6IDE2NC4zMDgoYSkoMykoaSksIDE2NC4zMDgoYSkoMykoaWkpKEEpLCAxNjQuMzA4KGEpKDQpKGlpKShBKSwgMTY0LjMwOChhKSg0KShpaSkoQyksIDE2NC4zMTIoYSkoMSkpLlwiO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnSWYgYSB0YXNrIGRlZmluaXRpb24gaGFzIGVsZXZhdGVkIHByaXZpbGVnZXMgaXQgaXMgYmVjYXVzZSB5b3UgaGF2ZSBzcGVjaWZpY2FsbHkgb3B0ZWQtaW4gdG8gdGhvc2UgY29uZmlndXJhdGlvbnMuIFRoaXMgcnVsZSBjaGVja3MgZm9yIHVuZXhwZWN0ZWQgcHJpdmlsZWdlIGVzY2FsYXRpb24gd2hlbiBhIHRhc2sgZGVmaW5pdGlvbiBoYXMgaG9zdCBuZXR3b3JraW5nIGVuYWJsZWQgYnV0IHRoZSBjdXN0b21lciBoYXMgbm90IG9wdGVkLWluIHRvIGVsZXZhdGVkIHByaXZpbGVnZXMuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBDaGVjayBFRlMgUmVzb3VyY2VzXG4gICAqIEBwYXJhbSBub2RlIHRoZSBJQ29uc3RydWN0IHRvIGV2YWx1YXRlXG4gICAqIEBwYXJhbSBpZ25vcmVzIGxpc3Qgb2YgaWdub3JlcyBmb3IgdGhlIHJlc291cmNlXG4gICAqL1xuICBwcml2YXRlIGNoZWNrRUZTKG5vZGU6IENmblJlc291cmNlLCBpZ25vcmVzOiBhbnkpIHtcbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdISVBBQS5TZWN1cml0eS1FRlNFbmNyeXB0ZWQnKSAmJlxuICAgICAgIWhpcGFhU2VjdXJpdHlFRlNFbmNyeXB0ZWQobm9kZSlcbiAgICApIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdISVBBQS5TZWN1cml0eS1FRlNFbmNyeXB0ZWQnO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgRUZTIGRvZXMgbm90IGhhdmUgZW5jcnlwdGlvbiBhdCByZXN0IGVuYWJsZWQgLSAoQ29udHJvbCBJRHM6IDE2NC4zMTIoYSkoMikoaXYpLCAxNjQuMzEyKGUpKDIpKGlpKSkuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ0JlY2F1c2Ugc2Vuc2l0aXZlIGRhdGEgY2FuIGV4aXN0IGFuZCB0byBoZWxwIHByb3RlY3QgZGF0YSBhdCByZXN0LCBlbnN1cmUgZW5jcnlwdGlvbiBpcyBlbmFibGVkIGZvciB5b3VyIEFtYXpvbiBFbGFzdGljIEZpbGUgU3lzdGVtIChFRlMpLic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogQ2hlY2sgRWxhc3RpQ2FjaGUgUmVzb3VyY2VzXG4gICAqIEBwYXJhbSBub2RlIHRoZSBJQ29uc3RydWN0IHRvIGV2YWx1YXRlXG4gICAqIEBwYXJhbSBpZ25vcmVzIGxpc3Qgb2YgaWdub3JlcyBmb3IgdGhlIHJlc291cmNlXG4gICAqL1xuICBwcml2YXRlIGNoZWNrRWxhc3RpQ2FjaGUobm9kZTogQ2ZuUmVzb3VyY2UsIGlnbm9yZXM6IGFueSkge1xuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoXG4gICAgICAgIGlnbm9yZXMsXG4gICAgICAgICdISVBBQS5TZWN1cml0eS1FbGFzdGljYWNoZVJlZGlzQ2x1c3RlckF1dG9tYXRpY0JhY2t1cCcsXG4gICAgICApICYmXG4gICAgICAhaGlwYWFTZWN1cml0eUVsYXN0aUNhY2hlUmVkaXNDbHVzdGVyQXV0b21hdGljQmFja3VwKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnSElQQUEuU2VjdXJpdHktRWxhc3RpY2FjaGVSZWRpc0NsdXN0ZXJBdXRvbWF0aWNCYWNrdXAnO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgRWxhc3RpQ2FjaGUgUmVkaXMgY2x1c3RlciBkb2VzIG5vdCByZXRhaW4gYXV0b21hdGljIGJhY2t1cHMgZm9yIGF0IGxlYXN0IDE1IGRheXMgLSAoQ29udHJvbCBJRHM6IDE2NC4zMDgoYSkoNykoaSksIDE2NC4zMDgoYSkoNykoaWkpKEEpLCAxNjQuMzA4KGEpKDcpKGlpKShCKSkuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ0F1dG9tYXRpYyBiYWNrdXBzIGNhbiBoZWxwIGd1YXJkIGFnYWluc3QgZGF0YSBsb3NzLiBJZiBhIGZhaWx1cmUgb2NjdXJzLCB5b3UgY2FuIGNyZWF0ZSBhIG5ldyBjbHVzdGVyLCB3aGljaCByZXN0b3JlcyB5b3VyIGRhdGEgZnJvbSB0aGUgbW9zdCByZWNlbnQgYmFja3VwLic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogQ2hlY2sgRWxhc3RpYyBCZWFuc3RhbGsgUmVzb3VyY2VzXG4gICAqIEBwYXJhbSBub2RlIHRoZSBJQ29uc3RydWN0IHRvIGV2YWx1YXRlXG4gICAqIEBwYXJhbSBpZ25vcmVzIGxpc3Qgb2YgaWdub3JlcyBmb3IgdGhlIHJlc291cmNlXG4gICAqL1xuICBwcml2YXRlIGNoZWNrRWxhc3RpY0JlYW5zdGFsayhub2RlOiBDZm5SZXNvdXJjZSwgaWdub3JlczogYW55KTogdm9pZCB7XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShcbiAgICAgICAgaWdub3JlcyxcbiAgICAgICAgJ0hJUEFBLlNlY3VyaXR5LUVsYXN0aWNCZWFuc3RhbGtFbmhhbmNlZEhlYWx0aFJlcG9ydGluZ0VuYWJsZWQnLFxuICAgICAgKSAmJlxuICAgICAgIWhpcGFhU2VjdXJpdHlFbGFzdGljQmVhbnN0YWxrRW5oYW5jZWRIZWFsdGhSZXBvcnRpbmdFbmFibGVkKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPVxuICAgICAgICAnSElQQUEuU2VjdXJpdHktRWxhc3RpY0JlYW5zdGFsa0VuaGFuY2VkSGVhbHRoUmVwb3J0aW5nRW5hYmxlZCc7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSBFbGFzdGljIEJlYW5zdGFsayBlbnZpcm9ubWVudCBkb2VzIG5vdCBoYXZlIGVuaGFuY2VkIGhlYWx0aCByZXBvcnRpbmcgZW5hYmxlZCAtIChDb250cm9sIElEOiAxNjQuMzEyKGIpKS4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnQVdTIEVsYXN0aWMgQmVhbnN0YWxrIGVuaGFuY2VkIGhlYWx0aCByZXBvcnRpbmcgZW5hYmxlcyBhIG1vcmUgcmFwaWQgcmVzcG9uc2UgdG8gY2hhbmdlcyBpbiB0aGUgaGVhbHRoIG9mIHRoZSB1bmRlcmx5aW5nIGluZnJhc3RydWN0dXJlLiBUaGVzZSBjaGFuZ2VzIGNvdWxkIHJlc3VsdCBpbiBhIGxhY2sgb2YgYXZhaWxhYmlsaXR5IG9mIHRoZSBhcHBsaWNhdGlvbi4gRWxhc3RpYyBCZWFuc3RhbGsgZW5oYW5jZWQgaGVhbHRoIHJlcG9ydGluZyBwcm92aWRlcyBhIHN0YXR1cyBkZXNjcmlwdG9yIHRvIGdhdWdlIHRoZSBzZXZlcml0eSBvZiB0aGUgaWRlbnRpZmllZCBpc3N1ZXMgYW5kIGlkZW50aWZ5IHBvc3NpYmxlIGNhdXNlcyB0byBpbnZlc3RpZ2F0ZS4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoXG4gICAgICAgIGlnbm9yZXMsXG4gICAgICAgICdISVBBQS5TZWN1cml0eS1FbGFzdGljQmVhbnN0YWxrTWFuYWdlZFVwZGF0ZXNFbmFibGVkJyxcbiAgICAgICkgJiZcbiAgICAgICFoaXBhYVNlY3VyaXR5RWxhc3RpY0JlYW5zdGFsa01hbmFnZWRVcGRhdGVzRW5hYmxlZChub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0hJUEFBLlNlY3VyaXR5LUVsYXN0aWNCZWFuc3RhbGtNYW5hZ2VkVXBkYXRlc0VuYWJsZWQnO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgRWxhc3RpYyBCZWFuc3RhbGsgZW52aXJvbm1lbnQgZG9lcyBub3QgaGF2ZSBtYW5hZ2VkIHVwZGF0ZXMgZW5hYmxlZCAtIChDb250cm9sIElEOiAxNjQuMzA4KGEpKDUpKGlpKShBKSkuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ0VuYWJsaW5nIG1hbmFnZWQgcGxhdGZvcm0gdXBkYXRlcyBmb3IgYW4gQW1hem9uIEVsYXN0aWMgQmVhbnN0YWxrIGVudmlyb25tZW50IGVuc3VyZXMgdGhhdCB0aGUgbGF0ZXN0IGF2YWlsYWJsZSBwbGF0Zm9ybSBmaXhlcywgdXBkYXRlcywgYW5kIGZlYXR1cmVzIGZvciB0aGUgZW52aXJvbm1lbnQgYXJlIGluc3RhbGxlZC4gS2VlcGluZyB1cCB0byBkYXRlIHdpdGggcGF0Y2ggaW5zdGFsbGF0aW9uIGlzIGEgYmVzdCBwcmFjdGljZSBpbiBzZWN1cmluZyBzeXN0ZW1zLic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogQ2hlY2sgRWxhc3RpYyBMb2FkIEJhbGFuY2VyIFJlc291cmNlc1xuICAgKiBAcGFyYW0gbm9kZSB0aGUgSUNvbnN0cnVjdCB0byBldmFsdWF0ZVxuICAgKiBAcGFyYW0gaWdub3JlcyBsaXN0IG9mIGlnbm9yZXMgZm9yIHRoZSByZXNvdXJjZVxuICAgKi9cbiAgcHJpdmF0ZSBjaGVja0VMQihub2RlOiBDZm5SZXNvdXJjZSwgaWdub3JlczogYW55KTogdm9pZCB7XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShcbiAgICAgICAgaWdub3JlcyxcbiAgICAgICAgJ0hJUEFBLlNlY3VyaXR5LUFMQkh0dHBEcm9wSW52YWxpZEhlYWRlckVuYWJsZWQnLFxuICAgICAgKSAmJlxuICAgICAgIWhpcGFhU2VjdXJpdHlBTEJIdHRwRHJvcEludmFsaWRIZWFkZXJFbmFibGVkKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnSElQQUEuU2VjdXJpdHktQUxCSHR0cERyb3BJbnZhbGlkSGVhZGVyRW5hYmxlZCc7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSBBTEIgZG9lcyBub3QgaGF2ZSBpbnZhbGlkIEhUVFAgaGVhZGVyIGRyb3BwaW5nIGVuYWJsZWQgLSAoQ29udHJvbCBJRHM6IDE2NC4zMTIoYSkoMikoaXYpLCAxNjQuMzEyKGUpKDEpLCAxNjQuMzEyKGUpKDIpKGkpLCAxNjQuMzEyKGUpKDIpKGlpKSkuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ0Vuc3VyZSB0aGF0IHlvdXIgQXBwbGljYXRpb24gTG9hZCBCYWxhbmNlcnMgKEFMQikgYXJlIGNvbmZpZ3VyZWQgdG8gZHJvcCBodHRwIGhlYWRlcnMuIEJlY2F1c2Ugc2Vuc2l0aXZlIGRhdGEgY2FuIGV4aXN0LCBlbmFibGUgZW5jcnlwdGlvbiBpbiB0cmFuc2l0IHRvIGhlbHAgcHJvdGVjdCB0aGF0IGRhdGEuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdISVBBQS5TZWN1cml0eS1BTEJIdHRwVG9IdHRwc1JlZGlyZWN0aW9uJykgJiZcbiAgICAgICFoaXBhYVNlY3VyaXR5QUxCSHR0cFRvSHR0cHNSZWRpcmVjdGlvbihub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0hJUEFBLlNlY3VyaXR5LUFMQkh0dHBUb0h0dHBzUmVkaXJlY3Rpb24nO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgIFwiVGhlIEFMQidzIEhUVFAgbGlzdGVuZXJzIGFyZSBub3QgY29uZmlndXJlZCB0byByZWRpcmVjdCB0byBIVFRQUyAtIChDb250cm9sIElEczogMTY0LjMxMihhKSgyKShpdiksIDE2NC4zMTIoZSkoMSksIDE2NC4zMTIoZSkoMikoaSksIDE2NC4zMTIoZSkoMikoaWkpKS5cIjtcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ1RvIGhlbHAgcHJvdGVjdCBkYXRhIGluIHRyYW5zaXQsIGVuc3VyZSB0aGF0IHlvdXIgQXBwbGljYXRpb24gTG9hZCBCYWxhbmNlciBhdXRvbWF0aWNhbGx5IHJlZGlyZWN0cyB1bmVuY3J5cHRlZCBIVFRQIHJlcXVlc3RzIHRvIEhUVFBTLiBCZWNhdXNlIHNlbnNpdGl2ZSBkYXRhIGNhbiBleGlzdCwgZW5hYmxlIGVuY3J5cHRpb24gaW4gdHJhbnNpdCB0byBoZWxwIHByb3RlY3QgdGhhdCBkYXRhLic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShpZ25vcmVzLCAnSElQQUEuU2VjdXJpdHktRUxCQUNNQ2VydGlmaWNhdGVSZXF1aXJlZCcpICYmXG4gICAgICAhaGlwYWFTZWN1cml0eUVMQkFDTUNlcnRpZmljYXRlUmVxdWlyZWQobm9kZSlcbiAgICApIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdISVBBQS5TZWN1cml0eS1FTEJBQ01DZXJ0aWZpY2F0ZVJlcXVpcmVkJztcbiAgICAgIGNvbnN0IGluZm8gPVxuICAgICAgICAnVGhlIENMQiBkb2VzIG5vdCB1dGlsaXplIGFuIFNTTCBjZXJ0aWZpY2F0ZSBwcm92aWRlZCBieSBBQ00gKEFtYXpvbiBDZXJ0aWZpY2F0ZSBNYW5hZ2VyKSAtIChDb250cm9sIElEczogMTY0LjMxMihhKSgyKShpdiksIDE2NC4zMTIoZSkoMSksIDE2NC4zMTIoZSkoMikoaSksIDE2NC4zMTIoZSkoMikoaWkpKS4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnQmVjYXVzZSBzZW5zaXRpdmUgZGF0YSBjYW4gZXhpc3QgYW5kIHRvIGhlbHAgcHJvdGVjdCBkYXRhIGF0IHRyYW5zaXQsIGVuc3VyZSBlbmNyeXB0aW9uIGlzIGVuYWJsZWQgZm9yIHlvdXIgRWxhc3RpYyBMb2FkIEJhbGFuY2luZy4gVXNlIEFXUyBDZXJ0aWZpY2F0ZSBNYW5hZ2VyIHRvIG1hbmFnZSwgcHJvdmlzaW9uIGFuZCBkZXBsb3kgcHVibGljIGFuZCBwcml2YXRlIFNTTC9UTFMgY2VydGlmaWNhdGVzIHdpdGggQVdTIHNlcnZpY2VzIGFuZCBpbnRlcm5hbCByZXNvdXJjZXMuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKFxuICAgICAgICBpZ25vcmVzLFxuICAgICAgICAnSElQQUEuU2VjdXJpdHktRUxCQ3Jvc3Nab25lQmFsYW5jaW5nRW5hYmxlZCcsXG4gICAgICApICYmXG4gICAgICAhaGlwYWFTZWN1cml0eUVMQkNyb3NzWm9uZUJhbGFuY2luZ0VuYWJsZWQobm9kZSlcbiAgICApIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdISVBBQS5TZWN1cml0eS1FTEJDcm9zc1pvbmVCYWxhbmNpbmdFbmFibGVkJztcbiAgICAgIGNvbnN0IGluZm8gPVxuICAgICAgICAnVGhlIENMQiBkb2VzIG5vdCBiYWxhbmNlIHRyYWZmaWMgYmV0d2VlbiBhdCBsZWFzdCAyIEF2YWlsYWJpbGl0eSBab25lcyAtIChDb250cm9sIElEczogMTY0LjMwOChhKSg3KShpKSwgMTY0LjMwOChhKSg3KShpaSkoQykpLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdUaGUgY3Jvc3Mtem9uZSBsb2FkIGJhbGFuY2luZyByZWR1Y2VzIHRoZSBuZWVkIHRvIG1haW50YWluIGVxdWl2YWxlbnQgbnVtYmVycyBvZiBpbnN0YW5jZXMgaW4gZWFjaCBlbmFibGVkIGF2YWlsYWJpbGl0eSB6b25lLic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShcbiAgICAgICAgaWdub3JlcyxcbiAgICAgICAgJ0hJUEFBLlNlY3VyaXR5LUVMQkRlbGV0aW9uUHJvdGVjdGlvbkVuYWJsZWQnLFxuICAgICAgKSAmJlxuICAgICAgIWhpcGFhU2VjdXJpdHlFTEJEZWxldGlvblByb3RlY3Rpb25FbmFibGVkKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnSElQQUEuU2VjdXJpdHktRUxCRGVsZXRpb25Qcm90ZWN0aW9uRW5hYmxlZCc7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSBBTEIsIE5MQiwgb3IgR0xCIGRvZXMgbm90IGhhdmUgZGVsZXRpb24gcHJvdGVjdGlvbiBlbmFibGVkIC0gKENvbnRyb2wgSURzOiAxNjQuMzA4KGEpKDcpKGkpLCAxNjQuMzA4KGEpKDcpKGlpKShDKSkuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ1VzZSB0aGlzIGZlYXR1cmUgdG8gcHJldmVudCB5b3VyIGxvYWQgYmFsYW5jZXIgZnJvbSBiZWluZyBhY2NpZGVudGFsbHkgb3IgbWFsaWNpb3VzbHkgZGVsZXRlZCwgd2hpY2ggY2FuIGxlYWQgdG8gbG9zcyBvZiBhdmFpbGFiaWxpdHkgZm9yIHlvdXIgYXBwbGljYXRpb25zLic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShpZ25vcmVzLCAnSElQQUEuU2VjdXJpdHktRUxCTG9nZ2luZ0VuYWJsZWQnKSAmJlxuICAgICAgIWhpcGFhU2VjdXJpdHlFTEJMb2dnaW5nRW5hYmxlZChub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0hJUEFBLlNlY3VyaXR5LUVMQkxvZ2dpbmdFbmFibGVkJztcbiAgICAgIGNvbnN0IGluZm8gPVxuICAgICAgICAnVGhlIEVMQiBkb2VzIG5vdCBoYXZlIGxvZ2dpbmcgZW5hYmxlZCAtIChDb250cm9sIElEOiAxNjQuMzEyKGIpKS4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICBcIkVsYXN0aWMgTG9hZCBCYWxhbmNpbmcgYWN0aXZpdHkgaXMgYSBjZW50cmFsIHBvaW50IG9mIGNvbW11bmljYXRpb24gd2l0aGluIGFuIGVudmlyb25tZW50LiBFbnN1cmUgRUxCIGxvZ2dpbmcgaXMgZW5hYmxlZC4gVGhlIGNvbGxlY3RlZCBkYXRhIHByb3ZpZGVzIGRldGFpbGVkIGluZm9ybWF0aW9uIGFib3V0IHJlcXVlc3RzIHNlbnQgdG8gVGhlIEVMQi4gRWFjaCBsb2cgY29udGFpbnMgaW5mb3JtYXRpb24gc3VjaCBhcyB0aGUgdGltZSB0aGUgcmVxdWVzdCB3YXMgcmVjZWl2ZWQsIHRoZSBjbGllbnQncyBJUCBhZGRyZXNzLCBsYXRlbmNpZXMsIHJlcXVlc3QgcGF0aHMsIGFuZCBzZXJ2ZXIgcmVzcG9uc2VzLlwiO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoaWdub3JlcywgJ0hJUEFBLlNlY3VyaXR5LUVMQlRsc0h0dHBzTGlzdGVuZXJzT25seScpICYmXG4gICAgICAhaGlwYWFTZWN1cml0eUVMQlRsc0h0dHBzTGlzdGVuZXJzT25seShub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0hJUEFBLlNlY3VyaXR5LUVMQlRsc0h0dHBzTGlzdGVuZXJzT25seSc7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSBDTEIgZG9lcyBub3QgcmVzdHJpY3QgaXRzIGxpc3RlbmVycyB0byBvbmx5IHRoZSBTU0wgYW5kIEhUVFBTIHByb3RvY29scyAtIChDb250cm9sIElEczogMTY0LjMxMihhKSgyKShpdiksIDE2NC4zMTIoZSkoMSksIDE2NC4zMTIoZSkoMikoaSksIDE2NC4zMTIoZSkoMikoaWkpKS4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnQmVjYXVzZSBzZW5zaXRpdmUgZGF0YSBjYW4gZXhpc3QsIGVuYWJsZSBlbmNyeXB0aW9uIGluIHRyYW5zaXQgdG8gaGVscCBwcm90ZWN0IHRoYXQgZGF0YS4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoaWdub3JlcywgJ0hJUEFBLlNlY3VyaXR5LUVMQnYyQUNNQ2VydGlmaWNhdGVSZXF1aXJlZCcpICYmXG4gICAgICAhaGlwYWFTZWN1cml0eUVMQnYyQUNNQ2VydGlmaWNhdGVSZXF1aXJlZChub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0hJUEFBLlNlY3VyaXR5LUVMQnYyQUNNQ2VydGlmaWNhdGVSZXF1aXJlZCc7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSBBTEIsIE5MQiwgb3IgR0xCIGxpc3RlbmVyIGRvZXMgbm90IHV0aWxpemUgYW4gU1NMIGNlcnRpZmljYXRlIHByb3ZpZGVkIGJ5IEFDTSAoQW1hem9uIENlcnRpZmljYXRlIE1hbmFnZXIpIC0gKENvbnRyb2wgSURzOiAxNjQuMzEyKGEpKDIpKGl2KSwgMTY0LjMxMihlKSgyKShpaSkpLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdCZWNhdXNlIHNlbnNpdGl2ZSBkYXRhIGNhbiBleGlzdCBhbmQgdG8gaGVscCBwcm90ZWN0IGRhdGEgYXQgdHJhbnNpdCwgZW5zdXJlIGVuY3J5cHRpb24gaXMgZW5hYmxlZCBmb3IgeW91ciBFbGFzdGljIExvYWQgQmFsYW5jaW5nLiBVc2UgQVdTIENlcnRpZmljYXRlIE1hbmFnZXIgdG8gbWFuYWdlLCBwcm92aXNpb24gYW5kIGRlcGxveSBwdWJsaWMgYW5kIHByaXZhdGUgU1NML1RMUyBjZXJ0aWZpY2F0ZXMgd2l0aCBBV1Mgc2VydmljZXMgYW5kIGludGVybmFsIHJlc291cmNlcy4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIENoZWNrIEVNUiBSZXNvdXJjZXNcbiAgICogQHBhcmFtIG5vZGUgdGhlIElDb25zdHJ1Y3QgdG8gZXZhbHVhdGVcbiAgICogQHBhcmFtIGlnbm9yZXMgbGlzdCBvZiBpZ25vcmVzIGZvciB0aGUgcmVzb3VyY2VcbiAgICovXG4gIHByaXZhdGUgY2hlY2tFTVIobm9kZTogQ2ZuUmVzb3VyY2UsIGlnbm9yZXM6IGFueSkge1xuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoaWdub3JlcywgJ0hJUEFBLlNlY3VyaXR5LUVNUktlcmJlcm9zRW5hYmxlZCcpICYmXG4gICAgICAhaGlwYWFTZWN1cml0eUVNUktlcmJlcm9zRW5hYmxlZChub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0hJUEFBLlNlY3VyaXR5LUVNUktlcmJlcm9zRW5hYmxlZCc7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSBFTVIgY2x1c3RlciBkb2VzIG5vdCBoYXZlIEtlcmJlcm9zIGVuYWJsZWQgLSAoQ29udHJvbCBJRHM6IDE2NC4zMDgoYSkoMykoaSksIDE2NC4zMDgoYSkoMykoaWkpKEEpLCAxNjQuMzA4KGEpKDMpKGlpKShCKSwgMTY0LjMwOChhKSg0KShpKSwgMTY0LjMwOChhKSg0KShpaSkoQSksIDE2NC4zMDgoYSkoNCkoaWkpKEIpLCAxNjQuMzA4KGEpKDQpKGlpKShDKSwgMTY0LjMxMihhKSgxKSkuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ1RoZSBhY2Nlc3MgcGVybWlzc2lvbnMgYW5kIGF1dGhvcml6YXRpb25zIGNhbiBiZSBtYW5hZ2VkIGFuZCBpbmNvcnBvcmF0ZWQgd2l0aCB0aGUgcHJpbmNpcGxlcyBvZiBsZWFzdCBwcml2aWxlZ2UgYW5kIHNlcGFyYXRpb24gb2YgZHV0aWVzLCBieSBlbmFibGluZyBLZXJiZXJvcyBmb3IgQW1hem9uIEVNUiBjbHVzdGVycy4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIENoZWNrIElBTSBSZXNvdXJjZXNcbiAgICogQHBhcmFtIG5vZGUgdGhlIElDb25zdHJ1Y3QgdG8gZXZhbHVhdGVcbiAgICogQHBhcmFtIGlnbm9yZXMgbGlzdCBvZiBpZ25vcmVzIGZvciB0aGUgcmVzb3VyY2VcbiAgICovXG4gIHByaXZhdGUgY2hlY2tJQU0obm9kZTogQ2ZuUmVzb3VyY2UsIGlnbm9yZXM6IGFueSk6IHZvaWQge1xuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoaWdub3JlcywgJ0hJUEFBLlNlY3VyaXR5LUlBTU5vSW5saW5lUG9saWN5JykgJiZcbiAgICAgICFoaXBhYVNlY3VyaXR5SUFNTm9JbmxpbmVQb2xpY3kobm9kZSlcbiAgICApIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdISVBBQS5TZWN1cml0eS1JQU1Ob0lubGluZVBvbGljeSc7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSBJQU0gR3JvdXAsIFVzZXIsIG9yIFJvbGUgY29udGFpbnMgYW4gaW5saW5lIHBvbGljeSAtIChDb250cm9sIElEczogMTY0LjMwOChhKSgzKShpKSwgMTY0LjMwOChhKSgzKShpaSkoQSksIDE2NC4zMDgoYSkoMykoaWkpKEIpLCAxNjQuMzA4KGEpKDQpKGkpLCAxNjQuMzA4KGEpKDQpKGlpKShBKSwgMTY0LjMwOChhKSg0KShpaSkoQiksIDE2NC4zMDgoYSkoNCkoaWkpKEMpLCAxNjQuMzEyKGEpKDEpKS4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnQVdTIHJlY29tbWVuZHMgdG8gdXNlIG1hbmFnZWQgcG9saWNpZXMgaW5zdGVhZCBvZiBpbmxpbmUgcG9saWNpZXMuIFRoZSBtYW5hZ2VkIHBvbGljaWVzIGFsbG93IHJldXNhYmlsaXR5LCB2ZXJzaW9uaW5nIGFuZCByb2xsaW5nIGJhY2ssIGFuZCBkZWxlZ2F0aW5nIHBlcm1pc3Npb25zIG1hbmFnZW1lbnQuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKFxuICAgICAgICBpZ25vcmVzLFxuICAgICAgICAnSElQQUEuU2VjdXJpdHktSUFNUG9saWN5Tm9TdGF0ZW1lbnRzV2l0aEFkbWluQWNjZXNzJyxcbiAgICAgICkgJiZcbiAgICAgICFoaXBhYVNlY3VyaXR5SUFNUG9saWN5Tm9TdGF0ZW1lbnRzV2l0aEFkbWluQWNjZXNzKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnSElQQUEuU2VjdXJpdHktSUFNUG9saWN5Tm9TdGF0ZW1lbnRzV2l0aEFkbWluQWNjZXNzJztcbiAgICAgIGNvbnN0IGluZm8gPVxuICAgICAgICAnVGhlIElBTSBwb2xpY3kgZ3JhbnRzIGFkbWluIGFjY2VzcyAtIChDb250cm9sIElEczogMTY0LjMwOChhKSgzKShpKSwgMTY0LjMwOChhKSgzKShpaSkoQSksIDE2NC4zMDgoYSkoMykoaWkpKEIpLCAxNjQuMzA4KGEpKDQpKGkpLCAxNjQuMzA4KGEpKDQpKGlpKShBKSwgMTY0LjMwOChhKSg0KShpaSkoQiksIDE2NC4zMDgoYSkoNCkoaWkpKEMpLCAxNjQuMzEyKGEpKDEpKS4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnQVdTIElkZW50aXR5IGFuZCBBY2Nlc3MgTWFuYWdlbWVudCAoSUFNKSBjYW4gaGVscCB5b3UgaW5jb3Jwb3JhdGUgdGhlIHByaW5jaXBsZXMgb2YgbGVhc3QgcHJpdmlsZWdlIGFuZCBzZXBhcmF0aW9uIG9mIGR1dGllcyB3aXRoIGFjY2VzcyBwZXJtaXNzaW9ucyBhbmQgYXV0aG9yaXphdGlvbnMsIHJlc3RyaWN0aW5nIHBvbGljaWVzIGZyb20gY29udGFpbmluZyBcIkVmZmVjdFwiOiBcIkFsbG93XCIgd2l0aCBcIkFjdGlvblwiOiBcIipcIiBvdmVyIFwiUmVzb3VyY2VcIjogXCIqXCIuIEFsbG93aW5nIHVzZXJzIHRvIGhhdmUgbW9yZSBwcml2aWxlZ2VzIHRoYW4gbmVlZGVkIHRvIGNvbXBsZXRlIGEgdGFzayBtYXkgdmlvbGF0ZSB0aGUgcHJpbmNpcGxlIG9mIGxlYXN0IHByaXZpbGVnZSBhbmQgc2VwYXJhdGlvbiBvZiBkdXRpZXMuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cblxuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoXG4gICAgICAgIGlnbm9yZXMsXG4gICAgICAgICdISVBBQS5TZWN1cml0eS1JQU1Qb2xpY3lOb1N0YXRlbWVudHNXaXRoRnVsbEFjY2VzcycsXG4gICAgICApICYmXG4gICAgICAhaGlwYWFTZWN1cml0eUlBTVBvbGljeU5vU3RhdGVtZW50c1dpdGhGdWxsQWNjZXNzKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnSElQQUEuU2VjdXJpdHktSUFNUG9saWN5Tm9TdGF0ZW1lbnRzV2l0aEZ1bGxBY2Nlc3MnO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgSUFNIHBvbGljeSBncmFudHMgZnVsbCBhY2Nlc3MgLSAoQ29udHJvbCBJRHM6IDE2NC4zMDgoYSkoMykoaSksIDE2NC4zMDgoYSkoMykoaWkpKEEpLCAxNjQuMzA4KGEpKDMpKGlpKShCKSwgMTY0LjMwOChhKSg0KShpKSwgMTY0LjMwOChhKSg0KShpaSkoQSksIDE2NC4zMDgoYSkoNCkoaWkpKEIpLCAxNjQuMzA4KGEpKDQpKGlpKShDKSwgMTY0LjMxMihhKSgxKSkuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ0Vuc3VyZSBJQU0gQWN0aW9ucyBhcmUgcmVzdHJpY3RlZCB0byBvbmx5IHRob3NlIGFjdGlvbnMgdGhhdCBhcmUgbmVlZGVkLiBBbGxvd2luZyB1c2VycyB0byBoYXZlIG1vcmUgcHJpdmlsZWdlcyB0aGFuIG5lZWRlZCB0byBjb21wbGV0ZSBhIHRhc2sgbWF5IHZpb2xhdGUgdGhlIHByaW5jaXBsZSBvZiBsZWFzdCBwcml2aWxlZ2UgYW5kIHNlcGFyYXRpb24gb2YgZHV0aWVzLic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShpZ25vcmVzLCAnSElQQUEuU2VjdXJpdHktSUFNVXNlckdyb3VwTWVtYmVyc2hpcCcpICYmXG4gICAgICAhaGlwYWFTZWN1cml0eUlBTVVzZXJHcm91cE1lbWJlcnNoaXAobm9kZSlcbiAgICApIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdISVBBQS5TZWN1cml0eS1JQU1Vc2VyR3JvdXBNZW1iZXJzaGlwJztcbiAgICAgIGNvbnN0IGluZm8gPVxuICAgICAgICAnVGhlIElBTSB1c2VyIGRvZXMgbm90IGJlbG9uZyB0byBhbnkgZ3JvdXAocykgLSAoQ29udHJvbCBJRHM6IDE2NC4zMDgoYSkoMykoaSksIDE2NC4zMDgoYSkoMykoaWkpKEEpLCAxNjQuMzA4KGEpKDMpKGlpKShCKSwgMTY0LjMwOChhKSg0KShpKSwgMTY0LjMwOChhKSg0KShpaSkoQSksIDE2NC4zMDgoYSkoNCkoaWkpKEIpLCAxNjQuMzA4KGEpKDQpKGlpKShDKSwgMTY0LjMxMihhKSgxKSkuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ0FXUyBJZGVudGl0eSBhbmQgQWNjZXNzIE1hbmFnZW1lbnQgKElBTSkgY2FuIGhlbHAgeW91IHJlc3RyaWN0IGFjY2VzcyBwZXJtaXNzaW9ucyBhbmQgYXV0aG9yaXphdGlvbnMsIGJ5IGVuc3VyaW5nIElBTSB1c2VycyBhcmUgbWVtYmVycyBvZiBhdCBsZWFzdCBvbmUgZ3JvdXAuIEFsbG93aW5nIHVzZXJzIG1vcmUgcHJpdmlsZWdlcyB0aGFuIG5lZWRlZCB0byBjb21wbGV0ZSBhIHRhc2sgbWF5IHZpb2xhdGUgdGhlIHByaW5jaXBsZSBvZiBsZWFzdCBwcml2aWxlZ2UgYW5kIHNlcGFyYXRpb24gb2YgZHV0aWVzLic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG5cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdISVBBQS5TZWN1cml0eS1JQU1Vc2VyTm9Qb2xpY2llcycpICYmXG4gICAgICAhaGlwYWFTZWN1cml0eUlBTVVzZXJOb1BvbGljaWVzKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnSElQQUEuU2VjdXJpdHktSUFNVXNlck5vUG9saWNpZXMnO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgSUFNIHBvbGljeSBpcyBhdHRhY2hlZCBhdCB0aGUgdXNlciBsZXZlbCAtIChDb250cm9sIElEczogMTY0LjMwOChhKSgzKShpKSwgMTY0LjMwOChhKSgzKShpaSkoQSksIDE2NC4zMDgoYSkoMykoaWkpKEIpLCAxNjQuMzA4KGEpKDQpKGkpLCAxNjQuMzA4KGEpKDQpKGlpKShBKSwgMTY0LjMwOChhKSg0KShpaSkoQiksIDE2NC4zMDgoYSkoNCkoaWkpKEMpLCAxNjQuMzEyKGEpKDEpKS4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnQXNzaWduaW5nIHByaXZpbGVnZXMgYXQgdGhlIGdyb3VwIG9yIHRoZSByb2xlIGxldmVsIGhlbHBzIHRvIHJlZHVjZSBvcHBvcnR1bml0eSBmb3IgYW4gaWRlbnRpdHkgdG8gcmVjZWl2ZSBvciByZXRhaW4gZXhjZXNzaXZlIHByaXZpbGVnZXMuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBDaGVjayBMYW1iZGEgUmVzb3VyY2VzXG4gICAqIEBwYXJhbSBub2RlIHRoZSBJQ29uc3RydWN0IHRvIGV2YWx1YXRlXG4gICAqIEBwYXJhbSBpZ25vcmVzIGxpc3Qgb2YgaWdub3JlcyBmb3IgdGhlIHJlc291cmNlXG4gICAqL1xuICBwcml2YXRlIGNoZWNrTGFtYmRhKG5vZGU6IENmblJlc291cmNlLCBpZ25vcmVzOiBhbnkpIHtcbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdISVBBQS5TZWN1cml0eS1MYW1iZGFDb25jdXJyZW5jeScpICYmXG4gICAgICAhaGlwYWFTZWN1cml0eUxhbWJkYUNvbmN1cnJlbmN5KG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnSElQQUEuU2VjdXJpdHktTGFtYmRhQ29uY3VycmVuY3knO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgTGFtYmRhIGZ1bmN0aW9uIGlzIG5vdCBjb25maWd1cmVkIHdpdGggZnVuY3Rpb24tbGV2ZWwgY29uY3VycmVudCBleGVjdXRpb24gbGltaXRzIC0gKENvbnRyb2wgSUQ6IDE2NC4zMTIoYikpLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgIFwiRW5zdXJlIHRoYXQgYSBMYW1iZGEgZnVuY3Rpb24ncyBjb25jdXJyZW5jeSBoaWdoIGFuZCBsb3cgbGltaXRzIGFyZSBlc3RhYmxpc2hlZC4gVGhpcyBjYW4gYXNzaXN0IGluIGJhc2VsaW5pbmcgdGhlIG51bWJlciBvZiByZXF1ZXN0cyB0aGF0IHlvdXIgZnVuY3Rpb24gaXMgc2VydmluZyBhdCBhbnkgZ2l2ZW4gdGltZS5cIjtcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdISVBBQS5TZWN1cml0eS1MYW1iZGFEbHEnKSAmJlxuICAgICAgIWhpcGFhU2VjdXJpdHlMYW1iZGFEbHEobm9kZSlcbiAgICApIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdISVBBQS5TZWN1cml0eS1MYW1iZGFEbHEnO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgTGFtYmRhIGZ1bmN0aW9uIGlzIG5vdCBjb25maWd1cmVkIHdpdGggYSBkZWFkLWxldHRlciBjb25maWd1cmF0aW9uIC0gKENvbnRyb2wgSUQ6IDE2NC4zMTIoYikpLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdOb3RpZnkgdGhlIGFwcHJvcHJpYXRlIHBlcnNvbm5lbCB0aHJvdWdoIEFtYXpvbiBTaW1wbGUgUXVldWUgU2VydmljZSAoQW1hem9uIFNRUykgb3IgQW1hem9uIFNpbXBsZSBOb3RpZmljYXRpb24gU2VydmljZSAoQW1hem9uIFNOUykgd2hlbiBhIGZ1bmN0aW9uIGhhcyBmYWlsZWQuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdISVBBQS5TZWN1cml0eS1MYW1iZGFJbnNpZGVWUEMnKSAmJlxuICAgICAgIWhpcGFhU2VjdXJpdHlMYW1iZGFJbnNpZGVWUEMobm9kZSlcbiAgICApIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdISVBBQS5TZWN1cml0eS1MYW1iZGFJbnNpZGVWUEMnO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgTGFtYmRhIGZ1bmN0aW9uIGlzIG5vdCBWUEMgZW5hYmxlZCAtIChDb250cm9sIElEczogMTY0LjMwOChhKSgzKShpKSwgMTY0LjMwOChhKSg0KShpaSkoQSksIDE2NC4zMDgoYSkoNCkoaWkpKEMpLCAxNjQuMzEyKGEpKDEpLCAxNjQuMzEyKGUpKDEpKS4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnQmVjYXVzZSBvZiB0aGVpciBsb2dpY2FsIGlzb2xhdGlvbiwgZG9tYWlucyB0aGF0IHJlc2lkZSB3aXRoaW4gYW4gQW1hem9uIFZQQyBoYXZlIGFuIGV4dHJhIGxheWVyIG9mIHNlY3VyaXR5IHdoZW4gY29tcGFyZWQgdG8gZG9tYWlucyB0aGF0IHVzZSBwdWJsaWMgZW5kcG9pbnRzLic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogQ2hlY2sgT3BlblNlYXJjaCBSZXNvdXJjZXNcbiAgICogQHBhcmFtIG5vZGUgdGhlIElDb25zdHJ1Y3QgdG8gZXZhbHVhdGVcbiAgICogQHBhcmFtIGlnbm9yZXMgbGlzdCBvZiBpZ25vcmVzIGZvciB0aGUgcmVzb3VyY2VcbiAgICovXG4gIHByaXZhdGUgY2hlY2tPcGVuU2VhcmNoKG5vZGU6IENmblJlc291cmNlLCBpZ25vcmVzOiBhbnkpIHtcbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdISVBBQS5TZWN1cml0eS1PcGVuU2VhcmNoRW5jcnlwdGVkQXRSZXN0JykgJiZcbiAgICAgICFoaXBhYVNlY3VyaXR5T3BlblNlYXJjaEVuY3J5cHRlZEF0UmVzdChub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0hJUEFBLlNlY3VyaXR5LU9wZW5TZWFyY2hFbmNyeXB0ZWRBdFJlc3QnO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgT3BlblNlYXJjaCBTZXJ2aWNlIGRvbWFpbiBkb2VzIG5vdCBoYXZlIGVuY3J5cHRpb24gYXQgcmVzdCBlbmFibGVkIC0gKENvbnRyb2wgSURzOiAxNjQuMzEyKGEpKDIpKGl2KSwgMTY0LjMxMihlKSgyKShpaSkpLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdCZWNhdXNlIHNlbnNpdGl2ZSBkYXRhIGNhbiBleGlzdCBhbmQgdG8gaGVscCBwcm90ZWN0IGRhdGEgYXQgcmVzdCwgZW5zdXJlIGVuY3J5cHRpb24gaXMgZW5hYmxlZCBmb3IgeW91ciBBbWF6b24gT3BlblNlYXJjaCBTZXJ2aWNlIChPcGVuU2VhcmNoIFNlcnZpY2UpIGRvbWFpbnMuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdISVBBQS5TZWN1cml0eS1PcGVuU2VhcmNoSW5WUENPbmx5JykgJiZcbiAgICAgICFoaXBhYVNlY3VyaXR5T3BlblNlYXJjaEluVlBDT25seShub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0hJUEFBLlNlY3VyaXR5LU9wZW5TZWFyY2hJblZQQ09ubHknO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgT3BlblNlYXJjaCBTZXJ2aWNlIGRvbWFpbiBpcyBub3QgcnVubmluZyB3aXRoaW4gYSBWUEMgLSAoQ29udHJvbCBJRHM6IDE2NC4zMDgoYSkoMykoaSksIDE2NC4zMDgoYSkoNCkoaWkpKEEpLCAxNjQuMzA4KGEpKDQpKGlpKShDKSwgMTY0LjMxMihhKSgxKSwgMTY0LjMxMihlKSgxKSkuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ1ZQQ3MgaGVscCBzZWN1cmUgeW91ciBBV1MgcmVzb3VyY2VzIGFuZCBwcm92aWRlIGFuIGV4dHJhIGxheWVyIG9mIHByb3RlY3Rpb24uJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdISVBBQS5TZWN1cml0eS1PcGVuU2VhcmNoTG9nc1RvQ2xvdWRXYXRjaCcpICYmXG4gICAgICAhaGlwYWFTZWN1cml0eU9wZW5TZWFyY2hMb2dzVG9DbG91ZFdhdGNoKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnSElQQUEuU2VjdXJpdHktT3BlblNlYXJjaExvZ3NUb0Nsb3VkV2F0Y2gnO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgT3BlblNlYXJjaCBTZXJ2aWNlIGRvbWFpbiBkb2VzIG5vdCBzdHJlYW0gZXJyb3IgbG9ncyAoRVNfQVBQTElDQVRJT05fTE9HUykgdG8gQ2xvdWRXYXRjaCBMb2dzIC0gKENvbnRyb2wgSURzOiAxNjQuMzA4KGEpKDMpKGlpKShBKSwgMTY0LjMxMihiKSkuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ0Vuc3VyZSBBbWF6b24gT3BlblNlYXJjaCBTZXJ2aWNlIGRvbWFpbnMgaGF2ZSBlcnJvciBsb2dzIGVuYWJsZWQgYW5kIHN0cmVhbWVkIHRvIEFtYXpvbiBDbG91ZFdhdGNoIExvZ3MgZm9yIHJldGVudGlvbiBhbmQgcmVzcG9uc2UuIERvbWFpbiBlcnJvciBsb2dzIGNhbiBhc3Npc3Qgd2l0aCBzZWN1cml0eSBhbmQgYWNjZXNzIGF1ZGl0cywgYW5kIGNhbiBoZWxwIHRvIGRpYWdub3NlIGF2YWlsYWJpbGl0eSBpc3N1ZXMuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKFxuICAgICAgICBpZ25vcmVzLFxuICAgICAgICAnSElQQUEuU2VjdXJpdHktT3BlblNlYXJjaE5vZGVUb05vZGVFbmNyeXB0aW9uJyxcbiAgICAgICkgJiZcbiAgICAgICFoaXBhYVNlY3VyaXR5T3BlblNlYXJjaE5vZGVUb05vZGVFbmNyeXB0aW9uKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnSElQQUEuU2VjdXJpdHktT3BlblNlYXJjaE5vZGVUb05vZGVFbmNyeXB0aW9uJztcbiAgICAgIGNvbnN0IGluZm8gPVxuICAgICAgICAnVGhlIE9wZW5TZWFyY2ggU2VydmljZSBkb21haW4gZG9lcyBub3QgaGF2ZSBub2RlLXRvLW5vZGUgZW5jcnlwdGlvbiBlbmFibGVkIC0gKENvbnRyb2wgSURzOiAxNjQuMzEyKGEpKDIpKGl2KSwgMTY0LjMxMihlKSgxKSwgMTY0LjMxMihlKSgyKShpKSwgMTY0LjMxMihlKSgyKShpaSkpLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdCZWNhdXNlIHNlbnNpdGl2ZSBkYXRhIGNhbiBleGlzdCwgZW5hYmxlIGVuY3J5cHRpb24gaW4gdHJhbnNpdCB0byBoZWxwIHByb3RlY3QgdGhhdCBkYXRhIHdpdGhpbiB5b3VyIEFtYXpvbiBPcGVuU2VhcmNoIFNlcnZpY2UgKE9wZW5TZWFyY2ggU2VydmljZSkgZG9tYWlucy4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIENoZWNrIFJEUyBSZXNvdXJjZXNcbiAgICogQHBhcmFtIG5vZGUgdGhlIElDb25zdHJ1Y3QgdG8gZXZhbHVhdGVcbiAgICogQHBhcmFtIGlnbm9yZXMgbGlzdCBvZiBpZ25vcmVzIGZvciB0aGUgcmVzb3VyY2VcbiAgICovXG4gIHByaXZhdGUgY2hlY2tSRFMobm9kZTogQ2ZuUmVzb3VyY2UsIGlnbm9yZXM6IGFueSk6IHZvaWQge1xuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoXG4gICAgICAgIGlnbm9yZXMsXG4gICAgICAgICdISVBBQS5TZWN1cml0eS1SRFNBdXRvbWF0aWNNaW5vclZlcnNpb25VcGdyYWRlRW5hYmxlZCcsXG4gICAgICApICYmXG4gICAgICAhaGlwYWFTZWN1cml0eVJEU0F1dG9tYXRpY01pbm9yVmVyc2lvblVwZ3JhZGVFbmFibGVkKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnSElQQUEuU2VjdXJpdHktUkRTQXV0b21hdGljTWlub3JWZXJzaW9uVXBncmFkZUVuYWJsZWQnO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgUkRTIERCIGluc3RhbmNlIGRvZXMgbm90IGhhdmUgYXV0b21hdGljIG1pbm9yIHZlcnNpb24gdXBncmFkZXMgZW5hYmxlZCAtIChDb250cm9sIElEOiAxNjQuMzA4KGEpKDUpKGlpKShBKSkuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ0VuYWJsZSBhdXRvbWF0aWMgbWlub3IgdmVyc2lvbiB1cGdyYWRlcyBvbiB5b3VyIEFtYXpvbiBSZWxhdGlvbmFsIERhdGFiYXNlIFNlcnZpY2UgKFJEUykgaW5zdGFuY2VzIHRvIGVuc3VyZSB0aGUgbGF0ZXN0IG1pbm9yIHZlcnNpb24gdXBkYXRlcyB0byB0aGUgUmVsYXRpb25hbCBEYXRhYmFzZSBNYW5hZ2VtZW50IFN5c3RlbSAoUkRCTVMpIGFyZSBpbnN0YWxsZWQsIHdoaWNoIG1heSBpbmNsdWRlIHNlY3VyaXR5IHBhdGNoZXMgYW5kIGJ1ZyBmaXhlcy4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoXG4gICAgICAgIGlnbm9yZXMsXG4gICAgICAgICdISVBBQS5TZWN1cml0eS1SRFNFbmhhbmNlZE1vbml0b3JpbmdFbmFibGVkJyxcbiAgICAgICkgJiZcbiAgICAgICFoaXBhYVNlY3VyaXR5UkRTRW5oYW5jZWRNb25pdG9yaW5nRW5hYmxlZChub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0hJUEFBLlNlY3VyaXR5LVJEU0VuaGFuY2VkTW9uaXRvcmluZ0VuYWJsZWQnO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgUkRTIERCIGluc3RhbmNlIGRvZXMgbm90IGVuaGFuY2VkIG1vbml0b3JpbmcgZW5hYmxlZCAtIChDb250cm9sIElEOiAxNjQuMzEyKGIpKS4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnRW5hYmxlIGVuaGFuY2VkIG1vbml0b3JpbmcgdG8gaGVscCBtb25pdG9yIEFtYXpvbiBSRFMgYXZhaWxhYmlsaXR5LiBUaGlzIHByb3ZpZGVzIGRldGFpbGVkIHZpc2liaWxpdHkgaW50byB0aGUgaGVhbHRoIG9mIHlvdXIgQW1hem9uIFJEUyBkYXRhYmFzZSBpbnN0YW5jZXMuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdISVBBQS5TZWN1cml0eS1SRFNJbnN0YW5jZUJhY2t1cEVuYWJsZWQnKSAmJlxuICAgICAgIWhpcGFhU2VjdXJpdHlSRFNJbnN0YW5jZUJhY2t1cEVuYWJsZWQobm9kZSlcbiAgICApIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdISVBBQS5TZWN1cml0eS1SRFNJbnN0YW5jZUJhY2t1cEVuYWJsZWQnO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgUkRTIERCIGluc3RhbmNlIGRvZXMgbm90IGhhdmUgYmFja3VwcyBlbmFibGVkIC0gKENvbnRyb2wgSURzOiAxNjQuMzA4KGEpKDcpKGkpLCAxNjQuMzA4KGEpKDcpKGlpKShBKSwgMTY0LjMwOChhKSg3KShpaSkoQikpLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdUaGUgYmFja3VwIGZlYXR1cmUgb2YgQW1hem9uIFJEUyBjcmVhdGVzIGJhY2t1cHMgb2YgeW91ciBkYXRhYmFzZXMgYW5kIHRyYW5zYWN0aW9uIGxvZ3MuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKFxuICAgICAgICBpZ25vcmVzLFxuICAgICAgICAnSElQQUEuU2VjdXJpdHktUkRTSW5zdGFuY2VEZWxldGlvblByb3RlY3Rpb25FbmFibGVkJyxcbiAgICAgICkgJiZcbiAgICAgICFoaXBhYVNlY3VyaXR5UkRTSW5zdGFuY2VEZWxldGlvblByb3RlY3Rpb25FbmFibGVkKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnSElQQUEuU2VjdXJpdHktUkRTSW5zdGFuY2VEZWxldGlvblByb3RlY3Rpb25FbmFibGVkJztcbiAgICAgIGNvbnN0IGluZm8gPVxuICAgICAgICAnVGhlIFJEUyBEQiBpbnN0YW5jZSBvciBBdXJvcmEgREIgY2x1c3RlciBkb2VzIG5vdCBoYXZlIGRlbGV0aW9uIHByb3RlY3Rpb24gZW5hYmxlZCAtIChDb250cm9sIElEczogMTY0LjMwOChhKSg3KShpKSwgMTY0LjMwOChhKSg3KShpaSkoQykpLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdFbnN1cmUgQW1hem9uIFJlbGF0aW9uYWwgRGF0YWJhc2UgU2VydmljZSAoQW1hem9uIFJEUykgaW5zdGFuY2VzIGFuZCBjbHVzdGVycyBoYXZlIGRlbGV0aW9uIHByb3RlY3Rpb24gZW5hYmxlZC4gVXNlIGRlbGV0aW9uIHByb3RlY3Rpb24gdG8gcHJldmVudCB5b3VyIEFtYXpvbiBSRFMgREIgaW5zdGFuY2VzIGFuZCBjbHVzdGVycyBmcm9tIGJlaW5nIGFjY2lkZW50YWxseSBvciBtYWxpY2lvdXNseSBkZWxldGVkLCB3aGljaCBjYW4gbGVhZCB0byBsb3NzIG9mIGF2YWlsYWJpbGl0eSBmb3IgeW91ciBhcHBsaWNhdGlvbnMuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdISVBBQS5TZWN1cml0eS1SRFNJbnN0YW5jZU11bHRpQXpTdXBwb3J0JykgJiZcbiAgICAgICFoaXBhYVNlY3VyaXR5UkRTSW5zdGFuY2VNdWx0aUFaU3VwcG9ydChub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0hJUEFBLlNlY3VyaXR5LVJEU0luc3RhbmNlTXVsdGlBelN1cHBvcnQnO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgbm9uLUF1cm9yYSBSRFMgREIgaW5zdGFuY2UgZG9lcyBub3QgaGF2ZSBtdWx0aS1BWiBzdXBwb3J0IGVuYWJsZWQgLSAoQ29udHJvbCBJRHM6IDE2NC4zMDgoYSkoNykoaSksIDE2NC4zMDgoYSkoNykoaWkpKEMpKS4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnTXVsdGktQVogc3VwcG9ydCBpbiBBbWF6b24gUmVsYXRpb25hbCBEYXRhYmFzZSBTZXJ2aWNlIChBbWF6b24gUkRTKSBwcm92aWRlcyBlbmhhbmNlZCBhdmFpbGFiaWxpdHkgYW5kIGR1cmFiaWxpdHkgZm9yIGRhdGFiYXNlIGluc3RhbmNlcy4gV2hlbiB5b3UgcHJvdmlzaW9uIGEgTXVsdGktQVogZGF0YWJhc2UgaW5zdGFuY2UsIEFtYXpvbiBSRFMgYXV0b21hdGljYWxseSBjcmVhdGVzIGEgcHJpbWFyeSBkYXRhYmFzZSBpbnN0YW5jZSwgYW5kIHN5bmNocm9ub3VzbHkgcmVwbGljYXRlcyB0aGUgZGF0YSB0byBhIHN0YW5kYnkgaW5zdGFuY2UgaW4gYSBkaWZmZXJlbnQgQXZhaWxhYmlsaXR5IFpvbmUuIEluIGNhc2Ugb2YgYW4gaW5mcmFzdHJ1Y3R1cmUgZmFpbHVyZSwgQW1hem9uIFJEUyBwZXJmb3JtcyBhbiBhdXRvbWF0aWMgZmFpbG92ZXIgdG8gdGhlIHN0YW5kYnkgc28gdGhhdCB5b3UgY2FuIHJlc3VtZSBkYXRhYmFzZSBvcGVyYXRpb25zIGFzIHNvb24gYXMgdGhlIGZhaWxvdmVyIGlzIGNvbXBsZXRlLic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShpZ25vcmVzLCAnSElQQUEuU2VjdXJpdHktUkRTSW5zdGFuY2VQdWJsaWNBY2Nlc3MnKSAmJlxuICAgICAgIWhpcGFhU2VjdXJpdHlSRFNJbnN0YW5jZVB1YmxpY0FjY2Vzcyhub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0hJUEFBLlNlY3VyaXR5LVJEU0luc3RhbmNlUHVibGljQWNjZXNzJztcbiAgICAgIGNvbnN0IGluZm8gPVxuICAgICAgICAnVGhlIFJEUyBEQiBpbnN0YW5jZSBhbGxvd3MgcHVibGljIGFjY2VzcyAtIChDb250cm9sIElEczogMTY0LjMwOChhKSgzKShpKSwgMTY0LjMwOChhKSg0KShpaSkoQSksIDE2NC4zMDgoYSkoNCkoaWkpKEMpLCAxNjQuMzEyKGEpKDEpLCAxNjQuMzEyKGUpKDEpKS4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnQW1hem9uIFJEUyBkYXRhYmFzZSBpbnN0YW5jZXMgY2FuIGNvbnRhaW4gc2Vuc2l0aXZlIGluZm9ybWF0aW9uLCBhbmQgcHJpbmNpcGxlcyBhbmQgYWNjZXNzIGNvbnRyb2wgaXMgcmVxdWlyZWQgZm9yIHN1Y2ggYWNjb3VudHMuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdISVBBQS5TZWN1cml0eS1SRFNMb2dnaW5nRW5hYmxlZCcpICYmXG4gICAgICAhaGlwYWFTZWN1cml0eVJEU0xvZ2dpbmdFbmFibGVkKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnSElQQUEuU2VjdXJpdHktUkRTTG9nZ2luZ0VuYWJsZWQnO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgUkRTIERCIGluc3RhbmNlIGRvZXMgbm90IGhhdmUgYWxsIENsb3VkV2F0Y2ggbG9nIHR5cGVzIGV4cG9ydGVkIC0gKENvbnRyb2wgSURzOiAxNjQuMzA4KGEpKDMpKGlpKShBKSwgMTY0LjMwOChhKSg1KShpaSkoQykpLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdUbyBoZWxwIHdpdGggbG9nZ2luZyBhbmQgbW9uaXRvcmluZyB3aXRoaW4geW91ciBlbnZpcm9ubWVudCwgZW5zdXJlIEFtYXpvbiBSZWxhdGlvbmFsIERhdGFiYXNlIFNlcnZpY2UgKEFtYXpvbiBSRFMpIGxvZ2dpbmcgaXMgZW5hYmxlZC4gV2l0aCBBbWF6b24gUkRTIGxvZ2dpbmcsIHlvdSBjYW4gY2FwdHVyZSBldmVudHMgc3VjaCBhcyBjb25uZWN0aW9ucywgZGlzY29ubmVjdGlvbnMsIHF1ZXJpZXMsIG9yIHRhYmxlcyBxdWVyaWVkLic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShpZ25vcmVzLCAnSElQQUEuU2VjdXJpdHktUkRTU3RvcmFnZUVuY3J5cHRlZCcpICYmXG4gICAgICAhaGlwYWFTZWN1cml0eVJEU1N0b3JhZ2VFbmNyeXB0ZWQobm9kZSlcbiAgICApIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdISVBBQS5TZWN1cml0eS1SRFNTdG9yYWdlRW5jcnlwdGVkJztcbiAgICAgIGNvbnN0IGluZm8gPVxuICAgICAgICAnVGhlIFJEUyBEQiBpbnN0YW5jZSBvciBBdXJvcmEgREIgY2x1c3RlciBkb2VzIG5vdCBoYXZlIHN0b3JhZ2UgZW5jcnlwdGVkIC0gKENvbnRyb2wgSURzOiAxNjQuMzEyKGEpKDIpKGl2KSwgMTY0LjMxMihlKSgyKShpaSkpLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdCZWNhdXNlIHNlbnNpdGl2ZSBkYXRhIGNhbiBleGlzdCBhdCByZXN0IGluIEFtYXpvbiBSRFMgREIgaW5zdGFuY2VzIGFuZCBjbHVzdGVycywgZW5hYmxlIGVuY3J5cHRpb24gYXQgcmVzdCB0byBoZWxwIHByb3RlY3QgdGhhdCBkYXRhLic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogQ2hlY2sgUmVkc2hpZnQgUmVzb3VyY2VzXG4gICAqIEBwYXJhbSBub2RlIHRoZSBJQ29uc3RydWN0IHRvIGV2YWx1YXRlXG4gICAqIEBwYXJhbSBpZ25vcmVzIGxpc3Qgb2YgaWdub3JlcyBmb3IgdGhlIHJlc291cmNlXG4gICAqL1xuICBwcml2YXRlIGNoZWNrUmVkc2hpZnQobm9kZTogQ2ZuUmVzb3VyY2UsIGlnbm9yZXM6IGFueSk6IHZvaWQge1xuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoaWdub3JlcywgJ0hJUEFBLlNlY3VyaXR5LVJlZHNoaWZ0QmFja3VwRW5hYmxlZCcpICYmXG4gICAgICAhaGlwYWFTZWN1cml0eVJlZHNoaWZ0QmFja3VwRW5hYmxlZChub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0hJUEFBLlNlY3VyaXR5LVJlZHNoaWZ0QmFja3VwRW5hYmxlZCc7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSBSZWRzaGlmdCBjbHVzdGVyIGRvZXMgbm90IGhhdmUgYXV0b21hdGVkIHNuYXBzaG90cyBlbmFibGVkIG9yIHRoZSByZXRlbnRpb24gcGVyaW9kIGlzIG5vdCBiZXR3ZWVuIDEgYW5kIDM1IGRheXMgLSAoQ29udHJvbCBJRHM6IDE2NC4zMDgoYSkoNykoaSksIDE2NC4zMDgoYSkoNykoaWkpKEEpLCAxNjQuMzA4KGEpKDcpKGlpKShCKSkuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ1RvIGhlbHAgd2l0aCBkYXRhIGJhY2stdXAgcHJvY2Vzc2VzLCBlbnN1cmUgeW91ciBBbWF6b24gUmVkc2hpZnQgY2x1c3RlcnMgaGF2ZSBhdXRvbWF0ZWQgc25hcHNob3RzLiBXaGVuIGF1dG9tYXRlZCBzbmFwc2hvdHMgYXJlIGVuYWJsZWQgZm9yIGEgY2x1c3RlciwgUmVkc2hpZnQgcGVyaW9kaWNhbGx5IHRha2VzIHNuYXBzaG90cyBvZiB0aGF0IGNsdXN0ZXIuIEJ5IGRlZmF1bHQsIFJlZHNoaWZ0IHRha2VzIGEgc25hcHNob3QgZXZlcnkgZWlnaHQgaG91cnMgb3IgZXZlcnkgNSBHQiBwZXIgbm9kZSBvZiBkYXRhIGNoYW5nZXMsIG9yIHdoaWNoZXZlciBjb21lcyBmaXJzdC4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoXG4gICAgICAgIGlnbm9yZXMsXG4gICAgICAgICdISVBBQS5TZWN1cml0eS1SZWRzaGlmdENsdXN0ZXJDb25maWd1cmF0aW9uJyxcbiAgICAgICkgJiZcbiAgICAgICFoaXBhYVNlY3VyaXR5UmVkc2hpZnRDbHVzdGVyQ29uZmlndXJhdGlvbihub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0hJUEFBLlNlY3VyaXR5LVJlZHNoaWZ0Q2x1c3RlckNvbmZpZ3VyYXRpb24nO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgUmVkc2hpZnQgY2x1c3RlciBkb2VzIG5vdCBoYXZlIGVuY3J5cHRpb24gb3IgYXVkaXQgbG9nZ2luZyBlbmFibGVkIC0gKENvbnRyb2wgSURzOiAxNjQuMzEyKGEpKDIpKGl2KSwgMTY0LjMxMihiKSwgMTY0LjMxMihlKSgyKShpaSkpLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdUbyBwcm90ZWN0IGRhdGEgYXQgcmVzdCwgZW5zdXJlIHRoYXQgZW5jcnlwdGlvbiBpcyBlbmFibGVkIGZvciB5b3VyIEFtYXpvbiBSZWRzaGlmdCBjbHVzdGVycy4gWW91IG11c3QgYWxzbyBlbnN1cmUgdGhhdCByZXF1aXJlZCBjb25maWd1cmF0aW9ucyBhcmUgZGVwbG95ZWQgb24gQW1hem9uIFJlZHNoaWZ0IGNsdXN0ZXJzLiBUaGUgYXVkaXQgbG9nZ2luZyBzaG91bGQgYmUgZW5hYmxlZCB0byBwcm92aWRlIGluZm9ybWF0aW9uIGFib3V0IGNvbm5lY3Rpb25zIGFuZCB1c2VyIGFjdGl2aXRpZXMgaW4gdGhlIGRhdGFiYXNlLic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShcbiAgICAgICAgaWdub3JlcyxcbiAgICAgICAgJ0hJUEFBLlNlY3VyaXR5LVJlZHNoaWZ0Q2x1c3Rlck1haW50ZW5hbmNlU2V0dGluZ3MnLFxuICAgICAgKSAmJlxuICAgICAgIWhpcGFhU2VjdXJpdHlSZWRzaGlmdENsdXN0ZXJNYWludGVuYW5jZVNldHRpbmdzKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnSElQQUEuU2VjdXJpdHktUmVkc2hpZnRDbHVzdGVyTWFpbnRlbmFuY2VTZXR0aW5ncyc7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSBSZWRzaGlmdCBjbHVzdGVyIGhhcyB2ZXJzaW9uIHVwZ3JhZGVzIGVuYWJsZWQsIGF1dG9tYXRlZCBzbmFwc2hvdCByZXRlbnRpb24gcGVyaW9kcyBlbmFibGVkLCBhbmQgYW4gZXhwbGljaXQgbWFpbnRlbmFuY2Ugd2luZG93IGNvbmZpZ3VyZWQgLSAoQ29udHJvbCBJRHM6IDE2NC4zMDgoYSkoNSkoaWkpKEEpLCAxNjQuMzA4KGEpKDcpKGlpKShBKSkuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ0Vuc3VyZSB0aGF0IEFtYXpvbiBSZWRzaGlmdCBjbHVzdGVycyBoYXZlIHRoZSBwcmVmZXJyZWQgc2V0dGluZ3MgZm9yIHlvdXIgb3JnYW5pemF0aW9uLiBTcGVjaWZpY2FsbHksIHRoYXQgdGhleSBoYXZlIHByZWZlcnJlZCBtYWludGVuYW5jZSB3aW5kb3dzIGFuZCBhdXRvbWF0ZWQgc25hcHNob3QgcmV0ZW50aW9uIHBlcmlvZHMgZm9yIHRoZSBkYXRhYmFzZS4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoaWdub3JlcywgJ0hJUEFBLlNlY3VyaXR5LVJlZHNoaWZ0Q2x1c3RlclB1YmxpY0FjY2VzcycpICYmXG4gICAgICAhaGlwYWFTZWN1cml0eVJlZHNoaWZ0Q2x1c3RlclB1YmxpY0FjY2Vzcyhub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0hJUEFBLlNlY3VyaXR5LVJlZHNoaWZ0Q2x1c3RlclB1YmxpY0FjY2Vzcyc7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSBSZWRzaGlmdCBjbHVzdGVyIGFsbG93cyBwdWJsaWMgYWNjZXNzIC0gKENvbnRyb2wgSURzOiAxNjQuMzA4KGEpKDMpKGkpLCAxNjQuMzA4KGEpKDQpKGlpKShBKSwgMTY0LjMwOChhKSg0KShpaSkoQyksIDE2NC4zMTIoYSkoMSksIDE2NC4zMTIoZSkoMSkpLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdBbWF6b24gUmVkc2hpZnQgY2x1c3RlcnMgY2FuIGNvbnRhaW4gc2Vuc2l0aXZlIGluZm9ybWF0aW9uIGFuZCBwcmluY2lwbGVzIGFuZCBhY2Nlc3MgY29udHJvbCBpcyByZXF1aXJlZCBmb3Igc3VjaCBhY2NvdW50cy4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoXG4gICAgICAgIGlnbm9yZXMsXG4gICAgICAgICdISVBBQS5TZWN1cml0eS1SZWRzaGlmdEVuaGFuY2VkVlBDUm91dGluZ0VuYWJsZWQnLFxuICAgICAgKSAmJlxuICAgICAgIWhpcGFhU2VjdXJpdHlSZWRzaGlmdEVuaGFuY2VkVlBDUm91dGluZ0VuYWJsZWQobm9kZSlcbiAgICApIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdISVBBQS5TZWN1cml0eS1SZWRzaGlmdEVuaGFuY2VkVlBDUm91dGluZ0VuYWJsZWQnO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgUmVkc2hpZnQgY2x1c3RlciBkb2VzIG5vdCBoYXZlIGVuaGFuY2VkIFZQQyByb3V0aW5nIGVuYWJsZWQgLSAoQ29udHJvbCBJRHM6IDE2NC4zMTIoZSkoMSkpLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdFbmhhbmNlZCBWUEMgcm91dGluZyBmb3JjZXMgYWxsIENPUFkgYW5kIFVOTE9BRCB0cmFmZmljIGJldHdlZW4gdGhlIGNsdXN0ZXIgYW5kIGRhdGEgcmVwb3NpdG9yaWVzIHRvIGdvIHRocm91Z2ggeW91ciBBbWF6b24gVlBDLiBZb3UgY2FuIHRoZW4gdXNlIFZQQyBmZWF0dXJlcyBzdWNoIGFzIHNlY3VyaXR5IGdyb3VwcyBhbmQgbmV0d29yayBhY2Nlc3MgY29udHJvbCBsaXN0cyB0byBzZWN1cmUgbmV0d29yayB0cmFmZmljLiBZb3UgY2FuIGFsc28gdXNlIFZQQyBmbG93IGxvZ3MgdG8gbW9uaXRvciBuZXR3b3JrIHRyYWZmaWMuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBDaGVjayBBbWF6b24gUzMgUmVzb3VyY2VzXG4gICAqIEBwYXJhbSBub2RlIHRoZSBJQ29uc3RydWN0IHRvIGV2YWx1YXRlXG4gICAqIEBwYXJhbSBpZ25vcmVzIGxpc3Qgb2YgaWdub3JlcyBmb3IgdGhlIHJlc291cmNlXG4gICAqL1xuICBwcml2YXRlIGNoZWNrUzMobm9kZTogQ2ZuUmVzb3VyY2UsIGlnbm9yZXM6IGFueSk6IHZvaWQge1xuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoXG4gICAgICAgIGlnbm9yZXMsXG4gICAgICAgICdISVBBQS5TZWN1cml0eS1TM0J1Y2tldExldmVsUHVibGljQWNjZXNzUHJvaGliaXRlZCcsXG4gICAgICApICYmXG4gICAgICAhaGlwYWFTZWN1cml0eVMzQnVja2V0TGV2ZWxQdWJsaWNBY2Nlc3NQcm9oaWJpdGVkKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnSElQQUEuU2VjdXJpdHktUzNCdWNrZXRMZXZlbFB1YmxpY0FjY2Vzc1Byb2hpYml0ZWQnO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgUzMgYnVja2V0IGRvZXMgbm90IHByb2hpYml0IHB1YmxpYyBhY2Nlc3MgdGhyb3VnaCBidWNrZXQgbGV2ZWwgc2V0dGluZ3MgLSAoQ29udHJvbCBJRHM6IDE2NC4zMDgoYSkoMykoaSksIDE2NC4zMDgoYSkoNCkoaWkpKEEpLCAxNjQuMzA4KGEpKDQpKGlpKShDKSwgMTY0LjMxMihhKSgxKSwgMTY0LjMxMihlKSgxKSkuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ0tlZXAgc2Vuc2l0aXZlIGRhdGEgc2FmZSBmcm9tIHVuYXV0aG9yaXplZCByZW1vdGUgdXNlcnMgYnkgcHJldmVudGluZyBwdWJsaWMgYWNjZXNzIGF0IHRoZSBidWNrZXQgbGV2ZWwuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdISVBBQS5TZWN1cml0eS1TM0J1Y2tldExvZ2dpbmdFbmFibGVkJykgJiZcbiAgICAgICFoaXBhYVNlY3VyaXR5UzNCdWNrZXRMb2dnaW5nRW5hYmxlZChub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0hJUEFBLlNlY3VyaXR5LVMzQnVja2V0TG9nZ2luZ0VuYWJsZWQnO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgUzMgQnVja2V0IGRvZXMgbm90IGhhdmUgc2VydmVyIGFjY2VzcyBsb2dzIGVuYWJsZWQgLSAoQ29udHJvbCBJRHM6IDE2NC4zMDgoYSkoMykoaWkpKEEpLCAxNjQuMzEyKGIpKS4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnQW1hem9uIFNpbXBsZSBTdG9yYWdlIFNlcnZpY2UgKEFtYXpvbiBTMykgc2VydmVyIGFjY2VzcyBsb2dnaW5nIHByb3ZpZGVzIGEgbWV0aG9kIHRvIG1vbml0b3IgdGhlIG5ldHdvcmsgZm9yIHBvdGVudGlhbCBjeWJlcnNlY3VyaXR5IGV2ZW50cy4gVGhlIGV2ZW50cyBhcmUgbW9uaXRvcmVkIGJ5IGNhcHR1cmluZyBkZXRhaWxlZCByZWNvcmRzIGZvciB0aGUgcmVxdWVzdHMgdGhhdCBhcmUgbWFkZSB0byBhbiBBbWF6b24gUzMgYnVja2V0LiBFYWNoIGFjY2VzcyBsb2cgcmVjb3JkIHByb3ZpZGVzIGRldGFpbHMgYWJvdXQgYSBzaW5nbGUgYWNjZXNzIHJlcXVlc3QuIFRoZSBkZXRhaWxzIGluY2x1ZGUgdGhlIHJlcXVlc3RlciwgYnVja2V0IG5hbWUsIHJlcXVlc3QgdGltZSwgcmVxdWVzdCBhY3Rpb24sIHJlc3BvbnNlIHN0YXR1cywgYW5kIGFuIGVycm9yIGNvZGUsIGlmIHJlbGV2YW50Lic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShcbiAgICAgICAgaWdub3JlcyxcbiAgICAgICAgJ0hJUEFBLlNlY3VyaXR5LVMzQnVja2V0UHVibGljUmVhZFByb2hpYml0ZWQnLFxuICAgICAgKSAmJlxuICAgICAgIWhpcGFhU2VjdXJpdHlTM0J1Y2tldFB1YmxpY1JlYWRQcm9oaWJpdGVkKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnSElQQUEuU2VjdXJpdHktUzNCdWNrZXRQdWJsaWNSZWFkUHJvaGliaXRlZCc7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSBTMyBCdWNrZXQgZG9lcyBub3QgcHJvaGliaXQgcHVibGljIHJlYWQgYWNjZXNzIHRocm91Z2ggaXRzIEJsb2NrIFB1YmxpYyBBY2Nlc3MgY29uZmlndXJhdGlvbnMgYW5kIGJ1Y2tldCBBQ0xzIC0gKENvbnRyb2wgSURzOiAxNjQuMzA4KGEpKDMpKGkpLCAxNjQuMzA4KGEpKDQpKGlpKShBKSwgMTY0LjMwOChhKSg0KShpaSkoQyksIDE2NC4zMTIoYSkoMSksIDE2NC4zMTIoZSkoMSkpLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdUaGUgbWFuYWdlbWVudCBvZiBhY2Nlc3Mgc2hvdWxkIGJlIGNvbnNpc3RlbnQgd2l0aCB0aGUgY2xhc3NpZmljYXRpb24gb2YgdGhlIGRhdGEuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKFxuICAgICAgICBpZ25vcmVzLFxuICAgICAgICAnSElQQUEuU2VjdXJpdHktUzNCdWNrZXRQdWJsaWNXcml0ZVByb2hpYml0ZWQnLFxuICAgICAgKSAmJlxuICAgICAgIWhpcGFhU2VjdXJpdHlTM0J1Y2tldFB1YmxpY1dyaXRlUHJvaGliaXRlZChub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0hJUEFBLlNlY3VyaXR5LVMzQnVja2V0UHVibGljV3JpdGVQcm9oaWJpdGVkJztcbiAgICAgIGNvbnN0IGluZm8gPVxuICAgICAgICAnVGhlIFMzIEJ1Y2tldCBkb2VzIG5vdCBwcm9oaWJpdCBwdWJsaWMgd3JpdGUgYWNjZXNzIHRocm91Z2ggaXRzIEJsb2NrIFB1YmxpYyBBY2Nlc3MgY29uZmlndXJhdGlvbnMgYW5kIGJ1Y2tldCBBQ0xzIC0gKENvbnRyb2wgSURzOiAxNjQuMzA4KGEpKDMpKGkpLCAxNjQuMzA4KGEpKDQpKGlpKShBKSwgMTY0LjMwOChhKSg0KShpaSkoQyksIDE2NC4zMTIoYSkoMSksIDE2NC4zMTIoZSkoMSkpLic7XG4gICAgICBjb25zdCBleHBsYW5hdGlvbiA9XG4gICAgICAgICdUaGUgbWFuYWdlbWVudCBvZiBhY2Nlc3Mgc2hvdWxkIGJlIGNvbnNpc3RlbnQgd2l0aCB0aGUgY2xhc3NpZmljYXRpb24gb2YgdGhlIGRhdGEuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdISVBBQS5TZWN1cml0eS1TM0J1Y2tldFJlcGxpY2F0aW9uRW5hYmxlZCcpICYmXG4gICAgICAhaGlwYWFTZWN1cml0eVMzQnVja2V0UmVwbGljYXRpb25FbmFibGVkKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnSElQQUEuU2VjdXJpdHktUzNCdWNrZXRSZXBsaWNhdGlvbkVuYWJsZWQnO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgUzMgQnVja2V0IGRvZXMgbm90IGhhdmUgcmVwbGljYXRpb24gZW5hYmxlZCAtIChDb250cm9sIElEczogMTY0LjMwOChhKSg3KShpKSwgMTY0LjMwOChhKSg3KShpaSkoQSksIDE2NC4zMDgoYSkoNykoaWkpKEIpKS4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnQW1hem9uIFNpbXBsZSBTdG9yYWdlIFNlcnZpY2UgKEFtYXpvbiBTMykgQ3Jvc3MtUmVnaW9uIFJlcGxpY2F0aW9uIChDUlIpIHN1cHBvcnRzIG1haW50YWluaW5nIGFkZXF1YXRlIGNhcGFjaXR5IGFuZCBhdmFpbGFiaWxpdHkuIENSUiBlbmFibGVzIGF1dG9tYXRpYywgYXN5bmNocm9ub3VzIGNvcHlpbmcgb2Ygb2JqZWN0cyBhY3Jvc3MgQW1hem9uIFMzIGJ1Y2tldHMgdG8gaGVscCBlbnN1cmUgdGhhdCBkYXRhIGF2YWlsYWJpbGl0eSBpcyBtYWludGFpbmVkLic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShcbiAgICAgICAgaWdub3JlcyxcbiAgICAgICAgJ0hJUEFBLlNlY3VyaXR5LVMzQnVja2V0U2VydmVyU2lkZUVuY3J5cHRpb25FbmFibGVkJyxcbiAgICAgICkgJiZcbiAgICAgICFoaXBhYVNlY3VyaXR5UzNCdWNrZXRTZXJ2ZXJTaWRlRW5jcnlwdGlvbkVuYWJsZWQobm9kZSlcbiAgICApIHtcbiAgICAgIGNvbnN0IHJ1bGVJZCA9ICdISVBBQS5TZWN1cml0eS1TM0J1Y2tldFNlcnZlclNpZGVFbmNyeXB0aW9uRW5hYmxlZCc7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSBTMyBCdWNrZXQgZG9lcyBub3QgaGF2ZSBkZWZhdWx0IHNlcnZlci1zaWRlIGVuY3J5cHRpb24gZW5hYmxlZCAtIChDb250cm9sIElEczogMTY0LjMxMihhKSgyKShpdiksIDE2NC4zMTIoYykoMiksIDE2NC4zMTIoZSkoMikoaWkpKS4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnQmVjYXVzZSBzZW5zaXRpdmUgZGF0YSBjYW4gZXhpc3QgYXQgcmVzdCBpbiBBbWF6b24gUzMgYnVja2V0cywgZW5hYmxlIGVuY3J5cHRpb24gdG8gaGVscCBwcm90ZWN0IHRoYXQgZGF0YS4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoaWdub3JlcywgJ0hJUEFBLlNlY3VyaXR5LVMzQnVja2V0VmVyc2lvbmluZ0VuYWJsZWQnKSAmJlxuICAgICAgIWhpcGFhU2VjdXJpdHlTM0J1Y2tldFZlcnNpb25pbmdFbmFibGVkKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnSElQQUEuU2VjdXJpdHktUzNCdWNrZXRWZXJzaW9uaW5nRW5hYmxlZCc7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSBTMyBCdWNrZXQgZG9lcyBub3QgaGF2ZSB2ZXJzaW9uaW5nIGVuYWJsZWQgLSAoQ29udHJvbCBJRHM6IDE2NC4zMDgoYSkoNykoaSksIDE2NC4zMDgoYSkoNykoaWkpKEEpLCAxNjQuMzA4KGEpKDcpKGlpKShCKSwgMTY0LjMxMihjKSgxKSwgMTY0LjMxMihjKSgyKSkuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ1VzZSB2ZXJzaW9uaW5nIHRvIHByZXNlcnZlLCByZXRyaWV2ZSwgYW5kIHJlc3RvcmUgZXZlcnkgdmVyc2lvbiBvZiBldmVyeSBvYmplY3Qgc3RvcmVkIGluIHlvdXIgQW1hem9uIFMzIGJ1Y2tldC4gVmVyc2lvbmluZyBoZWxwcyB5b3UgdG8gZWFzaWx5IHJlY292ZXIgZnJvbSB1bmludGVuZGVkIHVzZXIgYWN0aW9ucyBhbmQgYXBwbGljYXRpb24gZmFpbHVyZXMuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdISVBBQS5TZWN1cml0eS1TM0RlZmF1bHRFbmNyeXB0aW9uS01TJykgJiZcbiAgICAgICFoaXBhYVNlY3VyaXR5UzNEZWZhdWx0RW5jcnlwdGlvbktNUyhub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0hJUEFBLlNlY3VyaXR5LVMzRGVmYXVsdEVuY3J5cHRpb25LTVMnO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgUzMgQnVja2V0IGlzIG5vdCBlbmNyeXB0ZWQgd2l0aCBhIEtNUyBLZXkgYnkgZGVmYXVsdCAtIChDb250cm9sIElEczogMTY0LjMxMihhKSgyKShpdiksIDE2NC4zMTIoZSkoMikoaWkpKS4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnRW5zdXJlIHRoYXQgZW5jcnlwdGlvbiBpcyBlbmFibGVkIGZvciB5b3VyIEFtYXpvbiBTaW1wbGUgU3RvcmFnZSBTZXJ2aWNlIChBbWF6b24gUzMpIGJ1Y2tldHMuIEJlY2F1c2Ugc2Vuc2l0aXZlIGRhdGEgY2FuIGV4aXN0IGF0IHJlc3QgaW4gYW4gQW1hem9uIFMzIGJ1Y2tldCwgZW5hYmxlIGVuY3J5cHRpb24gYXQgcmVzdCB0byBoZWxwIHByb3RlY3QgdGhhdCBkYXRhLic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogQ2hlY2sgU2FnZU1ha2VyIFJlc291cmNlc1xuICAgKiBAcGFyYW0gbm9kZSB0aGUgSUNvbnN0cnVjdCB0byBldmFsdWF0ZVxuICAgKiBAcGFyYW0gaWdub3JlcyBsaXN0IG9mIGlnbm9yZXMgZm9yIHRoZSByZXNvdXJjZVxuICAgKi9cbiAgcHJpdmF0ZSBjaGVja1NhZ2VNYWtlcihub2RlOiBDZm5SZXNvdXJjZSwgaWdub3JlczogYW55KSB7XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShcbiAgICAgICAgaWdub3JlcyxcbiAgICAgICAgJ0hJUEFBLlNlY3VyaXR5LVNhZ2VNYWtlckVuZHBvaW50Q29uZmlndXJhdGlvbktNU0tleUNvbmZpZ3VyZWQnLFxuICAgICAgKSAmJlxuICAgICAgIWhpcGFhU2VjdXJpdHlTYWdlTWFrZXJFbmRwb2ludENvbmZpZ3VyYXRpb25LTVNLZXlDb25maWd1cmVkKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPVxuICAgICAgICAnSElQQUEuU2VjdXJpdHktU2FnZU1ha2VyRW5kcG9pbnRDb25maWd1cmF0aW9uS01TS2V5Q29uZmlndXJlZCc7XG4gICAgICBjb25zdCBpbmZvID1cbiAgICAgICAgJ1RoZSBTYWdlTWFrZXIgZW5kcG9pbnQgaXMgbm90IGVuY3J5cHRlZCB3aXRoIGEgS01TIGtleSAtIChDb250cm9sIElEczogMTY0LjMxMihhKSgyKShpdiksIDE2NC4zMTIoZSkoMikoaWkpKS4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnQmVjYXVzZSBzZW5zaXRpdmUgZGF0YSBjYW4gZXhpc3QgYXQgcmVzdCBpbiBTYWdlTWFrZXIgZW5kcG9pbnQsIGVuYWJsZSBlbmNyeXB0aW9uIGF0IHJlc3QgdG8gaGVscCBwcm90ZWN0IHRoYXQgZGF0YS4nO1xuICAgICAgQW5ub3RhdGlvbnMub2Yobm9kZSkuYWRkRXJyb3IoXG4gICAgICAgIHRoaXMuY3JlYXRlTWVzc2FnZShydWxlSWQsIGluZm8sIGV4cGxhbmF0aW9uKSxcbiAgICAgICk7XG4gICAgfVxuICAgIGlmIChcbiAgICAgICF0aGlzLmlnbm9yZVJ1bGUoXG4gICAgICAgIGlnbm9yZXMsXG4gICAgICAgICdISVBBQS5TZWN1cml0eS1TYWdlTWFrZXJOb3RlYm9va0luc3RhbmNlS01TS2V5Q29uZmlndXJlZCcsXG4gICAgICApICYmXG4gICAgICAhaGlwYWFTZWN1cml0eVNhZ2VNYWtlck5vdGVib29rSW5zdGFuY2VLTVNLZXlDb25maWd1cmVkKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnSElQQUEuU2VjdXJpdHktU2FnZU1ha2VyTm90ZWJvb2tJbnN0YW5jZUtNU0tleUNvbmZpZ3VyZWQnO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgU2FnZU1ha2VyIG5vdGVib29rIGlzIG5vdCBlbmNyeXB0ZWQgd2l0aCBhIEtNUyBrZXkgLSAoQ29udHJvbCBJRHM6IDE2NC4zMTIoYSkoMikoaXYpLCAxNjQuMzEyKGUpKDIpKGlpKSkuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ0JlY2F1c2Ugc2Vuc2l0aXZlIGRhdGEgY2FuIGV4aXN0IGF0IHJlc3QgaW4gU2FnZU1ha2VyIG5vdGVib29rLCBlbmFibGUgZW5jcnlwdGlvbiBhdCByZXN0IHRvIGhlbHAgcHJvdGVjdCB0aGF0IGRhdGEuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKFxuICAgICAgICBpZ25vcmVzLFxuICAgICAgICAnSElQQUEuU2VjdXJpdHktU2FnZU1ha2VyTm90ZWJvb2tOb0RpcmVjdEludGVybmV0QWNjZXNzJyxcbiAgICAgICkgJiZcbiAgICAgICFoaXBhYVNlY3VyaXR5U2FnZU1ha2VyTm90ZWJvb2tOb0RpcmVjdEludGVybmV0QWNjZXNzKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnSElQQUEuU2VjdXJpdHktU2FnZU1ha2VyTm90ZWJvb2tOb0RpcmVjdEludGVybmV0QWNjZXNzJztcbiAgICAgIGNvbnN0IGluZm8gPVxuICAgICAgICAnVGhlIFNhZ2VNYWtlciBub3RlYm9vayBkb2VzIG5vdCBkaXNhYmxlIGRpcmVjdCBpbnRlcm5ldCBhY2Nlc3MgLSAoQ29udHJvbCBJRHM6IDE2NC4zMDgoYSkoMykoaSksIDE2NC4zMDgoYSkoNCkoaWkpKEEpLCAxNjQuMzA4KGEpKDQpKGlpKShDKSwgMTY0LjMxMihhKSgxKSwgMTY0LjMxMihlKSgxKSkuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ0J5IHByZXZlbnRpbmcgZGlyZWN0IGludGVybmV0IGFjY2VzcywgeW91IGNhbiBrZWVwIHNlbnNpdGl2ZSBkYXRhIGZyb20gYmVpbmcgYWNjZXNzZWQgYnkgdW5hdXRob3JpemVkIHVzZXJzLic7XG4gICAgICBBbm5vdGF0aW9ucy5vZihub2RlKS5hZGRFcnJvcihcbiAgICAgICAgdGhpcy5jcmVhdGVNZXNzYWdlKHJ1bGVJZCwgaW5mbywgZXhwbGFuYXRpb24pLFxuICAgICAgKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogQ2hlY2sgU2VjcmV0cyBNYW5hZ2VyIFJlc291cmNlc1xuICAgKiBAcGFyYW0gbm9kZSB0aGUgSUNvbnN0cnVjdCB0byBldmFsdWF0ZVxuICAgKiBAcGFyYW0gaWdub3JlcyBsaXN0IG9mIGlnbm9yZXMgZm9yIHRoZSByZXNvdXJjZVxuICAgKi9cbiAgcHJpdmF0ZSBjaGVja1NlY3JldHNNYW5hZ2VyKG5vZGU6IENmblJlc291cmNlLCBpZ25vcmVzOiBhbnkpOiB2b2lkIHtcbiAgICBpZiAoXG4gICAgICAhdGhpcy5pZ25vcmVSdWxlKGlnbm9yZXMsICdISVBBQS5TZWN1cml0eS1TZWNyZXRzTWFuYWdlclVzaW5nS01TS2V5JykgJiZcbiAgICAgICFoaXBhYVNlY3VyaXR5U2VjcmV0c01hbmFnZXJVc2luZ0tNU0tleShub2RlKVxuICAgICkge1xuICAgICAgY29uc3QgcnVsZUlkID0gJ0hJUEFBLlNlY3VyaXR5LVNlY3JldHNNYW5hZ2VyVXNpbmdLTVNLZXknO1xuICAgICAgY29uc3QgaW5mbyA9XG4gICAgICAgICdUaGUgc2VjcmV0IGlzIG5vdCBlbmNyeXB0ZWQgd2l0aCBhIEtNUyBDdXN0b21lciBtYW5hZ2VkIGtleSAtIChDb250cm9sIElEczogMTY0LjMxMihhKSgyKShpdiksIDE2NC4zMTIoZSkoMikoaWkpKS4nO1xuICAgICAgY29uc3QgZXhwbGFuYXRpb24gPVxuICAgICAgICAnVG8gaGVscCBwcm90ZWN0IGRhdGEgYXQgcmVzdCwgZW5zdXJlIGVuY3J5cHRpb24gd2l0aCBBV1MgS2V5IE1hbmFnZW1lbnQgU2VydmljZSAoQVdTIEtNUykgaXMgZW5hYmxlZCBmb3IgQVdTIFNlY3JldHMgTWFuYWdlciBzZWNyZXRzLiBCZWNhdXNlIHNlbnNpdGl2ZSBkYXRhIGNhbiBleGlzdCBhdCByZXN0IGluIFNlY3JldHMgTWFuYWdlciBzZWNyZXRzLCBlbmFibGUgZW5jcnlwdGlvbiBhdCByZXN0IHRvIGhlbHAgcHJvdGVjdCB0aGF0IGRhdGEuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBDaGVjayBBbWF6b24gU05TIFJlc291cmNlc1xuICAgKiBAcGFyYW0gbm9kZSB0aGUgSUNvbnN0cnVjdCB0byBldmFsdWF0ZVxuICAgKiBAcGFyYW0gaWdub3JlcyBsaXN0IG9mIGlnbm9yZXMgZm9yIHRoZSByZXNvdXJjZVxuICAgKi9cbiAgcHJpdmF0ZSBjaGVja1NOUyhub2RlOiBDZm5SZXNvdXJjZSwgaWdub3JlczogYW55KTogdm9pZCB7XG4gICAgaWYgKFxuICAgICAgIXRoaXMuaWdub3JlUnVsZShpZ25vcmVzLCAnSElQQUEuU2VjdXJpdHktU05TRW5jcnlwdGVkS01TJykgJiZcbiAgICAgICFoaXBhYVNlY3VyaXR5U05TRW5jcnlwdGVkS01TKG5vZGUpXG4gICAgKSB7XG4gICAgICBjb25zdCBydWxlSWQgPSAnSElQQUEuU2VjdXJpdHktU05TRW5jcnlwdGVkS01TJztcbiAgICAgIGNvbnN0IGluZm8gPVxuICAgICAgICAnVGhlIFNOUyB0b3BpYyBkb2VzIG5vdCBoYXZlIEtNUyBlbmNyeXB0aW9uIGVuYWJsZWQgLSAoQ29udHJvbCBJRHM6IDE2NC4zMTIoYSkoMikoaXYpLCAxNjQuMzEyKGUpKDIpKGlpKSkuJztcbiAgICAgIGNvbnN0IGV4cGxhbmF0aW9uID1cbiAgICAgICAgJ0JlY2F1c2Ugc2Vuc2l0aXZlIGRhdGEgY2FuIGV4aXN0IGF0IHJlc3QgaW4gcHVibGlzaGVkIG1lc3NhZ2VzLCBlbmFibGUgZW5jcnlwdGlvbiBhdCByZXN0IHRvIGhlbHAgcHJvdGVjdCB0aGF0IGRhdGEuJztcbiAgICAgIEFubm90YXRpb25zLm9mKG5vZGUpLmFkZEVycm9yKFxuICAgICAgICB0aGlzLmNyZWF0ZU1lc3NhZ2UocnVsZUlkLCBpbmZvLCBleHBsYW5hdGlvbiksXG4gICAgICApO1xuICAgIH1cbiAgfVxuXG4gIC8vICAgLyoqXG4gIC8vICAgICogQ2hlY2sgVlBDIFJlc291cmNlc1xuICAvLyAgICAqIEBwYXJhbSBub2RlIHRoZSBJQ29uc3RydWN0IHRvIGV2YWx1YXRlXG4gIC8vICAgICogQHBhcmFtIGlnbm9yZXMgbGlzdCBvZiBpZ25vcmVzIGZvciB0aGUgcmVzb3VyY2VcbiAgLy8gICAgKi9cbiAgLy8gICBwcml2YXRlIGNoZWNrVlBDKG5vZGU6IENmblJlc291cmNlLCBpZ25vcmVzOiBhbnkpOiB2b2lkIHt9XG59XG4iXX0=