"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_elasticloadbalancing_1 = require("monocdk/aws-elasticloadbalancing");
const monocdk_1 = require("monocdk");
const nag_rules_1 = require("../../nag-rules");
/**
 * CLBs use ACM-managed certificates
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_elasticloadbalancing_1.CfnLoadBalancer) {
        //For each listener, ensure that it's utilizing an ACM SSL/HTTPS cert
        const listeners = monocdk_1.Stack.of(node).resolve(node.listeners);
        if (listeners != undefined) {
            //Iterate through listeners, checking if secured ACM certs are used
            for (const listener of listeners) {
                const resolvedListener = monocdk_1.Stack.of(node).resolve(listener);
                const listenerARN = nag_rules_1.NagRules.resolveIfPrimitive(node, resolvedListener.sslCertificateId);
                //Use the ARN to check if this is an ACM managed cert
                if (listenerARN == undefined) {
                    return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                }
                else {
                    const acmRegex = /^arn:[^:]+:acm:.+$/;
                    if (!acmRegex.test(listenerARN)) {
                        return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                    }
                }
            }
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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