"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_apigateway_1 = require("monocdk/aws-apigateway");
const monocdk_1 = require("monocdk");
const nag_rules_1 = require("../../nag-rules");
/**
 * Rest APIs have request validation enabled
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_apigateway_1.CfnRestApi) {
        const apiLogicalId = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.ref);
        let found = false;
        for (const child of monocdk_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_apigateway_1.CfnRequestValidator) {
                if (isMatchingRequestValidator(child, apiLogicalId)) {
                    found = true;
                    break;
                }
            }
        }
        if (!found) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
/**
 * Helper function to check whether a given Request Validator is associated with the given Rest API
 * @param node the CfnRequestValidator to check
 * @param apiLogicalId the Cfn Logical ID of the REST API
 * returns whether the CfnRequestValidator is associated with the given Rest API
 */
function isMatchingRequestValidator(node, apiLogicalId) {
    const resourceLogicalId = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.restApiId);
    if (resourceLogicalId === apiLogicalId) {
        return true;
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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