"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NagSuppressions = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const monocdk_1 = require("monocdk");
/**
 * Helper class with methods to add cdk-nag suppressions to cdk resources.
 *
 * @stability stable
 */
class NagSuppressions {
    /**
     * Apply cdk-nag suppressions to a Stack and optionally nested stacks.
     *
     * @param stack The Stack to apply the suppression to.
     * @param suppressions A list of suppressions to apply to the stack.
     * @param applyToNestedStacks Apply the suppressions to children stacks (default:false).
     * @stability stable
     */
    static addStackSuppressions(stack, suppressions, applyToNestedStacks = false) {
        const stacks = applyToNestedStacks
            ? stack.node.findAll().filter((x) => x instanceof monocdk_1.Stack)
            : [stack];
        stacks.forEach((s) => {
            var _b, _c;
            const newSuppressions = [];
            for (const suppression of suppressions) {
                if (suppression.reason.length >= 10) {
                    newSuppressions.push(suppression);
                }
                else {
                    throw Error(`${s.node.id}: The cdk_nag suppression for ${suppression.id} must have a reason of 10 characters or more. See https://github.com/cdklabs/cdk-nag#suppressing-a-rule for information on suppressing a rule.`);
                }
            }
            let currentSuppressions = (_c = (_b = s.templateOptions.metadata) === null || _b === void 0 ? void 0 : _b.cdk_nag) === null || _c === void 0 ? void 0 : _c.rules_to_suppress;
            currentSuppressions = Array.isArray(currentSuppressions)
                ? currentSuppressions
                : [];
            currentSuppressions.push(...newSuppressions);
            const dedupSuppressions = new Set();
            const result = currentSuppressions.filter((x) => !dedupSuppressions.has(JSON.stringify(x))
                ? dedupSuppressions.add(JSON.stringify(x))
                : false);
            if (s.templateOptions.metadata) {
                s.templateOptions.metadata.cdk_nag = {
                    rules_to_suppress: result,
                };
            }
            else {
                s.templateOptions.metadata = {
                    cdk_nag: { rules_to_suppress: result },
                };
            }
        });
    }
    /**
     * Add cdk-nag suppressions to a CfnResource and optionally its children.
     *
     * @param construct The IConstruct to apply the suppression to.
     * @param suppressions A list of suppressions to apply to the resource.
     * @param applyToChildren Apply the suppressions to children CfnResources  (default:false).
     * @stability stable
     */
    static addResourceSuppressions(construct, suppressions, applyToChildren = false) {
        var _b;
        const newSuppressions = [];
        for (const suppression of suppressions) {
            if (suppression.reason.length >= 10) {
                newSuppressions.push(suppression);
            }
            else {
                throw Error(`${construct.node.id}: The cdk_nag suppression for ${suppression.id} must have a reason of 10 characters or more. See https://github.com/cdklabs/cdk-nag#suppressing-a-rule for information on suppressing a rule.`);
            }
        }
        const constructs = applyToChildren ? construct.node.findAll() : [construct];
        for (const child of constructs) {
            const possibleL1 = child.node.defaultChild
                ? child.node.defaultChild
                : child;
            if (possibleL1 instanceof monocdk_1.CfnResource) {
                const resource = possibleL1;
                let currentSuppressions = (_b = resource.getMetadata('cdk_nag')) === null || _b === void 0 ? void 0 : _b.rules_to_suppress;
                currentSuppressions = Array.isArray(currentSuppressions)
                    ? currentSuppressions
                    : [];
                currentSuppressions.push(...newSuppressions);
                const dedupSuppressions = new Set();
                const result = currentSuppressions.filter((s) => !dedupSuppressions.has(JSON.stringify(s))
                    ? dedupSuppressions.add(JSON.stringify(s))
                    : false);
                resource.addMetadata('cdk_nag', {
                    rules_to_suppress: result,
                });
            }
        }
    }
    /**
     * Add cdk-nag suppressions to a CfnResource and optionally its children via its path.
     *
     * @param stack The Stack the construct belongs to.
     * @param path The path to the construct in the provided stack.
     * @param suppressions A list of suppressions to apply to the resource.
     * @param applyToChildren Apply the suppressions to children CfnResources  (default:false).
     * @stability stable
     */
    static addResourceSuppressionsByPath(stack, path, suppressions, applyToChildren = false) {
        for (const child of stack.node.findAll()) {
            const fixedPath = path.replace(/^\//, '');
            if (child.node.path === fixedPath ||
                child.node.path + '/Resource' === fixedPath) {
                NagSuppressions.addResourceSuppressions(child, suppressions, applyToChildren);
            }
        }
    }
}
exports.NagSuppressions = NagSuppressions;
_a = JSII_RTTI_SYMBOL_1;
NagSuppressions[_a] = { fqn: "monocdk-nag.NagSuppressions", version: "1.4.19" };
//# sourceMappingURL=data:application/json;base64,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