import * as ec2 from '@aws-cdk/aws-ec2';
import * as secretsmanager from '@aws-cdk/aws-secretsmanager';
import { IResource } from '@aws-cdk/core';
import { IClusterEngine } from './cluster-engine';
import { Endpoint } from './endpoint';
import { DatabaseProxy, DatabaseProxyOptions } from './proxy';
/**
 * Create a clustered database with a given number of instances.
 *
 * @stability stable
 */
export interface IDatabaseCluster extends IResource, ec2.IConnectable, secretsmanager.ISecretAttachmentTarget {
    /**
     * Identifier of the cluster.
     *
     * @stability stable
     */
    readonly clusterIdentifier: string;
    /**
     * Identifiers of the replicas.
     *
     * @stability stable
     */
    readonly instanceIdentifiers: string[];
    /**
     * The endpoint to use for read/write operations.
     *
     * @stability stable
     * @attribute EndpointAddress,EndpointPort
     */
    readonly clusterEndpoint: Endpoint;
    /**
     * Endpoint to use for load-balanced read-only operations.
     *
     * @stability stable
     * @attribute ReadEndpointAddress
     */
    readonly clusterReadEndpoint: Endpoint;
    /**
     * Endpoints which address each individual replica.
     *
     * @stability stable
     */
    readonly instanceEndpoints: Endpoint[];
    /**
     * The engine of this Cluster.
     *
     * May be not known for imported Clusters if it wasn't provided explicitly.
     *
     * @stability stable
     */
    readonly engine?: IClusterEngine;
    /**
     * Add a new db proxy to this cluster.
     *
     * @stability stable
     */
    addProxy(id: string, options: DatabaseProxyOptions): DatabaseProxy;
}
/**
 * Properties that describe an existing cluster instance.
 *
 * @stability stable
 */
export interface DatabaseClusterAttributes {
    /**
     * Identifier for the cluster.
     *
     * @stability stable
     */
    readonly clusterIdentifier: string;
    /**
     * The database port.
     *
     * @default - none
     * @stability stable
     */
    readonly port?: number;
    /**
     * The security groups of the database cluster.
     *
     * @default - no security groups
     * @stability stable
     */
    readonly securityGroups?: ec2.ISecurityGroup[];
    /**
     * Identifier for the instances.
     *
     * @default - no instance identifiers
     * @stability stable
     */
    readonly instanceIdentifiers?: string[];
    /**
     * Cluster endpoint address.
     *
     * @default - no endpoint address
     * @stability stable
     */
    readonly clusterEndpointAddress?: string;
    /**
     * Reader endpoint address.
     *
     * @default - no reader address
     * @stability stable
     */
    readonly readerEndpointAddress?: string;
    /**
     * Endpoint addresses of individual instances.
     *
     * @default - no instance endpoints
     * @stability stable
     */
    readonly instanceEndpointAddresses?: string[];
    /**
     * The engine of the existing Cluster.
     *
     * @default - the imported Cluster's engine is unknown
     * @stability stable
     */
    readonly engine?: IClusterEngine;
}
