"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseClusterEngine = exports.AuroraPostgresEngineVersion = exports.AuroraMysqlEngineVersion = exports.AuroraEngineVersion = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const parameter_group_1 = require("./parameter-group");
class ClusterEngineBase {
    constructor(props) {
        this.engineType = props.engineType;
        this.features = props.features;
        this.singleUserRotationApplication = props.singleUserRotationApplication;
        this.multiUserRotationApplication = props.multiUserRotationApplication;
        this.defaultPort = props.defaultPort;
        this.engineVersion = props.engineVersion;
        this.parameterGroupFamily = this.engineVersion ? `${this.engineType}${this.engineVersion.majorVersion}` : undefined;
    }
    bindToCluster(scope, options) {
        var _e;
        const parameterGroup = (_e = options.parameterGroup) !== null && _e !== void 0 ? _e : this.defaultParameterGroup(scope);
        return {
            parameterGroup,
            port: this.defaultPort,
            features: this.features,
        };
    }
}
class MySqlClusterEngineBase extends ClusterEngineBase {
    constructor(props) {
        super({
            ...props,
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_MULTI_USER,
            engineVersion: props.engineVersion ? props.engineVersion : { majorVersion: props.defaultMajorVersion },
        });
        this.engineFamily = 'MYSQL';
        this.supportedLogTypes = ['error', 'general', 'slowquery', 'audit'];
    }
    bindToCluster(scope, options) {
        var _e;
        const config = super.bindToCluster(scope, options);
        const parameterGroup = (_e = options.parameterGroup) !== null && _e !== void 0 ? _e : (options.s3ImportRole || options.s3ExportRole
            ? new parameter_group_1.ParameterGroup(scope, 'ClusterParameterGroup', {
                engine: this,
            })
            : config.parameterGroup);
        if (options.s3ImportRole) {
            parameterGroup === null || parameterGroup === void 0 ? void 0 : parameterGroup.addParameter('aurora_load_from_s3_role', options.s3ImportRole.roleArn);
        }
        if (options.s3ExportRole) {
            parameterGroup === null || parameterGroup === void 0 ? void 0 : parameterGroup.addParameter('aurora_select_into_s3_role', options.s3ExportRole.roleArn);
        }
        return {
            ...config,
            parameterGroup,
        };
    }
}
/**
 * The versions for the Aurora cluster engine (those returned by {@link DatabaseClusterEngine.aurora}).
 *
 * @stability stable
 */
class AuroraEngineVersion {
    constructor(auroraFullVersion, auroraMajorVersion = '5.6') {
        this.auroraFullVersion = auroraFullVersion;
        this.auroraMajorVersion = auroraMajorVersion;
    }
    /**
     * Create a new AuroraEngineVersion with an arbitrary version.
     *
     * @param auroraFullVersion the full version string, for example "5.6.mysql_aurora.1.78.3.6".
     * @param auroraMajorVersion the major version of the engine, defaults to "5.6".
     * @stability stable
     */
    static of(auroraFullVersion, auroraMajorVersion) {
        return new AuroraEngineVersion(auroraFullVersion, auroraMajorVersion);
    }
    static builtIn_5_6(minorVersion, addStandardPrefix = true) {
        return new AuroraEngineVersion(`5.6.${addStandardPrefix ? 'mysql_aurora.' : ''}${minorVersion}`);
    }
}
exports.AuroraEngineVersion = AuroraEngineVersion;
_a = JSII_RTTI_SYMBOL_1;
AuroraEngineVersion[_a] = { fqn: "@aws-cdk/aws-rds.AuroraEngineVersion", version: "1.141.0" };
/**
 * Version "5.6.10a".
 *
 * @stability stable
 */
AuroraEngineVersion.VER_10A = AuroraEngineVersion.builtIn_5_6('10a', false);
/**
 * Version "5.6.mysql_aurora.1.17.9".
 *
 * @stability stable
 */
AuroraEngineVersion.VER_1_17_9 = AuroraEngineVersion.builtIn_5_6('1.17.9');
/**
 * Version "5.6.mysql_aurora.1.19.0".
 *
 * @stability stable
 */
AuroraEngineVersion.VER_1_19_0 = AuroraEngineVersion.builtIn_5_6('1.19.0');
/**
 * Version "5.6.mysql_aurora.1.19.1".
 *
 * @stability stable
 */
AuroraEngineVersion.VER_1_19_1 = AuroraEngineVersion.builtIn_5_6('1.19.1');
/**
 * Version "5.6.mysql_aurora.1.19.2".
 *
 * @stability stable
 */
AuroraEngineVersion.VER_1_19_2 = AuroraEngineVersion.builtIn_5_6('1.19.2');
/**
 * Version "5.6.mysql_aurora.1.19.5".
 *
 * @stability stable
 */
AuroraEngineVersion.VER_1_19_5 = AuroraEngineVersion.builtIn_5_6('1.19.5');
/**
 * Version "5.6.mysql_aurora.1.19.6".
 *
 * @stability stable
 */
AuroraEngineVersion.VER_1_19_6 = AuroraEngineVersion.builtIn_5_6('1.19.6');
/**
 * Version "5.6.mysql_aurora.1.20.0".
 *
 * @stability stable
 */
AuroraEngineVersion.VER_1_20_0 = AuroraEngineVersion.builtIn_5_6('1.20.0');
/**
 * Version "5.6.mysql_aurora.1.20.1".
 *
 * @stability stable
 */
AuroraEngineVersion.VER_1_20_1 = AuroraEngineVersion.builtIn_5_6('1.20.1');
/**
 * Version "5.6.mysql_aurora.1.21.0".
 *
 * @stability stable
 */
AuroraEngineVersion.VER_1_21_0 = AuroraEngineVersion.builtIn_5_6('1.21.0');
/**
 * Version "5.6.mysql_aurora.1.22.0".
 *
 * @stability stable
 */
AuroraEngineVersion.VER_1_22_0 = AuroraEngineVersion.builtIn_5_6('1.22.0');
/**
 * Version "5.6.mysql_aurora.1.22.1".
 *
 * @stability stable
 */
AuroraEngineVersion.VER_1_22_1 = AuroraEngineVersion.builtIn_5_6('1.22.1');
/**
 * Version "5.6.mysql_aurora.1.22.1.3".
 *
 * @stability stable
 */
AuroraEngineVersion.VER_1_22_1_3 = AuroraEngineVersion.builtIn_5_6('1.22.1.3');
/**
 * Version "5.6.mysql_aurora.1.22.2".
 *
 * @stability stable
 */
AuroraEngineVersion.VER_1_22_2 = AuroraEngineVersion.builtIn_5_6('1.22.2');
class AuroraClusterEngine extends MySqlClusterEngineBase {
    constructor(version) {
        super({
            engineType: 'aurora',
            engineVersion: version
                ? {
                    fullVersion: version.auroraFullVersion,
                    majorVersion: version.auroraMajorVersion,
                }
                : undefined,
            defaultMajorVersion: '5.6',
        });
    }
    defaultParameterGroup(_scope) {
        // the default.aurora5.6 ParameterGroup is actually the default,
        // so just return undefined in this case
        return undefined;
    }
}
/**
 * The versions for the Aurora MySQL cluster engine (those returned by {@link DatabaseClusterEngine.auroraMysql}).
 *
 * @stability stable
 */
class AuroraMysqlEngineVersion {
    constructor(auroraMysqlFullVersion, auroraMysqlMajorVersion = '5.7') {
        this.auroraMysqlFullVersion = auroraMysqlFullVersion;
        this.auroraMysqlMajorVersion = auroraMysqlMajorVersion;
    }
    /**
     * Create a new AuroraMysqlEngineVersion with an arbitrary version.
     *
     * @param auroraMysqlFullVersion the full version string, for example "5.7.mysql_aurora.2.78.3.6".
     * @param auroraMysqlMajorVersion the major version of the engine, defaults to "5.7".
     * @stability stable
     */
    static of(auroraMysqlFullVersion, auroraMysqlMajorVersion) {
        return new AuroraMysqlEngineVersion(auroraMysqlFullVersion, auroraMysqlMajorVersion);
    }
    static builtIn_5_7(minorVersion, addStandardPrefix = true) {
        return new AuroraMysqlEngineVersion(`5.7.${addStandardPrefix ? 'mysql_aurora.' : ''}${minorVersion}`);
    }
    static builtIn_8_0(minorVersion) {
        return new AuroraMysqlEngineVersion(`8.0.mysql_aurora.${minorVersion}`, '8.0');
    }
}
exports.AuroraMysqlEngineVersion = AuroraMysqlEngineVersion;
_b = JSII_RTTI_SYMBOL_1;
AuroraMysqlEngineVersion[_b] = { fqn: "@aws-cdk/aws-rds.AuroraMysqlEngineVersion", version: "1.141.0" };
/**
 * Version "5.7.12".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_5_7_12 = AuroraMysqlEngineVersion.builtIn_5_7('12', false);
/**
 * Version "5.7.mysql_aurora.2.03.2".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_03_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.03.2');
/**
 * Version "5.7.mysql_aurora.2.03.3".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_03_3 = AuroraMysqlEngineVersion.builtIn_5_7('2.03.3');
/**
 * Version "5.7.mysql_aurora.2.03.4".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_03_4 = AuroraMysqlEngineVersion.builtIn_5_7('2.03.4');
/**
 * Version "5.7.mysql_aurora.2.04.0".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_04_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.0');
/**
 * Version "5.7.mysql_aurora.2.04.1".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_04_1 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.1');
/**
 * Version "5.7.mysql_aurora.2.04.2".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_04_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.2');
/**
 * Version "5.7.mysql_aurora.2.04.3".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_04_3 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.3');
/**
 * Version "5.7.mysql_aurora.2.04.4".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_04_4 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.4');
/**
 * Version "5.7.mysql_aurora.2.04.5".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_04_5 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.5');
/**
 * Version "5.7.mysql_aurora.2.04.6".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_04_6 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.6');
/**
 * Version "5.7.mysql_aurora.2.04.7".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_04_7 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.7');
/**
 * Version "5.7.mysql_aurora.2.04.8".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_04_8 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.8');
/**
 * Version "5.7.mysql_aurora.2.05.0".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_05_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.05.0');
/**
 * Version "5.7.mysql_aurora.2.06.0".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_06_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.06.0');
/**
 * Version "5.7.mysql_aurora.2.07.0".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_07_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.07.0');
/**
 * Version "5.7.mysql_aurora.2.07.1".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_07_1 = AuroraMysqlEngineVersion.builtIn_5_7('2.07.1');
/**
 * Version "5.7.mysql_aurora.2.07.2".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_07_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.07.2');
/**
 * Version "5.7.mysql_aurora.2.08.0".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_08_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.08.0');
/**
 * Version "5.7.mysql_aurora.2.08.1".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_08_1 = AuroraMysqlEngineVersion.builtIn_5_7('2.08.1');
/**
 * Version "5.7.mysql_aurora.2.08.2".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_08_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.08.2');
/**
 * Version "5.7.mysql_aurora.2.09.0".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_09_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.09.0');
/**
 * Version "5.7.mysql_aurora.2.09.1".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_09_1 = AuroraMysqlEngineVersion.builtIn_5_7('2.09.1');
/**
 * Version "5.7.mysql_aurora.2.09.2".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_09_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.09.2');
/**
 * Version "5.7.mysql_aurora.2.09.3".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_09_3 = AuroraMysqlEngineVersion.builtIn_5_7('2.09.3');
/**
 * Version "5.7.mysql_aurora.2.10.0".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_10_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.10.0');
/**
 * Version "5.7.mysql_aurora.2.10.1".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_2_10_1 = AuroraMysqlEngineVersion.builtIn_5_7('2.10.1');
/**
 * Version "8.0.mysql_aurora.3.01.0".
 *
 * @stability stable
 */
AuroraMysqlEngineVersion.VER_3_01_0 = AuroraMysqlEngineVersion.builtIn_8_0('3.01.0');
class AuroraMysqlClusterEngine extends MySqlClusterEngineBase {
    constructor(version) {
        super({
            engineType: 'aurora-mysql',
            engineVersion: version
                ? {
                    fullVersion: version.auroraMysqlFullVersion,
                    majorVersion: version.auroraMysqlMajorVersion,
                }
                : undefined,
            defaultMajorVersion: '5.7',
        });
    }
    defaultParameterGroup(scope) {
        return parameter_group_1.ParameterGroup.fromParameterGroupName(scope, 'AuroraMySqlDatabaseClusterEngineDefaultParameterGroup', `default.${this.parameterGroupFamily}`);
    }
}
/**
 * The versions for the Aurora PostgreSQL cluster engine (those returned by {@link DatabaseClusterEngine.auroraPostgres}).
 *
 * @stability stable
 */
class AuroraPostgresEngineVersion {
    constructor(auroraPostgresFullVersion, auroraPostgresMajorVersion, auroraPostgresFeatures) {
        this.auroraPostgresFullVersion = auroraPostgresFullVersion;
        this.auroraPostgresMajorVersion = auroraPostgresMajorVersion;
        this._features = {
            s3Import: (auroraPostgresFeatures === null || auroraPostgresFeatures === void 0 ? void 0 : auroraPostgresFeatures.s3Import) ? 's3Import' : undefined,
            s3Export: (auroraPostgresFeatures === null || auroraPostgresFeatures === void 0 ? void 0 : auroraPostgresFeatures.s3Export) ? 's3Export' : undefined,
        };
    }
    /**
     * Create a new AuroraPostgresEngineVersion with an arbitrary version.
     *
     * @param auroraPostgresFullVersion the full version string, for example "9.6.25.1".
     * @param auroraPostgresMajorVersion the major version of the engine, for example "9.6".
     * @stability stable
     */
    static of(auroraPostgresFullVersion, auroraPostgresMajorVersion, auroraPostgresFeatures) {
        jsiiDeprecationWarnings._aws_cdk_aws_rds_AuroraPostgresEngineFeatures(auroraPostgresFeatures);
        return new AuroraPostgresEngineVersion(auroraPostgresFullVersion, auroraPostgresMajorVersion, auroraPostgresFeatures);
    }
}
exports.AuroraPostgresEngineVersion = AuroraPostgresEngineVersion;
_c = JSII_RTTI_SYMBOL_1;
AuroraPostgresEngineVersion[_c] = { fqn: "@aws-cdk/aws-rds.AuroraPostgresEngineVersion", version: "1.141.0" };
/**
 * Version "9.6.8".
 *
 * @stability stable
 */
AuroraPostgresEngineVersion.VER_9_6_8 = AuroraPostgresEngineVersion.of('9.6.8', '9.6');
/**
 * Version "9.6.9".
 *
 * @stability stable
 */
AuroraPostgresEngineVersion.VER_9_6_9 = AuroraPostgresEngineVersion.of('9.6.9', '9.6');
/**
 * Version "9.6.11".
 *
 * @stability stable
 */
AuroraPostgresEngineVersion.VER_9_6_11 = AuroraPostgresEngineVersion.of('9.6.11', '9.6');
/**
 * Version "9.6.12".
 *
 * @stability stable
 */
AuroraPostgresEngineVersion.VER_9_6_12 = AuroraPostgresEngineVersion.of('9.6.12', '9.6');
/**
 * Version "9.6.16".
 *
 * @stability stable
 */
AuroraPostgresEngineVersion.VER_9_6_16 = AuroraPostgresEngineVersion.of('9.6.16', '9.6');
/**
 * Version "9.6.17".
 *
 * @stability stable
 */
AuroraPostgresEngineVersion.VER_9_6_17 = AuroraPostgresEngineVersion.of('9.6.17', '9.6');
/**
 * Version "9.6.18".
 *
 * @stability stable
 */
AuroraPostgresEngineVersion.VER_9_6_18 = AuroraPostgresEngineVersion.of('9.6.18', '9.6');
/**
 * Version "9.6.19".
 *
 * @stability stable
 */
AuroraPostgresEngineVersion.VER_9_6_19 = AuroraPostgresEngineVersion.of('9.6.19', '9.6');
/**
 * Version "10.4".
 *
 * @stability stable
 */
AuroraPostgresEngineVersion.VER_10_4 = AuroraPostgresEngineVersion.of('10.4', '10');
/**
 * Version "10.5".
 *
 * @stability stable
 */
AuroraPostgresEngineVersion.VER_10_5 = AuroraPostgresEngineVersion.of('10.5', '10');
/**
 * Version "10.6".
 *
 * @stability stable
 */
AuroraPostgresEngineVersion.VER_10_6 = AuroraPostgresEngineVersion.of('10.6', '10');
/**
 * Version "10.7".
 *
 * @stability stable
 */
AuroraPostgresEngineVersion.VER_10_7 = AuroraPostgresEngineVersion.of('10.7', '10', { s3Import: true });
/**
 * Version "10.11".
 *
 * @stability stable
 */
AuroraPostgresEngineVersion.VER_10_11 = AuroraPostgresEngineVersion.of('10.11', '10', { s3Import: true, s3Export: true });
/**
 * Version "10.12".
 *
 * @stability stable
 */
AuroraPostgresEngineVersion.VER_10_12 = AuroraPostgresEngineVersion.of('10.12', '10', { s3Import: true, s3Export: true });
/**
 * Version "10.13".
 *
 * @stability stable
 */
AuroraPostgresEngineVersion.VER_10_13 = AuroraPostgresEngineVersion.of('10.13', '10', { s3Import: true, s3Export: true });
/**
 * Version "10.14".
 *
 * @stability stable
 */
AuroraPostgresEngineVersion.VER_10_14 = AuroraPostgresEngineVersion.of('10.14', '10', { s3Import: true, s3Export: true });
/**
 * Version "10.16".
 *
 * @stability stable
 */
AuroraPostgresEngineVersion.VER_10_16 = AuroraPostgresEngineVersion.of('10.16', '10', { s3Import: true, s3Export: true });
/**
 * Version "10.18".
 *
 * @stability stable
 */
AuroraPostgresEngineVersion.VER_10_18 = AuroraPostgresEngineVersion.of('10.18', '10', { s3Import: true, s3Export: true });
/**
 * Version "11.4".
 *
 * @stability stable
 */
AuroraPostgresEngineVersion.VER_11_4 = AuroraPostgresEngineVersion.of('11.4', '11', { s3Import: true });
/**
 * Version "11.6".
 *
 * @stability stable
 */
AuroraPostgresEngineVersion.VER_11_6 = AuroraPostgresEngineVersion.of('11.6', '11', { s3Import: true, s3Export: true });
/**
 * Version "11.7".
 *
 * @stability stable
 */
AuroraPostgresEngineVersion.VER_11_7 = AuroraPostgresEngineVersion.of('11.7', '11', { s3Import: true, s3Export: true });
/**
 * Version "11.8".
 *
 * @stability stable
 */
AuroraPostgresEngineVersion.VER_11_8 = AuroraPostgresEngineVersion.of('11.8', '11', { s3Import: true, s3Export: true });
/**
 * Version "11.9".
 *
 * @stability stable
 */
AuroraPostgresEngineVersion.VER_11_9 = AuroraPostgresEngineVersion.of('11.9', '11', { s3Import: true, s3Export: true });
/**
 * Version "11.11".
 *
 * @stability stable
 */
AuroraPostgresEngineVersion.VER_11_11 = AuroraPostgresEngineVersion.of('11.11', '11', { s3Import: true, s3Export: true });
/**
 * Version "11.13".
 *
 * @stability stable
 */
AuroraPostgresEngineVersion.VER_11_13 = AuroraPostgresEngineVersion.of('11.13', '11', { s3Import: true, s3Export: true });
/**
 * Version "12.4".
 *
 * @stability stable
 */
AuroraPostgresEngineVersion.VER_12_4 = AuroraPostgresEngineVersion.of('12.4', '12', { s3Import: true, s3Export: true });
/**
 * Version "12.6".
 *
 * @stability stable
 */
AuroraPostgresEngineVersion.VER_12_6 = AuroraPostgresEngineVersion.of('12.6', '12', { s3Import: true, s3Export: true });
/**
 * Version "12.8".
 *
 * @stability stable
 */
AuroraPostgresEngineVersion.VER_12_8 = AuroraPostgresEngineVersion.of('12.8', '12', { s3Import: true, s3Export: true });
/**
 * Version "13.3".
 *
 * @stability stable
 */
AuroraPostgresEngineVersion.VER_13_3 = AuroraPostgresEngineVersion.of('13.3', '13', { s3Import: true, s3Export: true });
/**
 * Version "13.4".
 *
 * @stability stable
 */
AuroraPostgresEngineVersion.VER_13_4 = AuroraPostgresEngineVersion.of('13.4', '13', { s3Import: true, s3Export: true });
class AuroraPostgresClusterEngine extends ClusterEngineBase {
    constructor(version) {
        super({
            engineType: 'aurora-postgresql',
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_MULTI_USER,
            defaultPort: 5432,
            engineVersion: version
                ? {
                    fullVersion: version.auroraPostgresFullVersion,
                    majorVersion: version.auroraPostgresMajorVersion,
                }
                : undefined,
            features: version
                ? {
                    s3Import: version._features.s3Import ? AuroraPostgresClusterEngine.S3_IMPORT_FEATURE_NAME : undefined,
                    s3Export: version._features.s3Export ? AuroraPostgresClusterEngine.S3_EXPORT_FEATURE_NAME : undefined,
                }
                : {
                    s3Import: AuroraPostgresClusterEngine.S3_IMPORT_FEATURE_NAME,
                    s3Export: AuroraPostgresClusterEngine.S3_EXPORT_FEATURE_NAME,
                },
        });
        this.engineFamily = 'POSTGRESQL';
        this.defaultUsername = 'postgres';
        this.supportedLogTypes = ['postgresql'];
    }
    bindToCluster(scope, options) {
        var _e, _f, _g;
        const config = super.bindToCluster(scope, options);
        // skip validation for unversioned as it might be supported/unsupported. we cannot reliably tell at compile-time
        if ((_e = this.engineVersion) === null || _e === void 0 ? void 0 : _e.fullVersion) {
            if (options.s3ImportRole && !((_f = config.features) === null || _f === void 0 ? void 0 : _f.s3Import)) {
                throw new Error(`s3Import is not supported for Postgres version: ${this.engineVersion.fullVersion}. Use a version that supports the s3Import feature.`);
            }
            if (options.s3ExportRole && !((_g = config.features) === null || _g === void 0 ? void 0 : _g.s3Export)) {
                throw new Error(`s3Export is not supported for Postgres version: ${this.engineVersion.fullVersion}. Use a version that supports the s3Export feature.`);
            }
        }
        return config;
    }
    defaultParameterGroup(scope) {
        if (!this.parameterGroupFamily) {
            throw new Error('Could not create a new ParameterGroup for an unversioned aurora-postgresql cluster engine. ' +
                'Please either use a versioned engine, or pass an explicit ParameterGroup when creating the cluster');
        }
        return parameter_group_1.ParameterGroup.fromParameterGroupName(scope, 'AuroraPostgreSqlDatabaseClusterEngineDefaultParameterGroup', `default.${this.parameterGroupFamily}`);
    }
}
/**
 * feature name for the S3 data import feature
 */
AuroraPostgresClusterEngine.S3_IMPORT_FEATURE_NAME = 's3Import';
/**
 * feature name for the S3 data export feature
 */
AuroraPostgresClusterEngine.S3_EXPORT_FEATURE_NAME = 's3Export';
/**
 * A database cluster engine.
 *
 * Provides mapping to the serverless application
 * used for secret rotation.
 *
 * @stability stable
 */
class DatabaseClusterEngine {
    /**
     * Creates a new plain Aurora database cluster engine.
     *
     * @stability stable
     */
    static aurora(props) {
        jsiiDeprecationWarnings._aws_cdk_aws_rds_AuroraClusterEngineProps(props);
        return new AuroraClusterEngine(props.version);
    }
    /**
     * Creates a new Aurora MySQL database cluster engine.
     *
     * @stability stable
     */
    static auroraMysql(props) {
        jsiiDeprecationWarnings._aws_cdk_aws_rds_AuroraMysqlClusterEngineProps(props);
        return new AuroraMysqlClusterEngine(props.version);
    }
    /**
     * Creates a new Aurora PostgreSQL database cluster engine.
     *
     * @stability stable
     */
    static auroraPostgres(props) {
        jsiiDeprecationWarnings._aws_cdk_aws_rds_AuroraPostgresClusterEngineProps(props);
        return new AuroraPostgresClusterEngine(props.version);
    }
}
exports.DatabaseClusterEngine = DatabaseClusterEngine;
_d = JSII_RTTI_SYMBOL_1;
DatabaseClusterEngine[_d] = { fqn: "@aws-cdk/aws-rds.DatabaseClusterEngine", version: "1.141.0" };
/**
 * The unversioned 'aurora' cluster engine.
 *
 * **Note**: we do not recommend using unversioned engines for non-serverless Clusters,
 *    as that can pose an availability risk.
 *    We recommend using versioned engines created using the {@link aurora()} method
 *
 * @stability stable
 */
DatabaseClusterEngine.AURORA = new AuroraClusterEngine();
/**
 * The unversioned 'aurora-msql' cluster engine.
 *
 * **Note**: we do not recommend using unversioned engines for non-serverless Clusters,
 *    as that can pose an availability risk.
 *    We recommend using versioned engines created using the {@link auroraMysql()} method
 *
 * @stability stable
 */
DatabaseClusterEngine.AURORA_MYSQL = new AuroraMysqlClusterEngine();
/**
 * The unversioned 'aurora-postgresql' cluster engine.
 *
 * **Note**: we do not recommend using unversioned engines for non-serverless Clusters,
 *    as that can pose an availability risk.
 *    We recommend using versioned engines created using the {@link auroraPostgres()} method
 *
 * @stability stable
 */
DatabaseClusterEngine.AURORA_POSTGRESQL = new AuroraPostgresClusterEngine();
//# sourceMappingURL=data:application/json;base64,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