"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ValidationMethod = exports.Certificate = exports.CertificateValidation = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const certificatemanager_generated_1 = require("./certificatemanager.generated");
const util_1 = require("./util");
/**
 * How to validate a certificate.
 *
 * @stability stable
 */
class CertificateValidation {
    /** @param props Certification validation properties */
    constructor(props) {
        var _c;
        this.props = props;
        this.method = (_c = props.method) !== null && _c !== void 0 ? _c : ValidationMethod.EMAIL;
    }
    /**
     * Validate the certifcate with DNS.
     *
     * IMPORTANT: If `hostedZone` is not specified, DNS records must be added
     * manually and the stack will not complete creating until the records are
     * added.
     *
     * @param hostedZone the hosted zone where DNS records must be created.
     * @stability stable
     */
    static fromDns(hostedZone) {
        return new CertificateValidation({
            method: ValidationMethod.DNS,
            hostedZone,
        });
    }
    /**
     * Validate the certifcate with automatically created DNS records in multiple Amazon Route 53 hosted zones.
     *
     * @param hostedZones a map of hosted zones where DNS records must be created for the domains in the certificate.
     * @stability stable
     */
    static fromDnsMultiZone(hostedZones) {
        return new CertificateValidation({
            method: ValidationMethod.DNS,
            hostedZones,
        });
    }
    /**
     * Validate the certifcate with Email.
     *
     * IMPORTANT: if you are creating a certificate as part of your stack, the stack
     * will not complete creating until you read and follow the instructions in the
     * email that you will receive.
     *
     * ACM will send validation emails to the following addresses:
     *
     *   admin@domain.com
     *   administrator@domain.com
     *   hostmaster@domain.com
     *   postmaster@domain.com
     *   webmaster@domain.com
     *
     * For every domain that you register.
     *
     * @param validationDomains a map of validation domains to use for domains in the certificate.
     * @stability stable
     */
    static fromEmail(validationDomains) {
        return new CertificateValidation({
            method: ValidationMethod.EMAIL,
            validationDomains,
        });
    }
}
exports.CertificateValidation = CertificateValidation;
_a = JSII_RTTI_SYMBOL_1;
CertificateValidation[_a] = { fqn: "@aws-cdk/aws-certificatemanager.CertificateValidation", version: "1.91.0" };
/**
 * A certificate managed by AWS Certificate Manager.
 *
 * @stability stable
 */
class Certificate extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        let validation;
        if (props.validation) {
            validation = props.validation;
        }
        else { // Deprecated props
            if (props.validationMethod === ValidationMethod.DNS) {
                validation = CertificateValidation.fromDns();
            }
            else {
                validation = CertificateValidation.fromEmail(props.validationDomains);
            }
        }
        const allDomainNames = [props.domainName].concat(props.subjectAlternativeNames || []);
        const cert = new certificatemanager_generated_1.CfnCertificate(this, 'Resource', {
            domainName: props.domainName,
            subjectAlternativeNames: props.subjectAlternativeNames,
            domainValidationOptions: renderDomainValidation(validation, allDomainNames),
            validationMethod: validation.method,
        });
        this.certificateArn = cert.ref;
    }
    /**
     * Import a certificate.
     *
     * @stability stable
     */
    static fromCertificateArn(scope, id, certificateArn) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.certificateArn = certificateArn;
            }
        }
        return new Import(scope, id);
    }
}
exports.Certificate = Certificate;
_b = JSII_RTTI_SYMBOL_1;
Certificate[_b] = { fqn: "@aws-cdk/aws-certificatemanager.Certificate", version: "1.91.0" };
/**
 * Method used to assert ownership of the domain.
 *
 * @stability stable
 */
var ValidationMethod;
(function (ValidationMethod) {
    ValidationMethod["EMAIL"] = "EMAIL";
    ValidationMethod["DNS"] = "DNS";
})(ValidationMethod = exports.ValidationMethod || (exports.ValidationMethod = {}));
// eslint-disable-next-line max-len
function renderDomainValidation(validation, domainNames) {
    var _c, _d, _e;
    const domainValidation = [];
    switch (validation.method) {
        case ValidationMethod.DNS:
            for (const domainName of getUniqueDnsDomainNames(domainNames)) {
                const hostedZone = (_d = (_c = validation.props.hostedZones) === null || _c === void 0 ? void 0 : _c[domainName]) !== null && _d !== void 0 ? _d : validation.props.hostedZone;
                if (hostedZone) {
                    domainValidation.push({ domainName, hostedZoneId: hostedZone.hostedZoneId });
                }
            }
            break;
        case ValidationMethod.EMAIL:
            for (const domainName of domainNames) {
                const validationDomain = (_e = validation.props.validationDomains) === null || _e === void 0 ? void 0 : _e[domainName];
                if (!validationDomain && core_1.Token.isUnresolved(domainName)) {
                    throw new Error('When using Tokens for domain names, \'validationDomains\' needs to be supplied');
                }
                domainValidation.push({ domainName, validationDomain: validationDomain !== null && validationDomain !== void 0 ? validationDomain : util_1.apexDomain(domainName) });
            }
            break;
        default:
            throw new Error(`Unknown validation method ${validation.method}`);
    }
    return domainValidation.length !== 0 ? domainValidation : undefined;
}
/**
 * Removes wildcard domains (*.example.com) where the base domain (example.com) is present.
 * This is because the DNS validation treats them as the same thing, and the automated CloudFormation
 * DNS validation errors out with the duplicate records.
 */
function getUniqueDnsDomainNames(domainNames) {
    return domainNames.filter(domain => {
        return core_1.Token.isUnresolved(domain) || !domain.startsWith('*.') || !domainNames.includes(domain.replace('*.', ''));
    });
}
//# sourceMappingURL=data:application/json;base64,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