"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const sns = require("@aws-cdk/aws-sns");
const core_1 = require("@aws-cdk/core");
/**
 * Use an SQS queue as a subscription target
 */
class SqsSubscription {
    constructor(queue, props = {}) {
        this.queue = queue;
        this.props = props;
    }
    /**
     * Returns a configuration for an SQS queue to subscribe to an SNS topic
     */
    bind(topic) {
        // Create subscription under *consuming* construct to make sure it ends up
        // in the correct stack in cases of cross-stack subscriptions.
        if (!core_1.Construct.isConstruct(this.queue)) {
            throw new Error('The supplied Queue object must be an instance of Construct');
        }
        // add a statement to the queue resource policy which allows this topic
        // to send messages to the queue.
        this.queue.addToResourcePolicy(new iam.PolicyStatement({
            resources: [this.queue.queueArn],
            actions: ['sqs:SendMessage'],
            principals: [new iam.ServicePrincipal('sns.amazonaws.com')],
            conditions: {
                ArnEquals: { 'aws:SourceArn': topic.topicArn },
            },
        }));
        return {
            subscriberScope: this.queue,
            subscriberId: topic.node.uniqueId,
            endpoint: this.queue.queueArn,
            protocol: sns.SubscriptionProtocol.SQS,
            rawMessageDelivery: this.props.rawMessageDelivery,
            filterPolicy: this.props.filterPolicy,
            region: this.regionFromArn(topic),
            deadLetterQueue: this.props.deadLetterQueue,
        };
    }
    regionFromArn(topic) {
        // no need to specify `region` for topics defined within the same stack
        if (topic instanceof sns.Topic) {
            return undefined;
        }
        return core_1.Stack.of(topic).parseArn(topic.topicArn).region;
    }
}
exports.SqsSubscription = SqsSubscription;
//# sourceMappingURL=data:application/json;base64,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