"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const sns = require("@aws-cdk/aws-sns");
const core_1 = require("@aws-cdk/core");
/**
 * Use a Lambda function as a subscription target
 */
class LambdaSubscription {
    constructor(fn, props = {}) {
        this.fn = fn;
        this.props = props;
    }
    /**
     * Returns a configuration for a Lambda function to subscribe to an SNS topic
     */
    bind(topic) {
        // Create subscription under *consuming* construct to make sure it ends up
        // in the correct stack in cases of cross-stack subscriptions.
        if (!core_1.Construct.isConstruct(this.fn)) {
            throw new Error('The supplied lambda Function object must be an instance of Construct');
        }
        this.fn.addPermission(`AllowInvoke:${topic.node.uniqueId}`, {
            sourceArn: topic.topicArn,
            principal: new iam.ServicePrincipal('sns.amazonaws.com'),
        });
        return {
            subscriberScope: this.fn,
            subscriberId: topic.node.id,
            endpoint: this.fn.functionArn,
            protocol: sns.SubscriptionProtocol.LAMBDA,
            filterPolicy: this.props.filterPolicy,
            region: this.regionFromArn(topic),
            deadLetterQueue: this.props.deadLetterQueue,
        };
    }
    regionFromArn(topic) {
        // no need to specify `region` for topics defined within the same stack.
        if (topic instanceof sns.Topic) {
            return undefined;
        }
        return core_1.Stack.of(topic).parseArn(topic.topicArn).region;
    }
}
exports.LambdaSubscription = LambdaSubscription;
//# sourceMappingURL=data:application/json;base64,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