"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApplicationSignalsIntegration = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const ecs = require("aws-cdk-lib/aws-ecs");
const constructs_1 = require("constructs");
const constants = require("./constants");
const agent = require("./ecs-cloudwatch-agent");
const sdk = require("./ecs-sdk-instrumentation");
const inst = require("./instrumentation-versions");
/**
 * Class for integrating Application Signals into an ECS task definition.
 */
class ApplicationSignalsIntegration extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.mountVolumeName = 'opentelemetry-auto-instrumentation';
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_applicationsignals_alpha_ApplicationSignalsIntegrationProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, ApplicationSignalsIntegration);
            }
            throw error;
        }
        this.cloudWatchAgentSidecar = props.cloudWatchAgentSidecar;
        let runtimePlatformObj = props.instrumentation.runtimePlatform ?? props.taskDefinition.runtimePlatform;
        let cpuArch = ecs.CpuArchitecture.X86_64;
        let isWindows = false;
        if (runtimePlatformObj) {
            const runtimePlatform = runtimePlatformObj;
            if (runtimePlatform.operatingSystemFamily) {
                isWindows = runtimePlatform.operatingSystemFamily.isWindows();
                if (runtimePlatform.cpuArchitecture) {
                    cpuArch = runtimePlatform.cpuArchitecture;
                }
            }
        }
        const overrideEnvironments = [];
        if (props.serviceName) {
            // If service.name is also provided in OTEL_RESOURCE_ATTRIBUTES, then OTEL_SERVICE_NAME takes precedence.
            overrideEnvironments.push({
                name: constants.CommonExporting.OTEL_SERVICE_NAME,
                value: props.serviceName,
            });
        }
        overrideEnvironments.push(...props.overrideEnvironments ?? []);
        if (props.instrumentation.sdkVersion instanceof inst.JavaInstrumentationVersion) {
            this.sdkInjector = new sdk.JavaInjector(this.mountVolumeName, props.instrumentation.sdkVersion, overrideEnvironments);
        }
        else if (props.instrumentation.sdkVersion instanceof inst.PythonInstrumentationVersion) {
            this.sdkInjector = new sdk.PythonInjector(this.mountVolumeName, props.instrumentation.sdkVersion, overrideEnvironments);
        }
        else if (props.instrumentation.sdkVersion instanceof inst.DotnetInstrumentationVersion) {
            if (isWindows) {
                this.sdkInjector = new sdk.DotNetWindowsInjector(this.mountVolumeName, props.instrumentation.sdkVersion, overrideEnvironments);
            }
            else {
                this.sdkInjector = new sdk.DotNetLinuxInjector(this.mountVolumeName, props.instrumentation.sdkVersion, cpuArch, overrideEnvironments);
            }
        }
        else if (props.instrumentation.sdkVersion instanceof inst.NodeInstrumentationVersion) {
            this.sdkInjector = new sdk.NodeInjector(this.mountVolumeName, props.instrumentation.sdkVersion, overrideEnvironments);
        }
        this.mutateTaskDefinition(props.taskDefinition);
    }
    mutateTaskDefinition(taskDefinition) {
        taskDefinition.addVolume({
            name: this.mountVolumeName,
        });
        let defaultContainer = taskDefinition.defaultContainer;
        if (this.sdkInjector) {
            this.sdkInjector.renderDefaultContainer(taskDefinition);
            let initContainer = this.sdkInjector.injectInitContainer(taskDefinition);
            defaultContainer.addContainerDependencies({
                container: initContainer,
                condition: ecs.ContainerDependencyCondition.SUCCESS,
            });
        }
        if (this.cloudWatchAgentSidecar) {
            const cloudWatchAgent = new agent.CloudWatchAgentIntegration(this, 'CloudWatchAgentSidecar', {
                taskDefinition: taskDefinition,
                ...this.cloudWatchAgentSidecar,
            });
            defaultContainer.addContainerDependencies({
                container: cloudWatchAgent.agentContainer,
                condition: ecs.ContainerDependencyCondition.START,
            });
        }
        else {
            aws_cdk_lib_1.Annotations.of(this).addWarningV2(this.node.id, ' Application Signals functionality requires prior deployment of the CloudWatch Agent with appropriate security group settings. Missing or incorrect configurations will prevent successful collection of observability data.');
        }
    }
}
exports.ApplicationSignalsIntegration = ApplicationSignalsIntegration;
_a = JSII_RTTI_SYMBOL_1;
ApplicationSignalsIntegration[_a] = { fqn: "@aws-cdk/aws-applicationsignals-alpha.ApplicationSignalsIntegration", version: "2.202.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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