"use strict";
var _a, _b, _c, _d, _e, _f, _g;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NodeInjector = exports.DotNetWindowsInjector = exports.DotNetLinuxInjector = exports.DotNetInjector = exports.PythonInjector = exports.JavaInjector = exports.Injector = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ecs = require("aws-cdk-lib/aws-ecs");
const constants = require("./constants");
/**
 * Injector is a base class for all SDK injects to mutate the task definition
 * to inject the ADOT init container and configure the application container with
 * the necessary environment variables.
 */
class Injector {
    constructor(sharedVolumeName, instrumentationVersion, overrideEnvironments) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_applicationsignals_alpha_InstrumentationVersion(instrumentationVersion);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Injector);
            }
            throw error;
        }
        this.sharedVolumeName = sharedVolumeName;
        this.instrumentationVersion = instrumentationVersion;
        this.overrideEnvironments = overrideEnvironments;
    }
    /**
     * Inject ADOT SDK agent init container.
     * @param taskDefinition The TaskDefinition to render
     * @returns The created ContainerDefinition
     */
    injectInitContainer(taskDefinition) {
        const initContainer = taskDefinition.addContainer('adot-init', {
            image: ecs.ContainerImage.fromRegistry(this.instrumentationVersion.imageURI()),
            essential: false,
            command: this.command,
            cpu: 0,
            memoryLimitMiB: this.instrumentationVersion.memoryLimitMiB(),
        });
        initContainer.addMountPoints({
            sourceVolume: this.sharedVolumeName,
            containerPath: this.containerPath,
            // double check
            readOnly: false,
        });
        return initContainer;
    }
    /**
     * Render the application container for SDK instrumentation.
     * @param taskDefinition The TaskDefinition to render
     */
    renderDefaultContainer(taskDefinition) {
        let container = taskDefinition.defaultContainer;
        const envsToInject = {};
        const envsFromTaskDef = container.environment;
        for (const env of Injector.DEFAULT_ENVS) {
            envsToInject[env.name] = env.value;
        }
        this.injectAdditionalEnvironments(envsToInject, envsFromTaskDef);
        const envsToOverride = {};
        this.overrideAdditionalEnvironments(envsToOverride, envsFromTaskDef);
        for (const [key, value] of Object.entries(envsToInject)) {
            if (!envsFromTaskDef[key]) {
                container.addEnvironment(key, value);
            }
        }
        for (const [key, value] of Object.entries(envsToOverride)) {
            container.addEnvironment(key, value);
        }
        for (const env of this.overrideEnvironments ?? []) {
            container.addEnvironment(env.name, env.value);
        }
        if (!envsFromTaskDef[constants.CommonExporting.OTEL_SERVICE_NAME]) {
            let resourceAttributesVal = (envsFromTaskDef[constants.CommonExporting.OTEL_RESOURCE_ATTRIBUTES] || '');
            if (resourceAttributesVal.indexOf('service.name') < 0) {
                // Configure service.name to be task definition family name if undefined.
                container.addEnvironment(constants.CommonExporting.OTEL_SERVICE_NAME, taskDefinition.family);
            }
        }
        container.addMountPoints({
            sourceVolume: this.sharedVolumeName,
            containerPath: this.containerPath,
            // double check
            readOnly: false,
        });
    }
}
exports.Injector = Injector;
_a = JSII_RTTI_SYMBOL_1;
Injector[_a] = { fqn: "@aws-cdk/aws-applicationsignals-alpha.Injector", version: "2.202.0-alpha.0" };
Injector.DEFAULT_ENVS = [
    {
        name: constants.LogsExporting.OTEL_LOGS_EXPORTER,
        value: constants.LogsExporting.OTEL_LOGS_EXPORTER_NONE,
    },
    {
        name: constants.MetricsExporting.OTEL_METRICS_EXPORTER,
        value: constants.MetricsExporting.OTEL_METRICS_EXPORTER_NONE,
    },
    {
        name: constants.CommonExporting.OTEL_EXPORTER_OTLP_PROTOCOL,
        value: constants.CommonExporting.OTEL_EXPORTER_OTLP_PROTOCOL_HTTP_PROTOBUF,
    },
    {
        name: constants.CommonExporting.OTEL_AWS_APPLICATION_SIGNALS,
        value: constants.CommonExporting.OTEL_AWS_APPLICATION_SIGNALS_ENABLED,
    },
    {
        name: constants.CommonExporting.OTEL_AWS_APPLICATION_SIGNALS_EXPORTER_ENDPOINT,
        value: constants.CommonExporting.OTEL_AWS_APPLICATION_SIGNALS_EXPORTER_ENDPOINT_LOCAL_CWA,
    },
    {
        name: constants.TraceExporting.OTEL_EXPORTER_OTLP_TRACES_ENDPOINT,
        value: constants.TraceExporting.OTEL_EXPORTER_OTLP_TRACES_ENDPOINT_LOCAL_CWA,
    },
    {
        name: constants.TraceExporting.OTEL_TRACES_SAMPLER,
        value: constants.TraceExporting.OTEL_TRACES_SAMPLER_XRAY,
    },
    {
        name: constants.TraceExporting.OTEL_TRACES_SAMPLER_ARG,
        value: constants.TraceExporting.OTEL_TRACES_SAMPLER_ARG_LOCAL_CWA,
    },
    {
        name: constants.TraceExporting.OTEL_PROPAGATORS,
        value: constants.TraceExporting.OTEL_PROPAGATORS_APPLICATION_SIGNALS,
    },
];
/**
 * Java-specific implementation of the SDK injector.
 * Handles Java agent configuration and environment setup.
 */
class JavaInjector extends Injector {
    get command() {
        return ['cp', '/javaagent.jar', `${this.containerPath}/javaagent.jar`];
    }
    get containerPath() {
        return '/otel-auto-instrumentation';
    }
    injectAdditionalEnvironments(envsToInject, _envsFromTaskDef) {
        envsToInject[constants.JavaInstrumentation.JAVA_TOOL_OPTIONS] = ` -javaagent:${this.containerPath}/javaagent.jar`;
    }
    overrideAdditionalEnvironments(_envsToOverride, _overrideEnvironments) {
    }
}
exports.JavaInjector = JavaInjector;
_b = JSII_RTTI_SYMBOL_1;
JavaInjector[_b] = { fqn: "@aws-cdk/aws-applicationsignals-alpha.JavaInjector", version: "2.202.0-alpha.0" };
/**
 * Python-specific implementation of the SDK injector.
 * Handles Python auto-instrumentation setup and PYTHONPATH configuration.
 */
class PythonInjector extends Injector {
    get command() {
        return ['cp', '-a', '/autoinstrumentation/.', this.containerPath];
    }
    injectAdditionalEnvironments(envsToInject, _envsFromTaskDef) {
        for (const env of PythonInjector.PYTHON_ENVS) {
            envsToInject[env.name] = env.value;
        }
        envsToInject[constants.PythonInstrumentation.PYTHONPATH] = `${this.containerPath}/opentelemetry/instrumentation/auto_instrumentation:${this.containerPath}`;
    }
    get containerPath() {
        return '/otel-auto-instrumentation-python';
    }
    overrideAdditionalEnvironments(envsToOverride, envsFromTaskDef) {
        if (envsFromTaskDef[constants.PythonInstrumentation.PYTHONPATH]) {
            const pythonPath = envsFromTaskDef[constants.PythonInstrumentation.PYTHONPATH];
            envsToOverride[constants.PythonInstrumentation.PYTHONPATH] = `${this.containerPath}/opentelemetry/instrumentation/auto_instrumentation:${pythonPath}:${this.containerPath}`;
        }
    }
}
exports.PythonInjector = PythonInjector;
_c = JSII_RTTI_SYMBOL_1;
PythonInjector[_c] = { fqn: "@aws-cdk/aws-applicationsignals-alpha.PythonInjector", version: "2.202.0-alpha.0" };
PythonInjector.PYTHON_ENVS = [
    {
        name: constants.PythonInstrumentation.OTEL_PYTHON_DISTRO,
        value: constants.PythonInstrumentation.OTEL_PYTHON_DISTRO_AWS_DISTRO,
    },
    {
        name: constants.PythonInstrumentation.OTEL_PYTHON_CONFIGURATOR,
        value: constants.PythonInstrumentation.OTEL_PYTHON_CONFIGURATOR_AWS_CONFIGURATOR,
    },
];
/**
 * Base class for .NET SDK injectors.
 * Contains common .NET configuration settings used by both Windows and Linux implementations.
 */
class DotNetInjector extends Injector {
}
exports.DotNetInjector = DotNetInjector;
_d = JSII_RTTI_SYMBOL_1;
DotNetInjector[_d] = { fqn: "@aws-cdk/aws-applicationsignals-alpha.DotNetInjector", version: "2.202.0-alpha.0" };
DotNetInjector.DOTNET_COMMON_ENVS = [
    {
        name: constants.DotnetInstrumentation.OTEL_DOTNET_DISTRO,
        value: constants.DotnetInstrumentation.OTEL_DOTNET_DISTRO_AWS_DISTRO,
    },
    {
        name: constants.DotnetInstrumentation.OTEL_DOTNET_CONFIGURATOR,
        value: constants.DotnetInstrumentation.OTEL_DOTNET_CONFIGURATOR_AWS_CONFIGURATOR,
    },
    {
        name: constants.DotnetInstrumentation.OTEL_DOTNET_AUTO_PLUGINS,
        value: constants.DotnetInstrumentation.OTEL_DOTNET_AUTO_PLUGINS_ADOT,
    },
];
/**
 * Linux-specific implementation of the .NET SDK injector.
 * Handles CoreCLR profiler setup and paths for Linux environments.
 */
class DotNetLinuxInjector extends DotNetInjector {
    constructor(sharedVolumeName, instrumentationVersion, cpuArch, overrideEnvironments) {
        super(sharedVolumeName, instrumentationVersion, overrideEnvironments);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_applicationsignals_alpha_InstrumentationVersion(instrumentationVersion);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, DotNetLinuxInjector);
            }
            throw error;
        }
        this.cpuArch = cpuArch;
    }
    get command() {
        return ['cp', '-a', '/autoinstrumentation/.', this.containerPath];
    }
    injectAdditionalEnvironments(envsToInject, envsFromTaskDef) {
        if (envsFromTaskDef[constants.DotnetInstrumentation.OTEL_DOTNET_AUTO_HOME]) {
            // If OTEL_DOTNET_AUTO_HOME env var is already set, we will assume that .NET Auto-instrumentation is already configured.
            return;
        }
        for (const env of DotNetInjector.DOTNET_COMMON_ENVS) {
            envsToInject[env.name] = env.value;
        }
        for (const env of DotNetLinuxInjector.DOTNET_LINUX_ENVS) {
            envsToInject[env.name] = env.value;
        }
        envsToInject[constants.DotnetInstrumentation.CORECLR_PROFILER_PATH] = this.getCoreCLRProfilerPath();
        envsToInject[constants.DotnetInstrumentation.DOTNET_STARTUP_HOOKS] = `${this.containerPath}/net/OpenTelemetry.AutoInstrumentation.StartupHook.dll`;
        envsToInject[constants.DotnetInstrumentation.DOTNET_ADDITIONAL_DEPS] = `${this.containerPath}/AdditionalDeps`;
        envsToInject[constants.DotnetInstrumentation.OTEL_DOTNET_AUTO_HOME] = `${this.containerPath}`;
        envsToInject[constants.DotnetInstrumentation.DOTNET_SHARED_STORE] = `${this.containerPath}/store`;
    }
    get containerPath() {
        return '/otel-auto-instrumentation-dotnet';
    }
    overrideAdditionalEnvironments(_envsToOverride, _envsFromTaskDef) {
    }
    getCoreCLRProfilerPath() {
        const subPath = this.cpuArch == ecs.CpuArchitecture.ARM64 ? 'linux-arm64' : 'linux-x64';
        return `${this.containerPath}/${subPath}/OpenTelemetry.AutoInstrumentation.Native.so`;
    }
}
exports.DotNetLinuxInjector = DotNetLinuxInjector;
_e = JSII_RTTI_SYMBOL_1;
DotNetLinuxInjector[_e] = { fqn: "@aws-cdk/aws-applicationsignals-alpha.DotNetLinuxInjector", version: "2.202.0-alpha.0" };
DotNetLinuxInjector.DOTNET_LINUX_ENVS = [
    {
        name: constants.DotnetInstrumentation.CORECLR_ENABLE_PROFILING,
        value: constants.DotnetInstrumentation.CORECLR_ENABLE_PROFILING_ENABLED,
    },
    {
        name: constants.DotnetInstrumentation.CORECLR_PROFILER,
        value: constants.DotnetInstrumentation.CORECLR_PROFILER_OTEL,
    },
];
/**
 * Windows-specific implementation of the .NET SDK injector.
 * Handles CoreCLR profiler setup and paths for Windows environments.
 */
class DotNetWindowsInjector extends DotNetInjector {
    get command() {
        return ['CMD',
            '/c',
            'xcopy',
            '/e',
            'C:\\autoinstrumentation\\*',
            'C:\\otel-auto-instrumentation-dotnet',
            '&&',
            'icacls',
            'C:\\otel-auto-instrumentation-dotnet',
            '/grant',
            '*S-1-1-0:R',
            '/T'];
    }
    injectAdditionalEnvironments(envsToInject, envsFromTaskDef) {
        if (envsFromTaskDef[constants.DotnetInstrumentation.OTEL_DOTNET_AUTO_HOME]) {
            // If OTEL_DOTNET_AUTO_HOME env var is already set, we will assume that .NET Auto-instrumentation is already configured.
            return;
        }
        for (const env of DotNetInjector.DOTNET_COMMON_ENVS) {
            envsToInject[env.name] = env.value;
        }
        for (const env of DotNetWindowsInjector.DOTNET_WINDOWS_ENVS) {
            envsToInject[env.name] = env.value;
        }
        envsToInject[constants.DotnetInstrumentation.CORECLR_PROFILER_PATH] = `${this.containerPath}\\win-x64\\OpenTelemetry.AutoInstrumentation.Native.dll`;
        envsToInject[constants.DotnetInstrumentation.DOTNET_STARTUP_HOOKS] = `${this.containerPath}\\net\\OpenTelemetry.AutoInstrumentation.StartupHook.dll`;
        envsToInject[constants.DotnetInstrumentation.DOTNET_ADDITIONAL_DEPS] = `${this.containerPath}\\AdditionalDeps`;
        envsToInject[constants.DotnetInstrumentation.OTEL_DOTNET_AUTO_HOME] = `${this.containerPath}`;
        envsToInject[constants.DotnetInstrumentation.DOTNET_SHARED_STORE] = `${this.containerPath}\\store`;
    }
    get containerPath() {
        return 'C:\\otel-auto-instrumentation-dotnet';
    }
    overrideAdditionalEnvironments(_envsToOverride, _envsFromTaskDef) {
    }
}
exports.DotNetWindowsInjector = DotNetWindowsInjector;
_f = JSII_RTTI_SYMBOL_1;
DotNetWindowsInjector[_f] = { fqn: "@aws-cdk/aws-applicationsignals-alpha.DotNetWindowsInjector", version: "2.202.0-alpha.0" };
DotNetWindowsInjector.DOTNET_WINDOWS_ENVS = [
    {
        name: constants.DotnetInstrumentation.CORECLR_ENABLE_PROFILING,
        value: constants.DotnetInstrumentation.CORECLR_ENABLE_PROFILING_ENABLED,
    },
    {
        name: constants.DotnetInstrumentation.CORECLR_PROFILER,
        value: constants.DotnetInstrumentation.CORECLR_PROFILER_OTEL,
    },
];
/**
 * Node.js-specific implementation of the SDK injector.
 * Handles Node.js auto-instrumentation setup and NODE_OPTIONS configuration.
 */
class NodeInjector extends Injector {
    get command() {
        return ['cp', '-a', '/autoinstrumentation/.', this.containerPath];
    }
    injectAdditionalEnvironments(envsToInject, _envsFromTaskDef) {
        envsToInject[constants.NodeInstrumentation.NODE_OPTIONS] = ` --require ${this.containerPath}/autoinstrumentation.js`;
    }
    get containerPath() {
        return '/otel-auto-instrumentation-nodejs';
    }
    overrideAdditionalEnvironments(envsToOverride, envsFromTaskDef) {
        if (envsFromTaskDef[constants.NodeInstrumentation.NODE_OPTIONS]) {
            const originalNodeOptions = envsFromTaskDef[constants.NodeInstrumentation.NODE_OPTIONS] ?? '';
            let renderedNodeOptions = `${originalNodeOptions} --require ${this.containerPath}/autoinstrumentation.js`;
            envsToOverride[constants.NodeInstrumentation.NODE_OPTIONS] = renderedNodeOptions;
        }
    }
}
exports.NodeInjector = NodeInjector;
_g = JSII_RTTI_SYMBOL_1;
NodeInjector[_g] = { fqn: "@aws-cdk/aws-applicationsignals-alpha.NodeInjector", version: "2.202.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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