"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cluster = exports.ClientAuthentication = exports.ClientBrokerEncryption = exports.ClusterMonitoringLevel = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("aws-cdk-lib/aws-ec2");
const iam = require("aws-cdk-lib/aws-iam");
const kms = require("aws-cdk-lib/aws-kms");
const secretsmanager = require("aws-cdk-lib/aws-secretsmanager");
const core = require("aws-cdk-lib");
const cr = require("aws-cdk-lib/custom-resources");
const uniqueid_1 = require("constructs/lib/private/uniqueid");
const aws_msk_1 = require("aws-cdk-lib/aws-msk");
/**
 * A new or imported MSK Cluster.
 */
class ClusterBase extends core.Resource {
    /** Manages connections for the cluster */
    get connections() {
        if (!this._connections) {
            throw new Error('An imported Cluster cannot manage its security groups');
        }
        return this._connections;
    }
}
/**
 * The level of monitoring for the MSK cluster
 *
 * @see https://docs.aws.amazon.com/msk/latest/developerguide/monitoring.html#metrics-details
 */
var ClusterMonitoringLevel;
(function (ClusterMonitoringLevel) {
    /**
     * Default metrics are the essential metrics to monitor.
     */
    ClusterMonitoringLevel["DEFAULT"] = "DEFAULT";
    /**
     * Per Broker metrics give you metrics at the broker level.
     */
    ClusterMonitoringLevel["PER_BROKER"] = "PER_BROKER";
    /**
     * Per Topic Per Broker metrics help you understand volume at the topic level.
     */
    ClusterMonitoringLevel["PER_TOPIC_PER_BROKER"] = "PER_TOPIC_PER_BROKER";
    /**
     * Per Topic Per Partition metrics help you understand consumer group lag at the topic partition level.
     */
    ClusterMonitoringLevel["PER_TOPIC_PER_PARTITION"] = "PER_TOPIC_PER_PARTITION";
})(ClusterMonitoringLevel = exports.ClusterMonitoringLevel || (exports.ClusterMonitoringLevel = {}));
/**
 * Indicates the encryption setting for data in transit between clients and brokers.
 */
var ClientBrokerEncryption;
(function (ClientBrokerEncryption) {
    /**
     * TLS means that client-broker communication is enabled with TLS only.
     */
    ClientBrokerEncryption["TLS"] = "TLS";
    /**
     * TLS_PLAINTEXT means that client-broker communication is enabled for both TLS-encrypted, as well as plaintext data.
     */
    ClientBrokerEncryption["TLS_PLAINTEXT"] = "TLS_PLAINTEXT";
    /**
     * PLAINTEXT means that client-broker communication is enabled in plaintext only.
     */
    ClientBrokerEncryption["PLAINTEXT"] = "PLAINTEXT";
})(ClientBrokerEncryption = exports.ClientBrokerEncryption || (exports.ClientBrokerEncryption = {}));
/**
 * Configuration properties for client authentication.
 */
class ClientAuthentication {
    /**
     * @param saslProps - properties for SASL authentication
     * @param tlsProps - properties for TLS authentication
     */
    constructor(saslProps, tlsProps) {
        this.saslProps = saslProps;
        this.tlsProps = tlsProps;
    }
    /**
     * SASL authentication
     */
    static sasl(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_msk_alpha_SaslAuthProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.sasl);
            }
            throw error;
        }
        return new ClientAuthentication(props, undefined);
    }
    /**
     * TLS authentication
     */
    static tls(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_msk_alpha_TlsAuthProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.tls);
            }
            throw error;
        }
        return new ClientAuthentication(undefined, props);
    }
}
exports.ClientAuthentication = ClientAuthentication;
_a = JSII_RTTI_SYMBOL_1;
ClientAuthentication[_a] = { fqn: "@aws-cdk/aws-msk-alpha.ClientAuthentication", version: "2.27.0-alpha.0" };
/**
 * Create a MSK Cluster.
 *
 * @resource AWS::MSK::Cluster
 */
class Cluster extends ClusterBase {
    constructor(scope, id, props) {
        var _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s, _t, _u, _v, _w, _x, _y, _z, _0, _1, _2, _3, _4, _5, _6, _7, _8, _9, _10, _11, _12, _13, _14, _15, _16, _17, _18, _19, _20, _21, _22, _23;
        super(scope, id, {
            physicalName: props.clusterName,
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_msk_alpha_ClusterProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        const subnetSelection = props.vpc.selectSubnets(props.vpcSubnets);
        this._connections = new ec2.Connections({
            securityGroups: (_c = props.securityGroups) !== null && _c !== void 0 ? _c : [
                new ec2.SecurityGroup(this, 'SecurityGroup', {
                    description: 'MSK security group',
                    vpc: props.vpc,
                }),
            ],
        });
        if (subnetSelection.subnets.length < 2) {
            throw Error(`Cluster requires at least 2 subnets, got ${subnetSelection.subnets.length}`);
        }
        if (!core.Token.isUnresolved(props.clusterName) &&
            !/^[a-zA-Z0-9]+$/.test(props.clusterName) &&
            props.clusterName.length > 64) {
            throw Error('The cluster name must only contain alphanumeric characters and have a maximum length of 64 characters.' +
                `got: '${props.clusterName}. length: ${props.clusterName.length}'`);
        }
        if (((_e = (_d = props.clientAuthentication) === null || _d === void 0 ? void 0 : _d.saslProps) === null || _e === void 0 ? void 0 : _e.iam) && ((_g = (_f = props.clientAuthentication) === null || _f === void 0 ? void 0 : _f.saslProps) === null || _g === void 0 ? void 0 : _g.scram)) {
            throw Error('Only one client authentication method can be enabled.');
        }
        if (((_h = props.encryptionInTransit) === null || _h === void 0 ? void 0 : _h.clientBroker) ===
            ClientBrokerEncryption.PLAINTEXT &&
            props.clientAuthentication) {
            throw Error('To enable client authentication, you must enabled TLS-encrypted traffic between clients and brokers.');
        }
        else if (((_j = props.encryptionInTransit) === null || _j === void 0 ? void 0 : _j.clientBroker) ===
            ClientBrokerEncryption.TLS_PLAINTEXT &&
            (((_l = (_k = props.clientAuthentication) === null || _k === void 0 ? void 0 : _k.saslProps) === null || _l === void 0 ? void 0 : _l.scram) || ((_o = (_m = props.clientAuthentication) === null || _m === void 0 ? void 0 : _m.saslProps) === null || _o === void 0 ? void 0 : _o.iam))) {
            throw Error('To enable SASL/SCRAM or IAM authentication, you must only allow TLS-encrypted traffic between clients and brokers.');
        }
        const volumeSize = (_q = (_p = props.ebsStorageInfo) === null || _p === void 0 ? void 0 : _p.volumeSize) !== null && _q !== void 0 ? _q : 1000;
        // Minimum: 1 GiB, maximum: 16384 GiB
        if (volumeSize < 1 || volumeSize > 16384) {
            throw Error('EBS volume size should be in the range 1-16384');
        }
        const instanceType = props.instanceType
            ? this.mskInstanceType(props.instanceType)
            : this.mskInstanceType(ec2.InstanceType.of(ec2.InstanceClass.M5, ec2.InstanceSize.LARGE));
        const encryptionAtRest = ((_r = props.ebsStorageInfo) === null || _r === void 0 ? void 0 : _r.encryptionKey) ? {
            dataVolumeKmsKeyId: props.ebsStorageInfo.encryptionKey.keyId,
        }
            : undefined; // MSK will create the managed key
        const encryptionInTransit = {
            clientBroker: (_t = (_s = props.encryptionInTransit) === null || _s === void 0 ? void 0 : _s.clientBroker) !== null && _t !== void 0 ? _t : ClientBrokerEncryption.TLS,
            inCluster: (_v = (_u = props.encryptionInTransit) === null || _u === void 0 ? void 0 : _u.enableInCluster) !== null && _v !== void 0 ? _v : true,
        };
        const openMonitoring = ((_w = props.monitoring) === null || _w === void 0 ? void 0 : _w.enablePrometheusJmxExporter) || ((_x = props.monitoring) === null || _x === void 0 ? void 0 : _x.enablePrometheusNodeExporter)
            ? {
                prometheus: {
                    jmxExporter: ((_y = props.monitoring) === null || _y === void 0 ? void 0 : _y.enablePrometheusJmxExporter) ? { enabledInBroker: true }
                        : undefined,
                    nodeExporter: ((_z = props.monitoring) === null || _z === void 0 ? void 0 : _z.enablePrometheusNodeExporter) ? { enabledInBroker: true }
                        : undefined,
                },
            }
            : undefined;
        const loggingInfo = {
            brokerLogs: {
                cloudWatchLogs: {
                    enabled: ((_0 = props.logging) === null || _0 === void 0 ? void 0 : _0.cloudwatchLogGroup) !== undefined,
                    logGroup: (_2 = (_1 = props.logging) === null || _1 === void 0 ? void 0 : _1.cloudwatchLogGroup) === null || _2 === void 0 ? void 0 : _2.logGroupName,
                },
                firehose: {
                    enabled: ((_3 = props.logging) === null || _3 === void 0 ? void 0 : _3.firehoseDeliveryStreamName) !==
                        undefined,
                    deliveryStream: (_4 = props.logging) === null || _4 === void 0 ? void 0 : _4.firehoseDeliveryStreamName,
                },
                s3: {
                    enabled: ((_6 = (_5 = props.logging) === null || _5 === void 0 ? void 0 : _5.s3) === null || _6 === void 0 ? void 0 : _6.bucket) !== undefined,
                    bucket: (_8 = (_7 = props.logging) === null || _7 === void 0 ? void 0 : _7.s3) === null || _8 === void 0 ? void 0 : _8.bucket.bucketName,
                    prefix: (_10 = (_9 = props.logging) === null || _9 === void 0 ? void 0 : _9.s3) === null || _10 === void 0 ? void 0 : _10.prefix,
                },
            },
        };
        if (((_12 = (_11 = props.clientAuthentication) === null || _11 === void 0 ? void 0 : _11.saslProps) === null || _12 === void 0 ? void 0 : _12.scram) &&
            ((_14 = (_13 = props.clientAuthentication) === null || _13 === void 0 ? void 0 : _13.saslProps) === null || _14 === void 0 ? void 0 : _14.key) === undefined) {
            this.saslScramAuthenticationKey = new kms.Key(this, 'SASLKey', {
                description: 'Used for encrypting MSK secrets for SASL/SCRAM authentication.',
                alias: `msk/${props.clusterName}/sasl/scram`,
            });
            // https://docs.aws.amazon.com/kms/latest/developerguide/services-secrets-manager.html#asm-policies
            this.saslScramAuthenticationKey.addToResourcePolicy(new iam.PolicyStatement({
                sid: 'Allow access through AWS Secrets Manager for all principals in the account that are authorized to use AWS Secrets Manager',
                principals: [new iam.AnyPrincipal()],
                actions: [
                    'kms:Encrypt',
                    'kms:Decrypt',
                    'kms:ReEncrypt*',
                    'kms:GenerateDataKey*',
                    'kms:CreateGrant',
                    'kms:DescribeKey',
                ],
                resources: ['*'],
                conditions: {
                    StringEquals: {
                        'kms:ViaService': `secretsmanager.${core.Stack.of(this).region}.amazonaws.com`,
                        'kms:CallerAccount': core.Stack.of(this).account,
                    },
                },
            }));
        }
        let clientAuthentication;
        if ((_16 = (_15 = props.clientAuthentication) === null || _15 === void 0 ? void 0 : _15.saslProps) === null || _16 === void 0 ? void 0 : _16.iam) {
            clientAuthentication = {
                sasl: { iam: { enabled: props.clientAuthentication.saslProps.iam } },
            };
        }
        else if ((_18 = (_17 = props.clientAuthentication) === null || _17 === void 0 ? void 0 : _17.saslProps) === null || _18 === void 0 ? void 0 : _18.scram) {
            clientAuthentication = {
                sasl: {
                    scram: {
                        enabled: props.clientAuthentication.saslProps.scram,
                    },
                },
            };
        }
        else if (((_20 = (_19 = props.clientAuthentication) === null || _19 === void 0 ? void 0 : _19.tlsProps) === null || _20 === void 0 ? void 0 : _20.certificateAuthorities) !== undefined) {
            clientAuthentication = {
                tls: {
                    certificateAuthorityArnList: (_22 = (_21 = props.clientAuthentication) === null || _21 === void 0 ? void 0 : _21.tlsProps) === null || _22 === void 0 ? void 0 : _22.certificateAuthorities.map((ca) => ca.certificateAuthorityArn),
                },
            };
        }
        const resource = new aws_msk_1.CfnCluster(this, 'Resource', {
            clusterName: props.clusterName,
            kafkaVersion: props.kafkaVersion.version,
            numberOfBrokerNodes: props.numberOfBrokerNodes !== undefined ?
                subnetSelection.availabilityZones.length * props.numberOfBrokerNodes : subnetSelection.availabilityZones.length,
            brokerNodeGroupInfo: {
                instanceType,
                clientSubnets: subnetSelection.subnetIds,
                securityGroups: this.connections.securityGroups.map((group) => group.securityGroupId),
                storageInfo: {
                    ebsStorageInfo: {
                        volumeSize: volumeSize,
                    },
                },
            },
            encryptionInfo: {
                encryptionAtRest,
                encryptionInTransit,
            },
            configurationInfo: props.configurationInfo,
            enhancedMonitoring: (_23 = props.monitoring) === null || _23 === void 0 ? void 0 : _23.clusterMonitoringLevel,
            openMonitoring: openMonitoring,
            loggingInfo: loggingInfo,
            clientAuthentication: clientAuthentication,
        });
        this.clusterName = this.getResourceNameAttribute(core.Fn.select(1, core.Fn.split('/', resource.ref)));
        this.clusterArn = resource.ref;
        resource.applyRemovalPolicy(props.removalPolicy, {
            default: core.RemovalPolicy.RETAIN,
        });
    }
    /**
     * Reference an existing cluster, defined outside of the CDK code, by name.
     */
    static fromClusterArn(scope, id, clusterArn) {
        class Import extends ClusterBase {
            constructor() {
                super(...arguments);
                this.clusterArn = clusterArn;
                this.clusterName = core.Fn.select(1, core.Fn.split('/', clusterArn)); // ['arn:partition:kafka:region:account-id', clusterName, clusterId]
            }
        }
        return new Import(scope, id);
    }
    mskInstanceType(instanceType) {
        return `kafka.${instanceType.toString()}`;
    }
    /**
     * Get the ZooKeeper Connection string
     *
     * Uses a Custom Resource to make an API call to `describeCluster` using the Javascript SDK
     *
     * @param responseField Field to return from API call. eg. ZookeeperConnectString, ZookeeperConnectStringTls
     * @returns - The connection string to use to connect to the Apache ZooKeeper cluster.
     */
    _zookeeperConnectionString(responseField) {
        if (!this._clusterDescription) {
            this._clusterDescription = new cr.AwsCustomResource(this, 'ZookeeperConnect', {
                onUpdate: {
                    service: 'Kafka',
                    action: 'describeCluster',
                    parameters: {
                        ClusterArn: this.clusterArn,
                    },
                    physicalResourceId: cr.PhysicalResourceId.of('ZooKeeperConnectionString'),
                },
                policy: cr.AwsCustomResourcePolicy.fromSdkCalls({
                    resources: [this.clusterArn],
                }),
            });
        }
        return this._clusterDescription.getResponseField(`ClusterInfo.${responseField}`);
    }
    /**
     * Get the ZooKeeper Connection string
     *
     * Uses a Custom Resource to make an API call to `describeCluster` using the Javascript SDK
     *
     * @returns - The connection string to use to connect to the Apache ZooKeeper cluster.
     */
    get zookeeperConnectionString() {
        return this._zookeeperConnectionString('ZookeeperConnectString');
    }
    /**
     * Get the ZooKeeper Connection string for a TLS enabled cluster
     *
     * Uses a Custom Resource to make an API call to `describeCluster` using the Javascript SDK
     *
     * @returns - The connection string to use to connect to zookeeper cluster on TLS port.
     */
    get zookeeperConnectionStringTls() {
        return this._zookeeperConnectionString('ZookeeperConnectStringTls');
    }
    /**
     * Get the list of brokers that a client application can use to bootstrap
     *
     * Uses a Custom Resource to make an API call to `getBootstrapBrokers` using the Javascript SDK
     *
     * @param responseField Field to return from API call. eg. BootstrapBrokerStringSaslScram, BootstrapBrokerString
     * @returns - A string containing one or more hostname:port pairs.
     */
    _bootstrapBrokers(responseField) {
        if (!this._clusterBootstrapBrokers) {
            this._clusterBootstrapBrokers = new cr.AwsCustomResource(this, `BootstrapBrokers${responseField}`, {
                onUpdate: {
                    service: 'Kafka',
                    action: 'getBootstrapBrokers',
                    parameters: {
                        ClusterArn: this.clusterArn,
                    },
                    physicalResourceId: cr.PhysicalResourceId.of('BootstrapBrokers'),
                },
                policy: cr.AwsCustomResourcePolicy.fromSdkCalls({
                    resources: [this.clusterArn],
                }),
            });
        }
        return this._clusterBootstrapBrokers.getResponseField(responseField);
    }
    /**
     * Get the list of brokers that a client application can use to bootstrap
     *
     * Uses a Custom Resource to make an API call to `getBootstrapBrokers` using the Javascript SDK
     *
     * @returns - A string containing one or more hostname:port pairs.
     */
    get bootstrapBrokers() {
        return this._bootstrapBrokers('BootstrapBrokerString');
    }
    /**
     * Get the list of brokers that a TLS authenticated client application can use to bootstrap
     *
     * Uses a Custom Resource to make an API call to `getBootstrapBrokers` using the Javascript SDK
     *
     * @returns - A string containing one or more DNS names (or IP) and TLS port pairs.
     */
    get bootstrapBrokersTls() {
        return this._bootstrapBrokers('BootstrapBrokerStringTls');
    }
    /**
     * Get the list of brokers that a SASL/SCRAM authenticated client application can use to bootstrap
     *
     * Uses a Custom Resource to make an API call to `getBootstrapBrokers` using the Javascript SDK
     *
     * @returns - A string containing one or more dns name (or IP) and SASL SCRAM port pairs.
     */
    get bootstrapBrokersSaslScram() {
        return this._bootstrapBrokers('BootstrapBrokerStringSaslScram');
    }
    /**
     * A list of usersnames to register with the cluster. The password will automatically be generated using Secrets
     * Manager and the { username, password } JSON object stored in Secrets Manager as `AmazonMSK_username`.
     *
     * Must be using the SASL/SCRAM authentication mechanism.
     *
     * @param usernames - username(s) to register with the cluster
     */
    addUser(...usernames) {
        var _c;
        if (this.saslScramAuthenticationKey) {
            const MSK_SECRET_PREFIX = 'AmazonMSK_'; // Required
            const secrets = usernames.map((username) => new secretsmanager.Secret(this, `KafkaUser${username}`, {
                secretName: `${MSK_SECRET_PREFIX}${this.clusterName}_${username}`,
                generateSecretString: {
                    secretStringTemplate: JSON.stringify({ username }),
                    generateStringKey: 'password',
                },
                encryptionKey: this.saslScramAuthenticationKey,
            }));
            new cr.AwsCustomResource(this, `BatchAssociateScramSecrets${uniqueid_1.addressOf(usernames)}`, {
                onUpdate: {
                    service: 'Kafka',
                    action: 'batchAssociateScramSecret',
                    parameters: {
                        ClusterArn: this.clusterArn,
                        SecretArnList: secrets.map((secret) => secret.secretArn),
                    },
                    physicalResourceId: cr.PhysicalResourceId.of('CreateUsers'),
                },
                policy: cr.AwsCustomResourcePolicy.fromStatements([
                    new iam.PolicyStatement({
                        actions: ['kms:CreateGrant'],
                        resources: [(_c = this.saslScramAuthenticationKey) === null || _c === void 0 ? void 0 : _c.keyArn],
                    }),
                    new iam.PolicyStatement({
                        actions: ['kafka:BatchAssociateScramSecret'],
                        resources: [this.clusterArn],
                    }),
                ]),
            });
        }
        else {
            throw Error('Cannot create users if an authentication KMS key has not been created/provided.');
        }
    }
}
exports.Cluster = Cluster;
_b = JSII_RTTI_SYMBOL_1;
Cluster[_b] = { fqn: "@aws-cdk/aws-msk-alpha.Cluster", version: "2.27.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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