"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_secretsmanager_1 = require("monocdk/aws-secretsmanager");
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../../nag-pack");
/**
 * Secrets have automatic rotation scheduled
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_secretsmanager_1.CfnSecret) {
        const secretLogicalId = nag_pack_1.resolveResourceFromInstrinsic(node, node.ref);
        const secretTargetAttachmentLogicalIds = Array();
        const cfnSecretTargetAttachments = Array();
        const cfnRotationSchedules = Array();
        for (const child of monocdk_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_secretsmanager_1.CfnSecretTargetAttachment) {
                cfnSecretTargetAttachments.push(child);
            }
            else if (child instanceof aws_secretsmanager_1.CfnRotationSchedule) {
                cfnRotationSchedules.push(child);
            }
        }
        if (cfnRotationSchedules.length === 0) {
            return false;
        }
        let found = false;
        for (const child of cfnSecretTargetAttachments) {
            const attachmentLogicalId = getMatchingSecretTargetAttachment(child, secretLogicalId);
            if (attachmentLogicalId) {
                secretTargetAttachmentLogicalIds.push(attachmentLogicalId);
            }
        }
        for (const child of cfnRotationSchedules) {
            if (isMatchingRotationSchedule(child, secretLogicalId, secretTargetAttachmentLogicalIds)) {
                found = true;
                break;
            }
        }
        if (!found) {
            return false;
        }
    }
    return true;
}, 'name', { value: path_1.parse(__filename).name });
/**
 * Helper function to check whether a given Secret Target Attachment is associated with the given secret.
 * @param node The CfnTargetAttachment to check.
 * @param secretLogicalId The Cfn Logical ID of the secret.
 * Returns the Logical ID if the attachment if is associated with the secret, otherwise and empty string.
 */
function getMatchingSecretTargetAttachment(node, secretLogicalId) {
    const resourceSecretId = nag_pack_1.resolveResourceFromInstrinsic(node, node.secretId);
    if (secretLogicalId === resourceSecretId) {
        return nag_pack_1.resolveResourceFromInstrinsic(node, node.ref);
    }
    return '';
}
/**
 * Helper function to check whether a given Rotation Schedule is associated with the given secret.
 * @param node The CfnRotationSchedule to check.
 * @param secretLogicalId The Cfn Logical ID of the secret.
 * @param secretTargetAttachmentLogicalIds The Cfn Logical IDs of any Secret Target Attachments associated with the given secret.
 * Returns whether the CfnRotationSchedule is associated with the given secret.
 */
function isMatchingRotationSchedule(node, secretLogicalId, secretTargetAttachmentLogicalIds) {
    const resourceSecretId = nag_pack_1.resolveResourceFromInstrinsic(node, node.secretId);
    if (secretLogicalId === resourceSecretId ||
        secretTargetAttachmentLogicalIds.includes(resourceSecretId)) {
        if (monocdk_1.Stack.of(node).resolve(node.hostedRotationLambda) === undefined &&
            monocdk_1.Stack.of(node).resolve(node.rotationLambdaArn) === undefined) {
            return false;
        }
        const rotationRules = monocdk_1.Stack.of(node).resolve(node.rotationRules);
        if (rotationRules !== undefined) {
            const automaticallyAfterDays = monocdk_1.Stack.of(node).resolve(rotationRules.automaticallyAfterDays);
            if (automaticallyAfterDays !== undefined) {
                return true;
            }
        }
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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