"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_redshift_1 = require("monocdk/aws-redshift");
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../../nag-pack");
/**
 * Redshift clusters require TLS/SSL encryption
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_redshift_1.CfnCluster) {
        const clusterParameterGroupName = nag_pack_1.resolveResourceFromInstrinsic(node, node.clusterParameterGroupName);
        if (clusterParameterGroupName === undefined) {
            return false;
        }
        let found = false;
        for (const child of monocdk_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_redshift_1.CfnClusterParameterGroup) {
                if (isMatchingParameterGroup(child, clusterParameterGroupName)) {
                    found = true;
                    break;
                }
            }
        }
        if (!found) {
            return false;
        }
    }
    return true;
}, 'name', { value: path_1.parse(__filename).name });
/**
 * Helper function to check whether the Parameter Group enforces SSL and is associated with the given Redshift cluster
 * @param node the CfnClusterParameterGroup to check
 * @param parameterGroupName The name of the associated parameter group
 * returns whether the CfnClusterParameterGroup enforces SSL and is associated given Redshift cluster
 */
function isMatchingParameterGroup(node, parameterGroupName) {
    const parameterGroupLogicalId = nag_pack_1.resolveResourceFromInstrinsic(node, node.ref);
    if (parameterGroupName !== parameterGroupLogicalId ||
        node.parameters == undefined) {
        return false;
    }
    const parameters = monocdk_1.Stack.of(node).resolve(node.parameters);
    for (const parameter of parameters) {
        const resolvedParameter = monocdk_1.Stack.of(node).resolve(parameter);
        if (resolvedParameter.parameterName.toLowerCase() == 'require_ssl' &&
            resolvedParameter.parameterValue.toLowerCase() == 'true') {
            return true;
        }
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiUmVkc2hpZnRSZXF1aXJlVGxzU1NMLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vLi4vc3JjL3J1bGVzL3JlZHNoaWZ0L1JlZHNoaWZ0UmVxdWlyZVRsc1NTTC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUFBOzs7RUFHRTtBQUNGLCtCQUE2QjtBQUM3Qix1REFBNEU7QUFDNUUscUNBQTZDO0FBQzdDLDZDQUErRDtBQUUvRDs7O0dBR0c7QUFFSCxrQkFBZSxNQUFNLENBQUMsY0FBYyxDQUNsQyxDQUFDLElBQWlCLEVBQVcsRUFBRTtJQUM3QixJQUFJLElBQUksWUFBWSx5QkFBVSxFQUFFO1FBQzlCLE1BQU0seUJBQXlCLEdBQUcsd0NBQTZCLENBQzdELElBQUksRUFDSixJQUFJLENBQUMseUJBQXlCLENBQy9CLENBQUM7UUFDRixJQUFJLHlCQUF5QixLQUFLLFNBQVMsRUFBRTtZQUMzQyxPQUFPLEtBQUssQ0FBQztTQUNkO1FBQ0QsSUFBSSxLQUFLLEdBQUcsS0FBSyxDQUFDO1FBQ2xCLEtBQUssTUFBTSxLQUFLLElBQUksZUFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxJQUFJLENBQUMsT0FBTyxFQUFFLEVBQUU7WUFDakQsSUFBSSxLQUFLLFlBQVksdUNBQXdCLEVBQUU7Z0JBQzdDLElBQUksd0JBQXdCLENBQUMsS0FBSyxFQUFFLHlCQUF5QixDQUFDLEVBQUU7b0JBQzlELEtBQUssR0FBRyxJQUFJLENBQUM7b0JBQ2IsTUFBTTtpQkFDUDthQUNGO1NBQ0Y7UUFDRCxJQUFJLENBQUMsS0FBSyxFQUFFO1lBQ1YsT0FBTyxLQUFLLENBQUM7U0FDZDtLQUNGO0lBQ0QsT0FBTyxJQUFJLENBQUM7QUFDZCxDQUFDLEVBQ0QsTUFBTSxFQUNOLEVBQUUsS0FBSyxFQUFFLFlBQUssQ0FBQyxVQUFVLENBQUMsQ0FBQyxJQUFJLEVBQUUsQ0FDbEMsQ0FBQztBQUVGOzs7OztHQUtHO0FBQ0gsU0FBUyx3QkFBd0IsQ0FDL0IsSUFBOEIsRUFDOUIsa0JBQTBCO0lBRTFCLE1BQU0sdUJBQXVCLEdBQUcsd0NBQTZCLENBQUMsSUFBSSxFQUFFLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUM5RSxJQUNFLGtCQUFrQixLQUFLLHVCQUF1QjtRQUM5QyxJQUFJLENBQUMsVUFBVSxJQUFJLFNBQVMsRUFDNUI7UUFDQSxPQUFPLEtBQUssQ0FBQztLQUNkO0lBQ0QsTUFBTSxVQUFVLEdBQUcsZUFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUFDO0lBQzNELEtBQUssTUFBTSxTQUFTLElBQUksVUFBVSxFQUFFO1FBQ2xDLE1BQU0saUJBQWlCLEdBQUcsZUFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDLENBQUM7UUFDNUQsSUFDRSxpQkFBaUIsQ0FBQyxhQUFhLENBQUMsV0FBVyxFQUFFLElBQUksYUFBYTtZQUM5RCxpQkFBaUIsQ0FBQyxjQUFjLENBQUMsV0FBVyxFQUFFLElBQUksTUFBTSxFQUN4RDtZQUNBLE9BQU8sSUFBSSxDQUFDO1NBQ2I7S0FDRjtJQUNELE9BQU8sS0FBSyxDQUFDO0FBQ2YsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8qXG5Db3B5cmlnaHQgQW1hem9uLmNvbSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cblNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBBcGFjaGUtMi4wXG4qL1xuaW1wb3J0IHsgcGFyc2UgfSBmcm9tICdwYXRoJztcbmltcG9ydCB7IENmbkNsdXN0ZXIsIENmbkNsdXN0ZXJQYXJhbWV0ZXJHcm91cCB9IGZyb20gJ21vbm9jZGsvYXdzLXJlZHNoaWZ0JztcbmltcG9ydCB7IENmblJlc291cmNlLCBTdGFjayB9IGZyb20gJ21vbm9jZGsnO1xuaW1wb3J0IHsgcmVzb2x2ZVJlc291cmNlRnJvbUluc3RyaW5zaWMgfSBmcm9tICcuLi8uLi9uYWctcGFjayc7XG5cbi8qKlxuICogUmVkc2hpZnQgY2x1c3RlcnMgcmVxdWlyZSBUTFMvU1NMIGVuY3J5cHRpb25cbiAqIEBwYXJhbSBub2RlIHRoZSBDZm5SZXNvdXJjZSB0byBjaGVja1xuICovXG5cbmV4cG9ydCBkZWZhdWx0IE9iamVjdC5kZWZpbmVQcm9wZXJ0eShcbiAgKG5vZGU6IENmblJlc291cmNlKTogYm9vbGVhbiA9PiB7XG4gICAgaWYgKG5vZGUgaW5zdGFuY2VvZiBDZm5DbHVzdGVyKSB7XG4gICAgICBjb25zdCBjbHVzdGVyUGFyYW1ldGVyR3JvdXBOYW1lID0gcmVzb2x2ZVJlc291cmNlRnJvbUluc3RyaW5zaWMoXG4gICAgICAgIG5vZGUsXG4gICAgICAgIG5vZGUuY2x1c3RlclBhcmFtZXRlckdyb3VwTmFtZVxuICAgICAgKTtcbiAgICAgIGlmIChjbHVzdGVyUGFyYW1ldGVyR3JvdXBOYW1lID09PSB1bmRlZmluZWQpIHtcbiAgICAgICAgcmV0dXJuIGZhbHNlO1xuICAgICAgfVxuICAgICAgbGV0IGZvdW5kID0gZmFsc2U7XG4gICAgICBmb3IgKGNvbnN0IGNoaWxkIG9mIFN0YWNrLm9mKG5vZGUpLm5vZGUuZmluZEFsbCgpKSB7XG4gICAgICAgIGlmIChjaGlsZCBpbnN0YW5jZW9mIENmbkNsdXN0ZXJQYXJhbWV0ZXJHcm91cCkge1xuICAgICAgICAgIGlmIChpc01hdGNoaW5nUGFyYW1ldGVyR3JvdXAoY2hpbGQsIGNsdXN0ZXJQYXJhbWV0ZXJHcm91cE5hbWUpKSB7XG4gICAgICAgICAgICBmb3VuZCA9IHRydWU7XG4gICAgICAgICAgICBicmVhaztcbiAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICAgIGlmICghZm91bmQpIHtcbiAgICAgICAgcmV0dXJuIGZhbHNlO1xuICAgICAgfVxuICAgIH1cbiAgICByZXR1cm4gdHJ1ZTtcbiAgfSxcbiAgJ25hbWUnLFxuICB7IHZhbHVlOiBwYXJzZShfX2ZpbGVuYW1lKS5uYW1lIH1cbik7XG5cbi8qKlxuICogSGVscGVyIGZ1bmN0aW9uIHRvIGNoZWNrIHdoZXRoZXIgdGhlIFBhcmFtZXRlciBHcm91cCBlbmZvcmNlcyBTU0wgYW5kIGlzIGFzc29jaWF0ZWQgd2l0aCB0aGUgZ2l2ZW4gUmVkc2hpZnQgY2x1c3RlclxuICogQHBhcmFtIG5vZGUgdGhlIENmbkNsdXN0ZXJQYXJhbWV0ZXJHcm91cCB0byBjaGVja1xuICogQHBhcmFtIHBhcmFtZXRlckdyb3VwTmFtZSBUaGUgbmFtZSBvZiB0aGUgYXNzb2NpYXRlZCBwYXJhbWV0ZXIgZ3JvdXBcbiAqIHJldHVybnMgd2hldGhlciB0aGUgQ2ZuQ2x1c3RlclBhcmFtZXRlckdyb3VwIGVuZm9yY2VzIFNTTCBhbmQgaXMgYXNzb2NpYXRlZCBnaXZlbiBSZWRzaGlmdCBjbHVzdGVyXG4gKi9cbmZ1bmN0aW9uIGlzTWF0Y2hpbmdQYXJhbWV0ZXJHcm91cChcbiAgbm9kZTogQ2ZuQ2x1c3RlclBhcmFtZXRlckdyb3VwLFxuICBwYXJhbWV0ZXJHcm91cE5hbWU6IHN0cmluZ1xuKTogYm9vbGVhbiB7XG4gIGNvbnN0IHBhcmFtZXRlckdyb3VwTG9naWNhbElkID0gcmVzb2x2ZVJlc291cmNlRnJvbUluc3RyaW5zaWMobm9kZSwgbm9kZS5yZWYpO1xuICBpZiAoXG4gICAgcGFyYW1ldGVyR3JvdXBOYW1lICE9PSBwYXJhbWV0ZXJHcm91cExvZ2ljYWxJZCB8fFxuICAgIG5vZGUucGFyYW1ldGVycyA9PSB1bmRlZmluZWRcbiAgKSB7XG4gICAgcmV0dXJuIGZhbHNlO1xuICB9XG4gIGNvbnN0IHBhcmFtZXRlcnMgPSBTdGFjay5vZihub2RlKS5yZXNvbHZlKG5vZGUucGFyYW1ldGVycyk7XG4gIGZvciAoY29uc3QgcGFyYW1ldGVyIG9mIHBhcmFtZXRlcnMpIHtcbiAgICBjb25zdCByZXNvbHZlZFBhcmFtZXRlciA9IFN0YWNrLm9mKG5vZGUpLnJlc29sdmUocGFyYW1ldGVyKTtcbiAgICBpZiAoXG4gICAgICByZXNvbHZlZFBhcmFtZXRlci5wYXJhbWV0ZXJOYW1lLnRvTG93ZXJDYXNlKCkgPT0gJ3JlcXVpcmVfc3NsJyAmJlxuICAgICAgcmVzb2x2ZWRQYXJhbWV0ZXIucGFyYW1ldGVyVmFsdWUudG9Mb3dlckNhc2UoKSA9PSAndHJ1ZSdcbiAgICApIHtcbiAgICAgIHJldHVybiB0cnVlO1xuICAgIH1cbiAgfVxuICByZXR1cm4gZmFsc2U7XG59XG4iXX0=