"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_elasticloadbalancing_1 = require("monocdk/aws-elasticloadbalancing");
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../../nag-pack");
/**
 * CLBs use ACM-managed certificates
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_elasticloadbalancing_1.CfnLoadBalancer) {
        //For each listener, ensure that it's utilizing an ACM SSL/HTTPS cert
        const listeners = monocdk_1.Stack.of(node).resolve(node.listeners);
        if (listeners != undefined) {
            //Iterate through listeners, checking if secured ACM certs are used
            for (const listener of listeners) {
                const resolvedListener = monocdk_1.Stack.of(node).resolve(listener);
                const listenerARN = nag_pack_1.resolveIfPrimitive(node, resolvedListener.sslCertificateId);
                //Use the ARN to check if this is an ACM managed cert
                if (listenerARN == undefined) {
                    return false;
                }
                else {
                    const acmRegex = /^arn:[^:]+:acm:.+$/;
                    if (!acmRegex.test(listenerARN)) {
                        return false;
                    }
                }
            }
        }
    }
    return true;
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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