"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_elasticloadbalancingv2_1 = require("monocdk/aws-elasticloadbalancingv2");
const aws_wafv2_1 = require("monocdk/aws-wafv2");
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../../nag-pack");
/**
 * ALBs are associated with AWS WAFv2 web ACLs
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_elasticloadbalancingv2_1.CfnLoadBalancer) {
        const type = nag_pack_1.resolveIfPrimitive(node, node.type);
        if (type === undefined || type === 'application') {
            const loadBalancerLogicalId = nag_pack_1.resolveResourceFromInstrinsic(node, node.ref);
            let found = false;
            for (const child of monocdk_1.Stack.of(node).node.findAll()) {
                if (child instanceof aws_wafv2_1.CfnWebACLAssociation) {
                    if (isMatchingWebACLAssociation(child, loadBalancerLogicalId)) {
                        found = true;
                        break;
                    }
                }
            }
            if (!found) {
                return false;
            }
        }
    }
    return true;
}, 'name', { value: path_1.parse(__filename).name });
/**
 * Helper function to check whether a given Web ACL Association is associated with the given Load Balancer
 * @param node the CfnWebACLAssociation to check
 * @param loadBalancerLogicalId the Cfn Logical ID of the Load Balancer
 * returns whether the CfnWebACLAssociation is associates with the given Load Balancer
 */
function isMatchingWebACLAssociation(node, loadBalancerLogicalId) {
    const resourceLogicalId = JSON.stringify(monocdk_1.Stack.of(node).resolve(node.resourceArn));
    if (new RegExp(`${loadBalancerLogicalId}(?![\\w])`, 'gm').test(resourceLogicalId)) {
        return true;
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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