"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_backup_1 = require("monocdk/aws-backup");
const aws_efs_1 = require("monocdk/aws-efs");
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../../nag-pack");
/**
 * EFSs are part of AWS Backup plan(s)
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_efs_1.CfnFileSystem) {
        const fileSystemLogicalId = nag_pack_1.resolveResourceFromInstrinsic(node, node.ref);
        let found = false;
        for (const child of monocdk_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_backup_1.CfnBackupSelection) {
                if (isMatchingSelection(child, fileSystemLogicalId)) {
                    found = true;
                    break;
                }
            }
        }
        if (!found) {
            return false;
        }
    }
    return true;
}, 'name', { value: path_1.parse(__filename).name });
/**
 * Helper function to check whether the Backup Plan Selection contains the given File System
 * @param node the CfnBackupSelection to check
 * @param fileSystemLogicalId the Cfn Logical ID of the File System
 * returns whether the CfnBackupSelection contains the given File System
 */
function isMatchingSelection(node, fileSystemLogicalId) {
    const backupSelection = monocdk_1.Stack.of(node).resolve(node.backupSelection);
    const resources = monocdk_1.Stack.of(node).resolve(backupSelection.resources);
    if (Array.isArray(resources)) {
        for (const resource of resources) {
            const resolvedResource = JSON.stringify(monocdk_1.Stack.of(node).resolve(resource));
            if (new RegExp(`${fileSystemLogicalId}(?![\\w])`).test(resolvedResource)) {
                return true;
            }
        }
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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