"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_ecs_1 = require("monocdk/aws-ecs");
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../../nag-pack");
/**
 * Containers in ECS task definitions configured for host networking have 'privileged' set to true and a non-empty non-root 'user'
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_ecs_1.CfnTaskDefinition) {
        if (node.networkMode === aws_ecs_1.NetworkMode.HOST) {
            const containerDefinitions = monocdk_1.Stack.of(node).resolve(node.containerDefinitions);
            if (containerDefinitions !== undefined) {
                for (const containerDefinition of containerDefinitions) {
                    const resolvedDefinition = monocdk_1.Stack.of(node).resolve(containerDefinition);
                    const privileged = nag_pack_1.resolveIfPrimitive(node, resolvedDefinition.privileged);
                    const user = nag_pack_1.resolveIfPrimitive(node, resolvedDefinition.user);
                    if (privileged !== true || user === undefined) {
                        return false;
                    }
                    const rootIdentifiers = ['root', '0'];
                    const userParts = user.split(':');
                    for (const userPart of userParts) {
                        if (rootIdentifiers.includes(userPart.toLowerCase())) {
                            return false;
                        }
                    }
                }
            }
        }
    }
    return true;
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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