"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_ec2_1 = require("monocdk/aws-ec2");
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../../nag-pack");
const BLOCKED_PORTS = [20, 21, 3389, 3309, 3306, 4333];
/**
 * EC2 instances have all common TCP ports restricted for ingress IPv4 traffic
 * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/default-custom-security-groups.html
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_ec2_1.CfnSecurityGroup) {
        const ingressRules = monocdk_1.Stack.of(node).resolve(node.securityGroupIngress);
        if (ingressRules != undefined) {
            //For each ingress rule, ensure that it does not allow unrestricted SSH traffic.
            for (const rule of ingressRules) {
                const resolvedRule = monocdk_1.Stack.of(node).resolve(rule);
                for (const portNum of BLOCKED_PORTS) {
                    if (!testPort(node, resolvedRule, portNum)) {
                        return false;
                    }
                }
            }
        }
    }
    else if (node instanceof aws_ec2_1.CfnSecurityGroupIngress) {
        for (const portNum of BLOCKED_PORTS) {
            if (!testPort(node, node, portNum)) {
                return false;
            }
        }
    }
    return true;
}, 'name', { value: path_1.parse(__filename).name });
/**
 * Helper function to identify if the given port number is unrestricted
 * @param node the CfnResource to check
 * @param rule the CfnSecurityGroupIngress rule to check
 * @param portNum the number of the port to check
 */
function testPort(node, rule, portNum) {
    //Does this rule apply to TCP traffic?
    const ipProtocol = nag_pack_1.resolveIfPrimitive(node, rule.ipProtocol);
    const cidrIp = nag_pack_1.resolveIfPrimitive(node, rule.cidrIp);
    const fromPort = nag_pack_1.resolveIfPrimitive(node, rule.fromPort);
    const toPort = nag_pack_1.resolveIfPrimitive(node, rule.toPort);
    if (ipProtocol === 'tcp') {
        //Does this rule allow all IPv4 addresses (unrestricted access)?
        if (cidrIp != undefined && cidrIp.includes('/0')) {
            //Is a port range specified?
            if (fromPort != undefined && toPort != undefined) {
                if ((fromPort <= portNum && toPort >= portNum) ||
                    fromPort == -1 ||
                    toPort == -1) {
                    return false;
                }
            }
            else {
                if (fromPort == portNum) {
                    return false;
                }
            }
        }
    }
    //Are all ports allowed?
    if (ipProtocol === '-1') {
        return false;
    }
    return true;
}
//# sourceMappingURL=data:application/json;base64,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