"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_ec2_1 = require("monocdk/aws-ec2");
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../../nag-pack");
const EBS_OPTIMIZED_SUPPORTED = [
    'c1.xlarge',
    'c3.xlarge',
    'c3.2xlarge',
    'c3.4xlarge',
    'g2.2xlarge',
    'i2.xlarge',
    'i2.2xlarge',
    'i2.4xlarge',
    'm1.large',
    'm1.xlarge',
    'm2.2xlarge',
    'm2.4xlarge',
    'm3.xlarge',
    'm3.2xlarge',
    'r3.xlarge',
    'r3.2xlarge',
    'r3.4xlarge',
];
const DEFAULT_TYPE = 'm1.small';
/**
 * EC2 instance types that support EBS optimization and are not EBS optimized by default have EBS optimization enabled
 * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-optimized.html#previous
 *  @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_ec2_1.CfnInstance) {
        const instanceType = node.instanceType
            ? nag_pack_1.resolveIfPrimitive(node, node.instanceType)
            : DEFAULT_TYPE;
        const ebsOptimized = monocdk_1.Stack.of(node).resolve(node.ebsOptimized);
        if (EBS_OPTIMIZED_SUPPORTED.includes(instanceType) &&
            ebsOptimized !== true) {
            return false;
        }
    }
    return true;
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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