"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_backup_1 = require("monocdk/aws-backup");
const aws_ec2_1 = require("monocdk/aws-ec2");
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../../nag-pack");
/**
 * EBS volumes are part of AWS Backup plan(s)
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_ec2_1.CfnVolume) {
        const volumeLogicalId = nag_pack_1.resolveResourceFromInstrinsic(node, node.ref);
        let found = false;
        for (const child of monocdk_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_backup_1.CfnBackupSelection) {
                if (isMatchingSelection(child, volumeLogicalId)) {
                    found = true;
                    break;
                }
            }
        }
        if (!found) {
            return false;
        }
    }
    return true;
}, 'name', { value: path_1.parse(__filename).name });
/**
 * Helper function to check whether the Backup Plan Selection contains the given volume
 * @param node the CfnBackupSelection to check
 * @param volumeLogicalId the Cfn Logical ID of the volume
 * returns whether the CfnBackupSelection contains the given volume
 */
function isMatchingSelection(node, volumeLogicalId) {
    const backupSelection = monocdk_1.Stack.of(node).resolve(node.backupSelection);
    const resources = monocdk_1.Stack.of(node).resolve(backupSelection.resources);
    if (Array.isArray(resources)) {
        for (const resource of resources) {
            const resolvedResource = JSON.stringify(monocdk_1.Stack.of(node).resolve(resource));
            if (new RegExp(`${volumeLogicalId}(?![\\w])`).test(resolvedResource)) {
                return true;
            }
        }
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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