"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_applicationautoscaling_1 = require("monocdk/aws-applicationautoscaling");
const aws_dynamodb_1 = require("monocdk/aws-dynamodb");
const monocdk_1 = require("monocdk");
const nag_pack_1 = require("../../nag-pack");
/**
 * Provisioned capacity DynamoDB tables have auto-scaling enabled on their indexes
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_dynamodb_1.CfnTable) {
        if (nag_pack_1.resolveResourceFromInstrinsic(node, node.billingMode) !==
            aws_dynamodb_1.BillingMode.PAY_PER_REQUEST) {
            const indexWriteMatches = [''];
            const indexReadMatches = [''];
            const tableLogicalId = nag_pack_1.resolveResourceFromInstrinsic(node, node.ref);
            const tableName = monocdk_1.Stack.of(node).resolve(node.tableName);
            const globalSecondaryIndexes = monocdk_1.Stack.of(node).resolve(node.globalSecondaryIndexes);
            if (Array.isArray(globalSecondaryIndexes)) {
                globalSecondaryIndexes.forEach((gsi) => {
                    const resolvedGsi = monocdk_1.Stack.of(node).resolve(gsi);
                    indexWriteMatches.push(resolvedGsi.indexName);
                    indexReadMatches.push(resolvedGsi.indexName);
                });
            }
            for (const child of monocdk_1.Stack.of(node).node.findAll()) {
                if (child instanceof aws_applicationautoscaling_1.CfnScalableTarget) {
                    const writeMatchIndex = isMatchingScalableTarget(child, 'WriteCapacityUnits', tableLogicalId, tableName, indexWriteMatches);
                    if (writeMatchIndex !== -1) {
                        indexWriteMatches.splice(writeMatchIndex, 1);
                        continue;
                    }
                    const readMatchIndex = isMatchingScalableTarget(child, 'ReadCapacityUnits', tableLogicalId, tableName, indexReadMatches);
                    if (readMatchIndex !== -1) {
                        indexReadMatches.splice(readMatchIndex, 1);
                    }
                }
            }
            if (indexWriteMatches.length != 0 || indexReadMatches.length != 0) {
                return false;
            }
        }
    }
    return true;
}, 'name', { value: path_1.parse(__filename).name });
/**
 * Helper function to check whether the CfnScalableTarget is related to the given Table index
 * @param node the CfnScalableTarget to check
 * @param dimension the dimension of the CfnScalableTarget to check
 * @param tableLogicalId the Cfn Logical ID of the table
 * @param tableName the name of the table
 * @param indexNames the names of the indexes to check against
 * returns the location in the indexNames array of the matching index or -1 if no match is found
 */
function isMatchingScalableTarget(node, dimension, tableLogicalId, tableName, indexNames) {
    if (node.serviceNamespace === 'dynamodb') {
        let scalableDimension = '';
        const resourceId = JSON.stringify(monocdk_1.Stack.of(node).resolve(node.resourceId));
        for (let i = 0; i < indexNames.length; i++) {
            const regexes = Array();
            const indexName = indexNames[i];
            if (indexName === '') {
                regexes.push(`${tableLogicalId}.*`);
                if (tableName !== undefined) {
                    regexes.push(`${tableName}.*`);
                }
                scalableDimension = `dynamodb:table:${dimension}`;
            }
            else {
                regexes.push(`${tableLogicalId}.*index\/${indexName}(?![\\w\\-_\\.])`);
                if (tableName !== undefined) {
                    regexes.push(`${tableName}.*index\/${indexName}(?![\\w\\-_\\.])`);
                }
                scalableDimension = `dynamodb:index:${dimension}`;
            }
            const regex = new RegExp(regexes.join('|'), 'gm');
            if (node.scalableDimension === scalableDimension &&
                regex.test(resourceId)) {
                return i;
            }
        }
    }
    return -1;
}
//# sourceMappingURL=data:application/json;base64,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